/* xml++.h
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 */

#include <string>
#include <list>
#include <map>

#include <libxml/parser.h>
#include <libxml/tree.h>
#include <stdarg.h>
#include <stdio.h>

#ifndef __XML_H
#define __XML_H

class XMLTree;
class XMLNode;
typedef std::list<XMLNode *> XMLNodeList;
typedef XMLNodeList::iterator XMLNodeIterator;
typedef XMLNodeList::const_iterator XMLNodeConstIterator;
class XMLProperty;
typedef std::list<XMLProperty *> XMLPropertyList;
typedef XMLPropertyList::iterator XMLPropertyIterator;
typedef XMLPropertyList::const_iterator XMLPropertyConstIterator;
typedef std::map<std::string, XMLProperty *> XMLPropertyMap;

class XMLParserCallback {
public:
  virtual void start_document(void) { };
  virtual void end_document(void) { };
  virtual void start_element(const std::string &n, const XMLPropertyMap &p) { };
  virtual void end_element(const std::string &n) { };
  virtual void characters(const std::string &s) { };
  virtual void comment(const std::string &s) { };
  virtual void warning(const std::string &s) { };
  virtual void error(const std::string &s) { };
  virtual void fatal_error(const std::string &s) { };
};

template <class ParserCallback = XMLParserCallback>
class XMLParser {
private:
  XMLParserCallback *_parser_callback;
  xmlParserCtxtPtr _context;

  static xmlEntityPtr _get_entity(void *_parser, const xmlChar *n) {
    return xmlGetPredefinedEntity(n);
  };

  static void _start_document(void *_parser) {
    XMLParser *parser;

    parser = (XMLParser *) _parser;
    parser->start_document();
  };

  static void _end_document(void *_parser) {
    XMLParser *parser;

    parser = (XMLParser *) _parser;
    parser->end_document();
  };

  static void _start_element(void *_parser, const xmlChar *n,
				const xmlChar **p) {
    XMLParser *parser;
    XMLPropertyMap properties;

    if(p) {
      const xmlChar **cur;

      for(cur = p; cur && *cur; cur++) {
        std::string name, value;

        name = std::string((const char *) *cur++);
        value = std::string((const char *) *cur);

        properties[name] = new XMLProperty(name, value);
      }
    }

    parser = (XMLParser *) _parser;
    parser->start_element(std::string((const char *) n), properties);
  };

  static void _end_element(void *_parser, const xmlChar *n) {
    XMLParser *parser;

    parser = (XMLParser *) _parser;
    parser->end_element(std::string((const char *) n));
  };

  static void _characters(void *_parser, const xmlChar *s, int len) {
    XMLParser *parser;

    parser = (XMLParser *) _parser;
    parser->characters(std::string((const char *) s, len));
  };

  static void _comment(void *_parser, const xmlChar *s) {
    XMLParser *parser;

    parser = (XMLParser *) _parser;
    parser->comment(std::string((const char *) s));
  };

  static void _warning(void *_parser, const char *fmt, ...) {
    XMLParser *parser;
    va_list arg;
    char buff[1024];

    va_start(arg, fmt);
    vsprintf(buff, fmt, arg);
    va_end(arg);

    parser = (XMLParser *) _parser;
    parser->warning(std::string(buff));
  };

  static void _error(void *_parser, const char *fmt, ...) {
    XMLParser *parser;
    va_list arg;
    char buff[1024];

    va_start(arg, fmt);
    vsprintf(buff, fmt, arg);
    va_end(arg);

    parser = (XMLParser *) _parser;
    parser->error(std::string(buff));
  };

  static void _fatal_error(void *_parser, const char *fmt, ...) {
    XMLParser *parser;
    va_list arg;
    char buff[1024];

    va_start(arg, fmt);
    vsprintf(buff, fmt, arg);
    va_end(arg);

    parser = (XMLParser *) _parser;
    parser->fatal_error(std::string(buff));
  };

  void start_document(void) {
    if(_parser_callback)
      _parser_callback->start_document();
  };

  void end_document(void) {
    if(_parser_callback)
      _parser_callback->end_document();
  };

  void start_element(const std::string &n, const XMLPropertyMap &p) {
    if(_parser_callback)
      _parser_callback->start_element(n, p);
  };

  void end_element(const std::string &n) {
    if(_parser_callback)
      _parser_callback->end_element(n);
  };

  void characters(const std::string &s) {
    if(_parser_callback)
      _parser_callback->characters(s);
  };

  void comment(const std::string &s) {
    if(_parser_callback)
      _parser_callback->comment(s);
  };

  void warning(const std::string &s) {
    if(_parser_callback)
      _parser_callback->warning(s);
  };

  void error(const std::string &s) {
    if(_parser_callback)
      _parser_callback->error(s);
  };

  void fatal_error(const std::string &s) {
    if(_parser_callback)
      _parser_callback->fatal_error(s);
  };
public:
  XMLParser() {
    xmlSAXHandler sax_handler = {
      NULL,		// internalSubset
      NULL,		// isStandalone
      NULL,		// hasInternalSubset
      NULL,		// hasExternalSubset
      NULL,		// resolveEntity
      _get_entity,	// getEntity
      NULL,		// entityDecl
      NULL,		// notationDecl
      NULL,		// attributeDecl
      NULL,		// elementDecl
      NULL,		// unparsedEntityDecl
      NULL,		// setDocumentLocator
      _start_document,	// startDocument
      _end_document,	// endDocument
      _start_element,	// startElement
      _end_element,	// endElement
      NULL,		// reference
      _characters,	// characters
      NULL,		// ignorableWhitespace
      NULL,		// processingInstruction
      _comment,		// comment
      _warning,		// warning
      _error,		// error
      _fatal_error,	// fatalError
      NULL,		// getParameterEntity
      NULL,		// cdataBlock
      NULL		// externalSubset
    };

    _parser_callback = new ParserCallback;
    _context = xmlCreatePushParserCtxt(&sax_handler, this, NULL, 0, NULL);
  };

  ~XMLParser() {
    if(_parser_callback)
      delete _parser_callback;

    if(_context)
      xmlFreeParserCtxt(_context);
  };

  void parse_chunk(const std::string &s) {
    xmlParseChunk(_context, s.c_str(), s.length(), 0);
  };

  void finish(void) {
    xmlParseChunk(_context, NULL, 0, 1);
  };
};

class XMLTree {
private:
  bool _initialized;
  std::string _filename;
  XMLNode *_root;
  std::string _encoding;
  int _compression;
public:
  XMLTree() : _filename(), _root(NULL), _compression(0) { };
  XMLTree(const std::string &fn)
	: _filename(fn), _root(NULL), _compression(0) { read(); };
  XMLTree(const XMLTree *);
  ~XMLTree();
  bool initialized(void) const { return _initialized; };
  XMLNode *root(void) const { return _root; };
  XMLNode *set_root(XMLNode *n) { return _root = n; };
  const std::string & filename(void) const { return _filename; };
  const std::string & set_filename(const std::string &fn) { return _filename = fn; };
  const std::string & encoding(void) const { return _encoding; };
  const std::string & set_encoding(const std::string &);
  int compression(void) const { return _compression; };
  int set_compression(int);
  bool read(void);
  bool read(const std::string &fn) { set_filename(fn); return read(); };
  bool read_buffer(const std::string &);
  bool write(void) const;
  bool write(const std::string &fn) { set_filename(fn); return write(); };
  const std::string & write_buffer(void) const;
};

class XMLNode {
private:
  bool _initialized;
  std::string _name;
  bool _is_content;
  std::string _content;
  int _line;
  XMLNodeList _children;
  XMLPropertyList _proplist;
  XMLPropertyMap _prophash;
public:
  XMLNode(const std::string &);
  XMLNode(const std::string &, int line);
  XMLNode(const std::string &, const std::string &);
  XMLNode(const XMLNode *);
  ~XMLNode();
  bool initialized(void) const { return _initialized; };
  const std::string name(void) const { return _name; };
  int line() { return _line; };
  bool is_content(void) const { return _is_content; };
  const std::string & content(void) const { return _content; };
  const std::string & set_content(const std::string &);
  const XMLNodeList & children(const std::string & = std::string()) const;
  XMLNode *add_child(const std::string &);
  XMLNode *add_child(XMLNode *);
  XMLNode *add_content(const std::string & = std::string());
  void remove_child(XMLNode *);
  const XMLPropertyList & properties(void) const { return _proplist; };
  XMLProperty *property(const std::string &);
  const XMLProperty *property(const std::string &n) const
	{ return ((XMLNode *) this)->property(n); };
  XMLProperty *add_property(const std::string &, const std::string & = std::string());
  void remove_property(const std::string &);
};

class XMLProperty {
private:
  std::string _name;
  std::string _value;
public:
  XMLProperty(const std::string &n, const std::string &v = std::string())
	: _name(n), _value(v) { };
  const std::string & name(void) const { return _name; };
  const std::string & value(void) const { return _value; };
  const std::string & set_value(const std::string &v) { return _value = v; };
};

#endif /* __XML_H */

