// -*- C++ -*-

/* main.cc
 *
 * Copyright (C) 2002 The libxml++ development team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libxml++/libxml++.h>

#include <iostream>

void print_indentation(unsigned int indentation)
{
  for(unsigned int i = 0; i < indentation; ++i)
    std::cout << " ";
}

void print_node(const xmlpp::Node& node, unsigned int indentation = 0)
{
  print_indentation(indentation);
  std::cout << "Node name = " << node.name() << std::endl;
  print_indentation(indentation);
  std::cout << "     line = " << node.line() << std::endl;

  if(node.has_content())
  {
    print_indentation(indentation);
    std::cout << "content = " << node.content() << std::endl;
  }

  //Print attributes:
  xmlpp::Node::AttributeList attributes = node.attributes();
  for(xmlpp::Node::AttributeList::iterator iter = attributes.begin(); iter != attributes.end(); ++iter)
  {
    const xmlpp::Attribute* attribute = *iter;
    print_indentation(indentation);
    std::cout << "  Attribute " << attribute->name() << " = " << attribute->value() << std::endl;
  }

  //Recurse through child nodes:
  xmlpp::Node::NodeList list = node.children();
  for(xmlpp::Node::NodeList::iterator iter = list.begin(); iter != list.end(); ++iter)
  {
    print_node(*(*iter), indentation + 2); //recursive
  }
}

int main(int argc, char* argv[])
{
  if(argc > 1 )
  {
    std::string filepath = argv[1];
    
    try
    {
      xmlpp::DomParser parser(filepath);
      if(parser)
      {
        //Walk the tree:
        const xmlpp::Node* pNode = parser.get_root_node(); //deleted by DomParser.
        print_node(*pNode);
      }
    }
    catch(const std::exception& ex)
    {
      std::cout << "Exception caught: " << ex.what() << std::endl;
    }
  }

  return 0;
}

