/* xml++.cc
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 */

#include <libxml++/node.h>

namespace xmlpp {

Node::Node(const std::string& name)
 : _name(name), _line(0) {
  if(_name.empty())
    _initialized = false;
  else
    _initialized = true;
}

Node::Node(const std::string& name, int line)
 : _name(name), _line(line) {
  if(_name.empty())
    _initialized = false;
  else
    _initialized = true;
}

Node::Node(const std::string& name, const std::string& content)
 : _name(std::string()), _content(content), _line(0) {
  _initialized = true;
}

Node::Node(const Node* from)
: _initialized(false), _name(from->name()), _line(from->line())
{
  set_content(from->_content);

  for(AttributeList::const_iterator curattr = from->attributes().begin();
	  curattr != from->attributes().end();
	  ++curattr)
  {
    add_attribute((*curattr)->name(), (*curattr)->value());
  }

  for(NodeList::const_iterator curnode = from->children().begin();
	  curnode != from->children().end();
	  ++curnode)
  {
    add_child(new Node(*curnode));
  }

  _initialized = true;
}

Node::Node(xmlNodePtr node)
{
  if(node->name)
    _name = (char*) node->name;

  _line = XML_GET_LINE(node);

  if(XML_GET_CONTENT(node))
	  set_content((char*) XML_GET_CONTENT(node));

  for(xmlAttrPtr attr = node->properties;
		  attr;
		  attr = attr->next)
  {
	  add_attribute(attr);
  }

  for(xmlNodePtr child = node->children;
		  child;
		  child = child->next)
  {
	  add_child(new Node(child));
  }
}

Node::~Node() {
  NodeList::iterator curchild;

  for(NodeList::iterator i = _children.begin();
		  i != _children.end();
		  ++i)
  {
    delete *i;
  }

  for(AttributeList::iterator i = _attributes_list.begin();
		  i != _attributes_list.end();
		  ++i)
  {
    delete *i;
  }
}

void Node::set_content(const std::string& content)
{
  _content = content;
}

Node::NodeList Node::children(const std::string& name)
{
  if(name.empty())
    return _children;
  else
  {
    //Return only the nodes with the specified name:
    NodeList retval;
    for(NodeList::const_iterator cur = _children.begin(); cur != _children.end(); cur++)
      if((*cur)->name() == name)
        retval.insert(retval.end(), *cur);

     return retval;
  }
}

const Node::NodeList Node::children(const std::string& name) const
{
  return const_cast<Node*>(this)->children(name);
}

Node* Node::add_child(const std::string& name)
{
  Node* tmp = new Node(name);

  return add_child(tmp);
}

Node* Node::add_child(Node* node) {
  if(!node)
    return NULL;

  _children.insert(_children.end(), node);
  return node;
}

Node* Node::add_content(const std::string& content) {
  Node* tmp;

  tmp = new Node(std::string(), content);

  return add_child(tmp);
}

void Node::remove_child(Node* node)
{
  if(node)
  {
    _children.remove(node);
    delete node;
  }
}

Attribute* Node::attribute(const std::string& name) {
  AttributeMap::iterator i = _attributes_map.find(name);
  if(i == _attributes_map.end())
  {
    return NULL;
  }
  else
  {
    return i->second;
  }
}

Attribute* Node::add_attribute(const std::string& name, const std::string& value) {
  Attribute* tmp = 0;

  if(_attributes_map.find(name) != _attributes_map.end())
    return NULL;

  tmp = new Attribute(name, value);

  if(!tmp)
    return NULL;

  _attributes_map[tmp->name()] = tmp;
  _attributes_list.insert(_attributes_list.end(), tmp);

  return tmp;
}

Attribute* Node::add_attribute(xmlAttrPtr attr)
{
	if(_attributes_map.find( (const char*)attr->name ) != _attributes_map.end())
	{
		return 0;
	}

	Attribute* tmp = 0;

	if(tmp = new Attribute(attr))
	{
		_attributes_map[tmp->name()] = tmp;
		_attributes_list.insert(_attributes_list.end(), tmp);
	}

	return tmp;
}

void Node::remove_attribute(const std::string& name) {
  AttributeMap::iterator i = _attributes_map.find(name);
  if(i != _attributes_map.end())
  {
    Attribute* attribute = i->second;
    _attributes_list.remove(attribute);
    _attributes_map.erase(i);

    delete attribute;
  }
}

bool Node::initialized() const
{
  return _initialized;
}

const std::string& Node::name() const
{
  return _name;
}

int Node::line() const
{
  return _line;
}

bool Node::has_content() const
{
  return !(content().empty());
}

const std::string& Node::content() const
{
  if(!_content.empty())
    return _content;
  else
  {
    const NodeList& listNodes = children("text");
    if(listNodes.empty())
      return _content; //Which will be empty. We could return "" but this would break because we return a const std::string&. I wish we didn't. murrayc.
    else
    {
      Node* nodeContent = *(listNodes.begin());
      return nodeContent->content();
    }
  }
}


const Node::AttributeList& Node::attributes() const
{
  return _attributes_list;
}

const Attribute* Node::attribute(const std::string& name) const
{
  return const_cast<Node*>(this)->attribute(name);
}

void Node::write(xmlDocPtr doc, xmlNodePtr parent) const
{
  xmlNodePtr node = 0;

  if(!parent)
    node = doc->children = xmlNewDocNode(doc, NULL /* ns */, (xmlChar*) name().c_str(), 0);
  else
    node = xmlNewChild(parent, NULL /* ns */, (xmlChar*) name().c_str(), 0);

  if( ! _content.empty() )
  {
    node->type = XML_TEXT_NODE;
    xmlNodeSetContent(node, (const xmlChar*)_content.c_str());
  }

  //Add the attributes:
  for(AttributeList::const_iterator iter = _attributes_list.begin();
    iter != _attributes_list.end();
    ++iter)
  {
    const Attribute* attr = *iter;
    xmlSetProp(node, (xmlChar*)attr->name().c_str(), (xmlChar*)attr->value().c_str());
  }

  //Add the child nodes:
  for(NodeList::const_iterator iter = _children.begin();
     iter != _children.end();
     ++iter)
  {
    //Add the nodes to this parent node:
    (*iter)->write(doc, node);
  }

}

} // namespace xmlpp


