/* node.h
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 */

#ifndef __LIBXMLPP_NODE_H
#define __LIBXMLPP_NODE_H

#include <libxml++/attribute.h>
#include <list>
#include <map>

#include <libxml/tree.h>

namespace xmlpp {

/** Represents XML Nodes.
 * You should never new or delete Nodes. The Parser will create and manage them for you.
 */
class Node
{
public:
  typedef std::list<Node*> NodeList;
  typedef std::list<Attribute*> AttributeList;
  typedef std::map<std::string, Attribute*> AttributeMap;

  explicit Node(const std::string& name);
  Node(const std::string& name, int line);
  Node(const std::string& name, const std::string& content);
  explicit Node(const Node* from);
  explicit Node(xmlNodePtr node);
  ~Node();

  bool initialized() const;
  const std::string& name() const;
  int line() const;

  bool has_content() const;
  const std::string& content() const;
  void set_content(const std::string& content);

  NodeList children(const std::string& name = std::string());
  const NodeList children(const std::string& name = std::string()) const;

  Node* add_child(const std::string& name);
  Node* add_child(Node* node);
  Node* add_content(const std::string& name = std::string());

  /** Remove the child node.
   * @param node The child node to remove. This Node will be deleted and therefore unusable after calling this method.
   */
  void remove_child(Node* node);

  const AttributeList& attributes() const;
  Attribute* attribute(const std::string& name);
  const Attribute* attribute(const std::string& name) const;
  Attribute* add_attribute(const std::string& name, const std::string& value = std::string());

  /** Remove the child node.
   * @param node The child node to remove. The Attribute will be deleted by this method, so any references to it will be unusable.
   */
  void remove_attribute(const std::string& name);

  void write(xmlDocPtr doc, xmlNodePtr parent = 0) const;

private:
  Attribute* add_attribute(xmlAttrPtr attr);

  bool _initialized;
  std::string _name;
  std::string _content;
  int _line;
  Node::NodeList _children;
  AttributeList _attributes_list;
  AttributeMap _attributes_map;
};

} // namespace xmlpp

#endif //__LIBXMLPP_NODE_H



