/* xml++.cc
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 */

#include <libxml++/nodes/element.h>
#include <libxml++/nodes/textnode.h>
#include <libxml++/exceptions/internal_error.h>
#include <iostream>

namespace xmlpp {

Element::Element(const std::string& name)
  : Node(name)
{}

Element::Element(const Element* from)
  : Node(from)
{
  const AttributeList& attributes = from->get_attributes();
  for(AttributeList::const_iterator curattr = attributes.begin(); curattr != attributes.end(); ++curattr)
  {
    add_attribute((*curattr)->get_name(), (*curattr)->get_value());
  }
}

Element::Element(xmlNodePtr node)
  : Node(node)
{
  for(xmlAttrPtr attr = node->properties;
		  attr;
		  attr = attr->next)
  {
	  add_attribute(attr);
  }
}

Element::~Element()
{
  for(AttributeList::iterator i = _attributes_list.begin();
		  i != _attributes_list.end();
		  ++i)
  {
    delete *i;
  }
}

void Element::write(xmlDocPtr doc, xmlNodePtr parent) const
{ 
  xmlNodePtr node = 0;

  if(!parent)
    node = doc->children = xmlNewDocNode(doc, NULL /* ns */, (xmlChar*) get_name().c_str(), 0);
  else
  {
    //std::cout << "Element::write: xmlNewChild(): " << _name << std::endl;

    node = xmlNewChild(parent, NULL /* ns */, (xmlChar*) get_name().c_str(), 0);
    if(!node)
      throw internal_error("xmlNewChild() returned NULL");
  }

  if(node)
  {
    //Add the child nodes:
    for(NodeList::const_iterator iter = _children.begin();
       iter != _children.end();
       ++iter)
    {
      //Add the nodes to this parent node:

      //write() is a virtual function, implemented differently for each node type.
      (*iter)->write(doc, node);
    }

    //Add the attributes:
    for(AttributeList::const_iterator iter = _attributes_list.begin();
      iter != _attributes_list.end();
      ++iter)
    {
      const Attribute* attr = *iter;
      xmlSetProp(node, (xmlChar*)attr->get_name().c_str(), (xmlChar*)attr->get_value().c_str());
    }
  }

}

Attribute* Element::get_attribute(const std::string& name) {
  AttributeMap::iterator i = _attributes_map.find(name);
  if(i == _attributes_map.end())
  {
    return NULL;
  }
  else
  {
    return i->second;
  }
}

Attribute* Element::add_attribute(const std::string& name, const std::string& value) {
  Attribute* tmp = 0;

  if(_attributes_map.find(name) != _attributes_map.end())
    return NULL;

  tmp = new Attribute(name, value);

  if(!tmp)
    return NULL;

  _attributes_map[tmp->get_name()] = tmp;
  _attributes_list.insert(_attributes_list.end(), tmp);

  return tmp;
}

Attribute* Element::add_attribute(xmlAttrPtr attr)
{
	if(_attributes_map.find( (const char*)attr->name ) != _attributes_map.end())
	{
		return 0;
	}

	Attribute* tmp = 0;

	if(tmp = new Attribute(attr))
	{
		_attributes_map[tmp->get_name()] = tmp;
		_attributes_list.insert(_attributes_list.end(), tmp);
	}

	return tmp;
}

void Element::remove_attribute(const std::string& name) {
  AttributeMap::iterator i = _attributes_map.find(name);
  if(i != _attributes_map.end())
  {
    Attribute* attribute = i->second;
    _attributes_list.remove(attribute);
    _attributes_map.erase(i);

    delete attribute;
  }
}

const Element::AttributeList& Element::get_attributes() const
{
  return _attributes_list;
}

const Attribute* Element::get_attribute(const std::string& name) const
{
  return const_cast<Element*>(this)->get_attribute(name);
}



} // namespace xmlpp


