/* xml++.cc
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 */

#include <libxml++/nodes/node.h>
#include <libxml++/nodes/element.h>
#include <libxml++/nodes/commentnode.h>
#include <libxml++/nodes/textnode.h>
#include <libxml++/exceptions/internal_error.h>
#include <iostream>

namespace xmlpp {

Node::Node(const std::string& name)
 : _name(name), _line(0)
{
}


Node::Node(const Node* from)
: _name(from->get_name()), _line(from->get_line())
{
  const NodeList& children = from->get_children();
  for(NodeList::const_iterator curnode = children.begin(); curnode != children.end(); ++curnode)
  {
    //TODO: Create the appropriate type:
    add_child(new Node(*curnode));
  }
}

Node::Node(xmlNodePtr node)
{
  if(node->name)
    _name = (char*) node->name;

  _line = XML_GET_LINE(node);

  //Add child nodes:
  for(xmlNodePtr child = node->children;
		  child;
		  child = child->next)
  {
    Node* pChild = 0;

    //TODO: Deal with more types:
    if(child->type == XML_TEXT_NODE)
      pChild = new TextNode(child);
    else if(child->type == XML_COMMENT_NODE)
      pChild = new CommentNode(child);
    else
      pChild = new Element(child);
    
	  add_child(pChild);
  }
}

Node::~Node() {
  NodeList::iterator curchild;

  for(NodeList::iterator i = _children.begin();
		  i != _children.end();
		  ++i)
  {
    delete *i;
  }
}

Node::NodeList Node::get_children(const std::string& name)
{
  if(name.empty())
    return _children;
  else
  {
    //Return only the nodes with the specified name:
    NodeList retval;
    for(NodeList::const_iterator cur = _children.begin(); cur != _children.end(); cur++)
      if((*cur)->get_name() == name)
        retval.insert(retval.end(), *cur);

     return retval;
  }
}

const Node::NodeList Node::get_children(const std::string& name) const
{
  return const_cast<Node*>(this)->get_children(name);
}

Element* Node::add_child(const std::string& name)
{
  Element* tmp = new Element(name);

  return static_cast<Element*>(add_child(tmp));
}

Node* Node::add_child(Node* node) {
  if(!node)
    return NULL;

  _children.insert(_children.end(), node);
  return node;
}

TextNode* Node::add_content(const std::string& content)
{
  TextNode* tmp = new TextNode(content);

  return static_cast<TextNode*>(add_child(tmp));
}

void Node::remove_child(Node* node)
{
  if(node)
  {
    _children.remove(node);
    delete node;
  }
}

std::string Node::get_name() const
{
  return _name;
}

int Node::get_line() const
{
  return _line;
}

bool Node::has_content() const
{
  return get_child_content() != 0;
}

const TextNode* Node::get_child_content() const
{
  //TODO: Get by type:
  const NodeList& listNodes = get_children("text");
  if(listNodes.empty())
    return 0;
  else
  {
    const Node* nodeContent = *(listNodes.begin());
    return static_cast<const TextNode*>(nodeContent);
  }
}

TextNode* Node::get_child_content()
{
  //TODO: Get by type:
  NodeList listNodes = get_children("text");
  if(listNodes.empty())
    return 0;
  else
  {
    Node* nodeContent = *(listNodes.begin());
    return static_cast<TextNode*>(nodeContent);
  }
}

void Node::write(xmlDocPtr doc, xmlNodePtr parent) const
{ 
  xmlNodePtr node = 0;

  if(!parent)
    node = doc->children = xmlNewDocNode(doc, NULL /* ns */, (xmlChar*) get_name().c_str(), 0);
  else
  {
    //std::cout << "Node::write: xmlNewChild(): " << _name << std::endl;

    node = xmlNewChild(parent, NULL /* ns */, (xmlChar*) get_name().c_str(), 0);
    if(!node)
      throw internal_error("xmlNewChild() returned NULL");
  }

  if(node)
  {
    //Add the child nodes:
    for(NodeList::const_iterator iter = _children.begin();
       iter != _children.end();
       ++iter)
    {
      //Add the nodes to this parent node:

      //write() is a virtual function, implemented differently for each node type.
      (*iter)->write(doc, node);
    }
  }

}

void Node::set_child_content(const std::string& content)
{
  TextNode* node = get_child_content();
  if(!node)
    node = add_content(content);
  else
    node->set_content(content);
}

} // namespace xmlpp


