/* xml++.cc
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 */

#include "libxml++/parsers/domparser.h"

namespace xmlpp {

DomParser::DomParser()
: _doc(0), _pRootNode(0), _pDtd(0)
{
  //Start with an empty document:
  _doc = xmlNewDoc((xmlChar *) "1.0");
}

DomParser::DomParser(const std::string& filename)
: _doc(0), _pRootNode(0), _pDtd(0)
{
  parse_file(filename);
}

DomParser::~DomParser()
{
  release_underlying();
}

void DomParser::parse_file(const std::string& filename) throw(exception)
{
  release_underlying(); //Free any existing document.

  xmlLineNumbersDefault(1 /*TRUE*/); //Allow discovery of line numbers from Node. The default, of 0, is for libxml1 compatability.
  xmlKeepBlanksDefault(0 /*FALSE*/); //Don't use libxml1 compatability. This is the default anyway.

  _doc = xmlParseFile(filename.c_str());

  if(!_doc)
    throw parse_error("Document not well-formed.");
}

void DomParser::parse_memory(const std::string& contents) throw(exception)
{
  release_underlying(); //Free any existing document.

  xmlLineNumbersDefault(1);
  xmlKeepBlanksDefault(0);

  _doc = xmlParseMemory(contents.c_str(), contents.size());

  if(!_doc)
    throw parse_error("Document not well-formed.");
}

void DomParser::parse_stream(std::istream& in) throw(exception)
{
  release_underlying(); //Free any existing document.

  std::string line;

  xmlLineNumbersDefault(1);
  xmlKeepBlanksDefault(0);

  xmlParserCtxtPtr context = xmlCreatePushParserCtxt(
      NULL, // setting thoses two parameters to NULL force the parser
      NULL, // to create a document while parsing.
      NULL,
      0,
      ""); // here should come the filename. I don't know if it is a problem to let it empty

  while(getline(in, line))
  {
    // since getline does not get the line separator, we have to add it since the parser care
    // about layout in certain cases.
    line += '\n';

    xmlParseChunk(context, line.c_str(), line.length(), 0);
  }

  xmlParseChunk(context, NULL, 0, 1);

  _doc = context->myDoc;

  xmlFreeParserCtxt(context);

}

void DomParser::release_underlying()
{
  if(_pRootNode)
  {
    delete _pRootNode;
    _pRootNode = 0;
  }

  if(_pDtd)
  {
    delete _pDtd;
    _pDtd = 0;
  }

  if(_doc)
  {
    xmlFreeDoc(_doc);
    _doc = 0;
  }
}

Element* DomParser::get_root_node()
{
  if(!_pRootNode)
  {
    xmlNodePtr cnode = xmlDocGetRootElement(_doc);
    if(cnode)
      _pRootNode =  new Element(cnode);
  }

  return _pRootNode;
}

const Element* DomParser::get_root_node() const
{
  return const_cast<DomParser*>(this)->get_root_node();
}

Element* DomParser::set_root_node(const std::string& name, const std::string& ns_href, const std::string& ns_href_prefix)
{
  //Remove the existing root node:
  if(_pRootNode);
  {
    delete _pRootNode;
    _pRootNode = 0;
  }

  //Add the new root node:

  //Namespace:
  xmlNsPtr cNS = 0;
  if(!ns_href_prefix.empty())
    cNS = xmlNewNs((xmlNodePtr)_doc, (const xmlChar*)(ns_href.empty() ? 0 : ns_href.c_str()), (const xmlChar*)ns_href_prefix.c_str());

  xmlNodePtr cNode = xmlNewDocNode(_doc, cNS, (const xmlChar*)name.c_str(), 0 /* content */);
  xmlDocSetRootElement(_doc, cNode);
  //TODO: This probably invalidates the existing tree.

  return get_root_node();
}

DomParser::operator bool() const
{
  return _doc;
}

std::string DomParser::get_encoding() const
{
  std::string retval;

  if(_doc->encoding)
    retval = (const char*)_doc->encoding;

  return retval;
}

xmlDocPtr DomParser::dump_to_c_doc()
{
  //The C++ lists of child nodes and attributes might be out-of-sync with the original C tree.
  //But it's the results of the use of the C++ API that we care about,
  //so we rebuild the C tree, based on the C++ tree.
  //Then we can tell the xmlDoc to write itself.

  xmlDocPtr doc = xmlNewDoc((xmlChar *) "1.0");
  //xmlSetDocCompressMode(doc, _compression);

  Node* nodeRoot = get_root_node();
  if(!nodeRoot)
    throw exception("Attempt to write file without a root node.");

  nodeRoot->write(doc);

  //The Internal Subset:
  //Although it would make sense to do this before writing all the child nodes,
  //for some reason this is removed during the writing of the child nodes.
  if(_pDtd)
    xmlCreateIntSubset(doc, (xmlChar*)_pDtd->get_name().c_str(), (xmlChar*)_pDtd->get_external_id().c_str(), (xmlChar*)_pDtd->get_system_id().c_str());

  return doc;
}


void DomParser::write_to_file(const std::string& filename, const std::string& encoding) throw(exception)
{
  xmlDocPtr doc = dump_to_c_doc();

  int result = 0;
  if(!encoding.empty())
    result = xmlSaveFormatFileEnc(filename.c_str(), doc, encoding.c_str(), 1);
  else
    result = xmlSaveFormatFile(filename.c_str(), doc, 1);

  xmlFreeDoc(doc);

  if(result == -1)
    throw exception("write_to_file() failed.");
}

std::string DomParser::write_to_string(const std::string& encoding) throw(exception)
{
  xmlDocPtr doc = dump_to_c_doc();

  xmlChar* pchText = 0;
  int text_length = 0;
  if(!encoding.empty())
    xmlDocDumpMemoryEnc(doc, &pchText, &text_length, encoding.c_str());
  else
    xmlDocDumpMemory(doc, &pchText, &text_length);

  xmlFreeDoc(doc);

  if(!pchText)
    throw exception("write_to_string() failed.");

  return std::string((const char*)pchText, text_length);
}

void DomParser::set_internal_subset(const std::string& name, const std::string& external_id, const std::string& system_id)
{
  if(!_pDtd)
    delete _pDtd;

  _pDtd = new Dtd(name, external_id, system_id);
}

Dtd* DomParser::get_internal_subset()
{
  if(!_pDtd)
  {
    xmlDtdPtr cDtd = xmlGetIntSubset(_doc);
    _pDtd = new Dtd( (cDtd->name ? (const char*)cDtd->name : 0), (cDtd->ExternalID ? (const char*)cDtd->ExternalID : 0), (cDtd->SystemID ? (const char*)cDtd->SystemID : 0) );
  }

  return _pDtd;
}

const Dtd* DomParser::get_internal_subset() const
{
  return const_cast<DomParser*>(this)->get_internal_subset();
}



} // namespace xmlpp


