/* xml++.cc
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 */

#include "libxml++/parsers/domparser.h"
#include "libxml++/dtd.h"
#include "libxml++/nodes/element.h"
#include "libxml++/nodes/textnode.h"
#include "libxml++/nodes/commentnode.h"
#include <libxml/parserInternals.h>//For xmlCreateFileParserCtxt().

namespace xmlpp
{

DomParser::DomParser()
: _doc(0)
{
  //Start with an empty document:
  _doc = xmlNewDoc((xmlChar*) "1.0");
}

DomParser::DomParser(const std::string& filename, bool validate)
: _doc(0)
{
  set_validate(validate);
  parse_file(filename);
}

DomParser::~DomParser()
{ 
  release_underlying();
}

void DomParser::parse_file(const std::string& filename) throw(exception)
{
  release_underlying(); //Free any existing document.

  //These default values seem to affect the execution of xmlCreateFileParserCtxt(),
  //leading to extra blank text nodes from xmlParseDocument().
  //We set the same stuff per-context in initialize_context(), before calling xmlParseDocument().
  //but that seems to be too late.
  bool old_keepblanks = xmlKeepBlanksDefault(CONSTANT_KeepBlanksSetting); 

  //The following is based on the implementation of xmlParseFile(), in xmlSAXParseFileWithData():
  _context = xmlCreateFileParserCtxt(filename.c_str());

  //Restore the old value and trust initialize_context() to deal with this from now on:
  xmlKeepBlanksDefault(old_keepblanks); 


  if(_context->directory == NULL)
  {
    char* directory = xmlParserGetDirectory(filename.c_str());
    _context->directory = (char*) xmlStrdup((xmlChar*) directory);
  }

  parse_context();
}

void DomParser::parse_memory(const std::string& contents) throw(exception)
{
  release_underlying(); //Free any existing document.

  //See parse_file() about this - I haven't tested that it's necessary with xmlCreateMemoryParserCtxt() too,
  //but it probably is:
  bool old_keepblanks = xmlKeepBlanksDefault(CONSTANT_KeepBlanksSetting);

  //The following is based on the implementation of xmlParseFile(), in xmlSAXParseFileWithData():
  _context = xmlCreateMemoryParserCtxt(contents.c_str(), contents.size());

  xmlKeepBlanksDefault(old_keepblanks);

  parse_context();
}

void DomParser::parse_context() throw(exception)
{
  //The following is based on the implementation of xmlParseFile(), in xmlSAXParseFileWithData():
  //and the implementation of xmlParseMemory(), in xmlSaxParseMemoryWithData().
  initialize_context();

  xmlParseDocument(_context);
  _doc = _context->myDoc;

  check_for_exception();

  if(!_context->wellFormed)
  {
    release_underlying(); //Free _doc;
    throw parse_error("Document not well-formed.");
  }

  if(!_context->valid)
  {
    //This would probably not be reached, because the callback would create an exception,
    //which would have been thrown already by check_for_exception() above.
    
    release_underlying(); //Free _doc;
    throw validity_error("Document not valid.");
  }

  //Free the parse context, but keep the document alive so people can navigate the DOM tree:
  //TODO: Why not keep the context alive too?
  Parser::release_underlying();

  check_for_exception();
}


void DomParser::parse_stream(std::istream& in) throw(exception)
{
  release_underlying(); //Free any existing document.

  _context = xmlCreatePushParserCtxt(
      NULL, // setting thoses two parameters to NULL force the parser
      NULL, // to create a document while parsing.
      NULL,
      0,
      ""); // here should come the filename. I don't know if it is a problem to let it empty
  initialize_context();

  std::string line;
  while(getline(in, line))
  {
    // since getline does not get the line separator, we have to add it since the parser cares
    // about layout in certain cases.
    line += '\n';

    xmlParseChunk(_context, line.c_str(), line.length(), 0);
  }

  xmlParseChunk(_context, NULL, 0, 1);

  _doc = _context->myDoc;

  //Free the parse context, but keep the document alive so people can navigate the DOM tree:
  //TODO: Why not keep the context alive too?
  Parser::release_underlying();

  check_for_exception();
}

void DomParser::release_underlying()
{
  if(_doc)
  {
    xmlFreeDoc(_doc);
    _doc = 0;
  }

  Parser::release_underlying();
}

Element* DomParser::get_root_node()
{
  xmlNode* root = xmlDocGetRootElement(_doc);
  return root ? static_cast<Element*>(root->_private) : 0;
}

const Element* DomParser::get_root_node() const
{
  return const_cast<DomParser*>(this)->get_root_node();
}

Element* DomParser::set_root_node(const std::string& name, const std::string& ns_href, const std::string& ns_href_prefix)
{
  xmlNode* root = xmlNewDocNode(_doc, 0, (xmlChar*)name.c_str(), 0);

  if(!ns_href_prefix.empty())
    xmlNewNs(root,
             (xmlChar*)(ns_href.empty() ? 0 : ns_href.c_str()),
             (xmlChar*)ns_href_prefix.c_str());

  xmlDocSetRootElement(_doc, root);
  return get_root_node();
}

DomParser::operator bool() const
{
  return _doc;
}

std::string DomParser::get_encoding() const
{
  std::string retval;

  if(_doc->encoding)
    retval = (const char*)_doc->encoding;

  return retval;
}

void DomParser::write_to_file(const std::string& filename, const std::string& encoding) throw(exception)
{
  int result = 0;
  if(!encoding.empty())
    result = xmlSaveFormatFileEnc(filename.c_str(), _doc, encoding.c_str(), 1);
  else
    result = xmlSaveFormatFile(filename.c_str(), _doc, 1);

  if(result == -1)
    throw exception("write_to_file() failed.");
}

std::string DomParser::write_to_string(const std::string& encoding) throw(exception)
{
  xmlChar* pchText = 0;
  int text_length = 0;
  
  if(!encoding.empty())
    xmlDocDumpMemoryEnc(_doc, &pchText, &text_length, encoding.c_str());
  else
    xmlDocDumpMemory(_doc, &pchText, &text_length);

  if(!pchText)
    throw exception("write_to_string() failed.");

  return std::string((const char*)pchText, text_length);
}

void DomParser::set_internal_subset(const std::string& name, const std::string& external_id, const std::string& system_id)
{
  xmlDtd* dtd = xmlGetIntSubset(_doc);
  if (!dtd)
    xmlCreateIntSubset(_doc,
                       (xmlChar*)name.c_str(),
                       (xmlChar*)external_id.c_str(),
                       (xmlChar*)system_id.c_str());
  else
    throw internal_error("dtd already defined for this document");
}

Dtd* DomParser::get_internal_subset()
{
  xmlDtd* dtd = xmlGetIntSubset(_doc);
  return dtd ? static_cast<Dtd*>(dtd->_private) : 0;
}

const Dtd* DomParser::get_internal_subset() const
{
  return const_cast<DomParser*>(this)->get_internal_subset();
}




} // namespace xmlpp


