/* document.cc
 * this file is part of libxml++
 *
 * copyright (C) 2003 by Stefan Seefeld
 *
 * this file is covered by the GNU Lesser General Public License,
 * which should be included with libxml++ as the file COPYING.
 */

#include <libxml++/document.h>
#include "libxml++/dtd.h"
#include "libxml++/attribute.h"
#include "libxml++/nodes/element.h"
#include "libxml++/nodes/textnode.h"
#include "libxml++/nodes/commentnode.h"
#include <istream>

namespace xmlpp
{
  
Document* parse_file(const std::string& filename) throw(exception)
{
  // Allow discovery of line numbers from Node.
  xmlLineNumbersDefault(1);
  // Don't use libxml1 compatability. This is the default anyway.
  xmlKeepBlanksDefault(0);

  xmlDoc* doc = xmlParseFile(filename.c_str());

  if(!doc) throw parse_error("Document not well-formed.");
  else return new Document(doc);
}

Document* parse_memory(const std::string& contents) throw(exception)
{
  xmlLineNumbersDefault(1);
  xmlKeepBlanksDefault(0);

  xmlDoc* doc = xmlParseMemory(contents.c_str(), contents.size());

  if(!doc)
    throw parse_error("Document not well-formed.");
  else
    return new Document(doc);
}

Document* parse_stream(std::istream& is) throw(exception)
{
  xmlLineNumbersDefault(1);
  xmlKeepBlanksDefault(0);

  xmlParserCtxtPtr context = xmlCreatePushParserCtxt(0, 0, 0, 0, "");

  std::string line;
  while(std::getline(is, line))
  {
    line += '\n';
    xmlParseChunk(context, line.c_str(), line.length(), 0);
  }
  
  xmlParseChunk(context, 0, 0, 1);
  xmlDoc* doc = context->myDoc;
  xmlFreeParserCtxt(context);

  return new Document(doc);
}

}

namespace
{
  
void construct(xmlNode* node)
{
  switch (node->type)
  {
    case XML_ELEMENT_NODE:
    {
      node->_private = new xmlpp::Element(node);
      break;
    }
    case XML_ATTRIBUTE_NODE:
    {
      node->_private = new xmlpp::Attribute(node);
      break;
    }
    case XML_TEXT_NODE:
    {
      node->_private = new xmlpp::TextNode(node);
      break;
    }
    case XML_COMMENT_NODE:
    {
      node->_private = new xmlpp::CommentNode(node);
      break;
    }
    case XML_DTD_NODE:
    {
      node->_private = new xmlpp::Dtd(reinterpret_cast<xmlDtd*>(node));
      break;
    }
    case XML_DOCUMENT_NODE:
    {
      // do nothing ! in case of documents it's the wrapper that is the owner
      break;
    }
    default:
    {
      std::cout << "Warning: new node of unknown type " << node->type
                << " created !" << std::endl;
      break;
    }
  }
}

void destruct(xmlNode* node)
{
  bool bPrivateDeleted = false;
  if (node->type == XML_DTD_NODE)
  {
    xmlpp::Dtd* cppDtd = static_cast<xmlpp::Dtd*>(node->_private);
    if(cppDtd)
    {
      delete cppDtd;     
      bPrivateDeleted = true;
    }
  }
  else if (node->type == XML_DOCUMENT_NODE)
    // do nothing. See construct for an explanation
    ;
  else
  {
    xmlpp::Node* cppNode =  static_cast<xmlpp::Node*>(node->_private);
    if(cppNode)
    {
      delete cppNode;
      bPrivateDeleted = true;
    }
  }

  //This probably isn't necessary:
  if(bPrivateDeleted)
    node->_private = 0;
}

} //anonymous namespace

namespace xmlpp
{

Document::Init::Init()
{
   xmlRegisterNodeDefault(construct);
   xmlDeregisterNodeDefault(destruct);
}

Document::Init Document::_init;

Document::Document(const std::string& version)
  : _impl(xmlNewDoc((xmlChar*)version.c_str()))
{
  _impl->_private = this;
}

Document::Document(xmlDoc* doc)
  : _impl(doc)
{
  _impl->_private = this;
}

Document::~Document()
{
  xmlFreeDoc(_impl);
}

std::string Document::get_encoding() const
{
  std::string encoding;
  if(_impl->encoding) encoding = (char*)_impl->encoding;
  return encoding;
}

Dtd* Document::get_internal_subset() const
{
  xmlDtd* dtd = xmlGetIntSubset(_impl);
  if (!dtd)
    return 0;
    
  if (!dtd->_private)
    dtd->_private = new Dtd(dtd);
    
  return reinterpret_cast<Dtd*>(dtd->_private);
}

void Document::set_internal_subset(const std::string& name,
                                   const std::string& external_id,
                                   const std::string& system_id)
{
  xmlDtd* dtd = xmlCreateIntSubset(_impl,
				   (xmlChar*)name.c_str(),
				   (xmlChar*)external_id.c_str(),
				   (xmlChar*)system_id.c_str());
           
  if (dtd && !dtd->_private)
    dtd->_private = new Dtd(dtd);
}

Element* Document::get_root_node() const
{
  xmlNode* root = xmlDocGetRootElement(_impl);
  if (root == 0)
    return 0;
  else
    return reinterpret_cast<Element*>(root->_private);
}

Element* Document::create_root_node(const std::string& name,
                                    const std::string& ns_href,
                                    const std::string& ns_prefix)
{
  xmlNode* node = xmlNewDocNode(_impl, 0, (xmlChar*)name.c_str(), 0);
  if(!ns_prefix.empty())
     xmlNewNs(node,
              (xmlChar*)(ns_href.empty() ? 0 : ns_href.c_str()),
              (xmlChar*) ns_prefix.c_str());

  xmlDocSetRootElement(_impl, node);

  return get_root_node();
}

void Document::write_to_file(const std::string& filename, const std::string& encoding) throw(exception)
{
  int result = 0;
  if(!encoding.empty())
    result = xmlSaveFormatFileEnc(filename.c_str(), _impl, encoding.c_str(), 1);
  else
    result = xmlSaveFormatFile(filename.c_str(), _impl, 1);

  if(result == -1)
    throw exception("write_to_file() failed.");
}

std::string Document::write_to_string(const std::string& encoding)
  throw(exception)
{
  xmlChar* buffer = 0;
  int length = 0;
  
  if(!encoding.empty())
    xmlDocDumpMemoryEnc(_impl, &buffer, &length, encoding.c_str());
  else
    xmlDocDumpMemory(_impl, &buffer, &length);

  if(!buffer) throw exception("write_to_string() failed.");

  return std::string((char*)buffer, length);
}

} //namespace xmlpp


