/* xml++.cc
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 *
 * 2002/01/05 Valentin Rusu - fixed some potential buffer overruns
 * 2002/01/21 Valentin Rusu - added CDATA handlers
 */

#include "libxml++/parsers/saxparser.h"
#include <libxml/parserInternals.h> // for xmlCreateFileParserCtxt
#include <fstream> //For std::ifstream.
#include <cstdarg> //For va_list.
#include <iostream> //For cerr

namespace xmlpp {


SaxParser::SaxParser()
{
  xmlSAXHandler temp = {
    NULL,		// internalSubset
    NULL,		// isStandalone
    NULL,		// hasInternalSubset
    NULL,		// hasExternalSubset
    NULL,		// resolveEntity
    _callback_get_entity,	// getEntity
    NULL,		// entityDecl
    NULL,		// notationDecl
    NULL,		// attributeDecl
    NULL,		// elementDecl
    NULL,		// unparsedEntityDecl
    NULL,		// setDocumentLocator
    _callback_start_document,	// startDocument
    _callback_end_document,	// endDocument
    _callback_start_element,	// startElement
    _callback_end_element,	// endElement
    NULL,		// reference
    _callback_characters,	// characters
    NULL,		// ignorableWhitespace
    NULL,		// processingInstruction
    _callback_comment,		// comment
    _callback_warning,		// warning
    _callback_error,		// error
    _callback_fatal_error,	// fatalError
    NULL,		// getParameterEntity
    _callback_cdata_block,	// cdataBlock
    NULL		// externalSubset
  };
  _sax_handler = temp;
}

SaxParser::~SaxParser()
{
  release_underlying();
}

xmlEntityPtr SaxParser::on_get_entity(const xmlChar* name)
{
  return xmlGetPredefinedEntity(name);
}

void SaxParser::on_start_document()
{
}

void SaxParser::on_end_document()
{
}

void SaxParser::on_start_element(const std::string& name, const AttributeMap& attributes)
{
}

void SaxParser::on_end_element(const std::string& name)
{
}

void SaxParser::on_characters(const std::string& text)
{
}

void SaxParser::on_comment(const std::string& text)
{
}

void SaxParser::on_warning(const std::string& text)
{
}

void SaxParser::on_error(const std::string& text)
{
}

void SaxParser::on_fatal_error(const std::string& text)
{
  throw parse_error("Fatal error: " + text);
}

void SaxParser::on_cdata_block(const std::string& text)
{
}

// implementation of this function is inspired by the SAX documentation by James Henstridge.
// (http://www.daa.com.au/~james/gnome/xml-sax/implementing.html)
void SaxParser::parse()
{
  if(!_context)
    throw internal_error("Parse context not created.");

  _context->sax = & _sax_handler;
  _context->userData = this;

  initialize_context();
  
  xmlParseDocument(_context);

  if( (! _context->wellFormed)
      && (! _exception) )
    _exception = new parse_error("Document not well-formed");

  release_underlying();

  check_for_exception();
}

void SaxParser::parse_file(const std::string& filename) throw(exception)
{
  if(_context)
    throw parse_error("Attempt to start a second parse while a parse is in progress.");

  _context = xmlCreateFileParserCtxt(filename.c_str());
  parse();
}

void SaxParser::parse_memory(const std::string& contents) throw(exception)
{
  if(_context)
    throw parse_error("Attempt to start a second parse while a parse is in progress.");

  _context = xmlCreateMemoryParserCtxt(contents.c_str(), contents.length());
  parse();
}

void SaxParser::parse_stream(std::istream& in) throw(exception)
{
  if(_context)
    throw parse_error("Attempt to start a second parse while a parse is in progress.");

  _context = xmlCreatePushParserCtxt(
      &_sax_handler,
      this, // user_data
      NULL,
      0,
      ""); // This should be the filename. I don't know if it is a problem to leave it empty.

  initialize_context();

  std::string line;
  while( ( ! _exception )
      && getline(in, line))
  {
    // since getline does not get the line separator, we have to add it since the parser care
    // about layout in certain cases.
    line += '\n';

    xmlParseChunk(_context, line.c_str(), line.length(), 0);
  }

  if( ! _exception )
    xmlParseChunk(_context, NULL, 0, 1);

  release_underlying();

  check_for_exception();
}

void SaxParser::parse_chunk(const std::string& chunk) throw(exception)
{
  if(!_context)
  {
    _context = xmlCreatePushParserCtxt(
      &_sax_handler,
      this, // user_data
      NULL,
      0,
      ""); // here should come the filename. I don't know if it is a problem to let it empty

    initialize_context();
  }
  
  if(!_exception)
    xmlParseChunk(_context, chunk.c_str(), chunk.size(), 0);

  check_for_exception();
}


void SaxParser::release_underlying()
{
  if(_context)
    _context->sax = 0; //Prevent xmlFreeParserCtxt() from freeing it.

  Parser::release_underlying();
}

void SaxParser::finish_chunk_parsing() throw(exception)
{
  if(!_context)
  {
    _context = xmlCreatePushParserCtxt(
      &_sax_handler,
      this, // user_data
      NULL,
      0,
      ""); // This should be the filename. I don't know if it is a problem to leave it empty
  }
  
  if(!_exception)
    xmlParseChunk(_context, NULL, 0, 1);

  release_underlying();

  check_for_exception();
}

xmlEntityPtr SaxParser::_callback_get_entity(void* _parser, const xmlChar* name)
{
  SaxParser* parser = static_cast<SaxParser*>(_parser);
  return parser->on_get_entity(name);
}

void SaxParser::_callback_start_document(void* _parser)
{
  SaxParser* parser = (SaxParser*) _parser;
  try
  {
    parser->on_start_document();
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParser::_callback_end_document(void* _parser)
{
  SaxParser* parser = (SaxParser*) _parser;
  if(parser->_exception)
    return;

  try
  {
    parser->on_end_document();
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParser::_callback_start_element(void* _parser,
                                        const xmlChar* name,
                                        const xmlChar** p)
{
  SaxParser* parser = (SaxParser*) _parser;
  AttributeMap attributes;

  if(p)
    for(const xmlChar** cur = p; cur && *cur; cur += 2)
      attributes[(char*)*cur] = (char*)*(cur + 1);
   
  try
  {
    parser->on_start_element(std::string((const char*) name), attributes);
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParser::_callback_end_element(void* _parser, const xmlChar* name)
{
  SaxParser* parser = (SaxParser*) _parser;
  try
  {
    parser->on_end_element(std::string((const char*) name));
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParser::_callback_characters(void* _parser, const xmlChar* ch, int len)
{
  SaxParser* parser = (SaxParser*) _parser;
  try
  {
    parser->on_characters(std::string((const char*) ch, len));
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParser::_callback_comment(void* _parser, const xmlChar* value)
{
  SaxParser* parser = (SaxParser*) _parser;
  try
  {
    parser->on_comment(std::string((const char*) value));
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParser::_callback_warning(void* _parser, const char* fmt, ...)
{
  SaxParser* parser = (SaxParser*) _parser;

  va_list arg;
  char buff[1024]; //TODO: Larger/Shared

  va_start(arg, fmt);
  vsnprintf(buff, sizeof(buff)/sizeof(buff[0]), fmt, arg);
  va_end(arg);

  try
  {
    parser->on_warning(std::string(buff));
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParser::_callback_error(void* _parser, const char* fmt, ...)
{
  SaxParser* parser = (SaxParser*) _parser;
  va_list arg;
  char buff[1024]; //TODO: Larger/Shared

  if(parser->_exception)
    return;

  va_start(arg, fmt);
  vsnprintf(buff, sizeof(buff)/sizeof(buff[0]), fmt, arg);
  va_end(arg);

  try
  {
    parser->on_error(std::string(buff));
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParser::_callback_fatal_error(void* _parser, const char* fmt, ...)
{
  SaxParser* parser = (SaxParser*) _parser;
  va_list arg;
  char buff[1024]; //TODO: Larger/Shared

  va_start(arg, fmt);
  vsnprintf(buff, sizeof(buff)/sizeof(buff[0]), fmt, arg);
  va_end(arg);

  try
  {
    parser->on_fatal_error(std::string(buff));
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParser::_callback_cdata_block(void* _parser, const xmlChar* value, int len)
{
  SaxParser* parser = (SaxParser*) _parser;
  try
  {
    parser->on_cdata_block(std::string((const char*)value, len));
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  } 
}





} // namespace xmlpp


