/* xml++.cc
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 */

#include <libxml++/nodes/element.h>
#include <libxml++/nodes/textnode.h>
#include <libxml++/exceptions/internal_error.h>
#include <iostream>

namespace xmlpp
{

Element::Element(xmlNode* node)
: Node(node)
{}

Element::~Element()
{}

Attribute* Element::add_attribute(const std::string& name, const std::string& value)
{
  return set_attribute(name, value);
}

Element::AttributeList Element::get_attributes() const
{
  AttributeList attributes;
  for(xmlAttr* attr = cobj()->properties; attr; attr = attr->next)
  {
    attributes.push_back(reinterpret_cast<Attribute*>(attr->_private));
  }

  return attributes;
}

Attribute* Element::get_attribute(const std::string& name) const
{
  for(xmlAttr* attr = cobj()->properties; attr; attr = attr->next)
  {
    if(xmlStrEqual(attr->name, (xmlChar*)name.c_str()))
      return reinterpret_cast<Attribute*>(attr->_private);
  }

  return 0;
}

Attribute* Element::set_attribute(const std::string& name, const std::string& value)
{
  xmlAttr* attr = xmlSetProp(cobj(),
                             (xmlChar*)name.c_str(),
                             (xmlChar*)value.c_str());
  return reinterpret_cast<Attribute*>(attr->_private);
}

void Element::remove_attribute(const std::string& name)
{
  xmlUnsetProp(cobj(), (xmlChar*)name.c_str());
}

const TextNode* Element::get_child_content() const
{
  // FIXME: return only the first content node
  for(xmlNode* child = cobj()->children; child; child = child->next)
     if(child->type == XML_TEXT_NODE)
        return static_cast<TextNode*>(child->_private);

  return 0;
}

TextNode* Element::get_child_content()
{
  // FIXME: return only the first content node
  for(xmlNode* child = cobj()->children; child; child = child->next)
     if(child->type == XML_TEXT_NODE)
        return static_cast<TextNode*>(child->_private);

  return 0;
}

void Element::set_child_content(const std::string& content)
{
  TextNode* node = get_child_content();
  if(node)
    node->set_content(content);
  else
    add_content(content);
}

TextNode* Element::add_content(const std::string& content)
{
  if(cobj()->type == XML_ELEMENT_NODE)
  {
     xmlNode* node = xmlNewText((xmlChar*)content.c_str());
     xmlAddChild(cobj(), node);
     return static_cast<TextNode*>(node->_private);
  }
  return 0;
}

bool Element::has_content() const
{
  return get_child_content() != 0;
}



} //namespace xmlpp


