/* xml++.cc
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 */

#include "libxml++/parsers/domparser.h"
#include "libxml++/dtd.h"
#include "libxml++/nodes/element.h"
#include "libxml++/nodes/textnode.h"
#include "libxml++/nodes/commentnode.h"
#include "libxml++/exceptions/internal_error.h"
#include <libxml/parserInternals.h>//For xmlCreateFileParserCtxt().

namespace xmlpp
{

DomParser::DomParser()
: _doc(0)
{
  //Start with an empty document:
  _doc = new Document();
}

DomParser::DomParser(const std::string& filename, bool validate)
: _doc(0)
{
  set_validate(validate);
  parse_file(filename);
}

DomParser::~DomParser()
{ 
  release_underlying();
}

void DomParser::parse_file(const std::string& filename) throw(exception)
{
  release_underlying(); //Free any existing document.

  //These default values seem to affect the execution of xmlCreateFileParserCtxt(),
  //leading to extra blank text nodes from xmlParseDocument().
  //We set the same stuff per-context in initialize_context(), before calling xmlParseDocument().
  //but that seems to be too late.
  int old_keepblanks = xmlKeepBlanksDefault(CONSTANT_KeepBlanksSetting); 

  //The following is based on the implementation of xmlParseFile(), in xmlSAXParseFileWithData():
  _context = xmlCreateFileParserCtxt(filename.c_str());

  if(!_context)
  {
    throw internal_error("Couldn't create parsing context");
  }

  //Restore the old value and trust initialize_context() to deal with this from now on:
  xmlKeepBlanksDefault(old_keepblanks); 


  if(_context->directory == NULL)
  {
    char* directory = xmlParserGetDirectory(filename.c_str());
    _context->directory = (char*) xmlStrdup((xmlChar*) directory);
  }

  parse_context();
}

void DomParser::parse_memory(const std::string& contents) throw(exception)
{
  release_underlying(); //Free any existing document.

  //See parse_file() about this - I haven't tested that it's necessary with xmlCreateMemoryParserCtxt() too,
  //but it probably is:
  int old_keepblanks = xmlKeepBlanksDefault(CONSTANT_KeepBlanksSetting);

  //The following is based on the implementation of xmlParseFile(), in xmlSAXParseFileWithData():
  _context = xmlCreateMemoryParserCtxt(contents.c_str(), contents.size());

  if(!_context)
  {
    throw internal_error("Couldn't create parsing context");
  }

  xmlKeepBlanksDefault(old_keepblanks);

  parse_context();
}

void DomParser::parse_context() throw(exception)
{
  //The following is based on the implementation of xmlParseFile(), in xmlSAXParseFileWithData():
  //and the implementation of xmlParseMemory(), in xmlSaxParseMemoryWithData().
  initialize_context();

  xmlParseDocument(_context);
  _doc = new Document(_context->myDoc);

  check_for_exception();

  if(!_context->wellFormed)
  {
    release_underlying(); //Free _doc;
    throw parse_error("Document not well-formed.");
  }

  if(!_context->valid)
  {
    //This would probably not be reached, because the callback would create an exception,
    //which would have been thrown already by check_for_exception() above.
    
    release_underlying(); //Free _doc;
    throw validity_error("Document not valid.");
  }

  //Free the parse context, but keep the document alive so people can navigate the DOM tree:
  //TODO: Why not keep the context alive too?
  Parser::release_underlying();

  check_for_exception();
}


void DomParser::parse_stream(std::istream& in) throw(exception)
{
  release_underlying(); //Free any existing document.

  _context = xmlCreatePushParserCtxt(
      NULL, // setting thoses two parameters to NULL force the parser
      NULL, // to create a document while parsing.
      NULL,
      0,
      ""); // here should come the filename. I don't know if it is a problem to let it empty

  if(!_context)
  {
    throw internal_error("Couldn't create parsing context");
  }

  initialize_context();

  std::string line;
  while(std::getline(in, line))
  {
    // since getline does not get the line separator, we have to add it since the parser cares
    // about layout in certain cases.
    line += '\n';

    xmlParseChunk(_context, line.c_str(), line.length(), 0);
  }

  xmlParseChunk(_context, NULL, 0, 1);

  _doc = new Document(_context->myDoc);

  //Free the parse context, but keep the document alive so people can navigate the DOM tree:
  //TODO: Why not keep the context alive too?
  Parser::release_underlying();

  check_for_exception();
}

void DomParser::release_underlying()
{
  if(_doc)
  {
    delete _doc;
    _doc = 0;
  }

  Parser::release_underlying();
}

DomParser::operator bool() const
{
  return _doc != 0;
}

Document* DomParser::get_document()
{
  return _doc;
}

const Document* DomParser::get_document() const
{
  return _doc;
}

} // namespace xmlpp


