/* xml++.cc
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 */

#include "libxml++/parsers/parser.h"
#include <cstdarg> //For va_list.

namespace xmlpp {

Parser::Parser()
: _context(0), _exception(0), _validate(false)
{

}

Parser::~Parser()
{
  release_underlying();
}

void Parser::set_validate(bool val)
{
  _validate = val;
}

bool Parser::get_validate() const
{
  return _validate;
}

void Parser::initialize_context()
{
  //Disactivate any non-standards-compliant libxml1 features.
  //These are disactivated by default, but if we don't deactivate them for each context
  //then some other code which uses a global function, such as xmlKeepBlanksDefault(),
  // could cause this to use the wrong settings:
  _context->linenumbers = 1; // TRUE - This is the default anyway.
  _context->keepBlanks = CONSTANT_KeepBlanksSetting; // FALSE - This is the default anyway.

  //Turn on/off validation:
  _context->validate = (_validate ? 1 : 0);

  //Tell the validity context about the callbacks:
  //(These are only called if validation is on - see above)
  _context->vctxt.error = &_callback_validity_error;
  _context->vctxt.warning = &_callback_validity_warning;

  //Allow the _callback_validity_*() methods to retrieve the C++ instance:
  _context->_private = this;

  //Clear these temporary buffers too:
  _validate_error.erase();
  _validate_warning.erase();
}

void Parser::release_underlying()
{
  if(_context)
  {
    _context->_private = 0; //Not really necessary.
    
    xmlFreeParserCtxt(_context);
    _context = 0;
  }
}

void Parser::on_validity_error(const std::string& message)
{
  //Throw an exception later when the whole message has been received:
  _validate_error += message;
}

void Parser::on_validity_warning(const std::string& message)
{
  //Throw an exception later when the whole message has been received:
  _validate_warning += message;
}

void Parser::check_for_validity_messages()
{
  if(!_validate_error.empty())
  {
    if(!_exception)
      _exception = new validity_error("Validity error:\n" + _validate_error);

    _validate_error.erase();
  }

  if(!_validate_warning.empty())
  {
    if(!_exception)
      _exception = new validity_error("Validity warning:\n" + _validate_warning);

    _validate_warning.erase();
  }
}
  
void Parser::_callback_validity_error(void* _context, const char* msg, ...)
{
  //See xmlHTMLValidityError() in xmllint.c in libxml for more about this:
  
  xmlParserCtxtPtr context = (xmlParserCtxtPtr)_context;
  if(context)
  {
    Parser* parser = static_cast<Parser*>(context->_private);
    if(parser)
    {
      //Convert the ... to a string:
      va_list arg;
      char buff[1024]; //TODO: Larger/Shared

      va_start(arg, msg);
      vsnprintf(buff, sizeof(buff)/sizeof(buff[0]), msg, arg);
      va_end(arg);

      try
      {
        parser->on_validity_error(std::string(buff));
      }
      catch(const exception& e)
      {
        parser->handleException(e);
      }
    }
  }
  
}

void Parser::_callback_validity_warning(void* _context, const char* msg, ...)
{
  //See xmlHTMLValidityError() in xmllint.c in libxml for more about this:
  
  xmlParserCtxtPtr context = (xmlParserCtxtPtr)_context;
  if(context)
  {
    Parser* parser = static_cast<Parser*>(context->_private);
    if(parser)
    {
      //Convert the ... to a string:
      va_list arg;
      char buff[1024]; //TODO: Larger/Shared

      va_start(arg, msg);
      vsnprintf(buff, sizeof(buff)/sizeof(buff[0]), msg, arg);
      va_end(arg);

      try
      {
        parser->on_validity_warning(std::string(buff));
      }
      catch(const exception& e)
      {
        parser->handleException(e);
      }
    }
  }
}

void Parser::handleException(const exception& e)
{
  _exception = e.Clone();

  if(_context)
    xmlStopParser(_context);

  //release_underlying();
}

void Parser::check_for_exception()
{
  check_for_validity_messages();
  
  if(_exception)
  {
    exception* tmp = _exception;
    _exception = 0;
    tmp->Raise();
  }
}

} // namespace xmlpp


