/* xml++.cc
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 */

#include <libxml++/nodes/node.h>
#include <libxml++/exceptions/internal_error.h>

#include <libxml/xpath.h>
#include <libxml/tree.h>

namespace xmlpp
{

Node::Node(xmlNode* node)
  : _impl(node)
{
   _impl->_private = this;
}

Node::~Node()
{}

Node::NodeList Node::get_children(const std::string& name)
{
   xmlNode* child = _impl->children;
   if(!child)
     return NodeList();

   NodeList children;
   do
   {
      if(name.empty() || name == (char*)child->name)
         children.push_back(reinterpret_cast<Node*>(child->_private));
   }
   while((child = child->next));
   
   return children;
}

const Node::NodeList Node::get_children(const std::string& name) const
{
  return const_cast<Node*>(this)->get_children(name);
}

Element* Node::add_child(const std::string& name)
{
   if(_impl->type != XML_ELEMENT_NODE)
      throw internal_error("you can only add child nodes to element nodes");
      
   xmlNode* node = xmlAddChild(_impl, xmlNewNode(0, (xmlChar*)name.c_str()));
   return static_cast<Element*>(node->_private);
}

void Node::remove_child(Node* node)
{
  //TODO: Allow a node to be removed without deleting it, to allow it to be moved?
  //This would require a more complex memory management API.
  xmlUnlinkNode(node->cobj());
  xmlFreeNode(node->cobj()); //The C++ instance will be deleted in a callback.
}

std::string Node::get_name() const
{
  return _impl->name ? (char*)_impl->name : "";
}

void Node::set_name(const std::string & name)
{
  xmlNodeSetName( _impl, (const xmlChar *)name.c_str() );
}

int Node::get_line() const
{
   return XML_GET_LINE(_impl);
}


xmlNode* Node::cobj()
{
  return _impl;
}

const xmlNode* Node::cobj() const
{
  return _impl;
}

std::string Node::get_path() const
{
  xmlChar* path = xmlGetNodePath(_impl);
  std::string retn = path ? (char*)path : "";
  xmlFree(path);
  return retn;
}

NodeSet Node::find(const std::string& xpath) const
{
  xmlXPathContext* ctxt = xmlXPathNewContext(_impl->doc);
  ctxt->node = _impl;
  xmlXPathObject* result = xmlXPathEval((xmlChar*)xpath.c_str(), ctxt);

  if (result->type != XPATH_NODESET)
  {
    xmlXPathFreeObject(result);
    xmlXPathFreeContext(ctxt);
    throw internal_error("sorry, only nodeset result types supported for now.");
  }

  xmlNodeSet* nodeset = result->nodesetval;
  NodeSet nodes;
  if( nodeset ) {
    for (int i = 0; i != nodeset->nodeNr; ++i)
      nodes.push_back(static_cast<Node*>(nodeset->nodeTab[i]->_private));
  }
  else {
    // return empty set
  }
  xmlXPathFreeObject(result);
  xmlXPathFreeContext(ctxt);

  return nodes;
}

} //namespace xmlpp





