/* saxparser.h
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 */

#ifndef __LIBXMLPP_PARSERS_SAXPARSER_H
#define __LIBXMLPP_PARSERS_SAXPARSER_H

#include <libxml++/parsers/parser.h>

#include <list>
#include <map>
#include <memory>

extern "C" {
  struct _xmlSAXHandler;
  struct _xmlEntity;
};

namespace xmlpp {

/** SAX XML parser.
 * Derive your own class and override the on_*() methods.
 */
class SaxParser : public Parser
{
public:
  typedef std::map<std::string, std::string> AttributeMap;
  /** 
   * @param use_get_entity set to true when getEntity callback is going to be used.
   * In such a case on_get_entity MUST be overridden.
   */
  SaxParser(bool use_get_entity = false);
  virtual ~SaxParser();

  virtual void parse_file(const std::string& filename);
  virtual void parse_memory(const std::string& contents);
  virtual void parse_stream(std::istream& in);
  
  /** Parse a chunk of data.
   *
   * This lets you pass a document in small chunks, e.g. from a network
   * connection. The on_* virtual functions are called each time the chunks
   * provide enough information to advance the parser.
   *
   * The first call to parse_chunk will setup the parser. When the last chunk
   * has been parsed, call finish_chunk_parsing() to finish the parse.
   */
  virtual void parse_chunk(const std::string& chunk);

  /** Finish a chunk-wise parse.
   *
   * Call this after the last call to parse_chunk(). Don't use this function with
   * the other parsing methods.
   */
  virtual void finish_chunk_parsing();

protected:

  /** If you override this function see constructor parameter use_get_entity */
  virtual _xmlEntity* on_get_entity(const std::string& name);
  virtual void on_start_document();
  virtual void on_end_document();
  virtual void on_start_element(const std::string& name, const AttributeMap& attributes);
  virtual void on_end_element(const std::string& name);
  virtual void on_characters(const std::string& characters);
  virtual void on_comment(const std::string& text);
  virtual void on_warning(const std::string& text);
  virtual void on_error(const std::string& text);
  virtual void on_fatal_error(const std::string& text);
  virtual void on_cdata_block(const std::string& text);

  virtual void release_underlying();
  
private:
  virtual void parse();
  
  std::auto_ptr<_xmlSAXHandler> _sax_handler;

  friend struct SaxParserCallback;
};




} // namespace xmlpp

#endif //__LIBXMLPP_PARSERS_SAXPARSER_H



