/* document.cc
 * this file is part of libxml++
 *
 * copyright (C) 2003 by the libxml++ development team
 *
 * this file is covered by the GNU Lesser General Public License,
 * which should be included with libxml++ as the file COPYING.
 */

#include <libxml++/document.h>
#include <libxml++/dtd.h>
#include <libxml++/attribute.h>
#include <libxml++/nodes/element.h>
#include <libxml++/nodes/textnode.h>
#include <libxml++/nodes/commentnode.h>
#include <libxml++/nodes/cdatanode.h>
#include <libxml++/nodes/processinginstructionnode.h>
#include <libxml++/exceptions/internal_error.h>
#include <libxml++/keepblanks.h>

#include <libxml/tree.h>

#include <assert.h>

namespace
{

//Called by libxml whenever it constructs something,
//such as a node or attribute.
//This allows us to create a C++ instance for every C instance.
void on_libxml_construct(xmlNode* node)
{
  switch (node->type)
  {
    case XML_ELEMENT_NODE:
    {
      node->_private = new xmlpp::Element(node);
      break;
    }
    case XML_ATTRIBUTE_NODE:
    {
      node->_private = new xmlpp::Attribute(node);
      break;
    }
    case XML_TEXT_NODE:
    {
      node->_private = new xmlpp::TextNode(node);
      break;
    }
    case XML_COMMENT_NODE:
    {
      node->_private = new xmlpp::CommentNode(node);
      break;
    }
    case XML_CDATA_SECTION_NODE:
    {
      node->_private = new xmlpp::CdataNode(node);
      break;
    }
    case XML_PI_NODE:
    {
      node->_private = new xmlpp::ProcessingInstructionNode(node);
      break;
    }
    case XML_DTD_NODE:
    {
      node->_private = new xmlpp::Dtd(reinterpret_cast<xmlDtd*>(node));
      break;
    }
    case XML_DOCUMENT_NODE:
    {
      // do nothing ! in case of documents it's the wrapper that is the owner
      break;
    }
    default:
    {
      // good default for release versions
      node->_private = new xmlpp::ContentNode(node);
      assert(0 && "Warning: new node of unknown type created");
      break;
    }
  }
}

//Called by libxml whenever it destroys something
//such as a node or attribute.
//This allows us to delete the C++ instance for the C instance, if any.
void on_libxml_destruct(xmlNode* node)
{
  bool bPrivateDeleted = false;
  if (node->type == XML_DTD_NODE)
  {
    xmlpp::Dtd* cppDtd = static_cast<xmlpp::Dtd*>(node->_private);
    if(cppDtd)
    {
      delete cppDtd;     
      bPrivateDeleted = true;
    }
  }
  else if (node->type == XML_DOCUMENT_NODE)
    // do nothing. See on_libxml_construct for an explanation
    ;
  else
  {
    xmlpp::Node* cppNode =  static_cast<xmlpp::Node*>(node->_private);
    if(cppNode)
    {
      delete cppNode;
      bPrivateDeleted = true;
    }
  }

  //This probably isn't necessary:
  if(bPrivateDeleted)
    node->_private = 0;
}

} //anonymous namespace

namespace xmlpp
{

Document::Init::Init()
{
   xmlInitParser(); //Not always necessary, but necessary for thread safety.
   xmlRegisterNodeDefault(on_libxml_construct);
   xmlDeregisterNodeDefault(on_libxml_destruct);
}

Document::Init Document::_init;

Document::Document(const std::string& version)
  : _impl(xmlNewDoc((xmlChar*)version.c_str()))
{
  _impl->_private = this;
}

Document::Document(xmlDoc* doc)
  : _impl(doc)
{
  _impl->_private = this;
}

Document::~Document()
{
  xmlFreeDoc(_impl);
}

std::string Document::get_encoding() const
{
  std::string encoding;
  if(_impl->encoding)
    encoding = (char*)_impl->encoding;
    
  return encoding;
}

Dtd* Document::get_internal_subset() const
{
  xmlDtd* dtd = xmlGetIntSubset(_impl);
  if(!dtd)
    return 0;
    
  if(!dtd->_private)
    dtd->_private = new Dtd(dtd);
    
  return reinterpret_cast<Dtd*>(dtd->_private);
}

void Document::set_internal_subset(const std::string& name,
                                   const std::string& external_id,
                                   const std::string& system_id)
{
  xmlDtd* dtd = xmlCreateIntSubset(_impl,
				   (xmlChar*)name.c_str(),
				   (xmlChar*)external_id.c_str(),
				   (xmlChar*)system_id.c_str());
           
  if (dtd && !dtd->_private)
    dtd->_private = new Dtd(dtd);
}

Element* Document::get_root_node() const
{
  xmlNode* root = xmlDocGetRootElement(_impl);
  if(root == 0)
    return 0;
  else
    return reinterpret_cast<Element*>(root->_private);
}

Element* Document::create_root_node(const std::string& name,
                                    const std::string& ns_uri,
                                    const std::string& ns_prefix)
{
  xmlNode* node = xmlNewDocNode(_impl, 0, (xmlChar*)name.c_str(), 0);
  xmlDocSetRootElement(_impl, node);

  Element* element = get_root_node();
  
  if( !ns_uri.empty() )
  {
    element->set_namespace_declaration(ns_uri, ns_prefix);
    element->set_namespace(ns_prefix);
  }

  return element;
}

void Document::write_to_file(const std::string& filename, const std::string& encoding)
{
  KeepBlanks k(KeepBlanks::Default);
  int result = 0;
  
  if(!encoding.empty())
    result = xmlSaveFileEnc(filename.c_str(), _impl, encoding.c_str());
  else
    result = xmlSaveFile(filename.c_str(), _impl);

  if(result == -1)
    throw exception("write_to_file() failed.");
}

void Document::write_to_file_formatted(const std::string& filename, const std::string& encoding)
{
  KeepBlanks k(KeepBlanks::Default);
  xmlIndentTreeOutput = 1;
  int result = 0;
  
  if(!encoding.empty())
    result = xmlSaveFormatFileEnc(filename.c_str(), _impl, encoding.c_str(), 1);
  else
    result = xmlSaveFormatFile(filename.c_str(), _impl, 1);

  if(result == -1)
    throw exception("write_to_file_formatted() failed.");
}

std::string Document::write_to_string(const std::string& encoding)
{
  KeepBlanks k(KeepBlanks::Default);
  xmlChar* buffer = 0;
  int length = 0;
  
  if(!encoding.empty())
    xmlDocDumpMemoryEnc(_impl, &buffer, &length, encoding.c_str());
  else
    xmlDocDumpMemory(_impl, &buffer, &length);

  if(!buffer)
    throw exception("write_to_string() failed.");

  // Create a std::string copy of the buffer
  std::string result((char*)buffer, length);
  // Deletes the original buffer
  xmlFree(buffer);
  // Return a copy of the string
  return result;
}

std::string Document::write_to_string_formatted(const std::string& encoding)
{
  KeepBlanks k(KeepBlanks::Default);
  xmlIndentTreeOutput = 1;
  xmlChar* buffer = 0;
  int length = 0;

  if(!encoding.empty())
    xmlDocDumpFormatMemoryEnc(_impl, &buffer, &length, encoding.c_str(), 1);
  else
    xmlDocDumpFormatMemory(_impl, &buffer, &length, 1);

  if(!buffer)
    throw exception("write_to_string_formatted() failed.");

  // Create a std::string copy of the buffer
  std::string result((char*)buffer, length);
  // Deletes the original buffer
  xmlFree(buffer);
  // Return a copy of the string
  return result;
}

} //namespace xmlpp
