/* node.h
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 */

#ifndef __LIBXMLPP_NODES_ELEMENT_H
#define __LIBXMLPP_NODES_ELEMENT_H

#include <libxml++/nodes/node.h>
#include <libxml++/attribute.h>
#include <libxml++/nodes/commentnode.h>

namespace xmlpp
{

class Element : public Node
{
public:
  explicit Element(_xmlNode* node);
  virtual ~Element();

  /** This adds a namespace declaration to this node which will apply to this node and all children.
   * @param ns_uri The namespace to associate with the prefix, or to use as the default namespace if no prefix is specified.
   * @param ns_prefix The namespace prefix. If no prefix is specified then the namespace URI will be the default namespace.
   */
  void set_namespace_declaration(const std::string& ns_uri, const std::string& ns_prefix = std::string());

  AttributeList get_attributes() const;
  
  // FIXME: the following only returns explicitely provided
  // attributes, not default ones declared in the dtd.
  // TOOD: Is this still true? murrayc
  Attribute* get_attribute(const std::string& name,
                           const std::string& ns_prefix = std::string()) const;

  /** Set the value of the attribute with this name, and optionally with this namespace.
   * A matching attribute will be added if no matching attribute already exists.
   * For finer control, you might want to use get_attribute() and use the methods of the Attribute class.
   * @param name The name of the attribute whose value will change.
   * @param value The new value for the attribute
   * @param ns_prefix Namespace prefix. If the prefix has not been declared then this method will throw an exception.
   * @return The attribute that was changed, or 0 is no suitable Attribute was found.
   */
  Attribute* set_attribute(const std::string& name, const std::string& value,
                           const std::string& ns_prefix = std::string());

  /** Remove the attribute with this name, and optionally with this namespace.
   * @param name The name of the attribute to be removed
   * @param ns_prefix Namespace prefix. If specified, the attribute will be removed only if the attribute has this namespace.
   */
  void remove_attribute(const std::string& name,
                        const std::string& ns_prefix = std::string());
  
  
  // TODO: this should be reimplemented in two ways:
  // a) using iterators
  // b) returning a heap-allocated buffer containing the
  //    concatenation of all text node content
  //
  // for now only return the first text node (if any)
  TextNode* get_child_content();
  const TextNode* get_child_content() const;

  // Append a new text node
  TextNode* add_content(const std::string& content = std::string());
  void set_child_content(const std::string& content);
  bool has_content() const;

protected:
  std::string get_namespace_uri_for_prefix(const std::string& ns_prefix) const;
};

} // namespace xmlpp

#endif //__LIBXMLPP_NODES_ELEMENT_H
