/* node.h
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 */

#ifndef __LIBXMLPP_NODES_NODE_H
#define __LIBXMLPP_NODES_NODE_H

#include <libxml++/noncopyable.h>
#include <libxml++/exceptions/exception.h>
#include <string>
#include <list>
#include <map>
#include <vector>

extern "C" {
  struct _xmlNode;
};

namespace xmlpp {

class TextNode;
class Element;
class Attribute;

class Node;
typedef std::vector<Node*> NodeSet;

/** Represents XML Nodes.
 * You should never new or delete Nodes. The Parser will create and manage them for you.
 */
class Node : public NonCopyable
{
public:
  typedef std::list<Node*> NodeList;
  typedef std::list<Attribute*> AttributeList;

  explicit Node(_xmlNode* node);
  virtual ~Node();

  std::string get_name() const;
  void set_name(const std::string & name);

  /** Set the namespace prefix used by the node
   * If no such namespace prefix has been declared then this method will throw an exception.
   */
  void set_namespace(const std::string& ns_prefix);

  std::string get_namespace_prefix() const;
  std::string get_namespace_uri() const;

  int get_line() const;;
  
  NodeList get_children(const std::string& name = std::string());
  const NodeList get_children(const std::string& name = std::string()) const;

  /** Add a child element to this node
   * @param name The new node name
   * @param ns_prefix The namespace prefix. If the prefix has not been declared then this method will throw an exception.
   * @return The newly-created element
   */
  Element* add_child(const std::string& name,
                     const std::string& ns_prefix = std::string());

  /** Remove the child node.
   * @param node The child node to remove. This Node will be deleted and therefore unusable after calling this method.
   */
  void remove_child(Node* node);

  /** Import node(s) from another document under this node, without affecting the source node.
   * @param node The node to copy and insert under the current node.
   * @param recursive Whether to import the child nodes also. Defaults to true.
   * @return The newly-created node.
   */
  Node* import_node(const Node* node, bool recursive = true);

  std::string get_path() const;
  NodeSet find(const std::string& xpath) const;

  ///Access the underlying libxml implementation.
  _xmlNode* cobj();

  ///Access the underlying libxml implementation.
  const _xmlNode* cobj() const;

private:
  _xmlNode* _impl;
};

} // namespace xmlpp

#endif //__LIBXMLPP_NODES_NODE_H
