/* xml++.cc
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 *
 * 2002/01/05 Valentin Rusu - fixed some potential buffer overruns
 * 2002/01/21 Valentin Rusu - added CDATA handlers
 */

#include "libxml++/parsers/saxparser.h"
#include "libxml++/nodes/element.h"
#include "libxml++/keepblanks.h"

#include <libxml/parser.h>
#include <libxml/parserInternals.h> // for xmlCreateFileParserCtxt

#include <cstdarg> //For va_list.
#include <cassert> // for assert()
#include <iostream>

namespace xmlpp {

struct SaxParserCallback
{
  static xmlEntityPtr _get_entity(void* _parser, const xmlChar* name);
  static void _start_document(void* _parser);
  static void _end_document(void* _parser);
  static void _start_element(void* _parser, const xmlChar* name, const xmlChar** p);
  static void _end_element(void* _parser, const xmlChar* name);
  static void _characters(void* _parser, const xmlChar* ch, int len);
  static void _comment(void* _parser, const xmlChar* value);
  static void _warning(void* _parser, const char* fmt, ...);
  static void _error(void* _parser, const char* fmt, ...);
  static void _fatal_error(void* _parser, const char* fmt, ...);
  static void _cdata_block(void* _parser, const xmlChar* value, int len);
};



SaxParser::SaxParser(bool use_get_entity)
  : _sax_handler( new _xmlSAXHandler )
{
  xmlSAXHandler temp = {
    0,  // internalSubset
    0,  // isStandalone
    0,  // hasInternalSubset
    0,  // hasExternalSubset
    0,  // resolveEntity
    use_get_entity ? SaxParserCallback::_get_entity : 0, // getEntity
    0,  // entityDecl
    0,  // notationDecl
    0,  // attributeDecl
    0,  // elementDecl
    0,  // unparsedEntityDecl
    0,  // setDocumentLocator
    SaxParserCallback::_start_document, // startDocument
    SaxParserCallback::_end_document, // endDocument
    SaxParserCallback::_start_element, // startElement
    SaxParserCallback::_end_element, // endElement
    0,  // reference
    SaxParserCallback::_characters, // characters
    0,  // ignorableWhitespace
    0,  // processingInstruction
    SaxParserCallback::_comment,  // comment
    SaxParserCallback::_warning,  // warning
    SaxParserCallback::_error,  // error
    SaxParserCallback::_fatal_error, // fatalError
    0,  // getParameterEntity
    SaxParserCallback::_cdata_block, // cdataBlock
    0  // externalSubset
  };
  *_sax_handler = temp;
}

SaxParser::~SaxParser()
{
  release_underlying();
}

xmlEntityPtr SaxParser::on_get_entity(const std::string& name)
{
  assert(0); // This function shouldn't be called. If the callback of getEntity
             // is activated it must be overridden.
  return 0;
}

void SaxParser::on_start_document()
{
}

void SaxParser::on_end_document()
{
}

void SaxParser::on_start_element(const std::string& name, const AttributeMap& attributes)
{
}

void SaxParser::on_end_element(const std::string& name)
{
}

void SaxParser::on_characters(const std::string& text)
{
}

void SaxParser::on_comment(const std::string& text)
{
}

void SaxParser::on_warning(const std::string& text)
{
}

void SaxParser::on_error(const std::string& text)
{
}

void SaxParser::on_fatal_error(const std::string& text)
{
  throw parse_error("Fatal error: " + text);
}

void SaxParser::on_cdata_block(const std::string& text)
{
}

// implementation of this function is inspired by the SAX documentation by James Henstridge.
// (http://www.daa.com.au/~james/gnome/xml-sax/implementing.html)
void SaxParser::parse()
{
  if(!_context)
    throw internal_error("Parse context not created.");

  xmlSAXHandlerPtr old_sax = _context->sax;
  _context->sax = _sax_handler.get();
  _context->userData = this;

  initialize_context();
  
  xmlParseDocument(_context);

  _context->sax = old_sax;

  if( (! _context->wellFormed)
      && (! _exception) )
    _exception = new parse_error("Document not well-formed");

  release_underlying();

  check_for_exception();
}

void SaxParser::parse_file(const std::string& filename)
{
  if(_context)
    throw parse_error("Attempt to start a second parse while a parse is in progress.");

  KeepBlanks k(KeepBlanks::Default);

  _context = xmlCreateFileParserCtxt(filename.c_str());
  parse();
}

void SaxParser::parse_memory(const std::string& contents)
{
  if(_context)
    throw parse_error("Attempt to start a second parse while a parse is in progress.");

  KeepBlanks k(KeepBlanks::Default);

  _context = xmlCreateMemoryParserCtxt(contents.c_str(), contents.length());
  parse();
}

void SaxParser::parse_stream(std::istream& in)
{
  if(_context)
    throw parse_error("Attempt to start a second parse while a parse is in progress.");

  KeepBlanks k(KeepBlanks::Default);

  _context = xmlCreatePushParserCtxt(
      _sax_handler.get(),
      this, // user_data
      0,
      0,
      ""); // This should be the filename. I don't know if it is a problem to leave it empty.

  initialize_context();

  std::string line;
  while( ( ! _exception )
      && std::getline(in, line))
  {
    // since getline does not get the line separator, we have to add it since the parser care
    // about layout in certain cases.
    line += '\n';

    xmlParseChunk(_context, line.c_str(), line.length(), 0 /* don't terminate */);
  }

  if( ! _exception )
    xmlParseChunk(_context, 0 /* chunk */, 0 /* size */, 1 /* terminate (1 or 0) */); //This seems to be called just to terminate parsing.

  release_underlying();

  check_for_exception();
}

void SaxParser::parse_chunk(const std::string& chunk)
{
  KeepBlanks k(KeepBlanks::Default);

  if(!_context)
  {
    _context = xmlCreatePushParserCtxt(
      _sax_handler.get(),
      this, // user_data
      0,
      0,
      ""); // This should be the filename. I don't know if it is a problem to let it empty

    initialize_context();
  }
  
  if(!_exception)
    xmlParseChunk(_context, chunk.c_str(), chunk.size(), 0 /* don't terminate */);

  check_for_exception();
}


void SaxParser::release_underlying()
{
  Parser::release_underlying();
}

void SaxParser::finish_chunk_parsing()
{
  if(!_context)
  {
    _context = xmlCreatePushParserCtxt(
      _sax_handler.get(),
      this, // user_data
      0,
      0,
      ""); // This should be the filename. I don't know if it is a problem to leave it empty
  }
  
  if(!_exception)
    xmlParseChunk(_context, 0 /* chunk */, 0 /* size */, 1 /* terminate (1 or 0) */); //This seems to be called just to terminate parsing.

  release_underlying();

  check_for_exception();
}


xmlEntityPtr SaxParserCallback::_get_entity(void* _parser, const xmlChar* name)
{
  SaxParser* parser = static_cast<SaxParser*>(_parser);
  xmlEntityPtr result = 0;

  try
  {
    result = parser->on_get_entity((const char *)name);
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
  
  return result;
}


void SaxParserCallback::_start_document(void* _parser)
{
  SaxParser* parser = (SaxParser*)_parser;
  
  try
  {
    parser->on_start_document();
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParserCallback::_end_document(void* _parser)
{
  SaxParser* parser = (SaxParser*)_parser;

  if(parser->_exception)
    return;

  try
  {
    parser->on_end_document();
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParserCallback::_start_element(void* _parser,
                                        const xmlChar* name,
                                        const xmlChar** p)
{
  SaxParser* parser = (SaxParser*)_parser;
  SaxParser::AttributeMap attributes;

  if(p)
    for(const xmlChar** cur = p; cur && *cur; cur += 2)
      attributes[(char*)*cur] = (char*)*(cur + 1);
   
  try
  {
    parser->on_start_element(std::string((const char*) name), attributes);
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParserCallback::_end_element(void* _parser, const xmlChar* name)
{
  SaxParser* parser = (SaxParser*)_parser;
  
  try
  {
    parser->on_end_element(std::string((const char*) name));
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParserCallback::_characters(void* _parser, const xmlChar* ch, int len)
{
  SaxParser* parser = (SaxParser*)_parser;

  try
  {
    parser->on_characters(std::string((const char*) ch, len));
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParserCallback::_comment(void* _parser, const xmlChar* value)
{
  SaxParser* parser = (SaxParser*)_parser;
  
  try
  {
    parser->on_comment(std::string((const char*) value));
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParserCallback::_warning(void* _parser, const char* fmt, ...)
{
  SaxParser* parser = (SaxParser*)_parser;

  va_list arg;
  char buff[1024]; //TODO: Larger/Shared

  va_start(arg, fmt);
  vsnprintf(buff, sizeof(buff)/sizeof(buff[0]), fmt, arg);
  va_end(arg);

  try
  {
    parser->on_warning(std::string(buff));
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParserCallback::_error(void* _parser, const char* fmt, ...)
{
  SaxParser* parser = (SaxParser*)_parser;
  va_list arg;
  char buff[1024]; //TODO: Larger/Shared

  if(parser->_exception)
    return;

  va_start(arg, fmt);
  vsnprintf(buff, sizeof(buff)/sizeof(buff[0]), fmt, arg);
  va_end(arg);

  try
  {
    parser->on_error(std::string(buff));
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParserCallback::_fatal_error(void* _parser, const char* fmt, ...)
{
  SaxParser* parser = (SaxParser*)_parser;
  va_list arg;
  char buff[1024]; //TODO: Larger/Shared

  va_start(arg, fmt);
  vsnprintf(buff, sizeof(buff)/sizeof(buff[0]), fmt, arg);
  va_end(arg);

  try
  {
    parser->on_fatal_error(std::string(buff));
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  }
}

void SaxParserCallback::_cdata_block(void* _parser, const xmlChar* value, int len)
{
  SaxParser* parser = (SaxParser*)_parser;
  try
  {
    parser->on_cdata_block(std::string((const char*)value, len));
  }
  catch(const exception& e)
  {
    parser->handleException(e);
  } 
}





} // namespace xmlpp


