/* lpp --  A document formatter for PostScript-printers.
   version 3.44 (c) Copyright Fredric Ihren 1994, fred@nada.kth.se
   lpp is not freeware. Price information and registered copies can be
   obtained from
   Address: Hippokampos, c/o Ihren, Stagneliusvagen 30, S-112 59 Stholm, Sweden
   Telephone: +46 8 6568654 Email: fred@nada.kth.se
     15 Feb 1990. Major revision
     20 Aug 1991. 8-bit adjustment
     10 Oct 1991. Patches for SysV
      2 Dec 1991. $LC_CTYPE, $LC_TIME, optional new layout with -n, etc
     22 Jan 1992. New option -e"nn=name"
     19 Aug 1992. Pagecounter, structuring comments and new sequencing
     23 Aug 1992. -L -c -0. Rearrangement of functions.
     28 Sep 1992: tabs as T not spaces
      7 Oct 1992: letter, filetest for executables
     23 Oct 1992: numberlines, refuses if too many unprintable chrs
      1 Dec 1992: fixed -S bug
     14 Oct 1993: fixed breaking of long lines in ps-code (jmr bug fix)
     26 Oct 1993: 3.4 DSC 3.0
      3 Mar 1994: 3.41 added choice of tab expansion -ts -tx
     18 Mar 1994: 3.42 bugfix: wrapped tab at end of page
     12 Dec 1994: 3.43 bugfix: accept last page of lines ending with ^M
     14 Dec 1994: 3.44 added report flag -R and pagewidth flag -k
     13 Jul 1995: added USEUNAME for Solaris (from klas) */

#include <stdio.h>
#include <ctype.h>
#include <pwd.h>
#ifdef BSDF
#include <string.h>
#else
#include <string.h>
#include <sys/fcntl.h>
#endif /* BSD */
#include <sys/time.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include "lpp.h"

#ifdef LANG
#include <time.h>
#include <locale.h>
#include <nl_types.h>
#include <langinfo.h>
static char *lc_time, *lc_ctype=NULL;
static char lc_c[2];
static time_t ltid;
static struct tm *tid;
static struct dtconv *dtcp;
#endif /* LANG */

#ifndef NOMAGICK
#include <a.out.h>
#ifndef A_MAGIC1	/* must be a VM/UNIX system */
#	define A_MAGIC1	OMAGIC
#	define A_MAGIC2	NMAGIC
#	define A_MAGIC3	ZMAGIC
#	undef ARMAG
#	define ARMAG	0177545
#endif
#endif /* NOMAGICK */
  
#ifdef USEUNAME
#include <sys/utsname.h>
#endif

extern int getoptindex;		/* getopts argv index */
extern char *optarg;		/* getopts current opt char */
extern char *getoptp;           /* getopts pointer in arg */

static char *prog;		/* program name argv[0] */
static char *libdir;		/* place for prologue and help files */
static char *tempdir;		/* place for temp file */
static char OutName[200] = "";	/* filename for output */
static char flagstr[1000] = ""; /* storage for expanded LPP flags */
static char pagestr[1000] = ""; /* storage for page code and flags */
static int  xfiles = 0;
static char *xfile[10];         /* name of include postscript files */
static char miscstr[60] = "";	/* string for date and host information */
static char *PrinterName = NULL;
static char tempstr[256] = "";	/* temp storage for some flags and paths */
static char *eargv[60];         /* argv from environment and for spooler */
static int eargc;
static FILE *PSfile = NULL;	/* output ps file */

/* formatting variables */
#define LINELENGTHOUT 75        /* Maxlength for PostScript lines */
static int showctrl = FALSE;    /* Optional ^-printing of ctrl characters */
static int pipeout = FALSE;	/* output to stdout (-p-) */
int files = 0;	                /* counts output files */
static int nouser = FALSE;	/* Don't send standard user information */
int quiet = FALSE;	        /* suppress stderr info messages */
static int noprint = FALSE;	/* PSfile only */
static int nullout = FALSE;     /* No spool no store */
static int append = FALSE;	/* should next file simply be appended? */
static int extraff = FALSE;	/* formfeed between appended files? */
static int landscape = FALSE;   /* landscape mode */
static int guide = FALSE;       /* extra guide page */
static int nextguide = FALSE;
static int doubleside = FALSE;  /* Doubleside feature */
static int copies = 1;          /* Print copies copies of each page */
static int letter = FALSE;      /* Use US papersize letter 8.5" x 11" */
static int altlayout = FALSE;   /* Alternative layout */
static int numberlines = FALSE;
static int filterformfeed = FALSE;
static int swedish = FALSE;     /* Use Swedish encoded font */
static int report = FALSE;      /* Present values for -f,-k,-w, and -Y */
static float fontsize = 10.0;
static float baselineskip = 10.0;
static int pagelength = 72;
static int wraplength = 82;
static int killength = 0;
static int units = 1;           /* Number of logical pages */
static int tablength = 8;       /* Tabstop length in characters */
static int movtab = 1;          /* Tab with spaces ? (diff for prop font) */
static int lines = 0;           /* Total lines this page */
static int totlines = 0;        /* Total lines this file */
static int outlines = 0;        /* Total written lines this file */
static int pages = 0;           /* Pages this file */
static int totpages = 0;        /* Total pages */
static int outpages = 0;        /* Pages written to PSfile */
static int written = 0;         /* Anything of value this page */
static int begun = 0;           /* True when page has begun, after %%Page */
static int prevlen = 0;         /* Number of characters on this line */
static int outlen = 0;          /* Current linelength in PSfile */
static int unit = 0;            /* Current logical page -1 */
static int wrapped = 0;
static int startcolumn = 0;
static int zapat = 5;           /* zap output when this many unprintable chrs*/
static int zapcnt = 0;          /* counts unprintable chars */

/* variables for getnextc */
static char buf[BUFSIZ];
static int charsread = 0;
static int charsleft = 0;

/* terminator2 - a sighandler that exits after removing temp-file. */
void terminator2() 
  {
  if ((!pipeout) && (OutName != NULL) && (*OutName != '\0')) 
    (void) unlink(OutName);
  exit(1);
  }

void eject()
  {
  if (written)
    vfprintf(PSfile,"(%d)P\n",++pages), totpages++;
  if (guide)
   vfprintf(PSfile,"%%%%Page: - %d\n%s/page (%d) def /lines (%d) def doguide\n"
      ,++outpages,pagestr,pages,totlines), totpages++;
  }

void maybebeginpage()
  {
  if (!begun)
    vfprintf(PSfile,"%%%%Page: - %d\n%s1U S\n",++outpages,pagestr), begun = 1;
  }

void endpage()
  {
  if (++unit >= units)
   vfprintf(PSfile,"(%d)P\n",++pages), totpages++, written = begun = unit = 0;
  else vfprintf(PSfile,"R %d U S\n",unit+1);
  lines = 0;
  }

void maybebeginline()
  {
  if (!prevlen)
    if (numberlines) { outlen += 5;
      if (wrapped) fprints(PSfile,"(     ");
      else vfprintf(PSfile,(totlines > 999) ? "(%d " : "( %03d ",totlines + 1);
    }
    else fprintc(PSfile,'('), outlen++;
  wrapped = 0;
  }

void endline()
  {
  if (!prevlen && numberlines) maybebeginline(), prevlen++;
  if (prevlen)
    fprints(PSfile,")l\n"), written = 1;
  else fprints(PSfile,"nl\n");
  if (!wrapped) totlines++;
  outlines++; lines++; prevlen = outlen = 0; begun = 1;
  if (lines >= pagelength) endpage();
  }

int charcpyps(d,c) /* copy c to string s but convert to octal if needed */
register char *d,c;
  {
  if (c > 127) vsprintf(d,"\\%o",c);
  else if ((c < 32) && (c >= 0))
    if (showctrl)
      if (c == 28) (void) strcpy(d,"^\\\\"), prevlen++;
      else vsprintf(d,"^%c",c + 64), prevlen++, zapcnt++;
    else vsprintf(d,"\\%03o",c), zapcnt++;
  else if (c < 0) vsprintf(d,"\\%o",256 + c);
  else if (c == '(') (void) strcpy(d,"\\(");
  else if (c == ')') (void) strcpy(d,"\\)");
  else if (c == '\\') (void) strcpy(d,"\\\\");
  else { *d = c; *(d + 1) = '\0'; }
  return strlen(d);
  }

char *strcpyps(d,s) /* copy string s to d but convert 32>codes>127 to octal */
register char *d,*s;
  {
  register char *dc;
  dc = d; *d = '\0';
  while (*s) dc += charcpyps(dc,*s++);
  return d;
  }

int getnextc(infile,next)
FILE *infile;
char *next;
  {
  if (!charsleft) charsread = charsleft = read(fileno(infile),buf,sizeof buf);
  if (charsleft < 0) charsleft = 0;
  if (charsleft) {
    *next = *(buf + charsread - charsleft--);
    return(TRUE);
    }
  else return(FALSE);
  }

/* codefile - read inputfile and output it as compact PostScript to PSfile */
void codefile(inputfile)
FILE *inputfile;
  {
  char c, d[6];
  int i, temp, kill = 0;

  prevlen = outlen = wrapped = 0;
  while (getnextc(inputfile,&c))
    {
    if (outlen >= LINELENGTHOUT) {
      if (prevlen) fprints(PSfile,"\\\n");
      else fprintc(PSfile,'\n');
      outlen = 0;
    }
    switch(c)
      {
      case '\t':
        if (!kill)
          {
          maybebeginpage(); maybebeginline();
	  temp = tablength - (prevlen % tablength);
          prevlen += temp;
          if (!wraplength || (prevlen <= wraplength - startcolumn))
	    if (movtab)
              vfprintf(PSfile,")%d T(",prevlen + startcolumn), outlen += 7;
	    else
	    {
            for(i = 1; i <= temp; i++) fprintc(PSfile,' ');
	    outlen += temp;
    	    }
          else while (wraplength && (prevlen > wraplength - startcolumn))
            {
            temp = prevlen;
            wrapped = 1; endline(); maybebeginpage(); maybebeginline();
            prevlen = temp - wraplength + startcolumn;
	    if (movtab)
              vfprintf(PSfile,")%d T(",prevlen + startcolumn), outlen += 7;
	    else
              {
              for(i = wraplength + 1; i <= temp; i++) fprintc(PSfile,' ');
              outlen += temp - wraplength;
              }
            }
          }
        break;
      case '\b':
        if (prevlen && !kill)
          vfprintf(PSfile,")B%c",--prevlen ? '(' : ' '), outlen += 3;
        break;
      case '\f':
        if (!filterformfeed)
          {
          if (prevlen) fprints(PSfile,")s\n"), wrapped = 1;
          else maybebeginpage();
          endpage();
          prevlen = outlen = kill = 0;
          }
        break;
      case '\r':
        if (prevlen) fprints(PSfile,")CR\n"), written = 1;
        prevlen = outlen = kill = 0;
        break;
      case '\n':
        maybebeginpage();
        endline(); kill = 0;
        break;
      default:
        if (killength && (prevlen >= killength))
          kill = TRUE;
        else
          {
          if (wraplength && (prevlen >= wraplength - startcolumn))
            wrapped = 1, endline();
          maybebeginpage();
          maybebeginline();
          outlen += charcpyps(d,c);
          prevlen++;
          fprints(PSfile,d);
          }
        break;
      }
    }
    if (prevlen) endline();
    charsleft = 0; /* play it safe */
  }

/* cp - copy from file filename to stream stm */
cp(filename, stm)
char *filename;
register FILE *stm;
  {
  int ifp,ofp;
  register int count,lcount;
/*  char buf[BUFSIZ];  use buf for getnextc */

  (void) fflush(stm);
  ofp = fileno(stm);
  if ((ifp = open(filename, O_RDONLY, 0)) < 0)
    return(-1);
  lcount = 0;
  while ((count = read(ifp,buf,sizeof buf)) > 0)
    if (write(ofp,buf,lcount = count) != count)
      return(-2);
  if ((lcount) && (*(buf + lcount -1) != '\n')) /* avoid losing last line in */
    fprintc(stm,'\n');                       /* files not ended with newline */
  if (count < 0) return (-1);
  (void) close(ifp);
  (void) fflush(stm);
  return(0);
  }

void insertxfile()
  {
  while (xfiles)
    {
    if (strchr(strcpy(miscstr,xfile[--xfiles]),'.') == NULL)
      vstrcat(miscstr,".ps");
    if (cp(miscstr,PSfile))
      if (cp(strcat(strcat(strcpy(tempstr,libdir),"/"),miscstr),PSfile))
        vfprintf(stderr,"%s%s: cannot copy -x file %s\n",NL,prog,tempstr);
    }
  }

/* makePSfile - create output PSfile and print header. Dump prologue file
                into it followed by date and user info.    */
void makePSfile(FileName)
char *FileName;
  {
  long laban;
  if (noprint) 
    PSfile = pipeout ? stdout : fopen(OutName,"w");
  else 
    {
    (void) mktemp(strcat(strcpy(OutName,tempdir),LPPTMP));
    (void) umask(UMASK);
    PSfile = fopen(OutName,"w");
    }
  if (PSfile == NULL) 
    {
    vfprintf(stderr,"%s: cannot create output file %s\n",prog,OutName);
    exit(1);
    }
#ifdef LANG
  (void) strcpy(lc_c, "C");
  lc_time = setlocale(LC_TIME, "");
  if (lc_time == NULL) lc_time = lc_c;
  (void) time(&ltid);
  tid = localtime(&ltid);
  dtcp = localdtconv();
  (void) strftime(miscstr,0,"",NULL);   /* Dummy call */
  (void) strftime(miscstr,sizeof miscstr-1,dtcp->dtime_format,tid);
  lc_ctype = setlocale(LC_CTYPE, "");
  if (lc_ctype == NULL) lc_ctype = lc_c;
#else /* LANG */
  *miscstr = '\0';
  laban = time((long *) 0);
  (void) sscanf(ctime(&laban),"%[^\n]",miscstr);
#endif /* LANG */
  vfprintf(PSfile,
"%%!PS-Adobe-3.0\n\
%%%%Title: %s\n\
%%%%Creator: %s v3.44\n\
%%%%CreationDate: %s\n\
%%%%DocumentNeededResources: font Courier Times-Bold Helvetica%s\n\
%%%%BoundingBox: %s\n\
%%%%Pages: (atend)\n\
%%%%EndComments\n",FileName,prog,miscstr,pagestr,
    letter ? "24 24 587 768" : "24 24 570 818");
  if (cp(strcat(strcpy(tempstr,libdir),LPPPROLOG),PSfile))
    vfprintf(stderr,"%s: cannot copy prologue file %s%s\n",
      prog,libdir,LPPPROLOG),
    terminator2();
#ifdef LANG
  if (strncmp(lc_ctype,"Sw",2) == 0 || strncmp(lc_ctype,"sv",2) == 0)
    vfprintf(PSfile,"swedishdate\n%s\n",altlayout ?
      "/dtext (K|ad ) def /utext (av ) def /wtext (Senast {ndrad ) def" :
      "/dtext (Skickad: ) def /utext (K|ad av: ) def /wtext ([ndrad : ) def");
#endif /* LANG */
  vfprintf(PSfile,"/date (%s) def\n",strcpyps(tempstr,miscstr));
  if (!nouser)
    {
    char *p;
    struct passwd *pswd;          /* information about the user */
#ifdef USEUNAME
    struct utsname name;
    uname(&name);
    strcpy(miscstr, name.nodename);
#else
    (void) gethostname(miscstr,sizeof miscstr);
#endif
    pswd = getpwuid((int) getuid());
    if ((p = strchr(pswd->pw_gecos,',')) != NULL) *p = '\0'; /* first item */
    vfprintf(PSfile,"/user (%s <",strcpyps(tempstr,pswd->pw_gecos));
    fprints(PSfile,strcpyps(tempstr,pswd->pw_name));
    vfprintf(PSfile,"@%s>) def\n",strcpyps(tempstr,miscstr));
    }
  if (letter) fprints(PSfile,"letter ");
  if (altlayout) fprints(PSfile,"altlayout ");
  fprints(PSfile,"4:\n");                          /* default format */
  if (doubleside) fprints(PSfile,"statusdict /setduplexmode known\
 { statusdict begin true setduplexmode end } if\n");
  insertxfile();
  fprints(PSfile,"%%EndSetup\n");
  }

int filetest(file) /* returns nonzero if bad file */
char *file;
{
  struct stat statb;
#ifndef NOMAGICK
  struct exec execb;
#endif /* NOMAGICK */
  register int fd;
  if (access(file,4) < 0) {
    vfprintf(stderr,"%s%s: cannot access %s\n",NL,prog,file); return(-1);
  }
  if (stat(file,&statb) < 0) {
    vfprintf(stderr,"%s%s: cannot stat %s\n",NL,prog,file); return(-1);
  }
  if ((statb.st_mode & S_IFMT) == S_IFDIR) {
    vfprintf(stderr,"%s%s: %s is a directory.\n",NL,prog,file); return(-1);
  }
  if ((fd = open(file,O_RDONLY)) < 0) {
    vfprintf(stderr,"%s%s: cannot open %s\n",NL,prog,file); return(-1);
  }
#ifndef NOMAGICK
  if (read(fd,(char *) &execb,sizeof(execb)) == sizeof(execb))
    switch(execb.a_magic) {
    case A_MAGIC1:
    case A_MAGIC2:
    case A_MAGIC3:
#ifdef A_MAGIC4
    case A_MAGIC4:
#endif
      vfprintf(stderr,"%s%s: %s is an executable program and is unprintable.\n"
        ,NL,prog,file);
      (void) close(fd);
      return(-1);
    case ARMAG:
      vfprintf(stderr,"%s%s: %s is an unprintable archive file.\n",NL,prog,
        file);
      (void) close(fd);
      return(-1);
    }
#endif /* NOMAGICK */
  (void) close(fd);
  return(0);
}

/* dofile - Convert file FileName (or stdin if NULL) to PostScript on PSfile.
 *          If this is the first file then PSfile is properly
 *          initialized and PS prologue and setup is sent ahead of the file */
void dofile(FileName)
char *FileName;
  {
  struct stat filestat;
  long laban;
  FILE *infile;
  int isstdin;

  if (!(isstdin = (FileName == STDIN)))
    if (filetest(FileName)) terminator2();
    else (void) fstat(fileno(infile = fopen(FileName,"r")),&filestat);
  if (!files) makePSfile(isstdin ? "(stdin)" : FileName);
  if (append && files)
    {
    if (extraff) endpage();
    append = extraff = FALSE;
    fprints(PSfile,flagstr);
    }
  else
    {
    eject();
    guide = nextguide; nextguide = FALSE;
    *pagestr = '\0';
    vstrcat(pagestr,"J\n");
    pages = totlines = outlines = lines = written = begun = unit = zapcnt = 0;
    if (!isstdin)
      {
      vsprintf(ends(pagestr),"/title (%s) def\n",strcpyps(tempstr,FileName));
      laban = filestat.st_mtime;
      *miscstr = '\0';
#ifdef LANG
      (void) strftime(miscstr,sizeof miscstr-1,dtcp->dtime_format,
        localtime(&laban));
#else
      (void) sscanf(ctime(&laban),"%[^\n]",miscstr);
#endif /* LANG */
      vsprintf(ends(pagestr),"/wdate (%s) def\n",strcpyps(tempstr,miscstr));
      }
    if (*flagstr) vstrcat(pagestr,flagstr); /* fprintf(PSfile,flagstr); */
    }
/*  *flagstr='\0'; should be sent for all files */
  if (xfiles) { maybebeginpage(); insertxfile(); }
  if (!quiet) fprints(stderr,files ? "\n[" : "[");
  if (isstdin) codefile(stdin);
  else
    {
    if (!quiet) vfprintf(stderr,"%s ",FileName);
    codefile(infile); (void) fclose(infile);
    }
  if (!quiet)
    {
    vfprintf(stderr,"%d line%s ",outlines,outlines != 1 ? "s" : "");
    if (outlines != totlines) vfprintf(stderr,"(%d wrapped) ",
      outlines - totlines);
    vfprintf(stderr,"%d page%s] ",
      pages + written,(pages + written) != 1 ? "s" : "");
    }
  files++;
  if (zapat && (zapcnt >= zapat))
     vfprintf(stderr,"%s%s: too many unprintable characters. Is this really\
 a text file?\nForce printing with option -z0.\n",NL,prog), terminator2();
  }

void argtest(match,flag)
int match,flag;
  {
  if (!match)
    vfprintf(stderr,"%s%s: bad argument to option -%c.\n",NL,prog,flag),
      terminator2();
  }
  
void parseflags(ac,av)
int ac;
char **av;
  {
  int argp,tmp,tmp2;
  float ftmp, ftmp2;
  while ((argp = getopts(ac,av,ARGS)) != EOF)
    switch (argp)
      {
    case '#':
      argtest(sscanf(optarg,"%d",&copies),argp);
      vsprintf(ends(flagstr),"/#copies %s def\n",optarg);
      break;
    case '0':
      nullout = TRUE; break;
    case '2':
      doubleside = TRUE; break;
    case '4':
      vstrcat(flagstr,"4: ");
      pagelength = letter ? 67 : 72; fontsize = baselineskip = 10;
      wraplength = letter ? 85 : 82; units = 1; landscape = FALSE; break;
    case '5':
      vstrcat(flagstr,"5: ");
      pagelength = letter ? 73 : 66; wraplength = 82; units = 2;
      fontsize = baselineskip = letter ? 7.0 : 7.5; landscape = TRUE; break;
    case '6':
      vstrcat(flagstr,"6: ");
      pagelength = letter ? 66 : 72; fontsize = baselineskip = 5;
      wraplength = 82; units = 4; landscape = FALSE; break;
    case 'A':
      if (append) extraff = TRUE;
      append = TRUE; break;
    case 'a':
      if (files)
        vfprintf(stderr,"%s%s: option -a ignored after file.\n",NL,prog);
      else {
        letter = TRUE;
        if (pagelength == 72) { pagelength = 67; wraplength = 85; }
        }
      break;
    case 'B':
      vstrcat(flagstr,"/border? true def\n"); break;
    case 'b':
      vstrcat(flagstr,"/border? false def\n"); break;
    case 'C':
      vsprintf(ends(flagstr),"/comment (%s) def\n",strcpyps(tempstr,optarg));
      break;
    case 'c':
      tmp = units;
      if (2 == sscanf(optarg,"%d,%d",&units,&tmp2))
        ftmp2 = (int) 72.0 / 25.4 * tmp2;
      else if (1 == sscanf(optarg,"%d",&units)) ftmp2 = 14.2; /* default 5mm */
      else argtest(0,argp);
      tmp = (tmp == 1) ? wraplength :
        (letter ? (landscape ? 120 : 85) : (landscape ? 128 : 82));
      ftmp = ((float) tmp * 6 + ftmp2) / units;
      wraplength = (int) (ftmp - ftmp2) / 6 * 10 / fontsize;
      for(tmp = 0; tmp < units; tmp++)
        vsprintf(ends(flagstr),"%.1f c: ",tmp * ftmp);
      vsprintf(ends(flagstr),"%d %.1f %d %d %d format\n",units,fontsize,
        landscape ? 8 : 0,wraplength,pagelength);
      break;
    case 'D':
      vsprintf(ends(flagstr),"/date (%s) def\n",strcpyps(tempstr,optarg));
      break;
    case 'd':
      vstrcat(flagstr,"/date? false def\n"); break;
    case 'e':
      if (2 != sscanf(optarg,"%d=%s",&tmp,tempstr)) argtest(0,argp);
      vsprintf(ends(flagstr),"Encoding %d /%s put\n",tmp,tempstr);
      break;
    case 'F':
      vsprintf(ends(flagstr),"/%s F:\n",optarg);
      if (!files) vstrcat(strcat(pagestr," "),optarg);/* for %%Documentfonts */
      break;
    case 'f':
      vsprintf(ends(flagstr),"%s f: ",optarg);
      argtest(sscanf(optarg,"%f",&ftmp),argp);
      wraplength = wraplength * fontsize / ftmp;
      pagelength = pagelength * baselineskip / ftmp;
      baselineskip = fontsize = ftmp;
      if (report) vfprintf(stderr,
        "%s: %.1f pt font gives %d chars/line and %d lines/page.\n",prog,
        fontsize,wraplength,pagelength,wraplength);
      break;
    case 'G':
      nextguide = TRUE; break;
    case 'H':
      vsprintf(ends(flagstr),"/%s H:\n",optarg);
      if (!files) vstrcat(strcat(pagestr," "),optarg);/* for %%Documentfonts */
      break;
    case 'h': /* Help wanted */
      if (cp(strcat(strcpy(tempstr,libdir),LPPHELP),stdout))
        vfprintf(stderr,"%s: cannot open helpfile %s%s\n",prog,libdir,LPPHELP);
      exit(0);
      break;
    case 'I':
      vstrcat(flagstr,"/info? true def /border? true def\n"); break;
    case 'i':
      vstrcat(flagstr,"/info? false def /border? false def\n"); break;
    case 'K':
      argtest(sscanf(optarg,"%d",&killength),argp); break;
    case 'k':
      argtest(sscanf(optarg,"%d",&tmp),argp);
      ftmp = (float) tmp / (float) wraplength;
      ftmp2 = (float) wraplength / (float) tmp;
      pagelength = pagelength * ftmp;
      fontsize = fontsize * ftmp2;
      baselineskip = fontsize;
      wraplength = tmp;
      vsprintf(ends(flagstr),"%.1f f:\n",fontsize);
      if (report) vfprintf(stderr,
        "%s: %d chars/line gives %d lines/page and %.1f pt font.\n",prog,
        wraplength,pagelength,fontsize);
      break;
    case 'L':
      vstrcat(flagstr,"L: ");
      pagelength = letter ? 49 : 48; fontsize = baselineskip = 10;
      wraplength = letter ? 120 : 128; units = 1; landscape = TRUE; break;
    case 'l':
      argtest(sscanf(optarg,"%d",&pagelength),argp); break;
    case 'M':
      if (2 == sscanf(optarg,"%d,%d",&tmp,&tmp2))
        vsprintf(ends(flagstr),"%d %d M:\n",tmp,tmp2);
      else if (1 == sscanf(optarg,"%d",&tmp))
        vsprintf(ends(flagstr),"%d 0 M:\n",tmp);
      else argtest(0,argp);
      break;
    case 'N':
      numberlines = TRUE; startcolumn = 5; break;
    case 'n':
      if (files)
        vfprintf(stderr,"%s%s: option -n ignored after file.\n",NL,prog);
      else altlayout = TRUE;
      break;
    case 'O':
      argtest(sscanf(optarg,"%d",&wraplength),argp); break;
    case 'o':
      {
      int i,c[18];
      if (5 != sscanf(optarg,"%d %d %d %f %d",
        &units,&pagelength,&wraplength,&fontsize,&tmp)) argtest(0,argp);
      landscape = tmp & 8;
      if ((units*2+5) != sscanf(optarg,"%d %d %d %f %d %d %d %d %d %d %d\
 %d %d %d %d %d %d %d %d %d %d %d %d",&tmp2,&tmp2,&tmp2,&ftmp,&tmp2,
        &c[0],&c[1],&c[2],&c[3],&c[4],&c[5],&c[6],&c[7],&c[8],&c[9],&c[10],
        &c[11],&c[12],&c[13],&c[14],&c[15],&c[16],&c[17])) argtest(0,argp);
      for(i=0; i<(2*units); i++) vsprintf(ends(flagstr),"%d ",c[i]);
      vsprintf(ends(flagstr),"%d %.1f %d %d %d format\n",units,fontsize,
        tmp,wraplength,pagelength);
      }
      break;
    case 'P':
      PrinterName = optarg; break;
    case 'p':
      if (files)
        vfprintf(stderr,"%s%s: cannot change output after file argument.\n",
          NL,prog), terminator2();
      noprint = TRUE;
      if (!strcmp(strcpy(OutName,optarg),"-")) pipeout = TRUE;
      break;
    case 'q':
      quiet = TRUE; break;
    case 'R':
      report = TRUE; break;
    case 'r':
      vstrcat(flagstr,"r: "); break;
    case 'S':
      vsprintf(ends(flagstr),"%cS: ",swedish ? 'S' : ' '); swedish = TRUE;
      break;
    case 's':
      vstrcat(flagstr,"s: "); swedish = FALSE; break;
    case 'T':
      vsprintf(ends(flagstr),"/title (%s) def\n",strcpyps(tempstr,optarg));
      break;
    case 't':
      if (!strcmp(optarg,"s")) movtab = 0;
      else if (!strcmp(optarg,"x")) movtab = 1;
      else argtest(sscanf(optarg,"%d",&tablength),argp);
      break;
    case 'U':
      nouser = TRUE;
      vsprintf(ends(flagstr),"/user (%s) def\n",strcpyps(tempstr,optarg));
      break;
    case 'u':
      nouser = TRUE;
      vstrcat(flagstr,"/user () def\n"); break;
    case 'V':
      vstrcat(flagstr,"V: ");
      pagelength = letter ? 140 : 150; fontsize = baselineskip = 5;
      wraplength = 82; units = 2; landscape = FALSE; break;
    case 'W':
      vsprintf(ends(flagstr),"/wdate (%s) def\n",strcpyps(tempstr,optarg));
      break;
    case 'w':
      argtest(sscanf(optarg,"%d",&tmp),argp);
      baselineskip = pagelength * baselineskip / tmp;
      pagelength = tmp;
      wraplength = wraplength * fontsize / baselineskip;
      fontsize = baselineskip;
      vsprintf(ends(flagstr),"%.1f f:\n",fontsize);
      if (report) vfprintf(stderr,
        "%s: %d lines/page gives %d chars/line and %.1f pt font.\n",prog,
        pagelength,wraplength,fontsize);
      break;
    case 'X':
      vsprintf(ends(flagstr),"\n%s\n",optarg); break;
    case 'x':
      xfile[xfiles++] = optarg; break;
    case 'Y':
      vsprintf(ends(flagstr),"/Y %s def\n",optarg);
      argtest(sscanf(optarg,"%f",&ftmp),argp);
      pagelength = pagelength * baselineskip / ftmp; baselineskip = ftmp;
      if (report) vfprintf(stderr,
        "%s: %.1f pt baselineskip gives %d chars/line and %d lines/page.\n",
        prog,baselineskip,wraplength,pagelength);
      break;
    case 'z':
      /* filterformfeed = TRUE; break; */
      argtest(sscanf(optarg,"%d",&zapat),argp); break;
    case 'Z':
      showctrl = TRUE; break;
    case '?': /* Bad option */
      vfprintf(stderr,"Usage: %s %s\n",prog,USAGE);
      terminator2();
    default:
      break;
      }
  }

/* envget - a getenv. Returns the value from an environment variable unless it
 *          doesn't exist or has the value NULL, then it returns NULL.      */
char *envget(var)
char *var;
  {
  register char *val;
  if (((val = getenv(var)) == NULL) || (*val == '\0'))
    return ((char *) NULL);
  else return (val);
  }

/* parseenv -  parse args in environment variable LPP  */
void parseenv()
  {
  register char *p;

  if (p = envget("LPP"))
    {
    static char quote='\0';
    static int found=2;
    int got,ins;

    eargc = 1;
    while (*p)
      {
      ins = FALSE;
      if (found == 2)
        {
        while (*p == ' ') p++;
        eargv[eargc] = p;
        ins = TRUE;
        }
      for (got=FALSE, found=FALSE; (!found) && *p; got=TRUE, p++)
        {
        if (quote && (*p == quote)) found=1;
        if (!quote && ((*p == '"') || (*p == '\''))) found=1;
        if (!quote && (*p == ' ')) found=2;
        if (found) break;
        }
      if (got && ins) eargc++;
      if (found == 1)
        {
        quote = quote ? '\0' : *p; (void) strcpy(p,p+1);
        }
      else if (found == 2) *p++ = '\0';
      }
    if (quote)
      {
      vfprintf(stderr,"%s: unmatched %c in environment variable LPP \"%s\"\n",
        prog, quote, envget("LPP"));
      exit(1);
      }
    parseflags(eargc, eargv);
    if (eargc != getoptindex)
      {
      vfprintf(stderr,"%s: bad syntax in environment variable LPP \"%s\"\n",
        prog, envget("LPP"));
      exit(1);
      }
    }
  }

/* addarg - add argument to argv for the spooler */
void addarg(argv,argstr,argc)
char **argv;
char *argstr;
int *argc;
  {
  argv[(*argc)++] =  strcpy((char *) malloc((unsigned) (strlen(argstr) + 1)),
    argstr);
  argv[*argc] = '\0';
  }

/* flushPS - close the temporary PS file and maybe spool it.
             Number of pages and name of used printer or name of outfile is
             printed except when the 'quiet-flag' (-q) has been found or
             output is to stdout. */
void flushPS()
  {
  eject();         /* output last page if anything written on it */
  vfprintf(PSfile,
    "%%%%Trailer\n%%%%Pages: %d\n(%d page%s.\\n) print\n",
    outpages,totpages,totpages != 1 ? "s" : "");
  (void) fclose(PSfile);
  if (!quiet && (files > 1 || copies != 1))
    vfprintf(stderr,"\n%d page%s ",totpages*copies,
      totpages*copies != 1 ? "s" : "");
  if (nullout || !totpages || pipeout) fprintc(stderr,'\n');
  if (nullout || !totpages) (void) unlink(OutName);
  else if (noprint && !quiet && !pipeout)
    vfprintf(stderr,"stored in file %s\n",OutName);
  else if (!noprint)
    {
    eargc = 0;
    addarg(eargv,LPCOMMAND,&eargc);
    if (PrinterName == NULL) PrinterName = envget("PRINTER");
    if (PrinterName != NULL)
      {
      vsprintf(tempstr,LPDEVICE,PrinterName);
      addarg(eargv,tempstr,&eargc);
      }
    if (!quiet) vfprintf(stderr,"spooled to %s\n",
      PrinterName == NULL ? "(default)" : PrinterName);
#ifdef BSDF
    addarg(eargv,"-r",&eargc);
#else
    addarg(eargv,"-ops",&eargc);
    addarg(eargv,"-s",&eargc);
#endif /* BSD */
    addarg(eargv,OutName,&eargc);
    if (!fork())
      {
      (void) execvp(LPCOMMAND,eargv);
      vfprintf(stderr,"%s%s: cannot exec %s.\n",NL,prog,LPCOMMAND);
      exit(1);
      }
    }
  }

main(argc,argv)
int argc;
char *argv[];
  {
  char *p;
  
  prog = *eargv = *argv;                      /* name program is called by */
  if (signal(SIGINT,(void (*)())terminator2) == SIG_IGN)
    {
    (void) signal(SIGINT,SIG_IGN);
    (void) signal(SIGQUIT,SIG_IGN);
    (void) signal(SIGHUP,SIG_IGN);
    (void) signal(SIGTERM,SIG_IGN);
    }
  else
    {
    (void) signal(SIGQUIT,(void (*)())terminator2);
    (void) signal(SIGHUP, (void (*)())terminator2);
    (void) signal(SIGTERM,(void (*)())terminator2);
    }
  if ((libdir = envget("PSLIBDIR")) == NULL) libdir = LIBDIR;
  if ((tempdir = envget("PSTEMPDIR")) == NULL) tempdir = TMPDIR;
  parseenv();			/* parse flags in envariable LPP */
  getoptindex = 1;		/* reset getopt global, then  */
  parseflags(argc,argv);	/* parse options at start of commandline */
  for (; getoptindex < argc ; getoptindex++)     /* loop through commandline */
    if (strlen(p = argv[getoptindex]))
      if (*p == '-')	                  /* starts with '-', is it options? */
        if (strlen(p) == 1) dofile(STDIN);     /* if single '-' : read stdin */
        else
  	  {
          getoptp = ++p;
	  parseflags(argc,argv);
	  getoptindex--;                     /* don't loose arg after option */
	  }
      else dofile(p);
  if (!files) dofile(STDIN);         /* no file arguments, so implicit stdin */
  flushPS();
  return(0);
  }
