// -*- C++ -*-
/**
 * \file ControlDocument.C
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Edwin Leuven
 *
 * Full author contact details are available in file CREDITS
 */

#include <config.h>

#ifdef __GNUG__
#pragma implementation
#endif

#include "BufferView.h"
#include "ControlDocument.h"
#include "ViewBase.h"

#include "gettext.h"
#include "lyxfind.h"

#include "buffer.h"
#include "language.h"
#include "lyx_main.h"
#include "lyxtextclass.h"
#include "lyxtextclasslist.h"
#include "CutAndPaste.h"

#include "frontends/LyXView.h"
#include "frontends/Alert.h"

#include "support/lstrings.h"
#include "support/filetools.h"

#include "BoostFormat.h"

using std::endl;


ControlDocument::ControlDocument(LyXView & lv, Dialogs & d)
	: ControlDialogBD(lv, d), bp_(0)
{
}

ControlDocument::~ControlDocument()
{}


BufferParams & ControlDocument::params()
{
	lyx::Assert(bp_.get());
	return *bp_;
}


LyXTextClass ControlDocument::textClass()
{
	return textclasslist[bp_->textclass];
}


void ControlDocument::apply()
{
	if (!bufferIsAvailable())
		return;

	classApply();

	view().apply();
	setLanguage();
	buffer()->params = *bp_;

	lv_.view()->redoCurrentBuffer();

	buffer()->markDirty();

	lv_.message(_("Document settings applied"));
}


void ControlDocument::setParams()
{
	if (!bp_.get())
		bp_.reset(new BufferParams());

	/// Set the buffer parameters
	*bp_ = buffer()->params;
}


void ControlDocument::setLanguage()
{
	Language const * oldL = buffer()->params.language;
	Language const * newL = bp_->language;

	if (oldL != newL
	    && oldL->RightToLeft() == newL->RightToLeft()
	    && !lv_.buffer()->isMultiLingual())
		lv_.buffer()->changeLanguage(oldL, newL);
}


void ControlDocument::classApply()
{
	BufferParams & params = buffer()->params;
	lyx::textclass_type const old_class = params.textclass;
	lyx::textclass_type const new_class = bp_->textclass;

	// exit if nothing changes or if unable to load the new class
	if (new_class == old_class || !loadTextclass(new_class))
		return;

	// successfully loaded
	view().apply();
	buffer()->params = *bp_;

	lv_.message(_("Converting document to new document class..."));
	int ret = CutAndPaste::SwitchLayoutsBetweenClasses(
		old_class, new_class,
		&*(lv_.buffer()->paragraphs.begin()),
		lv_.buffer()->params);
	if (ret) {
		string s;
		if (ret == 1) {
			s = _("One paragraph couldn't be converted");
		} else {
#if USE_BOOST_FORMAT
			boost::format fmt(_("%1$s paragraphs couldn't be converted"));
			fmt % ret;
			s = fmt.str();
#else
			s += tostr(ret);
			s += _(" paragraphs couldn't be converted");
#endif
		}
		Alert::alert(_("Conversion Errors!"),s,
			     _("into chosen document class"));
	}
}


bool ControlDocument::loadTextclass(lyx::textclass_type tc) const
{
	bool const success = textclasslist[tc].load();
	if (!success) {
		// problem changing class
		// -- warn user (to retain old style)
		Alert::alert(_("Conversion Errors!"),
			     _("Errors loading new document class."),
			     _("Reverting to original document class."));
	}
	return success;
}


void ControlDocument::saveAsDefault()
{
	if (!Alert::askQuestion(_("Do you want to save the current settings"),
				_("for the document layout as default?"),
				_("(they will be valid for any new document)")))
		return;

	lv_.buffer()->params.preamble = bp_->preamble;

	string const fname = AddName(AddPath(user_lyxdir, "templates/"),
				     "defaults.lyx");
	Buffer defaults(fname);
	defaults.params = params();

	// add an empty paragraph. Is this enough?
	Paragraph * par = new Paragraph;
	par->layout(params().getLyXTextClass().defaultLayout());
	defaults.paragraphs.set(par);

	defaults.writeFile(defaults.fileName());

}
