-- Tutorial:Binomial Ideals
-- 
--
-- In this demo, we illustrate Macaulay2
-- by investigating the regularity
-- of ideals generated by three binomials.
-- General theory says that the regularity of
-- such an ideal can be as bad as double exponential
-- in the number of variables.  The Mayr-Meyer
-- examples are the prototype for ideals having
-- this high regularity.  However, other than
-- these complicated ideals, few examples having 
-- regularity worse than a complete intersection 
-- are known.  So our plan here is to try to find
-- interesting examples, and then analyze one such.
--
-- To start, define a ring with 8 variables
R = ZZ/32003[a..h]

-- Our plan is to generate binomials randomly.
--
-- A function which produces a random
-- binomial of degree d in the given ring $R$:
--$
genbinom = (R, d) -> (
    n := numgens R;
    product(d, i -> R_(random n)) 
      - product(d, i -> R_(random n)))

-- For example, a random binomial cubic:
genbinom(R,3)

-- Below is a function to produce an ideal in $R$
-- generated by $3$ random degree $d$ binomials.
-- Instead of producing an ideal, we have the routine
-- return a (cokernel) module.  The cokernel will
-- be more useful later.
--$
genideal = (R, d) ->
    coker matrix{{genbinom(R,d), 
           genbinom(R,d), 
           genbinom(R,d)}}

-- The resolution of one such ideal
M = genideal(R,5)
C = res M

-- View all of the maps in the resolution:
C.dd

-- Most ideals that we produce randomly will not
-- be particularly interesting.  We wish to filter out the 
-- undesirables.  The routine
-- below is our filter: it returns true only if
-- the ideal has codimension $2$, and its
-- regularity is at least that of a complete 
-- intersection ($3d-3$).
--$
isgood = (I,d) -> (
     codim I == 2
   and
     regularity I >= 3*d-3
     )

-- Most ideals will not be ``good'':
I = genideal(R,3)
isgood(I,3)
I

-- We now compute lots of these binomials and
-- only keep the ones of interest.
-- The following routine {\tt bunch} computes {\tt reps}
-- number of random ideals.  It returns a list
-- consisting
-- of ideals and the value {\tt null}.
--$
bunch = (R,d,reps) -> (
    apply(reps, i -> (
	    I := genideal(R,d);
	    if isgood(I,d) 
              then I 
              else null
            )
    ))

-- A set of $200$ examples.
II = bunch(R,4,200)

-- We use the {\tt select} command to prune out the null's.
II = select(II, i->i =!= null)

-- Notice that there are very few that satisfy all
-- of our criteria, but that the first one has regularity
-- $17$, which is two higher than the 
-- regularity of a complete intersection:
apply(II, I -> regularity I)

-- The betti numbers of the resolution of the first ideal:
C = res (II#0)
betti C

-- There is a syzygy with entries of degree $13$, 
-- (the generators of the ideal have degree $6$):
(C.dd_2)_{7}

-- We can do this all at once with
--$
bunch = (R,d,reps) -> (
    II := apply(reps, i -> (
	    I := genideal(R,d);
	    if isgood(I,d) 
              then I 
              else null));
    select(II, i -> i =!= null)
    )

-- Let's try this:
I1 = bunch(R, 5, 100)
apply(I1, regularity)

----------------------------------------
-- Investigating Components
----------------------------------------

--
-- Our plan now is to investigate the ideal 
-- with regularity $17$.  Here it is again:
R = ZZ/32003[a..h]
--$
I = ideal(
  b*c^3*d*h-a*c*d*f*g*h, 
  -a^2*d*e*g^2+b^4*c*h, 
  -a*b*c^2*e^2+b*d^2*f^2*h)
-- and the corresponding module
M = coker gens I

-- For example, what are the components of $I$?
-- By general theory, any associated primes of 
-- codimension $d$ are exactly the associated
-- primes of $Ext^d(M,R)$ of codimension $d$ (this
-- is the top dimensional part of this module).

-- First let's compute the resolution
C = res M

-- So the possible codimensions for associated
-- components are 2, 3, or 4.

-- The codimension two part is the easiest to obtain:

E2 = Ext^2(M,R)
I2 = ann E2
rad2 = radical I2
degree rad2

-- We can decompose this ideal. {\tt decompose I}
-- returns a list of the isolated primes of $I$.
comps2 = decompose rad2

-- One way to see these one per line:
scan(comps2, print)

-- So, there are six codimension two associated primes.
--
-- There are components of codimension $4$:

E4 = Ext^4(M,R)
codim E4
I4 = ann E4
comps4 = decompose I4
scan(comps4, print)

-- There are $8$ components of codimension $4$:
#comps4

-- These must all be embedded components, since they
-- have codimension greater than the number of equations.
--
-- By the way, we could also use {\tt decompose} on the 
-- entire ideal:
isolateds = decompose I
scan(isolateds, print)

-- This routine returns all of the isolated components
-- of $I$. There are $6$ components of codimension $2$, and $7$ of
-- codimension $3$.
cods = apply(isolateds, i -> codim i)
tally cods
-- Intersecting all of these gives the radical
Irad = intersect toSequence isolateds

betti Irad

-- The codimension $3$ part is the hardest computationally.
-- We already have the isolated primes of codim $3$.
isolated3 = select(isolateds, i -> codim i == 3)
rad3 = intersect toSequence isolated3
I3 = ann Ext^3(M,R)
codim I3
-- We now remove the isolated components, leaving
-- only the embedded primes.
emb3 = saturate(I3, rad3)
embedded3 = decompose emb3
scan(embedded3, print)
#embedded3
-- There are therefore $13$ embedded components 
-- of codimension three.
--
-- Another way to find the component(s) not
-- supported on some coordinate hyperplane
use R  -- sets the variables a..h back to elements in R
Igood = saturate(I, a*b*c*d*e*f*g*h)

-- This is a prime ideal, since it agrees with the known prime
-- ideal, the last element in the list {\tt isolateds}:
good = isolateds#(-1)
good == Igood

-- So there are the following components:
--
-- codim 2: (isolated): 6 linear spaces
--
-- codim 3: (isolated): 6 linear spaces, 1 binomial
--
-- codim 3: (embedded): 12 linear, 1 binomial
--
-- codim 4: (embedded): 8 linear spaces
