Matrix % Ideal := (f,I) -> (
     if isHomogeneous f and isHomogeneous I then (
	  m := max degrees source f;
	  g := gb(I,DegreeLimit=>m);
          f % g)
     else f % gb I)

document { sagbi,
     TT "sagbi(F:RingMap, nsteps:ZZ) --> RingMap",
     BR,NOINDENT,
     "  -- yields a ring map where the images of the variables forms
a subalgebra basis (sagbi) of image F.",
     TT "sagbihom(F:Matrix, maxdeg:ZZ) --> Matrix",
     BR,NOINDENT,
     "  -- yields a matrix whose entries form
a subalgebra basis (sagbi) of the subalgebra generated by the entries of
the matrix F.",
     PARA,
     "Currently, the first routine is based on the Macaulay classic version
     written by D. Eisenbud.  We (HT,MES) hope to improve the performance
     this week! The second routine computes a sagbi basis degree by degree,
     for a graded subalgebra.",
     EXAMPLE ///
     ///,
     CAVEAT "If F : R <-- S, then R and S should be polynomial
     rings, and not quotient rings.",
     PARA,
     "References: Kapur, D., Madlener, K. (1989). A completion procedure
for computing a canonical basis of a $k$-subalgebra.
Proceedings of {\it Computers and Mathematics 89}
(eds. Kaltofen and Watt), MIT, Cambridge, June 1989.",
PARA,
"Robbiano, L., Sweedler, M. (1990). Subalgebra bases,
in W.~Bruns, A.~Simis (eds.): {\sl Commutative Algebra},
Springer Lecture Notes in Mathematics {\bf 1430}, pp.~61--87.",
PARA,
"F. Ollivier, Canonical Bases: Relations with Standard bases, finiteness
conditions and applications to tame automorphisms, in Effective Methods
in Algebraic Geometry, Castiglioncello 1990, pp. 379-400, Progress in Math.
{\bf 94} Birkhauser, Boston (1991)",
PARA,
"B. Sturmfels, Groebner bases and Convex Polytopes, Univ. Lecture Series 8,
Amer Math Soc, Providence, 1996"
     }

-- 'split' returns two matrices of the same size as m1.
-- The first has zero entries where m2 has non-zero entries
-- and has the entries of m1 otherwise.  Similarly, the second
-- matrix returned has zero entries whenever m2 has zero enries, and
-- has the entries of m1 otherwise.

-- takes a one row matrix m of lead terms and returns in the ith column
-- the exponent vector of the ith monomial of m



split = (m1, m2) -> (
     map(target m1, source m1, (i,j) -> 
	  if m2_(i,j) == 0
	  then m1_(i,j)
	  else 0))

gbIsDone = (m) -> (
     -- only checks whether a 'non-syzygy' GB has completed
     m#?{false,0} and 
     (m#{false,0}).returnCode === 0)

autosubduction0 = (m, F, J, d) -> (
     -- m is the matrix whose entries we wish to reduce (subduct),
     -- these elements are all in the same degree d.
     -- F is the ringmap whose entries we may use to reduce
     -- J is the ideal (y_i - in(f_i)), which has a GB computed
     -- in degrees <= d.
     R := target F; -- also ring m
     S := source F;
     RS := ring J;
     RStoS = map(S, RS, (map(S^1, S^(elements(numgens R:-1)), 0)) | vars S);
     RtoRS = map(RS, R, (vars ring J)_{0..numgens R-1});
     gbJ := gb(J, DegreeLimit=>d);
     reduced = map(target m, source m, 0);
     while m != 0 do (
	 m = matrix entries m;  -- to fix degrees
         errorterm = leadTerm m;
         errorterm1 = RtoRS errorterm;
         errorterm2 = errorterm1 % gbJ;
         errorterm3 = RStoS errorterm2;
	 mm = split(m, errorterm3);
	 h := leadTerm(mm);
	 reduced = reduced + h;
	 m = m - h - F errorterm3;
	 );
     reduced = compress reduced;
     reduced = matrix entries reduced; -- fix degrees
     reduced
     )

TEST ///
R = ZZ/101[a,b,c]
S = ZZ/101[x,y,z,Degrees=>{2,2,2}]
RS = ZZ/101[a,b,c,x,y,z,Degrees=>{1,1,1,2,2,2}]
J = ideal(x-a^2, y-a*b, z-b^2)
gb(J, DegreeLimit=>4)
use R
F = map(R,S,{a^2-c^2, a*b-b*c, b^2-c^2})
m = matrix{{a^4+a^2*b^2, a^2*b^2}}
autosubduction1(m,F,J,4)
///


submatrixByDegrees = (m,d) -> (
    want := positions(0..numgens source m - 1, 
	             i -> (degrees source m)_i === {d});
    m_want)

submatrixBelowDegree = (m,d) -> (
    want := positions(0..numgens source m - 1, 
	             i -> (degrees source m)_i < {d});
    m_want)

rowReduce = (elems, d) -> (
     -- elems is a one row matrix of polynomials, all of degree d.
     -- return a (one row) matrix whose elements are row reduced
     -- CAUTION: HAVE TO FIND WAY TO PASS MONOMIAL ORDER
     R := ring elems;
     n := numgens R;
     M := monoid R;
     RH := (coefficientRing R)[Variables=>n+1, 
	  MonomialOrder => M.Options.MonomialOrder,
	  Degrees => append(M.Options.Degrees,{1})];
     RtoRH := map(RH,R,(vars RH)_{0..n-1});
     RHtoR := map(R,RH,vars R | matrix{{1_R}});
     elemsH := homogenize(RtoRH elems, RH_n);
     RHtoR gens gb(elemsH, DegreeLimit=>d))

///
S = QQ[a,b,c]
m = matrix{{a^2, a^2+a, b^2+a^2+1, a^2+b^2+3, a*b+a}}
rowReduce(m,2)
///


--------------------------------------------------------
--------------------------------------------------------
-- Inhomogeneous SAGBI bases ----
---------------------------------
sagbi = method()
sagbi(Matrix,ZZ) := (Gens, maxnloops) -> (
     --local R, G, S, RS, RStoS, Gmap, inGmap, J;
     --local d, maxdeg, nloops, Pending;
     R = ring Gens;
     maxdeg := maxnloops;
     Pending = new MutableList from elements(maxdeg+1:{});
     insertPending := (m) -> (
	  -- append the entries of the one row matrix 'm' to Pending.
	  i := 0;
	  lodeg := (degree m_(0,0))_0;
	  while i < numgens source m do (
	      f := m_(0,i);
	      e := (degree f)_0;
	      if e < lodeg then lodeg = e;
	      Pending#e = append(Pending#e, f);
	      i = i+1;
	      ));
     lowestDegree := () -> (
	  -- returns maxdeg+1 if Pending list is empty, otherwise
	  -- returns the smallest non-empty strictly positive degree.
	  i := 0;
	  while i <= maxdeg and Pending#i === {} do i=i+1;
	  i);
     appendToBasis := (m) -> (
	  R := ring m;
	  G = G | m;
          S = (coefficientRing R)[Variables=>numgens source G, 
	         Degrees=>degrees source G];
	  Gmap = map(R,S,G);
	  inGmap = map(R,S,leadTerm G);
	  nvars := numgens R + numgens S;
	  J = graphIdeal(inGmap, Variables=>nvars,
	             MonomialOrder => ProductOrder{numgens R,numgens S});
	  RS = ring J;
          RStoS = map(S, RS, (map(S^1, S^(numgens R), 0)) | vars S);
	  );
     grabLowestDegree := () -> (
	  -- assumes: lowest degree pending list is already autosubducted.
	  -- this row reduces this list, placing all of the
	  -- entries back into Pending, but then appends the lowest
	  -- degree part into the basis.
	  e := lowestDegree();
	  if e <= maxdeg then (
	       m = rowReduce(matrix{Pending#e}, e);
	       Pending#e = {};
	       insertPending m;
	       e = lowestDegree();
	       appendToBasis matrix{Pending#e};
	       Pending#e = {};
	       );
	  e);
     
     G = matrix(R, {{}});
     Gensmaxdeg = (max degrees source Gens)_0;
     Gens = compress submatrixBelowDegree(Gens, maxdeg+1);
     insertPending Gens;
     Pending#0 = {};
     d = grabLowestDegree();  -- initializes G 
     d = d+1;
     nloops := d;
     isdone := false;
     while nloops <= maxnloops and not isdone do time (
	  nloops = nloops+1;
	  << "--- degree " << d << " ----" << endl;
	  gbJ = time gb(J, DegreeLimit=>d);
	  spairs = mingens ideal selectInSubring(1, gens gbJ);
	  spairs = submatrixByDegrees(spairs, d);
	  spairs = Gmap(RStoS(spairs));
	  if Pending#d != {} then (
	       newgens = matrix{Pending#d};
	       spairs = spairs | newgens;
	       Pending#d = {};);
	  --newguys = time autosubduction0(spairs, Gmap, J, d);
	  newguys = time compress subduction(spairs, Gmap, J);
	  gb(J,DegreeLimit=>d);
	  if numgens source newguys > 0 
	  then (
     	       << "Generators added to sagbi basis!" << endl;
	       insertPending newguys;
	       d = grabLowestDegree();
	       )
	  else (
	       ngens := sum apply(elements Pending,i -> #i);
	       if ngens === 0 and gbIsDone J and d > Gensmaxdeg then (
	           isdone = true;
		   << "sagbi basis is finite!" << endl;
		   );
	      );
	  d = d+1;
	  );
     G)

///
kk = ZZ/101
R = kk[a,b,c]
F = matrix{{a+b+c-1, a^2+b^2+c^2-a, a^3+b^3+c^3-b}}
time sagbi(F,3)  -- best time 1.96 sec, 5.67 sec
///


--------------------------------------------------------
--------------------------------------------------------
TEST ///
restart
load "sagdraft.m2"
kk = ZZ/101

-- Random change of coordinates in a ring R and its effect on a matrix
G = random(R^1, R^(elements(numgens R:-1)))
Coordchange = map(R, R, G)
F = Coordchange F

-- ring of invariants of S_n
n = 3;
x = symbol x;
R = kk[x_0 .. x_(n-1)]; 
F = map(R^1, n, (j,i) -> sum apply(elements(x_0 .. x_(n-1)), x->x^(i+1)))
time sagbi(F,10)
-- (n=3) -- 4.13 sec
-- (n=4) -- 7.91 sec


-- k by k minors of a generic m by n matrix
minorsize = 2;
rowsize = 2;
colsize = 10;
matdim = rowsize * colsize - 1;
x = symbol x;
R = kk[x_0 .. x_matdim];
F = gens minors(minorsize,genericMatrix(R,x_0,rowsize,colsize))
time sagbi(F,100)
-- (2,3,3) -- 7.2 sec
-- (2,3,4) -- 118.07 sec
-- (2,3,5) -- long long time
-- (2,2,10) -- 134.66 sec


-- 'symmetric' quadratic artin ideal in 2x3 variables
R = kk[symbol a..symbol f]
I = mingens ((ideal(a,b,c))^2 + (ideal(d,e,f))^2 + (ideal(a+d,b+e,c+f))^2)
time sagbi(I,100) -- 


-- example with both finite and infinite sagbi bases
R = kk[symbol x,symbol y]   -- x>y gives infinite, y>x gives finite
F = matrix{{x, x*y-y^2, x*y^2}}
time sagbi(F,30)   -- MES, NEC 6200MX, 60.96 sec

R = kk[symbol y,symbol x]   -- x>y gives infinite, y>x gives finite
F = matrix{{x, x*y-y^2, x*y^2}}
time sagbi(F,1000)

R = kk[symbol x,symbol y]   -- Change of coordinates (i.e. random term order)
F = matrix{{x, x*y-y^2, x*y^2}}
G = random(R^1, R^(elements(2:-1)))
Coordchange = map(R, R, G)
F = Coordchange F
time sagbi(F,200)

-- invariants of A3, infinite sagbi bases, at least for lex order
-- it is infinite for all term orders.
R = kk[a,b,c]
F = matrix{{a+b+c, a*b+b*c+c*a, a*b*c, a^2*b+b^2*c+c^2*a}}
time sagbi(F,15) -- 

R = kk[a,b,c]
F = matrix{{a+b+c, a*b+b*c+c*a, a*b*c, a^2*b+b^2*c+c^2*a}}
time sagbi(F,30) -- 


-- Gr(2,5)
x = symbol x
y = symbol y
R = kk[x_1 .. x_10]
S = kk[y_1 .. y_10]
F = map(R,S,exteriorPower(2,genericMatrix(R,x_1,2,5)))


-- invariants of A^1, with a nilpotent action on A^n.

x = symbol x;
R = kk[t, x_1 .. x_n, MonomialOrder => Lex, 
     Degrees => append(1, elements reverse (1..n))];
-- HAVE TO CODE IN EXPONENTIATION

-- Invariants of A^1, with a nilpotent action on A^4.
x = symbol x
R = kk[t,x_1 .. x_4, MonomialOrder => Lex]
R = kk[t,x_1..x_4, MonomialOrder=>Lex, Degrees=>{1,4,3,2,1}]
R = kk[t,x_1..x_4, MonomialOrder=>ProductOrder{1,4}, Degrees=>{1,4,3,2,1}]
F = matrix{{x_4, 
	  t*x_4+x_3, 
	  t^2*x_4+2*t*x_3+2*x_2, 
	  t^3*x_4+3*t^2*x_3+6*t*x_2+6*x_1}}
time sagbi(F,30) -- 32.36 sec

-- Invariants of A^1, with a nilpotent action on A^5.
x = symbol x
R = QQ[t,x_1..x_5, MonomialOrder=>Lex, Degrees=>{1,5,4,3,2,1}]
R = kk[t,x_1..x_5, MonomialOrder=>Lex, Degrees=>{1,5,4,3,2,1}]
F = matrix{{x_5, 
	  t*x_5+x_4, 
	  t^2*x_5+2*t*x_4+2*x_3, 
	  t^3*x_5+3*t^2*x_4+6*t*x_3+6*x_2,
	  t^4*x_5+4*t^3*x_4+12*t^2*x_3+24*t*x_2+24*x_1}}
time sagbi(F,30) -- seems to take ALOT of time...

-- Invariants of A^1, with a nilpotent action on A^6.
x = symbol x
R = kk[t,x_1..x_6, MonomialOrder=>Lex, Degrees=>{1,6,5,4,3,2,1}]
F = matrix{{x_6, 
	  t*x_6+x_5, 
	  t^2*x_6+2*t*x_5+2*x_4, 
	  t^3*x_6+3*t^2*x_5+6*t*x_4+6*x_3,
	  t^4*x_6+4*t^3*x_5+12*t^2*x_4+24*t*x_3+24*x_2,
	  t^5*x_6+5*t^4*x_5+20*t^3*x_4+60*t^2*x_3+120*t*x_2+120*x_1}}
time sagbi(F,30) -- seems to take ALOT of time...

-- Invariants of A^1, with a nilpotent action on A^3.
x = symbol x;
t = symbol t;
R = kk[t,x_1,x_2,x_3, MonomialOrder=>Lex];
R = kk[t,x_1..x_3, MonomialOrder=>Lex, Degrees=>{1,3,2,1}]
F = matrix{{x_3, 
	  t*x_3+x_2, 
	  t^2*x_3+2*t*x_2+2*x_1}} 
time sagbi(F,200) 


-- invariants of SL_2 on V + V + Sym^2(V)
u = symbol u;
v = symbol v;
s = symbol s;
R = kk[u_1,u_2,v_1,v_2,s_0,s_1,s_2];
F = matrix{{u_2*v_1-u_1*v_2,
	  s_1^2-4*s_0*s_2,
	  s_0*u_2^2+s_2*u_1^2-s_1*u_1*u_2,
	  s_0*v_2^2+s_2*v_1^2-s_1*v_1*v_2,
	  2*s_0*u_2*v_2+2*s_2*u_1*v_1-s_1*(u_2*v_1+u_1*v_2)}}
time sagbi(F,30) -- 4.76 sec
///

///
Loop: D
  compute spairs of degree d of G, subduct
  add in degree d pending list
  
  row reduce this set of polynomials
  (result back into pending list, return lowest degree found).
  if lowest degree found is d:
      add these to G, remove from pending list.
      modify J, Gmap, RS, etc.
  if lowest degree is < d,
      d = this lowest degree
///

///
CHANGES FROM CODE MIKE SENT ME 6/25/97
1. Looping scenario is changed: we start at d=1 (therefore we will reach at
     most degree "maxnloop" elements)
2. Pending list is now maxdeg+1 long so that it can store elements of degree
     maxdeg.
3. We do not even put original generators of degree > maxdeg into Pending
     since they will never be used.
4. Degrees of RH are fixed to correspond with degrees of R (homogenizing)
5. The Pending list could be empty, in which case we 
6. CAREFUL: can only be done if original Pending list contains ALL generators.
7. To address 6, changed isDone criteria to require Pending list to contain
     all generators
8. Changed autosubduction to a normalform operation
///

/// 
CHANGES THAT NEED TO BE MADE
1. Need to pass along MonomialOrder to routine which homogenizes.
///



