/* $Header: /home/jcb/newmj/RCS/greedy.c,v 10.16 2001/01/22 00:49:00 jcb Rel $
 * greedy.c
 * This is a simple offensive player.
 * Options: --id n    use id n when connecting
 *          --server  addr   controller at address addr
 *          --punger   don't claim chows
 *          --chower   don't claim pungs
 *          --{pung,chow,pair,seq,sgl}base   strategy parameters
 *          --part{pung,chow}  ditto.
 *          --{default,punger,chower,concealed}weight  ditto
 *          --exposedpenalty   ditto
 */
/****************** COPYRIGHT STATEMENT **********************
 * This file is Copyright (c) 2000 by J. C. Bradfield.       *
 * Distribution and use is governed by the LICENCE file that *
 * accompanies this file.                                    *
 * The moral rights of the author are asserted.              *
 *                                                           *
 ***************** DISCLAIMER OF WARRANTY ********************
 * This code is not warranted fit for any purpose. See the   *
 * LICENCE file for further information.                     *
 *                                                           *
 *************************************************************/
 
static const char rcs_id[] = "$Header: /home/jcb/newmj/RCS/greedy.c,v 10.16 2001/01/22 00:49:00 jcb Rel $";

static int debugeval = 0;
#include <stdlib.h>
#include <stdio.h>
#include "client.h"
#include "sysdep.h"

#include "version.h"

Game *the_game;
int our_id = 0;
PlayerP our_player;
seats our_seat;

/* Here are some variables controlling our strategy */
/* These give the value of sets. First the base score is
   given, then for pairs and sequences is added
   part??? * ???base * (no. chances of completion)
*/
enum { Def=0,Punger=1,Chower=2, Concealed=3, Fast=4, NumStrat=5 };
static int curstrat = Def;
enum {pungbase=0,pairbase=1,chowbase=2,seqbase=3,sglbase=4,
	      partpung=5,partchow=6,exposedpungpenalty=7,exposedchowpenalty=8,
              mjbonus=9, kongbonus=10,weight=11};
static double strategies[NumStrat][weight+1] = {
  /* pungb  prb   chowb seqb sglb ptpu  ptchw   expp, expc mjb kongb weight*/
  { 12.0 , 4.0 , 12.0, 1.0, 0.05, 0.15, 1.0/12.0, 2.0, 2.0, 60.0, 3.0, 1.0 }, /* default */
  { 16.0 , 6.0 , 0.0, 0.0, 0.05, 0.15, 1.0/12.0, 1.0, 50.0, 60.0, 3.0, 1.0 }, /* pung */
  /* note that increasing pairbase to 4 seems to be key */
  { 0.0 , 6.0 , 14.0, 1.0, 0.0, 0.15, 1.0/12.0, 50.0, 1.0 , 60.0, 0.0, 1.0}, /* chow */
  { 18.0 , 4.0 , 18.0, 1.0, 0.05, 0.1, 1.0/18.0, 50.0, 50.0, 60.0, 0.0, 1.0 }, /* concealed */
  { 12.0 , 4.0 , 12.0, 1.0, 0.05, 0.15, 1.0/12.0, 1.0, 1.0, 60.0, 0.0, 1.0 }, /* fast */
};

/* the value of the non-default strategies must exceed the default
   by this amount for them to be chosen */
static double hysteresis = 4.0; 


static int easyeval = 0;
/* Used to note availability of tiles */
static int tilesleft[MaxTile];

static int despatch_line(char *line);
static void do_something(void);
static void check_discard(PlayerP p,int strat);
static Tile decide_discard(PlayerP p, double *score, int *newstrat);
static void update_tilesleft(CMsgUnion *m);
static double eval(Tile *tp, int strat, int reclevel, int ninc, int npr);
static double eval_tile(Tile *tp, Tile t);
static double evalhand(PlayerP p, int strat);
static int chances_to_win(PlayerP p);
/* copy old tile array into new */
#define tcopy(new,old) memcpy((void *)new,(void *)old,(MAX_CONCEALED+1)*sizeof(Tile))
/* Convenience function */
#define send_packet(m) client_send_packet(the_game,(PMsgMsg *)m)

static void usage(char *pname,char *msg) {
  fprintf(stderr,"%s: %s\nUsage: %s [ --id N ] [ --server ADDR ]\n\
  [ --punger ] [ --chower ]\n\
  [ --{pung,pair,chow,sgl}base N.N ] [ --part{pung,chow} N.N ]\n",
	  pname,msg,pname);
  exit(1);
}

int main(int argc, char *argv[]) {
  char buf[1000];
  char *l;
  int i;
  char *evalh = NULL ; /* just evaluate hand with default strategy
			  and all debugging, and exit */
  char *address = ":5000";

  /* options. I should use getopt ... */
  for (i=1;i<argc;i++) {
    if ( strcmp(argv[i],"--id") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --id");
      our_id = atoi(argv[i]);
    } else if ( strcmp(argv[i],"--server") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --server");
      address = argv[i];
    } else if ( strcmp(argv[i],"--address") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --address");
      address = argv[i];
    } else if ( strcmp(argv[i],"--debug") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --debug");
      debugeval = atoi(argv[i]);
    } else if ( strcmp(argv[i],"--eval") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --eval");
      evalh = argv[i];
      debugeval = 99;
    } else if ( strcmp(argv[i],"--pungbase") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --pungbase");
      strategies[curstrat][pungbase] = atof(argv[i]);
    } else if ( strcmp(argv[i],"--hysteresis") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --hysteresis");
      hysteresis = atof(argv[i]);
    } else if ( strcmp(argv[i],"--defaultweight") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --defaultweight");
      strategies[Def][weight] = atof(argv[i]);
    } else if ( strcmp(argv[i],"--pungerweight") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --pungerweight");
      strategies[Punger][weight] = atof(argv[i]);
    } else if ( strcmp(argv[i],"--chowerweight") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --chowerweight");
      strategies[Chower][weight] = atof(argv[i]);
    } else if ( strcmp(argv[i],"--fastweight") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --fastweight");
      strategies[Fast][weight] = atof(argv[i]);
    } else if ( strcmp(argv[i],"--concealedweight") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --concealedweight");
      strategies[Concealed][weight] = atof(argv[i]);
    } else if ( strcmp(argv[i],"--pairbase") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --pairbase");
      strategies[curstrat][pairbase] = atof(argv[i]);
    } else if ( strcmp(argv[i],"--chowbase") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --chowbase");
      strategies[curstrat][chowbase] = atof(argv[i]);
    } else if ( strcmp(argv[i],"--seqbase") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --seqbase");
      strategies[curstrat][seqbase] = atof(argv[i]);
    } else if ( strcmp(argv[i],"--sglbase") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --sglbase");
      strategies[curstrat][sglbase] = atof(argv[i]);
    } else if ( strcmp(argv[i],"--partpung") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --partpung");
      strategies[curstrat][partpung] = atof(argv[i]);
    } else if ( strcmp(argv[i],"--partchow") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --partchow");
      strategies[curstrat][partchow] = atof(argv[i]);
    } else if ( strcmp(argv[i],"--exposedpungpenalty") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --exposedpungpenalty");
      strategies[curstrat][exposedpungpenalty] = atof(argv[i]);
    } else if ( strcmp(argv[i],"--exposedchowpenalty") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --exposedchowpenalty");
      strategies[curstrat][exposedchowpenalty] = atof(argv[i]);
    }
    else {
      fprintf(stderr,argv[i]);
      usage(argv[0],"unknown option or argument");
    }
  }

  if ( evalh ) {
    Player pp;
    Game g;
    
    g.round = EastWind;
    the_game = &g;
    pp.wind = EastWind;
    initialize_player(&pp);
    set_player_tiles(&pp,evalh);
    our_player = &pp;
    evalhand(&pp,curstrat);
    exit(0);
  }

  the_game = client_init(address);
  if ( ! the_game ) exit(1);

  sprintf(buf,"Robot(%d)",getpid());

  client_connect(the_game,our_id,buf);

  while ( 1 ) {
   l = get_line(the_game->fd);
    if ( ! l ) {
      exit(1);
    }
    despatch_line(l);
  }
}

/* despatch_line: this is the mega-switch which deals with
   the input from the controller */
static int despatch_line(char *line) {
  CMsgMsg *cm;

  if ( line == NULL ) {

    warn("receive error on controller connexion\n");
    exit(1);
  }

  cm = decode_cmsg(line);
  if ( cm == NULL ) {
    warn("Protocol error on controller connexion; ignoring\n");
    return 0;
  }

  update_tilesleft((CMsgUnion *)cm);

  switch ( cm->type ) {
  case CMsgError:
    break; /* damn all we can do */
  case CMsgGameOver:
    exit(0);
  case CMsgInfoTiles:
    /* We ignore these. */
    break;
  case CMsgCanMahJong:
    /* Currently we ignore these, as we don't issue queries */
    break;
  case CMsgConnectReply:
    game_handle_cmsg(the_game,cm);
    our_id = the_game->players[0]->id;
    our_player = the_game->players[0];
    break;
    /* In these cases, our seat might have changed, so we need to calculate it */
  case CMsgGame:
  case CMsgNewRound:
  case CMsgNewHand:
    game_handle_cmsg(the_game,cm);
    our_seat = game_id_to_seat(the_game,our_id);
    curstrat = Def;
    break;
    /* in all these cases, game_handle_cmsg does all the work we want */
  case CMsgPlayer:
  case CMsgStopPlay:
  case CMsgClaimDenied:
  case CMsgPlayerDoesntClaim:
  case CMsgPlayerClaimsPung:
  case CMsgPlayerClaimsKong:
  case CMsgPlayerClaimsChow:
  case CMsgPlayerClaimsMahJong:
  case CMsgPlayerShowsTiles:
  case CMsgDangerousDiscard:
  case CMsgGameOption:
  case CMsgChangeManager:
  case CMsgMessage:
    game_handle_cmsg(the_game,cm);
    break;
  case CMsgHandScore:
    /* if that was the winner, we should start scoring our hand */
    if ( ( game_handle_cmsg(the_game,cm) 
	   == the_game->players[the_game->info.player]->id)
	 && the_game->active )
      do_something();
    break;
    /* after a Settlement or Washout message, do something: start next hand */
  case CMsgWashOut:
  case CMsgSettlement:
    game_handle_cmsg(the_game,cm);
    if ( the_game->active ) do_something();
    break;
    /* likewise after a washout */
    /* after a MahJong message, we should do something: namely
       start making our scoring sets. */
  case CMsgPlayerRobsKong:
  case CMsgPlayerMahJongs:
    game_handle_cmsg(the_game,cm);
    if ( the_game->active ) do_something();
    break;
    /* in the case of a PlayerDeclaresSpecials message, we need to
       do something if it is now our turn; but this isn't given
       by the affected id.
       However, if the state is Discarding, and no tiles have
       so far been discarded, we shouldn't do something
       now, since we are about to be asked to pause.
    */
  case CMsgPlayerDeclaresSpecial:
    game_handle_cmsg(the_game,cm);
    if ( the_game->info.player == our_seat && the_game->active
	 && ! ( the_game->state == Discarding && the_game->info.serial == 0 ))
      do_something();
    break;
    /* in these cases, we need to do something if the message
       is addressed to us. */
  case CMsgPlayerDraws:
  case CMsgPlayerDrawsLoose:
  case CMsgPlayerPungs:
  case CMsgPlayerKongs:
  case CMsgPlayerChows:
  case CMsgPlayerFormsClosedPung:
  case CMsgPlayerFormsClosedChow:
  case CMsgPlayerPairs:
  case CMsgPlayerFormsClosedPair:
  case CMsgPlayerSpecialSet:
  case CMsgPlayerFormsClosedSpecialSet:
  case CMsgSwapTile:
    if ( game_handle_cmsg(the_game,cm) == our_id && the_game->active)
      do_something();
    break;
    /* in this case, we need to do something else if it's not our turn! */
  case CMsgPlayerDiscards:
    if ( game_handle_cmsg(the_game,cm) != our_id && the_game->active)
      check_discard(our_player,curstrat);
    break;
    /* if this is us, we need to do something, and if it's 
       somebody else, we might be able to rob the kong */
  case CMsgPlayerDeclaresClosedKong:
  case CMsgPlayerAddsToPung:
    if ( game_handle_cmsg(the_game,cm) == our_id && the_game->active)
      do_something();
    else if ( the_game->active ) 
      check_discard(our_player,curstrat); /* actually this checks the kong */
    break;
    /* In this case, it depends on the state of the game */
  case CMsgStartPlay:
    /* We need to do something if the id is us, or 0. */
    { int id;
    id = game_handle_cmsg(the_game,cm);
    if ( id == our_id || id == 0 ) 
      do_something();
    }
    break;
    /* similarly */
  case CMsgPlayerReady:
    game_handle_cmsg(the_game,cm);
    if ( ! the_game->paused ) do_something();
    break;
  case CMsgPause:
    game_handle_cmsg(the_game,cm);
    do_something();
    break;
  case CMsgPlayerOptionSet:
    /* we don't recognize any options, so ignore it */
    break;
  }
  return 1;
}

/* do something when it's our turn. 
*/
static void do_something(void) {
  int i;

  /* if the game is paused, and we haven't said we're ready, say so */
  if ( the_game->paused ) {
    if ( !the_game->info.ready[our_seat] ) {
      PMsgReadyMsg pm;
      pm.type = PMsgReady;
      send_packet(&pm);
    }
    return;
  }

  /* if the game state is handcomplete, do nothing */
  if ( the_game->state == HandComplete ) return;

  /* If the game state is discarded, then it must mean this has
     been called in response to a StartPlay message after resuming
     an old hand. So actually we want to check the discard, unless
     of course we are the discarder, or we have already claimed. */
  if ( the_game->state == Discarded ) {
    if ( the_game->info.player != our_seat
	 && the_game->info.claims[our_seat] == UnknownClaim )
      check_discard(our_player,curstrat);
    return;
  }

  /* If the game state is Dealing, then we should not do anything.
   */
  if ( the_game->state == Dealing ) return;

  /* if we're called in declaring specials or discarding, but it's
     not our turn, do nothing */
  if ( (the_game->state == DeclaringSpecials
	|| the_game->state == Discarding)
       && the_game->info.player != our_seat ) return;

  /* if we're waiting to draw another tile, do nothing */
  if ( the_game->info.needs != FromNone ) return;

  /* if we have a special, declare it. N.B. we'll
     be called again as a result of this, so only look for first.
  */
  for ( i=0; i < our_player->num_concealed
	  && ! is_special(our_player->concealed[i]) ; i++);
  if ( i < our_player->num_concealed ) {
    PMsgDeclareSpecialMsg m;
    m.type = PMsgDeclareSpecial;
    m.tile = our_player->concealed[i];
    send_packet(&m);
    return;
  }
  /* OK, no specials */
  if ( the_game->state == DeclaringSpecials ) {
    PMsgDeclareSpecialMsg m;
    m.type = PMsgDeclareSpecial;
    m.tile = HiddenTile;
    send_packet(&m);
    return;
  }
  /* if the game is in the mahjonging state, and our hand is not declared,
     then we should declare a set. */
  if ( the_game->state == MahJonging ) {
    TileSet *tsp;
    PMsgUnion m;

    if ( pflag(our_player,HandDeclared) ) return;
    /* as courtesy, if we're not the winner, we shouldn't score until
       the winner has */
    if ( our_seat != the_game->info.player
	 && ! pflag(the_game->players[the_game->info.player],HandDeclared) ) return;

    /* get the list of possible decls */
    tsp = client_find_sets(our_player,
			   (the_game->info.player == our_seat
			    && the_game->info.pending)
			   ? the_game->info.tile : HiddenTile,
			   the_game->info.player == our_seat,
			   (PlayerP *)0);
    if ( !tsp && our_player->num_concealed > 0 ) {
      m.type = PMsgShowTiles;
      send_packet(&m);
      return;
    }
    /* just do the first one */
    switch ( tsp->type ) {
    case Kong:
      /* we can't declare a kong now, so declare the pung instead */
    case Pung:
      m.type = PMsgPung;
      m.pung.discard = 0;
      break;
    case Chow:
      m.type = PMsgChow;
      m.chow.discard = 0;
      m.chow.cpos = the_game->info.tile - tsp->tile;
      break;
    case Pair:
      m.type = PMsgPair;
      break;
    case ClosedPung:
      m.type = PMsgFormClosedPung;
      m.formclosedpung.tile = tsp->tile;
      break;
    case ClosedChow:
      m.type = PMsgFormClosedChow;
      m.formclosedchow.tile = tsp->tile;
      break;
    case ClosedPair:
      m.type = PMsgFormClosedPair;
      m.formclosedpair.tile = tsp->tile;
      break;
    case Empty: /* can't happen, just to suppress warning */
    case ClosedKong: /* ditto */
      ;
    }
    send_packet(&m);
    return;
  }
    
  /* if we can declare MahJong, do it */
  if ( player_can_mah_jong(our_player,HiddenTile) ) {
    PMsgMahJongMsg m;
    m.type = PMsgMahJong;
    m.discard = 0;
    send_packet(&m);
    return;
  } else if ( the_game->info.whence != FromDiscard ) {
    /* check for concealed kongs and melded kongs. Just declare them. */
    int i;
    double val;
    Player pc;
    val = evalhand(our_player,curstrat);
    /* a side effect of the above call is that our concealed tiles 
       are sorted (in reverse order), so we can avoid duplicating effort */
    for (i=0;i<our_player->num_concealed;i++) {
      /* don't look at same tile twice */
      if ( i && our_player->concealed[i] == our_player->concealed[i-1] ) continue;
      if ( player_can_declare_closed_kong(our_player,our_player->concealed[i]) ) {
	PMsgDeclareClosedKongMsg m;
	copy_player(&pc,our_player);
	player_forms_closed_pung(&pc,our_player->concealed[i]);
	if ( (evalhand(&pc,curstrat)+strategies[curstrat][kongbonus]) > val ) {
	  m.type = PMsgDeclareClosedKong;
	  m.tile = our_player->concealed[i];
	  send_packet(&m);
	  return;
	}
      }
    }
    /* Now check for pungs we can meld to */
    for (i=0;i<MAX_TILESETS;i++) {
      if ( our_player->tilesets[i].type == Pung
	   && player_can_add_to_pung(our_player,our_player->tilesets[i].tile) ) {
	PMsgAddToPungMsg m;
	copy_player(&pc,our_player);
	player_adds_to_pung(&pc,our_player->tilesets[i].tile);
	if ( evalhand(&pc,curstrat) > val ) {
	  m.type = PMsgAddToPung;
	  m.tile = our_player->tilesets[i].tile;
	  send_packet(&m);
	  return;
	}
      }
    }
  }
  /* if we get here, we have to discard */
  {
    PMsgDiscardMsg m;
    Tile t;
    int newstrat;
    
    t = decide_discard(our_player,NULL,&newstrat);

    m.type = PMsgDiscard;
    m.tile = t;
    curstrat = newstrat;
    m.calling = 0; /* we don't bother looking for original call */
    send_packet(&m);
    return;
  }
}

/* Check if we want the discard, and claim it.
   Arg is strategy.
   Also called to check whether a kong can be robbed */
static void check_discard(PlayerP p, int strat) {
  PMsgUnion m;
  double bestval,val;
  int canmj;
  char buf[100];

  if ( the_game->state == Discarding ) {
    /* this means we're being called to check whether a kong
       can be robbed. Since robbing a kong gets us an extra double,
       this is probably always worth doing, unless we're trying for
       some limit hand */
    if ( player_can_mah_jong(p,the_game->info.tile) ) {
      m.type = PMsgMahJong;
      m.mahjong.discard = the_game->info.serial;
    } else {
      m.type = PMsgNoClaim;
      m.noclaim.discard = the_game->info.serial;
    }
    send_packet(&m);
    return;
  }

  if ( debugeval ) {
    player_print_tiles(buf,p,0);
    printf("Hand: %s  ; discard %s\n",buf,tile_code(the_game->info.tile));
  }
  bestval = evalhand(p,strat)
    + strategies[strat][sglbase]*tilesleft[the_game->info.tile];
  if ( debugeval ) {
    printf("Hand value before claim %.3f\n",bestval);
  }

  canmj = player_can_mah_jong(p,the_game->info.tile);
  m.type = PMsgNoClaim;
  m.noclaim.discard = the_game->info.serial;
  if ( player_can_kong(p,the_game->info.tile) ) {
      Player pc;

      copy_player(&pc,p);
      player_pungs(&pc,the_game->info.tile);
      val = evalhand(&pc,strat) + strategies[strat][kongbonus];
      if ( debugeval ) {
	printf("Hand after kong  %.3f\n",val);
      }
      if ( val > bestval ) {
	m.type = PMsgKong;
	m.kong.discard = the_game->info.serial;
	bestval = val;
      }
      else 
	if ( debugeval ) {
	  printf("Chose not to kong\n");
	}
  } 
  if ( player_can_pung(p,the_game->info.tile) ) {
    Player pc;

    copy_player(&pc,p);
    player_pungs(&pc,the_game->info.tile);
    val = evalhand(&pc,strat);
    if ( debugeval ) {
      printf("Hand after pung  %.3f\n",val);
    }
    if ( val > bestval ) {
      m.type = PMsgPung;
      m.pung.discard = the_game->info.serial;
      bestval = val;
    }
    else 
      if ( debugeval ) {
	printf("Chose not to pung\n");
      }
    }
  if ( (canmj || our_seat == (the_game->info.player+1)%NUM_SEATS)
       && is_suit(the_game->info.tile) ) {
    ChowPosition cpos = (unsigned) -1;
    Player pc;
    int chowposs = 0;
    Tile t = the_game->info.tile;
    copy_player(&pc,p);
    if ( player_chows(&pc,t,Lower) ) {
      val = evalhand(&pc,strat);
      if ( debugeval ) {
	printf("Hand after lower chow: %.3f\n",val);
      }
      chowposs = 1;
      if ( val > bestval ) {
	bestval = val;
	cpos = Lower;
      }
      copy_player(&pc,p);
    }
    if ( player_chows(&pc,t,Middle) ) {
      val = evalhand(&pc,strat);
      if ( debugeval ) {
	printf("Hand after middle chow: %.3f\n",val);
      }
      chowposs = 1;
      if ( val > bestval ) {
	bestval = val;
	cpos = Middle;
      }
      copy_player(&pc,p);
    }
    if ( player_chows(&pc,t,Upper) ) {
      chowposs = 1;
      val = evalhand(&pc,strat);
      if ( debugeval ) {
	printf("Hand after upper chow: %.3f\n",val);
      }
      if ( val > bestval ) {
	bestval = val;
	cpos = Upper;
      }
      copy_player(&pc,p);
    }
    if ( cpos != (unsigned)-1 ) {
      m.type = PMsgChow;
      m.chow.discard = the_game->info.serial;
      m.chow.cpos = cpos;
    } 
    else
      if ( debugeval ) {
	if ( chowposs ) fprintf(stdout,"chose not to chow\n");
      }
  }
  /* pairing */
  if ( canmj ) {
    Player pc;
    copy_player(&pc,p);
    if ( player_pairs(&pc,the_game->info.tile) ) {
      val = evalhand(&pc,strat);
      if ( debugeval ) {
	printf("Hand after pair: %.3f\n",val);
      }
      if ( val > bestval ) {
	m.type = PMsgMahJong;
	m.mahjong.discard = the_game->info.serial;
	bestval = val;
      }
    }
    /* if we're following a concealed strategy, and we still have
       four chances (ex?cluding this one) of going out, then
       don't claim */
    if ( strat == Concealed ) {
      /* fprintf(stderr,"Calling on concealed strat, chances %d\n",
	 chances_to_win(p)); */
      if ( chances_to_win(p) >= 4 ) {
	m.type = PMsgNoClaim;
	m.noclaim.discard = the_game->info.serial;
      }
    }
    if ( m.type != PMsgNoClaim ) {
      m.type = PMsgMahJong;
      m.mahjong.discard = the_game->info.serial;
    } else {
      if ( debugeval ) {
	printf("CHOSE NOT TO MAHJONG\n");
      }
    }
  }
  if ( debugeval ) {
    printf("Result: %s",encode_pmsg((PMsgMsg *)&m));
  }
  send_packet(&m);
  return;
}


/* Here is a data structure to track the number of (assumed)
   available tiles elsewhere. The update fn should be called on every CMsg. */

static void update_tilesleft(CMsgUnion *m) {
  int i;
  /* note that we don't need to check whether the tile is blank,
     since the HiddenTile entry of tilesleft has no meaning.
  */
  switch ( m->type ) {
  case CMsgNewHand:
    for (i=0; i < MaxTile; i++) tilesleft[i] = 4;
    return;
  case CMsgPlayerDeclaresSpecial:
    return;
  case CMsgPlayerDraws:
    tilesleft[m->playerdraws.tile]--;
    return;
  case CMsgPlayerDrawsLoose:
    tilesleft[m->playerdrawsloose.tile]--;
    return;
  case CMsgPlayerDiscards:
    /* if this is us, we've already counted it */
    if ( m->playerdiscards.id != our_id )
      tilesleft[m->playerdiscards.tile]--;
    return;
  case CMsgPlayerPungs:
    /* if somebody else pungs, then two more tiles become dead
       (the discard was already noted).
       If we pung, nothing new is known */
    if ( m->playerpungs.id != our_id )
      tilesleft[m->playerpungs.tile] -= 2;
    return;
  case CMsgPlayerKongs:
    /* if somebody else kongs, then three more tiles become dead
       (the discard was already noted). */
    if ( m->playerkongs.id != our_id )
      tilesleft[m->playerkongs.tile] -= 3;
    return;
  case CMsgPlayerDeclaresClosedKong:
    if ( m->playerdeclaresclosedkong.id != our_id )
      tilesleft[m->playerdeclaresclosedkong.tile] -= 4;
    return;
  case CMsgPlayerAddsToPung:
    if ( m->playeraddstopung.id != our_id )
      tilesleft[m->playeraddstopung.tile]--;
    return;
  case CMsgPlayerChows:
    if ( m->playerchows.id != our_id ) {
      Tile t = m->playerchows.tile;
      ChowPosition c = m->playerchows.cpos;
      tilesleft[(c==Lower)?t+1:(c==Middle)?t-1:t-2]--;
      tilesleft[(c==Lower)?t+2:(c==Middle)?t+1:t-1]--;
    }
    return;
  case CMsgSwapTile:
    tilesleft[m->swaptile.oldtile]++;
    tilesleft[m->swaptile.newtile]--;
    return;
  default:
    return;
  }
}


/* This sorts the tiles (as below) into reverse order.
   Why reverse order? So that HiddenTile terminates the hand.
*/
static void tsort(Tile *tp) {
  int i,m;
  Tile t;
  /* bubblesort */
  m = 1;
  while ( m ) {
    m = 0;
    for (i=0; i<MAX_CONCEALED+1-1;i++)
      if ( tp[i] < tp[i+1] ) {
	t = tp[i];
	tp[i] = tp[i+1];
	tp[i+1] = t;
	m = 1;
      }
  }
}


/* is this tile a doubling tile (or scoring pair) */
#define is_doubler(t) (is_dragon(t) || (is_wind(t) && \
 (suit_of(t) == our_player->wind || suit_of(t) == the_game->round)))

/* This function attempts to evaluate a partial hand.
   The input is a (pointer to an) array of tiles
   (length MAX_CONCEALED+1; some will be blank).
   The value is a floating point number, calculated thus:
   for the first tile in the set, form all possible sets
   and partial sets (incl. singletons) with that tile;
   then recurse on the remainder. The value is then the
   max over the partial sets of the inherent value of the
   set (explained below) and the recursive value of the remaining hand.
   This would have the slightly awkward consequence that pairs get counted
   twice in pungs, since given 222... we see 22 2 and also 2 22.
   Hence, if we form a pung, we don't form a pair also.
   (Nothing special is done about kongs; it should be.)
   **** The input is sorted in place. ****
*/

/* debugging format:
   SSTT:mmmm+ (recursive)
   tttt(nnnn)
   where SS is a set code (Pu,Pr,Ch,In,Ou,Si),
   TT is the tile code, mmmm is the set value,
   tttt is the total, nnnn is the residual
   Hence the recursive call should indent all lines except the first
   by reclevel*11 characters, and the base case call should emit
   a newline.
*/
/* other parameters:
   strat: current strategy
   reclevel: recursion level. Should start at 0.
   ninc: number of incomplete sets (including pairs) in
     the chain so far.
   npr: number of pairs in the chain so far.
*/

static double eval(Tile *tp,int strat,
	    int reclevel,
	    int ninc,
	    int npr) {
  Tile copy[MAX_CONCEALED+1];
  int i;
  double mval = 0.0, val;
  char prefix[200];
  static char totbuf[200];
  char tb[20];
  int notfirst=0;

  if ( debugeval ) {
    /* The total buffer. A call at recursion level n writes the total into bytes
       11(n-1) to 11n-1  of the buffer (right justified in the first 4 chars; 
       spaces in the rest.
       A base case terminates the total buffer with null.
       When a second set is printed, or when we terminate at level zero,
       then the total buffer is printed.
    */

    for (i=0;i<reclevel*11;i++) prefix[i]=' ';
    prefix[i]=0;
  }

  tsort(tp);
  
  if ( tp[0] == HiddenTile ) {
    if ( debugeval > 1 ) {
      printf("\n"); 
      if (reclevel) sprintf(&totbuf[11*(reclevel-1)+4],"\n");
    }
    /* if there is one "incomplete" set and one pair, we have a complete
       hand */
    if ( ninc == 1 && npr == 1 ) val = strategies[strat][mjbonus];
    else val = 0.0;
    if ( debugeval > 1 ) {
      sprintf(tb,"%4.1f",val);
      strncpy(&totbuf[11*reclevel],tb,4);
    }
    return val;
  }

  /* does it pung? */
  if ( tp[1] == tp[0] ) {
    if ( tp[2] == tp[0] ) {
      val = 0.0;
      /* add the value of this set.
      */
      val += strategies[strat][pungbase];
      /* if we're trying for a no score hand, scoring pairs are bad */
      if ( is_doubler(tp[0]) ) val += (strat == Chower) ? -6.0 : 6.0;
      if ( is_major(tp[0]) ) val += 2.0;
      if ( debugeval > 1 ) {
	printf("%sPu%s:%4.1f+ ",notfirst++?prefix:"",tile_code(tp[0]),val);
      }
      /* remove the tiles and recurse */
      tcopy(copy,tp);
      copy[0] = copy[1] = copy[2] = HiddenTile;
      val += eval(copy,strat,reclevel+1,ninc,npr);
      if ( debugeval > 1 ) {
	sprintf(tb,"%4.1f",val);
	strncpy(&totbuf[11*reclevel],tb,4);
      }
      if ( val > mval ) mval = val;
    } else {
      val = 0.0;
      /* A pair is worth something for itself, plus something
	 for its chances of becoming a pung.
	 A doubler pair is worth an extra 2 points.
	 A major pair is worth an extra point.
	 Beware the risk of arranging effect that a chow and 
	 a pair is worth much more than a pung and an inner sequence,
	 so that we'd break a pung rather than a chow.
      */
      val += strategies[strat][pairbase];
      val += strategies[strat][partpung]*strategies[strat][pungbase] * tilesleft[tp[0]];
      if ( debugeval > 1 ) {
	printf("%sPr%s:%4.1f+ ",notfirst++?prefix:"",tile_code(tp[0]),val);
      }
      /* doublers are bad for noscore */
      if ( is_doubler(tp[0]) ) val += (strat == Chower) ? -8.0 : 2.0;
      if ( is_major(tp[0]) ) val += 1.0;
      /* remove the tiles and recurse */
      tcopy(copy,tp);
      copy[0] = copy[1] = HiddenTile;
      val += eval(copy,strat,reclevel+1,ninc+1,npr+1);
      if ( debugeval > 1 ) {
	sprintf(tb,"%4.1f",val);
	strncpy(&totbuf[11*reclevel],tb,4);
      }
      if ( val > mval ) mval = val;
    }
  }
  /* OK, now deal with chows. */
  if ( is_suit(tp[0]) ) {
    Tile t1,t2; int i1,i2; /* other tiles, and their indices */
    /* NB tiles are in reverse order!!!!! */
    t1 = tp[0]-1; t2 = tp[0]-2;
    for (i1=1;tp[i1] && tp[i1]!=t1;i1++);
    if ( ! tp[i1] ) i1=0;
    for (i2=1;tp[i2] && tp[i2]!=t2;i2++);
    if ( ! tp[i2] ) i2=0;

    /* if we have a chow */
    if ( i1 && i2 ) {
      val = 0.0;
      /* A chow is deemed to be worth 12 points also.
	 (Quick and dirty ... hike pungbonus to avoid chows.)
      */
     val += strategies[strat][chowbase];
     if ( debugeval > 1 ) {
       if ( notfirst ) printf("%s%s",prefix,&totbuf[11*reclevel]);
       printf("%sCh%s:%4.1f+ ",notfirst++?prefix:"",tile_code(tp[0]),val);
     }
      /* remove the tiles and recurse */
      tcopy(copy,tp);
      copy[0] = copy[i1] = copy[i2] = HiddenTile;
      val += eval(copy,strat,reclevel+1,ninc,npr);
      if ( debugeval > 1 ) {
	sprintf(tb,"%4.1f",val);
	strncpy(&totbuf[11*reclevel],tb,4);
      }
      if ( val > mval ) mval = val;
    }
    /* If we have an inner sequence... note that it's intentional
       that we do this as well, since maybe if we split the chow,
       we'll find a pung.
    */
    if ( i1 ) {
      val = 0.0;
      /* An inner sequence is worth the number of chances of completion,
	 allowing for the fact that on average, the tiles in right
	 and opposite and half the tiles in the wall will not be available
	 for completing a chow.
	 NOTE: this needs to change when we're nearly at MahJong.
      */
      if ( value_of(tp[0]) < 9 ) {
	val += strategies[strat][partchow]*strategies[strat][chowbase] * tilesleft[tp[0]+1];
	val += strategies[strat][seqbase];
      }
      if ( value_of(t1) > 1 ) {
	val += strategies[strat][partchow]*strategies[strat][chowbase] * tilesleft[t1-1];
	val += strategies[strat][seqbase];
      }
      if ( debugeval > 1 ) {
	if ( notfirst ) printf("%s%s",prefix,&totbuf[11*reclevel]);
	printf("%sIn%s:%4.1f+ ",notfirst++?prefix:"",tile_code(tp[0]),val);
      }
      /* remove the tiles and recurse */
      tcopy(copy,tp);
      copy[0] = copy[i1] = HiddenTile;
      val += eval(copy,strat,reclevel+1,ninc+1,npr);
      if ( debugeval > 1 ) {
	sprintf(tb,"%4.1f",val);
	strncpy(&totbuf[11*reclevel],tb,4);
      }
      if ( val > mval ) mval = val;
    }
    /* If we have a split sequence ... Here we don't do this if there's
       been a chow. Hmm, why not? I thought I had a reason. Fill it in
       sometime. */
    else if ( i2 ) {
      val = 0.0;
      /* Likewise */
      val += strategies[strat][seqbase];
      val += strategies[strat][partchow]*strategies[strat][chowbase] * tilesleft[t1];
      if ( debugeval > 1 ) {
	if ( notfirst ) printf("%s%s",prefix,&totbuf[11*reclevel]);
	printf("%sOu%s:%4.1f+ ",notfirst++?prefix:"",tile_code(tp[0]),val);
      }
      /* remove the tiles and recurse */
      tcopy(copy,tp);
      copy[0] = copy[i2] = HiddenTile;
      val += eval(copy,strat,reclevel+1,ninc+1,npr);
      if ( debugeval > 1 ) {
	sprintf(tb,"%4.1f",val);
	strncpy(&totbuf[11*reclevel],tb,4);
      }
      if ( val > mval ) mval = val;
    }
  }
  /* Finally, the score for a singleton. This is small:
     .25 the number of chances left, 
     plus (chances-1) for doublers. (The negativity here is intentional).
  */
  /* let's also add .25 times the number of neighbouring tiles */
  val = 0.0;
  val += strategies[strat][sglbase]
    * (tilesleft[tp[0]]
       + (((strat == Chower) && is_suit(tp[0])) 
	  ? (tilesleft[tp[0]-1]+ tilesleft[tp[0]+1]) : 0))
    * (strat == Chower ? 1 : 4);
  if ( is_doubler(tp[0]) ) val += tilesleft[tp[0]] -1.0 - 3*(strat == Chower);
  if ( debugeval > 1 ) {
    if ( notfirst ) printf("%s%s",prefix,&totbuf[11*reclevel]);
    printf("%sSi%s:%4.1f+ ",notfirst++?prefix:"",tile_code(tp[0]),val);
  }
  tcopy(copy,tp);
  copy[0] = HiddenTile;
  val += eval(copy,strat,reclevel+1,ninc+1,npr);
  if ( debugeval > 1 ) {
    sprintf(tb,"%4.1f",val);
    strncpy(&totbuf[11*reclevel],tb,4);
  }
  if ( val > mval ) mval = val;
  if ( debugeval > 1 ) {
    if ( reclevel ) {
      sprintf(tb,"(%4.1f) ",mval);
      strncpy(&totbuf[11*(reclevel-1)+4],tb,7);
    }
    else printf(totbuf);
  }
  return mval;
}


/* This is a simpler function: given an array and a tile,
   it evaluates the tile just by the positive basis of which
   sets it's part of. Try this out: it's not clear whether it
   comes to the same thing in the end */
static double eval_tile(Tile *tp, Tile t) {
  int i,n;
  double mval,val;

  mval = 0.0;
  /* How many of this tile? */
  for (n=i=0;i<MAX_CONCEALED+1;i++) if (tp[i] == t) n++;
  if ( n >=3 ) { /* arguably, if there are 4, it's less valuable ? */
    val = 0.0;
    val += strategies[curstrat][pungbase];
    if ( is_doubler(t) ) val += 6.0;
    if ( is_major(t) ) val += 2.0;
#if DEBUGEVAL
    printf("Pu%s:%4.1f+ ",tile_code(t),val);
#endif
    mval += val;
    } else if ( n >= 2) {
      val = 0.0;
      /* A pair is worth something for itself, plus something
	 for its chances of becoming a pung.
	 A doubler pair is worth an extra 2 points.
	 A major pair is worth an extra point.
	 Beware the risk of arranging effect that a chow and 
	 a pair is worth much more than a pung and an inner sequence,
	 so that we'd break a pung rather than a chow.
      */
      val += strategies[curstrat][pairbase];
      val += strategies[curstrat][partpung]*strategies[curstrat][pungbase] * tilesleft[t];
#if DEBUGEVAL
      printf("Pr%s:%4.1f+ ",tile_code(t),val);
#endif
      if ( is_doubler(t) ) val += 2.0;
      if ( is_major(t) ) val += 1.0;
      mval += val;
    }
  /* OK, now deal with chows. */
  if ( is_suit(t) ) {
    int a1=0,a2=0,b1=0,b2=0;

    /* are the adjacent tiles there? */
    if ( value_of(t) < 9 ) for (i=0;i<MAX_CONCEALED+1;i++) if ( tp[i] == t+1 ) a1++;
    if ( value_of(t) > 1 ) for (i=0;i<MAX_CONCEALED+1;i++) if ( tp[i] == t-1 ) b1++;
    if ( value_of(t) < 8 ) for (i=0;i<MAX_CONCEALED+1;i++) if ( tp[i] == t+2 ) a2++;
    if ( value_of(t) > 2 ) for (i=0;i<MAX_CONCEALED+1;i++) if ( tp[i] == t-2 ) b2++;
      
    /* is it part of a chow? */
    if ( (a1 && a2) || (b2 && b1) || (a1 && b1) ) {
      val = 0.0;
      /* A chow is deemed to be worth 12 points also.
      */
      val += strategies[curstrat][chowbase];
#if DEBUGEVAL
      printf("Ch%s:%4.1f+ ",tile_code(t),val);
#endif
      mval += val;
    } else if ( a1 || b1) {
      /* inner sequence */
      val = 0.0;
      if ( a1 && value_of(t) < 8 )
	val += strategies[curstrat][seqbase] + strategies[curstrat][partchow]*strategies[curstrat][chowbase] * tilesleft[t+2];
      if ( a1 && value_of(t) > 1 )
	val += strategies[curstrat][seqbase] + strategies[curstrat][partchow]*strategies[curstrat][chowbase] * tilesleft[t-1];
      if ( b1 && value_of(t) > 2 )
	val += strategies[curstrat][seqbase] + strategies[curstrat][partchow]*strategies[curstrat][chowbase] * tilesleft[t-2];
      if ( b1 && value_of(t) < 9 )
	val += strategies[curstrat][seqbase] + strategies[curstrat][partchow]*strategies[curstrat][chowbase] * tilesleft[t+1];
#if DEBUGEVAL
      printf("In%s:%4.1f+ ",tile_code(t),val);
#endif
      mval += val;
    } else if ( a2 || b2 ) {
      /* outer sequence */
      val = 0.0;
      if ( a2 )
	val += strategies[curstrat][seqbase] + strategies[curstrat][partchow]*strategies[curstrat][chowbase] * tilesleft[t+1];
      if ( b2 )
	val += strategies[curstrat][seqbase] + strategies[curstrat][partchow]*strategies[curstrat][chowbase] * tilesleft[t-1];
#if DEBUGEVAL
      printf("Ou%s:%4.1f+ ",tile_code(t),val);
#endif
      mval += val;
    }
  }
  val = 0.0;
  val += strategies[curstrat][sglbase]*tilesleft[t];
  if ( is_doubler(t) ) val += tilesleft[t] /* -1.0 */;
#if DEBUGEVAL
  printf("Si%s:%4.1f+ ",tile_code(t),val);
#endif
  mval += val;
#if DEBUGEVAL
  printf("= %4.1f\n",mval);
#endif
  return mval;
}

/* This function uses the above to evaluate a player's hand, including
   the completed sets. */
static double evalhand(PlayerP p, int strat) {
  Tile tcopy[MAX_CONCEALED+1];
  double val;
  int i, npr, ninc;

  for (i=0; i<p->num_concealed; i++)
    tcopy[i] = p->concealed[i];
  for ( ; i < MAX_CONCEALED+1; i++)
    tcopy[i] = HiddenTile;

  val = 0.0; 

  ninc = npr = 0; /* number of "incomplete"/pairs in hand */
  for (i=0; i<5; i++) {
    switch (p->tilesets[i].type) {
    case Chow:
      val -= strategies[strat][exposedchowpenalty];
    case ClosedChow:
      val += strategies[strat][chowbase];
      break;
    case Pung:
      val -= strategies[strat][exposedpungpenalty];
    case ClosedPung:
      val += strategies[strat][pungbase];
      break;
    case Kong:
      val -= strategies[strat][exposedpungpenalty];
    case ClosedKong:
      val += strategies[strat][pungbase];
      val += strategies[strat][kongbonus];
      break;
    case Pair: 
    case ClosedPair:
      ninc = npr = 1; /* for correct evaluation of pairs */
      val += strategies[strat][pairbase];
      break;
    default:
      ;
    }
  }
  val += eval(tcopy,strat,0,ninc,npr);
  return val*strategies[strat][weight];
}

/* compute the number of ways a calling hand can go out */
static int chances_to_win(PlayerP p) {
  Tile t;
  int n;

  t = HiddenTile;
  n = 0;
  while ( (t = tile_iterate(t,0)) != HiddenTile ) {
    if ( tilesleft[t] && player_can_mah_jong(p,t) )
      n += tilesleft[t];
  }
  return n;
}

/* compute tile to discard. Return value is tile.
   Second arg returns score of resulting hand.
   Third arg returns the new strategy.
*/
static Tile decide_discard(PlayerP p, double *score, int *newstrat) {
    /* how do we choose a tile to discard? 
       remove the tile, and evaluate the remaining hand.
       Discard the tile giving the greatest residual value.
    */
    double values[MAX_CONCEALED+1]; Tile tilesa[MAX_CONCEALED+1],tiles[MAX_CONCEALED+1] UNUSED;
    int i, best;
    int ostrat;
    int strat = curstrat;
    int dofast;
    double sval[NumStrat];
    char buf[80];
    Tile *t = p->concealed;


    for (i=0;i<MAX_CONCEALED+1;i++) values[i]=0;
    for (i=0;i<p->num_concealed;i++) tilesa[i] = t[i];
    for (;i<MAX_CONCEALED+1;i++) tilesa[i] = HiddenTile;

    /* strategy switching only after drawing tile from wall */
    if ( the_game->info.whence != FromDiscard ) {
      /* do we want to switch stragies ? */
      ostrat = strat;
      for ( i = Def ; i < NumStrat ; i++ ) {
	strat = i ;
	sval[i] = evalhand(p,strat);
      }
      strat = ostrat;
      /* if some other player has four sets declared, then switch to
	 Fast */
      dofast = 0;
      for ( i = 0 ; i < NUM_SEATS ; i++ ) {
	PlayerP p = the_game->players[i];
	if ( p == p ) continue;
	if ( p->num_concealed <= 1 ) dofast = 1;
      }
      if ( dofast ) {
	strat = Fast;
	if ( debugeval ) { printf("Using fast\n"); }
      } else {
	for ( i = Def ; i < NumStrat;i++) {
	  if ( (sval[i] > sval[ostrat] + hysteresis)
	       && sval[i] > sval[strat] )
	    if ( i != Concealed || sval[i] > 60.0 )
	      strat = i;
	}
      }
      if ( debugeval ) {
	if ( strat == Def && ostrat != Def )
	  printf("switching back to default");
	{      static char *ss[] = { "default","punger","chower", "conc", "fast" };
	for ( i = Def ; i < NumStrat ; i++ ) {
	  printf("%s %.02f ; ",ss[i],sval[i]);
	}
	printf("\n");
	if ( strat != ostrat ) {
	  printf("Switching to %s: ",ss[strat]);
	}
	}
      }
    }

    if ( debugeval ) {
      player_print_tiles(buf,p,0);
      printf("Hand: %s\n",buf);
    }
    best = 0;
    for (i=0;i<p->num_concealed;i++) {
      if ( debugeval ) {
	if ( easyeval ) printf("Trying %s: ",tile_code(tilesa[i]));
	else printf("Trying %s: \n",tile_code(tilesa[i]));
      }
      if ( easyeval ) {
	values[i] = eval_tile(tilesa,tilesa[i]);
	/* best tile to discard has the lowest score */
	if ( values[i] < values[best] ) best = i;
      } else {
	Player cp;
	copy_player(&cp,p);
	player_discards(&cp,tilesa[i]);
	values[i] = evalhand(&cp,strat);
	/* Best tile to discard leaves highest residual score */
	if ( values[i] > values[best] ) best = i;
      }
      if ( debugeval ) {
	if ( !easyeval )
	  printf("Tile %s has value %.1f\n",tile_code(tilesa[i]),values[i]);
      }
    }
    if ( debugeval ) {
      printf("Discarding %s\n",tile_code(tilesa[best]));
    }
    if ( score ) *score = values[best];
    if ( newstrat ) *newstrat = strat;
    return tilesa[best];
}
