/* $Header: /home/jcb/newmj/RCS/scoring.c,v 10.6 2001/02/10 21:32:37 jcb Exp $
 * scoring.c 
 * This file contains the routines to handle the scoring
 * of hands.
 */
/****************** COPYRIGHT STATEMENT **********************
 * This file is Copyright (c) 2000 by J. C. Bradfield.       *
 * Distribution and use is governed by the LICENCE file that *
 * accompanies this file.                                    *
 * The moral rights of the author are asserted.              *
 *                                                           *
 ***************** DISCLAIMER OF WARRANTY ********************
 * This code is not warranted fit for any purpose. See the   *
 * LICENCE file for further information.                     *
 *                                                           *
 *************************************************************/

static const char rcs_id[] = "$Header: /home/jcb/newmj/RCS/scoring.c,v 10.6 2001/02/10 21:32:37 jcb Exp $";

#include "tiles.h"
#include "player.h"
#include "controller.h"
#include "scoring.h"
#include "sysdep.h"

int no_special_scores = 0; /* disable scoring of flowers and seasons */

int limit = 1000; /* this will be an option */

/* score_of_tileset: takes a TileSetP, and returns its inherent score.
   It needs to know the wind of this player, and the prevailing wind. */
static Score score_of_tileset(TileSetP tp, TileWind own, TileWind prevailing) {
  static char buf[100];
  char tmp[50] = "";
  Score s;
  int n,d;

  buf[0] = '\000';

  if ( tp->type == Empty ) {
    /* I don't think this should happen */
    s.value = 0;
    s.explanation = "";
    warn("score_of_tileset: called on Empty tileset");
    return s;
  }

  d = is_major(tp->tile) ? 2 : 1;

  n = 0;
  switch ( tp->type ) {
  case Chow:
  case ClosedChow:
    n = 0; break;
  case Pung:
    n = 2*d;
    strcat(tmp,"open pung of ");
    strcat(tmp, (d==2) ? "majors" : "minors");
    break;
  case ClosedPung:
    n = 4*d;
    strcat(tmp,"closed pung of ");
    strcat(tmp, (d==2) ? "majors" : "minors");
    break;
  case Kong:
    n = 8*d;
    strcat(tmp,"open kong of ");
    strcat(tmp, (d==2) ? "majors" : "minors");
    break;
  case ClosedKong:
    n = 16*d;
    strcat(tmp,"closed kong of ");
    strcat(tmp, (d==2) ? "majors" : "minors");
    break;
  case Pair:
  case ClosedPair:
    n = 0;
    if ( is_dragon(tp->tile) ) {
      n = 2;
      strcat(tmp,"pair of dragons");
    } else if ( is_wind(tp->tile)
		&& (TileWind)value_of(tp->tile) == own ) {
      if ( own == prevailing ) {
	n = 4;
	strcat(tmp,"pair of own and prevailing wind");
      } else {
	n = 2;
	strcat(tmp,"pair of own wind");
      }
    } else if ( is_wind(tp->tile)
		&& (TileWind)value_of(tp->tile) == prevailing ) {
      n = 2;
      strcat(tmp,"pair of prevailing wind");
    }
    break;
  case Empty: break; /* to suppress warning */
  }
  s.value = n;
  s.explanation = "";
  if ( n > 0 ) {
    sprintf(buf,"%-12s %2d (%s)",tileset_string(tp),n,tmp);
    s.explanation = buf;
  }
  return s;
}

/* score_of_hand: score the given seat in the given game.
   In addition, set the danger flags to indicate if the player
   achieved any of the dangerous hands. */
Score score_of_hand(Game *g, seats s) {
  Score tot,tmp;
  static char buf[1000];
  char tbuf[100];
  PlayerP p;
  int winner;
  int i,doubles;
  Tile eyestile; int claimedpair;
  int numchows,numpungs,numkongs,allclosed,noscore,allhonours,
    allmajors,allterminals,dragonsets,windsets,dragonpairs,windpairs,
    numclosedpks,allgreen,allbamboo,allcharacter,allcircle,
    allbamboohonour,allcharacterhonour,allcirclehonour;

  
  p = g->players[s];
  assert(g->state == MahJonging);

  buf[0] = '\000';
  tot.explanation = buf;
  tot.value = 0;

  /* variables for detecting special hands */
  numchows = 0;
  numkongs = 0;
  numpungs = 0;
  allclosed = 1;
  noscore = 1;
  allterminals = 1;
  allhonours = 1;
  allmajors = 1;
  windsets = 0;
  dragonsets = 0;
  windpairs = 0;
  dragonpairs = 0;
  allgreen = 1;
  allbamboo = 1;
  allcharacter = 1;
  allcircle = 1;
  allbamboohonour = 1;
  allcharacterhonour = 1;
  allcirclehonour = 1;
  numclosedpks = 0; /* closed pungs/kongs */
  eyestile = HiddenTile; claimedpair = 0;

  /* Basic points */
  winner = 0;
  if ( s == g->info.player ) {
    sprintf(buf,"%-12s %2d (%s)\n","basic",20,"going Mah Jong");
    tot.value += 20;
    winner = 1;
  }

  if ( winner ) { 
    presetdflags(p,s);
    psetdflags(p,s,DangerEnd);
  }

  /* Now, get the points for the tilesets */
  /* Also, stash information for doubles */
  for ( i = 0; i < MAX_TILESETS; i++ ) {
    TileSetP t = (TileSetP) &p->tilesets[i];

    if ( t->type == Empty ) continue;

    tmp = score_of_tileset(t,
			   p->wind,
			   g->round);

    if ( tmp.value ) {
      noscore = 0;
      tot.value += tmp.value;
      strcat(buf,tmp.explanation);
      strcat(buf,"\n");
    }
    
    if ( t->type == Chow )
      numchows++,allclosed=0,allterminals=allhonours=allmajors=0;
    if ( t->type == ClosedChow )
      numchows++,allterminals=allhonours=allmajors=0;
    if ( t->type == Pung ) allclosed=0,numpungs++;
    if ( t->type == ClosedPung ) numpungs++,numclosedpks++;
    if ( t->type == Kong ) numkongs++,allclosed=0;
    if ( t->type == ClosedKong ) numkongs++,numclosedpks++;
    if ( t->type == Pair ) {
      allclosed=0;
      claimedpair = 1 ;
      eyestile = t->tile;
    }
    if ( t->type == ClosedPair ) {
      eyestile = t->tile;
    }

    if ( num_tiles_in_set(t) >= 3 ) {
      dragonsets += is_dragon(t->tile);
      windsets += is_wind(t->tile);
    } else {
      dragonpairs += is_dragon(t->tile);
      windpairs += is_wind(t->tile);
    }

    switch ( suit_of(t->tile) ) {
    case BambooSuit:
      allcharacter = allcharacterhonour = 0;
      allcircle = allcirclehonour = 0;
      switch ( t->type ) {
      case Chow:
      case ClosedChow:
	if ( value_of(t->tile) != 2 ) allgreen = 0;
	break;
      default:
	switch ( value_of(t->tile) ) {
	case 1:
	case 5:
	case 7:
	case 9:
	  allgreen = 0;
	default:
	  ;
	}
      }
      break;
    case CharacterSuit:
      allbamboo = allbamboohonour = 0;
      allcircle = allcirclehonour = 0;
      allgreen = 0;
      break;
    case CircleSuit:
      allbamboo = allbamboohonour = 0;
      allcharacter = allcharacterhonour = 0;
      allgreen = 0;
      break;
    case DragonSuit:
      allbamboo = allcharacter = allcircle = 0;
      if ( value_of(t->tile) != GreenDragon ) allgreen = 0;
      break;
    case WindSuit:
      allbamboo = allcharacter = allcircle = 0;
      allgreen = 0;
      break;
    default:
      warn("Strange suit seen in hand");
    }

    if ( !is_honour(t->tile) ) allhonours = 0;
    if ( !is_terminal(t->tile) ) allterminals = 0;
    if ( !is_major(t->tile) ) allmajors = 0;
  }

  /* Now points for flowers and seasons (which don't count
     to stop a noscore hand) */
  if ( p->num_specials > 0 && ! no_special_scores ) {
    tot.value += 4*p->num_specials;
    sprintf(tbuf,"%d %-10s %2d (%s)\n",p->num_specials,
	    p->num_specials > 1 ? "specials" : "special",
	    4*p->num_specials,"flowers and seasons");
    strcat(buf,tbuf);
  }

  /* odds and sods */
  if ( winner
       && g->info.whence == FromWall ) {
    tot.value += 2;
    sprintf(tbuf,"%-12s %2d (%s)\n","extra",2,"winning from wall");
    strcat(buf,tbuf);
  }

  /* fishing the eyes: if we claimed a pair to go out, then
     the tile had better match the discard.
     Otherwise, the tiles of the (necessarily closed) pair
     had better match the tile drawn from the wall.
  */
  if ( winner && eyestile == g->info.tile
       && ( claimedpair 
	    || g->info.whence == FromWall || g->info.whence == FromLoose ) ) {
    tot.value += 2;
    sprintf(tbuf,"%-12s %2d (%s)\n","extra",
	    is_major(eyestile) ? 4 : 2,"fishing the eyes");
    strcat(buf,tbuf);
  }

  /* this ridiculous amount of code deals with filling the only place.
     All this work for two points... */
  /* this is a sanity check; if this fails, the controller code
     has not put the right player in the caller slot */
  if ( winner ) {
    if ( p->id != gextras(g)->caller->id ) {
      warn("Wrong player found in caller slot of game structure!");
    } else {
      PlayerP pp = gextras(g)->caller;
      int n = 0;
      int i; Tile t;
      /* see how many tiles can complete the hand in theory */
      t = HiddenTile;
      while ( (t = tile_iterate(t,0)) != HiddenTile ) {
	/* if all four copies of the tile are exposed, it 
	   doesn't count as available */
	if ( g->exposed_tile_count[t] == 4 ) continue;
	i = player_can_mah_jong(pp,t);
	if ( i < 0 ) { warn("error in player_can_mah_jong while checking only place") ; }
	else n += i;
      }
      if ( n <= 1 ) {
	tot.value += 2;
	sprintf(tbuf,"%-12s %2d (%s)\n","extra",
		2,"filling the only place");
	strcat(buf,tbuf);
      }
    }
  }

  /* display the total points so far */
  sprintf(tbuf,"%-11s %3d\n","total pts",tot.value);
  strcat(buf,tbuf);

  /* convenience defn: braces to make it look like a fn */
#define dbl(n,ex) { doubles += n; sprintf(tbuf,"%-12s %2d (%s)\n", \
"doubles:",n,ex); strcat(buf,tbuf); }

  /* Now start looking for doubles from the tilesets */
  doubles = 0;
  for ( i = 0; i < MAX_TILESETS; i++ ) {
    TileSetP t = (TileSetP) &p->tilesets[i];

    if ( t->type == Empty ) continue;

    if ( num_tiles_in_set(t) >= 3 ) {
      if ( is_dragon(t->tile) ) dbl(1,"pung/kong of dragons");
      if ( is_wind(t->tile) && (TileWind)value_of(t->tile) == p->wind )
	dbl(1,"pung/kong of own wind");
      if ( is_wind(t->tile) && (TileWind)value_of(t->tile) == g->round )
	dbl(1,"pung/kong of prevailing wind");
    }
  }

/* We may want to suppress these when comparing strategies,
   as they introduce a lot of randomness */
  /* Flower and season doubles */
  if ( ! no_special_scores ) {
    int numown = 0;
    int numflowers = 0;
    int numseasons = 0;
    int i;

    for ( i = 0; i < p->num_specials; i++ ) {
      if ( suit_of(p->specials[i]) == FlowerSuit ) numflowers++;
      if ( suit_of(p->specials[i]) == SeasonSuit ) numseasons++;
      if ( (TileWind)value_of(p->specials[i]) == p->wind ) numown++;
    }
    if ( numown == 2 ) dbl(1,"own flower and season");
    if ( numflowers == 4 ) dbl(1,"all four flowers");
    if ( numseasons == 4 ) dbl(1,"all four seasons");
  }


  /* doubles applying to all hands */
  /* Little/Big Three Dragons */
  if ( dragonsets == 3 ) {
    dbl(2,"Big Three Dragons");
    if ( winner ) psetdflags(p,s,DangerDragon);
  } else if ( dragonsets == 2 && dragonpairs ) {
    dbl(1,"Little Three Dragons");
    if ( winner ) psetdflags(p,s,DangerDragon);
  }
  /* Little/Big Four Joys */
  if ( windsets == 4 ) {
    dbl(2,"Big Four Joys");
    if ( winner ) psetdflags(p,s,DangerWind);
  } else if ( windsets == 3 && windpairs ) {
    dbl(1,"Little Four Joys");
    if ( winner ) psetdflags(p,s,DangerWind);
  }

  /* three concealed pungs */
  if ( numclosedpks >= 3 ) { dbl(1,"three concealed pungs"); }

  /* other doubles applying only to the mahjong hand */
  if ( winner ) {
    if ( numchows == 0 ) dbl(1,"no chows");
    if ( noscore ) dbl(1,"no score hand");
    if ( allhonours ) {
      dbl(2,"all honours");
      psetdflags(p,s,DangerHonour);
    } else if ( allterminals ) {
      dbl(2,"all terminals");
      psetdflags(p,s,DangerTerminal);
    } else if ( allmajors ) {
      dbl(1,"all majors");
    }
    /* we may have claimed a discard for a special set */
    if ( g->info.whence == FromDiscard ) allclosed = 0;
    if ( allclosed ) dbl(1,"concealed hand");
    /* NB. If no suit tiles, then we already have all honours,
       and don't give another double! */
    if ( allbamboo || allcharacter || allcircle ) {
      dbl(3,"one suit only");
      if ( allbamboo ) psetdflags(p,s,DangerBamboo);
      if ( allcharacter ) psetdflags(p,s,DangerCharacter);
      if ( allcircle ) psetdflags(p,s,DangerCircle);
    } else if ( allbamboohonour || allcharacterhonour || allcirclehonour ) {
      dbl(1,"one suit with honours");
    }
    /* Should the following two be exclusive? */
    /* loose tile */
    if ( g->info.whence == FromLoose ) dbl(1,"winning with loose tile");
    /* last tile */
    if ( g->wall.live_used == g->wall.live_end ) dbl(1,"winning with last tile");
    /* robbing a kong */
    if ( g->info.whence == FromRobbedKong ) dbl(1,"robbing a kong");
    /* original call */
    if ( pflag(p,OriginalCall) ) dbl(1,"completing Original Call");
  }

  /* calculate the doubles */
  sprintf(tbuf,"%-12s %2d\n","total dbls",doubles);
  strcat(buf,tbuf);
  while ( doubles-- > 0 ) tot.value *= 2;
  if ( tot.value > limit ) {
    sprintf(tbuf,"%-9s %5d (over limit)\n","score:",tot.value);
    strcat(buf,tbuf);
    tot.value = limit;
  }

  /* having done all that work, we now look for limit hands.
     Note that all the danger signals except all green have already
     been set while we were looking for doubles */

  if ( winner ) {
    char *limname ;

    limname = NULL;
    if ( g->info.player == east && pflag(p,NoDiscard) ) {
      /* heaven's blessing */
      limname = "Heaven's Blessing";
    } else if ( g->info.player != east 
		&& pflag(p,NoDiscard)
		&& g->info.whence == FromDiscard ) {
      seats s;
      /* might be earth's blessing, but we need to check that
	 nobody else has discarded in between */
      limname = "Earth's Blessing";
      for ( s = south; s < NUM_SEATS; s++ ) {
	if ( !pflag(g->players[s],NoDiscard) ) limname = NULL;
      }
    } else if ( g->info.whence == FromLoose
		&& g->info.tile == make_tile(CircleSuit,5) ) {
      limname = "Gathering Plum Blossom from the Roof";
    } else if ( g->info.tile == make_tile(CircleSuit,1)
		&& g->wall.live_used == g->wall.live_end
		&& g->info.whence == FromWall ) {
      limname = "Catching the Moon from the Bottom of the Sea";
    } else if ( g->info.whence == FromRobbedKong
		&& g->info.tile == make_tile(BambooSuit,2) ) {
      limname = "Scratching a Carrying Pole";
    } else if ( game_flag(g,GFKongUponKong) ) {
      limname = "Kong upon Kong";
    } else if ( numkongs == 4 ) {
      limname = "Fourfold Plenty";
    } else if ( allclosed && numpungs+numkongs == 4 ) {
      limname = "Buried Treasure";
    } else if ( player_can_thirteen_wonders(p,HiddenTile) ) {
      limname = "Thirteen Unique Wonders";
    } else if ( dragonsets == 3 && numchows == 0 ) {
      limname = "Three Great Scholars";
    } else if ( windsets == 4 ) {
      limname = "Four Blessing o'er the Door";
    } else if ( allhonours ) {
      limname = "All Honours";
    } else if ( allterminals ) {
      limname = "Heads and Tails";
    } else if ( allclosed && (allbamboo || allcharacter || allcircle) ) {
      limname = "Concealed Clear Suit";
    } else if ( g->hands_as_east == 12 && g->info.player == east ) { /* sic */
      limname = "East's 13th consecutive win";
    } else if ( allgreen ) {
      limname = "Imperial Jade";
      psetdflags(p,s,DangerGreen);
    }
    /* Nine Gates */
    /* special hack here: a concealed Nine Gates is also an
       instance of Concealed Clear Suit, but people would much
       rather see the more exciting Nine Gates */
    if ( ! limname 
	 || ( allclosed && (allbamboo || allcharacter || allcircle) ) ) {
      char *l = "Nine Gates";
      PlayerP pp = gextras(g)->caller;
      
      if ( pp->id != p->id ) {
	warn("Wrong player found in caller slot of game structure!");
	l = NULL;
      } else {
	int i;
	if ( is_honour(g->info.tile) ) {
	  l = NULL;
	} else {
	  int s = suit_of(g->info.tile);
	  player_sort_tiles(pp);
	  if ( pp->num_concealed < 13 ) l = NULL;
	  for ( i = 0 ; i < 3 ; i++ ) {
	    if ( pp->concealed[i] != make_tile(s,1) ) l = NULL;
	  }
	  for ( i = 2 ; i < 11 ; i++ ) {
	    if ( pp->concealed[i] != make_tile(s,i-1) ) l = NULL;
	  }
	  for ( i = 11 ; i < 13 ; i++ ) {
	    if ( pp->concealed[i] != make_tile(s,9) ) l = NULL;
	  }
	}
      }
      if ( l ) limname = l;
    }
    /* The Wriggling Snake, another silly hand */
    if ( ! limname ) {
      int i;
      unsigned int s = suit_of(g->info.tile);
      int set1=0,set9=0,pair2=0,pair5=0,pair8=0,
	chow2=0,chow3=0,chow5=0,chow6=0;
      for ( i = 0 ; i < MAX_TILESETS ; i++ ) {
	if ( suit_of(p->tilesets[i].tile) != s ) continue;
	switch( p->tilesets[i].type ) {
	case Empty:
	  break;
	case Pung:
	case ClosedPung:
	case Kong:
	case ClosedKong:
	  set1 |= (value_of(p->tilesets[i].tile) == 1);
	  set9 |= (value_of(p->tilesets[i].tile) == 9);
	  break;
	case Pair:
	case ClosedPair:
	  pair2 |= (value_of(p->tilesets[i].tile) == 2);
	  pair5 |= (value_of(p->tilesets[i].tile) == 5);
	  pair8 |= (value_of(p->tilesets[i].tile) == 8);
	  break;
	case Chow:
	case ClosedChow:
	  chow2 |= (value_of(p->tilesets[i].tile) == 2);
	  chow3 |= (value_of(p->tilesets[i].tile) == 3);
	  chow5 |= (value_of(p->tilesets[i].tile) == 5);
	  chow6 |= (value_of(p->tilesets[i].tile) == 6);
	  break;
	}
      }
      if ( set1 && set9 &&
	   ( (pair2 && chow3 && chow6)
	     || (pair5 && chow2 && chow6)
	     || (pair8 && chow2 && chow5) ) ) limname = "Wriggling Snake";
    }	  
    
    if ( limname ) {
      sprintf(buf,"%-9s %5d (%s)\n","Limit",limit,limname);
      tot.value = limit;
    }
  }

  sprintf(tbuf,"%-9s %5d","SCORE:",tot.value);
  strcat(buf,tbuf);

  /* just return what we've got, to see if it's working */
  return tot;
}
