/* $Header: /home/jcb/newmj/RCS/sysdep.c,v 10.7 2001/01/29 17:46:17 jcb Exp $
 * sysdep.c
 * By intention, this file contains all functions that might need
 * fiddling with to cope with different variants of Unix.
 * In particular, all the networking code is in here.
 * Some parts of the other code assume POSIXish functions for file
 * descriptors and the like; this file is responsible for providing them
 * if they're not native. All such cases that occurred to me as I was
 * writing them can be found by searching for sysdep.c in the other
 * files.
 */
/****************** COPYRIGHT STATEMENT **********************
 * This file is Copyright (c) 2000 by J. C. Bradfield.       *
 * Distribution and use is governed by the LICENCE file that *
 * accompanies this file.                                    *
 * The moral rights of the author are asserted.              *
 *                                                           *
 ***************** DISCLAIMER OF WARRANTY ********************
 * This code is not warranted fit for any purpose. See the   *
 * LICENCE file for further information.                     *
 *                                                           *
 *************************************************************/

static const char rcs_id[] = "$Header: /home/jcb/newmj/RCS/sysdep.c,v 10.7 2001/01/29 17:46:17 jcb Exp $";

#include "sysdep.h"
#include <string.h>
#include <fcntl.h>

#ifndef WIN32
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/un.h>
/* this is modern, I think. */
#include <netdb.h>
#endif /* not WIN32 */

#include <stdlib.h>
#include <stdarg.h>
#include <time.h>
#include <signal.h>

/* warn: = fprintf(stderr,...), with automatic new line */
int warn(char *format,...) {
  va_list args;
  static char buf[256];
  int ret;
  int n;
  va_start(args,format);
  strncpy(buf,format,250);
  n = strlen(buf);
  /* do quick check for terminators */
#ifdef WIN32
  if ( buf[n-2] == '\r' ) {
    /* OK */
    ;
  } else if ( buf[n-1] == '\n' ) {
    /* already have unix terminator */
    buf[n-1] = '\r' ; buf[n] = '\n' ; buf[n+1] = 0 ;
  } else {
    /* no terminator */
    strcat(buf,"\r\n");
  }
#else
  if ( buf[n-1] != '\n' ) strcat(buf,"\n");
#endif
  ret = vfprintf(stderr,buf,args);
  va_end(args);
  return ret;
}

/* ignore the SIGPIPE signal. Return 0 on success, -1 on error */
int ignore_sigpipe(void) {
#ifdef WIN32
  return 0;
#else
  /* this is just posix at present */
  struct sigaction act;

  act.sa_handler = SIG_IGN;
  sigemptyset(&act.sa_mask);
  act.sa_flags = 0;
  return sigaction(SIGPIPE,&act,NULL);
#endif /* WIN32 */
}

/* functions to be applied by the socket routines */
static void *(*skt_open_transform)(SOCKET) = NULL;
static int (*skt_closer)(void *) = NULL;
static int (*skt_reader)(void *, void *, size_t) = NULL;
static int (*skt_writer)(void *, const void *,size_t) = NULL;
static int sockets_initialized = 0;

#ifdef WIN32
static void shutdown_sockets(void) {
  WSACleanup();
  /* I don't care what it returns */
}
#endif /* WIN32 */

/* initialize sockets */
int initialize_sockets(void *(*open_transform)(SOCKET),
		       int (*closer)(void *),
		       int (*reader)(void *, void *, size_t),
		       int (*writer)(void *, const void *,size_t)) {
  skt_open_transform = open_transform;
  skt_closer = closer;
  skt_reader = reader;
  skt_writer = writer;
#ifdef WIN32

  if ( ! sockets_initialized ) {
    WORD version;
    WSADATA data;
    int err;
    
    version = MAKEWORD( 2, 2 );
    
    err = WSAStartup( version, &data );
    if (err != 0) {
      return 0;
    }
    if ((LOBYTE( data.wVersion ) != 2) || (HIBYTE( data.wVersion ) != 2)) {
      WSACleanup();
      return 0;
    }
    if ( atexit(shutdown_sockets) != 0 ) {
      warn("couldn't register socket shutdown routine");
    }
  }
#endif /* WIN32 */

  sockets_initialized = 1;
  return 1;
}

/* this function takes a string and parses it as an address.
   It returns 0 for an Internet address, 1 for a Unix address.
   For Internet addresses, the host name and port are placed
   in the given variables (the string had better be long enough);
   for Unix, the file name is copied to the given variable.
*/
static int parse_address(const char *address,
			 char *ret_host,
			 unsigned short *ret_port,
			 char *ret_file) {
  if ( strchr(address,':') ) {
    /* grrr */
    if ( address[0] == ':' ) {
      strcpy(ret_host,"localhost");
      sscanf(address,":%hu",ret_port);
    } else {
      sscanf(address,"%[^:]:%hu",ret_host,ret_port);
    }
    return 0;
  } else {
    if ( ret_file) strcpy(ret_file,address);
    return 1;
  }
}

/* set_up_listening_socket:
   Set up a socket listening on the given address.
   Return its fd.
*/
SOCKET set_up_listening_socket(const char *address) {
  SOCKET sfd;
  struct protoent *prstruc = NULL;
  struct sockaddr_in inaddr;
#ifndef WIN32
  struct sockaddr_un unaddr;
#endif
  int unixsockets;
  unsigned short port;
  char name[256];
  int sockopt;

  if ( ! sockets_initialized ) initialize_sockets(NULL,NULL,NULL,NULL);

  unixsockets = 
#ifdef WIN32
    parse_address(address,name,&port,NULL);
#else
    parse_address(address,name,&port,unaddr.sun_path);
#endif /* WIN32 */

#ifdef WIN32
  if ( unixsockets ) {
    warn("unix sockets not available on Windows");
    return INVALID_SOCKET;
  }
#endif /* WINDOWS */

  if ( !unixsockets ) {
    prstruc = getprotobyname("tcp");
    if ( prstruc == NULL ) {
      perror("getprotobyname failed");
      return INVALID_SOCKET;
    }
  }
  sfd = socket(unixsockets ? AF_UNIX : AF_INET,
	       SOCK_STREAM,
	       unixsockets ? 0 : prstruc->p_proto);
  if ( sfd == INVALID_SOCKET ) {
    perror("socket failed");
    return INVALID_SOCKET;
  }

  sockopt=1;
  setsockopt(sfd,SOL_SOCKET,SO_REUSEADDR,(void *)&sockopt,sizeof(int));

#ifndef WIN32
  if ( unixsockets ) {
    unaddr.sun_family = AF_UNIX;
    unlink(unaddr.sun_path); /* need to check success FIXME */
    if ( bind(sfd,(struct sockaddr *)&unaddr,sizeof(struct sockaddr_un)) < 0 ) {
      perror("bind failed");
      return INVALID_SOCKET;
    }
  } else {
#else
  if ( 1 ) {
#endif /* WIN32 */
    inaddr.sin_family = AF_INET;
    inaddr.sin_port = htons(port);
    inaddr.sin_addr.s_addr = INADDR_ANY;
    if ( bind(sfd,(struct sockaddr *)&inaddr,sizeof(struct sockaddr_in)) < 0 ) {
      perror("bind failed");
      return INVALID_SOCKET;
    }
  }

  if ( listen(sfd,5) < 0 ) {
    perror("listen failed");
    return INVALID_SOCKET;
  }
  
  if ( skt_open_transform ) {
    return (SOCKET) skt_open_transform(sfd);
  } else {
    return sfd;
  }
}
  
  

/* accept_new_connection:
   A connection has arrived on the socket fd.
   Accept the connection, and return the new fd,
   or INVALID_SOCKET on error.
*/

SOCKET accept_new_connection(SOCKET fd) {
  SOCKET newfd;
  struct sockaddr saddr;
  struct linger l = { 1, 1000 } ; /* linger on close for 10 seconds */
  int saddrlen = sizeof(saddr);

  if ( ! sockets_initialized ) initialize_sockets(NULL,NULL,NULL,NULL);

  newfd = accept(fd,&saddr,&saddrlen);
  if ( newfd == INVALID_SOCKET ) { perror("accept failed"); }
  /* we should make sure that data is sent when this socket is closed */
  if ( setsockopt(newfd,SOL_SOCKET,SO_LINGER,(void *)&l,sizeof(l)) < 0 ) {
    warn("setsockopt failed");
  }
  if ( skt_open_transform ) {
    return (SOCKET) skt_open_transform(newfd);
  } else {
    return newfd;
  }
}

/* connect_to_host:
   Establish a connection to the given address.
   Return the file descriptor or INVALID_SOCKET on error.
   If unixsockets, only the port number is used in making the name
   The returned socket is marked close-on-exec .
*/

SOCKET connect_to_host(const char *address) {
  SOCKET fd;
  struct sockaddr_in inaddr;
#ifndef WIN32
  struct sockaddr_un unaddr;
#endif
  char name[512];
  int unixsockets;
  unsigned short port;
  struct hostent *hent = NULL;
  struct protoent *prstruc = NULL;

  if ( ! sockets_initialized ) initialize_sockets(NULL,NULL,NULL,NULL);

#ifdef WIN32
  unixsockets = parse_address(address,name,&port,NULL);
  if ( unixsockets ) {
    warn("Unix sockets not supported on Windows");
    return INVALID_SOCKET;
  }
#else
  unixsockets = parse_address(address,name,&port,unaddr.sun_path);
#endif /* WIN32 */

  if ( !unixsockets ) {
    hent = gethostbyname(name);
    
    if ( ! hent ) {
      perror("connect_to_host: gethostbyname failed");
      return INVALID_SOCKET;
    }
    
    prstruc = getprotobyname("tcp");
    if ( prstruc == NULL ) {
      perror("connect_to_host: getprotobyname failed");
      return INVALID_SOCKET;
    }
  }

  fd = socket(unixsockets ? AF_UNIX : AF_INET,
	      SOCK_STREAM,
	      unixsockets ? 0 : prstruc->p_proto);
  if ( fd == INVALID_SOCKET ) {
    perror("connect_to_host: socket failed");
    return INVALID_SOCKET;
  }

#ifndef WIN32
  if ( unixsockets ) {
    unaddr.sun_family = AF_UNIX;
    if ( connect(fd,(struct sockaddr *)&unaddr,sizeof(struct sockaddr_un)) < 0 ) {
      perror("connect_to_host: connect failed");
      return INVALID_SOCKET;
    }
  } else {
#else
  if ( 1 ) {
#endif /* WIN32 */
    inaddr.sin_family = AF_INET;
    inaddr.sin_port = htons(port);
    inaddr.sin_addr = *((struct in_addr *)hent->h_addr);
    if ( connect(fd,(struct sockaddr *)&inaddr,sizeof(struct sockaddr_in)) < 0 ) {
      perror("connect_to_host: connect failed");
      return INVALID_SOCKET;
    }
  }
  
#ifndef WIN32
  /* mark close on exec */
  fcntl(fd,F_SETFD,1);
#endif /* WIN32 */

  if ( skt_open_transform ) {
    return (SOCKET) skt_open_transform(fd);
  } else {
    return fd;
  }
}

/* close a network connection */
int close_socket(SOCKET s) {
  if ( skt_closer ) {
    return skt_closer((void *)s);
  } else {
    return closesocket(s);
  }
}


/* read a line, up to lf/crlf terminator, from the socket,
   and return it.
   If the line ends with an unescaped backslash, replace by newline,
   and append the next line.
   If there is an error or end of file before seeing a terminator,
   NULL is returned. At present, error includes reading a partial line.
   The returned string is valid until the next call of get_line.
*/
char *get_line(SOCKET fd) {
  int length;
  int offset=0;
  static char buffer[1024];

  while ( 1 ) {
    while ( ( (length
	       = (skt_reader ? skt_reader((void *)fd,buffer+offset,1)
		  : 
#ifdef WIN32
		  recv(fd,buffer+offset,1,0)
#else
		  read(fd,buffer+offset,1)
#endif
		  )) > 0 )
	    && buffer[offset]!='\n')
      offset += length;
    if ( offset == 0 ) return NULL;
    if ( length <= 0 ) {
      fprintf(stderr,"get_line read partial line");
      return NULL;
    }
    offset += length;
    buffer[offset] = '\0';
    /* now check for newline */
    if ( buffer[offset-1] == '\n' ) {
      if ( buffer[offset-2] == '\r' && buffer[offset-3] == '\\' ) {
        /* zap the CR */
	buffer[offset-2] = '\n';
	buffer[offset-1] = '\0';
	offset -= 1;
        /* and go onto next clause */
      }
      if ( buffer[offset-2] == '\\' ) {
	/* we have to decide whether this is an escaped backslash */
	int j = 1,k=offset-3;
	while ( k >= 0 && buffer[k] == '\\' ) j++;
	if ( j & 1 ) {
	  /* odd number, not escaped */
	  buffer[offset-2] = '\n' ;
	  buffer[--offset] = '\0' ;
	  /* read another line */
	} else break; /* return this line */
      } else break; /* return this line */
    } /* no newline, keep going */
  }
  return buffer;
}

/* write a line (which is assumed to have any terminator included)
   to the given socket. Return -1 on error, length of line on success.
   Replace internal newlines by backslash CRLF and write as lines.
*/
int put_line(SOCKET fd,char *line) {
  int length,n;
  int nn = 0; /* return value */
  char *newline;
  char buffer[1024];

  /* FIXME: should retry if only write partial line */
  while ( (length = strlen(line)) > 0 ) {
    buffer[0] = '\0';
    newline = strchr(line,'\n');
    if ( newline && newline[1] != '\0' ) {
      strncpy(buffer,line,length=(newline-line));
      buffer[length] = '\0';
      strcat(buffer,"\\\r\n");
      length += 3;
      line = newline+1;
    } else {
      strcpy(buffer,line);
      line += length;
    }
    n = skt_writer ? skt_writer((void *)fd,buffer,length)
      :
#ifdef WIN32
      send(fd,buffer,length,0)
#else
      write(fd,buffer,length)
#endif
      ;
    if ( n < length ) {
      perror("put_line: write failed");
      return -1;
    }
    nn += n;
  }
  return nn;
}

/* random integer from 0 to top inclusive */
static int seed = 0;

unsigned int rand_index(int top) {

  if ( seed == 0 ) {
    /* seed the generator */
    rand_seed(0);
  }

  return (unsigned int)((1.0+top)*rand()/(RAND_MAX+1.0));
}

/* and seed it */
void rand_seed(int s) {
  if ( s == 0 ) s = time(NULL);
  srand(seed = s);
}

/* feed a command to the system to be started in background.
   No fancy argument processing, just pass to shell or equiv.
   Return 1 on success, 0 on failure. */
int start_background_program(const char *cmd) {
  char buf[1024];
# ifdef WIN32
  int res;
  STARTUPINFO si;
  PROCESS_INFORMATION pi;
  
  strncpy(buf,cmd,1023); buf[1023] = 0;
  memset((void *)&si,0,sizeof(STARTUPINFO));
  si.cb = sizeof(STARTUPINFO);
  res = CreateProcess(NULL,buf,NULL,NULL,
		0,DETACHED_PROCESS,NULL,NULL,&si,&pi);
  if ( res ) {
    CloseHandle(pi.hProcess);
    CloseHandle(pi.hThread);
  } else {
    warn("Failed to start process %s",cmd);
  }
  return res;
# else
  strcpy(buf,cmd);
  strcat(buf," &");
# endif
  return ! system(buf);
}

/* stuff for Windows */
#ifdef WIN32
int gettimeofday(struct timeval *tv, void *tz UNUSED) {
  long sec, usec;
  LONGLONG ll;
  FILETIME ft;

  GetSystemTimeAsFileTime(&ft);
  ll = ft.dwHighDateTime;
  ll <<= 32;
  ll |= ft.dwLowDateTime;
  /* this magic number comes from Microsoft's knowledge base ' */
  ll -= 116447360000000000;  /* 100 nanoseconds since 1970 */
  ll /= 10; /* microseconds since 1970 */
  sec = (long)(ll / 1000000);
  usec = (long)(ll - 1000000*((LONGLONG) sec));
  if ( tv ) {
    tv->tv_sec = sec;
    tv->tv_usec = usec;
  }
  return 0;
}

char *getlogin(void) { 
  static char buf[128];
  DWORD len = 128;

  buf[0] = 0;
  GetUserName(buf,&len);
  return buf;
}
 
int getpid(void) {
  return GetCurrentProcessId();
}

unsigned int sleep(unsigned int t) { Sleep(1000*t); return 0; }

/* I don't think we really need submillisecond resolution ... */
unsigned int usleep(unsigned int t) { Sleep(t/1000); return 0; }

#endif /* WIN32 */
