/* $Header: /home/jcb/newmj/RCS/controller.c,v 10.19 2001/02/10 21:36:02 jcb Exp $
 * controller.c
 * This program implements the master controller, which accepts
 * connections from players and runs the game.
 * At present, this is designed around the assumption that it is
 * only running one game. Realistically, I see no immediate need
 * to be more general. However, I trust that nothing in the design
 * would make that awkward.
 */
/****************** COPYRIGHT STATEMENT **********************
 * This file is Copyright (c) 2000 by J. C. Bradfield.       *
 * Distribution and use is governed by the LICENCE file that *
 * accompanies this file.                                    *
 * The moral rights of the author are asserted.              *
 *                                                           *
 ***************** DISCLAIMER OF WARRANTY ********************
 * This code is not warranted fit for any purpose. See the   *
 * LICENCE file for further information.                     *
 *                                                           *
 *************************************************************/

static const char rcs_id[] = "$Header: /home/jcb/newmj/RCS/controller.c,v 10.19 2001/02/10 21:36:02 jcb Exp $";

#include <stdio.h>
#include <assert.h>
#include "controller.h"
#include "scoring.h"

#include "sysdep.h"

#include "version.h"

/* extra data in players */
typedef struct {
  /* option settings */
  int options[PONumOptions];
  /* this variable is set to one to indicate that the player
     has been disconnected or otherwise out of touch, and therefore needs
     to be told the current state of the game. */
  int disconnected;
  /* array of pointers to CMsgs issued in this hand. This is
     used when reconnecting. The array is 512, since the maximum
     number of messages that can (relevantly) be issued in a hand
     is around 4 per tile, but that's highly unlikely ever to be
     reached. This ought to be dynamic... */
  int histcount;
  CMsgMsg *history[512];
} PlayerExtras;

#define popts(p) ((PlayerExtras *)(p->userdata))->options
#define pextras(p) ((PlayerExtras *)(p->userdata))

/* this is The Game that we will use */

Game *the_game = NULL;

int num_connected_players;
#define NUM_PLAYERS 4

/* This array stores information about current connections.
   When a new connection arrives, we use the first free slot.
*/
#define MAX_CONNECTIONS 8
struct {
  int inuse; /* slot in use */
  SOCKET skt; /* the system level socket of this connection. */
  PlayerP player; /* player, if any */
  int seqno; /* sequence number of messages on this connection */
} connections[MAX_CONNECTIONS];


/* This is the fd of the socket on which we listen */
SOCKET socket_fd;

/* This is the master set of fds on which we are listening
   for events. If we're on some foul system that doesn't have
   this type, it is sysdep.h's responsibility to make sure we do.
*/

fd_set event_fds;

/* this is used by auxiliary functions to pass back an error message */
char *failmsg;

/* This is a logfile, if specified */
FILE *logfile = NULL;

/* forward declarations */
static void despatch_line(int cnx, char *line);
static int close_connection(int cnx);
static int new_connection(SOCKET skt); /* returns new connection number */
static void setup_maps(int cnx, PlayerP p);
static void remove_from_maps(int cnx);
static void send_packet(int cnx,CMsgMsg *m, int logit); /* not much used except in next two */
static void _send_id(int id,CMsgMsg *m, int logit);
static void _send_all(Game *g,CMsgMsg *m);
static void _send_others(Game *g, int id,CMsgMsg *m);
static void _send_seat(Game *g, seats s, CMsgMsg *m);
static void _send_other_seats(Game *g, seats s, CMsgMsg *m);
/* always casting is boring, so ... */
/* send_id logs (if logging is enabled */
#define send_id(id,m) _send_id(id,(CMsgMsg *)(m),1)
#define send_all(g,m) _send_all(g,(CMsgMsg *)(m))
#define send_others(g,id,m) _send_others(g,id,(CMsgMsg *)(m))
#define send_seat(g,s,m) _send_seat(g,s,(CMsgMsg *)(m))
#define send_other_seats(g,s,m) _send_other_seats(g,s,(CMsgMsg *)(m))
static int load_state(Game *g);
static int save_state(Game *g);

static int id_to_cnx(int id);
static int cnx_to_id(int cnx);

/* N.B. we do not use the game_id_to_player function, since
   we don't always have players in a game.
*/
static PlayerP id_to_player(int id);

/* A convenience definition. */
#define id_to_seat(id) game_id_to_seat(the_game,id)
/* and another */
#define handle_cmsg(g,m) game_handle_cmsg(g,(CMsgMsg *)m)

static int start_hand(Game *g);
static void check_claims(Game *g);
static void send_infotiles(PlayerP p);
static void score_hand(Game *g, seats s);
static void washout(char *reason);
static void handle_cnx_error(int cnx);

static int loadstate = 0; /* if we have to load state */
static int randomseats = 0; /* assign seats randomly */
static int debug = 0; /* allow debugging messages */
static int usehist = 1; /* if we keep history to allow resumption */
static int exit_on_disconnect = 0; /* if we die on losing a player */

static void usage(char *pname,char *msg) {
  fprintf(stderr,"%s: %s\nUsage: %s [ --server ADDR ]\
  [ --timeout N ]\n\
  [ --pause N ]\n\
  [ --random-seats ]\n\
  [ --debug ]\n\
  [ --exit-on-disconnect ]\n\
  [ --logfile FILE]\n\
  [ --no-special-scores ]\n\
  [ --seed N ]\n\
  [ --nohist ]\n",
	  pname,msg,pname);
  exit(1);
}

/* This global is used to specify a timeout for claims
   in milliseconds. If it is non-zero on entry to the select
   call, then it will be used as the timeout for select; and if
   data is read before timeout, it will be adjusted appropriately
   for the next call. If a select expires due to timeout, then
   the timeout_handler is called. */
static void timeout_handler(Game *g);
static int timeout = 0;
/* this is the user level timeout in seconds.*/
static int timeout_time = 15;
/* this is the minimum time between discards and draws,
   keep the game down to human speed */
static int min_time = 0; /* deciseconds */
/* and this is the value requested on the command line */
static int min_time_opt = 0;

int main(int argc, char *argv[]) {
  char *address = ":5000";
  char *logfilename = NULL;
  int seed = 0;
  int nfds; 
  struct timeval timenow, timethen, timetimeout;
  fd_set rfds, efds;
  int i;

  /* options. I should use getopt ... */
  for (i=1;i<argc;i++) {
    /* The load option is not currently available */
    /* if ( strcmp(argv[i],"--load") == 0 ) {
      loadstate = 1;
      } else */ if ( strcmp(argv[i],"--nohist") == 0 ) {
      usehist = 0;
    } else if ( strcmp(argv[i],"--debug") == 0 ) {
      debug = 1;
    } else if ( strcmp(argv[i],"--random-seats") == 0 ) {
      randomseats = 1;
    } else if ( strcmp(argv[i],"--exit-on-disconnect") == 0 ) {
      exit_on_disconnect = 1;
    } else if ( strcmp(argv[i],"--no-special-scores") == 0 ) {
      no_special_scores = 1;
    } else if ( strcmp(argv[i],"--address") == 0 ) {
      /* N.B. the --address option is retained only for backward compatibility */
      if ( ++i == argc ) usage(argv[0],"missing argument to --address");
      address = argv[i];
    } else if ( strcmp(argv[i],"--server") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --server");
      address = argv[i];
    } else if ( strcmp(argv[i],"--timeout") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --timeout");
      timeout_time = atoi(argv[i]);
    } else if ( strcmp(argv[i],"--pause") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --pause");
      min_time_opt = min_time = atoi(argv[i]); /* N.B. Time in deciseconds */
    } else if ( strcmp(argv[i],"--seed") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --seed");
      seed = atoi(argv[i]);
    } else if ( strcmp(argv[i],"--logfile") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --logfile");
      logfilename = argv[i];
    } else usage(argv[0],"unknown option or argument");
  }

  if ( logfilename ) {
    if ( strcmp(logfilename,"-") == 0 ) {
      logfile = stdout;
    } else {
      logfile = fopen(logfilename,"a");
      if ( ! logfile ) {
	perror("couldn't open logfile");
	exit(1);
      }
    }
  }

  /* we don't want to get SIGPIPE */
  ignore_sigpipe();

  socket_fd = set_up_listening_socket(address);
  if ( socket_fd == INVALID_SOCKET ) {
    perror("couldn't set up listening socket");
    exit(1);
  }

  /* stick it in the connection list */
  new_connection(socket_fd);
  
  /* seed the random number generator */
  rand_seed(seed);

  /* initialize data */
  num_connected_players = 0;

  /* malloc space for the game and four players.
     We will shortly load the state if we're resuming; if we're not
     resuming, initialization happens later */
  the_game = (Game *) malloc(sizeof(Game));
  if ( the_game == NULL ) {
    warn("Couldn't malloc game structure");
    exit(1);
  }
  memset((void *)the_game,0,sizeof(Game));
  the_game->userdata = malloc(sizeof(GameExtras));
  if ( the_game->userdata == NULL ) {
    warn("Couldn't malloc game extra structure");
    exit(1);
  }
  gextras(the_game)->caller = (PlayerP) malloc(sizeof(Player));
  if ( gextras(the_game)->caller == NULL ) {
    warn("Couldn't malloc game extra structure");
    exit(1);
  }


  for ( i=0 ; i < NUM_SEATS ; i++ ) {
    void *tmp;
    if ( (the_game->players[i] = (PlayerP) malloc(sizeof(Player)))
	 == NULL ) {
      warn("couldn't malloc player structure");
      exit(1);
    }
    memset((void *)the_game->players[i],0,sizeof(Player));
    if ( (tmp = malloc(sizeof(PlayerExtras)))
	 == NULL ) {
      warn("couldn't malloc player options");
      exit(1);
    }
    memset((void *)tmp,0,sizeof(PlayerExtras));
    set_player_userdata(the_game->players[i],tmp);
  }

  /* some game initialization happens here */
  the_game->protversion = PROTOCOL_VERSION; /* may be reduced by players */
  game_set_options_from_defaults(the_game);

  if ( loadstate ) {
    if ( load_state(the_game) == 0 ) {
      warn("Couldn't load the game state");
      loadstate = 0;
    }
  }

  /* enter the main loop */
  while ( 1 ) {
    struct timeval *tvp;
    rfds = efds = event_fds;

    tvp = NULL;
    if ( the_game->active && !the_game->paused && timeout > 0 ) {
      gettimeofday(&timethen,NULL);
      tvp = &timetimeout;
      tvp->tv_sec = timeout/1000; tvp->tv_usec = (timeout%1000)*1000;
    }
      
    nfds = select(32,&rfds,NULL,&efds,tvp); /* n, read, write, except, timeout */

    if ( tvp ) {
      gettimeofday(&timenow,NULL);
      timeout -= (timenow.tv_sec*1000 + timenow.tv_usec/1000)
	- (timethen.tv_sec*1000 + timethen.tv_usec/1000);
    }

    if ( nfds < 0 ) {
      perror("select failed");
      exit(1);
    }

    if ( nfds == 0 ) {
      timeout_handler(the_game);
      continue;
    }

    for ( i = 0 ; i < MAX_CONNECTIONS ; i++ ) {
      SOCKET fd;
      if ( ! connections[i].inuse ) continue;
      fd = connections[i].skt;

      if ( FD_ISSET(fd,&efds) ) {
	handle_cnx_error(i);
	continue;
      }
      if ( FD_ISSET(fd,&rfds) ) {
	if ( fd == socket_fd ) {
	  SOCKET newfd;

          newfd = accept_new_connection(fd);
	  if (newfd == INVALID_SOCKET) {
	    warn("failure in accepting connection");
	    exit(1);
	  }
	  /* add to the connection list */
	  if ( new_connection(newfd) < 0) {
	    CMsgErrorMsg em;
	    em.type = CMsgError;
	    em.seqno = 0;
	    em.error = "Unable to accept new connections";
	    put_line(newfd,encode_cmsg((CMsgMsg *)&em));
	    closesocket(newfd);
	  }
	}
	else {
	  char *line;
	  /* get_line will fail if there is only a partial line
	     available. This is wrong, since it is possible for
	     lines to arrive fragmentedly (not that it should 
	     normally happen). However, it would also be wrong to
	     wait forever for a partial line to complete. There
	     should be a timeout.
	  */
	  /* get_line provides us with a pointer to the text line,
	     which we can mess with as we like, provided we don't
	     expect it to survive the next call to get_line
	  */
	  line = get_line(connections[i].skt); /* errors dealt with by despatch line */
	  despatch_line(i,line);
	}
      }
    }
  }
}

/* function to act on Player Messages. NB it takes a connection number,
   not an id, since ids may be unassigned.
*/
static void handle_pmsg(PMsgMsg *pmp, int cnx);

/* despatch_line: process the line of input received on cnx. */

static void despatch_line(int cnx, char *line) {
  PMsgMsg *pmp;

  if ( logfile ) {
    fprintf(logfile,"<cnx%d %s",cnx,line);
  }

  if ( line == NULL ) {
    warn("receive error on cnx %d, player id %d\n",cnx, cnx_to_id(cnx));
    handle_cnx_error(cnx);
    return;
  }

  pmp = decode_pmsg(line);
  /* increment the sequence number */
  connections[cnx].seqno++;

  if ( pmp == NULL ) {
    CMsgErrorMsg em;
    em.type = CMsgError;
    em.error = "Protocol error";
    em.seqno = connections[cnx].seqno++;
    send_packet(cnx,(CMsgMsg *)&em,1);
    warn("Protocol error on cnx %d, player id %d; ignoring\n",cnx, cnx_to_id(cnx));
    return;
  }

  handle_pmsg(pmp,cnx);
}

/* little function to check that the minimum time has passed 
   since the last tile moving activity.
   The argument multiplies the usual delay time between the current
   call and the next call. It's used to increase the delay after
   claim implementations. bloody emacs ' */
static void check_min_time(float factor) {
  static struct timeval disctime, timenow;
  static int min_time_this_time;
  if ( min_time_this_time > 0 ) {
    int n;
    if ( disctime.tv_sec > 0 ) { /* not firsttime */
      gettimeofday(&timenow,NULL);
      n = (timenow.tv_sec * 1000 + timenow.tv_usec/1000)
	- (disctime.tv_sec * 1000 + disctime.tv_usec/1000);
      n = 100*min_time_this_time - n;
      if ( n > 0 ) usleep(1000*n);
    }
    gettimeofday(&disctime,NULL);
  }
  min_time_this_time = (int)(min_time*factor);
}

static void handle_pmsg(PMsgMsg *pmp, int cnx) 
{

  /* We mustn't act on most requests if the game is not active.
     There's a race possible otherwise: we can suspend the game, and
     meanwhile a player sends a request: we then act on this, but it
     doesn't get into the history records of the suspended players. */
  if ( ! the_game->active ) {
    CMsgErrorMsg em;
    em.type = CMsgError;
    em.error = "Game not active";
    em.seqno = connections[cnx].seqno;
    switch ( pmp->type ) {
    case PMsgSaveState:
    case PMsgConnect:
    case PMsgSetPlayerOption:
    case PMsgSendMessage:
    case PMsgQueryGameOption:
    case PMsgListGameOptions:
      break; /* these are OK */
    default:
      send_packet(cnx,(CMsgMsg *)&em,0);
      return;
    }
  }

  /* check for debugging messages */
  if ( ! debug && pmp->type >= DebugMsgsStart ) {
    CMsgErrorMsg em;
    em.type = CMsgError;
    em.seqno = connections[cnx].seqno;
    em.error = "Debugging not enabled";
    send_packet(cnx,(CMsgMsg *)&em,0);
    return;
  }

  switch ( pmp->type ) {
  case PMsgSaveState:
    { CMsgErrorMsg em;
    em.type = CMsgError;
    em.seqno = connections[cnx].seqno;
    em.error = "Saving state is currently broken";
    send_packet(cnx,(CMsgMsg *)&em,0);
    return;
    /* NOTREACHED */
    save_state(the_game);
    return;
    }
  case PMsgConnect:
    {
      PMsgConnectMsg *m = (PMsgConnectMsg *) pmp;
      PlayerP p = NULL;
      CMsgConnectReplyMsg cm;
      CMsgPlayerMsg thisplayer;
      CMsgGameMsg gamemsg;
      char *refusal = NULL;
      int i;

      /* start filling in reply message */
      cm.type = CMsgConnectReply;
      cm.pvers = PROTOCOL_VERSION;
      if ( m->pvers/1000 != PROTOCOL_VERSION/1000 ) refusal = "Protocol major version mismatch";
      if ( num_connected_players == NUM_PLAYERS )
	refusal = "Already have players";

      /* If we have a previous game state loaded, we need to associate
	 this player with one in the game. Match on ids */
      if ( loadstate ) {
	/* if no id specified, match on names */
	if ( m->last_id != 0 ) {
	  for ( i = 0; i < NUM_SEATS
		  && m->last_id != the_game->players[i]->id ; i++ ) ;
	} else {
	  for ( i = 0; i < NUM_SEATS
		  && (strcmp(m->name,the_game->players[i]->name) != 0) ; i++ ) ;
	}
	if ( i == NUM_SEATS ) {
	  refusal = "Can't find you in the resumed game";
	} else {
	  /* better check that this player isn't already connected */
	  if ( id_to_cnx(m->last_id) >= 0 )
	    refusal = "Your id is already connected";
	  else {
	    p = the_game->players[i];
	    set_player_name(p,m->name); /* not preserved by saving */
	    setup_maps(cnx,p); /* set up maps between id and cnx etc*/
	    num_connected_players++;
	  }
	}
      } else {
	/* ignore the lastid field, and assign this player the id
	   of its connection number+1. (num_connected_players has not
	   yet been incremented) */
	p = the_game->players[num_connected_players++];
	initialize_player(p);
	set_player_id(p,num_connected_players);
	set_player_name(p,m->name);
	setup_maps(cnx,p); /* set up maps between id and cnx etc*/
      }
      if (refusal) {
	cm.id = 0;
	cm.reason = refusal;
	send_packet(cnx,(CMsgMsg *)&cm,1);
	close_connection(cnx);
	return;
      }
      /* send the reply */
      cm.id = p->id;
      cm.reason = NULL;
      send_packet(cnx,(CMsgMsg *)&cm,1);

      /* reduce the game protocol version if this player can't cope */
      if ( m->pvers < the_game->protversion ) the_game->protversion = m->pvers;

      /* Yes, this is right: the fact that this player is connecting
	 means that it has been disconnected! */
      pextras(p)->disconnected = 1;

      /* Now we need to tell this player who's already here, and
	 tell the others about this one */
      thisplayer.type = CMsgPlayer;
      thisplayer.id = p->id;
      thisplayer.name = p->name;

      for ( i = 0; i < NUM_SEATS; i++ ) {
	CMsgPlayerMsg pm;
	PlayerP q;

	q = the_game->players[i];

	if ( q->id == 0 ) continue;
	
	if ( q->id == p->id ) continue;

	pm.type = CMsgPlayer;
	pm.id = q->id;
	pm.name = q->name;
	  
	send_packet(cnx,(CMsgMsg *)&pm,1);
	send_id(q->id,&thisplayer); /* fails harmlessly if not connected */
      }

      /* if not everybody is connected, just wait for the next */
      if ( num_connected_players < NUM_SEATS ) return;

      /* otherwise, set up the game state (if we aren't 
	 already in the middle of an interrupted game */

      if ( ! loadstate ) {
	GameOptionEntry goe;
	/* randomize the seats if desired */
	if ( randomseats ) {
	  int i,j,n;
	  PlayerP temp[NUM_SEATS];
	  for (i=0; i<NUM_SEATS; i++) temp[i] = the_game->players[i];
	  for (i=0; i<NUM_SEATS; i++) {
	    /* random number between 0 and seatsleft-1 */
	    n = rand_index(NUM_SEATS-(i+1));
	    /* skip already used */
	    j = 0;
	    while ( (!temp[j]) || j < n ) {
	      if ( !temp[j] ) n++;
	      j++;
	    }
	    the_game->players[i] = temp[j];
	    temp[j] = NULL;
	  }
	}
	the_game->state = HandComplete;
	the_game->info.player = noseat;
	the_game->round = EastWind;
	the_game->hands_as_east = 0;
	the_game->firsteast = the_game->players[east]->id;
	/* the_game->protversion is set in the connection process */
	the_game->manager = 0;
	game_set_options_from_defaults(the_game);
	/* the timeout time should be initialized from the command line option */
	goe = *game_get_option(the_game,GOTimeout,NULL);
	goe.value.optint = timeout_time;
	game_set_option(the_game,&goe);
      }
      the_game->cmsg_check = 1;

      gamemsg.type = CMsgGame;
      gamemsg.east = the_game->players[east]->id;
      gamemsg.south = the_game->players[south]->id;
      gamemsg.west = the_game->players[west]->id;
      gamemsg.north = the_game->players[north]->id;
      gamemsg.round = the_game->round;
      gamemsg.hands_as_east = the_game->hands_as_east;
      gamemsg.firsteast = the_game->firsteast;
      gamemsg.east_score = the_game->players[east]->cumulative_score;
      gamemsg.south_score = the_game->players[south]->cumulative_score;
      gamemsg.west_score = the_game->players[west]->cumulative_score;
      gamemsg.north_score = the_game->players[north]->cumulative_score;
      gamemsg.protversion = the_game->protversion;
      gamemsg.manager = the_game->manager;
      
      /* we only send the game message to the players who
	 have been disconnected. (In the usual case, this
	 is all players... We also send the history of
         the hand so far. */

      for ( i=0 ; i < NUM_SEATS; i++ ) {
	int j;
	PlayerP p = the_game->players[i];
	if ( ! pextras(p)->disconnected ) continue;
	pextras(p)->disconnected = 0;
	send_id(p->id,&gamemsg);
	/* we don't actually clear the history records
	   until the NewHand message is sent. But we don't
	   want to send history for a complete hand.
	   However, if the game is paused, we need to tell the player so. */
	if ( the_game->state == HandComplete ) {
	  if ( the_game->paused ) {
	    CMsgPauseMsg pm;
	    CMsgPlayerReadyMsg prm;
	    int i;
	    pm.type = CMsgPause;
	    pm.exempt = 0;
	    pm.requestor = 0;
	    pm.reason = the_game->paused;
	    send_id(p->id,&pm);
	    prm.type = CMsgPlayerReady;
	    for ( i = 0; i < NUM_SEATS; i++ ) {
	      if ( the_game->info.ready[i] ) {
		prm.id = the_game->players[i]->id;
		send_id(p->id,&prm);
	      }
	    }
	  }
	  continue;
	}
	if ( ! usehist ) {
	  CMsgErrorMsg em;
	  em.type = CMsgError;
	  em.seqno = connections[cnx].seqno;
	  em.error = "No history kept: resumption not supported";
	  send_all(the_game,&em);
	  warn(em.error);
	  exit(1);
	}
	for ( j=0; j < pextras(p)->histcount; j++ ) {
	  int sleeptime;
	  send_id(p->id,pextras(p)->history[j]);
	  /* this is an undocumented feature to make things 
	     a bit nicer for humans reconnecting: we'll 
	     unilaterally slow down the feed, by adding a 
	     0.15 second delay between items, or 1 second after
	     a claim implementation. */
	  switch ( pextras(p)->history[j]->type ) {
	  case CMsgPlayerPairs:
	  case CMsgPlayerChows:
	  case CMsgPlayerPungs:
	  case CMsgPlayerKongs:
	  case CMsgPlayerSpecialSet:
	    sleeptime = 1000; /* ms */
	    break;
	  default:
	    sleeptime = 150;
	  }
	  usleep(sleeptime*1000);
	}
      }

      /* Now we should tell somebody to do something. Namely:
	 In state HandComplete: tell everybody, and start a hand
	 if things aren't paused.
	 In state Dealing: everybody.
	 In Discarded, several players might want to do something,
	 so we don't specify the id.
	 Ditto in MahJonging.
         Otherwise, it's the info.player.
      */
      {
	CMsgStartPlayMsg sm;
	CMsgPauseMsg pm;
	sm.type = CMsgStartPlay;
	switch (the_game->state) {
	case HandComplete:
	case Dealing:
	case Discarded:
	case MahJonging:
	  sm.id = 0;
	  break;
	default:
	  sm.id = the_game->players[the_game->info.player]->id;
	  break;
	}
	/* now we should ask the players whether they are ready to
	   continue. We send the pause request *before* making the
	   game active */
	pm.type = CMsgPause;
	pm.reason = (the_game->state == HandComplete)
	  ? "to start hand" : "to continue play";
	pm.exempt = 0;
	pm.requestor = 0;
	handle_cmsg(the_game,&pm);
	send_all(the_game,&pm);
	send_all(the_game,&sm);
	/* and set the game active */
	the_game->active = 1;
      }


      /* now set loadstate */
      loadstate = 1;

      return;
    } /* end of case PMsgConnect */
  case PMsgRequestPause:
    { PMsgRequestPauseMsg *m = (PMsgRequestPauseMsg *)pmp;
      PlayerP p; int id; seats seat;
      CMsgPauseMsg pm;
      CMsgErrorMsg em;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      pm.type = CMsgPause;
      pm.exempt = 0;
      pm.requestor = id;
      pm.reason = m->reason;
      
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;

      if ( !(the_game->state == Discarding
	     || the_game->state == HandComplete) ) {
	em.error = "Not reasonable to pause at this point";
	send_id(id,&em);
	return;
      }
      
      send_all(the_game,&pm);
      break;
    } /* end of PMsgRequestPauseMsg */
  case PMsgSetPlayerOption:
    {
      PMsgSetPlayerOptionMsg *m = (PMsgSetPlayerOptionMsg *)pmp;
      PlayerP p; int id; seats seat;
      CMsgPlayerOptionSetMsg posm;
      CMsgErrorMsg em;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      posm.type = CMsgPlayerOptionSet;
      posm.option = m->option;
      posm.value = m->value;
      posm.text = m->text;
      posm.ack = 1;
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;

      if ( m->option == (unsigned)-1 ) m->option = POUnknown;
      if ( m->option == POUnknown ) em.error = "Unknown option";
      else switch ( m->option ) {
	/* validity checking  */
	/* boolean options */
      case POInfoTiles:
	if ( m->value < 0 || m->value > 1 ) em.error = "Bad value for InfoTiles";
	break;
	/* non-boolean options */
      case PODelayTime:
	/* players *can* request any positive value */
	if ( m->value < 0 ) em.error = "Bad value for DelayTime";
	break;
      default:
	;
      }
      if ( em.error ) {
	send_id(id,&em);
	return;
      }
      send_id(id,&posm);
      popts(p)[m->option] = m->value;
      /* action may now be required */
      switch ( m->option ) {
	int i;
      case PODelayTime:
	/* find the highest value requested by any player, or option */
	min_time = min_time_opt;
	for ( i = 0; i < NUM_SEATS; i++ ) {
	  if ( popts(the_game->players[i])[PODelayTime] > min_time )
	    min_time = popts(the_game->players[i])[PODelayTime];
	}
	/* highest reasonable value is 5 seconds */
	if ( min_time > 50 ) min_time = 50;
	break;
      default:
	;
      }
      return;
    }
  case PMsgReady:
    {
      PlayerP p; int id; seats seat;
      CMsgPlayerReadyMsg prm;
      CMsgErrorMsg em;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      prm.type = CMsgPlayerReady;
      prm.id = id;
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;

      /* if the player is already ready, ignore this message */
      if ( the_game->info.ready[seat] ) return;

      if ( handle_cmsg(the_game,&prm) < 0 ) {
	em.error = the_game->cmsg_err;
	send_id(id,&em);
	return;
      }

      send_all(the_game,&prm);

      /* if state is handcomplete and no longer pausing, 
	 start the next hand */
      if ( the_game->state == HandComplete && ! the_game->paused ) start_hand(the_game);
      /* and in the discarded or konging state, reinstate the timeout */
      if ( the_game->state == Discarded 
	   || (the_game->state == Discarding && the_game->info.konging ) )
	timeout = 1000*timeout_time;
      return;
    } /* end of case PMsgReady */
  case PMsgDeclareSpecial:
    {
      PMsgDeclareSpecialMsg *m = (PMsgDeclareSpecialMsg *) pmp;
      PlayerP p; int id; seats seat;
      CMsgErrorMsg em;
      CMsgPlayerDeclaresSpecialMsg pdsm;
      CMsgPlayerDrawsMsg  pdlm; /* FIXME */
      int res;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      pdlm.id = pdsm.id = id;
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      pdsm.type = CMsgPlayerDeclaresSpecial;
      pdlm.type = CMsgPlayerDraws;
      em.error = NULL;

      /* Legality checking is done by handle cmsg, so
	 just set up the cmsg and apply it */
      pdsm.tile = m->tile;

      res = handle_cmsg(the_game,&pdsm);

      if ( res < -1 ) {
	warn("Consistency error: giving up");
	exit(1);
      }
      if ( res < 0 ) {
	em.error = the_game->cmsg_err;
	send_id(id,&em);
	return;
      }
      check_min_time(1);
      send_all(the_game,&pdsm);
      if ( pdsm.tile == HiddenTile ) {
	/* if we've now started play (i.e. state == Discarding),
	   ask everybody (except east) if they're ready */
	if ( the_game->state == Discarding ) {
	  CMsgPauseMsg pm;
	  pm.type = CMsgPause;
	  pm.reason = "to start play";
	  pm.exempt = the_game->players[east]->id;
	  pm.requestor = 0;
	  if ( handle_cmsg(the_game,&pm) >= 0 ) {
	    send_all(the_game,&pm);
	  } else {
	    warn("Failed to pause at start of play: %s",the_game->cmsg_err);
	  }
	}
	return;
      }
      /* and now draw the replacement */
      /* draw another tile from the *live* wall
         FIXME: this should be controlled by an option */
      pdlm.tile = game_peek_tile(the_game);
      if ( pdlm.tile == ErrorTile ) {
	washout(NULL);
	return;
      }

      if ( the_game->state == Discarding ) {
	/* don't do this if declaring specials */
	/* stash a copy of the player, in case it goes mahjong */
	copy_player(gextras(the_game)->caller,p);
      }

      if ( handle_cmsg(the_game,&pdlm) < 0 ) {
	/* error should be impossible */
	warn("Consistency error: giving up");
	exit(1);
      }
      send_id(id,&pdlm);
      pdlm.tile = HiddenTile;
      check_min_time(1);
      send_others(the_game,id,&pdlm);
      send_infotiles(p);
      return;
      assert(0);
    } /* end of case PMsgDeclareSpecial */
  case PMsgDiscard:
    {
      PMsgDiscardMsg *m = (PMsgDiscardMsg *) pmp;
      PlayerP p; int id; seats seat;
      CMsgErrorMsg em;
      CMsgPlayerDiscardsMsg pdm;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;
      pdm.type = CMsgPlayerDiscards;
      pdm.id = id;
      pdm.tile = m->tile;
      pdm.discard = the_game->discard_serial;
      pdm.calling = m->calling;
      if ( handle_cmsg(the_game,&pdm) < 0 ) {
	em.error = the_game->cmsg_err;
	send_id(id,&em);
	return;
      }
      check_min_time(1);
      send_all(the_game,&pdm);
      send_infotiles(p);
      /* set the timeout */
      timeout = 1000*timeout_time;
      return;
    } /* end of case PMsgDiscard */
  case PMsgNoClaim:
    {
      PMsgNoClaimMsg *m = (PMsgNoClaimMsg *) pmp;
      PlayerP p; int id; seats seat;
      CMsgErrorMsg em;
      CMsgPlayerDoesntClaimMsg pdcm;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;
      pdcm.type = CMsgPlayerDoesntClaim;
      pdcm.id = id;
      pdcm.discard = m->discard;
      pdcm.timeout = 0;

      if ( handle_cmsg(the_game,&pdcm) < 0 ) {
	em.error = the_game->cmsg_err;
	send_id(id,&em);
	return;
      }
      /* handle_cmsg ignores noclaims in wrong state, so
	 we need to check it */
      if ( ! (the_game->state == Discarded 
	   || ( (the_game->state == Discarding 
		 || the_game->state == DeclaringSpecials)
		&& the_game->info.konging ) ) ) return;
      /* acknowledge to the player only */
      send_id(id,&pdcm);
      /* if all claims received, process */
      check_claims(the_game);
      return;
    } /* end of case PMsgNoClaim */
  case PMsgPung:
    {
      PMsgPungMsg *m = (PMsgPungMsg *) pmp;
      PlayerP p; int id; seats seat;
      CMsgErrorMsg em;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;

      if ( the_game->state != MahJonging ) {
	CMsgPlayerClaimsPungMsg pcpm;
	pcpm.type = CMsgPlayerClaimsPung;
	pcpm.id = id;
	pcpm.discard = m->discard;
	if ( handle_cmsg(the_game,&pcpm) < 0 ) {
	  em.error = the_game->cmsg_err;
	  send_id(id,&em);
	  return;
	}
	send_all(the_game,&pcpm);
	/* if all claims received, process */
	check_claims(the_game);
	return;
      } else { /* MahJonging */
	CMsgPlayerPungsMsg ppm;

	ppm.type = CMsgPlayerPungs;
	ppm.id = id;
	ppm.tile = the_game->info.tile;
	
	if ( handle_cmsg(the_game,&ppm) < 0 ) {
	  em.error = the_game->cmsg_err;
	  send_id(id,&em);
	  return;
	}
	check_min_time(1);
	send_all(the_game,&ppm);
	send_infotiles(p);
	/* if that came from a dangerous discard, tell the other players */
	if ( game_flag(the_game,GFDangerousDiscard) ) {
	  CMsgDangerousDiscardMsg ddm;
	  ddm.type = CMsgDangerousDiscard;
	  ddm.id = the_game->players[the_game->info.supplier]->id;
	  ddm.discard = the_game->info.serial;
	  ddm.nochoice = game_flag(the_game,GFNoChoice);
	  send_all(the_game,&ddm);
	}
	/* do scoring if we've finished */
	if ( pflag(p,HandDeclared) ) score_hand(the_game,seat);
	return;
      }
    } /* end of case PMsgPung */
  case PMsgPair:
    {
      PlayerP p; int id; seats seat;
      CMsgErrorMsg em;
      CMsgPlayerPairsMsg ppm;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;
      ppm.type = CMsgPlayerPairs;
      ppm.id = id;
      ppm.tile = the_game->info.tile;

      if ( handle_cmsg(the_game,&ppm) < 0 ) {
	em.error = the_game->cmsg_err;
	send_id(id,&em);
	return;
      }
      check_min_time(1);
      send_all(the_game,&ppm);
      send_infotiles(p);
      /* if that came from a dangerous discard, tell the other players */
      if ( game_flag(the_game,GFDangerousDiscard) ) {
	CMsgDangerousDiscardMsg ddm;
	ddm.type = CMsgDangerousDiscard;
	ddm.id = the_game->players[the_game->info.supplier]->id;
	ddm.discard = the_game->info.serial;
	ddm.nochoice = game_flag(the_game,GFNoChoice);
	send_all(the_game,&ddm);
      }
      /* do scoring if we've finished */
      if ( pflag(p,HandDeclared) ) score_hand(the_game,seat);
      return;
    } /* end of case PMsgPair */
  case PMsgSpecialSet:
    {
      PlayerP p; int id; seats seat;
      CMsgErrorMsg em;
      CMsgPlayerSpecialSetMsg pssm;
      char tiles[100];
      int i;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;
      pssm.type = CMsgPlayerSpecialSet;
      pssm.id = id;
      pssm.tile = the_game->info.tile;
	
      tiles[0] = '\000';
      for ( i = 0; i < p->num_concealed; i++ ) {
	if ( i > 0 ) strcat(tiles," ");
	strcat(tiles,tile_code(p->concealed[i]));
      }
      pssm.tiles = tiles;

      if ( handle_cmsg(the_game,&pssm) < 0 ) {
	em.error = the_game->cmsg_err;
	send_id(id,&em);
	return;
      }
      send_all(the_game,&pssm);
      send_infotiles(p);
      /* do scoring if we've finished */
      if ( pflag(p,HandDeclared) ) score_hand(the_game,seat);
      return;
    } /* end of case PMsgSpecialSet */
  case PMsgFormClosedPair:
    {
      PMsgFormClosedPairMsg *m = (PMsgFormClosedPairMsg *) pmp;
      CMsgPlayerFormsClosedPairMsg pfcpm;
      PlayerP p; int id; seats seat;
      CMsgErrorMsg em;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;
      pfcpm.type = CMsgPlayerFormsClosedPair;
      pfcpm.id = id;
      pfcpm.tile = m->tile;

      if ( handle_cmsg(the_game,&pfcpm) < 0 ) {
	em.error = the_game->cmsg_err;
	send_id(id,&em);
	return;
      }
      check_min_time(1);
      send_all(the_game,&pfcpm);
      send_infotiles(p);
      /* do scoring if we've finished */
      if ( pflag(p,HandDeclared) ) score_hand(the_game,seat);
      return;
    } /* end of case PMsgFormClosedPair */
  case PMsgFormClosedSpecialSet:
    {
      PlayerP p; int id; seats seat;
      CMsgErrorMsg em;
      CMsgPlayerFormsClosedSpecialSetMsg pfcssm;
      char tiles[100];
      int i;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;
      pfcssm.type = CMsgPlayerFormsClosedSpecialSet;
      pfcssm.id = id;
      tiles[0] = '\000';
      for ( i = 0; i < p->num_concealed; i++ ) {
	if ( i > 0 ) strcat(tiles," ");
	strcat(tiles,tile_code(p->concealed[i]));
      }
      pfcssm.tiles = tiles;

      if ( handle_cmsg(the_game,&pfcssm) < 0 ) {
	em.error = the_game->cmsg_err;
	send_id(id,&em);
	return;
      }
      check_min_time(1);
      send_all(the_game,&pfcssm);
      send_infotiles(p);
      /* do scoring if we've finished */
      if ( pflag(p,HandDeclared) ) score_hand(the_game,seat);
      return;
    } /* end of case PMsgFormClosedSpecialSet */
  case PMsgFormClosedPung:
    {
      PMsgFormClosedPungMsg *m = (PMsgFormClosedPungMsg *) pmp;
      CMsgPlayerFormsClosedPungMsg pfcpm;
      PlayerP p; int id; seats seat;
      CMsgErrorMsg em;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;
      pfcpm.type = CMsgPlayerFormsClosedPung;
      pfcpm.id = id;
      pfcpm.tile = m->tile;

      if ( handle_cmsg(the_game,&pfcpm) < 0 ) {
	em.error = the_game->cmsg_err;
	send_id(id,&em);
	return;
      }

      check_min_time(1);
      send_all(the_game,&pfcpm);
      send_infotiles(p);
      /* do scoring if we've finished */
      if ( pflag(p,HandDeclared) ) score_hand(the_game,seat);
      return;
    } /* end of case PMsgFormClosedPung */
  case PMsgFormClosedChow:
    {
      PMsgFormClosedChowMsg *m = (PMsgFormClosedChowMsg *) pmp;
      CMsgPlayerFormsClosedChowMsg pfccm;
      PlayerP p; int id; seats seat;
      CMsgErrorMsg em;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;
      pfccm.type = CMsgPlayerFormsClosedChow;
      pfccm.id = id;
      pfccm.tile = m->tile;
	
      if ( handle_cmsg(the_game,&pfccm) < 0 ) {
	em.error = the_game->cmsg_err;
	send_id(id,&em);
	return;
      }

      check_min_time(1);
      send_all(the_game,&pfccm);
      send_infotiles(p);
      /* do scoring if we've finished */
      if ( pflag(p,HandDeclared) ) score_hand(the_game,seat);
      return;
    } /* end of case PMsgFormClosedChow */
  case PMsgKong:
    {
      PMsgKongMsg *m = (PMsgKongMsg *) pmp;
      PlayerP p; int id; seats seat;
      CMsgPlayerClaimsKongMsg pckm;
      CMsgErrorMsg em;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;
      pckm.type = CMsgPlayerClaimsKong;
      pckm.id = id;
      pckm.discard = m->discard;
      if ( handle_cmsg(the_game,&pckm) < 0 ) {
	em.error = the_game->cmsg_err;
	send_id(id,&em);
	return;
      }
      send_all(the_game,&pckm);
      /* if all claims received, process */
      check_claims(the_game);
      return;
    } /* end of case PMsgKong */
  case PMsgChow:
    {
      PMsgChowMsg *m = (PMsgChowMsg *) pmp;
      PlayerP p; int id; seats seat;
      CMsgPlayerClaimsChowMsg pccm;
      CMsgErrorMsg em;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;
      pccm.type = CMsgPlayerClaimsChow;
      pccm.id = id;
      pccm.discard = m->discard;
      pccm.cpos = m->cpos;

      if ( the_game->state == Discarded ) {
	/* special case: if this player already
	   has a chow claim in, then this claim should
	   be to specify the previously unspecified position
	   after we've told the player that its claim has
	   succeeded.
	   So we don't need to announce it, just to update
	   the cpos field and re-process. */
	if ( the_game->info.claims[seat] == ChowClaim ) {
	  /* check this position is OK */
	  if ( ! player_can_chow(p,the_game->info.tile,m->cpos) ) {
	    em.error = "Can't chow in that position";
	    send_id(id,&em);
	    return;
	  }
	  the_game->info.cpos = m->cpos;
	  check_claims(the_game);
	  return;
	}
	if ( handle_cmsg(the_game,&pccm) < 0 ) {
	  em.error = the_game->cmsg_err;
	  send_id(id,&em);
	  return;
	}
	send_all(the_game,&pccm);
	/* if all claims received, process */
	check_claims(the_game);
	return;
      } else { /* This had better be mahjonging */
	CMsgPlayerChowsMsg pcm;

	pcm.type = CMsgPlayerChows;
	pcm.id = id;
	pcm.tile = the_game->info.tile;
	pcm.cpos = m->cpos;
	if ( handle_cmsg(the_game,&pcm) < 0 ) {
	  em.error = the_game->cmsg_err;
	  send_id(id,&em);
	  return;
	}
	check_min_time(1);
	send_all(the_game,&pcm);
	send_infotiles(p);
	/* if that came from a dangerous discard, tell the other players */
	if ( game_flag(the_game,GFDangerousDiscard) ) {
	  CMsgDangerousDiscardMsg ddm;
	  ddm.type = CMsgDangerousDiscard;
	  ddm.id = the_game->players[the_game->info.supplier]->id;
	  ddm.discard = the_game->info.serial;
	  ddm.nochoice = game_flag(the_game,GFNoChoice);
	  send_all(the_game,&ddm);
	}
	if ( pflag(p,HandDeclared) ) score_hand(the_game,seat);
	return;
      }
    } /* end of case PMsgChow */
  case PMsgDeclareWashOut:
    {
      PlayerP p; int id; seats seat;
      CMsgErrorMsg em;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;

      /* at present, we don't have any rules allowing a player
	 to declare a washout */
      em.error = "Can't declare a Wash-Out";
      send_id(id,&em);
      return;
    } /* end of case PMsgDeclareWashOut */
  case PMsgMahJong:
    {
      PMsgMahJongMsg *m = (PMsgMahJongMsg *) pmp;
      PlayerP p; int id; seats seat;
      CMsgPlayerClaimsMahJongMsg pcmjm;
      CMsgPlayerMahJongsMsg pmjm;
      CMsgErrorMsg em;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;
      pcmjm.type = CMsgPlayerClaimsMahJong;
      pcmjm.id = id;
      pmjm.type = CMsgPlayerMahJongs;
      pmjm.id = id;

      if ( the_game->state == Discarded
	   || ((the_game->state == Discarding
		|| the_game->state == DeclaringSpecials)
	       && the_game->info.konging ) ) {
	pcmjm.discard = m->discard;
	if ( handle_cmsg(the_game,&pcmjm) < 0 ) {
	  em.error = the_game->cmsg_err;
	  send_id(id,&em);
	  return;
	}
	send_all(the_game,&pcmjm);
	/* if all claims received, process */
	check_claims(the_game);
	return;
      } else { /* this should be discarding */
	pmjm.tile = the_game->info.tile; /* that's the tile drawn */
	if ( handle_cmsg(the_game,&pmjm) < 0 ) {
	  em.error = the_game->cmsg_err;
	  send_id(id,&em);
	  return;
	}
	send_all(the_game,&pmjm);
	/* the players will now start declaring their hands */
	return;
      }
    } /* end of case PMsgMahJong */
  case PMsgDeclareClosedKong:
    {
      PMsgDeclareClosedKongMsg *m = (PMsgDeclareClosedKongMsg *) pmp;
      PlayerP p; int id; seats seat;
      CMsgErrorMsg em;
      CMsgPlayerDeclaresClosedKongMsg pdckm;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      pdckm.id = id;
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      pdckm.type = CMsgPlayerDeclaresClosedKong;
      em.error = NULL;

      pdckm.tile = m->tile;
      pdckm.discard = the_game->discard_serial;

      if ( handle_cmsg(the_game,&pdckm) < 0 ) {
	em.error = the_game->cmsg_err;
	send_id(id,&em);
	return;
      }

      check_min_time(1);
      send_all(the_game,&pdckm);
      send_infotiles(p);

      /* now we need to wait for people to try to rob the kong */
      timeout = 1000*timeout_time;
      return;

    } /* end of case PMsgDeclareClosedKong */
  case PMsgAddToPung:
    {
      PMsgAddToPungMsg *m = (PMsgAddToPungMsg *) pmp;
      PlayerP p; int id; seats seat;
      CMsgErrorMsg em;
      CMsgPlayerAddsToPungMsg patpm;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      patpm.id = id;
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      patpm.type = CMsgPlayerAddsToPung;
      em.error = NULL;

      patpm.tile = m->tile;
      patpm.discard = the_game->discard_serial;
      if ( handle_cmsg(the_game,&patpm) < 0 ) {
	em.error = the_game->cmsg_err;
	send_id(id,&em);
	return;
      }

      check_min_time(1);
      send_all(the_game,&patpm);
      send_infotiles(p);

      /* now we need to wait for people to try to rob the kong */
      timeout = 1000*timeout_time;
      return;

      assert(0);
    } /* end of case PMsgAddToPung */
  case PMsgQueryMahJong:
    {
      PMsgQueryMahJongMsg *m = (PMsgQueryMahJongMsg *) pmp;
      PlayerP p; int id; seats seat;
      CMsgCanMahJongMsg cmjm;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      cmjm.type = CMsgCanMahJong;
      cmjm.tile = m->tile;
      cmjm.answer = player_can_mah_jong(p,m->tile);
      send_id(id,&cmjm);
      return;
    }
  case PMsgShowTiles:
    {
      PlayerP p; int id; seats seat;
      CMsgErrorMsg em;
      CMsgPlayerShowsTilesMsg pstm;

      id = cnx_to_id(cnx);
      p = id_to_player(id);
      seat = id_to_seat(id);

      /* fill in basic fields of possible replies */
      pstm.id = id;
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      pstm.type = CMsgPlayerShowsTiles;
      em.error = NULL;

      /* if there are no concealed tiles, just ignore the message;
	 this can only happen from the mahjonging player or by duplication */
      if ( p->num_concealed > 0 ) {
	char tiles[100];
	int i;
	
	tiles[0] = '\000';
	for ( i = 0; i < p->num_concealed; i++ ) {
	  if ( i > 0 ) strcat(tiles," ");
	  strcat(tiles,tile_code(p->concealed[i]));
	}
	pstm.tiles = tiles;
	if ( handle_cmsg(the_game,&pstm) < 0 ) {
	  em.error = the_game->cmsg_err;
	  send_id(id,&em);
	  return;
	}
	check_min_time(1);
	send_all(the_game,&pstm);
	score_hand(the_game,seat);
	return;
      }
      return;
    } /* end of case PMsgShowTiles */
  case PMsgSetGameOption:
    {
      PMsgSetGameOptionMsg *m = (PMsgSetGameOptionMsg *)pmp;
      PlayerP p; int id;
      CMsgErrorMsg em;
      CMsgGameOptionMsg gom;
      GameOptionEntry *goe;

      id = cnx_to_id(cnx);

      /* fill in basic fields of possible replies */
      gom.type = CMsgGameOption;
      gom.id = id;
      p = id_to_player(id);
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;

      /* do we have the option ? */
      goe = game_get_option(the_game,GOUnknown,m->optname);

      if ( ! goe ) {
	em.error = "Trying to set unknown option";
	send_id(id,&em);
	return;
      }
      /* if the option is actually unknown or end, ignore it */
      if ( goe->option == GOUnknown || goe->option == GOEnd ) break;

      if ( goe->enabled == 0 ) {
	em.error = "Option not available in this game";
	send_id(id,&em);
	return;
      }

      gom.optentry = *goe;

      switch ( goe->type ) {
      case GOTBool:
	if ( sscanf(m->optvalue,"%d",&gom.optentry.value.optbool) == 0 ) {
	  em.error = "No boolean value found for option";
	  send_id(id,&em);
	  return;
	}
	if ( gom.optentry.value.optbool != 0 
	     && gom.optentry.value.optbool != 1 ) {
	  em.error = "No boolean value found for option";
	  send_id(id,&em);
	  return;
	}
	break;
      case GOTInt:
	if ( sscanf(m->optvalue,"%d",&gom.optentry.value.optint) == 0 ) {
	  em.error = "No integer value found for option";
	  send_id(id,&em);
	  return;
	}
	break;
      case GOTScore:
	if ( sscanf(m->optvalue,"%d",&gom.optentry.value.optscore) == 0 ) {
	  em.error = "No score value found for option";
	  send_id(id,&em);
	  return;
	}
	break;
      case GOTString:
	strcpy(gom.optentry.value.optstring,m->optvalue);
	break;
      }

      /* specific validity checking not done by handle_cmsg */
      if ( gom.optentry.option == GOTimeout
	   && gom.optentry.value.optint < 0 ) {
	em.error = "Can't set negative timeout!";
	send_id(id,&em);
	return;
      }

      if ( handle_cmsg(the_game,&gom) < 0 ) {
	em.error = the_game->cmsg_err;
	send_id(id,&em);
      }
      send_all(the_game,&gom);
      if ( gom.optentry.option == GOTimeout ) {
	timeout_time = gom.optentry.value.optint;
      }
      break;
    }
  case PMsgQueryGameOption:
    {
      PMsgQueryGameOptionMsg *m = (PMsgQueryGameOptionMsg *)pmp;
      int id;
      CMsgErrorMsg em;
      CMsgGameOptionMsg gom;
      GameOptionEntry *goe;

      id = cnx_to_id(cnx);

      /* fill in basic fields of possible replies */
      gom.type = CMsgGameOption;
      gom.id = 0;
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;

      goe = game_get_option(the_game,GOUnknown,m->optname);
      if ( goe == NULL ) {
	em.error = "Option not known";
	send_id(id,&em);
	return;
      }
      gom.optentry = *goe;
      send_id(id,&gom);
      break;
    }
  case PMsgListGameOptions:
    {
      PMsgListGameOptionsMsg *m = (PMsgListGameOptionsMsg *)pmp;
      int id;
      CMsgGameOptionMsg gom;
      GameOptionEntry *goe;
      unsigned int i;

      id = cnx_to_id(cnx);

      /* fill in basic fields of possible replies */
      gom.type = CMsgGameOption;
      gom.id = 0;

      /* this relies on the fact that we know our option table
	 contains only known options in numerical order. This
	 would not necessarily be the case for clients, but it
	 is for us, since we don't allow unknown options to be set */
      for ( i = 1 ; i <= GOEnd ; i++ ) {
	goe = &the_game->options[i];
	if ( !goe->enabled && ! m->include_disabled ) continue;
	gom.optentry = *goe;
	send_id(id,&gom);
      }
      break;
    }
  case PMsgChangeManager:
    {
      PMsgChangeManagerMsg *m = (PMsgChangeManagerMsg *)pmp;
      int id;
      CMsgChangeManagerMsg cmm;
      CMsgErrorMsg em;

      id = cnx_to_id(cnx);

      /* fill in basic fields of possible replies */
      cmm.type = CMsgChangeManager;
      cmm.id = id;
      cmm.manager = m->manager;
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;

      if ( handle_cmsg(the_game,&cmm) < 0 ) {
	em.error = the_game->cmsg_err;
	send_id(id,&em);
	return;
      }
      send_all(the_game,&cmm);
      break;
    }
  case PMsgSendMessage:
    {
      PMsgSendMessageMsg *m = (PMsgSendMessageMsg *)pmp;
      int id;
      CMsgMessageMsg mm;
      CMsgErrorMsg em;

      id = cnx_to_id(cnx);

      mm.type = CMsgMessage;
      mm.sender = id;
      mm.addressee = m->addressee;
      mm.text = m->text;
      em.type = CMsgError;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;

      if ( mm.addressee == 0 ) {
	send_all(the_game,&mm);
      } else {
	if ( id_to_cnx(mm.addressee) < 0 ) {
	  em.error = "Addressee not found in game";
	  send_id(id,&em);
	  return;
	} else {
	  send_id(mm.addressee,&mm);
	}
      }
      break;
    }
  case PMsgSwapTile:
    {
      PMsgSwapTileMsg *m = (PMsgSwapTileMsg *) pmp;
      PlayerP p; int id; seats seat;
      CMsgErrorMsg em;
      CMsgSwapTileMsg stm;
      int i;

      id = cnx_to_id(cnx);

      /* fill in basic fields of possible replies */
      stm.type = CMsgSwapTile;
      stm.id = m->id;
      stm.oldtile = m->oldtile;
      stm.newtile = m->newtile;
      p = id_to_player(stm.id);
      seat = id_to_seat(stm.id);
      em.type = CMsgError ;
      em.seqno = connections[cnx].seqno;
      em.error = NULL;

      if ( p == NULL ) {
	em.error = "SwapTile: no such player";
	send_id(id,&em);
	return;
      }

      /* find the new tile in the wall. Because this is only used
	 for debugging and testing, we don't care that we're diving
	 inside the game structure!
	 Look for the tile from the end, so as to avoid depleting
	 the early wall.
      */
      for ( i = the_game->wall.dead_end-1 ;
	    the_game->wall.tiles[i] != m->newtile 
	      && i >= the_game->wall.live_used ;
	    i-- ) ;
      if ( i < the_game->wall.live_used ) em.error = "No new tile in wall";
      else {
	if ( handle_cmsg(the_game,&stm) < 0 ) {
	  em.error = the_game->cmsg_err;
	} else {
	  send_id(stm.id,&stm);
	  the_game->wall.tiles[i] = stm.oldtile;
	}
      }
      if ( em.error )
	send_id(id,&em);
      return;
    } /* end of case PMsgSwapTile */
  }
}



/* setup_maps: we need to be able to map between connections,
   player structures, and player ids.
   player to id is in the player structure, so we need the others.
   It's a pity we can't declare private variables here.
*/

/* set up data for a new connection */
static int new_connection(SOCKET skt) {
  int i;

  for (i= 0; i < MAX_CONNECTIONS; i++) {
    if ( connections[i].inuse ) continue;
    connections[i].inuse = 1;
    connections[i].skt = skt;
    connections[i].player = NULL;
    connections[i].seqno = 0;
    /* add to the event set */
    FD_SET(skt,&event_fds);
    return i;
  }
  warn("No space for new connection");
  return -1;
}

/* close connection */
static int close_connection(int cnx) {
  if ( connections[cnx].player ) {
    /* clear maps and decrement counter */
    remove_from_maps(cnx);
    num_connected_players--;
  }
  FD_CLR(connections[cnx].skt,&event_fds);
  closesocket(connections[cnx].skt);
  connections[cnx].inuse = 0;
  return 1;
}
static int cnx_to_id(int cnx) {
  PlayerP p;

  p = connections[cnx].player;
  return p ? p->id : -1;
}

static int id_to_cnx(int id) {
  int i;

  i = 0;
  while ( i < MAX_CONNECTIONS ) {
    if ( connections[i].inuse
	 && connections[i].player
	 && connections[i].player->id == id ) return i;
    i++;
  }
  return -1;
}

static PlayerP id_to_player(int id) {
  int f;

  f = id_to_cnx(id);
  if ( f < 0 ) return NULL;
  return connections[f].player;
}

static void setup_maps(int cnx, PlayerP p) {
  connections[cnx].player = p;
}

static void remove_from_maps(int cnx) {
  connections[cnx].player = (PlayerP) 0;
}

/* send_packet: send the given packet out on the given cnx, perhaps logging */
static void send_packet(int cnx,CMsgMsg *m, int logit) {
  char *l;

  if ( cnx < 0 ) return;

  l = encode_cmsg(m);
  if ( l == NULL ) {
    /* this shouldn't happen */
    warn("send_packet: protocol conversion failed");
    /* in fact, it so much shouldn't happen that we'll dump core */
    assert(0);
    return;
  }
  if ( logit && logfile ) {
    fprintf(logfile,">cnx%d %s",cnx,l);
    fflush(logfile);
  }
  if ( put_line(connections[cnx].skt,l) < 0 ) {
    warn("send_packet: write on cnx %d failed",cnx);
    /* maybe we should shutdown the descriptor here? */
    return;
  }
}

/* send player id a packet. Maybe log it.
   Enter the packet into the player's history, if appropriate */
static void _send_id(int id,CMsgMsg *m,int logit) {
  int cnx = id_to_cnx(id);
  if ( cnx < 0 ) return;
  send_packet(cnx,m,logit);
  if ( usehist && the_game->active ) {
    PlayerExtras *x = pextras(id_to_player(id));
    if ( ! x ) return;
    switch ( m->type ) {
      /* it's a lot easier to list the messages that DON'T
	 go into the history, and less error prone... */
    default:
      /* Gack. This results in lots of copies.
	 It's not clear how to fix this without adding
	 reference counts */
      x->history[x->histcount++] = cmsg_deepcopy(m);
      break;
    case CMsgCanMahJong:
    case CMsgClaimDenied:
    case CMsgConnectReply:
    case CMsgError:
    case CMsgGame:
    case CMsgGameOver:
    case CMsgInfoTiles:
    case CMsgNewRound:
    case CMsgPlayer:
    case CMsgMessage:
    case CMsgChangeManager:
    case CMsgStartPlay:
    case CMsgStopPlay:
      ;
    }
  } 
}

/* send to all players in a game, logging one copy */
static void _send_all(Game *g,CMsgMsg *m) {
  _send_id(g->players[east]->id,m,1);
  _send_id(g->players[south]->id,m,0);
  _send_id(g->players[west]->id,m,0);
  _send_id(g->players[north]->id,m,0);
}

/* send to all players in a game EXCEPT the id. Don't log it. */
static void _send_others(Game *g, int id,CMsgMsg *m) {
  if ( g->players[east]->id != id ) _send_id(g->players[east]->id,m,0);
  if ( g->players[south]->id != id ) _send_id(g->players[south]->id,m,0);
  if ( g->players[west]->id != id ) _send_id(g->players[west]->id,m,0);
  if ( g->players[north]->id != id ) _send_id(g->players[north]->id,m,0);
}

/* send to a particular seat in a game. Logit. */
static void _send_seat(Game *g, seats s, CMsgMsg *m) {
  _send_id(g->players[s]->id,m,1);
}

/* send to seats other than s.
   Don't log it: a more informative copy will have gone to
   another player. */
static void _send_other_seats(Game *g, seats s, CMsgMsg *m) {
  seats i;
  for ( i = 0 ; i < NUM_SEATS ; i++ )
    if ( i != s ) _send_id(g->players[i]->id,m,0);
}

/* start_hand: start a new hand.
   The game should be in state HandComplete.
   Send a NewHand message to each player.
   Shuffle and build the new wall.
   Deal the initial tiles to each player.
   Draw east's 14th tile.
   Enter state DeclaringSpecials, with info.player set to east.

   Return 1 on success.
   On error, return 0, and put human readable explanation in
   global variable failmsg.
*/

static int start_hand(Game *g) {
  int i;
  int stack,wall;
  CMsgNewHandMsg cm;

  cm.type = CMsgNewHand;

  if ( g->state != HandComplete ) {
    failmsg = "start_hand called when hand not complete.";
    return 0;
  }

  cm.east = g->players[east]->id; /* assume no change */

  /* Does the deal rotate? */
  if ( g->info.player != noseat /* there was a mah jong */
       && (g->info.player != east /* and either it wasn't east */
	   || g->hands_as_east == 13)) { /* or this was east's 13th win */
    
    /* does the round change? (the current south will be the next east;
       if that is the firsteast, the round changes) */
    if ( g->players[south]->id == g->firsteast ) {
      CMsgNewRoundMsg nrm;
      
      if ( g->round == NorthWind ) {
	seats i,j;
	CMsgGameOverMsg gom;
	gom.type = CMsgGameOver;
	send_all(g,&gom);
	fprintf(stdout,"Game over. Final scores:\n");
	/* May as well print these in the order of original easts */
	for (i=0;i<NUM_SEATS;i++) {
	  j = (i+1)%NUM_SEATS;
	  fprintf(stdout,"%5d (%-20s)  %5d\n",
		  g->players[j]->id,
		  g->players[j]->name,
		  g->players[j]->cumulative_score);
	}
#ifdef WIN32
	sleep(10); /* give the client time to close the connection */
#endif
	exit(0);
      }
      nrm.round = next_wind(g->round);
      nrm.type = CMsgNewRound;
      handle_cmsg(g,&nrm);
      send_all(g,&nrm);
    }
    /* rotate the seats. The new east is the current south */
    cm.east = g->players[south]->id;
  }

  /* we haven't yet processed the newhand message */

  /* now clear the players' history records */
  for ( i = 0 ; i < NUM_SEATS ; i++ ) {
    PlayerExtras *x = pextras(the_game->players[i]);
    while ( x->histcount > 0 ) {
      cmsg_deepfree(x->history[--x->histcount]);
    }
  }

  /* process and send a new hand msg */
  /* we're also supposed to choose the place where the deal starts.
     Just for fun, we'll follow the real procedure. */
  wall = 1+rand_index(5) + 1+rand_index(5);
  stack = wall + 1+rand_index(5) + 1+rand_index(5);
  stack++; /* the dice give the end of the dead wall */
  if ( stack > (144/NUM_SEATS)/2 ) {
    stack -= (144/NUM_SEATS)/2;
    wall++;
  }
  cm.start_wall = (wall-1) % NUM_SEATS;
  cm.start_stack = stack -1; /* convert from 1 counting to 0 counting */
  if ( handle_cmsg(g,&cm) < 0 ) {
    warn("handle_cmsg of NewHand message return error; dumping core");
    abort();
  }
  send_all(g,&cm);
  
  /* set up the wall. Should this be done by the game routine?
     I think not, on the whole. */
  random_tiles(g->wall.tiles);

  /* deal out the tiles into deal messages.
     We will actually deal in the traditional way: if we ever get
     round to implementing a semi-random shuffle, instead of a really
     random re-deal, this will make a difference.
  */
  { CMsgPlayerDrawsMsg pdm;
  int i,j;
  seats s;
  pdm.type = CMsgPlayerDraws;

  for ( j=0; j < 3; j++ ) {
    for ( s=east; s<=north; s++) {
      check_min_time(1);
      pdm.id = g->players[s]->id;
      for ( i = 0; i < 4; i++ ) {
	pdm.tile = game_draw_tile(g);
	send_id(pdm.id,&pdm);
	player_draws_tile(g->players[s],pdm.tile);
	pdm.tile = HiddenTile;
	send_others(g,pdm.id,&pdm);
      }
    }
  }
  for ( s=east; s<=north; s++) {
    check_min_time(1);
    pdm.id = g->players[s]->id;
    pdm.tile = game_draw_tile(g);
    send_id(pdm.id,&pdm);
    player_draws_tile(g->players[s],pdm.tile);
    pdm.tile = HiddenTile;
    send_others(g,pdm.id,&pdm);
  }
  /* at this point, we should copy east to the caller slot in case
     it wins with heaven's blessing; otherwise scoring.c will be
     unhappy. (Yes, this happened.) */
  copy_player(gextras(g)->caller,g->players[east]);

  s=east;
  pdm.id = g->players[s]->id;
  pdm.tile = game_draw_tile(g);
  check_min_time(1);
  send_id(pdm.id,&pdm);
  player_draws_tile(g->players[s],pdm.tile);
  pdm.tile = HiddenTile;
  send_others(g,pdm.id,&pdm);
  }

  /* send out info messages */
  send_infotiles(g->players[east]);
  send_infotiles(g->players[south]);
  send_infotiles(g->players[west]);
  send_infotiles(g->players[north]);

  /* Enter the next state and return */
  g->state = DeclaringSpecials;
  g->info.player = east;
  return 1;
}


/* timeout_handler: force no claims to be sent, and
   then check claims */
static void timeout_handler(Game *g) {
  seats i;
  CMsgPlayerDoesntClaimMsg pdcm;

  if ( g->state != Discarded
       && ! ( g->state == Discarding && g->info.konging ) ) {
    warn("timeout_handler called in unexpected state");
    return;
  }

  pdcm.type = CMsgPlayerDoesntClaim;
  pdcm.discard = g->info.serial;
  pdcm.timeout = 1;
  for ( i=0; i < NUM_SEATS; i++ ) {
    if ( i == g->info.player ) continue;
    if ( g->info.claims[i] != UnknownClaim ) continue;
    pdcm.id = g->players[i]->id;
    handle_cmsg(g,&pdcm);
    send_id(pdcm.id,&pdcm);
  }
  check_claims(g);
}


/* check_claims: when a player makes a claim on the discard,
   (or tries to rob a kong)
   we need to check whether all claims have been lodged, and
   if so, process them. This function does that */

static void check_claims(Game *g) {
  seats i, s, ds;
  CMsgClaimDeniedMsg cdm;

  if ( g->state != Discarded
       && ! ( (g->state == Discarding 
	       || g->state == DeclaringSpecials) 
	      && g->info.konging ) ) {
    warn("check_claims: called in wrong game state");
    return;
  }

  /* ds is the seat of the discarder/konger */
  ds = g->info.player;

  /* return if not all claims are lodged */
  for ( i = 0 ; i < NUM_SEATS ; i++ )
    if ( i != ds && g->info.claims[i] == UnknownClaim ) return;

  /* OK, process. First, cancel timeout */
  timeout = 0;

  /* s will be the successful claim */
  s = noseat ;
  /* is there a mah-jong? */
  for ( i = 0 ; i < NUM_SEATS ; i++ )
    if ( g->info.claims[(ds+i)%NUM_SEATS]
	  == MahJongClaim ) { s = (ds+i)%NUM_SEATS ; break ; }

  /* is there a kong? */
  if ( s == noseat )
    for ( i = 0 ; i < NUM_SEATS ; i++ )
      if ( g->info.claims[(ds+i)%NUM_SEATS]
	   == KongClaim ) { s = (ds+i)%NUM_SEATS ; break ; }

  /* is there a pung ? */
  if ( s == noseat )
    for ( i = 0 ; i < NUM_SEATS ; i++ )
      if ( g->info.claims[(ds+i)%NUM_SEATS]
	   == PungClaim ) { s = (ds+i)%NUM_SEATS ; break ; }
  
  /* or is there a chow? */
  if ( s == noseat )
    if ( g->info.claims[(ds+1)%NUM_SEATS]
	   == ChowClaim ) { s = (ds+1)%NUM_SEATS ; }

  /* finished checking; now process */

  if ( s == noseat ) {
    if ( g->state == Discarded ) {
      /* no claim; play passes to the next player, who draws */
      CMsgPlayerDrawsMsg m;
      s = (ds+1)%NUM_SEATS;
      m.type = CMsgPlayerDraws;
      m.id = g->players[s]->id;
      /* peek at the tile: we'll pass the CMsg to the game
	 to handle state changes */
      m.tile = game_peek_tile(g);
      
      if ( m.tile == ErrorTile ) {
	/* run out of wall; washout */
	washout(NULL);
	return;
      }
    
      /* stash a copy of the player, in case it goes mahjong */
      copy_player(gextras(g)->caller,g->players[s]);
      
      if ( handle_cmsg(g,&m) < 0 ) {
	warn("Consistency error drawing tile");
	exit(1);
      }
      check_min_time(1);
      send_seat(g,s,&m);
      m.tile = HiddenTile;
      send_other_seats(g,s,&m);
      send_infotiles(g->players[s]);
      return;
    } else {
      /* the player formed a kong, and (surprise) nobody tried to 
	 rob it */
      CMsgPlayerDrawsLooseMsg pdlm;

      pdlm.type = CMsgPlayerDrawsLoose;
      pdlm.id = g->players[ds]->id;
      /* find the loose tile. */
      pdlm.tile = game_peek_loose_tile(the_game);
      if ( pdlm.tile == ErrorTile ) {
	washout(NULL);
	return;
      }

      /* stash a copy of the player, in case it goes mahjong */
      copy_player(gextras(g)->caller,g->players[ds]);

      if ( handle_cmsg(the_game,&pdlm) < 0 ) {
	/* error should be impossible */
	warn("Consistency error: giving up");
	exit(1);
      }
      check_min_time(1);
      send_id(pdlm.id,&pdlm);
      pdlm.tile = HiddenTile;
      send_others(the_game,pdlm.id,&pdlm);
      send_infotiles(g->players[ds]);
      return;
    }
  }

  if ( g->info.claims[s] == ChowClaim ) {
    /* there can be no other claims, so just implement this one */
    CMsgPlayerChowsMsg pcm;

    pcm.type = CMsgPlayerChows;
    pcm.id = g->players[s]->id;
    pcm.tile = g->info.tile;
    pcm.cpos = g->info.cpos;

    /* if the cpos is AnyPos, then we instruct the player (only)
       to send in another chow claim, and just wait for it */
    if ( pcm.cpos == AnyPos ) {
      send_id(pcm.id,&pcm);
      return;
    }

    /* stash a copy of the player, in case it goes mahjong */
    copy_player(gextras(g)->caller,g->players[s]);

    if ( handle_cmsg(g,&pcm) < 0 ) {
      warn("Consistency error: failed to implement claimed chow.");
      exit(1);
    }

    check_min_time(2);
    send_all(g,&pcm);
    send_infotiles(g->players[s]);
    /* if that came from a dangerous discard, tell the other players */
    if ( game_flag(the_game,GFDangerousDiscard) ) {
      CMsgDangerousDiscardMsg ddm;
      ddm.type = CMsgDangerousDiscard;
      ddm.discard = the_game->info.serial;
      ddm.id = g->players[g->info.supplier]->id;
      ddm.nochoice = game_flag(g,GFNoChoice);
      send_all(the_game,&ddm);
    }
    return;
  }

  /* In the remaining cases, there may be other claims; send
     denial messages to them */
  cdm.type = CMsgClaimDenied;
  cdm.reason = "There was a prior claim";
  
  for ( i = 0 ; i < NUM_SEATS ; i++ ) {
    if ( i != ds && i != s && g->info.claims[i] != NoClaim) {
      cdm.id = g->players[i]->id;
      send_id(cdm.id,&cdm);
    }
  }

  if ( g->info.claims[s] == PungClaim ) {
    CMsgPlayerPungsMsg ppm;

    ppm.type = CMsgPlayerPungs;
    ppm.id = g->players[s]->id;
    ppm.tile = g->info.tile;

    /* stash a copy of the player, in case it goes mahjong */
    copy_player(gextras(g)->caller,g->players[s]);

    if ( handle_cmsg(g,&ppm) < 0 ) {
      warn("Consistency error: failed to implement claimed pung.");
      exit(1);
    }

    check_min_time(2);
    send_all(g,&ppm);
    send_infotiles(g->players[s]);
    /* if that came from a dangerous discard, tell the other players */
    if ( game_flag(the_game,GFDangerousDiscard) ) {
      CMsgDangerousDiscardMsg ddm;
      ddm.type = CMsgDangerousDiscard;
      ddm.id = g->players[g->info.supplier]->id;
      ddm.discard = the_game->info.serial;
      ddm.nochoice = game_flag(g,GFNoChoice);
      send_all(the_game,&ddm);
    }
    return;
  }

  if ( g->info.claims[s] == KongClaim ) {
    CMsgPlayerKongsMsg pkm;
    CMsgPlayerDrawsLooseMsg pdlm;

    pdlm.type = CMsgPlayerDrawsLoose;
    pkm.type = CMsgPlayerKongs;
    pdlm.id = pkm.id = g->players[s]->id;
    pkm.tile = g->info.tile;

    if ( handle_cmsg(g,&pkm) < 0 ) {
      warn("Consistency error: failed to implement claimed kong.");
      exit(1);
    }

    check_min_time(2);
    send_all(g,&pkm);

    /* find the loose tile. */
    pdlm.tile = game_peek_loose_tile(g);
    if ( pdlm.tile == ErrorTile ) {
      washout(NULL);
      return;
    }

    /* stash a copy of the player, in case it goes mahjong */
    copy_player(gextras(g)->caller,g->players[s]);

    if ( handle_cmsg(the_game,&pdlm) < 0 ) {
      /* error should be impossible */
      warn("Consistency error: giving up");
      exit(1);
    }
    check_min_time(1);
    send_id(pdlm.id,&pdlm);
    pdlm.tile = HiddenTile;
    send_others(g,pdlm.id,&pdlm);
    send_infotiles(g->players[s]);
    return;
  }

  if ( g->info.claims[s] == MahJongClaim ) {
    /* stash a copy of the player before it grabs the discard */
    copy_player(gextras(g)->caller,g->players[s]);

    if ( g->state == Discarded ) {
      /* the normal situation */
      CMsgPlayerMahJongsMsg pmjm;
      
      pmjm.type = CMsgPlayerMahJongs;
      pmjm.id = g->players[s]->id;
      pmjm.tile = HiddenTile;
      
      if ( handle_cmsg(g,&pmjm) < 0 ) {
	/* this should be impossible */
	CMsgErrorMsg em;
	em.type = CMsgError;
	em.seqno = connections[id_to_cnx(pmjm.id)].seqno;
	em.error = g->cmsg_err;
	send_id(pmjm.id,&em);
	return;
      }
      send_all(g,&pmjm);
      return;
    } else {
      /* the kong's been robbed */
      CMsgPlayerRobsKongMsg prkm;

      prkm.type = CMsgPlayerRobsKong;
      prkm.id = g->players[s]->id;
      prkm.tile = g->info.tile;

      if ( handle_cmsg(g,&prkm) < 0 ) {
	/* this should be impossible */
		CMsgErrorMsg em;
	em.type = CMsgError;
	em.seqno = connections[id_to_cnx(prkm.id)].seqno;
	em.error = g->cmsg_err;
	send_id(prkm.id,&em);
	return;
      }
      check_min_time(2);
      send_all(g,&prkm);
      return;
    }
  }

  /* NOTREACHED */
}


/* This function sends an InfoTiles message about the given player
   to all players who are receiving info tiles */
static void send_infotiles(PlayerP p) {
  static CMsgInfoTilesMsg itm;
  PlayerP q;
  static char buf[128];
  int i;

  itm.type = CMsgInfoTiles;
  itm.id = p->id;
  if ( popts(p)[POInfoTiles] ) {
    player_print_tiles(buf,p,0);
    itm.tileinfo = buf;
    send_id(itm.id,&itm);
  }
  player_print_tiles(buf,p,1);
  for ( i = 0 ; i < NUM_SEATS ; i++ ) {
    q = the_game->players[i];
    if ( q != p && popts(q)[POInfoTiles] ) {
      send_id(q->id,&itm);
    }
  }
}

/* This function is a quick hack to save the game state to a file.
   It will eventually be replaced by a more elegant function.
   However, the changes required to the connection procedure should
   apply also to the elegant version.
*/
static int save_state(Game *g) {
  FILE *fd;
  unsigned int i,j; /* unsigned to suppress warnings */
  unsigned char *u;
  char *pname;

  /* pro tem, just write to game.state in the current dir */
  fd = fopen("game.state","w");
  if ( ! fd ) return 0;
  /* the game structure starts with NUM_SEATS pointers (NB)
     to players. Write each player structure out as hex coded
     binary data.
  */
  for ( i=0 ; i < NUM_SEATS ; i++ ) {
    u = (unsigned char *) g->players[i];
    /* the player structure contains a pointer, namely the player's
       name. For now, we will (using illegal access) null this before
       writing, since when the players reconnect, they'll give their
       names, so we don't need to save the text.
    */
    pname = g->players[i]->name;
    ((Player *)(g->players[i]))->name = NULL;
    for (j=0; j < sizeof(Player); j++) {
      fprintf(fd,"%02X",u[j]);
    }
    ((Player *)(g->players[i]))->name = pname;
    fprintf(fd,"\n");
  }
  /* now print the rest of the game structure (no more pointers) */
  u = (unsigned char *) g;
  for ( j = NUM_SEATS*sizeof(PlayerP); j < sizeof(Game); j++ ) {
    fprintf(fd,"%02X",u[j]);
  }
  fprintf(fd,"\n");
  fclose(fd);
  return 1;
}

/* and load the state into the (pre-allocated) game pointed to by g.
   N.B. The player structures must also be allocated.
*/
static int load_state(Game *g) {
  FILE *fd;
  unsigned int i,j;
  int n;
  unsigned char *u;

  fd = fopen("game.state","r");
  if ( ! fd ) return 0;
  for ( i=0 ; i < NUM_SEATS ; i++ ) {
    u = (unsigned char *) g->players[i];
    for (j=0; j < sizeof(Player); j++) {
      fscanf(fd,"%2X",&n); u[j] = n;
    }
  }
  u = (unsigned char *) g;
  for ( j = NUM_SEATS*sizeof(PlayerP); j < sizeof(Game); j++ ) {
      fscanf(fd,"%2X",&n); u[j] = n;
  }
  fclose(fd);
  return 1;
}

/* This function is responsible for handling the scoring.
   It is called in the following circumstances:
   The mahjonging player has made all their sets, or
   a non-mahjonging player has issued a ShowTiles message
   (the showing of the tiles will already have been done).
   It then (a) computes the score for the hand, together
   with a text description of the computation, and announces
   it, and sets the HandDeclared flag for the player;
   (b) if all players are now declared, computes the settlements,
   announces the net change for each player, together with
   a text description of the calculation; implements the settlements;
   and moves to the state HandComplete, incrementing hands_as_east
   if necessary.
   Arguments: g is the game, s is the *seat* of the player on
   which something has happened.
*/
static void score_hand(Game *g, seats s) {
  PlayerP p = g->players[s];
  Score score;
  char buf[1000], tmp[100];
  static char *seatnames[] = { "East ", "South", "West ", "North" };
  static int changes[NUM_SEATS];
  CMsgHandScoreMsg hsm;
  CMsgSettlementMsg sm;
  unsigned int i,j;
  int m;
  int cannon=0;
  seats cannoner = noseat;

  buf[0] = 0;
  score = score_of_hand(g,s);
  hsm.type = CMsgHandScore;
  hsm.id = p->id;
  hsm.score = score.value;
  hsm.explanation = score.explanation;

  send_all(g,&hsm);

  psetflag(p,HandDeclared); /* probably already done */
  if ( s == g->info.player ) {
    psetflag(p,MahJongged);
  }
  set_player_hand_score(p,score.value);

  /* have we scored all the players ? */
  for (i=0; i < NUM_SEATS; i++)
    if ( g->players[i]->hand_score < 0 ) return;

  /* now compute the settlements. East pays and receives double. */

  for ( i = 0; i < NUM_SEATS; i++ ) changes[i] = 0;

  /* first the winner; let's re-use s and p */
  s = g->info.player;
  p = g->players[s];

  /* Firstly, check to see if somebody let off a cannon. At this point,
     we assume we know that the dflags field is a bit field! */
  for ( i = 0; i < NUM_SEATS; i++ ) {
    if ( i == s ) continue;
    if ( p->dflags[i] & p->dflags[s] ) {
      /* Only one person can be liable, so let's have a little
	 consistency check */
      if ( cannon ) {
	warn("Two people found letting off a cannon!");
	exit(1);
      }
      cannon=1;
      cannoner=i;
    }
  }

  if ( cannon ) {
    sprintf(tmp,"%s let off a cannon\n\n",seatnames[cannoner]);
    strcat(buf,tmp);

    m = g->players[s]->hand_score;
    if ( s == east ) {
      /* cannoner pays 6 times */
      m *= 6;
    } else {
      /* cannoner pays 4 times */
      m *= 4;
    }
    changes[s] += m;
    changes[cannoner] -= m;
    sprintf(tmp,"%s pays %s %5d\n",seatnames[cannoner],seatnames[s],m);
    strcat(buf,tmp);
  } else {
    for ( i = 0; i < NUM_SEATS; i++ ) {
      if ( i == s ) continue;
      m = g->players[s]->hand_score;
      if ( i*s == 0 ) m *= 2; /* sneaky test for one being east */
      changes[s] += m; changes[i] -= m;
      sprintf(tmp,"%s pays %s %5d\n",seatnames[i],seatnames[s],m);
      strcat(buf,tmp);
    }
  }
  sprintf(tmp,"%s GAINS %5d\n\n",seatnames[s],changes[s]);
  strcat(buf,tmp);

  /* and now the others */
  for ( i = 0; i < NUM_SEATS; i++ ) {
    if ( i == s ) continue;
    for ( j = i+1; j < NUM_SEATS; j++ ) {
      if ( j == s ) continue;
      if ( cannon ) continue;
      m = g->players[i]->hand_score - g->players[j]->hand_score;
      if ( i*j == 0 ) m *= 2;
      changes[i] += m; changes[j] -= m;
      if ( m > 0 ) {
	sprintf(tmp,"%s pays %s %5d\n",seatnames[j],seatnames[i],m);
      } else if ( m < 0 ) {
	sprintf(tmp,"%s pays %s %5d\n",seatnames[i],seatnames[j],-m);
      } else {
	sprintf(tmp,"%s and %s are even\n",seatnames[i],seatnames[j]);
      }
      strcat(buf,tmp);
    }
    sprintf(tmp,"%s %s %5d\n\n",seatnames[i],
	    changes[i] >= 0 ? "GAINS" : "LOSES",
	    abs(changes[i]));
    strcat(buf,tmp);
  }
  
  sm.type = CMsgSettlement;
  sm.east = changes[east];
  sm.south = changes[south];
  sm.west = changes[west];
  sm.north = changes[north];
  sm.explanation = buf;
  handle_cmsg(g,&sm);
  send_all(g,&sm);
  /* and now pause for the next hand */
  {
    CMsgPauseMsg pm;
    pm.type = CMsgPause;
    pm.exempt = 0;
    pm.requestor = 0;
    pm.reason = "to start next hand";
    handle_cmsg(g,&pm);
    send_all(g,&pm);
  }
}

/* implement washout. The argument is the reason for the washout;
   if NULL, the default reason of "wall exhausted" is assumed. */
static void washout(char *reason) {
  CMsgPauseMsg pm;
  CMsgWashOutMsg wom;
  wom.type = CMsgWashOut;
  if ( reason ) wom.reason = reason;
  else wom.reason = "wall exhausted";
  handle_cmsg(the_game,&wom);
  send_all(the_game,&wom);
  /* and now pause for the next hand */
  pm.type = CMsgPause;
  pm.exempt = 0;
  pm.requestor = 0;
  pm.reason = "to start next hand";
  handle_cmsg(the_game,&pm);
  send_all(the_game,&pm);
}

/* called for an error on an cnx. Removes the cnx,
   suspends the game if appropriate, etc */
static void handle_cnx_error(int cnx) {
  CMsgStopPlayMsg spm;
  char emsg[100];
  int id;
  /* for the moment, die noisily. Ultimately, just notify
     other players, and re-open the listening socket */
  if ( connections[cnx].skt == socket_fd ) {
    /* hard to see how this could happen, but ... */
    warn("Exception condition on listening fd\n");
    exit(1);
  }
  fprintf(stderr,"Exception/eof on cnx %d, player %d\n",
	  cnx, cnx_to_id(cnx));
  spm.type = CMsgStopPlay;
  /* We clear the player's entry before sending the message
     so that it fails harmlessly on the player's file instead
     of writing to a broken pipe. */
  id = cnx_to_id(cnx);
  close_connection(cnx);
  if ( id >= 0 ) {
    sprintf(emsg,"Lost connection to player %d%s",
	    id,exit_on_disconnect ? "quitting" : loadstate ? "; game suspended" : "");
    spm.reason = emsg;
    send_all(the_game,&spm);
    the_game->active = 0;
    timeout = 0; /* clear timeout */
  }
  if ( exit_on_disconnect ) exit(1) ;
}
