/* $Header: /home/jcb/newmj/RCS/gui.c,v 10.15 2001/02/10 21:41:40 jcb Exp $
 * gui.c
 * A graphical interface, using gtk.
 */
/****************** COPYRIGHT STATEMENT **********************
 * This file is Copyright (c) 2000 by J. C. Bradfield.       *
 * Distribution and use is governed by the LICENCE file that *
 * accompanies this file.                                    *
 * The moral rights of the author are asserted.              *
 *                                                           *
 ***************** DISCLAIMER OF WARRANTY ********************
 * This code is not warranted fit for any purpose. See the   *
 * LICENCE file for further information.                     *
 *                                                           *
 *************************************************************/


#include "gui.h"

static const char rcs_id[] = "$Header: /home/jcb/newmj/RCS/gui.c,v 10.15 2001/02/10 21:41:40 jcb Exp $";

/* local forward declarations */
static AnimInfo *adjust_wall_loose(void);
static gint check_claim_window(gpointer data);
static void create_wall(void);
static void clear_wall(void);
static int get_relative_posn(GtkWidget *w, GtkWidget *z, int *x, int *y);
static AnimInfo *playerdisp_discard(PlayerDisp *pd,Tile t);
static void playerdisp_init(PlayerDisp *pd, int ori);
static void playerdisp_update(PlayerDisp *pd, CMsgUnion *m);
static void playerdisp_clear_discards(PlayerDisp *pd);
static void rotate_pixmap(GdkPixmap *east, GdkPixmap **south,
			  GdkPixmap **west, GdkPixmap **north);
static void server_input_callback(gpointer data, gint source,
				  GdkInputCondition condition);
static void start_animation(AnimInfo *source,AnimInfo *target);
static void stdin_input_callback(gpointer data, gint source, 
				 GdkInputCondition condition);


/* This global variable contains XPM data for fallback tiles.
   It is defined in fbtiles.c */
extern char **fallbackpixmaps[];

/*********************
  GLOBAL VARIABLES 
*********************/

#ifndef TILEPIXMAPDIR
#define TILEPIXMAPDIR "./tiles"
#endif
static char *tilepixmapdir = TILEPIXMAPDIR; /* where to find tiles */
int debug = 0;
PlayerP our_player;
int our_id;
seats our_seat; /* our seat in the game */
/* convert from a game seat to a position on our display board */
#define game_to_board(s) ((s+NUM_SEATS-our_seat)%NUM_SEATS)
/* and conversely */
#define board_to_game(s) ((s+our_seat)%NUM_SEATS)
/* convert from game wall position to board wall position */
#define wall_game_to_board(i) ((i + wallstart) % 144)
/* and the other way */
#define wall_board_to_game(i) ((i + 144 - wallstart)%144)
Game *the_game;
int selected_button; /* the index of the user's selected
			     tile, or -1 if none */

/* tag for the server input callback */
static gint server_callback_tag;

/* These are used as time stamps. now_time() is the time since program
   start measured in milliseconds. start_tv is the start time.
*/

static struct timeval start_tv;
static int now_time(void) {
  struct timeval now_tv;
  gettimeofday(&now_tv,NULL);
  return (now_tv.tv_sec-start_tv.tv_sec)*1000
	  +(now_tv.tv_usec-start_tv.tv_usec)/1000;
}


int ptimeout = 15000; /* claim timeout time in milliseconds [global] */

/* These are the height and width of a tile (including
   the button border. The spacing is currently
   1/4 the width of an actual tile pixmap. */
static int tile_height = 0;
static int tile_width = 0;
static int tile_spacing = 0;

/* variable concerned with the display of the wall */
static int showwall = -1;

/* array of buttons for the tiles in the wall.
   button 0 is the top rightmost tile in our wall, and 
   we count clockwise as the tiles are dealt. */
static GtkWidget  *wall[144]; 
static int wallstart; /* where did the live wall start ? */
/* given a tile number in the wall, what orientation shd it have? */
#define wall_ori(n) (n < 144/4 ? 0 : n < 2*144/4 ? 3 : n < 3*144/4 ? 2 : 1)


/* Pixmaps for tiles. We need one for each orientation.
   The reason for this slightly sick procedure is so that the
   error tile can also have a pixmap here, with the indexing
   working as expected
*/
static GdkPixmap *_tilepixmaps[4][MaxTile+1];

/* this is global */
GdkPixmap **tilepixmaps[4] = {
  &_tilepixmaps[0][1], &_tilepixmaps[1][1],
  &_tilepixmaps[2][1], &_tilepixmaps[3][1] };

/* pixmaps for tong box, and mask */
static GdkPixmap *tongpixmaps[4][4]; /* [ori,wind-1] */
static GdkBitmap *tongmask;

GtkWidget *topwindow; /* main window */
GtkWidget *menubar; /* menubar */
GtkWidget *board; /* the table area itself */
GtkWidget *boardframe; /* fixed widget wrapping the board */
GtkStyle *tablestyle; /* for the dark green stuff */
GtkStyle *highlightstyle; /* to highlight tiles */
GtkWidget *highlittile = NULL ; /* the unique highlighted tile */
GtkWidget *dialoglowerbox; /* encloses dialogs when dialogs are below */
GdkFont *fixed_font; /* a fixed width font */
GdkFont *big_font; /* big font for claim windows */


/* This gives the width of a player display in units
   of tiles */

int pdispwidth = 0;
int square_aspect = 0; /* force a square table */
int animate = 0; /* do fancy animation */
/* the player display areas */
PlayerDisp pdisps[NUM_SEATS];

int calling = 0; /* disgusting global flag */

/* the widget in which we put discards */
GtkWidget *discard_area;
/* This is an allocation structure in which we
   note its allocated size, at some point when we
   know the allocation is valid.
*/
GtkAllocation discard_area_alloc;

GtkWidget *just_doubleclicked = NULL; /* yech yech yech. See doubleclicked */

/* space round edge of dialog boxes */
const gint dialog_border_width = 10;
/* horiz space between buttons */
const gint dialog_button_spacing = 10;
/* vert space between text and buttons etc */
const gint dialog_vert_spacing = 10;

/* space around player boxes. N.B. this should only
   apply to the outermost boxes */
const gint player_border_width = 10;

DialogPosition dialogs_position = DialogsUnspecified;

int pass_stdin = 0; /* 1 if commands are to read from stdin and
			      passed to the server. */

void usage(char *pname,char *msg) {
  fprintf(stderr,"%s: %s\nUsage: %s [ --id N ]\n\
  [ --server ADDR ]\n\
  [ --name NAME ]\n\
  [ --connect ]\n\
  [ --[no-]show-wall ]\n\
  [ --size N ]\n\
  [ --animate ]\n\
  [ --tile-pixmap-dir DIR ]\n\
  [ --dialogs-popup | --dialogs-below | --dialogs-central ]\n\
  [ --echo-server ]\n\
  [ --pass-stdin ]\n",
	  pname,msg,pname);
}

#ifdef WIN32
/* In Windows, we can't use sockets as they come, but have to convert
   them to glib channels. The following functions are passed to the
   sysdep.c initialize_sockets routine to do this. */

static void *skt_open_transform(SOCKET s) {
  GIOChannel *chan;

  chan = g_io_channel_unix_new(s); /* don't know what do if this fails */
  if ( ! chan ) {
    warn("Couldn't convert socket to channel");
    return NULL;
  }
  return (void *)chan;
}

static int skt_closer(void *chan) {
  g_io_channel_close((GIOChannel *)chan);
  g_io_channel_unref((GIOChannel *)chan); /* unref the ref at creation */
  return 1;
}

static int skt_reader(void *chan,void *buf, size_t n) {
  int len = 0;
  if ( g_io_channel_read((GIOChannel *)chan,buf,n,&len) 
       == G_IO_ERROR_NONE ) {
    /* fprintf(stderr,"g_io_channel_read returned %d bytes\r\n",len); */
    return len;
  }
  else {
    /* fprintf(stderr,"g_io_channel_read returned error\r\n"); */
    return -1;
  }
}

static int skt_writer(void *chan, const void *buf, size_t n) {
  int len = 0;
  if ( g_io_channel_write((GIOChannel *)chan, (void *)buf,n,&len) 
       == G_IO_ERROR_NONE ) 
    return len;
  else 
    return -1;
}

/* a glib level wrapper for the server input callback */

static gboolean gcallback(GIOChannel   *source UNUSED,
			     GIOCondition  condition UNUSED,
			    gpointer      data UNUSED) {
  server_input_callback(NULL,0,GDK_INPUT_READ);
  return 1;
}



#endif /* WIN32 */

int
main (int argc, char *argv[])
{
  GtkWidget *inboard, *tmpw, *outerframe;
  char *name = NULL;
  int do_connect = 0;
  static char *address = "localhost:5000";
  int i;
  
  /* options. I should use getopt ... */
  for (i=1;i<argc;i++) {
    if ( strcmp(argv[i],"--id") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --id");
      our_id = atoi(argv[i]);
    } else if ( strcmp(argv[i],"--server") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --server");
      address = argv[i];
    } else if ( strcmp(argv[i],"--name") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --name");
      name = argv[i];
    } else if ( strcmp(argv[i],"--connect") == 0 ) {
      do_connect = 1;
    } else if ( strcmp(argv[i],"--show-wall") == 0 ) {
      showwall = 1;
    } else if ( strcmp(argv[i],"--no-show-wall") == 0 ) {
      showwall = 0;
    } else if ( strcmp(argv[i],"--size") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --size");
      pdispwidth = atoi(argv[i]);
    } else if ( strcmp(argv[i],"--animate") == 0 ) {
      animate = 1;
    } else if ( strcmp(argv[i],"--tile-pixmap-dir") == 0 ) {
      if ( ++i == argc ) usage(argv[0],"missing argument to --tile-pixmap-dir");
      tilepixmapdir = argv[i];
    } else if ( strcmp(argv[i],"--echo-server") == 0 ) {
      debug = 1;
    } else if ( strcmp(argv[i],"--dialogs-popup") == 0 ) {
      dialogs_position = DialogsPopup;
    } else if ( strcmp(argv[i],"--dialogs-below") == 0 ) {
      dialogs_position = DialogsBelow;
    } else if ( strcmp(argv[i],"--dialogs-central") == 0 ) {
      dialogs_position = DialogsCentral;
    } else if ( strcmp(argv[i],"--pass-stdin") == 0 ) {
      pass_stdin = 1;
    } else {
      usage(argv[0],"unknown option or argument");
      exit(1);
    }
  }

#ifdef WIN32
  initialize_sockets(skt_open_transform,skt_closer,
		     skt_reader,skt_writer);
#endif

  gtk_init (&argc, &argv);
  
  /* load the extra fonts we need */
  /* It is completely $@#$@# insane that I can't specify these
     things with resources. Sometimes I think the gtk developers
     have never heard of networked systems. */
  /* try to get a fixed font. Courier for preference */
#ifdef WIN32
  fixed_font = gdk_font_load("courier new");
  if ( ! fixed_font ) fixed_font = gdk_font_load("courier");
#else
  fixed_font = gdk_font_load("-*-courier-medium-r-normal-*-*-120-*-*-*-*-*-*");
#endif
  /* if that didn't work, try fixed */
  if ( ! fixed_font )
    fixed_font = gdk_font_load("-misc-fixed-medium-r-normal-*-*-120-*-*-*-*-*-*");
  /* if that didn't work, try just "fixed" */
  if ( ! fixed_font )
    fixed_font = gdk_font_load("fixed");
  /* if that didn't work; what can we do? */
  if ( ! fixed_font )
    warn("unable to load a fixed font for scoring window etc");

  /* And now try to get a big font for claim windows. It seems that
     this does actually work on both X and Windows... */
  if ( !big_font ) {
    big_font = gdk_font_load("-*-helvetica-bold-r-*-*-*-180-*-*-*-*-iso8859-*");
  }
  if ( !big_font )
    big_font = gdk_font_load("-*-fixed-bold-r-*-*-*-140-*-*-*-*-iso8859-*");
  if ( !big_font )
    big_font = gdk_font_load("12x24");
  if ( !big_font )
    big_font = gdk_font_load("9x15bold");
  


  /* set the size if not given by user */
  if ( pdispwidth == 0 ) {
    pdispwidth = 19;
    if ( gdk_screen_width() <= 800 ) pdispwidth = 18;
  }
  /* choose a sensible value for show-wall */
  if ( showwall < 0 ) {
    showwall = ( gdk_screen_height() >= ((dialogs_position == DialogsBelow)
				 ? 1000 : 900));
  }

  if ( dialogs_position == DialogsUnspecified )
    dialogs_position = DialogsCentral;

  /* all the board widgets want a dark green background */
  tablestyle = gtk_style_copy(gtk_widget_get_default_style());
  gdk_color_parse("darkgreen",&(tablestyle->bg[GTK_STATE_NORMAL]));
  /* to highlight discarded tiles */
  highlightstyle = gtk_style_copy(gtk_widget_get_default_style());
  gdk_color_parse("red",&(highlightstyle->bg[GTK_STATE_NORMAL]));
  
  topwindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_signal_connect (GTK_OBJECT (topwindow), "delete_event",
                               GTK_SIGNAL_FUNC (gtk_main_quit), NULL);

  /* Why do we realize the window? Because we want to use it
     in creating pixmaps. */
  gtk_widget_realize(topwindow);
 
  /* create the tile pixmaps */
  {
    char pmfile[512];
    const char *code;
    const char *wl[] = { "E", "S", "W", "N" };
    GtkRequisition r;
    GtkWidget *b,*p;
    int usetpmd;
    Tile i;
    int numfailures = 0;

    /* specifying tilepixmapdir as the empty string is equivalent
       to having it null, meaning don't look for tiles */
    usetpmd = (tilepixmapdir && tilepixmapdir[0] != 0);

    for ( i = -1; i < MaxTile; i++ ) {
      if ( usetpmd ) {
	strcpy(pmfile,tilepixmapdir);
	strcat(pmfile,"/");
      }
      code = tile_code(i);
      /* only create the error pixmaps once! */
      if ( i >= 0 && strcmp(code,"XX") == 0 ) {
	tilepixmaps[0][i] = tilepixmaps[0][-1];
	tilepixmaps[1][i] = tilepixmaps[1][-1];
	tilepixmaps[2][i] = tilepixmaps[2][-1];
	tilepixmaps[3][i] = tilepixmaps[3][-1];
      }
      else {
	if ( usetpmd ) {
	  strcat(pmfile,code);
	  strcat(pmfile,".xpm");
	  tilepixmaps[0][i] = gdk_pixmap_create_from_xpm(topwindow->window,
						    NULL,
						    NULL,
						    pmfile);
	}
	/* If this fails, use the fallback data. The error tile
	   is in position 99, the others are in their natural position */
	if ( tilepixmaps[0][i] == NULL ) {
	  int j;
	  if ( i < 0 ) j = 99; else j = i;
	  tilepixmaps[0][i] =
	    gdk_pixmap_create_from_xpm_d(topwindow->window,
					 NULL,
					 NULL,fallbackpixmaps[j]);
	  numfailures++;
	}
	rotate_pixmap(tilepixmaps[0][i],
		      &tilepixmaps[1][i],
		      &tilepixmaps[2][i],
		      &tilepixmaps[3][i]);
      }
    }
    /* and the tong pixmaps */
    for (i=0; i<4; i++) {
      if ( usetpmd ) {
	strcpy(pmfile,tilepixmapdir);
	strcat(pmfile,"/");
	strcat(pmfile,"tong");
	strcat(pmfile,wl[i]);
	strcat(pmfile,".xpm");
	tongpixmaps[0][i] = gdk_pixmap_create_from_xpm(topwindow->window,
						    i>0 ? NULL : &tongmask,
						    NULL,
						    pmfile);
      }
      if ( tongpixmaps[0][i] == NULL ) {
	tongpixmaps[0][i] =
	  gdk_pixmap_create_from_xpm_d(topwindow->window,
					 i>0 ? NULL : &tongmask,
					 NULL,fallbackpixmaps[101+i]);
	numfailures++;
      }
      rotate_pixmap(tongpixmaps[0][i],
		      &tongpixmaps[1][i],
		      &tongpixmaps[2][i],
		      &tongpixmaps[3][i]);
    }

    if ( usetpmd && numfailures > 0 ) {
      char buf[50];
      sprintf(buf,"%d tile pixmaps not found: using fallbacks",numfailures);
      warn(buf);
    }

    /* and compute the tile spacing */
    gdk_window_get_size(tilepixmaps[0][0],&tile_spacing,NULL);
    tile_spacing /= 4;
    b = gtk_button_new();
    p = gtk_pixmap_new(tilepixmaps[0][HiddenTile],NULL);
    gtk_widget_show(p);
    gtk_container_add(GTK_CONTAINER(b),p);
    gtk_widget_size_request(b,&r);
    tile_width = r.width;
    tile_height = r.height;
    gtk_widget_destroy(b);
  }
    
  /* the outerframe contains the menubar and the board, and if
     dialogs are at the bottom, a box for them */
  outerframe = gtk_vbox_new(0,0);
  gtk_container_add(GTK_CONTAINER(topwindow),outerframe);
  gtk_widget_show(outerframe);
  menubar = menubar_create();
  gtk_box_pack_start(GTK_BOX(outerframe),menubar,0,0,0);

  /* The structure of the whole board is (at present)
     an hbox containing leftplayer, vbox, rightplayer
     with the vbox containing
     topplayer discardarea bottomplayer (us).
     The left/right players are actually sandwiched inside
     vboxes so that they will centre if we force a square
     aspect ratio.
  */

  boardframe = gtk_event_box_new();
  gtk_widget_set_style(boardframe,tablestyle);
  gtk_widget_show(boardframe);
  board = gtk_hbox_new(0,0);
  gtk_widget_set_style(board,tablestyle);
  gtk_widget_show(board);
  gtk_container_add(GTK_CONTAINER(boardframe),board);
  gtk_box_pack_start(GTK_BOX(outerframe),boardframe,0,0,0);

  if ( dialogs_position == DialogsBelow ) {
    GtkWidget *t;
    t = gtk_event_box_new(); /* so there's a window to have a style */
    gtk_widget_show(t);
    dialoglowerbox = gtk_hbox_new(0,0);
    gtk_container_add(GTK_CONTAINER(t),dialoglowerbox);
    gtk_box_pack_end(GTK_BOX(outerframe),t,0,0,0);
    gtk_widget_show(dialoglowerbox);
  }

  /* the left player */
  playerdisp_init(&pdisps[3],3);
  gtk_widget_show(pdisps[3].widget);
  tmpw = gtk_vbox_new(0,0);
  gtk_widget_set_style(tmpw,tablestyle);
  gtk_widget_show(tmpw);
  gtk_box_pack_start(GTK_BOX(board),tmpw,0,0,0);
  gtk_box_pack_start(GTK_BOX(tmpw),pdisps[3].widget,1,0,0);

  /* the inner box */
  inboard = gtk_vbox_new(0,0);
  gtk_widget_set_style(inboard,tablestyle);
  gtk_widget_show(inboard);
  /* The inner box needs to expand to fit the outer, or else
     it will shrink to fit the top and bottom players */
  gtk_box_pack_start(GTK_BOX(board),inboard,1,1,0);

  /* the right player */
  playerdisp_init(&pdisps[1],1);
  gtk_widget_show(pdisps[1].widget);
  tmpw = gtk_vbox_new(0,0);
  gtk_widget_set_style(tmpw,tablestyle);
  gtk_widget_show(tmpw);
  gtk_box_pack_start(GTK_BOX(board),tmpw,0,0,0);
  gtk_box_pack_start(GTK_BOX(tmpw),pdisps[1].widget,1,0,0);

  /* the top player */
  playerdisp_init(&pdisps[2],2);
  gtk_widget_show(pdisps[2].widget);
  gtk_box_pack_start(GTK_BOX(inboard),pdisps[2].widget,0,0,0);

  /* the discard area */
  discard_area = gtk_fixed_new();
  gtk_widget_set_style(discard_area,tablestyle);
  gtk_container_set_border_width(GTK_CONTAINER(discard_area),dialog_border_width);
  gtk_widget_show(discard_area);
  gtk_box_pack_start(GTK_BOX(inboard),discard_area,1,1,0);

  /* the bottom player (us) */
  playerdisp_init(&pdisps[0],0);
  gtk_widget_show(pdisps[0].widget);
  gtk_box_pack_end(GTK_BOX(inboard),pdisps[0].widget,0,0,0);


  /* At this point, all the boxes are where we want them.
     So to prevent everything being completely mucked up, we now
     lock the sizes of the player boxes.
     This works, without showing the window, because we're asking
     the widgets what they want */
  for (i=0;i<4;i++) {
    GtkRequisition r;
    GtkWidget *w;
    w = pdisps[i].widget;
    gtk_widget_size_request(w,&r);
    gtk_widget_set_usize(w,r.width,r.height);
  }

  /* if we are showing the wall, create it */
  if ( showwall ) create_wall();

  /* now we can force square aspect ratio */
  if ( square_aspect ) {
    GtkRequisition r;
    GtkWidget *w;
    w = board;
    gtk_widget_size_request(w,&r);
    gtk_widget_set_usize(w,r.width,r.width);
  }

  /* these dialogs may be part of the top level window,
     so they get created now */

  /* create the discard dialog */
  discard_dialog_init();

  /* and the chow dialog */
  chow_dialog_init();

  /* and the declaring specials dialog */
  ds_dialog_init();

  /* and create the turn dialog */
  turn_dialog_init();

  /* now create the scoring dialog */
  scoring_dialog_init();
      
  /* and the continue dialog */
  continue_dialog_init();

  /* and the open dialog */
  {
    char ht[256], pt[10], ft[256], idt[10];
    int usehost = 1;
    ht[0] = pt[0] = idt[0] = ft[0] = 0;
    if ( strchr(address,':') ) {
      /* grrr */
      if ( address[0] == ':' ) {
	strcpy(ht,"localhost");
	strcpy(pt,address+1);
      } else {
	sscanf(address,"%[^:]:%s",ht,pt);
      }
    } else {
      strcpy(ft,address);
      usehost = 0;
    }
    sprintf(idt,"%d",our_id);

    open_dialog_init(ht,pt,ft,idt,name,usehost);
  }

  /* It's essential that we map the main window now,
     since if we are resuming an interrupted game, messages
     will arrive quickly; and the playerdisp_discard
     needs to know the allocation of the discard area.
     If we look at the wrong time, before it's mapped, we get
     -1: the discard_area won't get its allocation until after
     the topwindow has got its real size from the WM */
  gtk_widget_show_now(topwindow);

  /* Unfortunately, this still doesn't work all the time.
     Maybe the allocation happens in response to a configure
     notify? 
     So we'll just iterate through the main loop until we
     find some sensible values in here. It seems that the non
     sensible values include 1. */
  while ( discard_area->allocation.width <= 1 )
    gtk_main_iteration();

  discard_area_alloc = discard_area->allocation;

  /* dissuade the discard area from shrinking */
  gtk_widget_set_usize(discard_area,discard_area_alloc.width,discard_area_alloc.height);

  /* The window is now built, and sizes calculated We now
     clear everything. I don't quite know what the order of events
     is here, but it seems to work with no flashing. */
  for (i=0;i<4;i++) playerdisp_update(&pdisps[i],NULL);

#if 0  /* not wanted at present, but left here so we remember how to do it */
  /* we create a new popup signal which we will send to dialogs when
     they're popped up and we want them to set focus to the appropriate
     button */
  gtk_object_class_user_signal_new(GTK_OBJECT_CLASS(gtk_type_class(gtk_type_from_name("GtkWidget"))), "popup", GTK_RUN_FIRST, gtk_signal_default_marshaller, GTK_TYPE_NONE,0);
#endif

  /* and the scoring message thing */
  textwindow_init();

  /* and the message window */
  messagewindow_init();

  /* and the status window */
  status_init();

  /* Argh argh double argh: we were doing this much earlier,
     but of course it's essential to do it now, not earlier,
     otherwise the above show_now will get server input, and
     things will happen before we've found discard_area_alloc */

  if ( pass_stdin ) gdk_input_add(0,GDK_INPUT_READ,stdin_input_callback,NULL);

  if ( do_connect ) open_connection(0,0);

  /* initialize time stamps */
  gettimeofday(&start_tv,NULL);

  gtk_main ();
  
  return 0;
}


void server_input_callback(gpointer data UNUSED,
                            gint              sourceparam UNUSED, 
			   GdkInputCondition conditionparam UNUSED) {
  char *l;
  CMsgUnion *m;
  int id;
  int i;

  l = get_line(the_game->fd);
  if ( debug ) fputs(l ? l : "NULL LINE\n",stdout);
  if ( ! l ) {
    fprintf(stderr,"receive error from controller\n");
    error_dialog_popup("Lost connection to server");
    close_connection();
    return;
  }

  m = (CMsgUnion *)decode_cmsg(l);

  if ( ! m ) {
    fprintf(stderr,"protocol error from controller\n");
    return;
  }

  if ( m->type == CMsgConnectReply ) {
    our_id = ((CMsgConnectReplyMsg *)m)->id;
    if ( our_id == 0 ) {
      char s[1024];
      close_connection();
      strcpy(s,"Server refused connection because: ");
      strcat(s,((CMsgConnectReplyMsg *)m)->reason);
      error_dialog_popup(s);
      return;
    } else {
      /* since we have a human driving us, ask the controller to slow down
	 a bit; especially if we're animating */
      /* Also, we need to know the claim timeout */
      PMsgSetPlayerOptionMsg spom;
      PMsgQueryGameOptionMsg qgom;
      spom.type = PMsgSetPlayerOption;
      spom.option = PODelayTime;
      spom.ack = 0;
      spom.value = animate ? 10 : 5; /* 0.5 or 1 second min delay */
      spom.text = NULL;
      send_packet(&spom);
      qgom.type = PMsgQueryGameOption;
      strcpy(qgom.optname,"Timeout");
      send_packet(&qgom);
    }
  }

  /* skip InfoTiles */
  if ( m->type == CMsgInfoTiles ) return;

  /* error messages */
  if ( m->type == CMsgError ) {
    error_dialog_popup(m->error.error);
    return;
  }    

  /* player options */
  if ( m->type == CMsgPlayerOptionSet ) {
    /* ignore it; we currently have no state corresponding
       to player options */
    return;
  }

  id = game_handle_cmsg(the_game,(CMsgMsg *)m);

  calling = 0; /* awful global flag to detect calling */
  /* This is the place where we take intelligent action in
     response to the game. As a matter of principle, we don't do this, 
     but there is an exception: if there is a kong to be robbed,
     we will check to see whether we can actually go out with the tile,
     and if we can't, we'll send in NoClaim directly, rather than
     asking the user */

  /* FIXME: we can't assume that we know whether we can go mahjong.
     The server might allow special winning hands that we don't know
     about. I suppose we should add a CanIMahJong? message to the
     protocol and use that.
  */

  if ( id != our_id 
       && (the_game->state == Discarding 
	   || the_game->state == DeclaringSpecials)
       && the_game->info.konging 
       && ! player_can_mah_jong(our_player,the_game->info.tile) ) {
    disc_callback(NULL,(gpointer) NoClaim); /* sends the message */
    the_game->info.claims[our_seat] = NoClaim; /* this is a cheat to stop
					   the claim window coming up later */
  }

  status_update(m->type == CMsgGameOver);

  if ( id == our_id ) player_sort_tiles(our_player);

  /* now deal with the display. There are basically two sorts
     of things to do: some triggered by a message, such as
     displaying draws, claims, etc; and some depending only on game state,
     such as making sure the right dialogues are up.
     We'll deal with message-dependent things first.
  */

  switch ( m->type ) {
  case CMsgError:
    /* this has been dealt with above, and cannot happen */
    warn("Error message in impossible place");
    break;
  case CMsgPlayerOptionSet:
    /* this has been dealt with above, and cannot happen */
    warn("PlayerOptionSet message in impossible place");
    break;
  case CMsgStopPlay:
    { char buf[512];
    strcpy(buf,"Server has suspended play because:\n");
    strcat(buf,(m->stopplay.reason));
    error_dialog_popup(buf);
    }
    break;
  case CMsgGameOption:
    /* update the timeout value */
    if ( m->gameoption.optentry.option == GOTimeout ) {
      ptimeout = 1000*m->gameoption.optentry.value.optint;
    }
    break;
  case CMsgGame:
    /* now we know the seating order: make the userdata field of
       each player point to the appropriate pdisp */
    our_seat = game_id_to_seat(the_game,our_id);
    for ( i = 0; i < NUM_SEATS; i++ ) {
      set_player_userdata(the_game->players[i],
			  (void *)&pdisps[game_to_board(i)]);
      /* and make the pdisp point to the player */
      pdisps[game_to_board(i)].player = the_game->players[i];
    }
    break;
  case CMsgNewRound:
    /* update seat when it might change */
    our_seat = game_id_to_seat(the_game,our_id);
    break;
  case CMsgNewHand:
    /* update seat when it might change */
    our_seat = game_id_to_seat(the_game,our_id);
    /* After a new hand message, clear the display, and
       change the labels in the scoring window */
    for ( i=0; i < 4; i++ ) {
      static char *windnames[] = { "East", "South", "West", "North" };
      playerdisp_update(&pdisps[i],m);
      gtk_label_set_text(GTK_LABEL(textlabels[i]),
			 windnames[board_to_game(i)]);
    }
    if ( showwall ) {
      wallstart = m->newhand.start_wall*(144/4) + m->newhand.start_stack*2;
      create_wall();
      adjust_wall_loose(); /* set the loose tiles in place */
    }
    break;
  case CMsgPlayerDiscards:
    /* if calling declaration, set flag */
    if ( m->playerdiscards.calling ) calling = 1;
    /* and then update */
  case CMsgPlayerFormsClosedPung:
  case CMsgPlayerDeclaresClosedKong:
  case CMsgPlayerAddsToPung:
  case CMsgPlayerFormsClosedChow:
  case CMsgPlayerFormsClosedPair:
  case CMsgPlayerShowsTiles:
  case CMsgPlayerFormsClosedSpecialSet:
  case CMsgPlayerDraws:
  case CMsgPlayerDrawsLoose:
  case CMsgPlayerDeclaresSpecial:
  case CMsgPlayerPairs:
  case CMsgPlayerChows:
  case CMsgPlayerPungs:
  case CMsgPlayerKongs:
  case CMsgPlayerSpecialSet:
  case CMsgPlayerClaimsPung:
  case CMsgPlayerClaimsKong:
  case CMsgPlayerClaimsChow:
  case CMsgPlayerClaimsMahJong:
  case CMsgSwapTile:
    playerdisp_update((PlayerDisp *)game_id_to_player(the_game,id)->userdata,m);
    break;
  case CMsgPlayerRobsKong:
    /* we need to update the victim */
    playerdisp_update(&pdisps[game_to_board(the_game->info.supplier)],m);
    break;
  case CMsgGameOver:
    close_connection(); /* this has to come first, so that
			   status_showraise doesn't try to update status info*/
    status_showraise();
    return;
    /* in the following case, we need do nothing (in this switch) */
  case CMsgInfoTiles:
  case CMsgConnectReply:
  case CMsgPlayer:
  case CMsgStartPlay:
  case CMsgPause:
  case CMsgPlayerReady:
  case CMsgClaimDenied:
  case CMsgPlayerDoesntClaim:
  case CMsgDangerousDiscard:
  case CMsgCanMahJong:
  case CMsgHandScore:
  case CMsgSettlement:
  case CMsgChangeManager:
    break;
  case CMsgMessage:
    { static char bigbuf[2048];
    expand_protocol_text(bigbuf,m->message.text,2048);
    gtk_text_insert(GTK_TEXT(messagetext),0,0,0,
		    game_id_to_player(the_game,m->message.sender)->name,-1);
    gtk_text_insert(GTK_TEXT(messagetext),0,0,0,
		    "> ",-1);
    gtk_text_insert(GTK_TEXT(messagetext),0,0,0,
		    m->message.text,-1);
    gtk_text_insert(GTK_TEXT(messagetext),0,0,0,
		    "\n",-1);
    showraise(messagewindow);
    }
    break;
  case CMsgPlayerMahJongs:
    /* this could be from hand.
       Clear all the texts in the scoring displays.
       Unselect tiles (if we're about to claim the discard,
       it's visually confusing seeing the first tile get selected;
       so instead, we'll select the first tile when we pop up
       the "declare closed sets" dialogue. */
    {
      int i;
      if ( the_game->info.player == our_seat ) 
	conc_callback(NULL,NULL);
      for (i=0;i<5;i++)
	gtk_editable_delete_text(GTK_EDITABLE(textpages[i]),0,-1);
    }
    playerdisp_update((PlayerDisp *)game_id_to_player(the_game,id)->userdata,m);
    break;
  case CMsgWashOut:
    /* display a message in the settlement window of the scoring info */
    {
      int i,npos=0;
      char *text = "This hand is a WASH-OUT";
      /* clear the scoring panes */
      for (i=0;i<5;i++)
	gtk_editable_delete_text(GTK_EDITABLE(textpages[i]),0,-1);
      gtk_editable_insert_text(GTK_EDITABLE(textpages[4]),
			     text,strlen(text),&npos);
      /* select the settlement pane */
      gtk_notebook_set_page(GTK_NOTEBOOK(scoring_notebook),4);
      showraise(textwindow);
    }
  }

  /* in the discarded state, the dialog box should be up if somebody
     else discarded and we haven't claimed. If we've claimed,
     it should be down.
     Further, if we have a chowclaim in and the pending flag is set,
     the chow dialog should be up. */
  if ( the_game->state == Discarded
       && the_game->info.player != our_seat ) {
    int s;
    /* which player display is it? */
    s = the_game->info.player; /* seat */
    s = (s+NUM_SEATS-our_seat)%NUM_SEATS; /* posn relative to us */
    if ( the_game->active && !the_game->paused && the_game->info.claims[our_seat] == UnknownClaim ) 
      discard_dialog_popup(the_game->info.tile,s,0);
    else
      gtk_widget_hide(discard_dialog->widget);
    if ( the_game->active && !the_game->paused && the_game->info.claims[our_seat] == ChowClaim
	 && the_game->info.pending )
      do_chow(NULL,(gpointer)AnyPos);
  }

  /* check whether to pop down the claim windows */
  {
    int i;
    for (i=0;i<4;i++) {
      if ( ! GTK_WIDGET_VISIBLE(pdisps[i].claimw) ) continue;
      check_claim_window(&pdisps[i]);
    }
  }

  /* in discarded or mahjonging states, the turn dialog should be down */
  if ( the_game->state == Discarded || the_game->state == MahJonging )
    gtk_widget_hide(turn_dialog);

  /* in the discarding state, make sure the chow and ds dialogs are down.
     If it's our turn, the turn dialog should be up (unless we are in a 
     konging state, when we're waiting for other claims), otherwise down.
     If there's a kong in progress, and we haven't claimed, the discard
     dialog should be up */
  if ( the_game->state == Discarding ) {
    gtk_widget_hide(chowdialog);
    gtk_widget_hide(ds_dialog);
    if ( the_game->active && !the_game->paused
	 && the_game->info.player == our_seat
	 && ! the_game->info.konging )
      turn_dialog_popup();
    else
      gtk_widget_hide(turn_dialog);
    if ( the_game->active && !the_game->paused && the_game->info.player != our_seat
	 && the_game->info.konging
	 && the_game->info.claims[our_seat] == UnknownClaim ) {
      seats s;
      /* which player display is it? */
      s = the_game->info.player; /* seat */
      s = (s+NUM_SEATS-our_seat)%NUM_SEATS; /* posn relative to us */
      discard_dialog_popup(the_game->info.tile,s,2);
    } else gtk_widget_hide(discard_dialog->widget);
  }

  /* If it is declaring specials time, and our turn,
     make sure the ds dialog is up. If it's not our turn,
     make sure it's down.
     Also make sure the turn and discard dialogs are down
     If somebody else has a kong in progress, and we haven't claimed,
     then the claim box should be up.
  */
  if ( the_game->active && !the_game->paused && the_game->state == DeclaringSpecials ) {
    if ( the_game->info.player == our_seat )
      ds_dialog_popup();
    else
      gtk_widget_hide(ds_dialog);
    gtk_widget_hide(turn_dialog);
    if ( the_game->info.konging 
	 && the_game->info.player != our_seat
	 && the_game->info.claims[our_seat] == UnknownClaim ) {
      seats s;
      /* which player display is it? */
      s = the_game->info.player; /* seat */
      s = (s+NUM_SEATS-our_seat)%NUM_SEATS; /* posn relative to us */
      discard_dialog_popup(the_game->info.tile,s,2);
    } else
      gtk_widget_hide(discard_dialog->widget);
  } else
    gtk_widget_hide(ds_dialog);

  
    
  /* In the mahjonging state, if we're the winner:
     If the pending flag is set, pop up the discard dialog,
     unless the chowdialog is up.
     Otherwise, pop down both it and the chow dialog.
     Pop down the turn dialog.
  */
  if ( the_game->state == MahJonging
       && the_game->info.player == our_seat ) {
    gtk_widget_hide(turn_dialog);
    if ( the_game->active  && !the_game->paused
	 && the_game->info.pending 
	 && ! GTK_WIDGET_VISIBLE(chowdialog) ) {
      int s;
      /* which player display is it? */
      s = the_game->info.supplier;
      s = (s+NUM_SEATS-our_seat)%NUM_SEATS; /* posn relative to us */
      discard_dialog_popup(the_game->info.tile,s,1);
    } else {
      gtk_widget_hide(discard_dialog->widget);
      gtk_widget_hide(chowdialog);
    }
  }

  /* In the mahjonging state, if our hand is not declared,
     and the pending flag is not set, and if either we
     are the winner or the winner's hand is declared
     popup the scoring dialog, otherwise down */
  if ( the_game->active && !the_game->paused
       && the_game->state == MahJonging
       && !the_game->info.pending
       && !pflag(our_player,HandDeclared)
       && (the_game->info.player == our_seat
	   || pflag(the_game->players[the_game->info.player],HandDeclared) ) ) {
    /* if we don't have a tile selected and we're the winner, select the first */
    if ( selected_button < 0 && the_game->info.player == our_seat ) 
      conc_callback(pdisps[0].conc[0],(gpointer)(0 | 128));
    scoring_dialog_popup();
  } else {
    gtk_widget_hide(scoring_dialog);
  }

  /* If the game is not active, or active but paused, 
     pop up the continue dialog */ 
  if ( (!the_game->active) || the_game->paused )
    continue_dialog_popup();
  else
    gtk_widget_hide(continue_dialog);

  /* make sure the discard dialog is down */
  if ( the_game->state == HandComplete ) gtk_widget_hide(discard_dialog->widget);


  /* after a HandScore message, stick the text in the display */
  if ( m->type == CMsgHandScore || m->type == CMsgSettlement ) {
    CMsgHandScoreMsg *hsm = (CMsgHandScoreMsg *)m;
    CMsgSettlementMsg *sm = (CMsgSettlementMsg *)m;
    int pos,npos=0;
    char *text;

    if ( m->type == CMsgHandScore ) {
      pos = (game_id_to_seat(the_game,hsm->id)+NUM_SEATS-our_seat)%NUM_SEATS;
      text = hsm->explanation;
    } else {
      pos = 4;
      text = sm->explanation;
    }

    gtk_editable_insert_text(GTK_EDITABLE(textpages[pos]),
			     text,strlen(text),&npos);
    if ( m->type == CMsgSettlement ) {
      gtk_widget_show(textwindow);
      gdk_window_show(textwindow->window); /* uniconify */
    }
  }
}

void stdin_input_callback(gpointer          data UNUSED,
			  gint              source, 
			  GdkInputCondition condition  UNUSED) {
  char *l;

  l = get_line(source);
  put_line(the_game->fd,l);
}


/* given an already allocated PlayerDisp* and an orientation, 
   initialize a player display.
   The player widget is NOT shown, but everything else (appropriate) is */
void playerdisp_init(PlayerDisp *pd, int ori) {
  /* here is the widget hierarchy we want:
     w -- the parent widget
       concealed  -- the concealed row
         csets[MAX_TILESETS]  -- five boxes which will have tiles packed in them
	 conc  -- the concealed & special tile subbox, with...
           conc[14] -- 14 tile buttons
           extras[8] -- more spaces for specials if we run out.
	                pdispwidth-14 are initially set, to get the spacing.
       exposed  -- the exposed row 
           esets[MAX_TILESETS] -- with sets, each of which has
                       tile buttons as appropriate.
           specbox --
             spec[8]  -- places for specials
	               (Not all of these can always be used; so we may
		       need to balance dynamically between this and extras.)
	     tongbox  -- the tong box pixmap widget 
  */
  GtkWidget *concealed, *exposed, *conc, *c, *b, *pm, *w,*specbox;
  int i;
  int tb,br,bl;
  int eori; /* orientation of exposed tiles, if different */

  pd->player = NULL;
  pd->orientation = ori;
  eori = flipori(ori);

  /* sometimes it's enough to know whether the player is top/bottom,
     or the left/right */
  tb = ! (ori%2); 
  /* and sometimes the bottom and right players are similar
     (they are to the increasing x/y of their tiles */
  br = (ori < 2);
  /* and sometimes the bottom and left are similar (their right
     is at the positive and of the box */
  bl = (ori == 0 || ori == 3);

  /* homogeneous to force exposed row to right height */
  if ( tb )
    w = gtk_vbox_new(1,tile_spacing);
  else
    w = gtk_hbox_new(1,tile_spacing);
  gtk_widget_set_style(w,tablestyle);

  pd->widget = w;
  gtk_container_set_border_width(GTK_CONTAINER(w),player_border_width);

  if ( tb )
    concealed = gtk_hbox_new(0,tile_spacing);
  else
    concealed = gtk_vbox_new(0,tile_spacing);
  gtk_widget_set_style(concealed,tablestyle);
  gtk_widget_show(concealed);
  /* the concealed box is towards the player */
  /* This box needs to stay expanded to the width of the surrounding,
     which should be fixed at start up.*/
  if ( br )
    gtk_box_pack_end(GTK_BOX(w),concealed,1,1,0);
  else
    gtk_box_pack_start(GTK_BOX(w),concealed,1,1,0);

  /* players put closed (nonkong) tilesets in to the left of their
     concealed tiles */
  for ( i=0; i < MAX_TILESETS; i++) {
    if ( tb )
      c = gtk_hbox_new(0,0);
    else
      c = gtk_vbox_new(0,0);
    if ( bl )
      gtk_box_pack_start(GTK_BOX(concealed),c,0,0,0);
    else
      gtk_box_pack_end(GTK_BOX(concealed),c,0,0,0);
    pd->csets[i].widget = c;
    tilesetbox_init(&pd->csets[i],ori,NULL,0);
  }

  if ( tb )
    conc = gtk_hbox_new(0,0);
  else 
    conc = gtk_vbox_new(0,0);
  gtk_widget_set_style(conc,tablestyle);
  gtk_widget_show(conc);

  /* the concealed tiles are to the right of the concealed sets */
  /* This also needs to stay expanded */
  if ( bl )
    gtk_box_pack_start(GTK_BOX(concealed),conc,1,1,0);
  else
    gtk_box_pack_end(GTK_BOX(concealed),conc,1,1,0);

  for ( i=0; i < 14; i++ ) {
    /* normally, just buttons, but for us toggle buttons */
    if ( ori == 0 )
      b = gtk_toggle_button_new();
    else
      b = gtk_button_new();
    /* we don't want any of the buttons taking the focus */
    GTK_WIDGET_UNSET_FLAGS(b,GTK_CAN_FOCUS);
    gtk_widget_show(b);
    pm = gtk_pixmap_new(tilepixmaps[ori][HiddenTile],NULL);
    gtk_widget_show(pm);
    gtk_container_add(GTK_CONTAINER(b),pm);
    /* the concealed tiles are placed from the player's left */
    if ( bl )
      gtk_box_pack_start(GTK_BOX(conc),b,0,0,0);
    else
     gtk_box_pack_end(GTK_BOX(conc),b,0,0,0);
    pd->conc[i] = b;
    /* if this is our player (ori = 0), attach a callback */
    if ( ori == 0 ) {
      gtk_signal_connect(GTK_OBJECT(b),"toggled",
			 GTK_SIGNAL_FUNC(conc_callback),(gpointer)i);
      gtk_signal_connect(GTK_OBJECT(b),"button_press_event",
			 GTK_SIGNAL_FUNC(doubleclicked),0);
    }
  }

  /* to the right of the concealed tiles, we will place
     up to 8 other tiles. These will be used for flowers
     and seasons when we run out of room in the exposed row.
     By setting pdispwidth-14 (default 5) of them initially,
     it also serves the purpose of fixing the row to be
     the desired 19 (currently) tiles wide */
  for ( i=0; i < 8; i++ ) {
    b = gtk_button_new();
    /* we don't want any of the buttons taking the focus */
    GTK_WIDGET_UNSET_FLAGS(b,GTK_CAN_FOCUS);
    if ( i < pdispwidth-14 ) gtk_widget_show(b);
    pm = gtk_pixmap_new(tilepixmaps[ori][HiddenTile],NULL);
    gtk_widget_show(pm);
    gtk_container_add(GTK_CONTAINER(b),pm);
    /* these extra tiles are placed from the player's right */
    if ( bl )
      gtk_box_pack_end(GTK_BOX(conc),b,0,0,0);
    else
      gtk_box_pack_start(GTK_BOX(conc),b,0,0,0);
    pd->extras[i] = b;
  }



  
  if ( tb )
    exposed = gtk_hbox_new(0,tile_spacing);
  else 
    exposed = gtk_vbox_new(0,tile_spacing);
  gtk_widget_set_style(exposed,tablestyle);
  gtk_widget_show(exposed);
  /* the exposed box is away from the player */
  if ( br )
    gtk_box_pack_start(GTK_BOX(w),exposed,0,0,0);
  else
    gtk_box_pack_end(GTK_BOX(w),exposed,0,0,0);

  /* the exposed tiles will be in front of the player, from
     the left */
  for ( i=0; i < MAX_TILESETS; i++ ) {
    if ( tb )
      c = gtk_hbox_new(0,0);
    else
      c = gtk_vbox_new(0,0);
    gtk_widget_set_style(c,tablestyle);
    gtk_widget_show(c);
    if ( bl )
      gtk_box_pack_start(GTK_BOX(exposed),c,0,0,0);
    else
      gtk_box_pack_end(GTK_BOX(exposed),c,0,0,0);
    pd->esets[i].widget = c;
    tilesetbox_init(&pd->esets[i],eori,NULL,0);
  }
  /* the special tiles are at the right of the exposed row */
  if ( tb )
    specbox = gtk_hbox_new(0,0);
  else
    specbox = gtk_vbox_new(0,0);
  gtk_widget_set_style(specbox,tablestyle);
  gtk_widget_show(specbox);
  if ( bl )
    gtk_box_pack_end(GTK_BOX(exposed),specbox,0,0,0);
  else
    gtk_box_pack_start(GTK_BOX(exposed),specbox,0,0,0);

  /* at the right of the spec box, we place a tongbox pixmap.
     This is not initially shown */
  pd->tongbox = gtk_pixmap_new(tongpixmaps[ori][east],tongmask);
  if ( bl )
    gtk_box_pack_end(GTK_BOX(specbox),pd->tongbox,0,0,0);
  else
    gtk_box_pack_start(GTK_BOX(specbox),pd->tongbox,0,0,0);

  for ( i=0 ; i < 8; i++ ) {
    b = gtk_button_new();
    /* we don't want any of the buttons taking the focus */
    GTK_WIDGET_UNSET_FLAGS(b,GTK_CAN_FOCUS);
    pm = gtk_pixmap_new(tilepixmaps[eori][HiddenTile],NULL);
    gtk_widget_show(pm);
    gtk_container_add(GTK_CONTAINER(b),pm);
    if ( bl )
      gtk_box_pack_end(GTK_BOX(specbox),b,0,0,0);
    else
      gtk_box_pack_start(GTK_BOX(specbox),b,0,0,0);
    pd->spec[i] = b;
  }

  /* the player's discard buttons are created as required,
     so just zero them */
  for ( i=0; i<32; i++ ) pd->discards[i] = NULL;
  pd->num_discards = 0;
  /* also initialize the info kept by the discard routine */
  pd->x = pd->y = 0;
  pd->row = 0;
  pd->plane = 0;
  for (i=0;i<5;i++) {
    pd->xmin[i] = 10000;
    pd->xmax[i] = 0;
  }

  /* Claim window */
  { 
    static GtkStyle *s;
    GdkColor c;
    if ( !s ) {
      s = gtk_style_copy(gtk_widget_get_default_style());
      gdk_color_parse("yellow",&c);
      s->bg[GTK_STATE_NORMAL] = c;
      if ( big_font ) s->font = big_font;
    }
    gtk_widget_push_style(s);
    pd->claimw = gtk_window_new(GTK_WINDOW_POPUP);
    /* should parametrize this some time */
    gtk_container_set_border_width(GTK_CONTAINER(pd->claimw),7);
    pd->claimlab = gtk_label_new("");
    pd->claim_serial = -1;
    pd->claim_time = -10000;
    gtk_widget_show(pd->claimlab);
    gtk_container_add(GTK_CONTAINER(pd->claimw),pd->claimlab);
    gtk_widget_pop_style();
  }
}

/* given a button, tile, and orientation, set the pixmap (and tooltip when
   we have them. Stash the tile in the user_data field. */
void button_set_tile(GtkWidget *b, Tile t, int ori) {
  gint32 ti;
  gtk_pixmap_set(GTK_PIXMAP(GTK_BIN(b)->child),
		 tilepixmaps[ori][t],NULL);
  ti = t;
  gtk_object_set_user_data(GTK_OBJECT(b),(gpointer)ti);
}

/* given a pointer to a TileSetBox, and an orientation,
   and a callback function and data
   initialize the box by creating the widgets and attaching
   the signal function.
*/
void tilesetbox_init(TileSetBox *tb, int ori,
		    GtkSignalFunc func,gpointer func_data) {
  int i;
  GtkWidget *b, *pm;

  for (i=0; i<4; i++) {
    if ( ! tb->tiles[i] ) {
      /* make a button */
      b = gtk_button_new();
      tb->tiles[i] = b;
      if ( func )
	gtk_signal_connect(GTK_OBJECT(b),"clicked",
			   func,func_data);
      /* we don't want any of the tiles taking the focus */
      GTK_WIDGET_UNSET_FLAGS(b,GTK_CAN_FOCUS);
      pm = gtk_pixmap_new(tilepixmaps[ori][HiddenTile],NULL);
      gtk_widget_show(pm);
      gtk_container_add(GTK_CONTAINER(b),pm);
      /* we want chows to grow from top and left for left and bottom */
      if ( ori == 0 || ori == 3 )
	gtk_box_pack_start(GTK_BOX(tb->widget),b,0,0,0);
      else 
	gtk_box_pack_end(GTK_BOX(tb->widget),b,0,0,0);
    }
  }

  tb->set.type = Empty;
}

/* given a pointer to a TileSetBox, and a pointer to a TileSet,
   and an orientation,
   make the box display the tileset.
*/
void tilesetbox_set(TileSetBox *tb, const TileSet *ts, int ori) {
  int n, i, ck;
  Tile t[4];

  if (tb->set.type == ts->type && tb->set.tile == ts->tile) {
    /* nothing to do, except show for safety */
    if ( ts->type == Empty )
      gtk_widget_hide(tb->widget);
    else
      gtk_widget_show(tb->widget);
    return;
  }

  tb->set = *ts;
  
  n = 2; /* for pairs etc */
  ck = 0; /* closed kong */
  switch (ts->type) {
  case ClosedKong:
    ck = 1;
  case Kong:
    n++;
  case Pung:
  case ClosedPung:
    n++;
  case Pair:
  case ClosedPair:
    for ( i=0 ; i < n ; i++ ) {
      t[i] = ts->tile;
      if ( ck && ( i == 0 || i == 3 ) ) t[i] = HiddenTile;
    }
    break;
  case Chow:
  case ClosedChow:
    n = 3;
    for (i=0; i<n; i++) {
      t[i] = ts->tile+i;
    }
    break;
  case Empty:
    n = 0;
  }

  for (i=0; i<n; i++) {
    button_set_tile(tb->tiles[i],t[i],ori);
    gtk_widget_show(tb->tiles[i]);
  }
  for ( ; i < 4; i++ )
    if ( tb->tiles[i] ) gtk_widget_hide(tb->tiles[i]);

  if ( ts->type == Empty )
    gtk_widget_hide(tb->widget);
  else
    gtk_widget_show(tb->widget);
}

/* utility used by the chow functions. Given a TileSetBox,
   highlight the nth tile by setting the others insensitive.
   (n counts from zero). */
void tilesetbox_highlight_nth(TileSetBox *tb,int n) {
  int i;
  for (i=0; i<4; i++)
    if ( tb->tiles[i] )
      gtk_widget_set_sensitive(tb->tiles[i],i==n);
}


/* two routines used internally */

/* update the concealed tiles of the playerdisp.
   Returns the *index* of the last tile matching the third argument.
*/
static int playerdisp_update_concealed(PlayerDisp *pd,Tile t) {
  PlayerP p = pd->player;
  int tindex = -1;
  int i;
  int cori,eori;

  cori = pd->orientation;
  eori = flipori(cori);
  if ( p && pflag(p,HandDeclared) ) cori = eori;

  for ( i=0 ; p && i < p->num_concealed ; i++ ) {
    button_set_tile(pd->conc[i],p->concealed[i],cori);
    gtk_widget_show(pd->conc[i]);
    if ( p->concealed[i] == t ) tindex = i;
  }
  for ( ; i < MAX_CONCEALED ; i++ ) {
    gtk_widget_hide(pd->conc[i]);
  }
  return tindex;
}

/* update the exposed tiles of the playerdisp.
   Returns the _button widget_ of the last tile that
   matched the second argument.
   The third argument causes special treatment: an exposed pung matching
   the tile is displayed as a kong.
*/
static GtkWidget *playerdisp_update_exposed(PlayerDisp *pd,Tile t,int robbingkong){
  int i,scount,excount,ccount,ecount,qtiles;
  GtkWidget *w;
  TileSet emptyset;
  PlayerP p = pd->player;
  int cori, eori; /* orientations of concealed and exposed tiles */

  cori = pd->orientation;
  eori = flipori(cori);
  emptyset.type = Empty;
  if ( p && pflag(p,HandDeclared) ) cori = eori;

  w = NULL; /* destination widget */
  /* We need to count the space taken up by the exposed tilesets,
     in order to know where to put the specials */
  qtiles = 0; /* number of quarter tile widths */
  ccount = ecount = 0;
  for ( i=0; p && i < MAX_TILESETS; i++) {
    switch (p->tilesets[i].type) {
    case ClosedPair:
    case ClosedChow:
      /* in the concealed row, so no space */
      /* NB, these are declared, they are oriented as
	 exposed sets, despite the name */
      tilesetbox_set(&pd->csets[ccount++],&p->tilesets[i],eori);
      break;
    case ClosedPung:
      /* in the special case that we are in the middle of robbing
	 a kong (13 wonders), and this is the robbed set, we keep
	 it in the exposed row displayed as a kong */
      if ( robbingkong && t == p->tilesets[i].tile) {
	TileSet ts = p->tilesets[i];
	ts.type = ClosedKong;
	tilesetbox_set(&pd->esets[ecount],&ts,eori);
	w = pd->esets[ecount].tiles[3];
	qtiles += 16; /* four tiles */
	ecount++;
      } else {
	tilesetbox_set(&pd->csets[ccount++],&p->tilesets[i],eori);
      }
      break;
    case ClosedKong:
    case Kong:
      tilesetbox_set(&pd->esets[ecount],&p->tilesets[i],eori);
      if ( t == p->tilesets[i].tile )
	w = pd->esets[ecount].tiles[3];
      qtiles += 16; /* four tiles */
      ecount++;
      break;
    case Pung:
      if ( robbingkong && t == p->tilesets[i].tile) {
	TileSet ts = p->tilesets[i];
	ts.type = Kong;
	tilesetbox_set(&pd->esets[ecount],&ts,eori);
	w = pd->esets[ecount].tiles[3];
	qtiles += 16; /* four tiles */
      } else {
	tilesetbox_set(&pd->esets[ecount],&p->tilesets[i],eori);
	if ( t == p->tilesets[i].tile )
	  w = pd->esets[ecount].tiles[2];
	qtiles += 12;
      }
      ecount++;
      break;
    case Chow:
      tilesetbox_set(&pd->esets[ecount],&p->tilesets[i],eori);
      if ( t >= p->tilesets[i].tile
	   && t <= p->tilesets[i].tile+2 )
	w = pd->esets[ecount].tiles[t-p->tilesets[i].tile];
      qtiles += 12;
      ecount++;
      break;
    case Pair:
      tilesetbox_set(&pd->esets[ecount],&p->tilesets[i],eori);
      if ( t == p->tilesets[i].tile )
	w = pd->esets[ecount].tiles[1];
      qtiles += 9; /* for the two tiles and space */
      ecount++;
      break;
    case Empty:
      ;
    }
  }
  for ( ;  ccount < MAX_TILESETS; )
    tilesetbox_set(&pd->csets[ccount++],&emptyset,eori);
  for ( ;  ecount < MAX_TILESETS; )
    tilesetbox_set(&pd->esets[ecount++],&emptyset,eori);
  /* if we are the dealer, the tongbox takes up
     about 1.5 tiles */
  if ( p && p->wind == EastWind ) {
    gtk_pixmap_set(GTK_PIXMAP(pd->tongbox),tongpixmaps[cori][the_game->round-1],
		   tongmask);
    gtk_widget_show(pd->tongbox);
    qtiles += 6;
  } else {
    gtk_widget_hide(pd->tongbox);
  }
  
  /* for the special tiles, we put as many as
     possible in the exposed row (specs),
     and then overflow into the concealed row */
  qtiles = (qtiles+3)/4; /* turn quarter tiles into tiles */
  scount = excount = 0;
  for ( i = 0; p && i < p->num_specials
	  && i <= pdispwidth - qtiles ; i++ ) {
    button_set_tile(pd->spec[i],p->specials[i],eori);
    gtk_widget_show(pd->spec[i]);
    if ( t == p->specials[i] ) w = pd->spec[i];
    scount++;
  }
  for ( ; p && i < p->num_specials ; i++ ) {
    button_set_tile(pd->extras[i-scount],p->specials[i],eori);
    gtk_widget_show(pd->extras[i-scount]);
    if ( t == p->specials[i] ) w = pd->extras[i-scount];
    excount++;
  }
  for ( ; scount < 8; scount ++ )
    gtk_widget_hide(pd->spec[scount]);
  for ( ; excount < 8; excount ++ )
    gtk_widget_hide(pd->extras[excount]);
  
  return w;
}

/* given a playerdisp, update the hand.
   The second argument gives the message prompting this.
   Animation is handled entirely in here.
*/


void playerdisp_update(PlayerDisp *pd, CMsgUnion *m) {
  int i;
  Tile t; int tindex;
  static AnimInfo srcs[4], dests[4];
  TileSet emptyset;
  int numanims = 0;
  static GtkWidget *robbedfromkong; /* yech */
  PlayerP p = pd->player;
  int deftile = -1; /* default tile to be discarded */
  int cori, eori; /* orientations of concealed and exposed tiles */
  int updated = 0; /* flag to say we've done our stuff */

  cori = pd->orientation;
  eori = flipori(cori);
  emptyset.type = Empty;
  if ( p && pflag(p,HandDeclared) ) cori = eori;

  if ( m == NULL ) {
    /* just update */
    playerdisp_update_concealed(pd,HiddenTile);
    playerdisp_update_exposed(pd,HiddenTile,0);
    return;
  }

  if ( m->type == CMsgNewHand && p) {
    playerdisp_clear_discards(pd);
    playerdisp_update_concealed(pd,HiddenTile);
    playerdisp_update_exposed(pd,HiddenTile,0);
    return;
  }

  if ( m->type == CMsgPlayerDraws || m->type == CMsgPlayerDrawsLoose) {
    updated = 1;
    t = (m->type == CMsgPlayerDraws) ? m->playerdraws.tile : 
      m->playerdrawsloose.tile;
    tindex = playerdisp_update_concealed(pd,t);
    assert(tindex >= 0);
    /* do we want to select a tile? Usually, but in declaring
       specials state, we want to select the rightmost tile
       if it's a special and otherwise none */
    if ( p == our_player ) {
      if ( the_game->state == DeclaringSpecials ) {
	if ( is_special(p->concealed[p->num_concealed-1]) ) {
	  deftile = p->num_concealed-1;
	  conc_callback(pd->conc[deftile],(gpointer)(deftile | 128));
	} else {
	  conc_callback(NULL,NULL);
	}
      } else {
	deftile = tindex;
	conc_callback(pd->conc[deftile],(gpointer)(deftile | 128));
      }
    }
    dests[numanims].target = pd->conc[tindex];
    dests[numanims].t = t;
    get_relative_posn(boardframe,dests[numanims].target,&dests[numanims].x,&dests[numanims].y);
    /* that was the destination, now the source */
    if ( showwall ) {
      if ( m->type == CMsgPlayerDraws ) {
	int i;
	/* wall has already been updated, so it's -1 */
	i = wall_game_to_board(the_game->wall.live_used-1);
	srcs[numanims].target = NULL;
	srcs[numanims].t = m->playerdraws.tile; /* we may as well see our tile now */
	srcs[numanims].ori = wall_ori(i);
	get_relative_posn(boardframe,wall[i],&srcs[numanims].x,&srcs[numanims].y);
	gtk_widget_destroy(wall[i]);
	wall[i] = NULL;
      } else {
	/* draws loose */
	srcs[numanims] = *adjust_wall_loose();
	/* in fact, we may as well see the tile if we're drawing */
	srcs[numanims].t = t;
      }
    } else {
      /* if we're animating, we'll just have the tile spring up from
	 nowhere! */
      srcs[numanims].target = NULL;
      srcs[numanims].t = t; /* we may as well see our tile now */
      srcs[numanims].ori = cori;
      /* these should be corrected for orientation */
      srcs[numanims].x = boardframe->allocation.width/2 - tile_width/2;
      srcs[numanims].y = boardframe->allocation.height/2 - tile_height/2;
    }
    numanims++;
  } /* end of Draws and DrawsLoose */
  if ( m->type == CMsgPlayerDiscards ) {
    updated = 1;
    /* update the concealed tiles */
    for ( i=0 ; i < p->num_concealed ; i++ ) {
      button_set_tile(pd->conc[i],p->concealed[i],cori);
      gtk_widget_show(pd->conc[i]);
    }
    for ( ; i < MAX_CONCEALED ; i++ ) {
      gtk_widget_hide(pd->conc[i]);
    }
    /* the source for animation is the previous rightmost tile
       for other players, or the selected tile, for us.
       Note that selected button may not be set, if we're
       replaying history. In that case, we won't bother to
       set the animation position, in the knowledge that 
       it won't actually be animated.
       This is far too convoluted. 
    */
    tindex = (p == our_player) ? selected_button : p->num_concealed;
    srcs[numanims].t = m->playerdiscards.tile;
    srcs[numanims].ori = eori;
    if ( tindex >= 0 ) {
      get_relative_posn(boardframe,pd->conc[tindex],&srcs[numanims].x,
			&srcs[numanims].y);
    }
    /* now display the new discard */
    dests[numanims] = *playerdisp_discard(pd,m->playerdiscards.tile);
    /* if we're not animating, highlight the tile */
    if ( ! (animate && the_game->active) ) {
      gtk_widget_set_style(dests[numanims].target,highlightstyle);
      highlittile = dests[numanims].target;
    }
    /* if we're discarding, clear the selected tile */
    if ( p == our_player ) conc_callback(NULL,NULL);
    numanims++;
  } /* end of CMsgPlayerDiscards */
  if ( m->type == CMsgPlayerDeclaresSpecial ) {
    updated = 1;
    if ( m->playerdeclaresspecial.tile != HiddenTile ) {
      GtkWidget *w = NULL;
      /* the source tile in this case is either the tile
	 after the rightmost concealed tile (for other players)
	 or the selected tile (for us) */
      if ( p == our_player ) {
	if ( selected_button >= 0 ) w = pd->conc[selected_button];
	/* else we're probably replaying history */
      } else {
	w = pd->conc[p->num_concealed];
      }
      if ( w ) get_relative_posn(boardframe,w,&srcs[numanims].x,&srcs[numanims].y);
      srcs[numanims].t = m->playerdeclaresspecial.tile;
      srcs[numanims].ori = cori;
      /* now update the special tiles, noting the destination */
      w = playerdisp_update_exposed(pd,m->playerdeclaresspecial.tile,0);
      assert(w);
      /* this forces a resize calculation so that the special tiles
	 are in place */
      gtk_widget_size_request(pd->widget,NULL);
      gtk_widget_size_allocate(pd->widget,&pd->widget->allocation);
      dests[numanims].target = w;
      dests[numanims].ori = eori;
      get_relative_posn(boardframe,w,&dests[numanims].x,&dests[numanims].y);
      numanims++; /* that's it */
    } else {
      /* blank tile. nothing to do except select a tile if it's now
	 our turn to declare */
      if ( the_game->state == DeclaringSpecials
	   && the_game->info.player == our_seat ) {
	if ( is_special(our_player->concealed[our_player->num_concealed-1]) ) {
	  deftile = our_player->num_concealed-1;
	  conc_callback(pdisps[0].conc[deftile],(gpointer)(deftile | 128));
	} else {
	  conc_callback(NULL,NULL);
	}
      }
    }
  } /* end of CMsgPlayerDeclaresSpecial */
  if ( m->type == CMsgPlayerClaimsPung 
	    || m->type == CMsgPlayerClaimsKong 
	    || m->type == CMsgPlayerClaimsChow 
	    || m->type == CMsgPlayerClaimsMahJong
            || m->type == CMsgPlayerMahJongs 
            || (m->type == CMsgPlayerDiscards
		&& m->playerdiscards.calling) ) {
    /* popup a claim window.
       This is a real pain, cos we have to work out where to put it.
       To keep it simple, we'll put it half way along the appropriate
       edge, centered two tileheights in. */
    const char *lab;
    GtkRequisition r;
    gint x,y,w,h;
    
    updated = 1;
    switch ( m->type ) {
    case CMsgPlayerClaimsPung: lab = "Pung!"; break;
    case CMsgPlayerClaimsKong: lab = "Kong!"; break;
    case CMsgPlayerClaimsChow: lab = "Chow!"; break;
    case CMsgPlayerMahJongs:
    case CMsgPlayerClaimsMahJong: lab = "Mah Jong!"; break;
    case CMsgPlayerDiscards: lab = "Calling!" ; break;
    default: assert(0);
    }
    gtk_label_set_text(GTK_LABEL(pd->claimlab),lab);
    pd->claim_serial = the_game->info.serial;
    pd->claim_time = now_time();
    /* if not already popped up */
    if ( !GTK_WIDGET_VISIBLE(pd->claimw) ) {
      gdk_window_get_deskrelative_origin(topwindow->window,&x,&y);
      /* gdk_window_get_size(topwindow->window,&w,&h); */
      gtk_widget_size_request(board,&r);
      w = r.width; h = r.height;
      gtk_widget_size_request(menubar,&r);
      y += r.height;
      gtk_widget_size_request(pd->claimw,&r);
      switch ( cori ) {
      case 0:
	x = x + w/2 - r.width/2;
	y = y + h - r.height/2 - 2*tile_height;
	break;
      case 1:
	x = x + w - r.width/2 - 2*tile_height;
	y = y + h/2 - r.height/2;
	break;
      case 2:
	x = x + w/2 - r.width/2;
	y = y - r.height/2 + 2*tile_height;
	break;
      case 3:
	x = x - r.width/2 + 2*tile_height;
	y = y + h/2 - r.height/2;
	break;
      }
      gtk_widget_set_uposition(pd->claimw,x,y);
      gtk_widget_show(pd->claimw);
    }
  } /* end of the claims */
  /* On receiving a successful claim, withdraw the last discard.
     And if mahjonging, select the first tile if this is us */
  if ( (m->type == CMsgPlayerChows
	&& ((CMsgPlayerChowsMsg *)m)->cpos != AnyPos)
       || m->type == CMsgPlayerPungs
       || m->type == CMsgPlayerKongs
       || m->type == CMsgPlayerPairs
       || m->type == CMsgPlayerSpecialSet ) {
    GtkWidget *w;

    updated = 1;
    /* withdraw discard, or use the saved widget if a kong was robbed */
    if ( the_game->info.whence == FromRobbedKong ) {
      srcs[numanims].t = the_game->info.tile;
      get_relative_posn(boardframe,robbedfromkong,
			&srcs[numanims].x,&srcs[numanims].y);
      playerdisp_update_exposed(&pdisps[game_to_board(the_game->info.supplier)],HiddenTile,0);
    } else {
      srcs[numanims] = *playerdisp_discard(&pdisps[game_to_board(the_game->info.supplier)],HiddenTile);
    }
    /* update exposed tiles, and get the destination widget */
    w = playerdisp_update_exposed(pd,the_game->info.tile,0);
    /* In the case of a special set, the destination tile will be
       in the concealed tiles */
    /* update the concealed tiles. At present we don't animate the
       movement of tiles from hand to exposed sets, so just update */
    if ( m->type == CMsgPlayerSpecialSet ) {
      w = 
	pd->conc[playerdisp_update_concealed(pd,the_game->info.tile)];
    } else {
      playerdisp_update_concealed(pd,HiddenTile);
    }
    assert(w);
    /* this should force a resize calculation so that 
       the declared tiles are in place. It seems to work. */
    gtk_widget_size_request(pd->widget,NULL);
    gtk_widget_size_allocate(pd->widget,&pd->widget->allocation);
    dests[numanims].target = w;
    srcs[numanims].ori = dests[numanims].ori = eori; 
    get_relative_posn(boardframe,w,&dests[numanims].x,&dests[numanims].y);
    numanims++; /* that's it */

    if ( the_game->state == MahJonging
	 && the_game->info.player == our_seat 
	 && p == our_player
	 && p->num_concealed > 0 )
      conc_callback(pd->conc[0],(gpointer)(0 | 128));
  } /* end of the successful claim cases */
  else if ( m->type == CMsgPlayerRobsKong ) {
    /* This should be called on the ROBBED player, so we
       can stash away the tile that will be robbed */
    robbedfromkong = playerdisp_update_exposed(pd,m->playerrobskong.tile,1);
    updated = 1;
  } /* end of robbing kong */
  /* if not yet handled, just update the tiles */
  if ( ! updated ) {
    playerdisp_update_concealed(pd,HiddenTile);
    playerdisp_update_exposed(pd,HiddenTile,0);
  }
  /* now kick off the animations (or not) */
  while ( numanims-- > 0 ) {
    if ( animate && the_game->active && the_game->state != Dealing ) {
      gtk_widget_unmap(dests[numanims].target);
      start_animation(&srcs[numanims],&dests[numanims]);
    } 
  }
}

/* utility function: given a gdkpixmap, create rotated versions
   and put them in the last three arguments.
*/
void rotate_pixmap(GdkPixmap *east,
			  GdkPixmap **south,
			  GdkPixmap **west,
			  GdkPixmap **north) {
  static GdkVisual *v = NULL;
  int w=-1, h=-1;
  int x, y;
  GdkImage *im;
  GdkPixmap *p1;
  GdkImage *im1;
  GdkGC *gc;
  
  if ( v == NULL ) {
    v = gdk_window_get_visual(topwindow->window);
  }

  gdk_window_get_size(east,&w,&h);
  /* remember to destroy these */
  im = gdk_image_get(east,0,0,w,h);
  gc = gdk_gc_new(east);

  /* do the south */
  im1 = gdk_image_new(GDK_IMAGE_NORMAL,v,h,w);
  for ( x=0; x < w; x++ )
    for ( y=0; y < h; y++ )
      gdk_image_put_pixel(im1,y,w-1-x,gdk_image_get_pixel(im,x,y));
  p1 = gdk_pixmap_new(NULL,h,w,im->depth);
  gdk_draw_image(p1,gc,im1,0,0,0,0,h,w);
  *south = p1;
  /* do the north */
  for ( x=0; x < w; x++ )
    for ( y=0; y < h; y++ )
      gdk_image_put_pixel(im1,h-1-y,x,gdk_image_get_pixel(im,x,y));
  p1 = gdk_pixmap_new(NULL,h,w,im->depth);
  gdk_draw_image(p1,gc,im1,0,0,0,0,h,w);
  *north = p1;

  /* and the west */
  gdk_image_destroy(im1);
  im1 = gdk_image_new(GDK_IMAGE_NORMAL,v,w,h);
  for ( x=0; x < w; x++ )
    for ( y=0; y < h; y++ )
      gdk_image_put_pixel(im1,w-1-x,h-1-y,gdk_image_get_pixel(im,x,y));
  p1 = gdk_pixmap_new(NULL,w,h,im->depth);
  gdk_draw_image(p1,gc,im1,0,0,0,0,w,h);
  *west = p1;

  /* clean up */
  gdk_image_destroy(im1);
  gdk_image_destroy(im);
  gdk_gc_destroy(gc);
}

/* new animation routines. The previous way was all very object-oriented,
   but not very good: having separate timeouts for every animated tile
   tends to slow things done. So we will instead maintain a global list
   of animations in progress (of which there will be at most five!),
   and have a single timeout that deals with all of them */
/* struct for recording information about an animation in progress */
struct AnimProg {
  GtkWidget *floater; /* the moving window. Set to NULL if this record
			 is not in use */
  gint x1,y1,x2,y2; /* start and end posns ***relative to root*** */
  gint steps, n; /* total number of steps, number completed */
  GtkWidget *target; /* target widget */
};

struct AnimProg animlist[5];
int animator_running = 0;

/* timeout function that moves all running animations by one step */
static gint discard_animator(gpointer data UNUSED) {
  struct AnimProg *animp;
  int i; int numinprogress = 0;

  for ( i = 0 ; i < 5 ; i++ ) {
    animp = &animlist[i];
    if ( ! animp->floater ) continue;
    if ( animp->n == 0 ) {
      gtk_widget_destroy(animp->floater);
      animp->floater = NULL;
      gtk_widget_show(animp->target);
      gtk_widget_map(animp->target);
    } else {
      numinprogress++;
      animp->n--;
      gdk_window_move(animp->floater->window,
		      (animp->n*animp->x1 
		       + (animp->steps-animp->n)*animp->x2)
		      /animp->steps,
		      (animp->n*animp->y1 
		       + (animp->steps-animp->n)*animp->y2)
		      /animp->steps);
    }
  }
  if ( numinprogress > 0 ) { return TRUE ; }
  else { animator_running = 0; return FALSE; }
}

/* kick off an animation from the given source and target info */
void start_animation(AnimInfo *source,AnimInfo *target) {
    GtkWidget *floater,*b;
    int i;
    struct AnimProg *animp;
    GtkWidget *p;
    gint xoff,yoff;

    /* find a free slot in the animation list */
    for ( i = 0 ; animlist[i].floater && i < 5 ; i++ ) ;
    if ( i == 5 ) {
      warn("no room in animation list; not animating");
      gtk_widget_map(target->target);
      return;
    }
    animp = &animlist[i];

    /* create the floating button */
    b = gtk_button_new();
    gtk_widget_show(b);
    GTK_WIDGET_UNSET_FLAGS(b,GTK_CAN_FOCUS);
    p = gtk_pixmap_new(tilepixmaps[0][HiddenTile],NULL);
    gtk_widget_show(p);
    gtk_container_add(GTK_CONTAINER(b),p);
    button_set_tile(b,source->t,source->ori);
    gtk_widget_show(b);
    floater = gtk_window_new(GTK_WINDOW_POPUP);
    gtk_container_add(GTK_CONTAINER(floater),b);
    gdk_window_get_origin(boardframe->window,&xoff,&yoff);
    animp->x1 = source->x+xoff;
    animp->x2 = target->x+xoff;
    animp->y1 = source->y+yoff;
    animp->y2 = target->y+yoff;
    /* how many steps? the frame interval is 20ms.
       We want it to move across the screen in about a second max.
       So say 1600 pixels in 1 second, i.e. 50 frames.
       However, we don't want it to move instantaneously over short distances,
       so let's say a minimum of 10 frames. Hence
       numframes = 10 + 40 * dist/1600 */
    animp->steps =  10 + floor(hypot(source->x-target->x,source->y-target->y))*40/1600;
    animp->n = animp->steps;
    animp->target = target->target;
    gtk_widget_set_uposition(floater,source->x+xoff,source->y+yoff);
    gtk_widget_show(floater);
    animp->floater = floater;
    if ( ! animator_running ) {
      animator_running = 1;
      gtk_timeout_add(20,discard_animator,(gpointer)NULL);
    }
}

/* playerdisp_clear_discards: clear the discard area for the player */
void playerdisp_clear_discards(PlayerDisp *pd) {
  int i;

  /* destroy ALL widgets, since there might be widgets after
     the current last discard (and will be, if the last mah jong
     was from the wall */
  for (i=0; i < 32; i++) {
    if ( pd->discards[i] ) {
      gtk_widget_destroy(pd->discards[i]);
      pd->discards[i] = NULL;
    }
  }
  highlittile = NULL;
  pd->num_discards = 0;
  pd->x = pd->y = pd->row = pd->plane = 0;
  for (i=0;i<5;i++) { pd->xmin[i] = 10000; pd->xmax[i] = 0; }
}
    
/* Given a player disp and a tile, display the tile as 
   the player's next discard. If HiddenTile, then withdraw
   the last discard. Also, if pd is NULL: assume pd is the
   last player on which we were called.
   Buttons are created as required.
   Tiles are packed in rows, putting a 1/4 tile spacing between them
   horizontally and vertically. 
   In the spirit of Mah Jong, each player grabs the next available space,
   working away from it; a little cleverness detects clashes.
   Returns an animinfo which, in the normal case, is the position
   of the new discard.
   In the withdrawing case, it's the discard being withdrawn.
*/
AnimInfo *playerdisp_discard(PlayerDisp *pd,Tile t) {
  static PlayerDisp *last_pd = NULL;
  static gint16 dw,dh; /* width and height of discard area */
  static gint16 bw,bh; /* width and height of tile button in ori 0 */
  static gint16 spacing;
  gint16 w,h; /* width and height of discard area viewed by this player */
  int row,plane; /* which row and plane are we in */
  int i;
  gint16 x,y,xx,yy;
  PlayerDisp *left,*right; /* our left and right neighbours */
  GtkWidget *b;
  int ori, eori;
  static AnimInfo anim;

  if ( pd == NULL ) pd = last_pd;
  last_pd = pd;
  ori = pd->orientation;
  eori = flipori(ori);

  if ( highlittile ) {
    gtk_widget_restore_default_style(highlittile);
    highlittile = NULL;
  }

  if ( t == HiddenTile ) {
    if ( pd == NULL ) {
      warn("Internal error: playerdisp_discard called to clear null discard");
      return NULL;
    }
    b = pd->discards[--pd->num_discards];
    if ( b == NULL ) {
      warn("Internal error: playerdisp_discard clearing nonexistent button");
      return NULL;
    }
    anim.t = (int) gtk_object_get_user_data(GTK_OBJECT(b));
    anim.target = b;
    anim.ori = eori;
    get_relative_posn(boardframe,b,&anim.x,&anim.y);
    gtk_widget_hide(b);
    return &anim;
  }

  ori = pd->orientation;
  eori  = flipori(ori);
  b = pd->discards[pd->num_discards];
  left = &pdisps[(ori+3)%4];
  right = &pdisps[(ori+1)%4];

  if ( b == NULL ) {
    GtkWidget *p;
    b = pd->discards[pd->num_discards] = gtk_button_new();
    GTK_WIDGET_UNSET_FLAGS(b,GTK_CAN_FOCUS);
    p = gtk_pixmap_new(tilepixmaps[0][HiddenTile],NULL);
    gtk_widget_show(p);
    gtk_container_add(GTK_CONTAINER(b),p);
    /* if this is the first time we've ever been called, we need
       to compute some information */
    if ( dw == 0 ) {
      /* This discard area alloc was computed at a known safe time
	 in the main routine */
      dw = discard_area_alloc.width
	- 2 * GTK_CONTAINER(discard_area)->border_width;
      dh = discard_area_alloc.height
	- 2 * GTK_CONTAINER(discard_area)->border_width;
      bw = tile_width;
      bh = tile_height;
      spacing = tile_spacing;
      /* if we're showing the wall, the effective dimensions are
	 is reduced by 2*(tileheight+3*tilespacing)  --
         each wall occupies tile_height+2*tile_spacing, and we
	 also want some space */
      if ( showwall ) {
	dw -= 2*(tile_height+3*tile_spacing);
	dh -= 2*(tile_height+3*tile_spacing);
      }
    }

    /* Now we need to work out where to put this button.
       To make life easier, we will always look at it as if
       we were at the bottom, and then rotate the answer later */
    /* we can't assume the discard area is square, so swap w and h
       if we're working for a side player */
    if ( ori == 0 || ori == 2 ) {
      w = dw; h = dh;
    } else {
      w = dh; h = dw;
    }
    /* get the positions from last time */
    x = pd->x; y = pd->y; row = pd->row; plane = pd->plane;
    /* working in the X coordinate system is too painful.
       We work in a normal Cartesian system and convert
       only at the last moment */
    /* x,y is the pos of the bottom left corner */
    /* repeat until success */
    while ( 1 ) {
      /* Does the current position intrude into the opposite area?
	 If so, we're stuffed, and need to think what to do */
      /* also if we've run out of rows */
      if ( row >= 5 || y+bh > h/2 ) {
	/* start stacking tiles */
	plane++;
	row = 0;
	x = 0 + plane*(bw+spacing)/2; /* two planes is enough, surely! */
	y = 0 + plane*(bh+spacing)/2;
	continue;
      }
      /* Does the current position intrude into the left hand neighbour?
	 The left hand neighbour has stored in its xmax array
	 the rightmost points it's used in each row. So for each of
	 its rows, we see if either our top left or top right intrude
	 on it, and if so we move along */
      for (i=0; i<5; i++) {
	if ( (
	      /* our top left is in the part occupied by the tile */
	      (x >= i*(bh+spacing) && x < i*(bh+spacing)+bh)
	      /* ditto for top right */
	      || (x+bw >= i*(bh+spacing) && x+bw < i*(bh+spacing)+bh) )
	     /* and our top intrudes */
	     && y+bh > h - left->xmax[i] )
	  x = i*(bh+spacing)+bh+spacing/2; /* half space all that's needed, I think */
      }
      
      /* Now see if the current position interferes with the right
	 neighbour similarly 
       */
      for (i=0; i<5; i++) {
	if ( (
	      /* our top left is in the column */
	      (x >= w-(i*(bh+spacing)+bh) && x < w - i*(bh+spacing))
	      /* ditto for top right */
	      || (x+bw >= w-(i*(bh+spacing)+bh) && x+bw < w - i*(bh+spacing)))
	     /* and our top intrudes */
	     && y+bh > right->xmin[i] ) {
	  /* we have to move up to the next row */
	  row++;
	  x = 0;
	  y += (bh+spacing);
	  i=-1; /* signal */
	  break;
	}
      }
      /* and restart the loop if we moved */
      if ( i < 0 ) continue;

      /* Phew, we've found it, if we get here */
      /* store our information for next time */
      pd->row = row;
      pd->plane = plane;
      pd->x = x+bw+spacing;
      pd->y = y;

      if ( x < pd->xmin[row] ) pd->xmin[row] = x;
      if ( x+bw > pd->xmax[row] ) pd->xmax[row] = x+bw;
      break;
    }
    
    /* Now we have to rotate for the appropriate player.
       This means giving a point for the top left corner.
       And switching y round */
    xx = x; yy = y;
    switch ( ori ) {
    case 0: x = xx; y = dh-yy-bh; break;
    case 1: y = dh - xx - bw; x = dw - yy - bh; break;
    case 2: x = dw - xx - bw; y = yy; break;
    case 3: y = xx; x = yy; break;
    }
    /* and if we are showing the wall, we have to shift everything
       down and right by the wall space */
    if ( showwall ) {
	x += (tile_height+3*tile_spacing);
	y += (tile_height+3*tile_spacing);
    }

    gtk_fixed_put(GTK_FIXED(discard_area),b,x,y);
    pd->dx[pd->num_discards] = x;
    pd->dy[pd->num_discards] = y;
  } else {
    x = pd->dx[pd->num_discards];
    y = pd->dy[pd->num_discards];
  }

  /* now we have a button created and in place: set it */
  button_set_tile(b,t,eori);

  /* now we need to compute the animation info to return */
  /* position is that of the button we've just created */
  /* unfortunately, if we've just created the button, it won't
     actually be in position (since it hasn't been realized),
     so we have to use our calculated position relative to the
     discard area */
  get_relative_posn(boardframe,discard_area,&anim.x,&anim.y);
  anim.x += x; anim.y += y;
  /* and don't forget the border of the discard area */
  anim.x += GTK_CONTAINER(discard_area)->border_width;
  anim.y += GTK_CONTAINER(discard_area)->border_width;
  anim.target = b;
  anim.t = t;
  anim.ori = eori;
  gtk_widget_show(b);
  pd->num_discards++;
  return &anim;
}

/* open_connection: connect to a server and announce,
   using values from the open dialog box where available.
   If data is one, start local server and players */
void open_connection(GtkWidget *w UNUSED, gpointer data) {
  char buf[256];
  int i;

  if ( GTK_WIDGET_SENSITIVE(openfile) ) {
    sprintf(buf,"%s",gtk_entry_get_text(GTK_ENTRY(openfiletext)));
  } else {
    sprintf(buf,"%s:%s",gtk_entry_get_text(GTK_ENTRY(openhosttext)),
	    gtk_entry_get_text(GTK_ENTRY(openporttext)));
  }

  /* start server ? */
  if ( data ) {
    char cmd[1024];
    char ibuf[10];

    strcpy(cmd,"mj-server --server ");
    strcat(cmd,buf);
    if ( ! GTK_TOGGLE_BUTTON(openallowdisconnectbutton)->active ) {
      strcat(cmd, " --exit-on-disconnect");
    }
    if ( GTK_TOGGLE_BUTTON(openrandomseatsbutton)->active ) {
      strcat(cmd, " --random-seats");
    }
    strcat(cmd, " --timeout ");
    sprintf(ibuf,"%d",
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(opentimeoutspinbutton)));
    strcat(cmd,ibuf);
    if ( ! start_background_program(cmd) ) {
      error_dialog_popup("Couldn't start server; unexpected failure");
      return;
    }
    sleep(2);
  }

  the_game = client_init(buf);
  if ( the_game == NULL ) {
    sprintf(buf,"Couldn't connect to server:\n%s",strerror(errno));
    if ( data ) {
      strcat(buf,"\nPerhaps server didn't start -- check error messages");
    }
    error_dialog_popup(buf);
    return;
  }
  our_player = the_game->players[0];

  gtk_widget_set_sensitive(openmenuentry,0);
  gtk_widget_set_sensitive(newgamemenuentry,0);
  gtk_widget_set_sensitive(closemenuentry,1);
  gtk_widget_hide(open_dialog);

#ifdef WIN32
  /* the socket routines have already converted the socket into
     a GIOChannel *. */
  server_callback_tag
    = g_io_add_watch((GIOChannel *)the_game->fd,
		     G_IO_IN|G_IO_ERR,gcallback,NULL);
#else  
  server_callback_tag
    = gdk_input_add(the_game->fd,GDK_INPUT_READ,server_input_callback,NULL);
#endif

  our_id = atoi(gtk_entry_get_text(GTK_ENTRY(openidtext)));
  client_connect(the_game,our_id,gtk_entry_get_text(GTK_ENTRY(opennametext)));

  if ( data ) {
    for ( i = 0 ; i < 3 ; i++ ) {
      if ( GTK_TOGGLE_BUTTON(openplayercheckboxes[i])->active ) {
	char cmd[1024];
	strcpy(cmd,"mj-player --server ");
	strcat(cmd,buf);
	strcat(cmd," ");
	strcat(cmd,gtk_entry_get_text(GTK_ENTRY(openplayeroptions[i])));
	if ( ! start_background_program(cmd) ) {
	  error_dialog_popup("Unexpected error starting player");
	  return;
	}
	sleep(2);
      }
    }
  }
}

/* shut down the connection */
void close_connection() {
  int i;
  gdk_input_remove(server_callback_tag);
  client_close(the_game);
  the_game = NULL;
  gtk_widget_set_sensitive(openmenuentry,1);
  gtk_widget_set_sensitive(newgamemenuentry,1);
  gtk_widget_set_sensitive(closemenuentry,0);
  /* clear display */
  for ( i=0; i < 4; i++ ) {
    pdisps[i].player = NULL;
    playerdisp_update(&pdisps[i],0);
    playerdisp_clear_discards(&pdisps[i]);
  }
  clear_wall();
  /* and pop down irrelevant dialogs that are up */
  gtk_widget_hide(chowdialog);
  gtk_widget_hide(ds_dialog);
  gtk_widget_hide(discard_dialog->widget);
  gtk_widget_hide(continue_dialog);
  gtk_widget_hide(turn_dialog);
  gtk_widget_hide(scoring_dialog);
}

/* convenience function: given a widget w and a descendant z, 
   give the position of z relative to w.
   To make sense of this function, one needs to know the insane
   conventions of gtk, which of course are entirely undocumented.
   To wit, the allocation of a widget is measured relative to its
   window, i.e. the window of the nearest ancestor widget that has one.
   The origin viewed *externally* of a widget is the top left of its
   border; the origin viewed *internally* is the top of the non-border.
   Returns true on success, false on failure */
static int get_relative_posn_aux(GtkWidget *w, GtkWidget *z, int *x, int *y) {
  GtkWidget *target = z;

  /* base case */
  if ( z == w ) {
    /* I don't think this should ever happen, unless we actually
       call the main function with z == w. Let's check */
    assert(0);
    return 1;
  }
  /* find the nearest ancestor of the target widget with a window */
  z = z->parent;
  while ( z && z != w && GTK_WIDGET_NO_WINDOW(z) ) z = z->parent;
  /* if z has a window, then the target position relative to
     z is its allocation */
  if ( ! z ) {
    /* something wrong */
    /* whoops. Original z wasn't a child of w */
    warn("get_relative_posn: z is not a descendant of w");
    return 0;
  } else if ( z == w ) {
    /* the position of the target is its allocation minus the
       allocation of w, if w is not a window; */
    *x += target->allocation.x;
    *y += target->allocation.y;
    if ( GTK_WIDGET_NO_WINDOW(w) ) {
      *x -= w->allocation.x;
      *y -= w->allocation.y;
      /* and minus the borderwidth of w */
      if ( GTK_IS_CONTAINER(w) ) {
	*x -= GTK_CONTAINER(w)->border_width;
	*y -= GTK_CONTAINER(w)->border_width;
      }
    }
    /* and that's it */
    return 1;
  } else if ( !GTK_WIDGET_NO_WINDOW(z) ) {
    *x += target->allocation.x;
    *y += target->allocation.y;
  } 
  /* if we get here, z is an ancestor window, but a descendant of
     w. So we have to pass through it, adding the borderwidth 
     if there is one. */
  if ( GTK_IS_CONTAINER(z) ) {
    /* add in the border */
    *x += GTK_CONTAINER(z)->border_width;
    *y += GTK_CONTAINER(z)->border_width;
  }
  /* now recurse */
  return get_relative_posn_aux(w,z,x,y);
}
  
int get_relative_posn(GtkWidget *w, GtkWidget *z, int *x, int *y) {
  *x = 0; *y = 0; return ((w == z) || get_relative_posn_aux(w,z,x,y));
}
  
/* utility function for wall display: given the tile in 
   wall  ori, stack j, and top layer (k = 0), bottom layer (k = 1)
   or loose tile (k = -1), return its x and y coordinates in the
   discard area.
   If the isdead argument is one, the tile is to be displaced as
   in the dead wall. This assumes that the board widget has a border
   into which we can shift the wall a little bit.
*/

static void wall_posn(int ori, int j, int k, int isdead, int *xp, int *yp) {
  int x=0, y=0;
  /* calculate basic position with respect to discard area */
  /* because there are possibly three stacks (the two layers
     and the loose tiles), which we want to display in a
     formalized perspective, by displacing layers by 
     tile_spacing from each other, each wall occupies 
     tile_height + 2*tile_spacing in the forward direction */
  switch (ori) {
  case 0:
    x = ((144/4)/2 - (j+1))*tile_width +tile_spacing;
    y = (144/4)/2 * tile_width  + (k+1)*tile_spacing ;
    break;
  case 1:
    x = (144/4)/2 * tile_width + (-k)*tile_spacing + 2*tile_spacing;
    y = tile_height + 2*tile_spacing + j*tile_width;
    break;
  case 2:
    x = tile_height + 2*tile_spacing + j*tile_width;
    y = (k+1)*tile_spacing;
    break;
  case 3:
    x = (k+1)*tile_spacing;
    y = ((144/4)/2 - (j+1))*tile_width + tile_spacing;
    break;
  }
  if ( isdead ) {
    switch ( wall_ori(wall_game_to_board(the_game->wall.live_end)) ) {
    case 0: x -= tile_spacing; break;
    case 1: y += tile_spacing; break;
    case 2: x += tile_spacing; break;
    case 3: y -= tile_spacing; break;
    }
  }
  *xp = x; *yp = y;
}

void clear_wall(void) {
  int i;
  /* destroy all existing widgets */
  for ( i=0; i<144; i++ ) {
    if ( wall[i] ) gtk_widget_destroy(wall[i]);
    wall[i] = NULL;
  }
}

/* create a new wall */
void create_wall(void) {
  int i,j,k,n,ori,x,y;
  GtkWidget *w;
  GtkWidget *pm;

  clear_wall();
  /* create buttons and place in discard area */
  for ( i = 0; i < 4; i++ ) {
    for ( j = 0; j < (144/4)/2 ; j++ ) {
      /* j is stack within wall */
      /* 0 is top, 1 is bottom */
      for (k=1; k >= 0 ; k-- ) {
	n = i*(144/4)+(2*j)+k;
	ori = wall_ori(n);
	wall_posn(ori,j,k,the_game 
		  && wall_board_to_game(n) >= the_game->wall.live_end,&x,&y);
	w = gtk_button_new();
	pm = gtk_pixmap_new(tilepixmaps[ori][HiddenTile],NULL);
	gtk_widget_show(pm);
	gtk_container_add(GTK_CONTAINER(w),pm);
	button_set_tile(w,HiddenTile,ori);
	gtk_widget_show(w);
	gtk_fixed_put(GTK_FIXED(discard_area),w,x,y);
	wall[n] = w;
      }
    }
  }
}

/* adjust the loose tiles */
AnimInfo *adjust_wall_loose () {
  int i,n,m,ori,j,x,y;
  AnimInfo *retval = NULL;
  static AnimInfo anim;
  /* destroy the tile just drawn (if any) */
  i = wall_game_to_board(the_game->wall.dead_end);
  if ( the_game->wall.dead_end < 144 ) {
    /* which tile gets drawn? If dead_end was even before
       (i.e. it's now odd), the tile drawn was the 2nd to last;
       otherwise it's the last */
    if ( the_game->wall.dead_end%2 == 1 ) {
      /* tile drawn was 2nd to last */
      i = wall_game_to_board(the_game->wall.dead_end-1);
    } else {
      /* tile drawn was actually beyond the then dead end */
      i = wall_game_to_board(the_game->wall.dead_end+1);
    }
    assert(i < 144 && wall[i]);
    anim.target = NULL;
    anim.t = HiddenTile;
    anim.ori = wall_ori(i);
    get_relative_posn(boardframe,wall[i],
		      &anim.x,&anim.y);
    retval = &anim;
    gtk_widget_destroy(wall[i]); 
    wall[i] = NULL;
  }
  /* if the number of loose tiles remaining is even, move the
     last two into position */
  if ( the_game->wall.dead_end%2 == 0 ) {
    /* number of the top loose tile */
    n = wall_game_to_board(the_game->wall.dead_end-2);
    /* it gets moved back three stacks */
    m = ((n-6)+144)%144;
    ori = wall_ori(m);
    j = (m - (144/4)*(m/(144/4)))/2;
    wall_posn(ori,j,-1,1,&x,&y);
    gtk_fixed_move(GTK_FIXED(discard_area),wall[n],x,y);
    gdk_window_raise(wall[n]->window);
    button_set_tile(wall[n],HiddenTile,ori);
    /* and now the bottom tile, gets moved back one stack */
    n = wall_game_to_board(the_game->wall.dead_end-1);
    m = ((n-2)+144)%144;
    ori = wall_ori(m);
    j = (m - (144/4)*(m/(144/4)))/2;
    wall_posn(ori,j,-1,1,&x,&y);
    gtk_fixed_move(GTK_FIXED(discard_area),wall[n],x,y);
    gdk_window_raise(wall[n]->window);
    button_set_tile(wall[n],HiddenTile,ori);
  }
  return retval;
}


/* check_claim_window: see whether a claim window should be
   popped down. This calls itself as a timeout, hence the prototype.
   If a claim window refers to an old discard, or we are in state
   handcomplete, pop it down unconditionally.
   Otherwise, if it's more than two seconds old, and we're not
   in the state Discarded, pop it down; if we're not in the discarded
   state, but it's less than two seconds old, check again later.
*/
gint check_claim_window(gpointer data) {
  PlayerDisp *pd = data;

  /* Unconditional popdown */
  if ( pd->claim_serial < the_game->info.serial
       || the_game->state == HandComplete ) {
    gtk_widget_hide(pd->claimw);
    pd->claim_serial = -1;
    pd->claim_time = -10000;
  } else if ( the_game->state != Discarded ) {
    int interval = now_time() - pd->claim_time;
    if ( interval > 2000 ) {
      /* pop down now */
      gtk_widget_hide(pd->claimw);
      pd->claim_serial = -1;
      pd->claim_time = -10000;
    } else {
      /* schedule a time out to check this window again when
	 the two seconds have expired */
      gtk_timeout_add(interval+100,check_claim_window,(gpointer)pd);
    }
  }
  return FALSE; /* don't run *this* timeout again! */
}
