#include "MediaStreamer.h"

void afterPlaying(void *cd) {

	MediaStreamer *ms = (MediaStreamer *)cd;
	ms->afterPlay();

}

/*
 * MediaStreamer():
 * Initalizes instance variables
 */

MediaStreamer::MediaStreamer() {

	TaskScheduler *scheduler = BasicTaskScheduler::createNew();
	env = BasicUsageEnvironment::createNew(*scheduler);

}

/*
 * next()
 * Moves the media list iterator to the next file in the list
 */

Media *MediaStreamer::next() {

	piter++;
	if (piter == playlist.end())
		piter = playlist.begin();

	return *piter;

}

/*
 * afterPlay()
 * Cleans up after a file is finished playing.
 * Starts playing the next file in the list.
 */

void MediaStreamer::afterPlay() {

	audioSink->stopPlaying();
	videoSink->stopPlaying();

	Medium::close(audioSource);
	Medium::close(videoSource);
	Medium::close(mpegDemux);

	PlayList::iterator iter = piter;
	if (++iter == playlist.end() && strcmp(parser.getLoop(), "yes"))
		;
	else
		play(next());

}

/*
 * start()
 * Call the playlist parser.
 * Initializes the socket structures with the correct audio/video group/port.
 * Begins playing the first file in the playlist.
 */

void MediaStreamer::start(char *config) {

	playlist = parser.parse(config);
	char *audioGroup = parser.getAudioGroup();
	int audioPort = parser.getAudioPort();
	char *videoGroup = parser.getVideoGroup();
	int videoPort = parser.getVideoPort();
	int ttl = parser.getTTL();

	struct in_addr rtpGroupAudio;
	struct in_addr rtpGroupVideo;
	
	rtpGroupAudio.s_addr = our_inet_addr(audioGroup);
	rtpGroupVideo.s_addr = our_inet_addr(videoGroup);
	Port rtpPortAudio(audioPort);
	Port rtcpPortAudio(audioPort+1);
	Port rtpPortVideo(videoPort);
	Port rtcpPortVideo(videoPort+1);

	Groupsock rtpGroupsockAudio(*env, rtpGroupAudio, rtpPortAudio, ttl);
	Groupsock rtcpGroupsockAudio(*env, rtpGroupAudio, rtcpPortAudio, ttl);
	Groupsock rtpGroupsockVideo(*env, rtpGroupVideo, rtpPortVideo, ttl);
	Groupsock rtcpGroupsockVideo(*env, rtpGroupVideo, rtcpPortVideo, ttl);

	audioSink = MPEGAudioRTPSink::createNew(*env, &rtpGroupsockAudio);
	videoSink = MPEGVideoRTPSink::createNew(*env, &rtpGroupsockVideo);

	int sessionBandAudio = 160;
	int sessionBandVideo = 4500;
	int maxCNAMElen = 100;
	unsigned char CNAME[maxCNAMElen+1];
	gethostname((char *)CNAME, maxCNAMElen);
	CNAME[maxCNAMElen] = '\0';
	RTCPInstance::createNew(*env, &rtcpGroupsockAudio, sessionBandAudio,
			CNAME, audioSink, NULL);
	RTCPInstance::createNew(*env, &rtcpGroupsockVideo, sessionBandVideo,
			CNAME, videoSink, NULL);

	piter = playlist.begin();
	play(*piter);

}	

/*
 * play()
 * Gets the type of media we are to play, and calls the correct handler.
 */

void MediaStreamer::play(Media *media) {

	int mediaType = mediaInfo.getType(media);

	printf("Playing %s\n", media->filename);

	if (mediaType == M_MP3) {
		playMP3(media);
	} else if (mediaType == M_UNKNOWN) {
		printf("%s: unknown file type.\n", media->filename);
		exit(0);
		//play(next());
	} else {
		playMPEG(media);
	}

	env->taskScheduler().blockMyself();

}

/*
 * playMP3()
 * Plays an mp3 file.
 */

void MediaStreamer::playMP3(Media *media) {

	audioSource = MyMP3FileSource::createNew(*env, media);

	if (audioSource == NULL) {

		printf("unable to open %s, skipping.\n", media->filename);
		play(next());

	} else {

		printf("Playing MP3 audio\n");
		audioSink->startPlaying(*audioSource, afterPlaying, this);

	}

}

/*
 * playMPEG()
 * play an MPEG movie.
 */

void MediaStreamer::playMPEG(Media *media) {

	ByteStreamFileSource *fs = MyByteStreamFileSource::createNew(*env, media);

	if (fs == NULL) {

		printf("unable to open %s, skipping.\n", media->filename);
		exit(0);
		play(next());

	} else {

		mpegDemux = MPEGDemux::createNew(*env, fs);
		if (mediaInfo.hasAudio && parser.getAudioPort()) {
			printf("Playing MPEG audio\n");
			FramedSource *audioES = mpegDemux->newAudioStream();
			audioSource = MPEGAudioStreamFramer::createNew(*env, audioES);
			audioSink->startPlaying(*audioSource, afterPlaying, this);
		}
		if (mediaInfo.hasVideo && parser.getVideoPort()) {
			printf("Playing MPEG video\n");
			FramedSource *videoES;
			if (mediaInfo.hasAudio)
				videoES = mpegDemux->newVideoStream();
			else
				videoES = fs;
			videoSource = MPEGVideoStreamFramer::createNew(*env, videoES);
			videoSink->startPlaying(*videoSource, afterPlaying, this);
		}

	}

}
