#!/usr/bin/perl -w
# $Id: graphdefanglib.pl,v 1.1 2003/01/22 14:36:23 dfs Exp $
#
# GraphDefang -- a set of tools to create graphs of your mimedefang
#                spam and virus logs.
#
# Written by:    John Kirkland
#                jpk@bl.org
#
# Copyright (c) 2002, John Kirkland
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or (at
# your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#=============================================================================

use strict;
use Time::Local;
use Date::Parse;
use Date::Format;
use Data::Dumper;
use File::ReadBackwards;

use MLDBM qw(DB_File Storable);
use Fcntl;

use File::Copy;	# for move() function

use GD::Graph::linespoints;
use GD::Graph::bars;

# X and Y Graph Sizes in pixels

my $X_GRAPH_SIZE = 700;
my $Y_GRAPH_SIZE = 300;

# Number of hours, days, and months in the hourly, daily, and monthly charts, respectively.

my $NUM_HOURS_SUMMARY = 48;
my $NUM_DAYS_SUMMARY = 60;
my $NUM_MONTH_SUMMARY = 24;

sub get_unixtime_by_timesummary($$) {
	my $timesummary = shift;
	my $unixtime = shift;
	
	# Get the number of seconds past the day for a given unixtime
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($unixtime);

	# zero out appropriate seconds,minutes,hours,days,etc...
	$sec = 0; #if ($timesummary =~ m/hourly|daily|monthly/);
	$min = 0; #if ($timesummary =~ m/hourly|daily|monthly/);
	$hour = 0 if ($timesummary ne 'hourly');
	$mday = 1 if ($timesummary eq 'monthly');
	
	# get unixtime for our new values
	$unixtime = timelocal($sec, $min, $hour, $mday, $mon, $year);

	return $unixtime;
}

sub trim_database() {

	my %data_db = ();
	my %data = ();

	tie (%data_db, 'MLDBM', $SUMMARYDB, O_RDWR|O_CREAT, 0644)
		or die "Can't open $SUMMARYDB:$!\n";
	%data = %data_db;

	untie %data_db;

	# Start the DB Trim

	my $now = time();
	my $trimcounter = 0;

	# Delete hourly data older than 2*$NUM_HOURS_SUMMARY hours

	my $deletetime = get_unixtime_by_timesummary('hourly',$now - 2*$NUM_HOURS_SUMMARY*60*60);

	foreach my $entrytime (keys %{$data{'hourly'}}) {
		if ($entrytime < $deletetime) {
			delete($data{'hourly'}{$entrytime});
			$trimcounter++;
		}
	}

	print STDERR "\tTrimmed $trimcounter 'hourly' entries from SummaryDB\n" if (!$QUIET);

	# Delete daily data older than 2*$NUM_DAYS_SUMMARY days

	$deletetime = get_unixtime_by_timesummary('daily',$now - 2*$NUM_DAYS_SUMMARY*60*60*24);
	$trimcounter=0;

	foreach my $entrytime (keys %{$data{'daily'}}) {
		if ($entrytime < $deletetime) {
			delete $data{'daily'}{$entrytime};
			$trimcounter++;
		}
	}

	print STDERR "\tTrimmed $trimcounter 'daily' entries from SummaryDB\n" if (!$QUIET);

	# Delete all but Top25 entries in hours, days, and months
	# other than the current one!
	
	my @DeleteTimes = ('hourly', 'daily'. 'monthly');
	$trimcounter = 0;

	foreach my $deletetime (@DeleteTimes) {
		my $nowdeletetime = get_unixtime_by_timesummary($deletetime,$now);
		foreach my $entrytime (keys %{$data{$deletetime}}) {
			if ($entrytime < $nowdeletetime ) {
			foreach my $event (keys %{$data{$deletetime}{$entrytime}}) {
				foreach my $type (keys %{$data{$deletetime}{$entrytime}{$event}}) {
					if ($type ne 'summary') {
						my %total = ();
						foreach my $value (keys %{$data{$deletetime}{$entrytime}{$event}{$type}}) {
							$total{$value} = $data{$deletetime}{$entrytime}{$event}{$type}{$value};
						}
						# Create list of top 25 items.
						my $i = 0;
						my %keep = ();
						foreach my $TopName (sort { $total{$b} <=> $total{$a} } keys %total) {
							$keep{$TopName} = 1;
							$i++;
							last if $i >= 25;
						}
						# delete the entries unless it is in the topList.
						foreach my $value (keys %{$data{$deletetime}{$entrytime}{$event}{$type}}) {
							if (!defined($keep{$value})) {
								delete $data{$deletetime}{$entrytime}{$event}{$type}{$value};
								$trimcounter++;
							}
						}
					}
				}
			}
			}
		}
	}

	print STDERR "\tTrimmed $trimcounter 'non top25' entries from SummaryDB\n" if (!$QUIET);
							

	# Backup the old summarydb file and recreate it using the new data

	move("$SUMMARYDB", "$SUMMARYDB.bak")
		or die "Can't move $SUMMARYDB to $SUMMARYDB.bak: $!";

	tie (%data_db, 'MLDBM', $SUMMARYDB, O_RDWR|O_CREAT, 0644)
		or die "Can't open $SUMMARYDB:$!\n";

	%data_db = %data;

	untie %data_db 
		or die "Database Save Failed... restore from $SUMMARYDB.bak:$!\n";

	unlink ("$SUMMARYDB.bak")
		or die "Can't unlink $SUMMARYDB.bak:$!\n";
}

sub read_and_summarize_data($) {
        my $fn = shift;
        my %data = ();
	my %data_db = ();

	# Temporary variables for lookup information	
	my %sendmail = ();
	my %spamd = ();

	my $NumNewLines = 0;;
	
	# Set graphtimes
	my @GraphTimes = ("hourly","daily","monthly");

	# Open SummaryDB
	if (!$NODB) {
		tie (%data_db, 'MLDBM', $SUMMARYDB, O_RDWR|O_CREAT, 0644)
			or die "Can't open $SUMMARYDB:$!\n";
		%data = %data_db;    
	}
       
	# Open log file 
	tie *ZZZ, 'File::ReadBackwards', $fn || die("can't open datafile: $!");

	# Get max unixtime value from DBM file 
	my $MaxDBUnixTime = 0;
	$MaxDBUnixTime = $data{'max'} if (defined($data{'max'}));

	print STDERR "\tMax Unixtime from SummaryDB:  $MaxDBUnixTime\n" if (!$QUIET and !$NODB);

# Sample Rows from mimedefang's md_log()

#Sep 28 21:55:50 westover mimedefang.pl[16803]: MDLOG,g8T2th86016917,mail_out,,,<mimedefang@westover.org>,<defang-admin@westover.org>,SPAM: 21.9, Make Money 100% RISK FREE!
#Sep 28 21:55:52 westover mimedefang.pl[16803]: MDLOG,g8T2th88016917,mail_out,,,<mimedefang@westover.org>,<defang-admin@westover.org>,SPAM: 20.3, Access Your PC from Anywhere - Download Now
#Sep 28 21:55:55 westover mimedefang.pl[16803]: MDLOG,g8T2th8A016917,mail_out,,,<mimedefang@westover.org>,<defang-admin@westover.org>,SPAM: 32.8, /ADV / The Best Business Opportunity on Net
#Sep 28 21:55:57 westover mimedefang.pl[16803]: MDLOG,g8T2th8C016917,mail_out,,,<mimedefang@westover.org>,<defang-admin@westover.org>,SPAM: 16.6, Get the lowest long distance rates available.
#Sep 28 21:56:02 westover mimedefang.pl[16803]: MDLOG,g8T2tt82016932,spam,38,203.167.97.19,<joisie69@hotmail.com>,<vernonm@westover.org>,I did not belive it....
#Sep 28 21:56:09 westover mimedefang.pl[16803]: MDLOG,g8T2u782016945,mail_out,,,<mimedefang@westover.org>,<defang-admin@westover.org>,SPAM: 38, I did not belive it....

# Sample Rows from spamd's syslog:

# SPAM:
#Oct  6 06:25:04 wbs sendmail[23124]: g96BP4C23124: from=<promo@spaceplanet.it>, size=1358, class=0, nrcpts=1, msgid=<200210051058.TZK5253@smtp.fastweb.it>, proto=ESMTP, daemon=MTA, relay=localhost.localdomain [127.0.0.1]
#Oct  6 06:25:04 wbs spamd[11318]: connection from localhost.localdomain [127.0.0.1] at port 38273
#Oct  6 06:25:04 wbs spamd[23129]: info: setuid to tex succeeded
#Oct  6 06:25:04 wbs spamd[23129]: processing message <200210051058.TZK5253@smtp.fastweb.it> for tex:502, expecting 1623 bytes.
#Oct  6 06:25:06 wbs spamd[23129]: identified spam (6.2/5.0) for tex:502 in 1.3 seconds, 1623 bytes.
#CLEAN MESSAGE
#Oct  6 06:40:28 wbs sendmail[23163]: g96BeSC23163: from=<root@moses.wbschool.net>, size=892, class=0, nrcpts=1, msgid=<200210061137.g96Bbv410245@moses.wbschool.net>, proto=ESMTP, daemon=MTA, relay=localhost.localdomain [127.0.0.1]
#Oct  6 06:40:28 wbs spamd[11318]: connection from localhost.localdomain [127.0.0.1] at port 38318
#Oct  6 06:40:28 wbs spamd[23168]: info: setuid to dottie succeeded
#Oct  6 06:40:28 wbs spamd[23168]: processing message <200210061137.g96Bbv410245@moses.wbschool.net> for dottie:509, expecting 1167 bytes.
#Oct  6 06:40:30 wbs spamd[23168]: clean message (-96.5/8.0) for dottie:509 in 1.5 seconds, 1167 bytes.

        while (<ZZZ>)
        {
                chomp;

		# Parse syslog line

		m/^(\S+\s+\d+\s+\d+:\d+:\d+)\s		# datestring -- 1
		(\S+)\s					# host -- 2
		(\S+?)					# program -- 3
		(?:\[(\d+)\])?:\s			# pid -- 4
		(?:\[ID\ \d+\ [a-z0-9]+\.[a-z]+\]\ )?	# Solaris stuff -- not used
		(.*)/x;					# text -- 5

		my $datestring = $1;
		my $host = $2;
		my $program = $3;
		my $pid = $4;
		my $text = $5;

		my $unixtime=str2time($datestring);

		# don't examine the line if it is greater than 5 minutes
		# older than the maximum time in our DB.  The 5 minutes
		# comes from the PID, From, and Relay caching with sendmail
		# and spamd that occurs below.
		last if ($unixtime < ($MaxDBUnixTime-60*5));

		my $event = '';
		my $value1 = '';
		my $value2 = '';
		my $sender = '';
		my $recipient = '';
		my $subject = '';

		my $FoundNewRow = 0;

		if ($program eq 'mimedefang.pl') {

			if ($text =~ m/^MDLOG,\S+?,(\S+?),(\S*?),(\S*?),(.*?),(.*?),(.*)$/ ) {

				# get values from regular expression

				# Only summarize data if it is newer than our current MaxDBUnixTime
				if ($unixtime > $MaxDBUnixTime) {

	              			$event = $1;
					$value1 = $2;
					$value2 = $3;
					$sender = $4;
					$recipient = $5;
					$subject = $6;

					$FoundNewRow = 1;
				}
			}

		} elsif ($program eq 'sendmail') {

			if ($text =~ m/^\S+: from=(.+), size=.+ msgid=(.+), proto=.+ relay=(.*)$/) {

                        	# our spamd times may be separated from our sendmail times by a processing
                        	# delay.  I've set the max potential processing delay to 5 minutes here
                        	if ($unixtime > ($MaxDBUnixTime-60*5)) {
                                	my $from = $1;
                                	my $msgid = $2;
                                	my $relay = $3;
					
					if (defined $spamd{$msgid}) {
						my $spamdpid = $spamd{$msgid};

						$event = $spamd{$spamdpid}{'event'};
						$sender = $from;
						$recipient = $spamd{$spamdpid}{'for'};
						$value1 = $spamd{$spamdpid}{'score'};
						$value2 = $relay;

						$FoundNewRow = 1;
                                        }

        			}
			} elsif ($text =~ m/^.* (\<\S+\>)\.\.\. User unknown$/) {
			# matching against: gAHC2iEA006157: <laches@moi.net>... User unknown

				if ($unixtime > $MaxDBUnixTime) {
					$event = 'user_unknown';
					$recipient = $1;
					
					# extract the domain from the unknown user's email address
					$recipient =~ m/<.*@(.*)>/;
					my $domain = $1;
					$value1 = "none";
					$value1 = $domain if defined($domain);

					$FoundNewRow = 1;
				}
			} elsif ($text =~ m/^.*ruleset=check_(?:rcpt|relay), .* relay=.*?(\d+\.\d+\.\d+\.\d+).* reject=(\d+) (\d\.\d.\d) .*$/) {
			#gAABdPd17840: ruleset=check_rcpt, arg1=<0206241317.54101.bj@bl.org>, relay=200-207-163-140.dsl.telesp.net.br [200.207.163.140], reject=550 5.0.0 <0206241317.54101.bj@bl.org>... We dont accept mail from br
			#gAF0bmd11725: ruleset=check_rcpt, arg1=<mparson@bl.org>, relay=[209.163.156.31], reject=451 4.1.8 Domain of sender address pqwa625@100pesos.com does not resolve
				
				if ($unixtime > $MaxDBUnixTime) {
					$event = "reject";
					$value1 = "$2 $3"; #reject code
					$value2 = $1; # relay addr

					$FoundNewRow = 1;
				}
				
			}
		} elsif ($program eq 'spamd') {
		
			if ($text =~ m/^processing message (.+) for (\S+):\d+, .+$/) {
				
				if ($unixtime > $MaxDBUnixTime) {
					my $msgid = $1;
					my $for = $2;

					$spamd{$pid}{'msgid'} = $msgid;
					$spamd{$pid}{'for'} = $for;	
					#provide a lookup for the sendmail parsing section
					$spamd{$msgid} = $pid;
				}
			} elsif ($text =~ m/^(\w+ \w+) \((.+)\/.+\) for .+$/) {

				if ($unixtime > $MaxDBUnixTime) {
					my $words = $1;
					my $score = $2;

					if ($words eq 'identified spam') { $event = 'spam' }
					elsif ($words eq 'clean message') { $event = 'non-spam' }

					$spamd{$pid}{'event'} = $event;
					$spamd{$pid}{'score'} = $score;
				}
			}
		} elsif ($program eq 'ipop3d') {
# Searching for Oct 17 21:58:33 westover ipop3d[21207]: Login user=ben host=c68.115.81.145.roc.mn.charter.com [68.115.81.145] nmsgs=0/0
			if ($text =~ m/^Login user=(\S+) host=(\S+)/) {
				
				if ($unixtime > $MaxDBUnixTime) {
					my $user = $1;
					my $host = $2;

					$event = 'ipop3d';
					$value1 = $user;
					$value2 = $host;

					$FoundNewRow = 1;
				}
			}
		} elsif ($program eq 'kernel') {
#Oct 27 04:02:30 hdnetwork kernel: gShield (default drop) IN=eth0 OUT= MAC=ff:ff:ff:ff:ff:ff:00:a0:c8:08:2b:2c:08:00 SRC=66.139.79.158 DST=216.201.156.32 LEN=60 TOS=0x00 PREC=0x00 TTL=52 ID=49735 DF PROTO=TCP SPT=51023 DPT=21 WINDOW=5840 RES=0x00 SYN URGP=0
                        if ($text =~ m/^gShield .* SRC=(\S+) DST=(\S+) .* DPT=(\d+)/) {
                                if ($unixtime > $MaxDBUnixTime) {
                                        my $src = $1;
                                        my $dst = $2;
                                        my $dpt = $3;

                                        $event =  'gShield';
                                        $sender = $src;
                                        $recipient = $dst;
                                        $value1 = $dpt;

                                        $FoundNewRow = 1;
                                }
                        }
                }

		if ($FoundNewRow) {
			# Increment Number of New Lines Found
			$NumNewLines++;
			
			# rollup hourly, daily, and monthly summaries for every variable
			foreach my $timesummary (@GraphTimes) {

				my $summarytime = get_unixtime_by_timesummary($timesummary, $unixtime);
				$data{$timesummary}{$summarytime}{$event}{'summary'}++;
				$data{$timesummary}{$summarytime}{$event}{'value1'}{$value1}++ 		if ($value1 ne '');
				$data{$timesummary}{$summarytime}{$event}{'value2'}{$value2}++ 		if ($value2 ne '');
				$data{$timesummary}{$summarytime}{$event}{'sender'}{$sender}++ 		if ($sender ne '');;
				$data{$timesummary}{$summarytime}{$event}{'recipient'}{$recipient}++ 	if ($recipient ne '');
				$data{$timesummary}{$summarytime}{$event}{'subject'}{$subject}++     	if ($subject ne '');

				# Store the maximum unixtime per timesummary for later reference
				$data{'max'} = $unixtime 
					if (!defined($data{'max'}) 
						or $unixtime > $data{'max'});
			} 
		}
	}
        close (ZZZ);

	if (!$NODB) {
		%data_db = %data;
		untie %data_db;
		print STDERR "\t$NumNewLines new log lines loaded\n" if (!$QUIET);
	}
        return %data;
}

sub graph($$) {
	my $settings = shift;
	my $data = shift;

	foreach my $grouping_time (@{$settings->{grouping_times}}) {

		$settings->{grouping_time} = $grouping_time;


		# Set the settings for the graph we've been asked to draw
		set_graph_settings($settings);

		print STDERR "\t$settings->{filename}\n" if (!$QUIET);

		# Get the data for the graph we've been asked to draw
		my @GraphData = get_graph_data($settings,$data);

        	# Draw Graph
        	if ($settings->{graph_type} eq 'line') {
                	draw_line_graph($settings,\@GraphData);
        	} elsif ($settings->{graph_type} eq 'stacked_bar') {
                	draw_stacked_bar_graph($settings,\@GraphData);
        	} else {
                	die ("Invalid GraphSettings{graph_type} = $settings->{graph_type}");
        	}
	}
}

sub set_graph_settings($) {

	my $settings = shift;

        # Set the graph title and filename according to the options chosen

        # Initialize the title and filename
        $settings->{chart_title} = "";
        $settings->{filename} = "";

	my $autotitle = "";

        # Add "Top N" to the beginning of the Title if necessary
        if ($settings->{top_n}) {
                $autotitle = "Top $settings->{top_n} ";
        } 

        # Set Data Type Title
	my $i = 0;
	foreach my $data_type (@{$settings->{data_types}}) {
		# Uppercase the first letter of the data_type
		$autotitle .= "\u$data_type";
		$settings->{filename} .= $data_type;
		$i++;
		if ( $i == ($#{$settings->{data_types}}) ) {
			$autotitle .= " and "
		} elsif ( $i < ($#{$settings->{data_types}}) ) {
			$autotitle .= ", "
		}
	}

        # Set Grouping Title
        if ($settings->{grouping} eq 'summary') {
                $autotitle = $autotitle . " Total Counts ";
        } elsif ($settings->{grouping} eq 'value1') {
		if (defined($settings->{value1_title})) {
                	$autotitle = $autotitle . " Counts by $settings->{value1_title}";
		} else {
			$autotitle = $autotitle . " Counts by Value1";
		}
        } elsif ($settings->{grouping} eq 'value2') {
		if (defined($settings->{value2_title})) {
                	$autotitle = $autotitle . " Counts by $settings->{value2_title}";
		} else {
			$autotitle = $autotitle . " Counts by Value2";
		}
        } elsif ($settings->{grouping} eq 'sender') {
                $autotitle = $autotitle . " Counts by Sender";
        } elsif ($settings->{grouping} eq 'recipient') {
                $autotitle = $autotitle . " Counts by Recipient";
        } elsif ($settings->{grouping} eq 'subject') {
                $autotitle = $autotitle . " Counts by Subject";
        } else {
                die ("Invalid settings{grouping} value");
        }

	# Put top_n in the filename?

	if ($settings->{top_n}) {
        	$settings->{filename} .= "_$settings->{top_n}";
	} else {
		$settings->{filename} .= "_";
	}

	$settings->{filename} .= "$settings->{grouping}_$settings->{graph_type}";

        # The final portion of the title will be set in the section below

        if ($settings->{grouping_time} eq 'hourly') {

                $settings->{x_axis_num_values}  = $NUM_HOURS_SUMMARY;   # Number of x-axis values on graph
                $settings->{x_axis_num_sec_incr}= 60*60;                # Incremental number of seconds represented by each x-axis value
                $settings->{x_axis_date_format} = "%h %d, %I%p";        # Format of date string on x-axis
                $settings->{x_label}            = 'Hours';
                $settings->{y_label}            = 'Counts per Hour';
                $settings->{chart_title}        = $autotitle . " per Hour (last $settings->{x_axis_num_values} hours)"
                                                unless defined($settings->{title});
                $settings->{filename}           = "hourly_" . $settings->{filename};

        } elsif ($settings->{grouping_time} eq 'daily') {

                $settings->{x_axis_num_values}  = $NUM_DAYS_SUMMARY;
                $settings->{x_axis_num_sec_incr}= 60*60*24;
                $settings->{x_axis_date_format} = "%h %d";
                $settings->{x_label}            = 'Days';
                $settings->{y_label}            = 'Counts per Day';
                $settings->{chart_title}              = $autotitle . " per Day (Last $settings->{x_axis_num_values} days)"
                                                unless defined($settings->{title});
                $settings->{filename}      = "daily_" . $settings->{filename};

        } elsif ($settings->{grouping_time} eq 'monthly') {

                $settings->{x_axis_num_values}  = $NUM_MONTH_SUMMARY;
                $settings->{x_axis_num_sec_incr}= 60*60*24*31;
                $settings->{x_axis_date_format} = "%h";
                $settings->{x_label}            = 'Months';
                $settings->{y_label}            = 'Counts per Month';
                $settings->{chart_title}              = $autotitle . " per Month (Last $settings->{x_axis_num_values} months)"
                                                unless defined($settings->{title});
                $settings->{filename}           = "monthly_" . $settings->{filename};
        }

	if (defined $settings->{filter_name}) {
		my $filter;
		($filter = $settings->{filter_name}) =~ s/\W/_/g;
		$settings->{chart_title} .= " filtered by $settings->{filter_name}";
		$settings->{filename} .= "_$filter";
	}

	$settings->{chart_title} = $settings->{title} if (defined($settings->{title}));
}


sub get_graph_data($$) {

	my $settings = shift;
	my $data = shift;

        # Calculate the date cutoff for our graph

        my $currenttime = time();
	my $cutofftime;

	if ($settings->{grouping_time} eq 'monthly') {
		my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($currenttime);
                # Decrement the month/year value n times
		for (my $i = 0; $i < $settings->{x_axis_num_values}-1; $i++) {
                       	if ($mon == 0) {
                       		$year--;
                               	$mon = 11;
                       	} else {
                               	$mon--;
			}
                }

                # get unixtime for our new values
                $cutofftime = timelocal($sec, $min, $hour, $mday, $mon, $year);

	} else {
        	$cutofftime = $currenttime - ($settings->{x_axis_num_sec_incr}*($settings->{x_axis_num_values}-1));
	}

	# Create Data Array for Graph

	my @GraphData = ();
	my @TopNNames = ();
	my %Total = ();
	my @Legend = ();

	# Handle data_types = 'all'
	my $allset;
	if ($settings->{'data_types'}[0] eq 'all') {
		$allset = 1;
		my %all;
		foreach my $date (keys %{$data->{$settings->{grouping_time}}}) {
			foreach my $data_type (keys %{$data->{$settings->{grouping_time}}{$date}}) {
				$all{$data_type} = 1;
			}
		}
		$settings->{'data_types'} = ();
		foreach my $key (sort keys %all) {
			push @{$settings->{'data_types'}}, $key;
		}
	}

	# Summarize totals across time interval
	for (my $time=$cutofftime; $time<=$currenttime; $time += $settings->{x_axis_num_sec_incr}) {

		my $date = get_unixtime_by_timesummary($settings->{grouping_time},$time);

		# Get total for summary grouping
		if ($settings->{'grouping'} eq 'summary') {

			foreach my $datatype (@{$settings->{'data_types'}}) {
				if (defined($data->{$settings->{grouping_time}}{$date}{$datatype}{'summary'})) {
					$Total{$datatype} += $data->
							{$settings->{grouping_time}}
							{$date}
							{$datatype}
							{'summary'};
				} else {
					$Total{$datatype} += 0;
				}
			}

		} else {
			# Get total for other groupings

			foreach my $datatype (@{$settings->{'data_types'}}) {
				foreach my $value (keys %{$data->
								{$settings->{grouping_time}}
								{$date}
								{$datatype}
								{$settings->{'grouping'}}} ) {
					$Total{'value'}{$value} += $data->
									{$settings->{grouping_time}}
									{$date}
									{$datatype}
									{$settings->{'grouping'}}
									{$value};	
					$Total{$date}{$value} += $data->
									{$settings->{grouping_time}}
									{$date}
									{$datatype}
									{$settings->{'grouping'}}
									{$value};
				}
			}	
		}
		# Recalculate the x_axis_num_sec_incr value if we are graphing monthly.
		# Determine the current month, increment it by one, and then get a time delta..
		if ($settings->{grouping_time} eq 'monthly') {
			my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($date);
			# Increment the month/year value
			if ($mon == 11) {
				$year++;
				$mon = 0;
			} else {
				$mon++;
			}

			# get unixtime for our new values
			my $newmonthtime = timelocal($sec, $min, $hour, $mday, $mon, $year);

			$settings->{x_axis_num_sec_incr} = $newmonthtime - $date;
                }
	}

	# Sort the TopNNames list so we have it largest to smallest and keep only the top N.
	if ($settings->{'grouping'} eq 'summary') {

		foreach my $datatype (@{$settings->{'data_types'}}) {
			push @Legend, "\u$datatype, Total = $Total{$datatype}";
		}
	} else {
		my $i=0;
		foreach my $TopNName (sort { $Total{'value'}{$b} <=> $Total{'value'}{$a} } keys %{$Total{'value'}} ) {
			if (!defined($settings->{'filter'}) or $TopNName =~ m/$settings->{'filter'}/i) {
				push @TopNNames, $TopNName;
				push @Legend, "$TopNName, Total=$Total{'value'}{$TopNName}";
				$i++;
			}
			last if (defined($settings->{'top_n'}) and $settings->{'top_n'} > 0  and $i >= $settings->{'top_n'} );
		}
	}
	
	# If we have no legend, create one so graph doesn't error
	push @Legend,"No values of this type!" if (!@Legend);

	@{$settings->{legend}} = @Legend;

	for (my $time=$cutofftime; $time<=$currenttime; $time += $settings->{x_axis_num_sec_incr}) {

		my $date = get_unixtime_by_timesummary($settings->{grouping_time},$time);
		my $datestring = time2str($settings->{x_axis_date_format},$date);

		my $i=0;
		push @{$GraphData[$i]}, $datestring;
		
		if ( $settings->{'grouping'} eq 'summary' ) {
			foreach my $datatype (@{$settings->{'data_types'}}) {

			# Data format:
			#$data{$timesummary}{$summarytime}{$event}{'summary'}++;
			#$data{$timesummary}{$summarytime}{$event}{'value1'}{$value1}++

				$i++;
				# Set any undefined values to 0 so GD::Graph
				# has something to graph
				if ( defined($data->
						{$settings->{grouping_time}}
						{$date}
						{$datatype}
						{'summary'}) ) {
					push @{$GraphData[$i]}, $data->
								{$settings->{grouping_time}}
								{$date}
								{$datatype}
								{'summary'};
				} else {
					push @{$GraphData[$i]}, 0;
				}
			}
		} else {
			# iterate over top_n values if they exist, else push 0
			if ($#TopNNames > -1) {
				foreach my $TopNName (@TopNNames) {
					$i++;
					# Set any undefined values to 0 so GD::Graph
					# has something to graph
					if ( defined ($Total{$date}{$TopNName}) ) {
						push @{$GraphData[$i]}, $Total{$date}{$TopNName};
					} else {
						push @{$GraphData[$i]}, 0;
					}	
				}
			} else {
				$i++;
				push @{$GraphData[$i]}, 0;
			}
		}
		# Recalculate the x_axis_num_sec_incr value if we are graphing monthly.
		# Determine the current month, increment it by one, and then get a time delta..
		if ($settings->{grouping_time} eq 'monthly') {
			my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($date);
			# Increment the month/year value
			if ($mon == 11) {
				$year++;
				$mon = 0;
			} else {
				$mon++;
			}

			# get unixtime for our new values
			my $newmonthtime = timelocal($sec, $min, $hour, $mday, $mon, $year);

			$settings->{x_axis_num_sec_incr} = $newmonthtime - $date;
                }
	}	

	@{$settings->{'data_types'}} = ('all') if ($allset);

	return @GraphData;
}

sub draw_line_graph($$) {
        my $settings = shift;
        my $data = shift;

        my $my_graph = new GD::Graph::linespoints($X_GRAPH_SIZE, $Y_GRAPH_SIZE);

        $my_graph->set(
                x_label                 => $settings->{x_label},
                y_label                 => $settings->{y_label},
                title                   => $settings->{chart_title},
                x_labels_vertical       => 1,
                x_label_position        => 1/2,

                bgclr                   => 'white',
                fgclr                   => 'gray',
                boxclr                  => 'lgray',

                y_tick_number           => 10,
                y_label_skip            => 2,
                long_ticks              => 1,
                marker_size             => 1,
                skip_undef              => 1,
                line_width              => 1,
                transparent             => 0,
        );

	$my_graph->set( dclrs => [ qw(lred lgreen lblue lyellow lpurple cyan lorange dred dgreen dblue dyellow dpurple) ] );


        $my_graph->set_legend( @{$settings->{legend}} );
        $my_graph->plot($data);
        save_chart($my_graph, "$OUTPUT_DIR/$settings->{filename}");
}

sub draw_stacked_bar_graph($$) {
        my $settings = shift;
        my $data = shift;

        my $my_graph = new GD::Graph::bars($X_GRAPH_SIZE, $Y_GRAPH_SIZE);

        $my_graph->set(
                x_label                 => $settings->{x_label},
                y_label                 => $settings->{y_label},
                title                   => $settings->{chart_title},
                x_labels_vertical       => 1,
                x_label_position        => 1/2,

                bgclr                   => 'white',
                fgclr                   => 'gray',
                boxclr                  => 'lgray',

                y_tick_number           => 10,
                y_label_skip            => 2,
                long_ticks              => 1,
                cumulate                => 1,
                transparent             => 0,
                correct_width           => 0,
        );
	
	$my_graph->set( dclrs => [ qw(lred lgreen lblue lyellow lpurple cyan lorange dred dgreen dblue dyellow dpurple) ] );

        $my_graph->set_legend( @{$settings->{legend}} );
        $my_graph->plot($data);
        save_chart($my_graph, "$OUTPUT_DIR/$settings->{filename}");
}

sub save_chart($$) {
        my $chart = shift or die "Need a chart!";
        my $name = shift or die "Need a name!";
        local(*OUT);

        my $ext = $chart->export_format;

        open(OUT, ">$name.png") or
                die "Cannot open $name.$ext for write: $!";
        binmode OUT;
        print OUT $chart->gd->png;
        close OUT;
}
1;
