/***********************************************************************
*
* md-mx-ctrl.c
*
* Command-line utility for talking to mimedefang-multiplexor directly
*
* Copyright (C) 2002-2003 by Roaring Penguin Software Inc.
*
* http://www.roaringpenguin.com
*
* This program may be distributed under the terms of the GNU General
* Public License, Version 2, or (at your option) any later version.
*
***********************************************************************/

static char const RCSID[] =
"$Id: md-mx-ctrl.c,v 1.8 2003/01/20 14:15:07 dfs Exp $";

#include "config.h"

#include <ctype.h>
#include <syslog.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <errno.h>
#include <stdarg.h>
#include <stdio.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifndef AF_LOCAL
#define AF_LOCAL AF_UNIX
#endif
/**********************************************************************
* %FUNCTION: MXCommand
* %ARGUMENTS:
*  sockname -- multiplexor socket name
*  cmd -- command to send
*  buf -- buffer for reply
*  len -- length of buffer
* %RETURNS:
*  0 if all went well, -1 on error.
* %DESCRIPTION:
*  Sends a command to the multiplexor and reads the answer back.
***********************************************************************/
static int
MXCommand(char const *sockname,
	  char const *cmd,
	  char *buf,
	  int len)
{
    int fd;
    struct sockaddr_un addr;
    int nleft, nwritten, nread;
    int n;

    fd = socket(AF_LOCAL, SOCK_STREAM, 0);
    if (fd < 0) {
	fprintf(stderr, "MXCommand: socket: %s\n", strerror(errno));
	return -1;
    }

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_LOCAL;
    strncpy(addr.sun_path, sockname, sizeof(addr.sun_path) - 1);

    if (connect(fd, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
	if (errno == EPERM || errno == EACCES) {
	    fprintf(stderr, "MXCommand: connect: %s\n", strerror(errno));
	} else {
	    fprintf(stderr,
		    "MXCommand: connect: %s: Is multiplexor running?\n",
		strerror(errno));
	}
	close(fd);
	return -1;
    }

    nwritten = 0;
    nleft = strlen(cmd);
    n = 0;
    while(nleft) {
	n = write(fd, cmd+nwritten, nleft);
	if (n < 0) {
	    if (errno == EINTR) continue;
	    break;
	}
	nwritten += n;
	nleft -= n;
    }
    if (n < 0) {
	fprintf(stderr, "MXCommand: write: %s: Is multiplexor running?\n",
		strerror(errno));
	close(fd);
	return -1;
    }

    /* Now read the answer */
    nread = 0;
    nleft = len;
    while(nleft) {
	n = read(fd, buf+nread, nleft);
	if (n <= 0) {
	    if (n < 0 && errno == EINTR) continue;
	    break;
	}
	nread += n;
	nleft -= n;
    }
    if (n < 0) {
	fprintf(stderr, "MXCommand: read: %s: Is multiplexor running?\n",
		strerror(errno));
	close(fd);
	return -1;
    }
    buf[nread] = 0;
    close(fd);
    return 0;
}

static int
doCmd(char const *sock, char const *cmd)
{
    char ans[4096];

    if (MXCommand(sock, cmd, ans, sizeof(ans)) < 0) {
	return EXIT_FAILURE;
    }
    printf("%s", ans);
    return EXIT_SUCCESS;
}

static int
doStatus(char const *sock)
{
    char ans[4096];
    char *s;
    int i, l;

    if (MXCommand(sock, "status\n", ans, sizeof(ans)) < 0) {
	return EXIT_FAILURE;
    }
    if (*ans != 'I' && *ans != 'K' && *ans != 'S' && *ans != 'B') {
	fprintf(stderr, "%s", ans);
	return EXIT_FAILURE;
    }

    /* Chop off message and activation count */
    s = ans;
    while (*s && *s != ' ') s++;
    *s = 0;

    l = strlen(ans);
    printf("Max slaves: %d\n", l);
    for (i=0; i<l; i++) {
	printf("Slave %d: ", i);
	switch(ans[i]) {
	case 'I':
	    printf("idle\n");
	    break;
	case 'K':
	    printf("killed\n");
	    break;
	case 'S':
	    printf("stopped\n");
	    break;
	case 'B':
	    printf("busy\n");
	    break;
	default:
	    printf("unknown state '%c'\n", ans[i]);
	}
    }
    return EXIT_SUCCESS;
}

static void
usage(void)
{
    fprintf(stderr, "Usage: md-mx-ctrl [options] command\n\n");
    fprintf(stderr, "Options:\n");
    fprintf(stderr, "-h              -- Display help\n");
    fprintf(stderr, "-s path         -- Specify path to multiplexor socket\n");
    fprintf(stderr, "\nPossible commands:\n");
    fprintf(stderr, "status          -- Display slave status\n");
    fprintf(stderr, "rawstatus       -- Display slave status in computer-readable format\n");
    fprintf(stderr, "reread          -- Re-read filter rules\n");
    fprintf(stderr, "msgs            -- Number of message processed since startup\n");
}

int
main(int argc, char *argv[])
{
    int c;

    char *sock = NULL;

    while ((c = getopt(argc, argv, "hs:")) != -1) {
	switch(c) {
	case 'h':
	    usage();
	    exit(EXIT_SUCCESS);
	case 's':
	    if (sock) free(sock);
	    sock = strdup(optarg);
	    if (!sock) {
		fprintf(stderr, "Out of memory\n");
		exit(EXIT_FAILURE);
	    }
	    break;
	default:
	    usage();
	    exit(EXIT_FAILURE);
	}
    }

    if (!argv[optind]) {
	usage();
	exit(EXIT_FAILURE);
    }
    if (!sock) {
	sock = SPOOLDIR "/mimedefang-multiplexor.sock";
    }

    if (!strcmp(argv[optind], "status")) {
	exit(doStatus(sock));
    } else if (!strcmp(argv[optind], "reread")) {
	exit(doCmd(sock, "reread\n"));
    } else if (!strcmp(argv[optind], "rawstatus")) {
	exit(doCmd(sock, "status\n"));
    } else if (!strcmp(argv[optind], "msgs")) {
	exit(doCmd(sock, "msgs\n"));
    }

    fprintf(stderr, "Unknown command '%s'\n", argv[optind]);
    exit(EXIT_FAILURE);
}
