/***********************************************************************
*
* mimedefang.c
*
* C interface to the attachment-filter program for stripping or altering
* MIME attachments in incoming Sendmail connections.
*
* Copyright (C) 2000 Roaring Penguin Software Inc.
* http://www.roaringpenguin.com
*
* This program may be distributed under the terms of the GNU General
* Public License, Version 2, or (at your option) any later version.
*
* This program was derived from the sample mail filter included in
* libmilter/README in the Sendmail 8.11 distribution.
***********************************************************************/

static char const RCSID[] =
"$Id: mimedefang.c,v 1.150 2003/02/14 18:28:29 dfs Exp $";

/* Define this to work around an M$ Outlook bug! */
/* #define CONVERT_EMBEDDED_CRS_IN_HEADERS 1 */

#define _BSD_SOURCE 1

#ifdef __linux__
/* On Linux, we need this defined to get fdopen.  On BSD, if we define
 * it, we don't get all the u_char, u_long, etc definitions.  GRR! */
#define _POSIX_SOURCE 1
#endif

#include "config.h"
#include "mimedefang.h"

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <pthread.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <time.h>
#include <sys/time.h>
#include <pwd.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <netinet/in.h>
#include <arpa/inet.h>

#include "libmilter/mfapi.h"

#include <sys/socket.h>
#include <sys/un.h>

#ifdef ENABLE_DEBUGGING
#include <signal.h>
#define DEBUG(x) x
#else
#define DEBUG(x) (void) 0
#endif

#define DEBUG_ENTER(func, line) DEBUG(syslog(LOG_DEBUG, "Entering %s (line %d)", func, line))
#define DEBUG_EXIT(func, line, ret) DEBUG(syslog(LOG_DEBUG, "Exiting %s (line %d) ret=%s", func, line, ret))

#define SCAN_BODY \
"MIMEDefang " VERSION " (www . roaringpenguin . com / mimedefang)"

char *scan_body = NULL;

#define KEY_FILE CONFDIR "/mimedefang-ip-key"

/* In debug mode, we do not delete working directories. */
int DebugMode = 0;

/* Log "eom" run-times */
int LogTimes = 0;

/* Run as this user */
static char *user = NULL;
extern int drop_privs(char const *user, uid_t uid, gid_t gid);

/* NOQUEUE */
static char *NOQUEUE = "NOQUEUE";

/* Header name for validating IP addresses */
static char ValidateHeader[256];

/* Keep track of private data -- file name and fp for writing e-mail body */
struct privdata {
    char *hostname;		/* Name of connecting host */
    char *hostip;		/* IP address of connecting host */
    char *sender;		/* Envelope sender */
    char *firstRecip;		/* Address of first recipient */
    char *dir;			/* Work directory */
    char *heloArg;		/* HELO argument */
    char *qid;                  /* Queue ID */
    FILE *fp;			/* File for message body */
    FILE *headerFP;		/* File for message headers */
    FILE *cmdFP;		/* File for commands */
    unsigned char headerPresent; /* Already an X-Scanned-By: header found */
    unsigned char validatePresent; /* Saw a relay-address validation header */
    unsigned char suspiciousBody; /* Suspicious characters in message body? */
    unsigned char lastWasCR;	/* Last char of body chunk was CR? */
    unsigned char filterFailed; /* Filter failed */
};

static void write_macro_value(SMFICTX *ctx,
			      char *macro);

static void log_perl_stderr(SMFICTX *ctx);
static sfsistat cleanup(SMFICTX *ctx);
static int IncrementRunningFilters(void);
static int DecrementRunningFilters(void);
static sfsistat mfclose(SMFICTX *ctx);

static char const *MultiplexorSocketName = NULL;
static char const *SubFilter = NULL;

static char const *FilterProgram = MIMEDEFANG_PL;

#define DATA ((struct privdata *) smfi_getpriv(ctx))

/* How many tries to get a unique directory name before giving up? */
#define MAXTRIES 8192

/* Size of chunk when replacing body */
#define CHUNK 4096

/* Number of file descriptors to close when forking */
#define CLOSEFDS 256

/* Number of running Perl filters */
static volatile int NumRunningFilters = 0;

/* Maximum allowable number of filters */
static int MaxRunningFilters = 10;

/* Mutex to protect access to NumRunningFilters */
static pthread_mutex_t FilterMutex = PTHREAD_MUTEX_INITIALIZER;

/* Mutex to protect mkdir() calls */
static pthread_mutex_t MkdirMutex = PTHREAD_MUTEX_INITIALIZER;

/* Do relay check? */
static int doRelayCheck = 0;

/* Do sender check? */
static int doSenderCheck = 0;

/* Do recipient check? */
static int doRecipientCheck = 0;

/* Keep directories around if multiplexor fails? */
static int keepFailedDirectories = 0;

/* Protect mkdir() with mutex? */
static int protectMkdirWithMutex = 0;

#ifdef ENABLE_DEBUGGING
/**********************************************************************
*%FUNCTION: handle_sig
*%ARGUMENTS:
* s -- signal number
*%RETURNS:
* Nothing
*%DESCRIPTION:
* Handler for SIGSEGV and SIGBUS -- logs a message and returns -- hopefully,
* we'll get a nice core dump the second time around
***********************************************************************/
static void
handle_sig(int s)
{
    syslog(LOG_ERR, "WHOA, NELLY!  Caught signal %d -- this is bad news.  Core dump at 11.", s);

    /* Default is terminate and core. */
    signal(s, SIG_DFL);

    /* Return and probably cause core dump */
}
#endif

/**********************************************************************
* %FUNCTION: log_perl_stderr
* %ARGUMENTS:
*  ctx -- filter context
* %RETURNS:
*  Nothing
* %DESCRIPTION:
*  If the STDERR file exists, log its contents.
***********************************************************************/
static void
log_perl_stderr(SMFICTX *ctx)
{
    char buffer[SMALLBUF];
    struct privdata *data = DATA;
    FILE *fp;

    if (!data) return;

    /* If STDERR file is there, log contents */
    snprintf(buffer, SMALLBUF, "%s/STDERR", data->dir);
    fp = fopen(buffer, "r");
    if (fp) {
	*buffer = 0;
	while(fgets(buffer, SMALLBUF, fp)) {
	    chomp(buffer);
	    if (*buffer) {
		syslog(LOG_INFO, "Perl stderr: %s", buffer);
	    }
	    *buffer = 0;
	}
	fclose(fp);
    }
}

/**********************************************************************
*%FUNCTION: closefiles
*%ARGUMENTS:
* None
*%RETURNS:
* Nothing
*%DESCRIPTION:
* Hack -- closes a whole bunch of file descriptors.  Use after a fork()
*         to conserve descriptors.
***********************************************************************/
static void
closefiles(void)
{
    int i;
    for (i=0; i<CLOSEFDS; i++) {
	(void) close(i);
    }
}

/**********************************************************************
*%FUNCTION: IncrementRunningFilters
*%ARGUMENTS:
* None
*%RETURNS:
* -1 if we've hit maximum number of filters, 0 otherwise.
*%DESCRIPTION:
* Increments number of running filters using a mutex to lock access.
***********************************************************************/
static int
IncrementRunningFilters(void)
{
    if (pthread_mutex_lock(&FilterMutex)) {
	DEBUG(syslog(LOG_INFO, "IncrementRunningFilters: error locking FilterMutex"));
	return -1;
    }
    if (NumRunningFilters >= MaxRunningFilters) {
	syslog(LOG_INFO,
	       "Maximum number of filters (%d) running", MaxRunningFilters);
	(void) pthread_mutex_unlock(&FilterMutex);
	return -1;
    }
    NumRunningFilters++;
    DEBUG(syslog(LOG_INFO, "New filter; now running %d filters", NumRunningFilters));
    (void) pthread_mutex_unlock(&FilterMutex);
    return 0;
}

/**********************************************************************
*%FUNCTION: DecrementRunningFilters
*%ARGUMENTS:
* None
*%RETURNS:
* -1 if there was an error; 0 otherwise.
*%DESCRIPTION:
* Decrements number of running filters using a mutex to lock access.
***********************************************************************/
static int
DecrementRunningFilters(void)
{
    if (pthread_mutex_lock(&FilterMutex)) {
	DEBUG(syslog(LOG_INFO, "DecrementRunningFilters: error locking FilterMutex"));
	return -1;
    }
    if (NumRunningFilters == 0) {
	/* Shouldn't happen! */
	DEBUG(syslog(LOG_ERR, "DecrementRunningFilters: NumRunningFilters is zero!"));
	(void) pthread_mutex_unlock(&FilterMutex);
	return -1;
    }
    NumRunningFilters--;
    DEBUG(syslog(LOG_INFO, "Filter finished; now running %d filters", NumRunningFilters));
    (void) pthread_mutex_unlock(&FilterMutex);
    return 0;
}

/**********************************************************************
*%FUNCTION: mfconnect
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
* hostname -- name of connecting host
* sa -- socket address of connecting host
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Allocates a private data structure for tracking this connection
***********************************************************************/
static sfsistat
mfconnect(SMFICTX *ctx, char *hostname, _SOCK_ADDR *sa)
{
    struct privdata *data;
    struct sockaddr_in *insa = (struct sockaddr_in *) sa;
    char *tmp;

    DEBUG_ENTER("mfconnect", __LINE__);

    /* Delete any existing context data */
    mfclose(ctx);

    /* If too many running filters, reject connection at this phase */
    if (MultiplexorSocketName) {
	int n = MXCheckFreeSlaves(MultiplexorSocketName);
	if (n <= 0) {
	    syslog(LOG_WARNING, "mfconnect: No free slaves");
	    DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
    } else {
	if (NumRunningFilters >= MaxRunningFilters) {
	    syslog(LOG_WARNING, "mfconnect: Maximum number of filters (%d) running",
		   MaxRunningFilters);
	    DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
    }

    data = malloc_with_log(sizeof *data);
    if (!data) {
	DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    data->hostname = NULL;
    data->hostip   = NULL;
    data->sender   = NULL;
    data->firstRecip = NULL;
    data->dir      = NULL;
    data->heloArg  = NULL;
    data->qid      = NOQUEUE;
    data->fp       = NULL;
    data->headerFP = NULL;
    data->cmdFP    = NULL;
    data->headerPresent  = 0;
    data->validatePresent = 0;
    data->suspiciousBody = 0;
    data->lastWasCR      = 0;
    data->filterFailed   = 0;

    /* Save private data */
    if (smfi_setpriv(ctx, data) != MI_SUCCESS) {
	free(data);
	/* Can't hurt... */
	smfi_setpriv(ctx, NULL);
	syslog(LOG_WARNING, "Unable to set private data pointer: smfi_setpriv failed");
	DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    if (hostname) {
	data->hostname = strdup_with_log(hostname);
	if (!data->hostname) {
	    mfclose(ctx);
	    DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
    } else {
	data->hostname = NULL;
    }

    /* Padding -- should be big enough for IPv6 addresses, but currently
       only support IPv4 -- see inet_ntoa below.*/
    if (!sa) {
	data->hostip = NULL;
    } else {
	data->hostip = malloc_with_log(65);
	if (!data->hostip) {
	    mfclose(ctx);
	    DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
	tmp = inet_ntoa(insa->sin_addr);
	if (!tmp) {
	    syslog(LOG_WARNING, "inet_ntoa failed: %m");
	    mfclose(ctx);
	    DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
	strncpy(data->hostip, tmp, 64);
	data->hostip[64] = 0;
    }

    data->dir = NULL;
    data->fp = NULL;
    data->headerFP = NULL;
    data->cmdFP = NULL;
    data->suspiciousBody = 0;
    data->lastWasCR = 0;

    DEBUG_EXIT("mfconnect", __LINE__, "SMFIS_CONTINUE");
    return SMFIS_CONTINUE;

}

/**********************************************************************
* %FUNCTION: helo
* %ARGUMENTS:
*  ctx -- Milter context
*  helohost -- argument to "HELO" or "EHLO" SMTP command
* %RETURNS:
*  SMFIS_CONTINUE
* %DESCRIPTION:
*  Stores the HELO argument in the private data area
***********************************************************************/
static sfsistat
helo(SMFICTX *ctx, char *helohost)
{
    struct privdata *data = DATA;
    if (!data) {
	DEBUG_EXIT("helo", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    if (data->heloArg) {
	free(data->heloArg);
	data->heloArg = NULL;
    }
    data->heloArg = strdup_with_log(helohost);
    return SMFIS_CONTINUE;
}

/**********************************************************************
*%FUNCTION: envfrom
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
* from -- list of arguments to "MAIL FROM:" SMTP command.
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Allocates a private data structure for tracking this message, and
* opens a temporary file for saving message body.
***********************************************************************/
static sfsistat
envfrom(SMFICTX *ctx, char **from)
{
    struct privdata *data;
    int fd;
    int tries;
    int success;
    char buffer[SMALLBUF];
    char buf2[SMALLBUF];
    time_t now = time(NULL);
    unsigned long ulnow = (unsigned long) now;
    char *queueid;

    DEBUG_ENTER("envfrom", __LINE__);
    /* Get the private context */
    data = DATA;
    if (!data) {
	syslog(LOG_WARNING, "envfrom: Unable to obtain private data from milter context");
	DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* Obtain the queue identifier */
    queueid = smfi_getsymval(ctx, "i");

    if (data->qid != NOQUEUE) {
	free(data->qid);
	data->qid = NOQUEUE;
    }
    if (queueid && *queueid) {
	data->qid = strdup_with_log(queueid);
	if (!data->qid) {
	    data->qid = NOQUEUE;
	    mfclose(ctx);
	    DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
    }

    /* Copy sender */
    if (data->sender) {
	free(data->sender);
    }
    data->sender = strdup_with_log(from[0]);
    if (!data->sender) {
	mfclose(ctx);
	DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* Old data lying around? */
    if (data->firstRecip) {
	free(data->firstRecip);
	data->firstRecip = NULL;
    }

    /* We need a nice clean directory all our own.  If two
     * threads get the same name, one of the mkdir() calls will fail.
     * On some systems, we might need to lock this region, hence
     * the MkdirMutex */

    success = 0;
    if (protectMkdirWithMutex) {
	if (pthread_mutex_lock(&MkdirMutex)) {
	    syslog(LOG_INFO, "Could not lock MkdirMutex");
	    DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
    }

    for (tries = 0; tries < MAXTRIES; tries++) {
	snprintf(buffer, SMALLBUF, "%s/mdefang-%lX-%d",
		 SPOOLDIR,
		 ulnow, tries);
	if (!mkdir(buffer, 0700)) {
	    success = 1;
	    break;
	}

	/* Apparently, mkdir on Solaris 8 can fail with EBADF */
	if ((errno != EEXIST) && (errno != EBADF)) {
	    break;
	}
    }

    if (protectMkdirWithMutex) {
	(void) pthread_mutex_unlock(&MkdirMutex);
    }

    /* Could not create temp. directory */
    if (!success) {
	syslog(LOG_WARNING, "%s: Could not create directory %s: %m",
	       data->qid, buffer);
	mfclose(ctx);
	DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    if (data->dir) free(data->dir);
    if (data->fp) fclose(data->fp);
    if (data->headerFP) fclose(data->headerFP);
    if (data->cmdFP) fclose(data->cmdFP);

    data->dir = NULL;
    data->fp = NULL;
    data->headerFP = NULL;
    data->cmdFP = NULL;
    data->headerPresent = 0;
    data->validatePresent = 0;
    data->filterFailed = 0;

    data->dir = strdup_with_log(buffer);

    if (!data->dir) {
	/* Don't forget to clean up directory... */
	rmdir(buffer);
	mfclose(ctx);
	DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* Check relaying and sender */
    if (MultiplexorSocketName) {
	if (doRelayCheck) {
	    int n = MXRelayOK(MultiplexorSocketName,
			      buf2, data->hostip, data->hostname,
			      data->heloArg, data->dir);
	    if (n == 0) {
		if (*buf2) {
		    smfi_setreply(ctx, "554", "5.7.1", buf2);
		}

		/* We reject connections from this relay */
		DEBUG_EXIT("envfrom", __LINE__, "SMFIS_REJECT");
		return SMFIS_REJECT;
	    }
	    if (n < 0) {
		if (*buf2) {
		    smfi_setreply(ctx, "450", "4.7.1", buf2);
		}
		DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
		return SMFIS_TEMPFAIL;
	    }
	    if (n == 2) {
		return SMFIS_ACCEPT;
	    }
	}
	if (doSenderCheck) {
	    int n = MXSenderOK(MultiplexorSocketName, buf2,
			       from[0], data->hostip, data->hostname,
			       data->heloArg, data->dir);
	    if (n == 0) {
		if (*buf2) {
		    smfi_setreply(ctx, "554", "5.7.1", buf2);
		}
		/* We reject connections from this sender */
		DEBUG_EXIT("envfrom", __LINE__, "SMFIS_REJECT");
		return SMFIS_REJECT;
	    }
	    if (n < 0) {
		if (*buf2) {
		    smfi_setreply(ctx, "450", "4.7.1", buf2);
		}
		DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
		return SMFIS_TEMPFAIL;
	    }
	    if (n == 2) {
		return SMFIS_ACCEPT;
	    }
	}
    }

    /* Open command file */
    snprintf(buf2, SMALLBUF, "%s/COMMANDS", buffer);
    if ((fd = open(buf2, O_CREAT | O_EXCL | O_APPEND | O_RDWR, 0600)) < 0 ||
	(data->cmdFP = fdopen(fd, "w+")) == NULL) {
	if (fd >= 0) {
	    syslog(LOG_WARNING, "%s: fdopen failed on %s: %m",
		   data->qid, buf2);
	    close(fd);
	} else {
	    syslog(LOG_WARNING, "%s: Could not create %s: %m",
		   data->qid, buf2);
	}
	cleanup(ctx);
	DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    if (queueid) {
	write_mx_command(data->cmdFP, 'Q', (unsigned char *) queueid);
    }

    /* Write host name and host IP */
    write_mx_command(data->cmdFP, 'H', (unsigned char *) data->hostname);
    write_mx_command(data->cmdFP, 'I', (unsigned char *) data->hostip);

    /* Write HELO value */
    if (data->heloArg) {
	write_mx_command(data->cmdFP, 'E', (unsigned char *) data->heloArg);
    }

    /* Synthesize filename for output */
    snprintf(buf2, SMALLBUF, "%s/INPUTMSG", buffer);
    if ((fd = open(buf2, O_CREAT | O_EXCL | O_APPEND | O_RDWR, 0600)) < 0 ||
	(data->fp = fdopen(fd, "w+")) == NULL) {
	if (fd >= 0) {
	    syslog(LOG_WARNING, "%s: fdopen failed on %s: %m",
		   data->qid, buf2);
	    close(fd);
	} else {
	    syslog(LOG_WARNING, "%s: Could not create %s: %m",
		   data->qid, buf2);
	}
	cleanup(ctx);
	DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    snprintf(buf2, SMALLBUF, "%s/HEADERS", buffer);
    if ((fd = open(buf2, O_CREAT | O_EXCL | O_APPEND | O_RDWR, 0600)) < 0 ||
	(data->headerFP = fdopen(fd, "w+")) == NULL) {
	if (fd >= 0) {
	    syslog(LOG_WARNING, "%s: fdopen failed on %s: %m",
		   data->qid, buf2);
	    close(fd);
	} else {
	    syslog(LOG_WARNING, "%s: Could not create %s: %m",
		   data->qid, buf2);
	}
	cleanup(ctx);
	DEBUG_EXIT("envfrom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* Write the sender */
    write_mx_command(data->cmdFP, 'S', (unsigned char *) from[0]);

    /* Write some macros */
    write_macro_value(ctx, "_");
    write_macro_value(ctx, "auth_authen");
    write_macro_value(ctx, "auth_author");
    write_macro_value(ctx, "auth_ssf");
    write_macro_value(ctx, "auth_type");
    write_macro_value(ctx, "cert_issuer");
    write_macro_value(ctx, "cert_subject");
    write_macro_value(ctx, "cipher");
    write_macro_value(ctx, "cipher_bits");
    write_macro_value(ctx, "daemon_name");
    write_macro_value(ctx, "i");
    write_macro_value(ctx, "if_addr");
    write_macro_value(ctx, "if_name");
    write_macro_value(ctx, "j");
    write_macro_value(ctx, "mail_addr");
    write_macro_value(ctx, "mail_host");
    write_macro_value(ctx, "mail_mailer");
    write_macro_value(ctx, "tls_version");
    write_macro_value(ctx, "verify");
    DEBUG_EXIT("envfrom", __LINE__, "SMFIS_CONTINUE");
    return SMFIS_CONTINUE;
}

/**********************************************************************
*%FUNCTION: rcptto
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
* to -- list of arguments to each RCPT_TO
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Saves recipient data
***********************************************************************/
static sfsistat
rcptto(SMFICTX *ctx, char **to)
{
    struct privdata *data = DATA;
    char ans[SMALLBUF];
    sfsistat retcode = SMFIS_CONTINUE;

    DEBUG_ENTER("rcptto", __LINE__);
    if (!data) {
	syslog(LOG_WARNING, "rcptto: Unable to obtain private data from milter context");
	DEBUG_EXIT("rcptto", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    /* Recipient check if enabled */
    if (doRecipientCheck) {
	int n;
	/* If this is first recipient, copy it */
	if (!data->firstRecip) {
	    data->firstRecip = strdup_with_log(to[0]);
	    if (!data->firstRecip) {
		DEBUG_EXIT("rcptto", __LINE__, "SMFIS_TEMPFAIL");
		return SMFIS_TEMPFAIL;
	    }
	}
	n = MXRecipientOK(MultiplexorSocketName, ans,
			  to[0], data->sender, data->hostip,
			  data->hostname, data->firstRecip, data->heloArg,
			  data->dir);
	if (n == 0) {
	    /* We reject to this recipient */
	    if (*ans) {
		smfi_setreply(ctx, "554", "5.7.1", ans);
	    }
	    DEBUG_EXIT("rcptto", __LINE__, "SMFIS_REJECT");
	    return SMFIS_REJECT;
	}
	if (n < 0) {
	    if (*ans) {
		smfi_setreply(ctx, "450", "4.7.1", ans);
	    }
	    DEBUG_EXIT("rcptto", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
	if (n == 2) {
	    retcode = SMFIS_ACCEPT;
	}
    }

    /* Write recipient line, only for recipients we accept! */
    write_mx_command(data->cmdFP, 'R', (unsigned char *) to[0]);

    DEBUG_EXIT("rcptto", __LINE__, "SMFIS_CONTINUE or SMFIS_ACCEPT");
    return retcode;
}

/**********************************************************************
*%FUNCTION: header
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
* headerf -- Header field name
* headerv -- Header value
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Writes the header to the temporary file
***********************************************************************/
static sfsistat
header(SMFICTX *ctx, char *headerf, char *headerv)
{
    struct privdata *data = DATA;
    int suspicious = 0;

    DEBUG_ENTER("header", __LINE__);
    if (!data) {
	syslog(LOG_WARNING, "header: Unable to obtain private data from milter context");
	DEBUG_EXIT("header", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* Write the header to the message file */
    suspicious = safeWriteHeader(data->fp, (unsigned char const *) headerf);
    fprintf(data->fp, ": ");
    suspicious |= safeWriteHeader(data->fp, (unsigned char const *) headerv);
    fprintf(data->fp, "\n");
    if (suspicious) {
	write_mx_command(data->cmdFP, '!', NULL);
    }

    /* Remove embedded newlines and save to our HEADERS file */
    chomp(headerf);
    chomp(headerv);
    fprintf(data->headerFP, "%s: %s\n", headerf, headerv);

    /* Check for subject -- special case */
    if (!strcasecmp(headerf, "subject")) {
	write_mx_command(data->cmdFP, 'U', (unsigned char *) headerv);
    } else if (!strcasecmp(headerf, "message-id")) {
	write_mx_command(data->cmdFP, 'X', (unsigned char *) headerv);
    }

    /* Check for existing identical X-Scanned-By: header */
    if (scan_body && *scan_body) {
	if (!strcasecmp(headerf, "X-Scanned-By") &&
	    !strcasecmp(headerv, scan_body)) {
	    data->headerPresent = 1;
	}
    }

    /* Check for validating IP header.  If found, write a J line
       to the file to reset the SMTP host address */
    if (ValidateHeader[0] && !strcmp(headerf, ValidateHeader)) {
	/* Make sure it looks like an IP address, though... */
	int n, a, b, c, d;
	char ipaddr[32];
	n = sscanf(headerv, "%d.%d.%d.%d", &a, &b, &c, &d);
	if (n == 4 &&
	    a >= 0 && a <= 255 &&
	    b >= 0 && b <= 255 &&
	    c >= 0 && c <= 255 &&
	    d >= 0 && d <= 255) {
	    sprintf(ipaddr, "%d.%d.%d.%d", a, b, c, d);
	    write_mx_command(data->cmdFP, 'J', (unsigned char *) ipaddr);
	    data->validatePresent = 1;
	}
    }

    DEBUG_EXIT("header", __LINE__, "SMFIS_CONTINUE");
    return SMFIS_CONTINUE;
}

/**********************************************************************
*%FUNCTION: eoh
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Writes a blank line to indicate the end of headers.
***********************************************************************/
static sfsistat
eoh(SMFICTX *ctx)
{
    struct privdata *data = DATA;

    DEBUG_ENTER("eoh", __LINE__);
    if (!data) {
	syslog(LOG_WARNING, "eoh: Unable to obtain private data from milter context");
	DEBUG_EXIT("eoh", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* We can close headerFP to save a descriptor */
    if (data->headerFP && (fclose(data->headerFP) == EOF)) {
	data->headerFP = NULL;
	syslog(LOG_WARNING, "%s: Error closing header descriptor: %m", data->qid);
	DEBUG_EXIT("eoh", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    data->headerFP = NULL;
    data->suspiciousBody = 0;
    data->lastWasCR = 0;

    fprintf(data->fp, "\n");
    DEBUG_EXIT("eoh", __LINE__, "SMFIS_CONTINUE");
    return SMFIS_CONTINUE;
}

/**********************************************************************
*%FUNCTION: body
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
* text -- a chunk of text from the mail body
* len -- length of chunk
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Writes a chunk of the body to the temporary file
***********************************************************************/
static sfsistat
body(SMFICTX *ctx, u_char *text, size_t len)
{
    struct privdata *data = DATA;

    DEBUG_ENTER("body", __LINE__);

    if (!data) {
	syslog(LOG_WARNING, "body: Unable to obtain private data from milter context");
	DEBUG_EXIT("body", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    /* Write to file and scan body for suspicious characters */
    if (len) {
	u_char *s = text;
	size_t n;

	/* If last was CR, and this is not LF, suspicious! */
	if (data->lastWasCR && *text != '\n') {
	    data->suspiciousBody = 1;
	}

	data->lastWasCR = 0;
	for (n=0; n<len; n++, s++) {
	    if (*s == '\r') {
		if (n == len-1) {
		    data->lastWasCR = 1;
		} else if (*(s+1) != '\n') {
		    data->suspiciousBody = 1;
		}
		continue;
	    }

	    /* Write char */
	    /* Cast is workaround for GCC bug on Solaris */
	    /* != 255 HACK is workaround for GCC bug! */
	    if (((int) putc(*s, data->fp)) == (int) EOF && *s != 255) {
		syslog(LOG_WARNING, "%s: putc failed: %m line %d",
		       data->qid, __LINE__);
		cleanup(ctx);
		DEBUG_EXIT("body", __LINE__, "SMFIS_TEMPFAIL");
		return SMFIS_TEMPFAIL;
	    }

	    /* Embedded NULL's are cause for concern */
	    if (!*s) {
		data->suspiciousBody = 1;
	    }
	}
    }
    DEBUG_EXIT("body", __LINE__, "SMFIS_CONTINUE");
    return SMFIS_CONTINUE;
}

/**********************************************************************
*%FUNCTION: eom
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* This is where all the action happens.  Called at end of message, it
* forks the Perl scanner which may or may not ask for the body to be
* replaced.
***********************************************************************/
static sfsistat
eom(SMFICTX *ctx)
{
    pid_t n;
    int status;
    char buffer[SMALLBUF];
    FILE *fp;
    struct privdata *data = DATA;
    int r;
    int problem = 0;
    int fd;
    int j;
    unsigned char chunk[CHUNK];
    char *hdr, *val, *count;

    struct timeval start, finish;

    DEBUG_ENTER("eom", __LINE__);
    if (LogTimes) {
	gettimeofday(&start, NULL);
    }

    /* Close output file */
    if (!data) {
	syslog(LOG_WARNING, "eom: Unable to obtain private data from milter context");
	DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }
    /* Signal suspicious body chars */
    if (data->suspiciousBody) {
	write_mx_command(data->cmdFP, '?', NULL);
    }

    /* Signal end of command file */
    write_mx_command(data->cmdFP, 'F', NULL);

    if (!data->fp || (fclose(data->fp) == EOF)) problem = 1;
    if (data->headerFP && (fclose(data->headerFP) == EOF)) problem = 1;
    if (data->cmdFP && (fclose(data->cmdFP) == EOF)) problem = 1;
    data->fp = NULL;
    data->headerFP = NULL;
    data->cmdFP = NULL;

    if (problem) {
	cleanup(ctx);
	DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
	return SMFIS_TEMPFAIL;
    }

    data->suspiciousBody = 0;
    data->lastWasCR = 0;

    /* Run the Perl filter */
    if (MultiplexorSocketName) {
	if (MXScanDir(MultiplexorSocketName, data->dir) < 0) {
	    data->filterFailed = 1;
	    /* If STDERR file is there, log contents */
	    log_perl_stderr(ctx);
	    cleanup(ctx);
	    DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
    } else {
	if (IncrementRunningFilters() < 0) {
	    /* Doh!  Too many running filters... Sorry, have to abandon at
	       this point */
	    cleanup(ctx);
	    DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	}
	n = fork();
	if (n < 0) {
	    syslog(LOG_WARNING, "%s: fork() failed -- cannot run filter",
		   data->qid);
	    DecrementRunningFilters();
	    cleanup(ctx);
	    DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
	    return SMFIS_TEMPFAIL;
	} else if (n == 0) {
	    /* In the child */
	    char const *ppath = FilterProgram;
	    char const *pname = strrchr(ppath, '/');
	    if (pname) {
		pname++;
	    } else {
		pname = ppath;
	    }
	    closefiles();

	    /* Direct stdin/stdout to /dev/null; stderr to STDERR
	       in work dir */
	    open("/dev/null", O_RDWR);
	    open("/dev/null", O_RDWR);

	    snprintf(buffer, SMALLBUF, "%s/STDERR", data->dir);
	    open(buffer, O_WRONLY|O_CREAT|O_TRUNC);
	    if (SubFilter) {
		execl(ppath, pname, "-f", SubFilter, data->dir, NULL);
	    } else {
		execl(ppath, pname, data->dir, NULL);
	    }

	    /* No good to try to use syslog -- closefiles closed
	       the log descriptor.  Instead print to stderr and
	       parent will pick it up and log it */
	    fprintf(stderr, "%s: Could not execute %s: %s\n",
		    data->qid, ppath, strerror(errno));
	    _exit(EXIT_FAILURE);
	} else {
	    /* In the parent */
	    if (waitpid(n, &status, 0) < 0) {
		DecrementRunningFilters();
		cleanup(ctx);
		DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
		return SMFIS_TEMPFAIL;
	    }
	    if (DecrementRunningFilters() < 0) {
		cleanup(ctx);
		DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
		return SMFIS_TEMPFAIL;
	    }
	}
    }

    /* If STDERR file is there, log contents */
    log_perl_stderr(ctx);

    /* Read the results file */
    snprintf(buffer, SMALLBUF, "%s/RESULTS", data->dir);
    fp = fopen(buffer, "r");
    if (!fp) {
	syslog(LOG_WARNING, "%s: Filter did not create RESULTS file", data->qid);
	data->filterFailed = 1;
	cleanup(ctx);
	return SMFIS_TEMPFAIL;
    }

    /* Process the commands in the results file */
    while (fgets(buffer, SMALLBUF, fp)) {
	chomp(buffer);
	switch(*buffer) {
	case 'B':
	    /* Bounce */
	    fclose(fp);
	    syslog(LOG_DEBUG, "%s: Bouncing because filter instructed us to",
		   data->qid);
	    percent_decode((unsigned char *) buffer+1);
	    smfi_setreply(ctx, "554", "5.7.1", buffer+1);
	    cleanup(ctx);
	    r = SMFIS_REJECT;
	    goto bail_out;

	case 'D':
	    /* Discard */
	    fclose(fp);
	    syslog(LOG_DEBUG, "%s: Discarding because filter instructed us to",
		   data->qid);
	    cleanup(ctx);
	    r = SMFIS_DISCARD;
	    goto bail_out;

	case 'T':
	    /* Tempfail */
	    fclose(fp);
	    syslog(LOG_DEBUG, "%s: Tempfailing because filter instructed us to",
		   data->qid);
	    percent_decode((unsigned char *) buffer+1);
	    if (*(buffer+1)) {
		smfi_setreply(ctx, "451", "4.7.1", buffer+1);
	    } else {
		smfi_setreply(ctx, "451", "4.7.1", "Please try again later");
	    }
	    cleanup(ctx);
	    r = SMFIS_TEMPFAIL;
	    goto bail_out;

	case 'C':
	    snprintf(buffer, SMALLBUF, "%s/NEWBODY", data->dir);
	    fd = open(buffer, O_RDONLY);
	    if (fd < 0) {
		syslog(LOG_WARNING, "%s: Could not open %s for reading: %m",
		       data->qid, buffer);
		close(fd);
		cleanup(ctx);
		DEBUG_EXIT("eom", __LINE__, "SMFIS_TEMPFAIL");
		return SMFIS_TEMPFAIL;
	    }
	    while ((j=read(fd, chunk, CHUNK)) > 0) {
		smfi_replacebody(ctx, chunk, j);
	    }
	    close(fd);
	    break;

	case 'M':
	    /* New content-type header */
	    percent_decode((unsigned char *) buffer+1);
	    if (strlen(buffer+1) > 0) {
		smfi_chgheader(ctx, "Content-Type", 1, buffer+1);
	    }
	    smfi_chgheader(ctx, "MIME-Version", 1, "1.0");
	    /* We've just changed the message to multi-part.  Better delete
	       other MIME headers which may be present */
	    smfi_chgheader(ctx, "Content-Disposition", 1, "");
	    smfi_chgheader(ctx, "Content-Transfer-Encoding", 1, "");
	    break;

	case 'H':
	    /* Add a header */
	    split_on_space(buffer+1, &hdr, &val);
	    if (hdr && val) {
		percent_decode((unsigned char *) hdr);
		percent_decode((unsigned char *) val);
		smfi_addheader(ctx, hdr, val);
	    }
	    break;

	case 'I':
	    /* Change a header */
	    split_on_space3(buffer+1, &hdr, &count, &val);
	    if (hdr && val && count) {
		percent_decode((unsigned char *) hdr);
		percent_decode((unsigned char *) count);
		percent_decode((unsigned char *) val);
		if (sscanf(count, "%d", &j) != 1 || j < 1) {
		    j = 1;
		}
		smfi_chgheader(ctx, hdr, j, val);
	    }
	    break;

	case 'J':
	    /* Delete a header */
	    split_on_space(buffer+1, &hdr, &count);
	    if (hdr && count) {
		percent_decode((unsigned char *) hdr);
		percent_decode((unsigned char *) count);
		if (sscanf(count, "%d", &j) != 1 || j < 1) {
		    j = 1;
		}
		smfi_chgheader(ctx, hdr, j, NULL);
	    }
	    break;

	case 'R':
	    /* Add a recipient */
	    percent_decode((unsigned char *) buffer+1);
	    smfi_addrcpt(ctx, buffer+1);
	    break;

	case 'S':
	    /* Delete a recipient */
	    percent_decode((unsigned char *) buffer+1);
	    smfi_delrcpt(ctx, buffer+1);
	    break;

	case 'F':
	    /* We're done */
	    break;

	default:
	    syslog(LOG_WARNING, "%s: Unknown command '%c' in RESULTS file",
		   data->qid, *buffer);
	}
	if (*buffer == 'F') break;
    }

    fclose(fp);

    if (!data->headerPresent) {
	if (scan_body && *scan_body) {
	    smfi_addheader(ctx, "X-Scanned-By", scan_body);
	}
    }

    /* Delete first validation header if it was present */
    if (ValidateHeader[0] && data->validatePresent) {
	smfi_chgheader(ctx, ValidateHeader, 1, NULL);
    }

    r = cleanup(ctx);

  bail_out:
    if (LogTimes) {
	long sec_diff, usec_diff;

	gettimeofday(&finish, NULL);

	sec_diff = finish.tv_sec - start.tv_sec;
	usec_diff = finish.tv_usec - start.tv_usec;

	if (usec_diff < 0) {
	    usec_diff += 1000000;
	    sec_diff--;
	}

	/* Convert to milliseconds */
	sec_diff = sec_diff * 1000 + (usec_diff / 1000);
	syslog(LOG_INFO, "%s: Filter time is %ldms", data->qid, sec_diff);
    }

    DEBUG(syslog(LOG_INFO, "Exiting eom (line %d) ret=%d", __LINE__, r));
    return r;
}

/**********************************************************************
*%FUNCTION: mfclose
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
*%RETURNS:
* SMFIS_ACCEPT
*%DESCRIPTION:
* Called when connection is closed.
***********************************************************************/
static sfsistat
mfclose(SMFICTX *ctx)
{
    struct privdata *data = DATA;

    DEBUG_ENTER("mfclose", __LINE__);
    cleanup(ctx);
    if (data) {
	if (data->fp)         fclose(data->fp);
	if (data->headerFP)   fclose(data->headerFP);
	if (data->cmdFP)      fclose(data->cmdFP);
	if (data->dir)        free(data->dir);
	if (data->hostname)   free(data->hostname);
	if (data->hostip)     free(data->hostip);
	if (data->sender)     free(data->sender);
	if (data->firstRecip) free(data->firstRecip);
	if (data->heloArg)    free(data->heloArg);
	if (data->qid && data->qid != NOQUEUE) free(data->qid);
	free(data);
    }
    smfi_setpriv(ctx, NULL);
    DEBUG_EXIT("mfclose", __LINE__, "SMFIS_CONTINUE");
    return SMFIS_CONTINUE;
}

/**********************************************************************
*%FUNCTION: mfabort
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Called if current message is aborted.  Just cleans up.
***********************************************************************/
static sfsistat
mfabort(SMFICTX *ctx)
{
    return cleanup(ctx);
}

/**********************************************************************
*%FUNCTION: cleanup
*%ARGUMENTS:
* ctx -- Sendmail filter mail context
*%RETURNS:
* SMFIS_TEMPFAIL or SMFIS_CONTINUE
*%DESCRIPTION:
* Cleans up temporary files.
***********************************************************************/
static sfsistat
cleanup(SMFICTX *ctx)
{
    sfsistat r = SMFIS_CONTINUE;
    struct privdata *data = DATA;
    int doRemove = 1;

    DEBUG_ENTER("cleanup", __LINE__);
    if (!data) {
	DEBUG_EXIT("cleanup", __LINE__, "SMFIS_CONTINUE");
	return r;
    }

    if (data->fp && (fclose(data->fp) == EOF)) {
	syslog(LOG_ERR, "%s: Failure in cleanup line %d: %m",
	       data->qid, __LINE__);
	r = SMFIS_TEMPFAIL;
    }
    data->fp = NULL;

    if (data->headerFP && (fclose(data->headerFP) == EOF)) {
	syslog(LOG_ERR, "%s: Failure in cleanup line %d: %m",
	       data->qid, __LINE__);
	r = SMFIS_TEMPFAIL;
    }
    data->headerFP = NULL;

    if (data->cmdFP && (fclose(data->cmdFP) == EOF)) {
	syslog(LOG_ERR, "%s: Failure in cleanup line %d: %m",
	       data->qid, __LINE__);
	r = SMFIS_TEMPFAIL;
    }
    data->cmdFP = NULL;

    /* Remove temporary files unless "-d" command-line switch given or
       SPOOLDIR/DO-NOT-DELETE-WORK-DIRS exists.  Also, if "-k" command-line
       switch was given and the filter failed, do not remove directory. */
    if (DebugMode ||
	(access(SPOOLDIR "/DO-NOT-DELETE-WORK-DIRS", F_OK) == 0)) {
	doRemove = 0;
    }

    if (keepFailedDirectories && data->filterFailed) {
	if (data->dir) {
	    syslog(LOG_WARNING, "%s: Filter failed.  Message is in %s",
		   data->qid, data->dir);
	}
	doRemove = 0;
    }

    if (doRemove) {
	if (data->dir) {
	    if (rm_r(data->dir) < 0) {
		syslog(LOG_ERR, "%s: Failure in cleanup line %d: %m",
		       data->qid, __LINE__);
		r = SMFIS_TEMPFAIL;
	    }
	}
    }

    if (data->dir) {
	free(data->dir);
	data->dir = NULL;
    }
    if (data->sender) {
	free(data->sender);
	data->sender = NULL;
    }
    if (data->firstRecip) {
	free(data->firstRecip);
	data->firstRecip = NULL;
    }

    DEBUG(syslog(LOG_INFO, "Exiting cleanup (line %d) ret=%s", __LINE__, (r == SMFIS_TEMPFAIL ? "SMFIS_TEMPFAIL" : "SMFIS_CONTINUE")));
    return r;
}

static struct smfiDesc filterDescriptor =
{
    "MIMEDefang-" VERSION,      /* Filter name */
    SMFI_VERSION,		/* Version code */

#if SMFI_VERSION == 1
    /* We can: add a header and may alter body and add/delete recipients*/
    SMFIF_MODHDRS|SMFIF_MODBODY|SMFIF_ADDRCPT|SMFIF_DELRCPT,
#else
    /* We can: Add headers, alter body, add/delete recipients, alter headers */
    SMFIF_ADDHDRS|SMFIF_CHGBODY|SMFIF_ADDRCPT|SMFIF_DELRCPT|SMFIF_CHGHDRS,
#endif

    mfconnect,			/* connection */
    helo,			/* HELO */
    envfrom,			/* MAIL FROM: */
    rcptto,			/* RCPT TO: */
    header,			/* Called for each header */
    eoh,			/* Called at end of headers */
    body,			/* Called for each body chunk */
    eom,			/* Called at end of message */
    mfabort,			/* Called on abort */
    mfclose			/* Called on connection close */
};

/**********************************************************************
* %FUNCTION: usage
* %ARGUMENTS:
*  None
* %RETURNS:
*  Nothing (exits)
* %DESCRIPTION:
*  Prints usage information
***********************************************************************/
static void
usage(void)
{
    fprintf(stderr, "mimedefang version %s\n", VERSION);
    fprintf(stderr, "Usage: mimedefang [options]\n");
    fprintf(stderr, "Options:\n");
    fprintf(stderr, "  -h                -- Print usage info and exit\n");
    fprintf(stderr, "  -n maxFilters     -- Maximum number of concurrent filters\n");
    fprintf(stderr, "  -U user           -- Run as user instead of root\n");
    fprintf(stderr, "  -p /path          -- Path to UNIX-domain socket for sendmail communication\n");
    fprintf(stderr, "  -d                -- Enable debugging (do not remove spool files)\n");
    fprintf(stderr, "  -k                -- Do not remove spool files if filter fails\n");
    fprintf(stderr, "  -m /path          -- Use multiplexor; use /path as UNIX-domain socket\n");
    fprintf(stderr, "  -f /dir/filter    -- Specify full path of filter program\n");
    fprintf(stderr, "  -F filter_path    -- Specify path to filter rules file\n");
    fprintf(stderr, "  -r                -- Do relay check before processing body (requires -m)\n");
    fprintf(stderr, "  -s                -- Do sender check before processing body (requires -m)\n");
    fprintf(stderr, "  -t                -- Do recipient checks before processing body (requires -m)\n");
    fprintf(stderr, "  -P file           -- Write process-ID of daemon to specified file\n");
    fprintf(stderr, "  -T                -- Log filter times to syslog\n");
    fprintf(stderr, "  -x string         -- Add string as X-Scanned-By header\n");
    fprintf(stderr, "  -M                -- Protect mkdir with mutex\n");
    exit(EXIT_FAILURE);
}

/**********************************************************************
* %FUNCTION: main
* %ARGUMENTS:
*  argc, argv -- the usual suspects
* %RETURNS:
*  Whatever smfi_main returns
* %DESCRIPTION:
*  Main program
***********************************************************************/
int
main(int argc, char **argv)
{
    int c;
    int maxConn;
    pid_t i;
    char *pidfile = NULL;
    struct passwd *pw = NULL;
    FILE *fp;

    /* Paranoia time */
    umask(077);

    /* Paranoia time II */
    if (getuid() != geteuid()) {
	fprintf(stderr, "ERROR: %s is NOT intended to run suid! Exiting.\n",
		argv[0]);
	exit(1);
    }

    if (getgid() != getegid()) {
	fprintf(stderr, "ERROR: %s is NOT intended to run sgid! Exiting.\n",
		argv[0]);
	exit(1);
    }

    /* Process command line options */
    while ((c = getopt(argc, argv, "hn:p:dm:f:srtkP:F:U:Tx:M")) != -1) {
	switch (c) {
	case 'M':
	    protectMkdirWithMutex = 1;
	    break;
	case 'x':
	    if (scan_body) {
		free(scan_body);
	    }
	    scan_body = strdup(optarg);
	    if (!scan_body) {
		fprintf(stderr, "Out of memory\n");
		exit(EXIT_FAILURE);
	    }
	    break;

	case 'T':
	    LogTimes = 1;
	    break;

	case 'U':
	    /* User to run as */
	    if (user) {
		free(user);
	    }
	    user = strdup(optarg);
	    if (!user) {
		fprintf(stderr, "Out of memory\n");
		exit(EXIT_FAILURE);
	    }
	    break;
	case 'F':
	    if (SubFilter) {
		free((void *) SubFilter);
	    }
	    SubFilter = strdup(optarg);
	    if (!SubFilter) {
		fprintf(stderr, "Out of memory\n");
		exit(EXIT_FAILURE);
	    }
	    break;
	case 'P':
	    /* Write our pid to this file */
	    if (pidfile != NULL) free(pidfile);

	    pidfile = strdup(optarg);
	    if (!pidfile) {
		fprintf(stderr, "Out of memory\n");
		exit(EXIT_FAILURE);
	    }
	    break;
	case 'k':
	    keepFailedDirectories = 1;
	    break;
	case 's':
	    doSenderCheck = 1;
	    break;
	case 'r':
	    doRelayCheck = 1;
	    break;
	case 't':
	    doRecipientCheck = 1;
	    break;
	case 'h':
	    usage();
	    break;
	case 'f':
	    /* Filter program */
	    if (optarg[0] != '/') {
		fprintf(stderr, "-f: You must supply an absolute path for filter program\n");
		exit(EXIT_FAILURE);
	    }
	    FilterProgram = strdup(optarg);
	    if (!FilterProgram) {
		fprintf(stderr, "strdup: Out of memory\n");
		exit(EXIT_FAILURE);
	    }
	    break;
	case 'm':
	    /* Multiplexor */
	    MultiplexorSocketName = strdup(optarg);
	    if (!MultiplexorSocketName) {
		fprintf(stderr, "strdup: Out of memory\n");
		exit(EXIT_FAILURE);
	    }
	    break;
	case 'd':
	    DebugMode = 1;
	    break;

	case 'n':
	    if (optarg == NULL || sscanf(optarg, "%d", &maxConn) != 1) {
		fprintf(stderr, "Usage: mimedefang -n max_filters\n");
		exit(EXIT_FAILURE);
	    }
	    if (maxConn <= 0) {
		fprintf(stderr, "Usage: mimedefang -n max_filters (max_filters > 0)\n");
		exit(EXIT_FAILURE);
	    }
	    MaxRunningFilters = maxConn;
	    break;
	case 'p':
	    if (optarg == NULL || *optarg == '\0') {
		(void) fprintf(stderr, "Illegal conn: %s\n",
			       optarg);
		exit(EXIT_FAILURE);
	    }
	    /* Remove socket from file system */
	    (void) remove(optarg);
	    if (smfi_setconn(optarg) != MI_SUCCESS) {
		fprintf(stderr, "Could not open connection %s: %s",
			optarg, strerror(errno));
		exit(EXIT_FAILURE);
	    }
	    break;
	default:
	    usage();
	    break;
	}
    }

    if (!scan_body) {
	scan_body = SCAN_BODY;
    }

    if (!MultiplexorSocketName) {
	fprintf(stderr, "WARNING: Use of mimedefang without the -m option is deprecated.\n");
	fprintf(stderr, "Support for non-multiplexor operation will be removed in a future release.\n");
	if (doSenderCheck) {
	    fprintf(stderr, "Cannot use -s option without -m\n");
	    doSenderCheck = 0;
	}
	if (doRelayCheck) {
	    fprintf(stderr, "Cannot use -r option without -m\n");
	    doRelayCheck = 0;
	}
	if (doRecipientCheck) {
	    fprintf(stderr, "Cannot use -t option without -m\n");
	    doRecipientCheck = 0;
	}
    }

    /* Look up user */
    if (user) {
	pw = getpwnam(user);
	if (!pw) {
	    fprintf(stderr, "Unknown user '%s'", user);
	    exit(EXIT_FAILURE);
	}
	if (drop_privs(user, pw->pw_uid, pw->pw_gid) < 0) {
	    fprintf(stderr, "Could not drop privileges: %s", strerror(errno));
	    exit(EXIT_FAILURE);
	}
	free(user);
    }

    /* Warn */
    if (!getuid() || !geteuid()) {
	fprintf(stderr,
		"ERROR: You must not run mimedefang as root.\n"
		"Use the -U option to set a non-root user.\n");
	exit(EXIT_FAILURE);
    }


    if (chdir(SPOOLDIR) < 0) {
	fprintf(stderr, "Unable to chdir(%s): %s\n",
		SPOOLDIR, strerror(errno));
	exit(EXIT_FAILURE);
    }

    /* Read key file if present */
    fp = fopen(KEY_FILE, "r");
    if (fp) {
	fgets(ValidateHeader, sizeof(ValidateHeader), fp);
	fclose(fp);
	chomp(ValidateHeader);
    } else {
	ValidateHeader[0] = 0;
    }
    if (smfi_register(filterDescriptor) == MI_FAILURE) {
	fprintf(stderr, "smfi_register failed\n");
	exit(EXIT_FAILURE);
    }

    /* Daemonize */
    i = fork();
    if (i < 0) {
	fprintf(stderr, "fork() failed\n");
	exit(EXIT_FAILURE);
    } else if (i != 0) {
	/* parent */
	exit(EXIT_SUCCESS);
    }
    setsid();
    signal(SIGHUP, SIG_IGN);
    i = fork();
    if (i < 0) {
	fprintf(stderr, "fork() failed\n");
	exit(EXIT_FAILURE);
    } else if (i != 0) {
	exit(EXIT_SUCCESS);
    }

    /* Write pid */
    if (pidfile) {
	FILE *fp = fopen(pidfile, "w");
	if (fp) {
	    fprintf(fp, "%d\n", (int) getpid());
	    fclose(fp);
	}
	free(pidfile);
    }

    (void) closelog();
    closefiles();

    /* Direct stdin/stdout/stderr to /dev/null */
    open("/dev/null", O_RDWR);
    open("/dev/null", O_RDWR);
    open("/dev/null", O_RDWR);

    openlog("mimedefang", LOG_PID, LOG_MAIL);

    if (ValidateHeader[0]) {
	syslog(LOG_DEBUG, "IP validation header is %s", ValidateHeader);
    }
#ifdef ENABLE_DEBUGGING
    signal(SIGSEGV, handle_sig);
    signal(SIGBUS, handle_sig);
#endif

    return smfi_main();
}

/**********************************************************************
* %FUNCTION: write_macro_value
* %ARGUMENTS:
*  ctx -- Sendmail milter context
*  macro -- name of a macro
* %RETURNS:
*  Nothing
* %DESCRIPTION:
*  Sends a command to Perl code to set a macro value
***********************************************************************/
static void
write_macro_value(SMFICTX *ctx,
		  char *macro)
{
    struct privdata *data;
    char *val;
    char buf[256];

    data = DATA;
    if (!data || !data->cmdFP) return;

    if (*macro && *(macro+1)) {
	/* Longer than 1 char -- use curlies */
	snprintf(buf, sizeof(buf), "{%s}", macro);
	val = smfi_getsymval(ctx, buf);
    } else {
	val = smfi_getsymval(ctx, macro);
    }
    if (!val) return;
    fprintf(data->cmdFP, "=");
    write_percent_encoded((unsigned char *)macro, data->cmdFP);
    fprintf(data->cmdFP, " ");
    write_percent_encoded((unsigned char *)val, data->cmdFP);
    fprintf(data->cmdFP, "\n");
}
