/***********************************************************************
*
* utils.c
*
* Utility functions for MIMEDefang
*
* Copyright (C) 2002-2003 Roaring Penguin Software Inc.
* http://www.roaringpenguin.com
*
* This program may be distributed under the terms of the GNU General
* Public License, Version 2, or (at your option) any later version.
*
***********************************************************************/

static char const RCSID[] =
"$Id: utils.c,v 1.18 2003/02/14 18:28:29 dfs Exp $";

#define _BSD_SOURCE 1

#ifdef __linux__
/* On Linux, we need this defined to get fdopen.  On BSD, if we define
 * it, we don't get all the u_char, u_long, etc definitions.  GRR! */
#define _POSIX_SOURCE 1
#endif

#include "config.h"
#include "mimedefang.h"

#include <ctype.h>
#include <syslog.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <errno.h>
#include <stdarg.h>

#ifndef AF_LOCAL
#define AF_LOCAL AF_UNIX
#endif

/**********************************************************************
* %FUNCTION: split_on_space
* %ARGUMENTS:
*  buf -- input buffer
*  first -- set to first word
*  rest -- set to everything following a space, or NULL if no space
* %RETURNS:
*  Nothing
* %DESCRIPTION:
*  Splits a line on whitespace.
***********************************************************************/
void
split_on_space(char *buf,
	       char **first,
	       char  **rest)
{
    *first = buf;
    *rest = NULL;
    while(*buf && !isspace(*buf)) buf++;
    if (*buf && isspace(*buf)) {
	*buf = 0;
	*rest = buf+1;
    }
}

/**********************************************************************
* %FUNCTION: split_on_space3
* %ARGUMENTS:
*  buf -- input buffer
*  first -- set to first word
*  second -- set to second word or NULL
*  rest -- set to everything following a space, or NULL if no space
* %RETURNS:
*  Nothing
* %DESCRIPTION:
*  Splits a line on whitespace.
***********************************************************************/
void
split_on_space3(char *buf,
		char **first,
		char **second,
		char **rest)
{
    *first = buf;
    *second = NULL;
    *rest = NULL;
    while(*buf && !isspace(*buf)) buf++;
    if (*buf && isspace(*buf)) {
	*buf = 0;
	*second = buf+1;
	buf++;
	while(*buf && !isspace(*buf)) buf++;
	if (*buf && isspace(*buf)) {
	    *buf = 0;
	    *rest = buf+1;
	}
    }
}
/**********************************************************************
* %FUNCTION: malloc_with_log
* %ARGUMENTS:
*  size -- amount of memory to allocate
* %RETURNS:
*  Allocated memory
* %DESCRIPTION:
*  Calls malloc, but syslogs an error on failure to allocate memory.
***********************************************************************/
void *
malloc_with_log(size_t s)
{
    void *p = malloc(s);
    if (!p) {
	syslog(LOG_WARNING, "Failed to allocate %lu bytes of memory",
	       (unsigned long) s);
    }
    return p;
}

/**********************************************************************
* %FUNCTION: strdup_with_log
* %ARGUMENTS:
*  s -- string to strdup
* %RETURNS:
*  A copy of s in malloc'd memory.
* %DESCRIPTION:
*  Calls strdup, but syslogs an error on failure to allocate memory.
***********************************************************************/
char *
strdup_with_log(char const *s)
{
    char *p = strdup(s);
    if (!p) {
	syslog(LOG_WARNING, "Failed to allocate %d bytes of memory in strdup",
	       (int) strlen(s)+1);
    }
    return p;
}

/**********************************************************************
*%FUNCTION: chomp
*%ARGUMENTS:
* str -- a string
*%RETURNS:
* Nothing
*%DESCRIPTION:
* Removes newlines and carriage-returns (if any) from str
***********************************************************************/
void
chomp(char *str)
{
    char *s, *t;
    s = str;
    for (t=str; *t; t++) {
	if (*t == '\n' || *t == '\r') continue;
	*s++ = *t;
    }
    *s = 0;
}

/**********************************************************************
* %FUNCTION: safeWriteHeader
* %ARGUMENTS:
*  fp -- file to write to
*  str -- a string value
* %RETURNS:
*  0 if header seems OK; 1 if suspicious character found.
* %DESCRIPTION:
*  Writes "str" to FP with the following changes:
*    CRLF -> written as-is
*    LFCR -> written as CRLF
*    CR   -> written as CRLF
*    LF   -> written as CRLF
***********************************************************************/
int
safeWriteHeader(FILE *fp,
		unsigned char const *str)
{
    int suspicious = 0;
    for(; *str; str++) {
	/* Do not write \r to header file */
	if (*str == '\r') {
	    if (*(str+1) != '\n') {
		suspicious = 1;
	    }
	    continue;
	}

	putc(*str, fp);
    }

    return suspicious;
}

/**********************************************************************
* %FUNCTION: MXCommand
* %ARGUMENTS:
*  sockname -- multiplexor socket name
*  cmd -- command to send
*  buf -- buffer for reply
*  len -- length of buffer
* %RETURNS:
*  0 if all went well, -1 on error.
* %DESCRIPTION:
*  Sends a command to the multiplexor and reads the answer back.
***********************************************************************/
int
MXCommand(char const *sockname,
	  char const *cmd,
	  char *buf,
	  int len)
{
    int fd;
    struct sockaddr_un addr;
    int nleft, nwritten, nread;
    int n;

    fd = socket(AF_LOCAL, SOCK_STREAM, 0);
    if (fd < 0) {
	syslog(LOG_ERR, "MXCommand: socket: %m");
	return -1;
    }

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_LOCAL;
    strncpy(addr.sun_path, sockname, sizeof(addr.sun_path) - 1);

    if (connect(fd, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
	syslog(LOG_ERR, "MXCommand: connect: %m: Is multiplexor running?");
	close(fd);
	return -1;
    }

    nwritten = 0;
    nleft = strlen(cmd);
    n = 0;
    while(nleft) {
	n = write(fd, cmd+nwritten, nleft);
	if (n < 0) {
	    if (errno == EINTR) continue;
	    break;
	}
	nwritten += n;
	nleft -= n;
    }
    if (n < 0) {
	syslog(LOG_ERR, "MXCommand: write: %m: Is multiplexor running?");
	close(fd);
	return -1;
    }

    /* Now read the answer */
    nread = 0;
    nleft = len;
    while(nleft) {
	n = read(fd, buf+nread, nleft);
	if (n <= 0) {
	    if (n < 0 && errno == EINTR) continue;
	    break;
	}
	nread += n;
	nleft -= n;
    }
    if (n < 0) {
	syslog(LOG_ERR, "MXCommand: read: %m: Is multiplexor running?");
	close(fd);
	return -1;
    }
    buf[nread] = 0;
    close(fd);
    return 0;
}

/**********************************************************************
* %FUNCTION: MXCheckFreeSlaves
* %ARGUMENTS:
*  sockname -- MX socket name
* %RETURNS:
*  >0 if there are free slaves, 0 if all slaves are busy, -1 if there
*  was an error.
* %DESCRIPTION:
*  Queries multiplexor for number of free slaves.
***********************************************************************/
int
MXCheckFreeSlaves(char const *sockname)
{
    char ans[SMALLBUF];
    int slaves;

    if (MXCommand(sockname, "free\n", ans, SMALLBUF-1) < 0) return -1;

    if (sscanf(ans, "%d", &slaves) != 1) return -1;
    return slaves;
}

/**********************************************************************
* %FUNCTION: MXScanDir
* %ARGUMENTS:
*  sockname -- MX socket name
*  dir -- directory to scan
* %RETURNS:
*  0 if scanning succeeded; -1 if there was an error.
* %DESCRIPTION:
*  Asks multiplexor to initiate a scan.
***********************************************************************/
int
MXScanDir(char const *sockname,
	  char const *dir)
{
    char cmd[SMALLBUF];
    char ans[SMALLBUF];
    int len;

    snprintf(cmd, SMALLBUF, "scan %s\n", dir);
    if (MXCommand(sockname, cmd, ans, SMALLBUF-1) < 0) return -1;

    if (!strcmp(ans, "ok\n")) return 0;

    len = strlen(ans);
    if (len > 0 && ans[len-1] == '\n') ans[len-1] = 0;
    syslog(LOG_ERR, "Error from multiplexor: %s", ans);
    return -1;
}

/**********************************************************************
* %FUNCTION: write_percent_encoded
* %ARGUMENTS:
*  str -- a string
*  fp -- file to write to
* %RETURNS:
*  Nothing
* %DESCRIPTION:
*  Writes a string to a file, escaping the following chars:
*  <= ascii-32; > ascii-126, %, \, ', ".  These are replaced with "%XY"
*  where X and Y are hex digits.
* "
***********************************************************************/
void
write_percent_encoded(unsigned char const *str,
		      FILE *fp)
{
    int c;
    while ((c = *str++) != 0) {
	if (c <= 32 || c > 126 || c == '%' || c == '\\' || c == '\'' || c == '"') {
	    fprintf(fp, "%%%02X", (unsigned int) c);
	} else {
	    fputc(c, fp);
	}
    }
}

/**********************************************************************
* %FUNCTION: percent_decode
* %ARGUMENTS:
*  buf -- a buffer with percent-encoded data
* %RETURNS:
*  Nothing
* %DESCRIPTION:
*  Decodes buf IN PLACE.
***********************************************************************/
void
percent_decode(unsigned char *buf)
{
    unsigned char *in = buf;
    unsigned char *out = buf;
    unsigned int val;

    while(*in) {
	if (*in == '%' && isxdigit(*(in+1)) && isxdigit(*(in+2))) {
	    sscanf(in+1, "%2x", &val);
	    *out++ = (unsigned char) val;
	    in += 3;
	    continue;
	}
	*out++ = *in++;
    }
    /* Copy terminator */
    *out = 0;
}

/**********************************************************************
* %FUNCTION: percent_encode
* %ARGUMENTS:
*  in -- input buffer to encode
*  out -- output buffer to place encoded data
*  outlen -- number of chars in output buffer.
* %RETURNS:
*  Number of chars written, not including trailing NULL.  Ranges from
*  0 to outlen-1
* %DESCRIPTION:
*  Encodes "in" into "out", writing at most (outlen-1) chars.  Then writes
*  trailing 0.
***********************************************************************/
int
percent_encode(unsigned char *in,
	       unsigned char *out,
	       int outlen)
{
    unsigned char tmp[8];
    int nwritten = 0;
    unsigned char c;

    if (outlen <= 0) {
	return 0;
    }
    if (outlen == 1) {
	*out = 0;
	return 0;
    }

    /* Do real work */
    while((c = *in++) != 0) {
	if (c <= 32 || c > 126 || c == '%' || c == '\\' || c == '\'' || c == '"') {
	    if (nwritten >= outlen-3) {
		break;
	    }
	    sprintf((char *) tmp, "%%%02X", (unsigned int) c);
	    *out++ = tmp[0];
	    *out++ = tmp[1];
	    *out++ = tmp[2];
	    nwritten += 3;
	} else {
	    *out++ = c;
	    nwritten++;
	}
	if (nwritten >= outlen-1) {
	    break;
	}
    }
    out[nwritten] = 0;
    return nwritten;
}

/**********************************************************************
* %FUNCTION: percent_encode_command
* %ARGUMENTS:
*  out -- output buffer
*  outlen -- length of output buffer
*  args -- arguments.  Each one is percent-encoded and space-separated from
*          previous.
* %RETURNS:
*  0 if everything fits; -1 otherwise.
* %DESCRIPTION:
*  Writes a series of space-separated, percent-encoded words to a buffer.
*  Buffer is terminated with "\n\0"
***********************************************************************/
int
percent_encode_command(char *out, int outlen, ...)
{
    va_list ap;
    int spaceleft = outlen-2;
    int first = 1;
    int len;
    char *arg;

    if (outlen < 2) return -1;

    va_start(ap, outlen);

    while ((arg = va_arg(ap, char *)) != NULL) {
	if (first) {
	    first = 0;
	} else {
	    if (spaceleft <= 0) {
		va_end(ap);
		return -1;
	    }
	    *out++ = ' ';
	    spaceleft--;
	}
	len = percent_encode((unsigned char *) arg, (unsigned char *) out, spaceleft);
	spaceleft -= len;
	out += len;
    }
    va_end(ap);
    *out++ = '\n';
    *out = 0;
    return 0;
}

/**********************************************************************
* %FUNCTION: MXRelayOK
* %ARGUMENTS:
*  sockname -- multiplexor socket name
*  msg -- buffer for holding error message, at least SMALLBUF chars
*  ip -- relay IP address
*  name -- relay name
*  helo -- argument to "HELO/EHLO" (may be NULL)
*  dir -- MIMEDefang working directory
* %RETURNS:
*  1 if it's OK to accept connections from this host; 0 if not, -1 if error.
*  If connection is rejected, error message *may* be set.
***********************************************************************/
int
MXRelayOK(char const *sockname,
	  char *msg,
	  char const *ip,
	  char const *name,
	  char const *helo,
	  char const *dir)
{
    char cmd[SMALLBUF];
    char ans[SMALLBUF];
    int len;

    *msg = 0;

    if (!ip || !*ip) {
	ip = "UNKNOWN";
    }
    if (!name || !*name) {
	name = ip;
    }
    if (!helo) {
	helo = "";
    }
    percent_encode_command(cmd, sizeof(cmd), "relayok", ip, name, helo,
			   dir, NULL);
    if (MXCommand(sockname, cmd, ans, SMALLBUF-1) < 0) return -1;
    if (!strcmp(ans, "ok -1\n")) return -1;
    if (!strcmp(ans, "ok 1\n")) return 1;
    if (!strcmp(ans, "ok 2\n")) return 2;
    if (!strcmp(ans, "ok 0\n")) return 0;

    chomp(ans);

    /* If rejection message is supplied, set it */
    len = strlen(ans);
    if (len >= 6 && !strncmp(ans, "ok 0 ", 5)) {
	strcpy(msg, ans+5);
	return 0;
    }

    if (len >= 7 && !strncmp(ans, "ok -1 ", 6)) {
	strcpy(msg, ans+6);
	return -1;
    }

    if (len > 0 && ans[len-1] == '\n') ans[len-1] = 0;
    syslog(LOG_ERR, "Error from multiplexor: %s", ans);
    return -1;

}

/**********************************************************************
* %FUNCTION: MXSenderOK
* %ARGUMENTS:
*  sockname -- socket name
*  msg -- buffer of at least SMALLBUF size for error message
*  sender -- sender's e-mail address
*  ip -- sending relay's IP address
*  name -- sending relay's host name
*  helo -- argument to "HELO/EHLO" (may be NULL)
*  dir -- MIMEDefang working directory
* %RETURNS:
*  1 if it's OK to accept messages from this sender; 0 if not, -1 if error or
*  we should tempfail.
*  If message is rejected, error message *may* be set.
***********************************************************************/
int
MXSenderOK(char const *sockname,
	   char *msg,
	   char const *sender,
	   char const *ip,
	   char const *name,
	   char const *helo,
	   char const *dir)
{
    char cmd[SMALLBUF];
    char ans[SMALLBUF];
    int len;

    *msg = 0;

    if (!sender || !*sender) {
	sender = "UNKNOWN";
    }

    if (!ip || !*ip) {
	ip = "UNKNOWN";
    }
    if (!name || !*name) {
	name = ip;
    }
    if (!helo) {
	helo = "";
    }
    percent_encode_command(cmd, sizeof(cmd), "senderok", sender, ip, name,
			   helo, dir, NULL);
    if (MXCommand(sockname, cmd, ans, SMALLBUF-1) < 0) return -1;
    if (!strcmp(ans, "ok -1\n")) return -1;
    if (!strcmp(ans, "ok 1\n")) return 1;
    if (!strcmp(ans, "ok 2\n")) return 2;
    if (!strcmp(ans, "ok 0\n")) return 0;

    chomp(ans);

    /* If rejection message is supplied, set failure code and return 0 */
    len = strlen(ans);
    if (len >= 6 && !strncmp(ans, "ok 0 ", 5)) {
	strcpy(msg, ans+5);
	return 0;
    }

    if (len >= 7 && !strncmp(ans, "ok -1 ", 6)) {
	strcpy(msg, ans+6);
	return -1;
    }

    if (len > 0 && ans[len-1] == '\n') ans[len-1] = 0;
    syslog(LOG_ERR, "Error from multiplexor: %s", ans);
    return -1;
}

/**********************************************************************
* %FUNCTION: MXRecipientOK
* %ARGUMENTS:
*  sockname -- multiplexor socket name
*  msg -- buffer of at least SMALLBUF size for error messages
*  recipient -- recipient e-mail address
*  sender -- sender's e-mail address
*  ip -- sending relay's IP address
*  name -- sending relay's host name
*  firstRecip -- first recipient of the message
*  helo -- argument to "HELO/EHLO" (may be NULL)
*  dir -- MIMEDefang working directory
* %RETURNS:
*  1 if it's OK to accept messages to this recipient; 0 if not, -1 if error.
*  If recipient is rejected, error message *may* be set.
***********************************************************************/
int
MXRecipientOK(char const *sockname,
	      char *msg,
	      char const *recipient,
	      char const *sender,
	      char const *ip,
	      char const *name,
	      char const *firstRecip,
	      char const *helo,
	      char const *dir)
{
    char cmd[SMALLBUF];
    char ans[SMALLBUF];
    int len;

    *msg = 0;

    if (!recipient || !*recipient) {
	recipient = "UNKNOWN";
    }

    if (!sender || !*sender) {
	sender = "UNKNOWN";
    }

    if (!ip || !*ip) {
	ip = "UNKNOWN";
    }
    if (!name || !*name) {
	name = ip;
    }

    if (!firstRecip || !*firstRecip) {
	firstRecip = "UNKNOWN";
    }
    if (!helo) {
	helo = "";
    }

    percent_encode_command(cmd, sizeof(cmd),
			   "recipok", recipient, sender, ip, name, firstRecip,
			   helo, dir, NULL);
    if (MXCommand(sockname, cmd, ans, SMALLBUF-1) < 0) return -1;
    if (!strcmp(ans, "ok -1\n")) return -1;
    if (!strcmp(ans, "ok 1\n")) return 1;
    if (!strcmp(ans, "ok 2\n")) return 2;
    if (!strcmp(ans, "ok 0\n")) return 0;

    chomp(ans);

    /* If rejection message is supplied, set failure code and return 0 */
    len = strlen(ans);
    if (len >= 6 && !strncmp(ans, "ok 0 ", 5)) {
	strcpy(msg, ans+5);
	return 0;
    }

    if (len >= 7 && !strncmp(ans, "ok -1 ", 6)) {
	strcpy(msg, ans+6);
	return -1;
    }

    if (len > 0 && ans[len-1] == '\n') ans[len-1] = 0;
    syslog(LOG_ERR, "Error from multiplexor: %s", ans);
    return -1;
}

/**********************************************************************
* %FUNCTION: write_mx_command
* %ARGUMENTS:
*  fp -- file to write to
*  cmd -- the command to write.  A single character.
*  buf -- command arguments
* %RETURNS:
*  Nothing
* %DESCRIPTION:
*  Writes a command for communicating with filter.
***********************************************************************/
void
write_mx_command(FILE *fp,
		 char cmd,
		 unsigned char const *buf)
{
    if (!fp) return;
    fprintf(fp, "%c", cmd);
    if (buf) {
	write_percent_encoded(buf, fp);
    }
    fprintf(fp, "\n");
}
