#include "guts.h"
#include <sys/file.h>
#include <sys/stat.h>
#include <pwd.h>

extern char *getenv(char *);

#define NPERUSER	3

#ifndef HIGHSCOREFILE
#define HIGHSCOREFILE	"/tmp/missile.scores"
#endif

#define USERNAME_WIDTH	8
#define FULLNAME_WIDTH	32
#define SCORE_WIDTH	8
#define LEVEL_WIDTH	3
#define USERNAME_POSN	0
#define FULLNAME_POSN	USERNAME_WIDTH
#define SCORE_POSN	(FULLNAME_POSN + FULLNAME_WIDTH)
#define LEVEL_POSN	(SCORE_POSN + SCORE_WIDTH + 1)
#define RECORD_END	(LEVEL_POSN + LEVEL_WIDTH)
#define RECORD_LENGTH	64

typedef struct {
	char username[USERNAME_WIDTH + 1];
	char fullname[FULLNAME_WIDTH + 1];
	int score;
	int level;
} ScoreLine;

ScoreLine *high_score;
int personal[NPERUSER];
int nhighscores;

int my_score_line = -1;

void
null_terminate(char *s, int len)
{
	s[len--] = '\0';
	while (len >= 0 && s[len] == ' ')
		s[len--] = '\0';
}

void
get_scores(FILE *fp)
{
	char record[RECORD_LENGTH];
	ScoreLine *line = high_score;
	int i;

	while (fread(record, sizeof record, 1, fp) == 1) {
		strncpy(line->username, record, USERNAME_WIDTH);
		null_terminate(line->username, USERNAME_WIDTH);
		strncpy(line->fullname, &record[FULLNAME_POSN], FULLNAME_WIDTH);
		null_terminate(line->fullname, FULLNAME_WIDTH);
		record[SCORE_POSN + SCORE_WIDTH] = '\0';
		line->score = atoi(&record[SCORE_POSN]);
		record[LEVEL_POSN + LEVEL_WIDTH] = '\0';
		line->level = atoi(&record[LEVEL_POSN]);
		line++;
	}
}

void
put_scores(FILE *fp)
{
	ScoreLine *line;
	char record[RECORD_LENGTH];

	fseek(fp, 0L, 0);
	for (line = high_score; line < &high_score[nhighscores]; line++) {
		fprintf(fp, "%-*s%-*s%*d %*d%*s\n",
			USERNAME_WIDTH, line->username,
			FULLNAME_WIDTH, line->fullname,
			SCORE_WIDTH, line->score,
			LEVEL_WIDTH, line->level,
			RECORD_LENGTH - RECORD_END - 1, "");
	}
}

int
update_scores(int new_score, int level)
{
	static char score_file_name[] = HIGHSCOREFILE;
	FILE *score_file;
	struct passwd *passwd_entry;
	int i, j, npersonal;
	char *t;
	ScoreLine new_line;
	struct stat sfstat;

	if ((passwd_entry = getpwuid(getuid())) == NULL)
		sprintf(new_line.username, "#%d", getuid());
	else
		(void) strcpy(new_line.username, passwd_entry->pw_name);
	if ((t = getenv("NAME")) == NULL) {
		if (passwd_entry == NULL)
			strcpy(new_line.fullname, "Who Knows?");
		else {
			t = index(passwd_entry->pw_gecos, ',');
			if (t != NULL)
				*t = '\0';
			strncpy(new_line.fullname, passwd_entry->pw_gecos,
				FULLNAME_WIDTH);
		}
	} else
		strncpy(new_line.fullname, t, FULLNAME_WIDTH);
	new_line.fullname[FULLNAME_WIDTH] = '\0';
	new_line.score = new_score * 100;
	new_line.level = level;

	if ((score_file = fopen(score_file_name, "r+")) == NULL) {
		if ((score_file = fopen(score_file_name, "w+")) == NULL) {
			fprintf(stderr, "missile: can't open score file \"%s\"",
				score_file_name);
			perror("");
			return FALSE;
		}
		fchmod(fileno(score_file), 0666);
	}
	flock(fileno(score_file), LOCK_EX);
	if (fstat(fileno(score_file), &sfstat) < 0)
		nhighscores = 0;
	else
		nhighscores = sfstat.st_size / RECORD_LENGTH;
	high_score =
		(ScoreLine *) malloc(sizeof(ScoreLine) * (nhighscores + 1));
	if (high_score == NULL) {
		perror("missile: malloc");
		fclose(score_file);
		return FALSE;
	}
	get_scores(score_file);
	if (new_score >= 0) {
		high_score[nhighscores].score = -1;
		for (i = 0; i <= nhighscores; i++) {
			if (new_line.score > high_score[i].score ||
					new_line.score == high_score[i].score &&
					new_line.level > high_score[i].level) {
				for (j = nhighscores - 1; j >= i; j--)
					high_score[j + 1] = high_score[j];
				high_score[i] = new_line;
				my_score_line = i;
				nhighscores++;
				break;
			}
		}
	}
	npersonal = 0;
	for (i = 0; i < nhighscores; i++)
		if (strcmp(high_score[i].username, new_line.username) == 0) {
			if (npersonal == NPERUSER) {
				for (j = i; j < nhighscores - 1; j++)
					high_score[j] = high_score[j + 1];
				if (my_score_line == i)
					my_score_line = -1;
				nhighscores--;
				break;
			} else
				personal[npersonal++] = i;
		}
	while (npersonal < NPERUSER)
		personal[npersonal++] = -1;

	if (my_score_line >= 0)
		put_scores(score_file);
	fclose(score_file);
	return my_score_line >= 0;
}

extern GC erase_gc, sta_gc, fall_gc;
extern XFontStruct *the_font;

#define border		4
#define bigborder	6

#define NALLTIME	15
#define NROWS		(4 + NALLTIME + NPERUSER)
#define NCOLUMNS	5

#define POSN_COL	0
#define UNAME_COL	1
#define FNAME_COL	2
#define SCORE_COL	3
#define LEVEL_COL	4

static int height[NROWS];
static int vposn[NROWS];
static int vsep[NROWS + 1];
static int width[NCOLUMNS];
static int hposn[NCOLUMNS];
static int hsep[NCOLUMNS + 1];
static int ascent, overall_height, overall_width;

static char uname_label[] = "User";
static char fname_label[] = "Full Name";
static char score_label[] = "Score";
static char level_label[] = "Level";

static void
calculate_posns()
{
	int i, posn;
	int x, y;

	posn = 0;
	for (i = 0; ; i++) {
		posn += vsep[i];
		if (i == NROWS)
			break;
		vposn[i] = posn;
		posn += height[i];
	}
	overall_height = posn;
	posn = 0;
	for (i = 0; ; i++) {
		posn += hsep[i];
		if (i == NCOLUMNS)
			break;
		hposn[i] = posn;
		posn += width[i];
	}
	overall_width = posn;
	x = (MAXX - overall_width) / 2;
	y = (MAXY - overall_height) / 2;
	for (i = 0; i < NROWS; i++)
		vposn[i] += y;
	for (i = 0; i < NCOLUMNS; i++)
		hposn[i] += x;
}

int
maximum(int a, int b)
{
	if (a > b)
		return a;
	else
		return b;
}

static void
left_string(GC gc, int col, int row, char *s)
{
	XDrawString(display, win, gc, hposn[col], vposn[row] + ascent,
		s, strlen(s));
}

static void
right_string(GC gc, int col, int row, char *s)
{
	int nc, w;

	nc = strlen(s);
	w = XTextWidth(the_font, s, nc);
	XDrawString(display, win, gc,
		hposn[col] + width[col] - w, vposn[row] + ascent,
		s, nc);
}

static void
number(int n, int col, int row)
{
	char nbuf[10];

	sprintf(nbuf, "%d", n);
	right_string(sta_gc, col, row, nbuf);
}

static void
draw_labels(int row, int do_name)
{
	if (do_name)
		left_string(sta_gc, UNAME_COL, row, uname_label);
	left_string(sta_gc, FNAME_COL, row, fname_label);
	right_string(sta_gc, SCORE_COL, row, score_label);
	right_string(sta_gc, LEVEL_COL, row, level_label);
}

static void
draw_score_line(int do_name, int row, int s)
{
	if (s == my_score_line) {
		XFillRectangle(display, win, fall_gc,
			hposn[0] - 2, vposn[row] - 1,
			hposn[NCOLUMNS - 1] + width[NCOLUMNS - 1]
				- hposn[0] + 4,
			height[row] + 2);
	}
	number(s + 1, POSN_COL, row);
	if (do_name)
		left_string(sta_gc, UNAME_COL, row, high_score[s].username);
	left_string(sta_gc, FNAME_COL, row, high_score[s].fullname);
	number(high_score[s].score, SCORE_COL, row);
	number(high_score[s].level, LEVEL_COL, row);
}

void
display_scores()
{
	static char alltime_label[] = "High Scores";
	static char personal_label[] = "Personal High Scores";
	int i, n;
	int digit_width, letter_width, name_width;
	int line_height;
	XEvent event;

	ascent = the_font->ascent;
	line_height = ascent + the_font->descent;
	for (i = 0; i <= NROWS; i++) {
		vsep[i] = 0;
		height[i] = line_height;
	}
	digit_width = XTextWidth(the_font, "0", 1);
	letter_width = XTextWidth(the_font, "m", 1);
	n = 10;
	i = 1;
	while (n < nhighscores) {
		n *= 10;
		i++;
	}
	width[POSN_COL] = digit_width * i;
	hsep[UNAME_COL] = letter_width;
	width[UNAME_COL] = letter_width * USERNAME_WIDTH;
	hsep[FNAME_COL] = 2 * letter_width;
	width[FNAME_COL] = letter_width * FULLNAME_WIDTH / 2;
	for (i = 0; i < NALLTIME && i < nhighscores; i++) {
		name_width = XTextWidth(the_font, high_score[i].fullname,
			strlen(high_score[i].fullname));
		if (width[FNAME_COL] < name_width)
			width[FNAME_COL] = name_width;
	}
	hsep[SCORE_COL] = 2 * letter_width;
	width[SCORE_COL] = maximum(
		digit_width * SCORE_WIDTH,
		XTextWidth(the_font, score_label, strlen(score_label))
	);
	hsep[LEVEL_COL] = 2 * letter_width;
	width[LEVEL_COL] = maximum(
		digit_width * LEVEL_WIDTH,
		XTextWidth(the_font, level_label, strlen(level_label))
	);
	hsep[0] = bigborder + border;
	hsep[NCOLUMNS] = bigborder + border;
	vsep[0] = bigborder;
	vsep[2] = border;
	vsep[NALLTIME+2] = 2 * border;
	vsep[NALLTIME+4] = border;
	vsep[NROWS] = bigborder + border;

	calculate_posns();

	XFillRectangle(display, win, erase_gc,
		hposn[0] - hsep[0], vposn[0] - vsep[0],
		overall_width, overall_height);
	XDrawRectangle(display, win, sta_gc,
		hposn[0] - hsep[0], vposn[0] - vsep[0],
		overall_width - 1, overall_height - 1);
	XDrawRectangle(display, win, sta_gc,
		hposn[0] - border, vposn[2] - border,
		overall_width - 2 * bigborder - 1,
		vposn[NALLTIME + 1] - vposn[2] + height[NALLTIME + 1] +
		    2 * border - 1);
	XDrawRectangle(display, win, sta_gc,
		hposn[0] - border, vposn[NALLTIME + 4] - border,
		overall_width - 2 * bigborder - 1,
		vposn[NROWS - 1] - vposn[NALLTIME + 4] + height[NROWS - 1] +
		    2 * border - 1);

	XSetForeground(display, fall_gc, colors[YELLOW]);
	XSetForeground(display, erase_gc, colors[RED]);

	left_string(erase_gc, 0, 0, alltime_label);
	draw_labels(1, TRUE);
	for (i = 0; i < NALLTIME && i < nhighscores; i++)
		draw_score_line(TRUE, i + 2, i);
	left_string(erase_gc, 0, NALLTIME + 2, personal_label);
	draw_labels(NALLTIME + 3, FALSE);
	for (i = 0; i < NPERUSER && personal[i] >= 0; i++)
		draw_score_line(FALSE, i + NALLTIME + 4, personal[i]);
	do {
		XNextEvent(display, &event);
	} while (event.type != ButtonPress && event.type != KeyPress);
}
