/* 
 *  imdct_kni.c
 *
 *	Copyright (C) Aaron Holtzman - May 1999
 *
 *  This file is part of ac3dec, a free Dolby AC-3 stream decoder.
 *	
 *  ac3dec is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  ac3dec is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 *  Modified for using with mplayer by Nick Kurshev <nickols_k@mail.ru>
 */
 
extern void fft_128p_kni(complex_t *);
extern void fft_64p_kni(complex_t *);

/* Delay buffer for time domain interleaving */
static float xcos_sin_sse[128 * 4] __attribute__((aligned(16)));

extern void imdct512_pre_ifft_twiddle_kni(const int *pmt, complex_t *buf, float *data, float *xcos_sin_sse);
extern void imdct512_post_ifft_twiddle_kni(complex_t *buf, float *xcos_sin_sse);
extern void imdct512_window_delay_kni(complex_t *buf, float *data_ptr, float *window_prt, float *delay_prt);
extern void imdct512_window_delay_nol_kni(complex_t *buf, float *data_ptr, float *window_prt, float *delay_prt);

void imdct_init (void)
{
  int i;
  float scale = 255.99609372;

  for (i=0; i < 128; i++) {
	float xcos_i = cos(2.0f * M_PI * (8*i+1)/(8*N)) * scale;
       	float xsin_i = sin(2.0f * M_PI * (8*i+1)/(8*N)) * scale;
       	xcos_sin_sse[i * 4] = xcos_i;
       	xcos_sin_sse[i * 4 + 1] = -xsin_i;
       	xcos_sin_sse[i * 4 + 2] = -xsin_i;
       	xcos_sin_sse[i * 4 + 3] = -xcos_i;
  }
}


static void imdct_do_512 (float data[], float delay[])
{
	imdct512_pre_ifft_twiddle_kni (pm128, buf, data, xcos_sin_sse);
	fft_128p_kni (buf);
	imdct512_post_ifft_twiddle_kni (buf, xcos_sin_sse);
	imdct512_window_delay_kni (buf, data, window, delay);
}


static void imdct_do_512_nol (float data[], float delay[])
{
	imdct512_pre_ifft_twiddle_kni (pm128, buf, data, xcos_sin_sse);  
	fft_128p_kni (buf);
	imdct512_post_ifft_twiddle_kni (buf, xcos_sin_sse);
	imdct512_window_delay_nol_kni (buf, data, window, delay);
}

static void imdct_do_256 (float data[],float delay[])
{
	int i, j, k;
	int p, q;

	float tmp_a_i;
	float tmp_a_r;

	float *data_ptr;
	float *delay_ptr;
	float *window_ptr;

	complex_t *buf1, *buf2;

	buf1 = &buf[0];
	buf2 = &buf[64];

// Pre IFFT complex multiply plus IFFT complex conjugate
	for (k=0; k<64; k++) { 
		/* X1[k] = X[2*k]	*/
		/* X2[k] = X[2*k+1]	*/

		j = pm64[k];
		p = 2 * (128-2*j-1);
		q = 2 * (2 * j);

		/* Z1[k] = (X1[128-2*k-1] + j * X1[2*k]) * (xcos2[k] + j * xsin2[k]); */
		buf1[k].re =         data[p] * xcos2[j] - data[q] * xsin2[j];
		buf1[k].im = -1.0f * (data[q] * xcos2[j] + data[p] * xsin2[j]);
		/* Z2[k] = (X2[128-2*k-1] + j * X2[2*k]) * (xcos2[k] + j * xsin2[k]); */
		buf2[k].re =          data[p + 1] * xcos2[j] - data[q + 1] * xsin2[j];
		buf2[k].im = -1.0f * ( data[q + 1] * xcos2[j] + data[p + 1] * xsin2[j]);
	}

	fft_64p_kni(&buf1[0]);
	fft_64p_kni(&buf2[0]);

#ifdef DEBUG
       //DEBUG FFT
#if 0
	printf ("Post FFT, buf1\n");
	for (i=0; i < 64; i++)
		printf("%d %f %f\n", i, buf_1[i].re, buf_1[i].im);
	printf ("Post FFT, buf2\n");
	for (i=0; i < 64; i++)
		printf("%d %f %f\n", i, buf_2[i].re, buf_2[i].im);
#endif
#endif


	// Post IFFT complex multiply 
	for( i=0; i < 64; i++) {
		tmp_a_r =  buf1[i].re;
		tmp_a_i = -buf1[i].im;
		buf1[i].re =(tmp_a_r * xcos2[i])  -  (tmp_a_i  * xsin2[i]);
		buf1[i].im =(tmp_a_r * xsin2[i])  +  (tmp_a_i  * xcos2[i]);
		tmp_a_r =  buf2[i].re;
		tmp_a_i = -buf2[i].im;
		buf2[i].re =(tmp_a_r * xcos2[i])  -  (tmp_a_i  * xsin2[i]);
		buf2[i].im =(tmp_a_r * xsin2[i])  +  (tmp_a_i  * xcos2[i]);
	}
	
	data_ptr = data;
	delay_ptr = delay;
	window_ptr = window;

	/* Window and convert to real valued signal */
	for(i=0; i< 64; i++) { 
		*data_ptr++  = -buf1[i].im      * *window_ptr++ + *delay_ptr++;
		*data_ptr++  =  buf1[64-i-1].re * *window_ptr++ + *delay_ptr++;
	}

	for(i=0; i< 64; i++) {
		*data_ptr++  = -buf1[i].re      * *window_ptr++ + *delay_ptr++;
		*data_ptr++  =  buf1[64-i-1].im * *window_ptr++ + *delay_ptr++;
	}
	
	delay_ptr = delay;

	for(i=0; i< 64; i++) {
		*delay_ptr++ = -buf2[i].re      * *--window_ptr;
		*delay_ptr++ =  buf2[64-i-1].im * *--window_ptr;
	}

	for(i=0; i< 64; i++) {
		*delay_ptr++ =  buf2[i].im      * *--window_ptr;
		*delay_ptr++ = -buf2[64-i-1].re * *--window_ptr;
	}
}


/**
 *
 **/

static void imdct_do_256_nol (float data[], float delay[])
{
	int i, j, k;
	int p, q;

	float tmp_a_i;
	float tmp_a_r;

	float *data_ptr;
	float *delay_ptr;
	float *window_ptr;

	complex_t *buf1, *buf2;

	buf1 = &buf[0];
	buf2 = &buf[64];

       /* Pre IFFT complex multiply plus IFFT cmplx conjugate */
	for(k=0; k<64; k++) {
               /* X1[k] = X[2*k]  */
               /* X2[k] = X[2*k+1]     */
               j = pm64[k];
               p = 2 * (128-2*j-1);
               q = 2 * (2 * j);

               /* Z1[k] = (X1[128-2*k-1] + j * X1[2*k]) * (xcos2[k] + j * xsin2[k]); */
               buf1[k].re =         data[p] * xcos2[j] - data[q] * xsin2[j];
               buf1[k].im = -1.0f * (data[q] * xcos2[j] + data[p] * xsin2[j]);
               /* Z2[k] = (X2[128-2*k-1] + j * X2[2*k]) * (xcos2[k] + j * xsin2[k]); */
               buf2[k].re =          data[p + 1] * xcos2[j] - data[q + 1] * xsin2[j];
               buf2[k].im = -1.0f * ( data[q + 1] * xcos2[j] + data[p + 1] * xsin2[j]);
       }


       fft_64p_kni(&buf1[0]);
       fft_64p_kni(&buf2[0]);

#ifdef DEBUG
	//DEBUG FFT
#if 0
       printf("Post FFT, buf1\n");
       for (i=0; i < 64; i++)
               printf("%d %f %f\n", i, buf_1[i].re, buf_1[i].im);
       printf("Post FFT, buf2\n");
       for (i=0; i < 64; i++)
               printf("%d %f %f\n", i, buf_2[i].re, buf_2[i].im);
#endif
#endif

       /* Post IFFT complex multiply */
       for( i=0; i < 64; i++) {
               /* y1[n] = z1[n] * (xcos2[n] + j * xs in2[n]) ; */
               tmp_a_r =  buf1[i].re;
               tmp_a_i = -buf1[i].im;
               buf1[i].re =(tmp_a_r * xcos2[i])  -  (tmp_a_i  * xsin2[i]);
               buf1[i].im =(tmp_a_r * xsin2[i])  +  (tmp_a_i  * xcos2[i]);
               /* y2[n] = z2[n] * (xcos2[n] + j * xsin2[n]) ; */
               tmp_a_r =  buf2[i].re;
               tmp_a_i = -buf2[i].im;
               buf2[i].re =(tmp_a_r * xcos2[i])  -  (tmp_a_i  * xsin2[i]);
               buf2[i].im =(tmp_a_r * xsin2[i])  +  (tmp_a_i  * xcos2[i]);
       }
      
       data_ptr = data;
       delay_ptr = delay;
       window_ptr = window;

       /* Window and convert to real valued signal, no overlap */
       for(i=0; i< 64; i++) {
               *data_ptr++  = -buf1[i].im      * *window_ptr++;
               *data_ptr++  =  buf1[64-i-1].re * *window_ptr++;
       }

       for(i=0; i< 64; i++) {
               *data_ptr++  = -buf1[i].re      * *window_ptr++ + *delay_ptr++;
               *data_ptr++  =  buf1[64-i-1].im * *window_ptr++ + *delay_ptr++;
       }

       delay_ptr = delay;

       for(i=0; i< 64; i++) {
               *delay_ptr++ = -buf2[i].re      * *--window_ptr;
               *delay_ptr++ =  buf2[64-i-1].im * *--window_ptr;
       }

       for(i=0; i< 64; i++) {
               *delay_ptr++ =  buf2[i].im      * *--window_ptr;
               *delay_ptr++ = -buf2[64-i-1].re * *--window_ptr;
	}
}
