
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/ioctl.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>

#include "libvo2.h"
#include "img_format.h"

#include "../drivers/mga_vid.h"

// Driver info:
static vo2_info_t info = {
        "Matrox G200/G4x0 mga_vid.o",
        "mga",
        "Arpad Gereoffy <arpi@thot.banki.hu>",
        "requires mga_vid.o kernel module"
};

// Local, driver-dependent data:  (do not use globals! -> reentrancy)
typedef struct {
    int f;
    int mga_next_frame;
    mga_vid_config_t mga_vid_config;
    vo2_surface_t surfaces[4];
} priv_t;

#include "vo2_def.h"

// open hardware/lib, get capabilities
// this function will be called first, before any other control() or start() calls
// return:  1=success  0=failed
vo2_handle_t* vo2_init_mga(){
    vo2_handle_t* h;
    char *devname="/dev/mga_vid";
    int f;

    printf("vo2_mga: init()\n");

	f = open(devname,O_RDWR);
	if(f == -1)
	{
		printf("Couldn't open %s\n",devname); 
		return(NULL);
	}

    h=new_handle();
    if(!h) return NULL;
    ((priv_t*)h->priv)->f=f;
    ((priv_t*)h->priv)->mga_vid_config.num_frames=4;
    ((priv_t*)h->priv)->mga_next_frame=0;

    return h;
}

#define PRIV ((priv_t*)p)

// control (get/set/query) device parameters
//  for example: query supported pixel formats, en/disable double buffering,
//  query hw/sw scaling capabilities, switch window/fullscreen,
//  get best matching resolution for a given image size etc...
static int control(void *p, int cmd, void* param){
    printf("vo2_mga: control(0x%X)\n",cmd);
    switch(cmd){
    case VO2CTRL_QUERY_SURFACE_FAST: return VO2_FALSE;
    case VO2CTRL_QUERY_SURFACE_DIRECT: return VO2_TRUE;
    case VO2CTRL_GET_MAX_SURFACES: return 4;
    case VO2CTRL_QUERY_FORMAT:
	switch((int)param){
        case IMGFMT_YV12:
        case IMGFMT_I420:
        case IMGFMT_IYUV:
        case IMGFMT_YUY2:
        case IMGFMT_UYVY:
	    return VO2_TRUE;
	}
	return VO2_FALSE;
    case VO2CTRL_QUERY_SWSCALE: return VO2_FALSE;
    case VO2CTRL_QUERY_HWSCALE: return VO2_TRUE;
    }
    return VO2_UNKNOWN;
}

// start drawing (set video mode, allocate image buffers etc.)
// w,h: requested surface size (actual size may be larger!!!)
// format: IMGFMT_* requested surface pixel format
// surfaces: 0=audo 1..x=number of requested surfaces/buffers
// flags: various things, like fullscreen, sw/hw zoom and vidmode change
// return: 0=fail (fail if pixel format or buffering not supported)
//         n=number of surfaces allocated
static int start(void *p, int w,int h,int format,int surfaces,int flags){
    int width=((w+31)&(~31));
    int height=h;
    int planar=0;

    // open window / switch vidmode, set up surfaces etc...
    printf("vo2_mga: start() %dx%d  %s  %d  0x%X\n",w,h,vo_format_name(format),surfaces,flags);
    
    if(surfaces){
	if(surfaces>4) return 0; // can't make more than 4 surfaces! (yet)
	PRIV->mga_vid_config.num_frames=surfaces;
    }

        switch(format){
        case IMGFMT_YV12:
	    planar=1;
	    PRIV->mga_vid_config.frame_size = width * height + (width * height) / 2;
            PRIV->mga_vid_config.format=MGA_VID_FORMAT_YV12; break;
        case IMGFMT_I420:
	    planar=1;
	    PRIV->mga_vid_config.frame_size = width * height + (width * height) / 2;
            PRIV->mga_vid_config.format=MGA_VID_FORMAT_I420; break;
        case IMGFMT_IYUV:
	    planar=1;
	    PRIV->mga_vid_config.frame_size = width * height + (width * height) / 2;
            PRIV->mga_vid_config.format=MGA_VID_FORMAT_IYUV; break;
        case IMGFMT_YUY2:
	    PRIV->mga_vid_config.frame_size = width * height * 2;
            PRIV->mga_vid_config.format=MGA_VID_FORMAT_YUY2; break;
        case IMGFMT_UYVY:
	    PRIV->mga_vid_config.frame_size = width * height * 2;
            PRIV->mga_vid_config.format=MGA_VID_FORMAT_UYVY; break;
        default: 
            printf("mga: invalid output format %0X\n",format);
            return 0;
        }

        PRIV->mga_vid_config.colkey_on=0;

	PRIV->mga_vid_config.src_width = w;
	PRIV->mga_vid_config.src_height= h;
	PRIV->mga_vid_config.dest_width = w;
	PRIV->mga_vid_config.dest_height= h;
	PRIV->mga_vid_config.x_org= 0; // (720-mga_vid_config.dest_width)/2;
	PRIV->mga_vid_config.y_org= 0; // (576-mga_vid_config.dest_height)/2;
	
	PRIV->mga_vid_config.version=MGA_VID_VERSION;

	if (ioctl(PRIV->f,MGA_VID_CONFIG,&PRIV->mga_vid_config))
	{
		perror("Error in mga_vid_config ioctl()");
                printf("Your mga_vid driver version is incompatible with this MPlayer version!\n");
		return 0;
	}

	{ char * ptr=(char*)mmap(0,PRIV->mga_vid_config.frame_size*PRIV->mga_vid_config.num_frames,PROT_WRITE,MAP_SHARED,PRIV->f,0);
	  int i;
	  if(!ptr) {
	      printf("mmap failed\n");
	      return 0;
	  }

	  // clear the buffer:
	  memset(ptr,0x80,PRIV->mga_vid_config.frame_size*PRIV->mga_vid_config.num_frames);

	  // create surfaces:
	  for(i=0;i<PRIV->mga_vid_config.num_frames;i++){
	      vo2_surface_t* s=&PRIV->surfaces[i];
	      s->w=w; s->h=h;
	      s->x=s->y=0;
	      s->img[0]=ptr;
	      if(planar){
	          s->img[1]=ptr+width*height;
	          s->img[2]=s->img[1]+width*height/4;
		  s->stride[0]=width;
	          s->stride[1]=s->stride[2]=s->stride[0]/2;
		  s->bpp=12;
	      } else {
		  s->stride[0]=2*width;
		  s->bpp=16;
	      }
	      s->format=format;
	      ptr+=PRIV->mga_vid_config.frame_size;
	  }
	}

	// enable display:
	ioctl(PRIV->f,MGA_VID_ON,0);
	  
    return PRIV->mga_vid_config.num_frames;
}

static int stop(void *p){
    // stop rendering, close device
    printf("vo2_mga: stop()\n");
    ioctl( PRIV->f,MGA_VID_OFF,0 );
    return 1;
}

// get destination surface (for direct rendering or generic draw_ functions)
// num: number of frame. 0 = temporary frame - can be dropped/modified
//                       1-2 = static frames - should not be modified
// Note:  mpeg will use 0,1,2 frames for B,Pf,Pb  (or fallback to 0-only)
//        win32 will use only 0
static vo2_surface_t* get_surface(void *p, int num){
    printf("vo2_mga: get_surface(%d) -> %d\n",num,PRIV->mga_next_frame);
    return &PRIV->surfaces[PRIV->mga_next_frame];
}

static void flip_image(void *p, int num){
    // we can assume that num is valid (get_surface(num) will return non-NULL)

    printf("vo2_mga: flip_image(%d)\n",num);

	ioctl(PRIV->f,MGA_VID_FSEL,&PRIV->mga_next_frame);
	PRIV->mga_next_frame=(PRIV->mga_next_frame+1)%PRIV->mga_vid_config.num_frames;
//	vid_data=frames[mga_next_frame];
  
    // let's show surface[num]

//    if(double_buffering) current_surface=current_surface^1; // swap buffers
}

