/*
 *  MPEG Real Time Encoder
 *  MPEG-1 Audio Layer II Tables
 *
 *  Copyright (C) 1999-2000 Michael H. Schimek
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* $Id: tables.c,v 1.1 2001/12/04 23:55:53 mswitch Exp $ */

#include "mpeg.h"

const int
sampling_freq_value[MPEG_VERSIONS][4] =
{
	{ 11025, 12000, 8000 },
	{ 0 },
	{ 22050, 24000, 16000 },
	{ 44100, 48000, 32000 }
};

const int
bit_rate_value[MPEG_VERSIONS][16] =
{
	{ 0, 8000, 16000, 24000, 32000, 40000, 48000, 56000, 64000, 80000, 96000, 112000, 128000, 144000, 160000 },
	{ 0 },
	{ 0, 8000, 16000, 24000, 32000, 40000, 48000, 56000, 64000, 80000, 96000, 112000, 128000, 144000, 160000 },
	{ 0, 32000, 48000, 56000, 64000, 80000, 96000, 112000, 128000, 160000, 192000, 224000, 256000, 320000, 384000 }
};

/*
 *  Some subbands have the same properties, this information is stored here
 *  to keep the tables small
 */
const unsigned char
subband_group[TABLES][SBLIMIT] =
{
	{ 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3 },
	{ 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3 },
	{ 4, 4, 5, 5, 5, 5, 5, 5 },
	{ 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5 },
	{ 6, 6, 6, 6, 5, 5, 5, 5, 5, 5, 5, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7 }
};

/*
 *  Bit allocation index 0..15 to bits per sample encoded
 */
const unsigned char
bits_table[NUM_SG][MAX_BA_INDICES] =
{
	{ 0, 5, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16 },
	{ 0, 5, 7, 3, 10, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 16 },
	{ 0, 5, 7, 3, 10, 4, 5, 16 },
	{ 0, 5, 7, 16 },
	{ 0, 5, 7, 10, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
	{ 0, 5, 7, 10, 4, 5, 6, 7 },
	{ 0, 5, 7, 3, 10, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14 },
	{ 0, 5, 7, 10 }
};

/*
 *  1 << n indicates packed encoding for subband n (first 5, 7, 10 in bits_table)
 */
const unsigned char
pack_table[NUM_SG] = { 0x02, 0x16, 0x16, 0x06, 0x0E, 0x0E, 0x16, 0x0E };

/*
 *  Quantization step size
 */
const unsigned int
steps_table[NUM_SG][MAX_BA_INDICES] =
{
	{ 0, 3, 7, 15, 31, 63, 127, 255, 511, 1023, 2047, 4095, 8191, 16383, 32767, 65535 },
	{ 0, 3, 5, 7, 9, 15, 31, 63, 127, 255, 511, 1023, 2047, 4095, 8191, 65535 },
	{ 0, 3, 5, 7, 9, 15, 31, 65535 },
	{ 0, 3, 5, 65535 },
	{ 0, 3, 5, 9, 15, 31, 63, 127, 255, 511, 1023, 2047, 4095, 8191, 16383, 32767 }, 
	{ 0, 3, 5, 9, 15, 31, 63, 127 },
	{ 0, 3, 5, 7, 9, 15, 31, 63, 127, 255, 511, 1023, 2047, 4095, 8191, 16383 },
	{ 0, 3, 5, 9 }
};

const unsigned char
quant_table[NUM_SG][MAX_BA_INDICES] =
{
	{ 0, 0, 2, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16 },
	{ 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 16 },
	{ 0, 0, 1, 2, 3, 4, 5, 16 },
	{ 0, 0, 1, 16 },
	{ 0, 0, 1, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
	{ 0, 0, 1, 3, 4, 5, 6, 7 },
	{ 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14 },
	{ 0, 0, 1, 3 }
};

/*
 *  ISO/IEC 11172-3 Table 3-C.5 Layer II Signal-To-Noise Ratios
 *  Steps 0, 3, 5, 7, 9, 15, 31, 63, ...
 */
const float
SNR[18] =
{
	0.00, 7.00, 11.00, 16.00, 20.84, 25.28, 31.59, 37.75, 43.84,
        49.89, 55.93, 61.96, 67.98, 74.01, 80.03, 86.05, 92.01, 98.01
};

/*
 *  ISO/IEC 11172-3 Table 3-C.1 Coefficients of the Analysis Window.
 */
const double
C[512] =
{
     0.000000000, -0.000000477, -0.000000477, -0.000000477,
    -0.000000477, -0.000000477, -0.000000477, -0.000000954,
    -0.000000954, -0.000000954, -0.000000954, -0.000001431,
    -0.000001431, -0.000001907, -0.000001907, -0.000002384,
    -0.000002384, -0.000002861, -0.000003338, -0.000003338,
    -0.000003815, -0.000004292, -0.000004768, -0.000005245,
    -0.000006199, -0.000006676, -0.000007629, -0.000008106,
    -0.000009060, -0.000010014, -0.000011444, -0.000012398,
    -0.000013828, -0.000014782, -0.000016689, -0.000018120,
    -0.000019550, -0.000021458, -0.000023365, -0.000025272,
    -0.000027657, -0.000030041, -0.000032425, -0.000034809,
    -0.000037670, -0.000040531, -0.000043392, -0.000046253,
    -0.000049591, -0.000052929, -0.000055790, -0.000059605,
    -0.000062943, -0.000066280, -0.000070095, -0.000073433,
    -0.000076771, -0.000080585, -0.000083923, -0.000087261,
    -0.000090599, -0.000093460, -0.000096321, -0.000099182,
     0.000101566,  0.000103951,  0.000105858,  0.000107288,
     0.000108242,  0.000108719,  0.000108719,  0.000108242,
     0.000106812,  0.000105381,  0.000102520,  0.000099182,
     0.000095367,  0.000090122,  0.000084400,  0.000077724,
     0.000069618,  0.000060558,  0.000050545,  0.000039577,
     0.000027180,  0.000013828, -0.000000954, -0.000017166,
    -0.000034332, -0.000052929, -0.000072956, -0.000093937,
    -0.000116348, -0.000140190, -0.000165462, -0.000191212,
    -0.000218868, -0.000247478, -0.000277042, -0.000307560,
    -0.000339031, -0.000371456, -0.000404358, -0.000438213,
    -0.000472546, -0.000507355, -0.000542164, -0.000576973,
    -0.000611782, -0.000646591, -0.000680923, -0.000714302,
    -0.000747204, -0.000779152, -0.000809669, -0.000838757,
    -0.000866413, -0.000891685, -0.000915051, -0.000935555,
    -0.000954151, -0.000968933, -0.000980854, -0.000989437,
    -0.000994205, -0.000995159, -0.000991821, -0.000983715,
     0.000971317,  0.000953674,  0.000930786,  0.000902653,
     0.000868797,  0.000829220,  0.000783920,  0.000731945,
     0.000674248,  0.000610352,  0.000539303,  0.000462532,
     0.000378609,  0.000288486,  0.000191689,  0.000088215,
    -0.000021458, -0.000137329, -0.000259876, -0.000388145,
    -0.000522137, -0.000661850, -0.000806808, -0.000956535,
    -0.001111031, -0.001269817, -0.001432419, -0.001597881,
    -0.001766682, -0.001937389, -0.002110004, -0.002283096,
    -0.002457142, -0.002630711, -0.002803326, -0.002974033,
    -0.003141880, -0.003306866, -0.003467083, -0.003622532,
    -0.003771782, -0.003914356, -0.004048824, -0.004174709,
    -0.004290581, -0.004395962, -0.004489899, -0.004570484,
    -0.004638195, -0.004691124, -0.004728317, -0.004748821,
    -0.004752159, -0.004737377, -0.004703045, -0.004649162,
    -0.004573822, -0.004477024, -0.004357815, -0.004215240,
    -0.004049301, -0.003858566, -0.003643036, -0.003401756,
     0.003134727,  0.002841473,  0.002521515,  0.002174854,
     0.001800537,  0.001399517,  0.000971317,  0.000515938,
     0.000033379, -0.000475883, -0.001011848, -0.001573563,
    -0.002161503, -0.002774239, -0.003411293, -0.004072189,
    -0.004756451, -0.005462170, -0.006189346, -0.006937027,
    -0.007703304, -0.008487225, -0.009287834, -0.010103703,
    -0.010933399, -0.011775017, -0.012627602, -0.013489246,
    -0.014358521, -0.015233517, -0.016112804, -0.016994476,
    -0.017876148, -0.018756866, -0.019634247, -0.020506859,
    -0.021372318, -0.022228718, -0.023074150, -0.023907185,
    -0.024725437, -0.025527000, -0.026310921, -0.027073860,
    -0.027815342, -0.028532982, -0.029224873, -0.029890060,
    -0.030526638, -0.031132698, -0.031706810, -0.032248020,
    -0.032754898, -0.033225536, -0.033659935, -0.034055710,
    -0.034412861, -0.034730434, -0.035007000, -0.035242081,
    -0.035435200, -0.035586357, -0.035694122, -0.035758972,
     0.035780907,  0.035758972,  0.035694122,  0.035586357,
     0.035435200,  0.035242081,  0.035007000,  0.034730434,
     0.034412861,  0.034055710,  0.033659935,  0.033225536,
     0.032754898,  0.032248020,  0.031706810,  0.031132698,
     0.030526638,  0.029890060,  0.029224873,  0.028532982,
     0.027815342,  0.027073860,  0.026310921,  0.025527000,
     0.024725437,  0.023907185,  0.023074150,  0.022228718,
     0.021372318,  0.020506859,  0.019634247,  0.018756866,
     0.017876148,  0.016994476,  0.016112804,  0.015233517,
     0.014358521,  0.013489246,  0.012627602,  0.011775017,
     0.010933399,  0.010103703,  0.009287834,  0.008487225,
     0.007703304,  0.006937027,  0.006189346,  0.005462170,
     0.004756451,  0.004072189,  0.003411293,  0.002774239,
     0.002161503,  0.001573563,  0.001011848,  0.000475883,
    -0.000033379, -0.000515938, -0.000971317, -0.001399517,
    -0.001800537, -0.002174854, -0.002521515, -0.002841473,
     0.003134727,  0.003401756,  0.003643036,  0.003858566,
     0.004049301,  0.004215240,  0.004357815,  0.004477024,
     0.004573822,  0.004649162,  0.004703045,  0.004737377,
     0.004752159,  0.004748821,  0.004728317,  0.004691124,
     0.004638195,  0.004570484,  0.004489899,  0.004395962,
     0.004290581,  0.004174709,  0.004048824,  0.003914356,
     0.003771782,  0.003622532,  0.003467083,  0.003306866,
     0.003141880,  0.002974033,  0.002803326,  0.002630711,
     0.002457142,  0.002283096,  0.002110004,  0.001937389,
     0.001766682,  0.001597881,  0.001432419,  0.001269817,
     0.001111031,  0.000956535,  0.000806808,  0.000661850,
     0.000522137,  0.000388145,  0.000259876,  0.000137329,
     0.000021458, -0.000088215, -0.000191689, -0.000288486,
    -0.000378609, -0.000462532, -0.000539303, -0.000610352,
    -0.000674248, -0.000731945, -0.000783920, -0.000829220,
    -0.000868797, -0.000902653, -0.000930786, -0.000953674,
     0.000971317,  0.000983715,  0.000991821,  0.000995159,
     0.000994205,  0.000989437,  0.000980854,  0.000968933,
     0.000954151,  0.000935555,  0.000915051,  0.000891685,
     0.000866413,  0.000838757,  0.000809669,  0.000779152,
     0.000747204,  0.000714302,  0.000680923,  0.000646591,
     0.000611782,  0.000576973,  0.000542164,  0.000507355,
     0.000472546,  0.000438213,  0.000404358,  0.000371456,
     0.000339031,  0.000307560,  0.000277042,  0.000247478,
     0.000218868,  0.000191212,  0.000165462,  0.000140190,
     0.000116348,  0.000093937,  0.000072956,  0.000052929,
     0.000034332,  0.000017166,  0.000000954, -0.000013828,
    -0.000027180, -0.000039577, -0.000050545, -0.000060558,
    -0.000069618, -0.000077724, -0.000084400, -0.000090122,
    -0.000095367, -0.000099182, -0.000102520, -0.000105381,
    -0.000106812, -0.000108242, -0.000108719, -0.000108719,
    -0.000108242, -0.000107288, -0.000105858, -0.000103951,
     0.000101566,  0.000099182,  0.000096321,  0.000093460,
     0.000090599,  0.000087261,  0.000083923,  0.000080585,
     0.000076771,  0.000073433,  0.000070095,  0.000066280,
     0.000062943,  0.000059605,  0.000055790,  0.000052929,
     0.000049591,  0.000046253,  0.000043392,  0.000040531,
     0.000037670,  0.000034809,  0.000032425,  0.000030041,
     0.000027657,  0.000025272,  0.000023365,  0.000021458,
     0.000019550,  0.000018120,  0.000016689,  0.000014782,
     0.000013828,  0.000012398,  0.000011444,  0.000010014,
     0.000009060,  0.000008106,  0.000007629,  0.000006676,
     0.000006199,  0.000005245,  0.000004768,  0.000004292,
     0.000003815,  0.000003338,  0.000003338,  0.000002861,
     0.000002384,  0.000002384,  0.000001907,  0.000001907,
     0.000001431,  0.000001431,  0.000000954,  0.000000954,
     0.000000954,  0.000000954,  0.000000477,  0.000000477,
     0.000000477,  0.000000477,  0.000000477,  0.000000477
};

const struct absthr_rec
absthr[6][134] =
{
    /*
     *  From ISO/IEC 11172-3 Table 3-D.1d.:  Frequencies, Critical Band Rates and Absolute Threshold
     *  Table is valid for Layer II at a sampling rate of 32 kHz
     */
    {
	{ 133, 32000.0 },

	{ 1, 58.23 },
	{ 2, 33.44 },
	{ 3, 24.17 },
	{ 4, 19.20 },
	{ 5, 16.05 },
	{ 6, 13.87 },
	{ 7, 12.26 },
	{ 8, 11.01 },
	{ 9, 10.01 },
	{ 10, 9.20 },
	{ 11, 8.52 },
	{ 12, 7.94 },
	{ 13, 7.44 },
	{ 14, 7.00 },
	{ 15, 6.62 },
	{ 16, 6.28 },
	{ 17, 5.97 },
	{ 18, 5.70 },
	{ 19, 5.44 },
	{ 20, 5.21 },
	{ 21, 5.00 },
	{ 22, 4.80 },
	{ 23, 4.62 },
	{ 24, 4.45 },
	{ 25, 4.29 },
	{ 26, 4.14 },
	{ 27, 4.00 },
	{ 28, 3.86 },
	{ 29, 3.73 },
	{ 30, 3.61 },
	{ 31, 3.49 },
	{ 32, 3.37 },
	{ 33, 3.26 },
	{ 34, 3.15 },
	{ 35, 3.04 },
	{ 36, 2.93 },
	{ 37, 2.83 },
	{ 38, 2.73 },
	{ 39, 2.63 },
	{ 40, 2.53 },
	{ 41, 2.42 },
	{ 42, 2.32 },
	{ 43, 2.22 },
	{ 44, 2.12 },
	{ 45, 2.02 },
	{ 46, 1.92 },
	{ 47, 1.81 },
	{ 48, 1.71 },
	{ 50, 1.49 },
	{ 52, 1.27 },
	{ 54, 1.04 },
	{ 56, 0.80 },
	{ 58, 0.55 },
	{ 60, 0.29 },
	{ 62, 0.02 },
	{ 64, -0.25 },
	{ 66, -0.54 },
	{ 68, -0.83 },
	{ 70, -1.12 },
	{ 72, -1.43 },
	{ 74, -1.73 },
	{ 76, -2.04 },
	{ 78, -2.34 },
	{ 80, -2.64 },
	{ 82, -2.93 },
	{ 84, -3.22 },
	{ 86, -3.49 },
	{ 88, -3.74 },
	{ 90, -3.98 },
	{ 92, -4.20 },
	{ 94, -4.40 },
	{ 96, -4.57 },
	{ 100, -4.82 },
	{ 104, -4.96 },
	{ 108, -4.97 },
	{ 112, -4.86 },
	{ 116, -4.63 },
	{ 120, -4.29 },
	{ 124, -3.87 },
	{ 128, -3.39 },
	{ 132, -2.86 },
	{ 136, -2.31 },
	{ 140, -1.77 },
	{ 144, -1.24 },
	{ 148, -0.74 },
	{ 152, -0.29 },
	{ 156, 0.12 },
	{ 160, 0.48 },
	{ 164, 0.79 },
	{ 168, 1.06 },
	{ 172, 1.29 },
	{ 176, 1.49 },
	{ 180, 1.66 },
	{ 184, 1.81 },
	{ 188, 1.95 },
	{ 192, 2.08 },
	{ 200, 2.33 },
	{ 208, 2.59 },
	{ 216, 2.86 },
	{ 224, 3.17 },
	{ 232, 3.51 },
	{ 240, 3.89 },
	{ 248, 4.31 },
	{ 256, 4.79 },
	{ 264, 5.31 },
	{ 272, 5.88 },
	{ 280, 6.50 },
	{ 288, 7.19 },
	{ 296, 7.93 },
	{ 304, 8.75 },
	{ 312, 9.63 },
	{ 320, 10.58 },
	{ 328, 11.60 },
	{ 336, 12.71 },
	{ 344, 13.90 },
	{ 352, 15.18 },
	{ 352, 16.54 },
	{ 368, 18.01 },
	{ 376, 19.57 },
	{ 384, 21.23 },
	{ 382, 23.01 },
	{ 400, 24.90 },
	{ 408, 26.90 },
	{ 416, 29.03 },
	{ 424, 31.28 },
	{ 432, 33.67 },
	{ 440, 36.19 },
	{ 448, 38.86 },
	{ 456, 41.67 },
	{ 464, 44.63 },
	{ 472, 47.76 },
	{ 480, 51.04 }
    },

    /*
     *  From ISO/IEC 11172-3 Table 3-D.1e.:  Frequencies, Critical Band Rates and Absolute Threshold
     *  Table is valid for Layer II at a sampling rate of 44.1 kHz
     */
    {
	{ 131, 44100.0 },

	{ 1, 45.05 },
	{ 2, 25.87 },
	{ 3, 18.70 },
	{ 4, 14.85 },
	{ 5, 12.41 },
	{ 6, 10.72 },
	{ 7, 9.47 },
	{ 8, 8.50 },
	{ 9, 7.73 },
	{ 10, 7.10 },
	{ 11, 6.56 },
	{ 12, 6.11 },
	{ 13, 5.72 },
	{ 14, 5.37 },
	{ 15, 5.07 },
	{ 16, 4.79 },
	{ 17, 4.55 },
	{ 18, 4.32 },
	{ 19, 4.11 },
	{ 20, 3.92 },
	{ 21, 3.74 },
	{ 22, 3.57 },
	{ 23, 3.40 },
	{ 24, 3.25 },
	{ 25, 3.10 },
	{ 26, 2.95 },
	{ 27, 2.81 },
	{ 28, 2.67 },
	{ 29, 2.53 },
	{ 30, 2.39 },
	{ 31, 2.25 },
	{ 32, 2.11 },
	{ 33, 1.97 },
	{ 34, 1.83 },
	{ 35, 1.68 },
	{ 36, 1.53 },
	{ 37, 1.38 },
	{ 38, 1.23 },
	{ 39, 1.07 },
	{ 40, 0.90 },
	{ 41, 0.74 },
	{ 42, 0.56 },
	{ 43, 0.39 },
	{ 44, 0.21 },
	{ 45, 0.02 },
	{ 46, -0.17 },
	{ 47, -0.36 },
	{ 48, -0.56 },
	{ 50, -0.96 },
	{ 52, -1.37 },
	{ 54, -1.79 },
	{ 56, -2.21 },
	{ 58, -2.63 },
	{ 60, -3.03 },
	{ 62, -3.41 },
	{ 64, -3.77 },
	{ 66, -4.09 },
	{ 68, -4.37 },
	{ 70, -4.60 },
	{ 72, -4.78 },
	{ 74, -4.91 },
	{ 76, -4.97 },
	{ 78, -4.98 },
	{ 80, -4.92 },
	{ 82, -4.81 },
	{ 84, -4.65 },
	{ 86, -4.43 },
	{ 88, -4.17 },
	{ 90, -3.87 },
	{ 92, -3.54 },
	{ 94, -3.19 },
	{ 96, -2.82 },
	{ 100, -2.06 },
	{ 104, -1.33 },
	{ 108, -0.64 },
	{ 112, -0.04 },
	{ 116, 0.47 },
	{ 120, 0.89 },
	{ 124, 1.23 },
	{ 128, 1.51 },
	{ 132, 1.74 },
	{ 136, 1.93 },
	{ 140, 2.11 },
	{ 144, 2.28 },
	{ 148, 2.45 },
	{ 152, 2.63 },
	{ 156, 2.82 },
	{ 160, 3.03 },
	{ 164, 3.25 },
	{ 168, 3.49 },
	{ 172, 3.74 },
	{ 176, 4.02 },
	{ 180, 4.32 },
	{ 184, 4.64 },
	{ 188, 4.98 },
	{ 192, 5.35 },
	{ 200, 6.15 },
	{ 208, 7.07 },
	{ 216, 8.10 },
	{ 224, 9.25 },
	{ 232, 10.54 },
	{ 240, 11.97 },
	{ 248, 13.56 },
	{ 256, 15.30 },
	{ 264, 17.23 },
	{ 272, 19.33 },
	{ 280, 21.64 },
	{ 288, 24.15 },
	{ 296, 26.88 },
	{ 304, 29.84 },
	{ 312, 33.04 },
	{ 320, 36.51 },
	{ 328, 40.24 },
	{ 336, 44.26 },
	{ 344, 48.58 },
	{ 352, 53.21 },
	{ 360, 58.17 },
	{ 368, 63.48 },
	{ 376, 69.13 },
	{ 384, 69.13 },
	{ 392, 69.13 },
	{ 400, 69.13 },
	{ 408, 69.13 },
	{ 416, 69.13 },
	{ 424, 69.13 },
	{ 432, 69.13 },
	{ 440, 69.13 },
	{ 448, 69.13 },
	{ 456, 69.13 },
	{ 464, 69.13 }
    },

    /*
     *  ISO/IEC 11172-3 Table 3-D.1f.:  Frequencies, Critical Band Rates and Absolute Threshold
     *  Table is valid for Layer II at a sampling rate of 48.0 kHz
     */
    {
	{ 127, 48000.0 },

	{ 1, 42.10 },
	{ 2, 24.17 },
	{ 3, 17.47 },
	{ 4, 13.87 },
	{ 5, 11.60 },
	{ 6, 10.01 },
	{ 7, 8.84 },
	{ 8, 7.94 },
	{ 9, 7.22 },
	{ 10, 6.62 },
	{ 11, 6.12 },
	{ 12, 5.70 },
	{ 13, 5.33 },
	{ 14, 5.00 },
	{ 15, 4.71 },
	{ 16, 4.45 },
	{ 17, 4.21 },
	{ 18, 4.00 },
	{ 19, 3.79 },
	{ 20, 3.61 },
	{ 21, 3.43 },
	{ 22, 3.26 },
	{ 23, 3.09 },
	{ 24, 2.93 },
	{ 25, 2.78 },
	{ 26, 2.63 },
	{ 27, 2.47 },
	{ 28, 2.32 },
	{ 29, 2.17 },
	{ 30, 2.02 },
	{ 31, 1.86 },
	{ 32, 1.71 },
	{ 33, 1.55 },
	{ 34, 1.38 },
	{ 35, 1.21 },
	{ 36, 1.04 },
	{ 37, 0.86 },
	{ 38, 0.67 },
	{ 39, 0.49 },
	{ 40, 0.29 },
	{ 41, 0.09 },
	{ 42, -0.11 },
	{ 43, -0.32 },
	{ 44, -0.54 },
	{ 45, -0.75 },
	{ 46, -0.97 },
	{ 47, -1.20 },
	{ 48, -1.43 },
	{ 50, -1.88 },
	{ 52, -2.34 },
	{ 54, -2.79 },
	{ 56, -3.22 },
	{ 58, -3.62 },
	{ 60, -3.98 },
	{ 62, -4.30 },
	{ 64, -4.57 },
	{ 66, -4.77 },
	{ 68, -4.91 },
	{ 70, -4.98 },
	{ 72, -4.97 },
	{ 74, -4.90 },
	{ 76, -4.76 },
	{ 78, -4.55 },
	{ 80, -4.29 },
	{ 82, -3.99 },
	{ 84, -3.64 },
	{ 86, -3.26 },
	{ 88, -2.86 },
	{ 90, -2.45 },
	{ 92, -2.04 },
	{ 94, -1.63 },
	{ 96, -1.24 },
	{ 100, -0.51 },
	{ 104, 0.12 },
	{ 108, 0.64 },
	{ 112, 1.06 },
	{ 116, 1.39 },
	{ 120, 1.66 },
	{ 124, 1.88 },
	{ 128, 2.08 },
	{ 132, 2.27 },
	{ 136, 2.46 },
	{ 140, 2.65 },
	{ 144, 2.86 },
	{ 148, 3.09 },
	{ 152, 3.33 },
	{ 156, 3.60 },
	{ 160, 3.89 },
	{ 164, 4.20 },
	{ 168, 4.54 },
	{ 172, 4.91 },
	{ 176, 5.31 },
	{ 180, 5.73 },
	{ 184, 6.18 },
	{ 188, 6.67 },
	{ 192, 7.19 },
	{ 200, 8.33 },
	{ 208, 9.63 },
	{ 216, 11.08 },
	{ 224, 12.71 },
	{ 232, 14.53 },
	{ 240, 16.54 },
	{ 248, 18.77 },
	{ 256, 21.23 },
	{ 264, 23.94 },
	{ 272, 26.90 },
	{ 280, 30.14 },
	{ 288, 33.67 },
	{ 296, 37.51 },
	{ 304, 41.67 },
	{ 312, 46.17 },
	{ 320, 51.04 },
	{ 328, 56.29 },
	{ 336, 61.94 },
	{ 344, 68.00 },
	{ 352, 68.00 },
	{ 360, 68.00 },
	{ 368, 68.00 },
	{ 376, 68.00 },
	{ 384, 68.00 },
	{ 392, 68.00 },
	{ 400, 68.00 },
	{ 408, 68.00 },
	{ 416, 68.00 },
	{ 424, 68.00 },
	{ 432, 68.00 }
    },

    /*
     *  From ISO/IEC 13818-3 Table D.1d. - Frequencies, critical band rates and absolute threshold
     *  Table is valid for Layer II at a sampling rate of 16 kHz.
     */
    {
	{ 133, 16000.0 },

	{ 1, 68.00 },
	{ 2, 58.23 },
	{ 3, 42.10 },
	{ 4, 33.44 },
	{ 5, 27.97 },
	{ 6, 24.17 },
	{ 7, 21.36 },
	{ 8, 19.20 },
	{ 9, 17.47 },
	{ 10, 16.05 },
	{ 11, 14.87 },
	{ 12, 13.87 },
	{ 13, 13.01 },
	{ 14, 12.26 },
	{ 15, 11.60 },
	{ 16, 11.01 },
	{ 17, 10.49 },
	{ 18, 10.01 },
	{ 19, 9.59 },
	{ 20, 9.20 },
	{ 21, 8.84 },
	{ 22, 8.52 },
	{ 23, 8.22 },
	{ 24, 7.94 },
	{ 25, 7.68 },
	{ 26, 7.44 },
	{ 27, 7.22 },
	{ 28, 7.00 },
	{ 29, 6.81 },
	{ 30, 6.62 },
	{ 31, 6.44 },
	{ 32, 6.28 },
	{ 33, 6.12 },
	{ 34, 5.97 },
	{ 35, 5.83 },
	{ 36, 5.70 },
	{ 37, 5.57 },
	{ 38, 5.44 },
	{ 39, 5.33 },
	{ 40, 5.21 },
	{ 41, 5.10 },
	{ 42, 5.00 },
	{ 43, 4.90 },
	{ 44, 4.80 },
	{ 45, 4.71 },
	{ 46, 4.62 },
	{ 47, 4.53 },
	{ 48, 4.45 },
	{ 50, 4.29 },
	{ 52, 4.14 },
	{ 54, 4.00 },
	{ 56, 3.86 },
	{ 58, 3.73 },
	{ 60, 3.61 },
	{ 62, 3.49 },
	{ 64, 3.37 },
	{ 66, 3.26 },
	{ 68, 3.15 },
	{ 70, 3.04 },
	{ 72, 2.93 },
	{ 74, 2.83 },
	{ 76, 2.73 },
	{ 78, 2.63 },
	{ 80, 2.53 },
	{ 82, 2.42 },
	{ 84, 2.32 },
	{ 86, 2.22 },
	{ 88, 2.12 },
	{ 90, 2.02 },
	{ 92, 1.92 },
	{ 94, 1.81 },
	{ 96, 1.71 },
	{ 100, 1.49 },
	{ 104, 1.27 },
	{ 108, 1.04 },
	{ 112, 0.80 },
	{ 116, 0.55 },
	{ 120, 0.29 },
	{ 124, 0.02 },
	{ 128, -0.25 },
	{ 132, -0.54 },
	{ 136, -0.83 },
	{ 140, -1.12 },
	{ 144, -1.43 },
	{ 148, -1.73 },
	{ 152, -2.04 },
	{ 156, -2.34 },
	{ 160, -2.64 },
	{ 164, -2.93 },
	{ 168, -3.22 },
	{ 172, -3.49 },
	{ 176, -3.74 },
	{ 180, -3.98 },
	{ 184, -4.20 },
	{ 188, -4.40 },
	{ 192, -4.57 },
	{ 200, -4.82 },
	{ 208, -4.96 },
	{ 216, -4.97 },
	{ 224, -4.86 },
	{ 232, -4.63 },
	{ 240, -4.29 },
	{ 248, -3.87 },
	{ 256, -3.39 },
	{ 264, -2.86 },
	{ 272, -2.31 },
	{ 280, -1.77 },
	{ 288, -1.24 },
	{ 296, -0.74 },
	{ 304, -0.29 },
	{ 312, 0.12 },
	{ 320, 0.48 },
	{ 328, 0.79 },
	{ 336, 1.06 },
	{ 344, 1.29 },
	{ 352, 1.49 },
	{ 360, 1.66 },
	{ 368, 1.81 },
	{ 376, 1.95 },
	{ 384, 2.08 },
	{ 392, 2.21 },
	{ 400, 2.33 },
	{ 408, 2.46 },
	{ 416, 2.59 },
	{ 424, 2.72 },
	{ 432, 2.86 },
	{ 440, 3.01 },
	{ 448, 3.17 },
	{ 456, 3.33 },
	{ 464, 3.51 },
	{ 472, 3.69 },
	{ 480, 3.89 }
    },

    /*
     *  From ISO/IEC 13818-3 Table D.1e. - Frequencies, critical band rates and absolute threshold
     *  Table is valid for Layer II at a sampling rate of 22.05 kHz.
     */
    {
 	{ 133, 22050.0 },

	{ 1, 68.00 },
	{ 2, 45.05 },
	{ 3, 32.57 },
	{ 4, 25.87 },
	{ 5, 21.63 },
	{ 6, 18.70 },
	{ 7, 16.52 },
	{ 8, 14.85 },
	{ 9, 13.51 },
	{ 10, 12.41 },
	{ 11, 11.50 },
	{ 12, 10.72 },
	{ 13, 10.05 },
	{ 14, 9.47 },
	{ 15, 8.96 },
	{ 16, 8.50 },
	{ 17, 8.10 },
	{ 18, 7.73 },
	{ 19, 7.40 },
	{ 20, 7.10 },
	{ 21, 6.82 },
	{ 22, 6.56 },
	{ 23, 6.33 },
	{ 24, 6.11 },
	{ 25, 5.91 },
	{ 26, 5.72 },
	{ 27, 5.54 },
	{ 28, 5.37 },
	{ 29, 5.22 },
	{ 30, 5.07 },
	{ 31, 4.93 },
	{ 32, 4.79 },
	{ 33, 4.67 },
	{ 34, 4.55 },
	{ 35, 4.43 },
	{ 36, 4.32 },
	{ 37, 4.21 },
	{ 38, 4.11 },
	{ 39, 4.01 },
	{ 40, 3.92 },
	{ 41, 3.83 },
	{ 42, 3.74 },
	{ 43, 3.65 },
	{ 44, 3.57 },
	{ 45, 3.48 },
	{ 46, 3.40 },
	{ 47, 3.33 },
	{ 48, 3.25 },
	{ 50, 3.10 },
	{ 52, 2.95 },
	{ 54, 2.81 },
	{ 56, 2.67 },
	{ 58, 2.53 },
	{ 60, 2.39 },
	{ 62, 2.25 },
	{ 64, 2.11 },
	{ 66, 1.97 },
	{ 68, 1.83 },
	{ 70, 1.68 },
	{ 72, 1.53 },
	{ 74, 1.38 },
	{ 76, 1.23 },
	{ 78, 1.07 },
	{ 80, 0.90 },
	{ 82, 0.74 },
	{ 84, 0.56 },
	{ 86, 0.39 },
	{ 88, 0.21 },
	{ 90, 0.02 },
	{ 92, -0.17 },
	{ 94, -0.36 },
	{ 96, -0.56 },
	{ 100, -0.96 },
	{ 104, -1.38 },
	{ 108, -1.79 },
	{ 112, -2.21 },
	{ 116, -2.63 },
	{ 120, -3.03 },
	{ 124, -3.41 },
	{ 128, -3.77 },
	{ 132, -4.09 },
	{ 136, -4.37 },
	{ 140, -4.60 },
	{ 144, -4.78 },
	{ 148, -4.91 },
	{ 152, -4.97 },
	{ 156, -4.98 },
	{ 160, -4.92 },
	{ 164, -4.81 },
	{ 168, -4.65 },
	{ 172, -4.43 },
	{ 176, -4.17 },
	{ 180, -3.87 },
	{ 184, -3.54 },
	{ 188, -3.19 },
	{ 192, -2.82 },
	{ 200, -2.06 },
	{ 208, -1.32 },
	{ 216, -0.64 },
	{ 224, -0.04 },
	{ 232, 0.47 },
	{ 240, 0.89 },
	{ 248, 1.23 },
	{ 256, 1.51 },
	{ 264, 1.74 },
	{ 272, 1.93 },
	{ 280, 2.11 },
	{ 288, 2.28 },
	{ 296, 2.46 },
	{ 304, 2.63 },
	{ 312, 2.82 },
	{ 320, 3.03 },
	{ 328, 3.25 },
	{ 336, 3.49 },
	{ 344, 3.74 },
	{ 352, 4.02 },
	{ 360, 4.32 },
	{ 368, 4.64 },
	{ 376, 4.98 },
	{ 384, 5.35 },
	{ 392, 5.74 },
	{ 400, 6.15 },
	{ 408, 6.60 },
	{ 416, 7.07 },
	{ 424, 7.57 },
	{ 432, 8.10 },
	{ 440, 8.66 },
	{ 448, 9.25 },
	{ 456, 9.88 },
	{ 464, 10.54 },
	{ 472, 11.24 },
	{ 480, 11.97 }
    },

    /*
     *  From ISO/IEC 13818-3 Table D.1f. - Frequencies, critical band rates and absolute threshold
     *  Table is valid for Layer II at a sampling rate of 24 kHz.
     */
    {
	{ 133, 24000.0 },

	{ 1, 68.00 },
	{ 2, 42.10 },
	{ 3, 30.43 },
	{ 4, 24.17 },
	{ 5, 20.22 },
	{ 6, 17.47 },
	{ 7, 15.44 },
	{ 8, 13.87 },
	{ 9, 12.62 },
	{ 10, 11.60 },
	{ 11, 10.74 },
	{ 12, 10.01 },
	{ 13, 9.39 },
	{ 14, 8.84 },
	{ 15, 8.37 },
	{ 16, 7.94 },
	{ 17, 7.56 },
	{ 18, 7.22 },
	{ 19, 6.90 },
	{ 20, 6.62 },
	{ 21, 6.36 },
	{ 22, 6.12 },
	{ 23, 5.90 },
	{ 24, 5.70 },
	{ 25, 5.50 },
	{ 26, 5.33 },
	{ 27, 5.16 },
	{ 28, 5.00 },
	{ 29, 4.85 },
	{ 30, 4.71 },
	{ 31, 4.58 },
	{ 32, 4.45 },
	{ 33, 4.33 },
	{ 34, 4.21 },
	{ 35, 4.10 },
	{ 36, 4.00 },
	{ 37, 3.89 },
	{ 38, 3.79 },
	{ 39, 3.70 },
	{ 40, 3.61 },
	{ 41, 3.51 },
	{ 42, 3.43 },
	{ 43, 3.34 },
	{ 44, 3.26 },
	{ 45, 3.17 },
	{ 46, 3.09 },
	{ 47, 3.01 },
	{ 48, 2.93 },
	{ 50, 2.78 },
	{ 52, 2.63 },
	{ 54, 2.47 },
	{ 56, 2.32 },
	{ 58, 2.17 },
	{ 60, 2.02 },
	{ 62, 1.86 },
	{ 64, 1.71 },
	{ 66, 1.55 },
	{ 68, 1.38 },
	{ 70, 1.21 },
	{ 72, 1.04 },
	{ 74, 0.86 },
	{ 76, 0.67 },
	{ 78, 0.49 },
	{ 80, 0.29 },
	{ 82, 0.09 },
	{ 84, -0.11 },
	{ 86, -0.32 },
	{ 88, -0.54 },
	{ 90, -0.75 },
	{ 92, -0.97 },
	{ 94, -1.20 },
	{ 96, -1.43 },
	{ 100, -1.88 },
	{ 104, -2.34 },
	{ 108, -2.79 },
	{ 112, -3.22 },
	{ 116, -3.62 },
	{ 120, -3.98 },
	{ 124, -4.30 },
	{ 128, -4.57 },
	{ 132, -4.77 },
	{ 136, -4.91 },
	{ 140, -4.98 },
	{ 144, -4.97 },
	{ 148, -4.90 },
	{ 152, -4.76 },
	{ 156, -4.55 },
	{ 160, -4.29 },
	{ 164, -3.99 },
	{ 168, -3.64 },
	{ 172, -3.26 },
	{ 176, -2.86 },
	{ 180, -2.45 },
	{ 184, -2.04 },
	{ 188, -1.63 },
	{ 192, -1.24 },
	{ 200, -0.51 },
	{ 208, 0.12 },
	{ 216, 0.64 },
	{ 224, 1.06 },
	{ 232, 1.39 },
	{ 240, 1.66 },
	{ 248, 1.88 },
	{ 256, 2.08 },
	{ 264, 2.27 },
	{ 272, 2.46 },
	{ 280, 2.65 },
	{ 288, 2.86 },
	{ 296, 3.09 },
	{ 304, 3.33 },
	{ 312, 3.60 },
	{ 320, 3.89 },
	{ 328, 4.20 },
	{ 336, 4.54 },
	{ 344, 4.91 },
	{ 352, 5.31 },
	{ 360, 5.73 },
	{ 368, 6.18 },
	{ 376, 6.67 },
	{ 384, 7.19 },
	{ 392, 7.74 },
	{ 400, 8.33 },
	{ 408, 8.96 },
	{ 416, 9.63 },
	{ 424, 10.33 },
	{ 432, 11.08 },
	{ 440, 11.87 },
	{ 448, 12.71 },
	{ 456, 13.59 },
	{ 464, 14.53 },
	{ 472, 15.51 },
	{ 480, 16.54 }
    }
};
