/*
    libfame - Fast Assembly MPEG Encoder Library
    Copyright (C) 2000-2001 Vivien Chappelier

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "fame.h"
#include "cpuflags.h"
#include "fame_profile_mpeg1.h"
#include "fame_profile_mpeg4_simple.h"
#include "fame_profile_mpeg4_shape.h"
#include "fame_syntax.h"
#include "fame_syntax_mpeg1.h"
#include "fame_syntax_mpeg4.h"
#include "fame_shape.h"
#include "fame_motion.h"
#include "fame_motion_none.h"
#include "fame_motion_fourstep.h"
#include "fame_motion_pmvfast.h"
#include "fame_encoder_mpeg.h"
#include "fame_decoder_mpeg.h"
#include "fame_rate.h"

/* version information */
const unsigned int libfame_major_version = LIBFAME_MAJOR_VERSION,
                   libfame_minor_version = LIBFAME_MINOR_VERSION,
                   libfame_micro_version = LIBFAME_MICRO_VERSION;
const char libfame_version[] = LIBFAME_VERSION;

struct _fame_private_t_ {
  /* built-in objects */
  fame_profile_mpeg1_t *profile_mpeg1;
  fame_profile_mpeg4_simple_t *profile_mpeg4_simple;
  fame_profile_mpeg4_shape_t *profile_mpeg4_shape;
  fame_encoder_mpeg_t *encoder_mpeg;
  fame_decoder_mpeg_t *decoder_mpeg;
  fame_motion_none_t *motion_none;
  fame_motion_pmvfast_t *motion_pmvfast;
  fame_motion_fourstep_t *motion_fourstep;
  fame_syntax_mpeg1_t *syntax_mpeg1;
  fame_syntax_mpeg4_t *syntax_mpeg4;
  fame_shape_t *shape;
  fame_rate_t *rate;
};

/*  fame_open                                                                */
/*                                                                           */
/*  Description:                                                             */
/*    Create a new context to use the library.                               */
/*                                                                           */
/*  Arguments:                                                               */
/*    None.                                                                  */
/*                                                                           */
/*  Return value:                                                            */
/*    fame_context_t * : a handle needed by all other library functions      */

fame_context_t * fame_open()
{
  fame_context_t *context;
  
  /* Initialize context */
  context = (fame_context_t *) malloc(sizeof(fame_context_t));
    
  /* Build built_in object list */
  context->type_list = NULL;
  context->priv = (struct _fame_private_t_ *) malloc(sizeof(struct _fame_private_t_));
  context->priv->profile_mpeg1 = FAME_NEW(fame_profile_mpeg1_t);
  context->priv->profile_mpeg4_simple = FAME_NEW(fame_profile_mpeg4_simple_t);
  context->priv->profile_mpeg4_shape = FAME_NEW(fame_profile_mpeg4_shape_t);
  context->priv->encoder_mpeg = FAME_NEW(fame_encoder_mpeg_t);
  context->priv->decoder_mpeg = FAME_NEW(fame_decoder_mpeg_t);
  context->priv->motion_none = FAME_NEW(fame_motion_none_t);
  context->priv->motion_pmvfast = FAME_NEW(fame_motion_pmvfast_t);
  context->priv->motion_fourstep = FAME_NEW(fame_motion_fourstep_t);
  context->priv->syntax_mpeg1 = FAME_NEW(fame_syntax_mpeg1_t);
  context->priv->syntax_mpeg4 = FAME_NEW(fame_syntax_mpeg4_t);
  context->priv->shape = FAME_NEW(fame_shape_t);
  context->priv->rate = FAME_NEW(fame_rate_t);

  /* built-in profiles */
  fame_register(context, "profile", FAME_OBJECT(context->priv->profile_mpeg1));
  fame_register(context, "profile/mpeg1", FAME_OBJECT(context->priv->profile_mpeg1));
  fame_register(context, "profile/mpeg4", FAME_OBJECT(context->priv->profile_mpeg4_simple));
  fame_register(context, "profile/mpeg4/simple", FAME_OBJECT(context->priv->profile_mpeg4_simple));
  fame_register(context, "profile/mpeg4/shape", FAME_OBJECT(context->priv->profile_mpeg4_shape));
  /* built-in encoders */
  fame_register(context, "encoder", FAME_OBJECT(context->priv->encoder_mpeg));
  fame_register(context, "encoder/mpeg", FAME_OBJECT(context->priv->encoder_mpeg));
  /* built-in decoders */
  fame_register(context, "decoder", FAME_OBJECT(context->priv->decoder_mpeg));
  fame_register(context, "decoder/mpeg", FAME_OBJECT(context->priv->decoder_mpeg));
  /* built-in motion estimators */
  fame_register(context, "motion", FAME_OBJECT(context->priv->motion_pmvfast));
  fame_register(context, "motion/none", FAME_OBJECT(context->priv->motion_none));
  fame_register(context, "motion/pmvfast", FAME_OBJECT(context->priv->motion_pmvfast));
  fame_register(context, "motion/fourstep", FAME_OBJECT(context->priv->motion_fourstep));
  /* built-in bitstream syntax writers */
  fame_register(context, "syntax", FAME_OBJECT(context->priv->syntax_mpeg1));
  fame_register(context, "syntax/mpeg1", FAME_OBJECT(context->priv->syntax_mpeg1));
  fame_register(context, "syntax/mpeg4", FAME_OBJECT(context->priv->syntax_mpeg4));
  /* built-in shape coders */
  fame_register(context, "shape", FAME_OBJECT(context->priv->shape));
  /* built-in rate controllers */
  fame_register(context, "rate", FAME_OBJECT(context->priv->rate));

  return(context);
}

/*  fame_register                                                            */
/*                                                                           */
/*  Description:                                                             */
/*    Register a type to the library and associate an object with it.        */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_context_t * context: the context handle returned by fame_open     */
/*    char const *type: the type to register                                 */
/*    fame_object_t * object: the object to be associated with this type     */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

void fame_register(fame_context_t *context,
		   char const *type,
		   fame_object_t *object)
{
  fame_list_t * next = context->type_list;

  if(fame_get_object(context, type))
    fame_unregister(context, type);
  context->type_list = (fame_list_t *) malloc(sizeof(fame_list_t));
  context->type_list->next = next;
  context->type_list->type = type;
  context->type_list->item = object;
}

/*  fame_unregister                                                          */
/*                                                                           */
/*  Description:                                                             */
/*    Remove a type from the library.                                        */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_context_t * context: the context handle returned by fame_open     */
/*    char const *type: the type to be unregistered                          */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

void fame_unregister(fame_context_t * context, char const *type)
{
  fame_list_t *list, *last;

  for(last = list = context->type_list; list; list = list->next) {
    if(!strcmp(list->type, type)) {
      if(last == list)
	context->type_list = list->next;
      else
	last->next = list->next;
      free(list);
      return;
    }
    last = list;
  }
}

/*  fame_get_object                                                          */
/*                                                                           */
/*  Description:                                                             */
/*    Retrieve the object associated to the given type from the library.     */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_context_t * context: the context handle returned by fame_open     */
/*    char const *type: the type of the object to be retrieved               */
/*                                                                           */
/*  Return value:                                                            */
/*    fame_object_t *: the object associated with type or NULL if not found. */

fame_object_t *fame_get_object(fame_context_t * context, char const *type)
{
  fame_list_t *list;

  for(list = context->type_list; list; list = list->next) {
    if(!strcmp(list->type, type))
      return(list->item);
  }
  return(NULL);
}

/*  fame_init                                                                */
/*                                                                           */
/*  Description:                                                             */
/*    Initialize the library                                                 */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_context_t * context: the context handle returned by fame_open     */
/*    fame_parameters_t *p: the parameters used to initialize (see fame.h)   */
/*    unsigned char * buffer: the output buffer                              */
/*    int size: the size of the output buffer                                */
/*                                                                           */
/*  Notes:                                                                   */
/*    The output buffer must be large enough to contain an encoded frame.    */
/*    There is no check to detect buffer overflow.                           */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

void fame_init(fame_context_t * context, 
	       fame_parameters_t *p,
	       unsigned char *buffer,
	       unsigned int size)
{
  unsigned long arch_flags;

  /* Print information message */
  if(p->verbose) {
    FAME_INFO("libfame %s Copyright (C) 2000-2001 Vivien Chappelier\n",
	      LIBFAME_VERSION);
    FAME_INFO("This library is provided under the terms of the LGPL. "
	      "See COPYING for details\n");
  }

  /* Choose profile */
  context->profile = fame_get_object(context, "profile");

  if(context->profile == NULL)
    FAME_FATAL("could not find 'profile'\n");

  if(p->verbose) {
    FAME_INFO("%s %dx%d @ %d/%d fps %s sequence %d%% quality\n",
	      context->profile->name,
	      p->width, p->height,
	      p->frame_rate_num,
	      p->frame_rate_den,
	      p->coding,
	      p->quality);
  }

  FAME_PROFILE(context->profile)->init(FAME_PROFILE(context->profile), context, p, buffer, size);

  arch_flags = cpuflags();

#if defined(HAS_MMX)
  if(arch_flags & X86_HAS_MMX) {
    if(p->verbose)
      FAME_INFO("Using MMX arithmetic\n");
  } else {
    FAME_FATAL("MMX not detected!\n"
	       "Consider recompiling without --enable-mmx in configure\n");
  }
#else
  if(p->verbose)
    FAME_INFO("Using floating point arithmetic\n");
#endif
}

/*  fame_encode_frame                                                        */
/*                                                                           */
/*  Description:                                                             */
/*    Encode a single frame.                                                 */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_context_t * context: the context handle returned by fame_open     */
/*    fame_yuv_t * yuv: the input frame in raw YUV format (YV12 planar)      */
/*    unsigned char * mask: the input mask (0 = transparent, 255 = opaque)   */
/*                                                                           */
/*  Return value:                                                            */
/*    int : the number of bytes written to buffer                            */

int fame_encode_frame(fame_context_t *context,
		      fame_yuv_t *yuv,
		      unsigned char *mask)
{
  return(FAME_PROFILE(context->profile)->encode(FAME_PROFILE(context->profile), yuv, mask));
}

/*  fame_close                                                               */
/*                                                                           */
/*  Description:                                                             */
/*    Flush remaining encoded data and cleanup everything.                   */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_context_t * context: the context handle returned by fame_open     */
/*                                                                           */
/*  Return value:                                                            */
/*    int : the number of bytes written to buffer                            */

int fame_close(fame_context_t *context)
{
  int bytes_written = 0;
  fame_list_t *l, *p;

  if(context->profile && FAME_PROFILE(context->profile)->close)
    bytes_written = FAME_PROFILE(context->profile)->close(FAME_PROFILE(context->profile));

  if(context->type_list != NULL) {
    l = context->type_list;

    while(l->next != NULL) {
      p = l;
      l = l->next;
      free(p);
    }
    free(l);
  }
  
  FAME_DELETE(context->priv->profile_mpeg1);
  FAME_DELETE(context->priv->profile_mpeg4_simple);
  FAME_DELETE(context->priv->profile_mpeg4_shape);
  FAME_DELETE(context->priv->encoder_mpeg);
  FAME_DELETE(context->priv->decoder_mpeg);
  FAME_DELETE(context->priv->motion_none);
  FAME_DELETE(context->priv->motion_pmvfast);
  FAME_DELETE(context->priv->motion_fourstep);
  FAME_DELETE(context->priv->syntax_mpeg1);
  FAME_DELETE(context->priv->syntax_mpeg4);
  FAME_DELETE(context->priv->shape);
  FAME_DELETE(context->priv->rate);
  free(context->priv);
 
  free(context);

  return(bytes_written);
}


