/*
    libfame - Fast Assembly MPEG Encoder Library
    Copyright (C) 2000-2001 Vivien Chappelier

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/**************************** half-pixel interpolation************************/

/*  half_interpolate                                                         */
/*                                                                           */
/*  Description:                                                             */
/*    Compute half-pel resolution frames from reference frame.               */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_encoder_t *encoder: the encoder                                   */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void inline half_interpolate(int width,
				    int height,
				    fame_yuv_t **ref,
				    int rounding)
{
  int x, y, w, h;
  int pitch;
  unsigned char *ref00, *ref01, *ref10, *ref11;

  /* Note: reference frame is allocated a little larger */
  /*       to allow overflow when doing interpolation   */
  /*       and thus avoid a special case for borders.   */
  /* Note: interpolation is done for Cr and Cb components   */
  /*       too, as it is used when computing the difference */
  /*       in motion compensation, even if it's not used    */
  /*       in motion estimation.                            */

  /* Y component */
  w = width;
  h = height;
  ref00 = ref[0]->y;
  ref01 = ref[1]->y;
  ref10 = ref[2]->y;
  ref11 = ref[3]->y;
  pitch = width;

  for(y = 0; y < h; y++) {
    for(x = 0; x < w; x++) {
      *ref01 = ((unsigned) ref00[0] +
		(unsigned) ref00[1] + 1 - rounding) >> 1;
      *ref10 = ((unsigned) ref00[0] +
		(unsigned) ref00[pitch] + 1 - rounding) >> 1;
      *ref11 = ((unsigned) ref00[0] +
		(unsigned) ref00[1] +
		(unsigned) ref00[pitch] +
		(unsigned) ref00[pitch+1] + 2 - rounding) >> 2;
      ref00++;
      ref01++;
      ref10++;
      ref11++;
    }
  }

  /* U component */
  w = width >> 1;
  h = height >> 1;
  ref00 = ref[0]->u;
  ref01 = ref[1]->u;
  ref10 = ref[2]->u;
  ref11 = ref[3]->u;
  pitch = width >> 1;

  for(y = 0; y < h; y++) {
    for(x = 0; x < w; x++) {
      *ref01 = ((unsigned) ref00[0] +
		(unsigned) ref00[1] + 1 - rounding) >> 1;
      *ref10 = ((unsigned) ref00[0] +
		(unsigned) ref00[pitch] + 1 - rounding) >> 1;
      *ref11 = ((unsigned) ref00[0] +
		(unsigned) ref00[1] +
		(unsigned) ref00[pitch] +
		(unsigned) ref00[pitch+1] + 2 - rounding) >> 2;
      ref00++;
      ref01++;
      ref10++;
      ref11++;
    }
  }

  /* V component */
  w = width >> 1;
  h = height >> 1;
  ref00 = ref[0]->v;
  ref01 = ref[1]->v;
  ref10 = ref[2]->v;
  ref11 = ref[3]->v;
  pitch = width >> 1;

  for(y = 0; y < h; y++) {
    for(x = 0; x < w; x++) {
      *ref01 = ((unsigned) ref00[0] +
		(unsigned) ref00[1] + 1 - rounding) >> 1;
      *ref10 = ((unsigned) ref00[0] +
		(unsigned) ref00[pitch] + 1 - rounding) >> 1;
      *ref11 = ((unsigned) ref00[0] +
		(unsigned) ref00[1] +
		(unsigned) ref00[pitch] +
		(unsigned) ref00[pitch+1] + 2 - rounding) >> 2;
      ref00++;
      ref01++;
      ref10++;
      ref11++;
    }
  }
}
