/*
    libfame - Fast Assembly MPEG Encoder Library
    Copyright (C) 2000-2001 Vivien Chappelier

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "fame.h"
#include "fame_profile.h"
#include "fame_profile_mpeg4_shape.h"
#include "fame_encoder.h"
#include "fame_decoder.h"
#include "fame_motion.h"
#include "fame_syntax.h"
#include "fame_shape.h"

static void profile_mpeg4_shape_init(fame_profile_t *profile,
				     fame_context_t *context,
				     fame_parameters_t *params,
				     unsigned char *buffer,
				     unsigned int size);

FAME_CONSTRUCTOR(fame_profile_mpeg4_shape_t)
{
  fame_profile_mpeg_t_constructor(FAME_PROFILE_MPEG(this));
  FAME_OBJECT(this)->name = "MPEG-4 shape profile";
  this->FAME_OVERLOADED(init) = FAME_PROFILE(this)->init;
  FAME_PROFILE(this)->init = profile_mpeg4_shape_init;
  return(this);
}

/*  profile_mpeg4_shape_init                                                 */
/*                                                                           */
/*  Description:                                                             */
/*    Initialize the profile.                                                */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_profile_t *profile: the profile to initialize                     */
/*    fame_parameters_t *params: the parameters for initialization           */
/*    unsigned char *buffer: the buffer to output data                       */
/*    unsigned int size: the size of the output buffer                       */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void profile_mpeg4_shape_init(fame_profile_t *profile,
				     fame_context_t *context,
				     fame_parameters_t *params,
				     unsigned char *buffer,
				     unsigned int size)
{
  fame_profile_mpeg_t *profile_mpeg = FAME_PROFILE_MPEG(profile);

  fame_register(context, "encoder", fame_get_object(context, "encoder/mpeg"));
  fame_register(context, "decoder", fame_get_object(context, "decoder/mpeg"));
  fame_register(context, "motion", fame_get_object(context, "motion"));
  fame_register(context, "syntax", fame_get_object(context, "syntax/mpeg4"));
  fame_register(context, "shape",  fame_get_object(context, "shape"));
  FAME_PROFILE_MPEG(profile)->motion_flags =
    FAME_MOTION_SUBPEL_SEARCH | 
    FAME_MOTION_UNRESTRICTED_SEARCH | 
    FAME_MOTION_BLOCK_SEARCH |
    FAME_MOTION_FLIP_ROUNDING;

  FAME_PROFILE_MPEG(profile)->shape_flags =
    (params->shape_quality == 100) ? FAME_SHAPE_LOSSLESS : 0;
  FAME_PROFILE_MPEG(profile)->syntax_flags =
    FAME_SYNTAX_ARBITRARY_SHAPE |
    ((params->shape_quality == 100) ? FAME_SYNTAX_LOSSLESS_SHAPE : 0);

  FAME_PROFILE_MPEG4_SHAPE(profile)->FAME_OVERLOADED(init)(profile, context, params, buffer, size);

  if(profile_mpeg->decoder == NULL)
    FAME_ERROR("Could not find decoder object");
  if(profile_mpeg->encoder == NULL)
    FAME_ERROR("Could not find encoder object");
  if(profile_mpeg->motion == NULL)
    FAME_ERROR("Could not find motion object");
  if(profile_mpeg->syntax == NULL)
    FAME_ERROR("Could not find syntax object");
  if(profile_mpeg->shape == NULL)
    FAME_ERROR("Could not find shape object");
}
