#!/usr/local/bin/perl -w
#
# routers.cgi : Version 1.0 
#
# Tested with RRDTool v1.0.7, v1.0.13, v1.0.24
#
# Create router monitoring pages 
#
# for all routers, we have
#     IP address(D), DNS name(D), Cisco name(S), SNMP community, 
#     description, MRTG conf
#     list of interfaces: description, RRD file(M), bandwidth(S)(M)
#
#  Now we use cookies to preserve the user's last selection of 
# router, graphs and scaling (when we get around to it).
# The cookies only specify the user's Default preferences.
#
# ASSUMPTIONS:
# 1. Each router is defined in a separate file
# 2. All .conf files are in the same directory
# 3. All .rrd files are to be placed in the same directory
# 4. The router's (short) name is the first word of the Title: line 
# 5. The router's bandwidth (if known) is set in the MaxBytes: line
# 6. You dont use the default [_] for MaxBytes
# 7. All your interfaces are defined as ifno:community@host
#    and not using OIDs
# 8. You want all values reported in bits/s, although they are stored 
#    in bytes/s
# 9. The icons for this CGI are kept in the URL $iconurl

# Hopefully, the next version will make less assumptions.  Most of these
# assuptions will hold for most people who used the config_maker scripts

use strict;
use CGI qw/:standard *ul/;

my ($VERSION ) = "1.0";
my ($APPURL  ) = "http://ukpavu99.adsweu.com/pub/";

##############################################################################
# You MUST set this to the location of the configuration file!
#
my ($conffile) = "/usr/local/etc/webdev.conf";
##############################################################################

my ($mtype, $gtype, $defgtype, $defrouter, $defif, $q, $cookie, @cookies);
my (%routers, %interfaces, %gtypes, @gorder, $querystring, $statistic);
my (%headeropts, @cfgfiles);
my ($pagetype);
my (%config) = ();

########################################################################
# You MAY configure the lines below if you require
########################################################################

%gtypes = ( d=>"Daily",w=>"Weekly", m=>"Monthly",y=>"Yearly", 
	dwmy=>"Full", dwmys=>"Summary" );
@gorder = qw/d w m y dwmy dwmys/;

########################################################################
# Nothing else to configure after this line
########################################################################

# initialize CGI
$q = new CGI;
$q->import_names('CGI');

my $meurl = $q->url();

#################################
# For sorting

sub numerically { $a <=> $b; }
sub bydesc { 
	my ( $da, $db ) = ($config{"routerdesc-$a"}, $config{"routerdesc-$b"});
	$da = $a if ( ! $da );
	$db = $b if ( ! $db );
	$da cmp $db; 
}

#################################
# Read in confgiuration file

# readconf: pass it a list of section names
sub readconf(@)
{
	my ($inlist, $i, @secs, $sec);
	
	@secs = @_;
	%config = ();

	# set defaults
	%config = (
		'routers.cgi-confpath' => ".",
		'routers.cgi-cfgfiles' => "*.conf"
	);

	( open CFH, "<".$conffile ) || do {
		print $q->header({-expires=>"now"});	
		print $q->start_html({ -title => "Error" });	
		print "Cannot read config file $conffile.";
		print $q->end_html();
		exit(1);
	};

	$inlist=0;
	$sec = "";
	while( <CFH> ) {
		/^\s*#/ && next;
		/\[(\S*)\]/ && do { 
			$sec = $1;
			$inlist=0;	
			foreach $i ( @secs ) {
				if ( $i eq $1 ) { $inlist=1; last; }
			}
			next ;
		};
		if ( $inlist ) { /(\w+)\s*=\s*(\S.*)$/ and $config{"$sec-$1"}=$2; }
	}
	
	# get list of configuration files
	@cfgfiles = glob("$config{'routers.cgi-confpath'}/$config{'routers.cgi-cfgfiles'}");
}

#################################
# Read in files

my $router = "";

# build routers hash, set default router.
sub read_routers()
{
  my ($fname, $router);
# for all .conf files, /^Title[\B+]: (\w+)\W/i
#    key=$1 
  foreach  $fname ( @cfgfiles ) {
	open( CFG, $fname ) or next;
	$router = ""; # just starting
	while( <CFG> ) {
		if( /^Title\[\S+\]:\s+(\w+)\W/i ) { 	
			$router=$1;
			$routers{$router}=$fname ; 
			if( $config{"routerdesc-$router"} ) { last; }
			$config{"routerdesc-$router"}=$1; # set default description
			next; 
		}
		if( /^ShortDesc\[\S+\]:\s+(.+)$/i and $router ) {
			$config{"routerdesc-$router"}=$1;
			next;
		}
	}
# really, we should populate the interfaces hash here, too.
	close( CFG );
  }
# By this point, we have populated the %routers hash
}

# build interfaces hash for current router $router
my $interface = "";

sub read_cfg()
{ #7
#    Read in all the lines...
#                          /^Target[(\S+)]:\s+([0-9]+):(\w+)@(\S+)/i
#    rrd=lc($1).rrd, ifno=$2, community=$3, address=$4
#                          /^Title[(\S+)]:\s+((\S+).*)$/i
#    rrd=lc($1).rrd, desc=$2, shdesc=$3
#                          /^MaxBytes[(\S+)]:\s+(\w+)/i
#    max=$2
#                          /^ShortDesc[(\S+)]:\s+(.*)$/i
#    rrd=lc($1).rrd, shdesc=$2
# 
my($fname, $curif);

%interfaces = (); # blank it out ready
$fname = $routers{$router};
unless( open( CFG, $fname ) ) { return; }
# here we have to do the necessary
$curif="";
while ( <CFG> ) {
	if( /^Target\[(\S+)\]:\s+([0-9]+):(\w+)@(\S+)/i ) {
#    rrd=lc($1).rrd, ifno=$2, community=$3, address=$4
		$interfaces{$2} = { ifno=>"$2", rrd=>lc($1).".rrd",
			community=>$3, address=>$4 };
#		$interface = $2 if ( ! $interface ); # set default
		$curif = $2;
		next;
	}
	if( /^Title\[\S+\]:\s+((\S+).*)$/i and $curif ) {
		$interfaces{$curif}{desc}=$1;
		$interfaces{$curif}{shdesc}=$2; # first default
		if( $1 =~ /.*:\s*(\w+)/ ) {
			$interfaces{$curif}{shdesc}=$1; # second default
		}
		next;
	}
	if( /^MaxBytes\[\S+\]:\s+(\S+)/i and $curif ) {
		$interfaces{$curif}{max} = $1 * 8; # convert to bits
		next;
	}
	$curif="" if( /Target/ );
}
close ( CFG );
# Now, read the corresponding .ok file, if available.
# this contains lines of the format ^([0-9]+):.+=\s*(\S.*)$
# where $1 = interface and $2 = interface shortrdesc

$fname =~ s/.\w+$/.ok/;
unless( open( CFG, $fname ) ) { return; }
while ( <CFG> ) {
		/^([0-9]+):.+=\s*(\S[^:]*)/ ;
		$interfaces{$1}{shdesc}=$2 if( $interfaces{$1}{rrd} ); # preferred
}
close ( CFG );
}

#################################
# Top menu

sub do_head()
{
print $q->start_html({ -title => "Router Statistics" ,
					   -bgcolor => "d0d0d0"});

print $q->table( { -border=>"0", -width=>"100%" },
  $q->Tr( { -valign=>"TOP", -width=>"100%" }, 
    $q->td({ -align=>"LEFT", -width=>150, -valign=>"TOP" }, 
      $q->a({ href=>$config{'web-backurl'}, target=>"_top"}, 
      $q->img({ src=>"${config{'routers.cgi-iconpath'}}mainmenu-red.gif", alt=>"Main Menu", border=>0,
		width=>100, height=>20 }))."\n"
	.$q->br.$q->a({href=>"javascript:parent.graph.location.reload(true)"},
		$q->img({src=>"${config{'routers.cgi-iconpath'}}refresh.gif",alt=>"Refresh", border=>"0",
		width=>100, height=>20}))
    )."\n"
    .$q->td({ -align=>"CENTER" }, $q->h1("Router Statistics"))."\n"
    .$q->td({ -align=>"RIGHT", -width=>200, -valign=>"TOP" }, 
		$q->img({ src=>"${config{'routers.cgi-iconpath'}}rrdtool.gif", alt=>"RRDTool"}))."\n"
  )
),"\n";

# Finish off the page
print $q->end_html();
}

###########################
# Side menu

# $mtype specified 'routers' (list routers) or 'options' (list options)


sub do_menu()
{
my ($options,$iflabel);
my ($rtrdesc);

$options = "&xgtype=$gtype&xmtype=$mtype";

# Start it off
print $q->start_html({ -title => "Router Statistics",
					   -bgcolor => "d0d0d0", nowrap => "yes"});

print "<DIV NOWRAP>\n";

# Main stuff and links
if ( $mtype eq "options" ) {
	print $q->a({ href=>"$meurl?page=menu&xgtype=$gtype&xmtype=routers&rtr=$router&if=$interface", target=>"_self",
		onMouseOver=>"{images[0].src='${config{'routers.cgi-iconpath'}}routers-dn-w.gif'; window.status='Show list of routers';}", 
		onmouseout=>"{images[0].src='${config{'routers.cgi-iconpath'}}routers-dn.gif'; window.status='';}" },
		$q->img({ src=>"${config{'routers.cgi-iconpath'}}routers-dn.gif", alt=>"Routers", border=>0 ,
		width=>100, height=>20})),"\n";
	# list options
	print $q->br, $q->img({ src=>"${config{'routers.cgi-iconpath'}}interfaces.gif", alt=>"Interfaces",
		width=>100, height=>20 }), $q->br,"\n";
	foreach ( sort numerically keys( %interfaces ) ) {
		$iflabel = $interfaces{$_}{shdesc};
		$iflabel = "#$_" unless ( $iflabel );
		print $q->img({ src=>"${config{'routers.cgi-iconpath'}}interface-sm.gif",
			width=>15, height=>15  }),"&nbsp;";
		if ( $interface eq $_ ) {
			print $q->b($iflabel),$q->br,"\n";
		} else {
			print $q->a({ href=>"$meurl?page=graph&rtr=$router&xmtype=$mtype&xgtype=$gtype&if=$_", target=>"graph" }, $iflabel ),$q->br,"\n";
		}
	}
	if( ($interface ne "none") and ($router ne "none") ) {
		print $q->img({ src=>"${config{'routers.cgi-iconpath'}}menu-sm.gif",
			width=>15, height=>15  }),"&nbsp;";
		if( $interface eq "summary" ) {
			print $q->b("Summary"),$q->br,"\n";
		} else {
			print $q->a({ href=>"$meurl?page=graph&rtr=$router&xmtype=$mtype&xgtype=$gtype&if=summary", target=>"graph" },"Summary"),$q->br,"\n";	
		}
		print $q->img({ src=>"${config{'routers.cgi-iconpath'}}menu-sm.gif",
			width=>15, height=>15  }),"&nbsp;";
		if( $interface eq "info" ) {
			print $q->b("Info"),$q->br,"\n";
		} else {
			print $q->a({ href=>"$meurl?page=graph&rtr=$router&xmtype=$mtype&xgtype=$gtype&if=info", target=>"graph" },"Info"),$q->br,"\n";	
		}
	}

	print "\n";
	print $q->br, $q->img({ src=>"${config{'routers.cgi-iconpath'}}graphs.gif", alt=>"Graphs",
		width=>100, height=>20 }), $q->br,"\n";
	foreach ( @gorder ) {
		print $q->img({ src=>"${config{'routers.cgi-iconpath'}}graph-sm.gif",
			width=>15, height=>15  }),"&nbsp;";
		if($gtype eq $_) {
			print $q->b($gtypes{$_}),$q->br,"\n";
		} elsif(( $interface eq "summary" and (length > 1 ))
			or ( $router eq "none" )) {
			print $gtypes{$_},$q->br,"\n";
		} else {
			print $q->a({ href=>"$meurl?page=graph&rtr=$router&xmtype=$mtype&xgtype=$_&if=$interface", target=>"graph" }, $gtypes{$_} ),$q->br,"\n";
		}
	}
	print $q->br, $q->img({ src=>"${config{'routers.cgi-iconpath'}}otherstuff.gif", alt=>"Other Stuff",
		width=>100, height=>20 }), $q->br,"\n";
	print $q->img({ src=>"${config{'routers.cgi-iconpath'}}user-sm.gif",
			width=>15, height=>15  }),"&nbsp;";
	print $q->a({href=>"$meurl?page=config&rtr=$router&xmtype=$mtype&xgtype=$gtype", target=>"graph"}, 
		"Preferences"),$q->br,"\n";
	print $q->img({ src=>"${config{'routers.cgi-iconpath'}}query.gif" }),"&nbsp;";
	print $q->a({href=>"$meurl?page=help&rtr=$router&xmtype=$mtype&xgtype=$gtype", target=>"graph"}, 
		"Information"),$q->br,"\n";
} else {
	print $q->a({ href=>"$meurl?page=menu&xgtype=$gtype&xmtype=options&rtr=$router&if=$interface", target=>"_self" ,
		onMouseOver=>"{images[0].src='${config{'routers.cgi-iconpath'}}options-dn-w.gif'; window.status='Show display options';}", 
		onmouseout=>"{images[0].src='${config{'routers.cgi-iconpath'}}options-dn.gif'; window.status='';}" },
		$q->img({ src=>"${config{'routers.cgi-iconpath'}}options-dn.gif", alt=>"Options", border=>0,
			height=>20, width=>100 })),"\n";
	# list routers
	print $q->br, $q->img({ src=>"${config{'routers.cgi-iconpath'}}routers.gif", alt=>"Routers",
			height=>20, width=>100 }), $q->br,"\n";
	foreach ( sort bydesc keys(%routers) ) {
		$rtrdesc = $config{"routerdesc-$_"};
		$rtrdesc = $_ if(! $rtrdesc );
		$rtrdesc =~ s/ /\&nbsp\;/g; # stop breaking of line on spaces

		print $q->img({ src=>"${config{'routers.cgi-iconpath'}}router-sm.gif",
			width=>15, height=>15  }),"&nbsp;";
		if( $_ eq $router ) {
			print $q->b($rtrdesc),$q->br,"\n";
		} else {
			print $q->a({ href=>"$meurl?page=graph&rtr=$_$options", 	
				target=>"graph" },$rtrdesc ),$q->br,"\n";
		}
	}
}

# Finish off the page
print "</DIV>",$q->end_html();
}

############################
# Main frame set

sub do_main()
{
	print <<"EOT"
<HTML><HEAD><TITLE>Router Monitor</TITLE></HEAD>
<FRAMESET border=0 marginheight=1 marginwidth=1 rows=60,*>
 <FRAME name=head src=$meurl?page=head noresize scrolling=no>
 <FRAMESET border=0 marginheight=1 marginwidth=1 cols=150,*>
  <FRAME name=menu src=$meurl?page=menu&rtr=$router&if=$interface&xgtype=$gtype&xmtype=$mtype scrolling=auto nowrap>
  <FRAME name=graph src=$meurl?page=graph&rtr=$router&if=$interface&xgtype=$gtype&xmtype=$mtype noresize scrolling=auto>
 </FRAMESET>
</FRAMESET>
<NOFRAMES>
 <BODY>
  Sorry, I'm too lazy to support non-frames browsers.  Upgrade to Netscape4.
 </BODY>
</NOFRAMES>
</HTML>
EOT
}

##########################
# Graph panel

my(@params);

sub rtr_params(@)
{
	my($ds0,$ds1,$mds0,$mds1);
	my($lin, $lout, $mlin, $mlout);
	my($dwmy,$interface) = @_;
	my($sin, $sout);
	my($l);

	$sin="In\\: %6.2lf %s"; $sout="Out\\:%6.2lf %s";
	
	if ( $dwmy =~ /s/ ) {
		$lin=""; $lout="";
		$mlin=""; $mlout="";
	} else {
		$lin = ":Inbound traffic"; $lout = ":Outbound traffic\\l";
		$mlin = ":Peak Inbound"; $mlout = ":Peak Outbound\\l";
	}
	$lin = substr( $lin."                    ",0,30 ) if ($lin);
	$mlin = substr( $mlin."                    ",0,30 ) if ($mlin);

	($ds0, $ds1) = ("in", "out");

	push @params,
		"DEF:in=$config{'routers.cgi-dbpath'}/$interfaces{$interface}{rrd}:ds0:AVERAGE",
		"DEF:out=$config{'routers.cgi-dbpath'}/$interfaces{$interface}{rrd}:ds1:AVERAGE",
		"CDEF:fin=in,8,*", "CDEF:fout=out,8,*";
	($ds0, $ds1) = ("fin", "fout");
#	now for the peaks stuff
#	if ( $dwmy !~ /d/ ) {
		($mds0, $mds1) = ("min", "mout");
		push @params,
			"DEF:min=$config{'routers.cgi-dbpath'}/$interfaces{$interface}{rrd}:ds0:MAX",
			"DEF:mout=$config{'routers.cgi-dbpath'}/$interfaces{$interface}{rrd}:ds1:MAX",
			"CDEF:fmin=min,8,*", "CDEF:fmout=mout,8,*";
		($mds0, $mds1) = ("fmin", "fmout");
#	}
	if( $interfaces{$interface}{max} && $dwmy !~ /s/ ) {
		push @params,
			"CDEF:pcin=$ds0,100,*,$interfaces{$interface}{max},/",
			"CDEF:pcout=$ds1,100,*,$interfaces{$interface}{max},/",
			"CDEF:mpcin=$mds0,100,*,$interfaces{$interface}{max},/",
			"CDEF:mpcout=$mds1,100,*,$interfaces{$interface}{max},/";
	}

#	now for the actual lines
	if( $dwmy !~ /d/ ) {
		push @params,
			"LINE1:$mds0#006600$mlin",
			"LINE1:$mds1#ff00ff$mlout";
	}
	push @params,
		"AREA:$ds0#00cc00$lin",
		"LINE1:$ds1#0000ff$lout";

# data unavailable
	push @params,
		"CDEF:down=in,UN,INF,0,IF","AREA:down#d0d0d0";
# the max line
	if( $interfaces{$interface}{max} ) {
		my( $lmax ) = "";
		if( $interfaces{$interface}{max} and $dwmy !~ /s/ ) {
			if( $interfaces{$interface}{max} > 1000000 ) {
			$lmax =":100% Bandwidth (".($interfaces{$interface}{max}/1000000)."Mbps)\\l";
			} else {
			$lmax =":100% Bandwidth (".($interfaces{$interface}{max}/1000)."kbps)\\l";
			}
		}
		push @params, "HRULE:$interfaces{$interface}{max}#ff0000$lmax"		
	}
#	now for the labels at the bottom
	if( $dwmy !~ /s/ ) {
		if( $interfaces{$interface}{max} ) {
			push @params,
		"GPRINT:$mds1:MAX:Max $sout", "GPRINT:mpcout:MAX:(%2.0lf%%)",
		"GPRINT:$ds1:AVERAGE:Avg $sout", "GPRINT:pcout:AVERAGE:(%2.0lf%%)",
		"GPRINT:$ds1:LAST:Current $sout", "GPRINT:pcout:LAST:(%2.0lf%%)\\l",
		"GPRINT:$mds0:MAX:Max $sin", "GPRINT:mpcin:MAX:(%2.0lf%%)",
		"GPRINT:$ds0:AVERAGE:Avg $sin", "GPRINT:pcin:AVERAGE:(%2.0lf%%)",
		"GPRINT:$ds0:LAST:Current $sin" , "GPRINT:pcin:LAST:(%2.0lf%%)\\l" ;
		} else {
			push @params,
			"GPRINT:$mds1:MAX:Max $sout",
			"GPRINT:$ds1:AVERAGE:Avg $sout",
			"GPRINT:$ds1:LAST:Current $sout\\l",
			"GPRINT:$mds0:MAX:Max $sin",
			"GPRINT:$ds0:AVERAGE:Avg $sin",
			"GPRINT:$ds0:LAST:Current $sin\\l" ;
		}
	}
}

sub make_graph(@)
{
	my ($e, $thisgraph, $thisurl);
	my ($tstr, $width, $gtitle);
	
	my($dwmy,$interface) = @_;

	$tstr = "Daily" if( $dwmy =~ /d/ ) ;
	$tstr = "Weekly" if( $dwmy =~ /w/ ) ;
	$tstr = "Monthly" if( $dwmy =~ /m/ ) ;
	$tstr = "Yearly" if( $dwmy =~ /y/ ) ;

	$gtitle = $interfaces{$interface}{desc};
	$gtitle = $router.":".$interface 
		if ( ! $gtitle or ((length($gtitle)>25) and ($dwmy =~ /s/)));
	$gtitle = $gtitle." ".$tstr;

	@params = ();
	$thisurl   = "$config{'routers.cgi-graphurl'}/${router}.${interface}-$dwmy.gif";
	$thisgraph = "$config{'routers.cgi-graphpath'}/${router}.${interface}-$dwmy.gif";
	if ( $dwmy =~ /s/ ) { $width = 200 } else { $width = 400 }
	push @params, $thisgraph;
	push @params, qw/--lazy -l 0 --interlaced/;
	push @params, "-u", $interfaces{$interface}{max} 
		if( $interfaces{$interface}{max} );
# could have added a "-r" there to enforce the upper limit rigidly
#	push @params, "-v", $config{$statistic}{-v};
	push @params, "-w", $width;
#	push @params, "-s", (-5 * $width)."m"  if ( $dwmy =~ /d/ );
	push @params, "-s", (-25 * $width)."m" if ( $dwmy =~ /w/ );
	push @params, "-s", (-2 * $width)."h"  if ( $dwmy =~ /m/ );
	push @params, "-s", (-1 * $width)."d"   if ( $dwmy =~ /y/ );
	push @params,"--x-grid","HOUR:1:HOUR:24:HOUR:2:0:%k"  if ( $dwmy =~ /d/ );
	push @params,"--x-grid","HOUR:6:DAY:1:DAY:1:86400:%a" if ( $dwmy =~ /w/ );
	push @params,"--title", $gtitle;

	rtr_params($dwmy,$interface);

	RRDs::graph(@params);
	$e = RRDs::error();
	if ( $e ) {
		print $q->p("RRD Error: $e"),"\n";
		print $q->p("Router: $router"),$q->p("Interface: $interface"),
			$q->ul($q->li(keys(%interfaces)));
	} else {
		print $q->img({ src => $thisurl, alt => $gtitle, border => 0});
	}
}

# This is for the summary of interfaces view
sub do_summary()
{
# Start off.  We use onload() and Javascript to force reload the 
# lefthand (menu) panel.
my ($javascript, $e);
my ($thisgraph, $thisurl, $options);
my ($rrd, $curif, $lastupdate);

$javascript = "function LoadMenu() 
{ 
parent.menu.location = \"$meurl?page=menu&rtr=$router&if=$interface&xmtype=$mtype&xgtype=$gtype\"; 
}";

print $q->start_html({ -title => "Router Monitor", 
	-expires => "+5s",  -script => $javascript,
	-onload => "LoadMenu()" }),"\n";
#
# Now for the RRD stuff
require 'RRDs.pm';

$options="rtr=$router&xgtype=$gtype&xmtype=$mtype";

print "<TABLE border=0 width=100% align=center>\n";

foreach $curif ( sort numerically keys(%interfaces) ) {
	print "<TR WIDTH=100% VALIGN=TOP>";

	print "<TD VALIGN=TOP>";

	$rrd = "$config{'routers.cgi-dbpath'}/$interfaces{$curif}{rrd}";
	# Last update stuff.
	$lastupdate = RRDs::last($rrd);
	$e = RRDs::error();
	if( $e ) {
		print $q->p($q->b("Error reading RRD database $rrd"),$q->br,$e),"\n";
		print "</TD><TD>\n";
		print $q->b($interfaces{$curif}{desc}),$q->br;
		print $q->dump;
	} else {
		make_graph($gtype."s",$curif);
		print "</TD><TD>\n";
		print $q->a({href=>"$meurl?page=graph&if=$curif&$options", 
			target=>"graph"}, $q->b($interfaces{$curif}{desc}));
		print $q->br,$q->br,"Last update: ".scalar(localtime($lastupdate)),
			$q->br,"\n";
		print "Bandwidth: ".($interfaces{$curif}{max}/1000)."k bits/s",
			$q->br,"\n"
			if ( $interfaces{$curif}{max} );
		print "Address: ".$interfaces{$curif}{address},$q->br,"\n"
			if ( $interfaces{$curif}{address} );
		print "Interface #: ".$curif,$q->br,"\n";
	}
	print "</TD></TR>\n";
} # foreach
print "</TABLE>\n";
print $q->hr;
print "\n",$q->a({href=>"javascript:location.reload(true)"},
	$q->img({src=>"${config{'routers.cgi-iconpath'}}refresh.gif",alt=>"Refresh",border=>"0"})),$q->br,"\n";
print $q->hr,$q->small("routers.cgi Version $VERSION : Steve Shipway 2000 : ".$q->a({ href=>$APPURL, target=>"_top" },$APPURL)),"\n";
}

sub do_graph()
{
# Start off.  We use onload() and Javascript to force reload the 
# lefthand (menu) panel.
my ($javascript, $e);
my ($thisgraph, $thisurl);
my ($rrd, $curif);

$javascript = "function LoadMenu() 
{ 
parent.menu.location = \"$meurl?page=menu&rtr=$router&if=$interface&xmtype=$mtype&xgtype=$gtype\"; 
}";

print $q->start_html({ -title => "Router Monitor", 
	-expires => "+5s",  -script => $javascript,
	-onload => "LoadMenu()" }),"\n";
#
# Test stuff that is commented out as it all works perfectly
#print $q->dump;
#
# Now for the RRD stuff
require 'RRDs.pm';
# Now, we have to do this differently depending on which gtype we have
# We do a switch for the different graphs.
# We have to call RRD to create them, and the IMG tag is created ready to
# stuff into the page!  Oooh!
if ( $interface eq "summary" ) {
	$curif = (keys(%interfaces))[0];
	$rrd = "$config{'routers.cgi-dbpath'}/$interfaces{$curif}{rrd}";
} else {
	$rrd = "$config{'routers.cgi-dbpath'}/$interfaces{$interface}{rrd}";
}

# Last update stuff.
my $lastupdate = RRDs::last($rrd);
$e = RRDs::error();
if( $e ) {
	print $q->p($q->b("Error reading RRD database $rrd"),$q->br,$e),"\n";
	print $q->dump;
	print $q->p("Router: $router"),$q->p("Interface: $interface"),
		$q->ul($q->li(keys(%interfaces)));
} else {
	my $suffix = ( $gtype =~ /s/ ) ? "s" : "";
	make_graph("d$suffix",$interface) if ( $gtype =~ /d/ );
	make_graph("w$suffix",$interface) if ( $gtype =~ /w/ );
	print $q->br,"\n" if ( length($gtype) > 2 );
	make_graph("m$suffix",$interface) if ( $gtype =~ /m/ );
	make_graph("y$suffix",$interface) if ( $gtype =~ /y/ );
	print $q->br,"\n" if ( length($gtype) > 2 );

	print $q->p("Last update: ".scalar(localtime($lastupdate))),"\n";
	print $q->hr;
	print "\n",$q->a({href=>"javascript:location.reload(true)"},
		$q->img({src=>"${config{'routers.cgi-iconpath'}}refresh.gif",alt=>"Refresh",border=>"0"})),$q->br,"\n";
print $q->hr,$q->small("routers.cgi Version $VERSION : Steve Shipway 2000 : ".$q->a({ href=>$APPURL, target=>"_top" },$APPURL)),"\n";
}

# Finish off the page (this does the ending body and html tags)
print $q->end_html();
}

# Information on this router
sub do_info()
{
# Start off.  We use onload() and Javascript to force reload the 
# lefthand (menu) panel.
my ($javascript);

$javascript = "function LoadMenu() 
{ 
parent.menu.location = \"$meurl?page=menu&rtr=$router&if=info&xmtype=$mtype&xgtype=$gtype\"; 
}";

print $q->start_html({ -title => "Router Monitor", 
	-expires => "+5s",  -script => $javascript,
	-onload => "LoadMenu()" }),"\n";

# Here we build up a page of info, with lotsalinks.
print $q->h2("Router Information"),"\n";

print $q->a({href=>"$meurl?page=graph&rtr=$router&if=summary&xgtype=$gtype&xmtype=$mtype",
	target=>"graph"},
	$q->b("$router: ".$config{"routerdesc-$router"})),$q->br,"\n";
print "<UL>";
foreach (sort numerically keys (%interfaces)) {
	print $q->li( $q->a( 
		{href=>"$meurl?page=graph&rtr=$router&if=$_&xgtype=$gtype&xmtype=$mtype",
		target=>"graph"},
		"$_: $interfaces{$_}{desc}"
	)),"\n";
}
print "</UL>\n";

# Finish off the page (this does the ending body and html tags)
print $q->end_html();
}

sub do_help()
{
	my $javascript;
	$javascript = "function LoadMenu() 
	{ 
	parent.menu.location = \"$meurl?page=menu\"; 
	}";
	print $q->start_html({-script => $javascript, -onload => "LoadMenu()" });

	print $q->h1("Information and Help");

	print <<EOT
Updates to the routers.cgi script may be obtained from
<a href=http://ukpavu99.adsweu.com/pub/>here</a>.
<P>
Users may add their own help text in here.
<P>
EOT
;
	print $q->p,$q->hr,"\n";
	print $q->small("Created by Steve Shipway");
	print $q->end_html();
}

# set cookies etc. for defaults.
# the way we do this is by refreshing ourself with extra parameters.
# The existence of the extra parameters causes the cookie to be set.
sub do_config()
{
	my ( $javascript );
	my ( %routerdesc );

	$javascript = "function LoadMenu() 
	{ 
	parent.menu.location = \"$meurl?page=menu&xmtype=$mtype&rtr=none&if=none\"; 
	}";
	print $q->start_html({-script => $javascript, -onload => "LoadMenu()" });

	print $q->h2("Personal Preferences"),"\n";

	if( $CGI::defrouter ) {
		print $q->p($q->b("Options have been saved.")),"\n";
	}

	print $q->p("Options set here will persist over future invocations of this script.  Note that this uses cookies, so you must have them enabled.");

#	print $q->hr;
#	print "Cookies set: ",$q->cookie,$q->br,"\n";
#	print "Router: ",$q->cookie('router')," Gtype: ",$q->cookie('gtype'),
#		$q->br,"\n";
	print $q->hr;
	print "<FORM METHOD=GET ACTION=$meurl>\n";
	
# now a couple of hidden fields to preserve mtype and page
	print $q->hidden({ name=>"page", value=>"config" }),"\n",
		$q->hidden({ name=>"xmtype", value=>$mtype }) ,"\n";

# Now the main fields, defrouter and defgtype.  Dropdown lists.  In a table.

	%routerdesc = ();
	foreach ( keys( %routers ) ) {
		$routerdesc{$_} = "$_";
		$routerdesc{$_} .= " : ".$config{"routerdesc-$_"}
			if($config{"routerdesc-$_"} ne $_);
	}
	print $q->table( { -border=>0 },
		$q->Tr( { -border=>"0", align=>"left" } ,
			$q->td("Default router:" )."\n", $q->td( 
				$q->popup_menu( {name=>"defrouter", 
					values=>[sort bydesc keys(%routers)],
					labels=>{%routerdesc}, default=>$q->cookie('router')})
			)."\n" 
		),
		$q->Tr( { -border=>"0", align=>"left" } ,
			$q->td("Default interface:" )."\n", $q->td( 
				$q->popup_menu({ name=>"defif", values=>["first","summary","info"],
					labels=>{first=>"First", summary=>"Summary page",
						info=>"Info Page"}, 
					default=>$q->cookie('if') })
			)."\n" 
		),
		$q->Tr( { -border=>"0", align=>"left" } ,
			$q->td("Default graph type:" )."\n", $q->td( 
				$q->popup_menu({ name=>"defgtype", values=>[@gorder],
					labels=>{%gtypes}, default=>$q->cookie('gtype') })
			)."\n" 
		),
		$q->Tr( { align=>"left" },
			$q->td(""),$q->td(
				$q->submit
			)."\n"
		)
	),"\n";

	print "</FORM>";

	print $q->br({clear=>"BOTH"}),$q->hr,"\n";
	print $q->small("Created and maintained by Steve Shipway");
	print $q->end_html();
}

###########################
# If we get a bad page request

sub do_bad()
{
	print $q->h1("Eeek, an error!");
	print $q->start_html();
	print $q->dump;
	print $q->end_html();
}

########################################################################
# Initialise parameters

# get these sections from the conf file.
readconf('routers.cgi','web','routerdesc'); 

# Anyone giving us a cookie?
$defgtype = $q->cookie('gtype');
$defgtype = $gorder[0] if(! $defgtype);
$defrouter = (sort bydesc keys(%routers))[0];
$defrouter = $q->cookie('router') if($q->cookie('router'));
#print "defrouter=",$defrouter,"\n";

$router="";
read_routers();
$router = ($CGI::rtr)?$CGI::rtr:$defrouter ;
$router = (sort bydesc keys(%routers))[0] if (! $router );
read_cfg();
$defif = $q->cookie('if');
$defif = (sort numerically keys(%interfaces))[0] if( ! $defif or ($defif eq "first") );
$gtype = ($CGI::xgtype)?$CGI::xgtype:$defgtype ;
$mtype = ($CGI::xmtype)?$CGI::xmtype:"routers" ;
$interface = "";
$interface = ($q->param('if'))?$q->param('if'):$defif ;
$interface = "" if(! $interface);
if (($interface ne "summary") and ($router ne "none") and ($interface ne "info")
		and (! $interfaces{$interface})) {
	$interface = $defif;
}
if ( ! $gtypes{$gtype} or 
	( $interface eq "summary" and (length ($gtype) > 1) )) {
	$gtype = $gorder[0];
}

# Start the page off
$pagetype="";
$pagetype=$CGI::page if( $CGI::page );

if( $pagetype eq "graph" ) {
	$headeropts{-expires} = "+5s";
	$headeropts{-Refresh} = "300";
}
$headeropts{-target} = $pagetype if( $pagetype =~ /head|menu|graph/ );
$headeropts{-target} = "graph" if ( $pagetype =~ /graph|help|info|config/ );
$headeropts{-target} = "_top" if ( !$pagetype );
if ( $pagetype eq "config" ) {
	push @cookies, $q->cookie( -name=>'gtype', -value=>$CGI::defgtype, 
		-path=>$q->url(-absolute=>1), -expires=>"+10y" ) 
			if( $CGI::defgtype );
	push @cookies, $q->cookie( -name=>'router', -value=>$CGI::defrouter, 
		-path=>$q->url(-absolute=>1), -expires=>"+10y" ) 
			if( $CGI::defrouter );
	push @cookies, $q->cookie( -name=>'if', -value=>$CGI::defif, 
		-path=>$q->url(-absolute=>1), -expires=>"+10y" ) 
			if( $CGI::defif );
	$headeropts{-cookie} = [@cookies] if(@cookies);
}

print $q->header( %headeropts );

#
# Now, we check the passed parameters to find out what sort of page to
# serve up.  If we can't work out which one to do, then we just serve the
# index page
if($pagetype) {
	for($pagetype) {
		/head/ and do {	do_head(); last; };
		/menu/ and do { do_menu(); last; };
		/graph/ and do { 
			if( $interface eq "summary") {
				do_summary();
			} elsif ( $interface eq "info" ) {
				do_info();
			} else {
				do_graph();
			}
			last; 
		};
		/help/ and do { do_help(); last; };
		/main/ and do { do_main(); last; };
		/info/ and do { do_info(); last; };
		/summary/ and do { do_summary(); last; };
		/config/ and do { do_config(); last; };
		do_bad();
	}
} else { do_main() }

exit 0;
