C
      SUBROUTINE PANGEN(NDIM,XB,XBP,YB,YBP,SB,NB,NBEL,NPAN,CVPAR,CTERAT)
C
C---------------------------------------------------------------
C     Set paneling distribution from input airfoil geometry.
C     The results are kept in the COMMON block in AIRPAN.INC
C     for the other panel routines.
C 
C     Input:
C
C      NDIM      first dimension of XB,YB,SB
C      XB,YB(..) input airfoil coordinates (2nd index goes over elements)
C      XBP,YBP   derivative arrays for spline evaluation
C      SB(..)    arc length for XB,YB
C      NB(.)     number of points in XB,YB
C      NBEL      number of elements
C
C      NPAN(.)   number of panel nodes to be generated for each element
C      CVPAR(.)  sets how much surface curvature attracts nodes
C      CTERAT(.) ratio of TE to LE panel spacing (approximately)
C
C
C     Output:
C
C      NDIM      first dimension of XA,YA,SA
C      XA,YA(..) panel node coordinates  (2nd index goes over elements)
C      SA(..)    arc length for XA,YA
C---------------------------------------------------------------
      INCLUDE 'AIRPAN.INC'
C
      DIMENSION XB(NDIM,NBEL), XBP(NDIM,NBEL), 
     &          YB(NDIM,NBEL), YBP(NDIM,NBEL), SB(NDIM,NBEL)
      DIMENSION NB(NBEL), NPAN(NBEL)
      DIMENSION CVPAR(NBEL), CTERAT(NBEL)
C
C---- Number of temporary nodes for panel distribution calculation
C       exceeds the specified panel number by factor of IPFAC.
      PARAMETER (IPFACX=3)
C
C---- allocate work arrays for the paneling algorithm
      PARAMETER (IWX=IPFACX*IQX)
      DIMENSION W1(IWX),W2(IWX),W3(IWX),W4(IWX),W5(IWX),W6(IWX)
      DIMENSION SNEW(IWX)
C
C
      NEL = NBEL
C
      DO 9000 IEL=1, NEL
C
      IF(NB(IEL).GT.IWX) STOP 'PANGEN: Array overflow. Increase IQX.'
C
      DO 1 I=1, IQX
        LINVRS(I,IEL) = .FALSE.
    1 CONTINUE
C
C---- locate LE point arc length value
      CALL LEFIND(SBLE,XB(1,IEL),XBP(1,IEL),
     &                 YB(1,IEL),YBP(1,IEL),SB(1,IEL),NB(IEL))
C
C---- set element leading and trailing edge coordinates
      XLE = SEVAL(SBLE,XB(1,IEL),XBP(1,IEL),SB(1,IEL),NB(IEL))
      YLE = SEVAL(SBLE,YB(1,IEL),YBP(1,IEL),SB(1,IEL),NB(IEL))
      XTE = 0.5*(XB(1,IEL)+XB(NB(IEL),IEL))
      YTE = 0.5*(YB(1,IEL)+YB(NB(IEL),IEL))
C
C---- set element chord
      CHORD(IEL) = SQRT( (XTE-XLE)**2 + (YTE-YLE)**2 )
C
C---- set number of panels
      NA(IEL) = NPAN(IEL)
      IF(NA(IEL) .GE. IQX)
     &  STOP 'PANGEN: Too many panels. Increase IQX in AIRPAN.INC'
C
C---- remove one slot for each corner point added later
      DO 2 IB=1, NB(IEL)-1
        IF(SB(IB,IEL) .EQ. SB(IB+1,IEL)) NA(IEL) = NA(IEL)-1
    2 CONTINUE
C
C---- set up non-dimensionalized curvature array
      DO 3 I=1, NB(IEL)
        W5(I) = ABS( 
     &    CURV(SB(I,IEL),
     &         XB(1,IEL),XBP(1,IEL),
     &         YB(1,IEL),YBP(1,IEL),SB(1,IEL),NB(IEL)) )*CHORD(IEL)
    3 CONTINUE
C
C---- nondimensionalized curvature at LE point
      CVLE = ABS( CURV(SBLE,
     &                 XB(1,IEL),XBP(1,IEL),
     &                 YB(1,IEL),YBP(1,IEL),
     &                 SB(1,IEL),NB(IEL)) )*CHORD(IEL)
C
C
C---- set curvature attraction coefficient actually used
      CC = 5.0 * CVPAR(IEL)
C
C---- set artificial curvature at TE to bunch panels there
      CVTE = CVLE * CTERAT(IEL)
      W5(1)       = CVTE
      W5(NB(IEL)) = CVTE
C
C
C**** smooth curvature array for smoother panel size distribution  ****
C
      SMOOL = 0.015
CCC      CALL ASK('Enter curvature smoothing length/c\',3,SMOOL)
      SMOOSQ = (SMOOL*CHORD(IEL))**2
C
C---- set up tri-diagonal system for smoothed curvatures
      W2(1) = 1.0
      W3(1) = 0.0
      DO 4 I=2, NB(IEL)-1
        DSM = SB(I  ,IEL) - SB(I-1,IEL)
        DSP = SB(I+1,IEL) - SB(I  ,IEL)
        DSO = 0.5*(SB(I+1,IEL) - SB(I-1,IEL))
C
        IF(DSM.EQ.0.0 .OR. DSP.EQ.0.0) THEN
         W1(I) = 0.0
         W2(I) = 1.0
         W3(I) = 0.0
        ELSE
         W1(I) =  SMOOSQ * (         - 1.0/DSM) / DSO
         W2(I) =  SMOOSQ * ( 1.0/DSP + 1.0/DSM) / DSO  +  1.0
         W3(I) =  SMOOSQ * (-1.0/DSP          ) / DSO
        ENDIF
    4 CONTINUE
C
      W1(NB(IEL)) = 0.0
      W2(NB(IEL)) = 1.0
C
C---- fix curvature at LE point by modifying equations adjacent to LE
      DO 5 I=2, NB(IEL)-1
        IF(SB(I,IEL).EQ.SBLE) THEN
C------- if node falls right on LE point, fix curvature there
         W1(I) = 0.
         W2(I) = 1.0
         W3(I) = 0.
         W5(I) = CVLE
         GO TO 51
        ELSE IF(SB(I-1,IEL).LT.SBLE .AND. SB(I,IEL).GT.SBLE) THEN
C------- modify equation at node just before LE point
         DSM = SB(I-1,IEL) - SB(I-2,IEL)
         DSP = SBLE        - SB(I-1,IEL)
         DSO = 0.5*(SBLE - SB(I-2,IEL))
C
         W1(I-1) =  SMOOSQ * (         - 1.0/DSM) / DSO
         W2(I-1) =  SMOOSQ * ( 1.0/DSP + 1.0/DSM) / DSO  +  1.0
         W3(I-1) =  0.
         W5(I-1) = W5(I-1) + SMOOSQ*CVLE/(DSP*DSO)
C
C------- modify equation at node just after LE point
         DSM = SB(I  ,IEL) - SBLE
         DSP = SB(I+1,IEL) - SB(I,IEL)
         DSO = 0.5*(SB(I+1,IEL) - SBLE)
         W1(I) =  0.
         W2(I) =  SMOOSQ * ( 1.0/DSP + 1.0/DSM) / DSO  +  1.0
         W3(I) =  SMOOSQ * (-1.0/DSP          ) / DSO
         W5(I) = W5(I) + SMOOSQ*CVLE/(DSM*DSO)
C
         GO TO 51
        ENDIF
    5 CONTINUE
   51 CONTINUE
C
C---- solve for smoothed curvature array W5
      CALL TRISOL(W2,W1,W3,W5,NB(IEL))
C
C---- find max curvature
      CVMAX = 0.
      DO 61 I=1, NB(IEL)
        CVMAX = MAX( CVMAX , ABS(W5(I)) )
   61 CONTINUE
C
C---- normalize curvature array
      DO 62 I=1, NB(IEL)
        W5(I) = W5(I) / CVMAX
   62 CONTINUE
C
C---- spline curvature array
      CALL SEGSPL(W5,W6,SB(1,IEL),NB(IEL))
C
C---- Set initial guess for node positions uniform in s.
C     More nodes than specified (by factor of IPFAC) are 
C     temporarily used  for more reliable convergence.
      DO 7 K=1, IPFACX
        IPFAC = IPFACX - K + 1
        NN = IPFAC*(NA(IEL)-1)+1
        IF(NN .LE. IWX) GO TO 71
 7    CONTINUE
      STOP 'PANGEN: Work array overflow.  Increase IQX.'
 71   CONTINUE
C
      DSAVG = SB(NB(IEL),IEL)/FLOAT(NN-1)
      DO 8 I=1, NN
        SNEW(I) = DSAVG * FLOAT(I-1)
    8 CONTINUE
C
C---- Newton iteration loop for new node positions
      DO 10 ITER=1, 20
C
C------ set up tri-diagonal system for node position deltas
        CV1  = SEVAL(SNEW(1),W5,W6,SB(1,IEL),NB(IEL))
        CV2  = SEVAL(SNEW(2),W5,W6,SB(1,IEL),NB(IEL))
        CVS1 = DEVAL(SNEW(1),W5,W6,SB(1,IEL),NB(IEL))
        CVS2 = DEVAL(SNEW(2),W5,W6,SB(1,IEL),NB(IEL))
        CAVM = SQRT(CV1**2 + CV2**2)
        DO 110 I=2, NN-1
          DSM = SNEW(I) - SNEW(I-1)
          DSP = SNEW(I) - SNEW(I+1)
          CV3  = SEVAL(SNEW(I+1),W5,W6,SB(1,IEL),NB(IEL))
          CVS3 = DEVAL(SNEW(I+1),W5,W6,SB(1,IEL),NB(IEL))
          CAVP = SQRT(CV3**2 + CV2**2)
          FM = CC*CAVM + 1.0
          FP = CC*CAVP + 1.0
C
C-------- lower, main, and upper diagonals
          W1(I) =      -FM  +  CC*            DSM/CAVM  * CV1*CVS1
          W2(I) =  FP + FM  +  CC*(DSP/CAVP + DSM/CAVM) * CV2*CVS2
          W3(I) = -FP       +  CC* DSP/CAVP             * CV3*CVS3
C
C-------- righthand side, requiring that
C         (1 + C*curv)*deltaS is equal on both sides of node i
          W4(I) = -DSP*FP - DSM*FM
C
          CV1 = CV2
          CV2 = CV3
          CVS1 = CVS2
          CVS2 = CVS3
          CAVM = CAVP
  110   CONTINUE
C
C------ fix endpoints (at TE)
        W2(1) = 1.0
        W3(1) = 0.0
        W4(1) = 0.0
        W1(NN) = 0.0
        W2(NN) = 1.0
        W4(NN) = 0.0
C
C------ solve for changes W4 in node position arc length values
        CALL TRISOL(W2,W1,W3,W4,NN)
C
C------ find under-relaxation factor so as to keep nodes from changing order
        RLX = 1.0
        DMAX = 0.0
        DO 12 I=1, NN-1
          DS  = SNEW(I+1) - SNEW(I)
          DDS = W4(I+1) - W4(I)
          DSRAT = 1.0 + RLX*DDS/DS
          IF(DSRAT.GT.4.0) RLX = (4.0-1.0)*DS/DDS
          IF(DSRAT.LT.0.2) RLX = (0.2-1.0)*DS/DDS
          DMAX = MAX(ABS(W4(I)),DMAX)
   12   CONTINUE
C
C------ update node position
        DO 13 I=2, NN-1
          SNEW(I) = SNEW(I) + RLX*W4(I)
   13   CONTINUE
C
CCC        IF(RLX.EQ.1.0) WRITE(*,*) DMAX
CCC        IF(RLX.NE.1.0) WRITE(*,*) DMAX,'    RLX =',RLX
        IF(ABS(DMAX).LT.1.E-3) GO TO 19
   10 CONTINUE
      WRITE(*,*) 'Paneling convergence failed.  Continuing anyway...'
C
   19 CONTINUE
C
C---- set new panel node coordinates
      DO 20 I=1, NA(IEL)
        IND = IPFAC*(I-1) + 1
        SA(I,IEL) = SNEW(IND)
        XA(I,IEL) = SEVAL(SNEW(IND),XB(1,IEL),XBP(1,IEL),
     &                   SB(1,IEL),NB(IEL))
        YA(I,IEL) = SEVAL(SNEW(IND),YB(1,IEL),YBP(1,IEL),
     &                   SB(1,IEL),NB(IEL))
   20 CONTINUE
C
C---- set arc length array for panel node coordinates
ccc   CALL SCALC(XA(1,IEL),YA(1,IEL),SA(1,IEL),NA(IEL))
C
C---- go over input airfoil coordinates, checking for corners (double points)
      NCORN = 0
      DO 25 IB=1, NB(IEL)-1
        IF(SB(IB,IEL) .EQ. SB(IB+1,IEL)) THEN
C------- found one ! ... Find panel which contains corner
         NCORN = NCORN+1
         XBCORN = XB(IB,IEL)
         YBCORN = YB(IB,IEL)
         SBCORN = SB(IB,IEL)
         DO 252 I=1, NA(IEL)
C
C--------- keep stepping until first node past corner
           IF(SA(I,IEL) .LE. SBCORN) GO TO 252
C
C---------- move remainder of panel nodes to make room for additional node
            DO 2522 J=NA(IEL), I, -1
              XA(J+1,IEL) = XA(J,IEL)
              YA(J+1,IEL) = YA(J,IEL)
              SA(J+1,IEL) = SA(J,IEL)
              LINVRS(J+1,IEL) = LINVRS(J,IEL)
 2522       CONTINUE
            NA(IEL) = NA(IEL)+1
            IF(NA(IEL) .GT. IQX-1)
     &       STOP 'PANGEN: Too many panels. Increase IQX in AIRPAN.INC'
            XA(I,IEL) = XBCORN
            YA(I,IEL) = YBCORN
            SA(I,IEL) = SBCORN
C
C---------- shift nodes adjacent to corner to keep panel sizes comparable
            IF(I-2 .GE. 1) THEN
             SA(I-1,IEL) = 0.5*(SA(I,IEL) + SA(I-2,IEL))
             XA(I-1,IEL) = SEVAL(SA(I-1,IEL),XB(1,IEL),XBP(1,IEL),
     &                           SB(1,IEL),NB(IEL))
             YA(I-1,IEL) = SEVAL(SA(I-1,IEL),YB(1,IEL),YBP(1,IEL),
     &                           SB(1,IEL),NB(IEL))
            ENDIF
C
            IF(I+2 .LE. NA(IEL)) THEN
             SA(I+1,IEL) = 0.5*(SA(I,IEL) + SA(I+2,IEL))
             XA(I+1,IEL) = SEVAL(SA(I+1,IEL),XB(1,IEL),XBP(1,IEL),
     &                           SB(1,IEL),NB(IEL))
             YA(I+1,IEL) = SEVAL(SA(I+1,IEL),YB(1,IEL),YBP(1,IEL),
     &                           SB(1,IEL),NB(IEL))
            ENDIF
C
C---------- mark panel nodes from corner to 80% to TE as inverse nodes
            DOTP = XBP(IB,IEL)*(XTE-XLE) + YBP(IB,IEL)*(YTE-YLE)
            IF(DOTP .GT. 0.0) THEN
C----------- bottom surface ...
             DO 2524 J=I+1, NA(IEL)-2
               FRAC = (SA(J,IEL)-SA(I,IEL))/(SA(NA(IEL),IEL)-SA(I,IEL))
               IF(FRAC .LT. 0.80) LINVRS(J,IEL) = .TRUE.
 2524        CONTINUE
            ELSE
C----------- top surface
             DO 2526 J=I-1, 3, -1
               FRAC = (SA(J,IEL)-SA(I,IEL))/(SA(1,IEL)-SA(I,IEL))
               IF(FRAC .LT. 0.80) LINVRS(J,IEL) = .TRUE.
 2526        CONTINUE
            ENDIF
C
C---------- go on to next input geometry point to check for corner
            GO TO 25
C
  252    CONTINUE
        ENDIF
   25 CONTINUE
C
ccc   CALL SCALC (XA(1,IEL), YA(1,IEL),SA(1,IEL),NA(IEL))
      CALL SPLINE(XA(1,IEL),XAP(1,IEL),SA(1,IEL),NA(IEL))
      CALL SPLINE(YA(1,IEL),YAP(1,IEL),SA(1,IEL),NA(IEL))
C
C---- set normal vectors
      DO 30 I=1, NA(IEL)
        NX(I,IEL) =  YAP(I,IEL)
        NY(I,IEL) = -XAP(I,IEL)
   30 CONTINUE
C
C---- calculate panel size ratios (user info)
      DSMIN =  1000.0
      DSMAX = -1000.0
      DO 40 I=1, NA(IEL)-1
        DS = SA(I+1,IEL) - SA(I,IEL)
        DSMIN = MIN(DSMIN,DS)
        DSMAX = MAX(DSMAX,DS)
   40 CONTINUE
C
      DSMIN = DSMIN*FLOAT(NA(IEL)-1)/SA(NA(IEL),IEL)
      DSMAX = DSMAX*FLOAT(NA(IEL)-1)/SA(NA(IEL),IEL)
CCC      WRITE(*,*) 'DSmin/DSavg = ',DSMIN,'     DSmax/DSavg = ',DSMAX
C
 9000 CONTINUE ! with next element
C
C---- set up system pointers
      IQ0(1) = 0
      DO 50 IEL=1, NEL-1
        IQ0(IEL+1) = IQ0(IEL) + NA(IEL)
   50 CONTINUE
C
      NTOT = 0
      DO 52 IEL=1, NEL
        NTOT = NTOT + NA(IEL)
   52 CONTINUE
C
C---- check if arrays are big enough for panel matrix solution
      NMAT = NTOT + NEL
      IF(NMAT .GT. IAX) THEN
       WRITE(*,*) 'Arrays will overflow'
       WRITE(*,*) 'Increase IAX in AIRPAN.INC to ', NMAT
      ENDIF
C
C---- determine if TE is blunt or sharp, calculate TE geometry parameters
      CALL TECALC
C
C      DO 60 IEL=1, NEL
C        IF(SHARP(IEL)) THEN
C         WRITE(*,1101) IEL
CCCC         WRITE(*,1200) NA(IEL), CVPAR(IEL), CTERAT(IEL), CTRRAT(IEL),
CCCC     &              XSREF1(IEL), XSREF2(IEL), XPREF1(IEL), XPREF2(IEL)
C        ELSE
C         WRITE(*,1102) IEL
CCCC         WRITE(*,1200) NA(IEL), CVPAR(IEL), CTERAT(IEL), CTRRAT(IEL),
CCCC     &              XSREF1(IEL), XSREF2(IEL), XPREF1(IEL), XPREF2(IEL)
C        ENDIF
C   60 CONTINUE
C 1101 FORMAT(' Element',I2,'   Sharp trailing edge')
C 1102 FORMAT(' Element',I2,'   Blunt trailing edge')
CC
C 1200 FORMAT(/' Paneling parameters used:'
C     &       /' Number of panel nodes     ' , I3
C     &       /' Panel bunching parameter  ' , F6.3
C     &       /' TE/LE panel density ratio ' , F6.3
C     &       /' Refined area/LE panel density ratio  ' , F6.3
C     &       /' Suction  side refined area x/c limits' , 2F6.3
C     &       /' Pressure side refined area x/c limits' , 2F6.3 /)
CC
C---- there's no factored Aij matrix for the new geometry
      LQAIJ = .FALSE.
C
      RETURN
C
      END ! PANGEN


      SUBROUTINE TECALC
C--------------------------------------------
C     Calculates total and projected TE gap 
C     areas and TE panel strengths.
C--------------------------------------------
      INCLUDE 'AIRPAN.INC'
C
      DO 900 IEL=1, NEL
C
      DXTE(IEL) = XA(1,IEL) - XA(NA(IEL),IEL)
      DYTE(IEL) = YA(1,IEL) - YA(NA(IEL),IEL)
      DXS = 0.5*(-XAP(1,IEL) + XAP(NA(IEL),IEL))
      DYS = 0.5*(-YAP(1,IEL) + YAP(NA(IEL),IEL))
C
C---- normal and streamwise projected TE gap areas
      ANTE(IEL) = DXS*DYTE(IEL) - DYS*DXTE(IEL)
      ASTE(IEL) = DXS*DXTE(IEL) + DYS*DYTE(IEL)
C
C---- total TE gap area
      DSTE(IEL) = SQRT(DXTE(IEL)**2 + DYTE(IEL)**2)
C
      SHARP(IEL) = DSTE(IEL) .LT. 0.0001*CHORD(IEL)
C
      IF(SHARP(IEL)) THEN
       SCS = 1.0
       SDS = 0.0
      ELSE
       SCS = ANTE(IEL)/DSTE(IEL)
       SDS = ASTE(IEL)/DSTE(IEL)
      ENDIF
C
C---- TE panel source and vorticity strengths
CCC      SIGTE(IEL) = 0.5*(GAM(1,IEL) - GAM(NA(IEL),IEL))*SCS
CCC      GAMTE(IEL) = -.5*(GAM(1,IEL) - GAM(NA(IEL),IEL))*SDS
C
  900 CONTINUE
C
      RETURN
      END ! TECALC


 
      SUBROUTINE AVAIRF(YGROUND)
      INCLUDE 'AIRPAN.INC'
      LOGICAL BELOWG, ELSAME
C------------------------------------------------
C     Sets angles of vectors from first airfoil 
C     contour points J=1 to all surface points.
C------------------------------------------------
C
C---- airfoil below ground plane?
      BELOWG = YA(1,NEL) .LT. YGROUND
C
      IF(YGROUND.EQ.999.0) THEN
C------ no image airfoil
C
        DO 10 JEL=1, NEL
          JELT(JEL) = JEL
          YSGN(JEL) = 1.0
          YDEL(JEL) = 0.0
 10     CONTINUE
        NELT = NEL
C
      ELSE
C------ image airfoil is present
C
        IF(BELOWG) THEN
          DO 21 JEL=1, NEL
            JELT(NEL-JEL+1) =  JEL
            YSGN(NEL-JEL+1) = -1.0
            YDEL(NEL-JEL+1) =  2.0*YGROUND
            JELT(NEL+JEL  ) =  JEL
            YSGN(NEL+JEL  ) =  1.0
            YDEL(NEL+JEL  ) =  0.0
 21       CONTINUE
        ELSE
          DO 22 JEL=1, NEL
            JELT(    JEL  ) =  JEL
            YSGN(    JEL  ) =  1.0
            YDEL(    JEL  ) =  0.0
            JELT(NEL+JEL  ) =  NEL-JEL+1
            YSGN(NEL+JEL  ) = -1.0
            YDEL(NEL+JEL  ) =  2.0*YGROUND
 22       CONTINUE
        ENDIF
        NELT = 2*NEL
C
      ENDIF
C
C
C---- go over all J=1 panel points on all elements, real and image
      DO 900 JT=1, NELT
C
C---- index of real element defining this real or image element
      JEL = JELT(JT)
C
C---- airfoil point
      XJ1 = XA(1,JEL)
      YJ1 = YA(1,JEL)*YSGN(JT) + YDEL(JT)
C
C---- go over all field points (actually, airfoil surface nodes)
      DO 100 IEL=1, NEL
C
        ELSAME = IEL.EQ.JEL .AND. YSGN(JT).GT.0.0
C
C------ set angle from J=1 point to first field point
        I = 1
        RXN = XA(I,IEL) - XJ1
        RYN = YA(I,IEL) - YJ1
C
        IF(IEL.EQ.1) THEN
C
C------- first element --- just use ATAN2
         IF(ELSAME) THEN
          AVECA(I,IEL,JT) = ATAN2(NX(1,IEL),NY(1,IEL))
         ELSE
          AVECA(I,IEL,JT) = ATAN2(RXN,RYN)
         ENDIF
C
        ELSE
C
C------- lower elements --- make sure they use the same branch as upper element
         IELM = IEL-1
         ATMP = AVECA(NA(IELM),IELM,JT)
C
         IF(ELSAME) THEN
          AVECA(I,IEL,JT) = ATANC(NX(I,IEL),NY(I,IEL),ATMP)
         ELSE
          AVECA(I,IEL,JT) = ATANC(RXN,RYN,ATMP)
         ENDIF
C
        ENDIF
C
C------ set angles to remaining field points so that a branch cut is not crossed
        IF(SHARP(IEL) .AND. ELSAME) THEN
C
         DO 40 I=2, NA(IEL)-1
           RXN = XA(I,IEL) - XJ1
           RYN = YA(I,IEL) - YJ1
           AVECA(I,IEL,JT) = ATANC(RXN,RYN,AVECA(I-1,IEL,JT))
   40    CONTINUE
         I = NA(IEL)
         AVECA(I,IEL,JT) = ATANC(NX(I,IEL),NY(I,IEL),
     &                            AVECA(I-1,IEL,JT))
        ELSE
C
         DO 42 I=2, NA(IEL)
           RXN = XA(I,IEL) - XJ1
           RYN = YA(I,IEL) - YJ1
           AVECA(I,IEL,JT) = ATANC(RXN,RYN,AVECA(I-1,IEL,JT))
   42    CONTINUE
C
        ENDIF
C
  100 CONTINUE
  900 CONTINUE
C
      RETURN
      END ! AVAIRF


      SUBROUTINE AVSPAC(XI,YI, ANGL)
      INCLUDE 'AIRPAN.INC'
      DIMENSION ANGL(2*IEX)
C
C---- go over all J=1 panel points on all elements, real and image
      DO 900 JT=1, NELT
C
C---- index of real element defining this real or image element
      JEL = JELT(JT)
C
C---- airfoil point
      XJ1 = XA(1,JEL)
      YJ1 = YA(1,JEL)*YSGN(JT) + YDEL(JT)
C
C---- set angle from J=1 point to field point
      RXN = XI - XJ1
      RYN = YI - YJ1
      ANGL(JT) = ATAN2(RXN,RYN)
C
  900 CONTINUE
C
      RETURN
      END ! AVSPAC


      SUBROUTINE PANSOL(YGROUND)
C-------------------------------------------------------------
C     Calculates two surface vorticity (gamma) distributions
C     for alpha = 0,90  degrees.  These can be later 
C     superimposed for specified alpha or CL.
C-------------------------------------------------------------
      INCLUDE 'AIRPAN.INC'
      DIMENSION ANGL1(2*IEX)
C
      QINF = 1.0
C
      COSA = COS(ALFA)
      SINA = SIN(ALFA)
C
C---- calculate position vector angles for panel routines
      CALL AVAIRF(YGROUND)
C
      WRITE(*,*)
      WRITE(*,*) 'Calculating unit vorticity distributions ...'
C
C---- go over alements
      DO 9000 IEL=1, NEL
C
C---- system row indices for 1st and last point (panel node) on this element
      JQ1 = IQ0(IEL) + 1
      JQN = IQ0(IEL) + NA(IEL)
C
C---- Set up matrix system for  Psi = Psio  on airfoil surface.
C-    The unknowns are (dGamma)i  or  (dSigma)i ,  and dPsio.
      DO 20 I=1, NA(IEL)
C
C------ system row index for this control point (panel node)
        IQ = IQ0(IEL) + I
C
C------ set angles from all first points on all elements to this control point
        DO 205 JT=1, NELT
          ANGL1(JT) = AVECA(I,IEL,JT)
  205   CONTINUE
C
C------ calculate Psi[Gamma, cos(a), sin(a)]  for current node
        CALL PSIPHI(I,IEL,XA(I,IEL),YA(I,IEL),ANGL1,PSII,PHII,UII,VII)
C
C------ set  Res = Psi( 0) - PsiO
        RES1 = PSI_CA
C
C------ set  Res = Psi(90) - PsiO
        RES2 = PSI_SA
C
C------ dRes/dGamma  or  dRes/dSigma
        DO 210 JEL=1, NEL
          DO 2105 J=1, NA(JEL)
            JQ = IQ0(JEL) + J
            AIJ(IQ,JQ) = PSI_G(J,JEL)
 2105     CONTINUE
  210   CONTINUE
C
C------ dRes/dPsio
        DO 220 JEL=1, NEL
          JQ = NTOT + JEL
          AIJ(IQ,JQ) = 0.0
  220   CONTINUE
        AIJ(IQ,NTOT+IEL) = -1.0
C
C------ local source contribution to Psi
        IF(LINVRS(I,IEL)) AIJ(IQ,IQ) = AIJ(IQ,IQ) - 1.0
C
        DQ(IQ,1) = -RES1
        DQ(IQ,2) = -RES2
C
 20   CONTINUE
C
C---- set Kutta condition
      RES = 0.0
CCC       = GAM(1) + GAM(N)
C
      IQ = NTOT + IEL
C
      DO 30 JEL=1, NEL
        DO 301 J=1, NA(JEL)
          JQ = IQ0(JEL) + J
          AIJ(IQ,JQ) = 0.0
  301   CONTINUE
        JQ = NTOT + JEL
        AIJ(IQ,JQ) = 0.0
   30 CONTINUE
C
      AIJ(IQ,JQ1) = 1.0
      AIJ(IQ,JQN) = 1.0
C
      DQ(IQ,1) = -RES
      DQ(IQ,2) = -RES
C
      IF(SHARP(IEL)) THEN
C----- TE Gamma extrapolation if  1st and Nth  points coincide
       IQ = IQ0(IEL) + NA(IEL)
C
       TM = XA(3,IEL) + XA(NA(IEL)-2,IEL)
       TO = XA(2,IEL) + XA(NA(IEL)-1,IEL)
       TP = XA(1,IEL) + XA(NA(IEL)  ,IEL)
C
       DM = 1.0 / (TP - TO)
       DP = 1.0 / (TO - TM)
C
       DO 40 JEL=1, NEL
         DO 401 J=1, NA(JEL)
           JQ = IQ0(IEL) + J
           AIJ(IQ,JQ) = 0.0
  401    CONTINUE
         JQ = NTOT + JEL
         AIJ(IQ,JQ) = 0.0
   40  CONTINUE
C
       RES = 0.0
C          = ( (GAM(1)-GAM(N  )) - (GAM(2)-GAM(N-1)) ) * DP 
C    &     - ( (GAM(2)-GAM(N-1)) - (GAM(3)-GAM(N-2)) ) * DM
C
       AIJ(IQ,JQ1  ) =   DP
       AIJ(IQ,JQ1+1) = -(DP+DM)
       AIJ(IQ,JQ1+2) =   DM
       AIJ(IQ,JQN  ) = -DP
       AIJ(IQ,JQN-1) =  DP+DM
       AIJ(IQ,JQN-2) = -DM
C
C----- redefine influence coefficients for any inverse points
       DO 42 J = 1, 3
         JQ = IQ0(IEL) + J
         IF(LINVRS(J,IEL)) THEN
          CALL GMILIN(J,IEL,JL,JR,GMI_GL,GMI_GR)
          JQL = IQ0(IEL) + JL
          JQR = IQ0(IEL) + JR
          AIJ(IQ,JQL) = AIJ(IQ,JQL) + AIJ(IQ,JQ)*GMI_GL
          AIJ(IQ,JQR) = AIJ(IQ,JQR) + AIJ(IQ,JQ)*GMI_GR
          AIJ(IQ,JQ) = 0.0
         ENDIF
   42  CONTINUE
C
       DO 43 J = NA(IEL)-2, NA(IEL)
         JQ = IQ0(IEL) + J
         IF(LINVRS(J,IEL)) THEN
          CALL GMILIN(J,IEL,JL,JR,GMI_GL,GMI_GR)
          JQL = IQ0(IEL) + JL
          JQR = IQ0(IEL) + JR
          AIJ(IQ,JQL) = AIJ(IQ,JQL) + AIJ(IQ,JQ)*GMI_GL
          AIJ(IQ,JQR) = AIJ(IQ,JQR) + AIJ(IQ,JQ)*GMI_GR
          AIJ(IQ,JQ) = 0.0
         ENDIF
   43  CONTINUE
C
       DQ(IQ,1) = -RES
       DQ(IQ,2) = -RES
C
      ENDIF
C
 9000 CONTINUE
C
C---- solve system for the two vorticity distributions
      NMAT = NTOT + NEL
C
C---- LU-factor coefficient matrix AIJ
      CALL LUDCMP(IAX,NMAT,AIJ,AIJPIV)
      LQAIJ = .TRUE.
C
C---- solve system for the two vorticity distributions
      CALL BAKSUB(IAX,NMAT,AIJ,AIJPIV,DQ(1,1))
      CALL BAKSUB(IAX,NMAT,AIJ,AIJPIV,DQ(1,2))
C
C---- set alpha=0,90 surface vorticity for this geometry
      DO 50 IEL=1, NEL
        DO 501 I=1, NA(IEL)
          IQ = IQ0(IEL) + I
          GAMU(I,1,IEL) = DQ(IQ,1)
          GAMU(I,2,IEL) = DQ(IQ,2)
          SIGU(I,1,IEL) = 0.0
          SIGU(I,2,IEL) = 0.0
 501    CONTINUE
C
C------ reset source and vorticity at any inverse points
        DO 502 I=1, NA(IEL)
          IF(LINVRS(I,IEL)) THEN
           CALL GMILIN(I,IEL,JL,JR,GMI_GL,GMI_GR)
           SIGU(I,1,IEL) = GAMU(I,1,IEL)
           SIGU(I,2,IEL) = GAMU(I,2,IEL)
           GAMU(I,1,IEL) = GMI_GL*GAMU(JL,1,IEL) + GMI_GR*GAMU(JR,1,IEL)
           GAMU(I,2,IEL) = GMI_GL*GAMU(JL,2,IEL) + GMI_GR*GAMU(JR,2,IEL)
          ENDIF
 502    CONTINUE
C
C------ set Psio0 and Psio90
        I = NA(IEL)+1
        IQ = NTOT + IEL
        GAMU(I,1,IEL) = DQ(IQ,1)
        GAMU(I,2,IEL) = DQ(IQ,2)
 50   CONTINUE
C
C---- set circulations for alpha=0,90
      DO 60 IEL=1, NEL
        CIRC1 = 0.0
        CIRC2 = 0.0
        DO 602 I=2, NA(IEL)
          DS = SA(I,IEL) - SA(I-1,IEL)
          CIRC1 = CIRC1 + DS*0.5*(GAMU(I,1,IEL) + GAMU(I-1,1,IEL))
          CIRC2 = CIRC2 + DS*0.5*(GAMU(I,2,IEL) + GAMU(I-1,2,IEL))
 602    CONTINUE
        CIRCU(1,IEL) = CIRC1
        CIRCU(2,IEL) = CIRC2
C
        IF(SHARP(IEL)) THEN
         SCS = 1.0
         SDS = 0.0
        ELSE
         SCS = ANTE(IEL)/DSTE(IEL)
         SDS = ASTE(IEL)/DSTE(IEL)
        ENDIF
C
C------ TE panel source and vorticity strengths
        GAMTE1 = -.5*(GAMU(1,1,IEL) - GAMU(NA(IEL),1,IEL))*SDS
        GAMTE2 = -.5*(GAMU(1,2,IEL) - GAMU(NA(IEL),2,IEL))*SDS
        CIRCU(1,IEL) = CIRCU(1,IEL) + GAMTE1*DSTE(IEL)
        CIRCU(2,IEL) = CIRCU(2,IEL) + GAMTE2*DSTE(IEL)
C
 60   CONTINUE
C
      RETURN
      END ! PANSOL
      
      
      
      SUBROUTINE PANSEN(YGROUND)
C------------------------------------------------
C     Calculates the sensitivities of gamma to 
C     the individual element position changes.
C------------------------------------------------
      INCLUDE 'AIRPAN.INC'
      DIMENSION ANGL1(2*IEX)
C     
      QINF = 1.0
C     
      COSA = COS(ALFA)
      SINA = SIN(ALFA)
C     
      IF(.NOT.LQAIJ) CALL PANSOL(YGROUND)
C     
      WRITE(*,*)
      WRITE(*,*) 'Calculating sensitivities to element movements ...'
C     
      NRHS = 4*NEL
C     
      DO 10 IQ=1, NMAT
        DO 102 L=1, NRHS
          DQ(IQ,L) = 0.
 102    CONTINUE
 10   CONTINUE
C     
C---- go over alements
      DO 9000 IEL=1, NEL
C     
C---- system row indices for 1st and last point (panel node) on this element
      JQ1 = IQ0(IEL) + 1
      JQN = IQ0(IEL) + NA(IEL)
C     
C---- Set up matrix system for  Psi = Psio  on airfoil surface.
C-    The unknowns are (dGamma)i  or  (dSigma)i ,  and dPsio.
      DO 20 I=1, NA(IEL)
C     
C------system row index for this control point (panel node)
        IQ = IQ0(IEL) + I
C     
C------set angles from all first points on all elements to this control point
        DO 205 JT=1, NELT
          ANGL1(JT) = AVECA(I,IEL,JT)
 205    CONTINUE
C     
C------calculate Psi and dPsi/dGamma array for current node
        CALL PSIPHI(I,IEL,XA(I,IEL),YA(I,IEL),ANGL1,PSII,PHII,UII,VII)
C     
C------set  Res = Psi - PsiO  sensitivities wrt element movements
        DO 225 JEL=1, NEL
          L = 4*JEL
          DQ(IQ,L-3) = PSI_XE(JEL)
          DQ(IQ,L-2) = PSI_YE(JEL)
          DQ(IQ,L-1) = PSI_FE(JEL)
          DQ(IQ,L  ) = PSI_AE(JEL)
 225    CONTINUE
C     
 20   CONTINUE
C     
C---- set Kutta condition
      IQ = NTOT + IEL
      DO 21 L=1, NRHS
        DQ(IQ,L) = 0.
 21   CONTINUE
C     
      IF(SHARP(IEL)) THEN
C-----TE Gamma extrapolation if  1st and Nth  points coincide
        IQ = IQ0(IEL) + NA(IEL)
        DO 25 L=1, NRHS
          DQ(IQ,L) = 0.
 25     CONTINUE
      ENDIF
C     
 9000 CONTINUE
C     
C---- back-substitute for the sensitivity vectors
      DO 40 L=1, NRHS
        CALL BAKSUB(IAX,NMAT,AIJ,AIJPIV,DQ(1,L))
 40   CONTINUE
C     
C---- set alpha=0,90 surface vorticity for this geometry
      DO 50 IEL=1, NEL
        DO 501 I=1, NA(IEL)
          IQ = IQ0(IEL) + I
          DO 5010 JEL=1, NEL
            L = 4*JEL
            GAMX(I,JEL,IEL) = -DQ(IQ,L-3)
            GAMY(I,JEL,IEL) = -DQ(IQ,L-2)
            GAMF(I,JEL,IEL) = -DQ(IQ,L-1)
            GAMA(I,JEL,IEL) = -DQ(IQ,L  )
            SIGX(I,JEL,IEL) = 0.0
            SIGY(I,JEL,IEL) = 0.0
            SIGF(I,JEL,IEL) = 0.0
            SIGA(I,JEL,IEL) = 0.0
 5010     CONTINUE
 501    CONTINUE
C     
C------reset source and vorticity at any inverse points
        DO 502 I=1, NA(IEL)
          IF(LINVRS(I,IEL)) THEN
            CALL GMILIN(I,IEL,JL,JR,GMI_GL,GMI_GR)
            DO 5020 JEL=1, NEL
              SIGX(I,JEL,IEL) = GAMX(I,JEL,IEL)
              SIGY(I,JEL,IEL) = GAMY(I,JEL,IEL)
              SIGF(I,JEL,IEL) = GAMF(I,JEL,IEL)
              SIGA(I,JEL,IEL) = GAMA(I,JEL,IEL)
              GAMX(I,JEL,IEL) = GMI_GL*GAMX(JL,JEL,IEL)
     &             + GMI_GR*GAMX(JR,JEL,IEL)
              GAMY(I,JEL,IEL) = GMI_GL*GAMY(JL,JEL,IEL)
     &             + GMI_GR*GAMY(JR,JEL,IEL)
              GAMF(I,JEL,IEL) = GMI_GL*GAMF(JL,JEL,IEL)
     &             + GMI_GR*GAMF(JR,JEL,IEL)
              GAMA(I,JEL,IEL) = GMI_GL*GAMA(JL,JEL,IEL)
     &             + GMI_GR*GAMA(JR,JEL,IEL)
 5020       CONTINUE
          ENDIF
 502    CONTINUE
C     
C------set Psio0 and Psio90
        I = NA(IEL)+1
        IQ = NTOT + IEL
        DO 504 JEL=1, IEL
          L = 4*JEL
          GAMX(I,JEL,IEL)  = DQ(IQ,L-3)
          GAMY(I,JEL,IEL)  = DQ(IQ,L-2)
          GAMF(I,JEL,IEL)  = DQ(IQ,L-1)
          GAMA(I,JEL,IEL)  = DQ(IQ,L  )
 504    CONTINUE
 50   CONTINUE
C     
C---- set circulations for alpha=0,90
      DO 60 IEL=1, NEL
        DO 602 JEL=1, NEL
          CIRCXS = 0.0
          CIRCYS = 0.0
          CIRCFS = 0.0
          CIRCAS = 0.0
          DO 6021 I=2, NA(IEL)
            DS = SA(I,IEL) - SA(I-1,IEL)
            CIRCXS = CIRCXS + DS*0.5*(GAMX(I,JEL,IEL)+GAMX(I-1,JEL,IEL))
            CIRCYS = CIRCYS + DS*0.5*(GAMY(I,JEL,IEL)+GAMY(I-1,JEL,IEL))
            CIRCFS = CIRCFS + DS*0.5*(GAMF(I,JEL,IEL)+GAMF(I-1,JEL,IEL))
            CIRCAS = CIRCAS + DS*0.5*(GAMA(I,JEL,IEL)+GAMA(I-1,JEL,IEL))
 6021     CONTINUE
          CIRCX(JEL,IEL) = CIRCXS
          CIRCY(JEL,IEL) = CIRCYS
          CIRCF(JEL,IEL) = CIRCFS
          CIRCA(JEL,IEL) = CIRCAS
 602    CONTINUE
C     
        IF(SHARP(IEL)) THEN
          SCS = 1.0
          SDS = 0.0
        ELSE
          SCS = ANTE(IEL)/DSTE(IEL)
          SDS = ASTE(IEL)/DSTE(IEL)
        ENDIF
C     
C------TE panel source and vorticity strengths
        DO 604 JEL=1, NEL
          GAMTEX = -.5*(GAMX(1,JEL,IEL) - GAMX(NA(IEL),JEL,IEL))*SDS
          GAMTEY = -.5*(GAMY(1,JEL,IEL) - GAMY(NA(IEL),JEL,IEL))*SDS
          GAMTEF = -.5*(GAMF(1,JEL,IEL) - GAMF(NA(IEL),JEL,IEL))*SDS
          GAMTEA = -.5*(GAMA(1,JEL,IEL) - GAMA(NA(IEL),JEL,IEL))*SDS
          CIRCX(JEL,IEL) = CIRCX(JEL,IEL) + GAMTEX*DSTE(IEL)
          CIRCY(JEL,IEL) = CIRCY(JEL,IEL) + GAMTEY*DSTE(IEL)
          CIRCF(JEL,IEL) = CIRCF(JEL,IEL) + GAMTEF*DSTE(IEL)
          CIRCA(JEL,IEL) = CIRCA(JEL,IEL) + GAMTEA*DSTE(IEL)
 604    CONTINUE
C     
 60   CONTINUE
C     
      RETURN
      END ! PANSEN


      SUBROUTINE XYUPAN(I,IEL,XX,YY,SS,UI)
      INCLUDE 'AIRPAN.INC'
C
      SS = SA(I,IEL)
      XX = XA(I,IEL)
      YY = YA(I,IEL)
      UI = GAM(I,IEL)
C
      RETURN
      END ! XYUPAN



      SUBROUTINE GMILIN(I,IEL,JL,JR,GMI_GL,GMI_GR)
      INCLUDE 'AIRPAN.INC'
C
C---- find "direct" endpoints JL,JR of the inverse segment containing I
      DO 10 JL=I-1, 1, -1
        IF(.NOT. LINVRS(JL,IEL)) GO TO 11
   10 CONTINUE
   11 CONTINUE
C
      DO 20 JR=I+1, NA(IEL)
        IF(.NOT. LINVRS(JR,IEL)) GO TO 21
   20 CONTINUE
   21 CONTINUE
C
C---- GAM(I,IEL) is defined to be linearly interpolated between JL, JR
      GMI_GL = (SA(JR,IEL) - SA(I ,IEL)) / (SA(JR,IEL) - SA(JL,IEL))
      GMI_GR = (SA(I ,IEL) - SA(JL,IEL)) / (SA(JR,IEL) - SA(JL,IEL))
C
cC---- GAM(I,IEL) is interpolated between JL, JR via high-order polynomial
c      FLIN = (SA(I ,IEL) - SA(JL,IEL)) / (SA(JR,IEL) - SA(JL,IEL))
c      FCUB = 3.0*FLIN**2 - 2.0*FLIN**3
cC
c      IF(XA(JR,IEL) .GT. XA(JL,IEL)) THEN
cC----- lower surface
c       GMI_GL = 1.0 - FCUB**6
c       GMI_GR =       FCUB**6
c      ELSE
cC----- upper surface
c       GMI_GL =       (1.0 - FCUB)**6
c       GMI_GR = 1.0 - (1.0 - FCUB)**6
c      ENDIF
C
      RETURN
      END



      SUBROUTINE PSIPHI(I,IE,XI,YI,ANGL1,PSII,PHII,UI,VI)
C--------------------------------------------------------------------------
C     Calculates current streamfunction Psi and potential Phi at 
C     field point XI,YI due to freestream and all bound vorticity 
C     Gam on the airfoil. 
C
C  Input:
C    I      node index of panel node at XI,YI      ( 0 if not a panel node)
C    IE     element index of panel node at XI,YI   ( 0 if not a panel node)
C    XI,YI  point coordinates
C    ANGL1  angles from point XI,YI to J=1 points on all elements
C
C   Output:
C    PSII   streamfunction at XI,YI
C    PHII   potential at XI,YI
C    UI,VI  velocity components at XI,YI
C
C    PSI_G   dPsi/dGam
C    PHI_G   dPhi/dGam
C
C    PSI_XE  dPsi/dXelement
C    PSI_YE  dPsi/dYelement
C    PSI_FE  dPsi/dFelement
C    PSI_AE  dPsi/dAelement
C
C    PHI_XE  dPhi/d...
C--------------------------------------------------------------------------
      INCLUDE 'AIRPAN.INC'
      LOGICAL ELSAME
      DIMENSION ANGL1(2*IEX)
      DIMENSION AVEC(IQX)
      COMPLEX*8  ZI,ZO,ZP, DZ,DZINV,
     &           DZO, DZP, GDZO, GDZP,
     &           FIS,FIS_ZI,FIS_ZO,FIS_ZP,
     &           FID,FID_ZI,FID_ZO,FID_ZP,
     &           FI , FI_ZI, FI_ZO, FI_ZP, FI_DZ, FI_DS,
     &           FI_SD,FI_GS,FI_GD,FI_SIG,FI_GAM,
     &           COMSUM,COMDIF
C
      DATA PI, QOPI / 3.1415926535897932384, 0.079577471545947671 /
C
      IO = I
      IEL = IE
C
      COSA = COS(ALFA)
      SINA = SIN(ALFA)
C
      PSII = 0.
      PHII = 0.
      UI   = 0.
      VI   = 0.
C
      DO 3 JEL=1, NEL
        DO 31 JO=1, NA(JEL)
          PSI_G(JO,JEL) = 0.
          PHI_G(JO,JEL) = 0.
   31   CONTINUE
        PSI_XE(JEL) = 0.
        PSI_YE(JEL) = 0.
        PSI_FE(JEL) = 0.
        PSI_AE(JEL) = 0.
        PHI_XE(JEL) = 0.
        PHI_YE(JEL) = 0.
        PHI_FE(JEL) = 0.
        PHI_AE(JEL) = 0.
    3 CONTINUE
C
C
      ZI = CMPLX( XI , YI )
C
C---- go over all elements, adding up contributions to Psi and Phi
      DO 900 JT=1, NELT
C
      JEL = JELT(JT)
C
      IF(SHARP(JEL)) THEN
       SCS = 1.0
       SDS = 0.0
      ELSE
       SCS = ANTE(JEL)/DSTE(JEL)
       SDS = ASTE(JEL)/DSTE(JEL)
      ENDIF
C
      ELSAME = IEL.EQ.JEL .AND. YSGN(JT).GT.0.0
C
C---- set angle AVEC of vectors from XI,YI point to all (JO,JEL) points 
C     (this array is set now to allow the  DO 10  loop to vectorize)
C
      IF(ELSAME .AND. IO.NE.0) THEN
C----- we're on the element which contains the field point
C
C----- set angles for points up to the field point
       AVEC(1) = ANGL1(JT)
       DO 61 JO=2, IO-1
         RX1 = XI - XA(JO,IEL)
         RY1 = YI - YA(JO,IEL)
         AVEC(JO) = ATANC(RX1,RY1,AVEC(JO-1))
   61  CONTINUE
C
C----- angle at the field point uses normal vector
       IF(IO.GT.1) AVEC(IO) = ATANC(NX(IO,IEL),NY(IO,IEL),AVEC(IO-1))
C
C----- set angles for points past the field point
       IF(SHARP(IEL) .AND. (IO.EQ.1 .OR. IO.EQ.NA(IEL))) THEN
         DO 62 JO=IO+1, NA(IEL)-1
           RX1 = XI - XA(JO,IEL)
           RY1 = YI - YA(JO,IEL)
           AVEC(JO) = ATANC(RX1,RY1,AVEC(JO-1))
   62    CONTINUE
         JO = NA(IEL)
         AVEC(JO) = ATANC(NX(JO,IEL),NY(JO,IEL),AVEC(JO-1))
       ELSE
         DO 63 JO=IO+1, NA(IEL)
           RX1 = XI - XA(JO,IEL)
           RY1 = YI - YA(JO,IEL)
           AVEC(JO) = ATANC(RX1,RY1,AVEC(JO-1))
   63    CONTINUE
       ENDIF
C
      ELSE
C----- the field point is not on the current element JEL
C
C----- just set all the angles
       AVEC(1) = ANGL1(JT)
       DO 8 JO=2, NA(JEL)
         RX1 = XI -  XA(JO,JEL)
         RY1 = YI - (YA(JO,JEL)*YSGN(JT) + YDEL(JT))
         AVEC(JO) = ATANC(RX1,RY1,AVEC(JO-1))
    8  CONTINUE
C
      ENDIF
C
C---- set first point angle again for TE panel streamfunction evaluation
      IF(IO .EQ. 0) THEN
       JO = 1
       RX1 = XI -  XA(JO,JEL)
       RY1 = YI - (YA(JO,JEL)*YSGN(JT) + YDEL(JT))
       JO = NA(JEL)+1
       AVEC(JO) = ATANC(RX1,RY1,AVEC(JO-1))
      ELSE
       JO = NA(JEL)+1
       AVEC(JO) = AVEC(1)
      ENDIF
C
C---- see if first and last angles are the same
      DAVEC = AVEC(NA(JEL)+1) - AVEC(1) 
      IF(ABS(DAVEC) .GT. 0.01) THEN
C----- they are not ... we're inside element JEL
       UI = 0.0
       VI = 0.0
       RETURN
      ENDIF
C
C---- go over all panels in JEL'th element
      DO 10 JO=1, NA(JEL)
        JP = JO+1
C
        JM = JO-1
        JQ = JP+1
C
        IF(JO.EQ.1) THEN
         JM = JO
        ELSE IF(JO.EQ.NA(JEL)-1) THEN
         JQ = JP
        ELSE IF(JO.EQ.NA(JEL)  ) THEN
         JP = 1
         IF((XA(JO,JEL)-XA(JP,JEL))**2 + (YA(JO,JEL)-YA(JP,JEL))**2
     &      .LT. 1.0E-8) GO TO 12
        ENDIF
C
        ZO = CMPLX( XA(JO,JEL) , YA(JO,JEL)*YSGN(JT) + YDEL(JT) )
        ZP = CMPLX( XA(JP,JEL) , YA(JP,JEL)*YSGN(JT) + YDEL(JT) )
C
        DZ = ZP - ZO
        DS = ABS(DZ)
C
        DZINV = 1.0 / DZ
C
        DZO = ZO - ZI
        DZP = ZP - ZI
C
        DZOSQ =  REAL(DZO)**2 + IMAG(DZO)**2
        DZPSQ =  REAL(DZP)**2 + IMAG(DZP)**2
C
        GO = 0.
        GP = 0.
        IF(DZOSQ .NE. 0.0) GO = 0.5*LOG( DZOSQ )
        IF(DZPSQ .NE. 0.0) GP = 0.5*LOG( DZPSQ )
C
        TO = -AVEC(JO)   + 0.5*PI
        TP = -AVEC(JO+1) + 0.5*PI
C
        GDZO = CMPLX( GO , TO )
        GDZP = CMPLX( GP , TP )
C
        IF(JO.EQ.NA(JEL)) GO TO 11
C
C------ calculate vortex panel contribution to complex potential
        FIS    =  DZP*GDZP - DZO*GDZO  -  DZ
        FIS_ZI =     -GDZP +     GDZO
        FIS_ZO =                -GDZO
        FIS_ZP =      GDZP
C
        FID    =  0.5*(DZO**2-DZP**2)
     &         +  (GDZO-GDZP)* DZO*DZP  + DZ*(DZO+DZP)
        FID_ZI = -(GDZO-GDZP)*(DZO+DZP) - DZ* 2.0
        FID_ZO =  (GDZO-GDZP)     *DZP  + DZ
        FID_ZP =  (GDZO-GDZP)* DZO      + DZ
C
        SDIF = SIG(JP,JEL) - SIG(JO,JEL)
        GSUM = GAM(JP,JEL) + GAM(JO,JEL)
        GDIF = GAM(JP,JEL) - GAM(JO,JEL)
C
C------ set complex sources
        COMSUM = CMPLX( 2.0*SDIF , GSUM*DS*YSGN(JT) )
        COMDIF = CMPLX( 0.0E0    , GDIF*DS*YSGN(JT) )
C
        FI    = QOPI*(COMSUM*FIS    + COMDIF*FID   *DZINV)*DZINV
        FI_DZ =-QOPI*(COMSUM*FIS + 2.*COMDIF*FID   *DZINV)*DZINV**2
        FI_DS = QOPI*(  GSUM*FIS +      GDIF*FID   *DZINV)*DZINV
     &        * (0.0,1.0)*YSGN(JT)
C
        FI_ZI = QOPI*(COMSUM*FIS_ZI + COMDIF*FID_ZI*DZINV)*DZINV
        FI_ZO = QOPI*(COMSUM*FIS_ZO + COMDIF*FID_ZO*DZINV)*DZINV - FI_DZ
        FI_ZP = QOPI*(COMSUM*FIS_ZP + COMDIF*FID_ZP*DZINV)*DZINV + FI_DZ
C
C
        FI_SD = QOPI*CMPLX(2.0E0,0.0E0      )*FIS*DZINV
        FI_GS = QOPI*CMPLX(0.0E0,DS*YSGN(JT))*FIS*DZINV
        FI_GD = QOPI*CMPLX(0.0E0,DS*YSGN(JT))*FID*DZINV**2
C
        PSII = PSII + IMAG(FI)
        PHII = PHII + REAL(FI)
C
        PSI_SP =             + IMAG(FI_SD)
        PSI_SO =             - IMAG(FI_SD)
        PSI_GP = IMAG(FI_GS) + IMAG(FI_GD)
        PSI_GO = IMAG(FI_GS) - IMAG(FI_GD)
C
        PHI_SP =             + REAL(FI_SD)
        PHI_SO =             - REAL(FI_SD)
        PHI_GP = REAL(FI_GS) + REAL(FI_GD)
        PHI_GO = REAL(FI_GS) - REAL(FI_GD)
C
        PSI_XE(IEL) = PSI_XE(IEL) + IMAG(FI_ZI)
        PSI_YE(IEL) = PSI_YE(IEL) + REAL(FI_ZI)
        PSI_FE(IEL) = PSI_FE(IEL) + IMAG(FI_ZI*ZI)
        PSI_AE(IEL) = PSI_AE(IEL) + REAL(FI_ZI*ZI)
C
        PSI_XE(JEL) = PSI_XE(JEL) + IMAG(FI_ZO    + FI_ZP   )
        PSI_YE(JEL) = PSI_YE(JEL) + REAL(FI_ZO    + FI_ZP   )
        PSI_FE(JEL) = PSI_FE(JEL) + IMAG(FI_ZO*ZO + FI_ZP*ZP + FI_DS*DS)
        PSI_AE(JEL) = PSI_AE(JEL) + REAL(FI_ZO*ZO + FI_ZP*ZP)
C
        PHI_XE(IEL) = PHI_XE(IEL) + REAL(FI_ZI   )
        PHI_YE(IEL) = PHI_YE(IEL) - IMAG(FI_ZI   )
        PHI_FE(IEL) = PHI_FE(IEL) + REAL(FI_ZI*ZI)
        PHI_AE(IEL) = PHI_AE(IEL) - IMAG(FI_ZI*ZI)
C
        PHI_XE(JEL) = PHI_XE(JEL) + REAL(FI_ZO    + FI_ZP   )
        PHI_YE(JEL) = PHI_YE(JEL) - IMAG(FI_ZO    + FI_ZP   )
        PHI_FE(JEL) = PHI_FE(JEL) + REAL(FI_ZO*ZO + FI_ZP*ZP + FI_DS*DS)
        PHI_AE(JEL) = PHI_AE(JEL) - IMAG(FI_ZO*ZO + FI_ZP*ZP)
C
C
C------ increment  dPsi/dGam  and/or  dPsi/dSig  influence coefficients
        IF(LINVRS(JP,JEL)) THEN
         CALL GMILIN(JP,JEL,JL,JR,GP_GL,GP_GR)
C
         PSI_G(JP,JEL) = PSI_G(JP,JEL) + PSI_SP
         PSI_G(JL,JEL) = PSI_G(JL,JEL) + PSI_GP*GP_GL
         PSI_G(JR,JEL) = PSI_G(JR,JEL) + PSI_GP*GP_GR
C
         PHI_G(JP,JEL) = PHI_G(JP,JEL) + PHI_SP
         PHI_G(JL,JEL) = PHI_G(JL,JEL) + PHI_GP*GP_GL
         PHI_G(JR,JEL) = PHI_G(JR,JEL) + PHI_GP*GP_GR
        ELSE
         PSI_G(JP,JEL) = PSI_G(JP,JEL) + PSI_GP
         PHI_G(JP,JEL) = PHI_G(JP,JEL) + PHI_GP
        ENDIF
C
        IF(LINVRS(JO,JEL)) THEN
         CALL GMILIN(JO,JEL,JL,JR,GO_GL,GO_GR)
C
         PSI_G(JO,JEL) = PSI_G(JO,JEL) + PSI_SO
         PSI_G(JL,JEL) = PSI_G(JL,JEL) + PSI_GO*GO_GL
         PSI_G(JR,JEL) = PSI_G(JR,JEL) + PSI_GO*GO_GR
C
         PHI_G(JO,JEL) = PHI_G(JO,JEL) + PHI_SO
         PHI_G(JL,JEL) = PHI_G(JL,JEL) + PHI_GO*GO_GL
         PHI_G(JR,JEL) = PHI_G(JR,JEL) + PHI_GO*GO_GR
        ELSE
         PSI_G(JO,JEL) = PSI_G(JO,JEL) + PSI_GO
         PHI_G(JO,JEL) = PHI_G(JO,JEL) + PHI_GO
        ENDIF
C
        UI   = UI + REAL(FI_ZI)
        VI   = VI - IMAG(FI_ZI)
C
   10 CONTINUE ! with next panel
C
   11 CONTINUE
C
      FIS    =  DZP*GDZP - DZO*GDZO  -  DZ
      FIS_ZI =     -GDZP +     GDZO
      FIS_ZO =           - 1.0-GDZO  + 1.0
      FIS_ZP =  1.0+GDZP             - 1.0
C
      SIGTE(JEL) =  0.5*SCS*(GAM(JP,JEL) - GAM(JO,JEL))
      GAMTE(JEL) = -0.5*SDS*(GAM(JP,JEL) - GAM(JO,JEL))
C
      COMSUM = CMPLX( SIGTE(JEL) , GAMTE(JEL)*YSGN(JT) ) * DS
C
      FI    = QOPI*2.0*COMSUM*FIS   *DZINV
      FI_DZ =-QOPI*2.0*COMSUM*FIS   *DZINV*DZINV
      FI_DS = QOPI*2.0       *FIS   *DZINV
     &                *CMPLX( SIGTE(JEL) , GAMTE(JEL)*YSGN(JT) )
C
      FI_ZI = QOPI*2.0*COMSUM*FIS_ZI*DZINV
      FI_ZO = QOPI*2.0*COMSUM*FIS_ZO*DZINV - FI_DZ
      FI_ZP = QOPI*2.0*COMSUM*FIS_ZP*DZINV + FI_DZ
C
      FI_SIG = QOPI*2.0*CMPLX(DS   ,0.0E0      )*FIS*DZINV
      FI_GAM = QOPI*2.0*CMPLX(0.0E0,DS*YSGN(JT))*FIS*DZINV
C
C---- TE panel contribution to Psi and Phi
      PSII = PSII + IMAG(FI)
      PHII = PHII + REAL(FI)
C
C---- increment dPsi/dGam influence coefficients
      PSI_G(JP,JEL) = PSI_G(JP,JEL) + 0.5*SCS*IMAG(FI_SIG)
     &                              - 0.5*SDS*IMAG(FI_GAM)
      PSI_G(JO,JEL) = PSI_G(JO,JEL) - 0.5*SCS*IMAG(FI_SIG)
     &                              + 0.5*SDS*IMAG(FI_GAM)
C
      PHI_G(JP,JEL) = PHI_G(JP,JEL) + 0.5*SCS*REAL(FI_SIG)
     &                              - 0.5*SDS*REAL(FI_GAM)
      PHI_G(JO,JEL) = PHI_G(JO,JEL) - 0.5*SCS*REAL(FI_SIG)
     &                              + 0.5*SDS*REAL(FI_GAM)
C
      PSI_XE(IEL) = PSI_XE(IEL) + IMAG(FI_ZI)
      PSI_YE(IEL) = PSI_YE(IEL) + REAL(FI_ZI)
      PSI_FE(IEL) = PSI_FE(IEL) + IMAG(FI_ZI*ZI)
      PSI_AE(IEL) = PSI_AE(IEL) + REAL(FI_ZI*ZI)
C
      PSI_XE(JEL) = PSI_XE(JEL) + IMAG(FI_ZO    + FI_ZP   )
      PSI_YE(JEL) = PSI_YE(JEL) + REAL(FI_ZO    + FI_ZP   )
      PSI_FE(JEL) = PSI_FE(JEL) + IMAG(FI_ZO*ZO + FI_ZP*ZP + FI_DS*DS)
      PSI_AE(JEL) = PSI_AE(JEL) + REAL(FI_ZO*ZO + FI_ZP*ZP)
C
      PHI_XE(IEL) = PHI_XE(IEL) + REAL(FI_ZI   )
      PHI_YE(IEL) = PHI_YE(IEL) - IMAG(FI_ZI   )
      PHI_FE(IEL) = PHI_FE(IEL) + REAL(FI_ZI*ZI)
      PHI_AE(IEL) = PHI_AE(IEL) - IMAG(FI_ZI*ZI)
C
      PHI_XE(JEL) = PHI_XE(JEL) + REAL(FI_ZO    + FI_ZP   )
      PHI_YE(JEL) = PHI_YE(JEL) - IMAG(FI_ZO    + FI_ZP   )
      PHI_FE(JEL) = PHI_FE(JEL) + REAL(FI_ZO*ZO + FI_ZP*ZP + FI_DS*DS)
      PHI_AE(JEL) = PHI_AE(JEL) - IMAG(FI_ZO*ZO + FI_ZP*ZP)
C
C
      UI   = UI   + REAL(FI_ZI)
      VI   = VI   - IMAG(FI_ZI)
C
   12 CONTINUE
C
  900 CONTINUE ! with next element
C
 1000 CONTINUE ! with next image
C
C**** Freestream terms
      FI    = QINF*CMPLX(COSA,-SINA)*ZI
      FI_ZI = QINF*CMPLX(COSA,-SINA)
C
      PSII = PSII + IMAG(FI)
      PHII = PHII + REAL(FI)
C
      UI   = UI   + REAL(FI_ZI)
      VI   = VI   - IMAG(FI_ZI)
C
      PSI_CA =  QINF*YI
      PSI_SA = -QINF*XI
      PHI_CA =  QINF*XI
      PHI_SA =  QINF*YI
C
      PSI_XE(IEL) = PSI_XE(IEL) + IMAG(FI_ZI)
      PSI_YE(IEL) = PSI_YE(IEL) + REAL(FI_ZI)
      PSI_FE(IEL) = PSI_FE(IEL) + IMAG(FI_ZI*ZI)
      PSI_AE(IEL) = PSI_AE(IEL) + REAL(FI_ZI*ZI)
C
      PHI_XE(IEL) = PHI_XE(IEL) + REAL(FI_ZI   )
      PHI_YE(IEL) = PHI_YE(IEL) - IMAG(FI_ZI   )
      PHI_FE(IEL) = PHI_FE(IEL) + REAL(FI_ZI*ZI)
      PHI_AE(IEL) = PHI_AE(IEL) - IMAG(FI_ZI*ZI)
C
      RETURN
      END ! PSIPHI



      SUBROUTINE ALSET(ANGLE)
C----------------------------------------------------------
C     Sets surface vorticity distribution for specified
C     freestream flow angle ANG.  The total configuration
C     circulation is returned in CIRTOT.
C----------------------------------------------------------
      INCLUDE 'AIRPAN.INC'
C
      ALFA = ANGLE
      SINA = SIN(ALFA)
      COSA = COS(ALFA)
C
C---- set surface vorticity distributions and add up circulation
      CIRC = 0.0
      DO 20 IEL=1, NEL
C
C------ superimpose  alpha = 0,90  distributions for this alpha
        DO 210 I=1, NA(IEL)
          GAM(I,IEL) = GAMU(I,1,IEL)*COSA + GAMU(I,2,IEL)*SINA
          SIG(I,IEL) = SIGU(I,1,IEL)*COSA + SIGU(I,2,IEL)*SINA
ccc          write(*,*) i, xa(i,iel), ya(i,iel), gam(i,iel), sig(i,iel)
  210   CONTINUE
C
        I = NA(IEL)+1
        PSIO(IEL) = GAMU(I,1,IEL)*COSA + GAMU(I,2,IEL)*SINA
C
        IF(SHARP(IEL)) THEN
         SCS = 1.0
         SDS = 0.0
        ELSE
         SCS = ANTE(IEL)/DSTE(IEL)
         SDS = ASTE(IEL)/DSTE(IEL)
        ENDIF
C
C------ TE panel source and vorticity strengths
        SIGTE(IEL) = 0.5*(GAM(1,IEL) - GAM(NA(IEL),IEL))*SCS
        GAMTE(IEL) = -.5*(GAM(1,IEL) - GAM(NA(IEL),IEL))*SDS
C
C------ element circulation
        CIRCEL(IEL) = CIRCU(1,IEL)*COSA + CIRCU(2,IEL)*SINA
C
        CIRC = CIRC + CIRCEL(IEL)
C
   20 CONTINUE
C
      CIRTOT = CIRC
C
      RETURN
      END ! ALSET


      SUBROUTINE CLSET(CLSPEC,ANGLE,MINF)
      INCLUDE 'AIRPAN.INC'
C
C---- find alpha needed to achieve specified CL
C
      ALFA = ANGLE
      ITMAX = 10
C
C---- Newton loop for alpha to get specified inviscid CL
      DO 100 ITAL=1, ITMAX
C
        COSA = COS(ALFA)
        SINA = SIN(ALFA)
        DO 10 IEL=1, NEL
          DO 101 I=1, NA(IEL)
            GAM(I,IEL) = COSA*GAMU(I,1,IEL) + SINA*GAMU(I,2,IEL)
  101     CONTINUE
   10   CONTINUE
C
C------ get corresponding CL and CL_alpha
        CALL CLCALC(CL,CL_ALF,CM,MINF,CLEL,CDEL,CMEL,0.25,0.0,.FALSE.)
C
        DALFA = (CLSPEC - CL) / CL_ALF
        ALFA = ALFA + DALFA
C
        IF(ABS(DALFA).LE.1.0E-6) GO TO 110
  100 CONTINUE
      WRITE(*,*) 'CLSET:  CL convergence failed'
  110 CONTINUE
C
      ANGLE = ALFA
C
      RETURN
      END ! CLSET


      SUBROUTINE CLCALC(CLT,CLT_ALF,CMT,MINF,CLELT,CDELT,CMELT,
     &                  XCEN,YCEN,LDXY)
C-----------------------------------------------------------
C     Integrates surface pressures to get CL and CM
C     Also calculates dCL/dAlpha for prescribed-CL routines
C-----------------------------------------------------------
      INCLUDE 'AIRPAN.INC'
      DIMENSION CLELT(IEX),CDELT(IEX),CMELT(IEX)
      LOGICAL LDXY
C
cC---- moment centroid location
c      XCEN = 0.25
c      YCEN = 0.
C
      SINA = SIN(ALFA)
      COSA = COS(ALFA)
C
      BETA = SQRT(1.0 - MINF**2)
      BFAC = 0.5*MINF**2 / (1.0 + BETA)
C
      CL = 0.
      CD = 0.
      CM = 0.
C
      CL_ALF = 0.
C
      DO 9000 IEL=1, NEL
C
      CLEL(IEL) = 0.0
      CDEL(IEL) = 0.0
      CMEL(IEL) = 0.0
C
      I = 1
      X1   =  XA(I,IEL)
      Y1   =  YA(I,IEL)
      GAM1 = GAM(I,IEL)
      IF(LDXY) CALL XYG(I,IEL,X1,Y1,GAM1)
C
      CGINC = 1.0 - (GAM1/QINF)**2
      CPG1 = CGINC / (BETA + BFAC*CGINC)
C
      GAM_ALF = -SINA*GAMU(I,1,IEL) + COSA*GAMU(I,2,IEL)
      CPI_GAM = -2.0*GAM1/QINF**2
      CPC_CPI = (1.0 - BFAC*CPG1)/ (BETA + BFAC*CGINC)
      CPG1_A = CPC_CPI*CPI_GAM*GAM_ALF
C
      DO 10 I=1, NA(IEL)
        IP = I+1
        IF(I.EQ.NA(IEL)) IP = 1
C
        X2   =  XA(IP,IEL)
        Y2   =  YA(IP,IEL)
        GAM2 = GAM(IP,IEL)
        IF(LDXY) CALL XYG(I,IEL,X2,Y2,GAM2)
C
        CGINC = 1.0 - (GAM2/QINF)**2
        CPG2 = CGINC / (BETA + BFAC*CGINC)
C
        GAM_ALF = -SINA*GAMU(IP,1,IEL) + COSA*GAMU(IP,2,IEL)
        CPI_GAM = -2.0*GAM2/QINF**2
        CPC_CPI = (1.0 - BFAC*CPG2)/ (BETA + BFAC*CGINC)
        CPG2_A = CPC_CPI*CPI_GAM*GAM_ALF
C
        DX = (X2-X1)*COSA + (Y2-Y1)*SINA
        DY = (Y2-Y1)*COSA - (X2-X1)*SINA
        DG = CPG2 - CPG1
C
        AX = (0.5*(X2+X1)-XCEN)*COSA  +  (0.5*(Y2+Y1)-YCEN)*SINA
        AY = (0.5*(Y2+Y1)-YCEN)*COSA  -  (0.5*(X2+X1)-XCEN)*SINA
        AG = 0.5*(CPG2 + CPG1)
C
        DX_A = (Y2-Y1)*COSA - (X2-X1)*SINA
        AG_A = 0.5*(CPG2_A + CPG1_A)
C
        CLEL(IEL) = CLEL(IEL) + DX* AG
        CDEL(IEL) = CDEL(IEL) - DY* AG
        CMEL(IEL) = CMEL(IEL) - DX*(AG*AX + DG*DX/12.0)
     &                        - DY*(AG*AY + DG*DY/12.0)
C
        CL_ALF = CL_ALF + DX*AG_A + AG*DX_A
C
        X1 = X2
        Y1 = Y2
        GAM1 = GAM2
        CPG1 = CPG2
        CPG1_A = CPG2_A
   10 CONTINUE
C
      CL = CL + CLEL(IEL)
      CD = CD + CDEL(IEL)
      CM = CM + CMEL(IEL)
C
      CLELT(IEL) = CLEL(IEL)
      CDELT(IEL) = CDEL(IEL)
      CMELT(IEL) = CMEL(IEL)
C
 9000 CONTINUE
C
      CLT = CL
      CMT = CM
      CLT_ALF = CL_ALF
C
      RETURN
      END ! CLCALC



      SUBROUTINE CLRXYA(IEL)
      INCLUDE 'AIRPAN.INC'
C
      DXEL(IEL) = 0.
      DYEL(IEL) = 0.
      DFEL(IEL) = 0.
      DAEL(IEL) = 0.
C
      RETURN
      END



      SUBROUTINE ADDXYA(IEL,DX,DY,DF,DA)
      INCLUDE 'AIRPAN.INC'
C
      EFE = EXP(DF)
      SAE = SIN(-DA)
      CAE = COS(-DA)
      DXE = DXEL(IEL)
      DYE = DYEL(IEL)
C
      DXEL(IEL) = DXE*CAE - DYE*SAE + DX
      DYEL(IEL) = DXE*SAE + DYE*CAE + DY
      DFEL(IEL) = DFEL(IEL) + DF
      DAEL(IEL) = DAEL(IEL) - DA
C
      RETURN
      END


      SUBROUTINE XYG(IO,IEL,XX,YY,GG)
C--------------------------------------
C     Returns perturbed position XX,YY
C     and vorticity GG of node IO,IEL.
C--------------------------------------
      INCLUDE 'AIRPAN.INC'
C
      I = IO
      L = IEL
C
      EDF = EXP(DFEL(L))
      SDA = SIN(DAEL(L))
      CDA = COS(DAEL(L))
C
      XF = EDF*XA(I,L)
      YF = EDF*YA(I,L)
C
      XX = CDA*XF - SDA*YF + DXEL(L)
      YY = SDA*XF + CDA*YF + DYEL(L)
C
      GG = GAM(I,L)
      DO 10 JEL=1, NEL
        GG = GG + GAMX(I,JEL,L)*DXEL(JEL)
     &          + GAMY(I,JEL,L)*DYEL(JEL)
     &          + GAMF(I,JEL,L)*DFEL(JEL)
     &          + GAMA(I,JEL,L)*DAEL(JEL)
 10   CONTINUE
C
      RETURN
      END



      SUBROUTINE VCENT(XCEN,YCEN,DOUX,DOUY)
C------------------------------------------------
C     Calculates vorticity centroid coordinates
C------------------------------------------------
      INCLUDE 'AIRPAN.INC'
C
      V1SUM = 0.0
      V2SUM = 0.0
      X1SUM = 0.0
      X2SUM = 0.0
      Y1SUM = 0.0
      Y2SUM = 0.0
C
      DO 100 IEL=1, NEL
        DO 10 I=2, NA(IEL)
          DS = SA(I,IEL) - SA(I-1,IEL)
          V1AVG = DS * (GAMU(I,1,IEL) + GAMU(I-1,1,IEL) )*0.5
          V2AVG = DS * (GAMU(I,2,IEL) + GAMU(I-1,2,IEL) )*0.5
          XAVG = ( XA(I,IEL) +  XA(I-1,IEL) )*0.5
          YAVG = ( YA(I,IEL) +  YA(I-1,IEL) )*0.5
C
          V1SUM = V1SUM + V1AVG
          V2SUM = V2SUM + V2AVG
          X1SUM = X1SUM + V1AVG*XAVG
          X2SUM = X2SUM + V2AVG*XAVG
          Y1SUM = Y1SUM + V1AVG*YAVG
          Y2SUM = Y2SUM + V2AVG*YAVG
   10   CONTINUE
C
C------ TE panel vorticity
        VTE1 = -.5*(GAMU(1,1,IEL) - GAMU(NA(IEL),1,IEL))*ASTE(IEL)
        VTE2 = -.5*(GAMU(1,2,IEL) - GAMU(NA(IEL),2,IEL))*ASTE(IEL)
C
        XAVG = ( XA(NA(IEL),IEL) +  XA(1,IEL) )*0.5
        YAVG = ( YA(NA(IEL),IEL) +  YA(1,IEL) )*0.5
C
        V1SUM = V1SUM + VTE1
        V2SUM = V2SUM + VTE2
        X1SUM = X1SUM + VTE1*XAVG
        X2SUM = X2SUM + VTE2*XAVG
        Y1SUM = Y1SUM + VTE1*YAVG
        Y2SUM = Y2SUM + VTE2*YAVG
  100 CONTINUE
C
      A11 =  V1SUM
      A12 = -V2SUM
      A21 =  V2SUM
      A22 =  V1SUM
C
      RHS1 = X1SUM - Y2SUM
      RHS2 = X2SUM + Y1SUM
C
      DET   =  A11 *A22  - A12 *A21
      VCENX = (RHS1*A22  - A12 *RHS2) / DET
      VCENY = (A11 *RHS2 - RHS1*A21 ) / DET
C
      SINA = SIN(ALFA)
      COSA = COS(ALFA)
C
      DOUX =   (Y1SUM - VCENY*V1SUM)*COSA**2
     &       - (X2SUM - VCENX*V2SUM)*SINA**2
     &       + (Y2SUM - VCENY*V2SUM)*SINA*COSA
     &       - (X1SUM - VCENX*V1SUM)*COSA*SINA
      DOUY = - (X1SUM - VCENX*V1SUM)*COSA**2
     &       - (Y2SUM - VCENY*V2SUM)*SINA**2
     &       - (X2SUM - VCENX*V2SUM)*SINA*COSA
     &       - (Y1SUM - VCENY*V1SUM)*COSA*SINA
C
      XCEN = VCENX
      YCEN = VCENY
C
      RETURN
      END ! VCENT



      SUBROUTINE CPLIMS(IEL,CPMIN,CPMAX)
      INCLUDE 'AIRPAN.INC'
C
      IF(IEL.EQ.0) THEN
       IEL1 = 1
       IEL2 = NEL
      ELSE
       IEL1 = IEL
       IEL2 = IEL
      ENDIF
C
      CPMIN = 1.0-GAM(1,IEL1)**2
      CPMAX = 1.0-GAM(1,IEL1)**2
C
      DO 10 JEL=IEL1, IEL2
        DO 105 I=1, NA(JEL)
          CPMIN = MIN( CPMIN , 1.0-GAM(I,JEL)**2 )
          CPMAX = MAX( CPMAX , 1.0-GAM(I,JEL)**2 )
 105    CONTINUE
   10 CONTINUE
C
      RETURN
      END



      SUBROUTINE CPPLOT(XOFF,XFAC,POFF,PFAC,MINF,LDXY,ILIN)
C---------------------------------------------------------
C     Plots inviscid -Cp distribution on airfoil elements
C---------------------------------------------------------
      INCLUDE 'AIRPAN.INC'
      LOGICAL LDXY
C
C---- overlay work arrays to save storage
      DIMENSION XPLT(IQX), YPLT(IQX)
      EQUIVALENCE (DQ(1,1),XPLT(1)), (DQ(1,IRX/2),YPLT(1))
C
C
      BETA = SQRT(1.0 - MINF**2)
      BFAC = 0.5*MINF**2 / (1.0 + BETA)
C
      CALL NEWPEN(2)
C
      DO 10 IEL=1, NEL
        DO 105 I=1, NA(IEL)
          XBAR =  XA(I,IEL)
          YBAR =  YA(I,IEL)
          GAMB = GAM(I,IEL)
          IF(LDXY) CALL XYG(I,IEL,XBAR,YBAR,GAMB)
C
          CPINC = 1.0 - (GAMB/QINF)**2
          CPCOM = CPINC / (BETA + BFAC*CPINC)
C
          XPLT(I) = XBAR
          YPLT(I) = -CPCOM
 105    CONTINUE
C
        CALL XYLINE(NA(IEL),XPLT,YPLT,XOFF,XFAC,POFF,PFAC,ILIN)
 10   CONTINUE
C
      RETURN
      END ! CPPLOT



      SUBROUTINE PANPLT(SFAC,XOFF,YOFF)
C-----------------------------------------------------
C     Shows panel nodes on current airfoil geometry
C-----------------------------------------------------
      INCLUDE 'AIRPAN.INC'
      XMOD(XTMP) = SFAC * (XTMP - XOFF)
      YMOD(YTMP) = SFAC * (YTMP - YOFF)
C
      DSN = 0.0075
C
      DO 10 L=1, NEL
        DO 110 I=2, NA(L)
          CALL PLOT(XMOD(XA(I-1,L)),YMOD(YA(I-1,L)),3)
          CALL PLOT(XMOD(XA(I  ,L)),YMOD(YA(I  ,L)),2)
          CALL PLOT(XMOD(XA(I  ,L)) + DSN*NX(I,L),
     &              YMOD(YA(I  ,L)) + DSN*NY(I,L),2)
 110    CONTINUE
   10 CONTINUE
C
      RETURN
      END


