
      SUBROUTINE BLSYS( SIMI, TRAN, TURB, WAKE,
     &                  PAR, GAM, HSUTH,
     &                  AMCRIT, XIT,
     &                  BULE, 
     &                  VAR1 , VAR2 ,
     &                  VJ1  , VJ2  ,
     &                  PJ1  , PJ2  ,
     &                  VS1  , VS2  , VSP, VSXIT, VSRES )
C-------------------------------------------------------------------------
C
C     Sets up the Newton system governing the current Xi-interval:
C
C |      |   |       ||dCt|   |       ||dCt|   |     |       |     ||    |
C |dVSRES| = |  VS1  ||dTh| + |  VS2  ||dTh| + |VSXIT|dXIT + | VSP ||dPAR|
C |      |   |       ||dDs|   |       ||dDs|   |     |       |     ||    |
C                     |dUe|            |dUe|                        |    |
C                     |dUw|            |dUw|
C                     |dRh|            |dRh|
C                     |dCv|            |dCv|
C                     |dXi|            |dXi|
C                     |dr |            |dr |
C                     |db |            |db |
C                     |dAm|            |dAm|
C                     |dmw|_1          |dmw|_2
C            
C    3x1        3x12   12x1       3x12  12x1      3x1           3x4   4x1  
C
C     The three rows in the system contain:
C
C       1) amplification equation (or lag equation if TURB or TRAN)
C       2) x-momentum equation
C       3) KE shape parameter equation
C
C     The variables of the system (Ct,Th...) are described below.
C
C     The routine essentially takes the system from SUBROUTINE BLDIF,
C     but adds special treatment for similarity and transition intervals.
C
C.........................................................................
C
C  Input:
C  ------
C  SIMI       .TRUE. if similarity station
C  TRAN       .TRUE. if transition interval
C  TURB       .TRUE. if turbulent interval
C  WAKE       .TRUE. if wake interval
C  PAR( . )   parameter...                     2
C     (LSH)     Htot  stagnation enthalpy = a_o /(gam-1)   (same units as Ue^2)
C     (LSR)     Rtot  stagnation density  = rho_o
C     (LRE)     REtot stagnation Reynolds number / unit length
C                ( = rho_o a_o / mu_o )  ; a_o = stagnation speed of sound
C     (LRO)     rot   rotation rate    (radians/time_unit)
C  GAM        cp/cv
C  HSUTH      Sutherland's enthalpy (= cp x 110K, in same units as Ue^2)
C  AMCRIT     critical amplification ratio, used only if TRAN=t
C  XIT        transition Xi location, used only if TRAN=t
C  BULE       similarity parameter x/Ue dUe/dx,  used only if SIMI=t
C
C  VAR1,2( . )      primary variable...
C        (ICT)  Ct  max shear stress^1/2
C        (ITH)  Th  momentum thickness 
C        (IDS)  Ds  displacement thickness 
C        (IUE)  Ue  outer velocity
C        (IUW)  Uw  inner velocity (wall, wake centerline, etc.)
C        (IRH)  Rh  edge density
C        (ICV)  Cv  displacement surface curvature, + concave 
C        (IXI)  Xi  arc-length BL coordinate
C        (IRR)  r   radius
C        (IBB)  b   streamtube thickness
C        (IAM)  Am  amplification variable
C        (IMW)  mw  wall suction mass flow rate
C
C  VJ1,2      secondary variable and derivative arrays (see BLVAR header)
C  PJ1,2      secondary variable derivatives wrt parameters
C
C Output:
C -------
C    VS1,2   Jacobian matrix arrays   dVSRES/dVAR1,2
C    VSP     Jacobian wrt parameters
C    VSXIT   Jacobian wrt XIT  (nonzero only for transition interval)
C    VSRES   residual vector
C
C-------------------------------------------------------------------------
      IMPLICIT REAL (A-H,M,O-Z)
      INCLUDE 'INDEX.INC'
C
      LOGICAL SIMI, TRAN, TURB, WAKE
C
      DIMENSION PAR(LTOT), VAR1(ITOT), VAR2(ITOT)
      DIMENSION VJ1(0:ITOT,JTOT), VJ2(0:ITOT,JTOT),
     &          PJ1(  LTOT,JTOT), PJ2(  LTOT,JTOT)
      DIMENSION VS1(3,ITOT), VS2(3,ITOT), 
     &          VSP(3,LTOT), VSXIT(3), VSRES(3)
C
C---- local arrays
      DIMENSION VART(ITOT), 
     &          VART_XIT(ITOT),
     &          VART_PAR(ITOT,LTOT),
     &          VART_VAR1(ITOT,ITOT),
     &          VART_VAR2(ITOT,ITOT)
      DIMENSION CTT_VART(ITOT), CTT_PAR(LTOT),
     &          CTT_VAR1(ITOT),
     &          CTT_VAR2(ITOT)
      DIMENSION VJT(0:ITOT,JTOT),
     &          PJT(  LTOT,JTOT)
C
      DIMENSION VL1(3,ITOT), VLT(3,ITOT), VLP(3,LTOT), VLRES(3)
      DIMENSION VTT(3,ITOT), VT2(3,ITOT), VTP(3,LTOT), VTRES(3)
C
C
      DO 10 K=1, 3
        VSRES(K)   = 0.
        DO 101 L=1, ITOT
          VS1(K,L) = 0.
          VS2(K,L) = 0.
 101    CONTINUE
        DO 102 L=1, LTOT
          VSP(K,L) = 0.
 102    CONTINUE
        VSXIT(K) = 0.
 10   CONTINUE
C
C
C---- unpack parameters
      HTOT  = PAR(LSH)
      RTOT  = PAR(LSR)
      RETOT = PAR(LRE)
      ROT   = PAR(LRO)
C
C
      IF(SIMI) THEN
C====== similarity interval
C
C------ set up similarity system using "2" variables only
        IDIF = 0
        CALL BLDIF(IDIF, BULE,ROT,AMCRIT,
     &             VAR2,VAR2, VJ2,VJ2, PJ2,PJ2,
     &             VS1, VS2, VSP, VSRES)
C
C------ BLDIF was given "2" variables in place of "1" variables,
C-      so lump VS1 into VS2
        DO 20 K=1, 3
          DO 202 L=1, ITOT
            VS2(K,L) = VS1(K,L) + VS2(K,L)
            VS1(K,L) = 0.
 202      CONTINUE
 20     CONTINUE
C
      ELSE IF(TRAN) THEN
C====== transition interval
C
C------ interpolate primary variables at XI1,XI2 to transition point XIT,
C-      giving  VARTi( XIT, VAR1j, VAR2j )
        CALL VTRSET(XIT, VAR1, VAR2, 
     &              VART,
     &              VART_XIT ,
     &              VART_VAR1, VART_VAR2 )
C
C----------------------------------------------------
C-      laminar sub-interval  XI1..XIT
C
        IDIF = 1
C
C------ set laminar secondary variables at XIT
        IVAR = 1
        CALL BLVAR(IVAR,PAR,GAM,HSUTH, 0.0,0.0,
     &             VART,VJT,PJT)
C
C------ set up equation system for laminar sub-interval
        CALL BLDIF(IDIF, BULE,ROT,AMCRIT,
     &             VAR1,VART, VJ1,VJT, PJ1,PJT,
     &             VL1, VLT, VLP, VLRES)
C
C------ clear out amplification equation for laminar sub-interval
C-      (it is implicitly solved in TRCHEK, which defines XIT)
        K = 1
        VLRES(K) = 0.
        DO 40 L=1, ITOT
          VL1(K,L) = 0.
          VLT(K,L) = 0.
 40     CONTINUE
        DO 41 L=1, LTOT
          VLP(K,L) = 0.
 41     CONTINUE
C
C----------------------------------------------------
C-      turbulent sub-interval  XIT..XI2
C
        IDIF = 2
C
C------ set turbulent secondary variables at XIT
        IVAR = 2
        CALL BLVAR(IVAR, PAR ,GAM,HSUTH, 0.0,0.0,
     &             VART,VJT,PJT)
C
C------ set initial Ctau^1/2 at XIT
        CALL CTINI(VJT,PJT, CTT,CTT_VART,CTT_PAR)
C
C------ resolve CTT(VART(VAR1,VAR2,XIT),PAR)  into  CTT(VAR1,VAR2,XIT,PAR)
        DO 42 L=1, ITOT
          CTT_VAR1(L) = 0.
          CTT_VAR2(L) = 0.
          DO 422 LL=1, ITOT
            CTT_VAR1(L) = CTT_VAR1(L) + CTT_VART(LL)*VART_VAR1(LL,L)
            CTT_VAR2(L) = CTT_VAR2(L) + CTT_VART(LL)*VART_VAR2(LL,L)
 422      CONTINUE
 42     CONTINUE
C
        CTT_XIT = 0.
        DO 43 LL=1, ITOT
          CTT_XIT = CTT_XIT + CTT_VART(LL)*VART_XIT(LL)
 43     CONTINUE
C
CCC     CTT_PAR(L) = CTT_PAR(L)
C
C
C------ place initial Ctau^1/2 and its derivatives into VART.. arrays
        VART(ICT) = CTT
        DO 44 L=1, ITOT
          VART_VAR1(ICT,L) = CTT_VAR1(L)
          VART_VAR2(ICT,L) = CTT_VAR2(L)
 44     CONTINUE
        DO 46 L=1, LTOT
          VART_PAR(ICT,L) = CTT_PAR(L)
 46     CONTINUE
        VART_XIT(ICT) = CTT_XIT
C
C
C------ recalculate secondary "T" variables with new Ctau^1/2
C-      (assumes that VJT,PJT used in CTINI above will not change)
        CALL BLVAR(IVAR, PAR, GAM,HSUTH, 0.0,0.0,
     &             VART,VJT,PJT)
C
C------ set up equation system for turbulent sub-interval
        CALL BLDIF(IDIF, BULE,ROT,AMCRIT,
     &             VART,VAR2, VJT,VJ2, PJT,PJ2,
     &             VTT, VT2, VTP, VTRES)
C
C----------------------------------------------------
C-      add up laminar and turbulent sub-interval equations,
C-      the resulting system being only in terms of 1,2 variables.
C
        DO 50 K=1, 3
          VSRES(K)   = VLRES(K) + VTRES(K)
C
          DO 502 L=1, ITOT
            VS1(K,L) = VL1(K,L)
            VS2(K,L) =            VT2(K,L)
 502      CONTINUE
          DO 503 L=1, LTOT
            VSP(K,L) = VLP(K,L) + VTP(K,L)
 503      CONTINUE
          VSXIT(K) = 0.
C
C-------- resolve VART(VAR1,VAR2,PAR,XIT) dependency
          DO 504 L=1, ITOT
            RES_VART = VLT(K,L) + VTT(K,L)
            DO 5042 LL=1, ITOT
              VS1(K,LL) = VS1(K,LL) + RES_VART*VART_VAR1(L,LL)
              VS2(K,LL) = VS2(K,LL) + RES_VART*VART_VAR2(L,LL)
 5042       CONTINUE
            DO 5043 LL=1, LTOT
              VSP(K,LL) = VSP(K,LL) + RES_VART*VART_PAR(L,LL)
 5043       CONTINUE
            VSXIT(K)    = VSXIT(K)  + RES_VART*VART_XIT(L)
 504      CONTINUE
C
 50     CONTINUE
C
      ELSE
C====== laminar, turbulent, or wake interval
C
C------ set interval-type flag for calling BLDIF
        IDIF = 1
        IF(TURB) IDIF = 2
        IF(WAKE) IDIF = 3
C
        CALL BLDIF(IDIF, BULE, ROT,AMCRIT,
     &             VAR1,VAR2, VJ1,VJ2, PJ1,PJ2,
     &             VS1, VS2, VSP, VSRES)
C
      ENDIF
C
      RETURN
      END ! BLSYS



      SUBROUTINE VTRSET(XIT, VAR1, VAR2, 
     &                  VART,
     &                  VART_XIT ,
     &                  VART_VAR1,VART_VAR2 )
C--------------------------------------------------------------------
C     Interpolates external variables to transition point.
C
C     Input:  XIT           Transition location (forced of free)
C             VAR1(.)       Variables at XI1, indexed via INDEX.INC
C             VAR2(.)       Variables at XI2,    "
C
C     Output: VART(.)       Variables at XIT, indexed same as VAR1,2
C             VART_XIT (.)  dVART(.)/dXIT
C             VART_VAR1(..) dVART(.)/dVAR1(.)
C             VART_VAR2(..) dVART(.)/dVAR2(.)
C-------------------------------------------------------------------
      IMPLICIT REAL (A-H,M,O-Z)
      INCLUDE 'INDEX.INC'
      DIMENSION VAR1(ITOT), VAR2(ITOT)
      DIMENSION VART(ITOT),
     &          VART_XIT(ITOT),
     &          VART_VAR1(ITOT,ITOT), 
     &          VART_VAR2(ITOT,ITOT)
C
      XI1 = VAR1(IXI)
      XI2 = VAR2(IXI)
C
C---- weighting factor for interpolation to transition point
      WF     = (XIT-XI1)/(XI2-XI1)
      WF_XIT =  1.0     /(XI2-XI1)
      WF_XI1 = (WF -1.0)/(XI2-XI1)
      WF_XI2 = (   -WF )/(XI2-XI1)
C
      DO 10 I=1, ITOT
        VART_XIT(I) = 0.
        DO 105 L=1, ITOT
          VART_VAR1(I,L) = 0.
          VART_VAR2(I,L) = 0.
 105    CONTINUE
 10   CONTINUE
C
C---- interpolate variables to transition point and set derivatives
      DO 20 I=1, ITOT
        VART(I) = WF*VAR2(I) + (1.0-WF)*VAR1(I)
        VART_WF =    VAR2(I)       -    VAR1(I)
C
        VART_XIT(I)      =  VART_WF*WF_XIT
        VART_VAR1(I,IXI) =  VART_WF*WF_XI1
        VART_VAR2(I,IXI) =  VART_WF*WF_XI2
C
        VART_VAR1(I,I) = VART_VAR1(I,I) + (1.0-WF)
        VART_VAR2(I,I) = VART_VAR2(I,I) +      WF
 20   CONTINUE
C
C---- set transition point exactly
C-    (only linear interpolation will return XIT as itself)
      VART(IXI)     = XIT
      VART_XIT(IXI) = 1.0
      DO 30 L=1, ITOT
        VART_VAR1(IXI,L) = 0.
        VART_VAR2(IXI,L) = 0.
 30   CONTINUE
C
      RETURN
      END ! VTRSET



      SUBROUTINE CTINI(VJ,PJ, CT,CT_VAR,CT_PAR)
C--------------------------------------------------------
C     Defines initial shear-stress coefficient Ctau^1/2
C     and its sensitivities from variables in VJ array.
C--------------------------------------------------------
      IMPLICIT REAL (A-H,M,O-Z)
      INCLUDE 'INDEX.INC'
      DIMENSION CT_VAR(ITOT), CT_PAR(LTOT)
      DIMENSION VJ(0:ITOT,JTOT),PJ(LTOT,JTOT)
C
C---- set equilibrium Ctau^1/2, kinematic shape parameter Hk, and Me^2
      CQ = VJ(0,JCQ)
      HK = VJ(0,JHK)
      MS = VJ(0,JMS)
C
C---- scale factor for initial Ctau^1/2
      CTR    = 1.8*EXP(-3.3/(HK-1.0))
      CTR_HK = CTR * 3.3/(HK-1.0)**2
      CTR_MS = 0.
C
c      CTR = 0.7
c      CTR_HK = 0.0
c      CTR_MS = 0.0
C
C---- set initial Ctau^1/2 and its sensitivities
      CT = CTR*CQ
      DO 20 L=1, ITOT
        CT_VAR(L) = CTR*VJ(L,JCQ) + CQ*CTR_HK*VJ(L,JHK)
     &                            + CQ*CTR_MS*VJ(L,JMS)
 20   CONTINUE
      DO 21 L=1, LTOT
        CT_PAR(L) = CTR*PJ(L,JCQ) + CQ*CTR_HK*PJ(L,JHK)
     &                            + CQ*CTR_MS*PJ(L,JMS)
 21   CONTINUE
C
      RETURN
      END ! CTINI








