
      SUBROUTINE TRCHEK(VAR1,VAR2, VJ1,VJ2, PJ1,PJ2,
     &                  AMCRIT, XIFORC, 
     &                  KTRAN,
     &                  XIT, XIT_VAR1, XIT_VAR2, XIT_PAR, XIT_XIF,
     &                  AMPL2 )
C----------------------------------------------------------------------
C     General, all-purpose, transition-handling routine.
C
C   * Tests for transition in interval XI1..XI2
C   * Returns the amplification ratio AMPL2 implied by current variables
C   * Sets transition location XIT if free or forced transition occurs
C   * Sets derivatives of XIT with respect to primary variables
C
C......................................................................
C
C  Input:
C  ------
C    VAR1,2    primary variable arrays for interval endpoints 1,2
C    VJ1,2     secondary variable and derivative arrays
C    AMCRIT    critical amplification ratio
C    XIFORC    forced-transition XI location
C
C  Output:
C  -------
C    KTRAN     = 0  if no transition
C              = 1  if transition was triggered by trip
C              = 2  if transition was triggered by e^n criterion
C              = 3  if transition was triggered by Abu-Ghannam--Shaw crit.
C    XIT       transition location  (valid only if KTRAN > 0)  
C    XIT_VAR1  dXIT/dVAR1(1:ITOT)   ( = 0  if forced transition )
C    XIT_VAR2  dXIT/dVAR2(1:ITOT)   ( = 0  if forced transition )
C    XIT_XIF   dXIT/dXIFORC         ( = 1  if forced transition,
C                                     = 0  if free   transition )
C    AMPL2     amplification at XI2 (meaningful only if KTRAN = 0)
C
C
C    Note:  XIT_XIF is indicator of whether transition is free or forced
C
C......................................................................
C
C     This routine integrates the amplification equation over XI1..XI2,
C
C        AM2 - AM1  =  (XI2 - XI1) dN/dx
C
C     where dN/dx is evaluated as an average over the XI1..XI2
C     interval, or the laminar sub-interval XI1..XIT if XIT < XI2.
C
C     XIT is implicitly defined by the amplification equation
C     taken over the laminar sub-interval XI1..XIT:
C
C       AMCRIT - AM1  =  (XIT - XI1) dN/dx
C
C
C     The actual XIT value may be returned as the forced location XIFORC 
C     if this is smaller.
C
C
C     As a side-effect, this routine solves for AM2 (returned via AMPL2).
C
C----------------------------------------------------------------------
      IMPLICIT REAL (A-H,M,O-Z)
      INCLUDE 'INDEX.INC'
C
      DIMENSION VAR1(ITOT), VAR2(ITOT)
      DIMENSION VJ1(0:ITOT,JTOT), VJ2(0:ITOT,JTOT)
      DIMENSION PJ1(  LTOT,JTOT), PJ2(  LTOT,JTOT)
      DIMENSION XIT_VAR1(ITOT), XIT_VAR2(ITOT), XIT_PAR(LTOT)
C
      LOGICAL TRFREE, TRFORC, LAGS
C
C
C---- unpack required primary and secondary variables
      TH1 = VAR1(ITH)
      XI1 = VAR1(IXI)
      RR1 = VAR1(IRR)
      AM1 = VAR1(IAM)
C
      TH2 = VAR2(ITH)
      XI2 = VAR2(IXI)
      RR2 = VAR2(IRR)
      AM2 = VAR2(IAM)
C
      HK1 = VJ1(0,JHK)
      RT1 = VJ1(0,JRT)
C
      HK2 = VJ2(0,JHK)
      RT2 = VJ2(0,JRT)
C
C
C---- max-change limits
      DAMMAX = 1.0
      DXIMAX = (XI2-XI1) * 0.05
C
C---- convergence limits
ccc      DAMEPS =             1.0E-6
ccc      DXIEPS = (XI2-XI1) * 1.0E-6
c###
      DAMEPS =             1.0D-12
      DXIEPS = (XI2-XI1) * 1.0D-12
C
C---- check for internal error
      if(am1.ge.amcrit .and. am2.ge.amcrit) then
        write(*,*) '? TRCHEK  X1,XA,X2:', XI1, XIA   , XI2
        write(*,*) '          N1,NC,N2:', AM1, AMCRIT, AM2
      endif
C
C
C---- clear all XIT derivatives for possible return
      DO 5 L=1, ITOT
        XIT_VAR1(L) = 0.
        XIT_VAR2(L) = 0.
 5    CONTINUE
      DO 6 L=1, LTOT
        XIT_PAR(L) = 0.
 6    CONTINUE
      XIT_XIF = 0.
C
C
C---- assume that no transition will occur
      KTRAN = 0
C
C---- solve implicit system for amplification ratio AM2
      DO 100 ITER=1, 25
C
C---- define second endpoint XIA of laminar sub-interval XI1..XIA
      IF(AM2 .GE. AMCRIT) THEN
C------ transition is present: XIA is transition location XIT
        XIA     = XI1 + (XI2-XI1)*(AMCRIT-AM1)/(AM2-AM1)
        XIA_XI1 = 1.0       -     (AMCRIT-AM1)/(AM2-AM1)
        XIA_XI2 =                 (AMCRIT-AM1)/(AM2-AM1)
        XIA_AM1 =     - (XI2-XI1)             /(AM2-AM1)
     &                + (XI2-XI1)*(AMCRIT-AM1)/(AM2-AM1)**2
        XIA_AM2 =     - (XI2-XI1)*(AMCRIT-AM1)/(AM2-AM1)**2
        XIA_AMC =       (XI2-XI1)             /(AM2-AM1)
      ELSE
C------ transition not present: XIA is XI2
        XIA     = XI2
        XIA_XI1 = 0.
        XIA_XI2 = 1.0
        XIA_AM1 = 0.
        XIA_AM2 = 0.
        XIA_AMC = 0.
      ENDIF
C
C---- check for internal error
      if(xia.lt.xi1 .or. xia.gt.xi2) then
        write(*,*) '? TRCHEK  X1,XA,X2:', XI1, XIA   , XI2
        write(*,*) '          N1,NC,N2:', AM1, AMCRIT, AM2
      endif
C
      CALL AXSET(HK1,   TH1,   RT1,   AM1,   XI1,
     &           HK2,   TH2,   RT2,   AM2,   XI2,   XIA, AMCRIT,
     &    AX, AX_HK1,AX_TH1,AX_RT1,AX_AM1,AX_XI1,
     &        AX_HK2,AX_TH2,AX_RT2,AX_AM2,AX_XI2,AX_XIA, AX_AMC,
     &    LAGS )
C
C
C---- execute Newton iteration for AM2
      DS = (XI2-XI1)*(RR1+RR2)*0.5
      RES     = AM2 - AM1 - DS* AX
      RES_AM2 = 1.0       - DS*(AX_AM2 + AX_XIA*XIA_AM2)
C
      DAM2 = -RES/RES_AM2
C
C---- implied XIA change
      DXIA = XIA_AM2*DAM2
C
C---- check if converged
      IF(ABS(DXIA) .LT. DXIEPS .AND.
     &   ABS(DAM2) .LT. DAMEPS       ) GO TO 101
C
ccc      if(iter.ge.15) then
c        WRITE(*,6700) XI1, XIA, XI2, AM1, AMCRIT, AM2, AX*0.5*(RR1+RR2)
c        write(*,6900) hk1, 1000.0*th1, rt1, am1, xi1,
c     &                hk2, 1000.0*th2, rt2, am2, xi2, xia, ax
c 6900   format(1x,5f12.6, /1x, 5f12.6, 2x,2f12.6)
ccc      endif

C---- set Newton step limiter
      RLX = 1.0
      IF(RLX*ABS(DXIA) .GT. DXIMAX) RLX = DXIMAX/ABS(DXIA)
      IF(RLX*ABS(DAM2) .GT. DAMMAX) RLX = DAMMAX/ABS(DAM2)
C
C---- Newton step
      AM2 = AM2 + RLX*DAM2
C
 100  CONTINUE
      WRITE(*,*) 'TRCHEK: Convergence failed.  dX,dN =', DXIA, DAM2
ccc      WRITE(*,6700) XI1, XIA, XI2, AM1, AMCRIT, AM2, AX*0.5*(RR1+RR2)
 6700 FORMAT(1X,'X:', 3F9.5,'  N:',3F7.3,'   dN/dX:', F8.3)
C
 101  CONTINUE
C
C---- return converged AM2 value
      AMPL2 = AM2
C
C---- test for free or forced transition
      TRFREE = AM2 .GE. AMCRIT
      TRFORC = XI2 .GE. XIFORC .AND. XI1.LT.XIFORC
C
C---- if no transition, just return with new AMPL2 value
      IF( .NOT. (TRFORC .OR. TRFREE) ) RETURN
C
C---- resolve if both forced and free transition
      IF(TRFREE .AND. TRFORC) THEN
       TRFORC = XIA .GE. XIFORC
       TRFREE = XIA .LT. XIFORC
      ENDIF
C
C
      IF(TRFORC) THEN
C
C------ forced transition... XIT is prescribed
        XIT     = XIFORC
        XIT_XIF = 1.0
        KTRAN = 1
C
      ELSE
C
C------ free transition... XIT is same as XIA
C
C------ set all derivatives of residual RES to be held stationary
        DS     = (XI2-XI1)*(RR1+RR2)*0.5
        DS_XI1 =     -     (RR1+RR2)*0.5
        DS_XI2 =           (RR1+RR2)*0.5
        DS_RR1 = (XI2-XI1)          *0.5
        DS_RR2 = (XI2-XI1)          *0.5
C
CCC     RES     = AM2 - AM1 - DS* AX
C
        RES_HK1 =           - DS* AX_HK1
        RES_TH1 =           - DS* AX_TH1
        RES_RT1 =           - DS* AX_RT1
        RES_AM1 =     - 1.0 - DS*(AX_AM1 + AX_XIA*XIA_AM1)
        RES_XI1 =           - DS*(AX_XI1 + AX_XIA*XIA_XI1)
     &                      - DS_XI1*AX
        RES_RR1 =           - DS_RR1*AX
C
        RES_HK2 =           - DS* AX_HK2
        RES_TH2 =           - DS* AX_TH2
        RES_RT2 =           - DS* AX_RT2
        RES_AM2 = 1.0       - DS*(AX_AM2 + AX_XIA*XIA_AM2)
        RES_XI2 =           - DS*(AX_XI2 + AX_XIA*XIA_XI2)
     &                      - DS_XI2*AX
        RES_RR2 =           - DS_RR2*AX
C
CCC     RES_AMC =           - DS*(         AX_XIA*XIA_AMC)
C
C------ set XIT and its derivatives,
C-      using  dAM2/d( ) = -RES_( )/RES_AM2  from the condition  dRES = 0
        XIT     = XIA
        XIT_HK1 =         - XIA_AM2*(RES_HK1/RES_AM2)
        XIT_TH1 =         - XIA_AM2*(RES_TH1/RES_AM2)
        XIT_RT1 =         - XIA_AM2*(RES_RT1/RES_AM2)
        XIT_AM1 = XIA_AM1 - XIA_AM2*(RES_AM1/RES_AM2)
        XIT_XI1 = XIA_XI1 - XIA_AM2*(RES_XI1/RES_AM2)
        XIT_RR1 =         - XIA_AM2*(RES_RR1/RES_AM2)
        XIT_HK2 =         - XIA_AM2*(RES_HK2/RES_AM2)
        XIT_TH2 =         - XIA_AM2*(RES_TH2/RES_AM2)
        XIT_RT2 =         - XIA_AM2*(RES_RT2/RES_AM2)
        XIT_AM2 = XIA_AM2 - XIA_AM2*(RES_AM2/RES_AM2)
        XIT_XI2 = XIA_XI2 - XIA_AM2*(RES_XI2/RES_AM2)
        XIT_RR2 =         - XIA_AM2*(RES_RR2/RES_AM2)
CCC     XIT_AMC = XIA_AMC - XIA_AM2*(RES_AMC/RES_AM2)
C-      note: (XIT_AM2 = 0 as it should, since AM2 is not a free parameter)
C
C------ store total XIT derivatives with respect to primary variables
        DO L=1, ITOT
          XIT_VAR1(L) = XIT_HK1*VJ1(L,JHK) + XIT_RT1*VJ1(L,JRT)
          XIT_VAR2(L) = XIT_HK2*VJ2(L,JHK) + XIT_RT2*VJ2(L,JRT)
        ENDDO
        XIT_VAR1(ITH) = XIT_VAR1(ITH) + XIT_TH1
        XIT_VAR1(IAM) = XIT_VAR1(IAM) + XIT_AM1
        XIT_VAR1(IXI) = XIT_VAR1(IXI) + XIT_XI1
        XIT_VAR1(IRR) = XIT_VAR1(IRR) + XIT_RR1
        XIT_VAR2(ITH) = XIT_VAR2(ITH) + XIT_TH2
        XIT_VAR2(IAM) = XIT_VAR2(IAM) + XIT_AM2
        XIT_VAR2(IXI) = XIT_VAR2(IXI) + XIT_XI2
        XIT_VAR2(IRR) = XIT_VAR2(IRR) + XIT_RR2
C
        DO L=1, LTOT
          XIT_PAR(L) = XIT_HK1*PJ1(L,JHK) + XIT_RT1*PJ1(L,JRT)
     &               + XIT_HK2*PJ2(L,JHK) + XIT_RT2*PJ2(L,JRT)
        ENDDO
C
        IF(LAGS) THEN
         KTRAN = 3
        ELSE
         KTRAN = 2
        ENDIF
      ENDIF
C
      RETURN
      END ! TRCHEK



      SUBROUTINE AXSET(
     &         HK1,   TH1,   RT1,   AM1,   XI1,
     &         HK2,   TH2,   RT2,   AM2,   XI2,    XIT, AMCRIT,
     &  AX, AX_HK1,AX_TH1,AX_RT1,AX_AM1,AX_XI1,
     &      AX_HK2,AX_TH2,AX_RT2,AX_AM2,AX_XI2, AX_XIT, AX_AMC,
     &  LAGS )
C-----------------------------------------------------------------------
C     Returns average amplification rate AX over XI1..XIT sub-interval.
C-----------------------------------------------------------------------
      LOGICAL LAGS
C
C---- parameters for modified Abu-Ghannam--Shaw transition criterion
      DATA ACON, BCON / 0.10 , 0.30 /
C
C---- set amplification rates AX1,AX2, at interval endpoints
      CALL DAMPL( HK1, TH1, RT1, AX1, AX1_HK1, AX1_TH1, AX1_RT1 )
      CALL DAMPL( HK2, TH2, RT2, AX2, AX2_HK2, AX2_TH2, AX2_RT2 )
C
C---- set fraction for interpolating to XIT
      FXT     = (XIT-XI1)/(XI2-XI1)
      FXT_XI1 = (FXT-1.0)/(XI2-XI1)
      FXT_XI2 = -FXT     /(XI2-XI1)
      FXT_XIT =  1.0     /(XI2-XI1)
C
C
C---- set variables averaged over laminar sub-interval 1..T
      AXA     = AX1 + 0.5*FXT*(AX2-AX1)
      AXA_AX1 = 1.0 - 0.5*FXT
      AXA_AX2 =       0.5*FXT
      AXA_FXT =       0.5    *(AX2-AX1)
C
      THA     = TH1 + 0.5*FXT*(TH2-TH1)
      THA_TH1 = 1.0 - 0.5*FXT
      THA_TH2 =       0.5*FXT
      THA_FXT =       0.5    *(TH2-TH1)
C
      AMA     = AM1 + 0.5*FXT*(AM2-AM1)
      AMA_AM1 = 1.0 - 0.5*FXT
      AMA_AM2 =       0.5*FXT
      AMA_FXT =       0.5    *(AM2-AM1)
C
C---- make sure interpolated envelope amplification is not negative
C-     ( just in case FXT is outside 0.0 .. 1.0 )
      IF(AXA .LE. 0.0) THEN
       AXA     = 0.0
       AXA_AX1 = 0.
       AXA_AX2 = 0.
       AXA_FXT = 0.
      ENDIF
C
C---- small additional term DAX to ensure  dN/dx > 0  near  N = Ncrit
      ARG = MIN( 20.0*(AMCRIT-AMA) , 20.0 )
      IF(ARG.LE.0.0) THEN
       EXN     = 1.0
       EXN_AMC = 0.
       EXN_AMA = 0.
      ELSE
       EXN     = EXP(-ARG)
       EXN_AMC = -20.0*EXN
       EXN_AMA =  20.0*EXN
      ENDIF
C
      DAX     = EXN     * 0.001/THA
      DAX_AMC = EXN_AMC * 0.001/THA
      DAX_AMA = EXN_AMA * 0.001/THA
      DAX_THA = -DAX/THA
C----------------------
c        DAX     = 0.
c        DAX_AMC = 0.
c        DAX_AMA = 0.
c        DAX_THA = 0.
C----------------------
C
C
C---- set additional term GAX due to strong freestream turbulence
C-    (mimics the  Abu-Ghannam & Shaw  bypass transition model)
C
C---- set AGS critical Rtheta for start of transition
      TNH1     = TANH(10.0/(HK1-1.0) - 5.5)
      TNH1_HK1 = (TNH1**2 - 1.0) * 10.0/(HK1-1.0)**2
      RTS1     = 155.0 + 89.0*(0.25*TNH1 + 1.0) * AMCRIT**1.25
      RTS1_HK1 =         89.0*(0.25*TNH1_HK1  ) * AMCRIT**1.25
      RTS1_AMC =         89.0*(0.25*TNH1 + 1.0) * AMCRIT**0.25 * 1.25
C
      TNH2     = TANH(10.0/(HK2-1.0) - 5.5)
      TNH2_HK2 = (TNH2**2 - 1.0) * 10.0/(HK2-1.0)**2
      RTS2     = 155.0 + 89.0*(0.25*TNH2 + 1.0) * AMCRIT**1.25
      RTS2_HK2 =         89.0*(0.25*TNH2_HK2  ) * AMCRIT**1.25
      RTS2_AMC =         89.0*(0.25*TNH2 + 1.0) * AMCRIT**0.25 * 1.25
C
C---- set cubic function argument centered on the AGS critical Rtheta
      FR1     = ( RT1/RTS1 - 1.0 )/BCON  +  0.5
      FR1_RT1 = ( 1.0/RTS1       )/BCON
      FR1_HK1 = (-RT1/RTS1**2    )/BCON * RTS1_HK1
      FR1_AMC = (-RT1/RTS1**2    )/BCON * RTS1_AMC
C
      FR2     = ( RT2/RTS2 - 1.0 )/BCON  +  0.5
      FR2_RT2 = ( 1.0/RTS2       )/BCON
      FR2_HK2 = (-RT2/RTS2**2    )/BCON * RTS2_HK2
      FR2_AMC = (-RT2/RTS2**2    )/BCON * RTS2_AMC
C-------------------
c      FR1 = 0.0
c      FR2 = 0.0
C-------------------
C
C
      IF(FR1 .LE. 0.0) THEN
        GAX1     = 0.0
        GAX1_HK1 = 0.
        GAX1_RT1 = 0.
        GAX1_TH1 = 0.
        GAX1_AMC = 0.
      ELSEIF(FR1 .GT. 1.0) THEN
        GAX1     = 0.5*ACON/TH1
        GAX1_HK1 = 0.
        GAX1_RT1 = 0.
        GAX1_TH1 = -GAX1/TH1
        GAX1_AMC = 0.
      ELSE
        GAX1     = (3.0*FR1**2 - 2.0*FR1**3) * 0.5*ACON/TH1
        GAX1_HK1 = (6.0*FR1    - 6.0*FR1**2) * 0.5*ACON/TH1 * FR1_HK1
        GAX1_RT1 = (6.0*FR1    - 6.0*FR1**2) * 0.5*ACON/TH1 * FR1_RT1
        GAX1_TH1 = -GAX1/TH1
        GAX1_AMC = (6.0*FR1    - 6.0*FR1**2) * 0.5*ACON/TH1 * FR1_AMC
      ENDIF
C
      IF(FR2 .LE. 0.0) THEN
        GAX2     = 0.0
        GAX2_HK2 = 0.
        GAX2_RT2 = 0.
        GAX2_TH2 = 0.
        GAX2_AMC = 0.
      ELSEIF(FR2 .GT. 1.0) THEN
        GAX2     = 0.5*ACON/TH2
        GAX2_HK2 = 0.
        GAX2_RT2 = 0.
        GAX2_TH2 = -GAX2/TH2
        GAX2_AMC = 0.
      ELSE
        GAX2     = (3.0*FR2**2 - 2.0*FR2**3) * 0.5*ACON/TH2
        GAX2_HK2 = (6.0*FR2    - 6.0*FR2**2) * 0.5*ACON/TH2 * FR2_HK2
        GAX2_RT2 = (6.0*FR2    - 6.0*FR2**2) * 0.5*ACON/TH2 * FR2_RT2
        GAX2_TH2 = -GAX2/TH2
        GAX2_AMC = (6.0*FR2    - 6.0*FR2**2) * 0.5*ACON/TH2 * FR2_AMC
      ENDIF
C
C---- set flag indicating whether AGS model is active
      LAGS = MAX(FR1,FR2) .GT. 0.2
C
C
C---- set total amplification rate and derivatives
      AX     = AXA             + DAX             + GAX1     + GAX2
      AX_FXT = AXA_FXT         + DAX_THA*THA_FXT
     &                         + DAX_AMA*AMA_FXT
C
      AX_AMC =                   DAX_AMC         + GAX1_AMC + GAX2_AMC
C
      AX_HK1 = AXA_AX1*AX1_HK1                   + GAX1_HK1
      AX_RT1 = AXA_AX1*AX1_RT1                   + GAX1_RT1
      AX_TH1 = AXA_AX1*AX1_TH1 + DAX_THA*THA_TH1 + GAX1_TH1
      AX_AM1 =                   DAX_AMA*AMA_AM1
      AX_XI1 =  AX_FXT*FXT_XI1
C
      AX_HK2 = AXA_AX2*AX2_HK2                              + GAX2_HK2
      AX_RT2 = AXA_AX2*AX2_RT2                              + GAX2_RT2
      AX_TH2 = AXA_AX2*AX2_TH2 + DAX_THA*THA_TH2            + GAX2_TH2
      AX_AM2 =                   DAX_AMA*AMA_AM2
      AX_XI2 =  AX_FXT*FXT_XI2
C
      AX_XIT =  AX_FXT*FXT_XIT
C
      RETURN
      END


 
      SUBROUTINE DAMPL( HK, TH, RT, AX, AX_HK, AX_TH, AX_RT )
C==============================================================
C     Amplification rate routine for envelope e^n method.
C     Reference: 
C                Drela, M., Giles, M.,
C               "Viscous/Inviscid Analysis of Transonic and 
C                Low Reynolds Number Airfoils", 
C                AIAA Journal, Oct. 1987.
C
C     NEW VERSION.   March 1991
C          - m(H) correlation made valid up to H=20
C          - non-similar profiles are used for H > 5 
C            in lieu of Falkner-Skan profiles.  These are
C            more representative of separation bubble profiles.
C--------------------------------------------------------------
C
C     input :   HK     kinematic shape parameter
C               TH     momentum thickness
C               RT     momentum-thickness Reynolds number
C
C     output:   AX     envelope spatial amplification rate
C               AX_(.) sensitivity of AX to parameter (.)
C
C
C     Usage: The log of the envelope amplitude N(x) is 
C            calculated by integrating AX (= dN/dx) with 
C            respect to the streamwise distance x.
C                      x
C                     /
C              N(x) = | AX(H(x),Th(x),Rth(x)) dx
C                     /
C                      0
C            The integration can be started from the leading
C            edge since AX will be returned as zero when RT
C            is below the critical Rtheta.  Transition occurs
C            when N(x) reaches Ncrit (Ncrit= 9 is "standard").
C==============================================================
      IMPLICIT REAL (A-H,M,O-Z)
      DATA DGR / 0.08 /
C
      HMI = 1.0/(HK - 1.0)
      HMI_HK = -HMI**2
C
C---- log10(Critical Rth) -- H   correlation for Falkner-Skan profiles
      AA    = 2.492*HMI**0.43
      AA_HK =   (AA/HMI)*0.43 * HMI_HK
C
      BB    = TANH(14.0*HMI - 9.24)
      BB_HK = (1.0 - BB*BB) * 14.0 * HMI_HK
C
      GRCRIT = AA    + 0.7*(BB + 1.0)
      GRC_HK = AA_HK + 0.7* BB_HK
C
C
      GR = LOG10(RT)
      GR_RT = 1.0 / (2.3025851*RT)
C
      IF(GR .LT. GRCRIT-DGR) THEN
C
C----- no amplification for Rtheta < Rcrit
       AX    = 0.
       AX_HK = 0.
       AX_TH = 0.
       AX_RT = 0.
C
      ELSE
C
C----- Set steep cubic ramp used to turn on AX smoothly as Rtheta 
C-     exceeds Rcrit (previously, this was done discontinuously).
C-     The ramp goes between  -DGR < log10(Rtheta/Rcrit) < DGR
C
       RNORM = (GR - (GRCRIT-DGR)) / (2.0*DGR)
       RN_HK =     -  GRC_HK       / (2.0*DGR)
       RN_RT =  GR_RT              / (2.0*DGR)
C
       IF(RNORM .GE. 1.0) THEN
        RFAC    = 1.0
        RFAC_HK = 0.
        RFAC_RT = 0.
       ELSE
        RFAC    = 3.0*RNORM**2 - 2.0*RNORM**3
        RFAC_RN = 6.0*RNORM    - 6.0*RNORM**2
C
        RFAC_HK = RFAC_RN*RN_HK
        RFAC_RT = RFAC_RN*RN_RT
       ENDIF
C
C
C----- set envelope amplification rate with respect to Rtheta
C-       DADR = d(N)/d(Rtheta) = f(H)
C
       ARG    = 3.87*HMI    - 2.52
       ARG_HK = 3.87*HMI_HK
C
       EX    = EXP(-ARG**2)
       EX_HK = EX * (-2.0*ARG*ARG_HK)
C
       DADR    = 0.028*(HK-1.0) - 0.0345*EX
       DADR_HK = 0.028          - 0.0345*EX_HK
C
C
C----- set conversion factor from d/d(Rtheta) to d/dx
C-       AF = Theta d(Rtheta)/dx = f(H)
C
       BRG = -20.0*HMI
       AF = -0.05 + 2.7*HMI -  5.5*HMI**2 + 3.0*HMI**3 + 0.1*EXP(BRG)
       AF_HMI =     2.7     - 11.0*HMI    + 9.0*HMI**2 - 2.0*EXP(BRG)
       AF_HK = AF_HMI*HMI_HK
C
C
C----- set amplification rate with respect to x, 
C-     with RFAC shutting off amplification when below Rcrit
C
       AX    = (AF   *DADR/TH                ) * RFAC
       AX_HK = (AF_HK*DADR/TH + AF*DADR_HK/TH) * RFAC
     &       + (AF   *DADR/TH                ) * RFAC_HK
       AX_TH = -AX/TH
       AX_RT = (AF   *DADR/TH                ) * RFAC_RT
C
      ENDIF
C
      RETURN
      END ! DAMPL


