
      SUBROUTINE BLVAR( ITYP,
     &                  PAR, GAM, HSUTH, DW, DWTE,
     &                  VAR,  VJ, PJ )
C-------------------------------------------------------------------------
C     Calculates secondary variables and their derivatives 
C     with respect to the primary variables CT, TH,  ...AM , and
C     with respect to the parameters Htot, Rtot, REtot, rot
C
C     Input:  flags and parameters
C       ITYP       1=laminar, 2=turbulent, 3=wake
C       PAR( . )   parameter...
C          (LSH)      Htot  stagnation enthalpy  (same units as Ue^2)
C          (LSR)      Rtot  stagnation density
C          (LRE)      REtot stagnation Reynolds number / unit length
C          (LRO)      rot   rotation rate  (radians / unit time)
C       GAM      cp/cv
C       HSUTH    Sutherland's enthalpy (= cp x 110K, in same units as Ue^2)
C       DW       wake "gap" for blunt-TE near-wake model
C       DWTE     blunt-TE height
C
C     Input:  primary variables
C       VAR(ICT)   Ctau^1/2     (max shear stress coefficient)^1/2
C       VAR(ITH)   Theta        momentum thickness
C       VAR(IDS)   Delta*       displacement thickness
C       VAR(IUE)   Ue           edge velocity
C       VAR(IUW)   Uw           wall velocity (or at wake centerline)
C       VAR(IRH)   Rhoe         edge density
C       VAR(ICV)   Curv         displacement surface curvature, + concave 
C       VAR(IXI)   xi           streamwise coordinate
C       VAR(IRR)   r            radius
C       VAR(IBB)   b            streamtube thickness
C       VAR(IAM)   Am           amplification variable
C       VAR(IMW)   mwall        wall suction mass flow rate
C
C
C      Output: secondary variables and derivatives
C       VJ(0,JRH)  Rhoe         edge density
C       VJ(0,JMS)  Me^2         edge Mach^2
C       VJ(0,JHK)  Hk           kinematic shape parameter
C       VJ(0,JHS)  H*           KE shape parameter
C       VJ(0,JHC)  H**          density-flux shape parameter
C       VJ(0,JHD)  Hr           density shape parameter
C       VJ(0,JUS)  Us           apparent slip velocity
C       VJ(0,JCF)  Cf           skin friction coefficient
C       VJ(0,JDI)  2CD/H*       dissipation coefficient x 2/H*
C       VJ(0,JUQ)  (ln Ue)_x    equilibrium pressure gradient
C       VJ(0,JCQ)  CtauEQ^1/2   equilibrium shear stress coefficient
C       VJ(0,JDE)  Delta        BL thickness
C       VJ(0,JRT)  Rtheta       momentum-thickness Reynolds number
C
C       VJ(IUE,JRH)  d(Rhoe)/d(Ue)
C       VJ(IRR,JRH)  d(Rhoe)/d(r)
C
C       VJ(ITH,JHK)  d(Hk)/d(Theta)
C       VJ(IDS,JHK)  d(Hk)/d(Delta*)
C       ... etc
C
C
C       PJ(LSH,JRH)   d(Rhoe)/dHTOT
C       PJ(LSR,JRH)   d(Rhoe)/dRTOT
C       PJ(LRE,JRH)   d(Rhoe)/dREYN
C       PJ(LRO,JRH)   d(Rhoe)/dROT
C       ... etc
C
C-------------------------------------------------------------------------
      IMPLICIT REAL (A-H,M,O-Z)
      INCLUDE 'BLPAR.INC'
      INCLUDE 'INDEX.INC'
      DIMENSION PAR(LTOT), VAR(ITOT), VJ(0:ITOT,JTOT), PJ(LTOT,JTOT)
C
C---- unpack parameters and variables
      HTOT  = PAR(LSH)
      RTOT  = PAR(LSR)
      RETOT = PAR(LRE)
      ROT   = PAR(LRO)
C
 5    CONTINUE
      CT = VAR(ICT)
      TH = VAR(ITH)
      DS = VAR(IDS)
      UE = VAR(IUE)
      UW = VAR(IUW)
      RH = VAR(IRH)
      CV = VAR(ICV)
      XI = VAR(IXI)
      RR = VAR(IRR)
      BB = VAR(IBB)
      AM = VAR(IAM)
      MW = VAR(IMW)
C
      GM1 = GAM - 1.0
C
C---- static enthalpy  HE( HTOT UE RR )
      HE    = HTOT - 0.5*UE**2 + 0.5*(RR*ROT)**2
      HE_UE =      -     UE
      HE_RR =                         RR*ROT**2
      HE_SH = 1.0
      HE_RO =                         RR**2*ROT
C
C---- edge Mach^2   MS( UE RR SH RO )
      MS    = UE**2 /(GM1*HE)
      MS_UE = UE*2.0/(GM1*HE) - (MS/HE)*HE_UE
      MS_RR =                 - (MS/HE)*HE_RR
      MS_SH =                 - (MS/HE)*HE_SH
      MS_RO =                 - (MS/HE)*HE_RO
C
C---- edge density  RHO( RH UE RR SH SR RO )
C
c----- rho is isentropically related to Ue
c      RHO = RTOT*(HE/HTOT)**(1.0/GM1)
c      RHO_RH = 0.
c      RHO_UE = RHO/(GM1*HE) * HE_UE
c      RHO_RR = RHO/(GM1*HE) * HE_RR
c      RHO_SH = RHO*(HE_SH/HE - 1.0/HTOT) / GM1
c      RHO_SR = RHO/RTOT
c      RHO_RO = RHO*(HE_RO/HE           ) / GM1
C
C---- rho is an independent variable
      RHO    = RH
      RHO_RH = 1.0
      RHO_UE = 0.
      RHO_RR = 0.
      RHO_SH = 0.
      RHO_SR = 0.
      RHO_RO = 0.
C
C
C---- stagnation viscosity
      MUTOT = RTOT*SQRT(GM1*HTOT)/RETOT
      MT_SR =     MUTOT/RTOT
      MT_SH = 0.5*MUTOT/HTOT
      MT_RE =    -MUTOT/RETOT

C---- edge viscosity
      MU    = SQRT((HE/HTOT)**3) * (HTOT+HSUTH)/(HE+HSUTH) * MUTOT
      MU_MT = SQRT((HE/HTOT)**3) * (HTOT+HSUTH)/(HE+HSUTH)
C
      MU_UE = MU*(1.5/HE   - 1.0/(HE  +HSUTH)) * HE_UE
      MU_RR = MU*(1.5/HE   - 1.0/(HE  +HSUTH)) * HE_RR
      MU_SR =                                            MU_MT*MT_SR
      MU_SH = MU*(1.5/HE   - 1.0/(HE  +HSUTH)) * HE_SH + MU_MT*MT_SH
     &      - MU*(1.5/HTOT - 1.0/(HTOT+HSUTH))
      MU_RE =                                            MU_MT*MT_RE
      MU_RO = MU*(1.5/HE   - 1.0/(HE  +HSUTH)) * HE_RO
C
C---- momentum thickness Reynolds number
      RT    = TH*RHO*UE/MU
      RT_TH =    RHO*UE/MU
      RT_RH =                RT*(RHO_RH/RHO           )
      RT_UE = TH*RHO   /MU + RT*(RHO_UE/RHO - MU_UE/MU)
      RT_RR =                RT*(RHO_RR/RHO - MU_RR/MU)
      RT_SH =                RT*(RHO_SH/RHO - MU_SH/MU)
      RT_SR =                RT*(RHO_SR/RHO - MU_SR/MU)
      RT_RE =                RT*(           - MU_RE/MU)
      RT_RO =                RT*(RHO_RO/RHO - MU_RO/MU)
C
C---- suction coefficient (Rho x v)wall / (Rho x U)edge
      CM    = -MW/(UE*RHO)
      CM_RH = -(CM/RHO)*RHO_RH
      CM_UE = -(CM/RHO)*RHO_UE - CM/UE
      CM_MW =                  - 1.0/(UE*RHO)
      CM_RR = -(CM/RHO)*RHO_RR
      CM_SH = -(CM/RHO)*RHO_SH
      CM_SR = -(CM/RHO)*RHO_SR
      CM_RO = -(CM/RHO)*RHO_RO
C
C---- shape parameter, excluding wake gap in total displacement thickness
      H     =  DS/TH  -  RR*DW/TH
      H_TH  =  -H/TH
      H_DS  = 1.0/TH
      H_RR  =         -     DW/TH
C
C---- kinematic shape parameter
      CALL HKIN( H, MS, GAM, HK, HK_H, HK_MS )
      HK_TH = HK_H*H_TH
      HK_DS = HK_H*H_DS
      HK_UE =             HK_MS*MS_UE
      HK_RR = HK_H*H_RR + HK_MS*MS_RR
      HK_SH =             HK_MS*MS_SH
      HK_RO =             HK_MS*MS_RO
C
      IF(ITYP.NE.3 .AND. HK.LT.1.01) THEN
C------ limit Hk for solid-wall case
        HK    = 1.01
        HK_TH = 0.
        HK_DS = 0.
        HK_UE = 0.
        HK_RR = 0.
        HK_SH = 0.
        HK_RO = 0.
      ENDIF
C
C=================================================
C---- shape parameter correlations
C
C---- H* = TS/TH    KE-thickness shape parameter
      IF(ITYP.EQ.1) THEN
       CALL HSL( HK, RT, MS, HS, HS_HK, HS_RT, HS_MS  )
      ELSE
       CALL HST( HK, RT, MS, HS, HS_HK, HS_RT, HS_MS  )
      ENDIF
      HS_TH = HS_HK*HK_TH + HS_RT*RT_TH
      HS_DS = HS_HK*HK_DS
      HS_RH =               HS_RT*RT_RH
      HS_UE = HS_HK*HK_UE + HS_RT*RT_UE + HS_MS*MS_UE
      HS_RR = HS_HK*HK_RR + HS_RT*RT_RR + HS_MS*MS_RR
      HS_SH = HS_HK*HK_SH + HS_RT*RT_SH + HS_MS*MS_SH
      HS_SR =               HS_RT*RT_SR
      HS_RE =               HS_RT*RT_RE
      HS_RO = HS_HK*HK_RO + HS_RT*RT_RO + HS_MS*MS_RO
C
C---- H** = D**/TH    density flux-thickness shape parameter
c      CALL HCT( HK, MS, GAM, HC, HC_HK, HC_MS )
c      HC_TH = HC_HK*HK_TH
c      HC_DS = HC_HK*HK_DS
c      HC_UE = HC_HK*HK_UE + HC_MS*MS_UE
c      HC_RR = HC_HK*HK_RR + HC_MS*MS_RR
c      HC_SH = HC_HK*HK_SH + HC_MS*MS_SH
c      HC_SR = 0.
c      HC_RE = 0.
c      HC_RO = HC_HK*HK_RO + HC_MS*MS_RO
C
C---- exact relation for  HC(MS HS)  assuming adiabatic flow
      HC    = 0.5*GM1*MS*HS
      HC_MS = 0.5*GM1   *HS
      HC_HS = 0.5*GM1*MS
C
      HC_TH = HC_HS*HS_TH
      HC_DS = HC_HS*HS_DS
      HC_RH = HC_HS*HS_RH
      HC_UE = HC_HS*HS_UE + HC_MS*MS_UE
      HC_RR = HC_HS*HS_RR + HC_MS*MS_RR
      HC_SH = HC_HS*HS_SH + HC_MS*MS_SH
      HC_SR = HC_HS*HS_SR
      HC_RE = HC_HS*HS_RE
      HC_RO = HC_HS*HS_RO + HC_MS*MS_RO
C
C---- Hr = T_rho/TH    density-thickness shape parameter
      CALL HDT( HK, MS, GAM, HD, HD_HK, HD_MS )
      HD_TH = HD_HK*HK_TH
      HD_DS = HD_HK*HK_DS
      HD_UE = HD_HK*HK_UE + HD_MS*MS_UE
      HD_RR = HD_HK*HK_RR + HD_MS*MS_RR
      HD_SH = HD_HK*HK_SH + HD_MS*MS_SH
      HD_RO = HD_HK*HK_RO + HD_MS*MS_RO
C
C=================================================
C---- BL thickness (delta) from simplified Green's correlation
C-    (used only as length scale in lag equation)
C
      DE    = (3.15 + 1.72/(HK-1.0)   )*TH  +  DS
      DE_HK = (     - 1.72/(HK-1.0)**2)*TH
      DE_TH = (3.15 + 1.72/(HK-1.0))
      DE_DS =                                  1.0
C
      DE_TH = DE_HK*HK_TH + DE_TH
      DE_DS = DE_HK*HK_DS + DE_DS
      DE_UE = DE_HK*HK_UE
      DE_RR = DE_HK*HK_RR
      DE_SH = DE_HK*HK_SH
      DE_RO = DE_HK*HK_RO
C
      IF(DE .GT. 15.0*TH) THEN
       DE    = 15.0*TH
       DE_TH = 15.0
       DE_DS = 0.
       DE_UE = 0.
       DE_RR = 0.
       DE_SH = 0.
       DE_RO = 0.
      ENDIF
C
C=================================================
C---- normalized slip velocity  us/qe  (centerline velocity for wake)
C
C---- US( HS HK H )
      US    = 0.5*HS*( 1.0 - (HK-1.0)/(GBCON*H) )
      US_HS = 0.5   *( 1.0 - (HK-1.0)/(GBCON*H) )
      US_HK = 0.5*HS*(     -  1.0    /(GBCON*H) )
      US_H  = 0.5*HS*        (HK-1.0)/(GBCON*H**2)
C
C---- US( TH DS ... RO )
      US_TH = US_HS*HS_TH + US_HK*HK_TH + US_H*H_TH
      US_DS = US_HS*HS_DS + US_HK*HK_DS + US_H*H_DS
      US_RH = US_HS*HS_RH
      US_UE = US_HS*HS_UE + US_HK*HK_UE
      US_RR = US_HS*HS_RR + US_HK*HK_RR + US_H*H_RR
      US_SH = US_HS*HS_SH + US_HK*HK_SH
      US_SR = US_HS*HS_SR
      US_RE = US_HS*HS_RE
      US_RO = US_HS*HS_RO + US_HK*HK_RO
C
      IF(ITYP.NE.3 .AND. US.GT.0.97) THEN
CCC       WRITE(*,*) 'BLVAR: Us clamped:', US
       US    = 0.97
       US_TH = 0.
       US_DS = 0.
       US_RH = 0.
       US_UE = 0.
       US_RR = 0.
       US_SH = 0.
       US_SR = 0.
       US_RE = 0.
       US_RO = 0.
      ENDIF
C
c      IF(ITYP.EQ.3 .AND. US.GT.0.99995) THEN
cCCC       WRITE(*,*) 'BLVAR: Wake Us clamped:', US
c       US    = 0.99995
c       US_TH = 0.
c       US_DS = 0.
c       US_RH = 0.
c       US_UE = 0.
c       US_RR = 0.
c       US_SH = 0.
c       US_SR = 0.
c       US_RE = 0.
c       US_RO = 0.
c      ENDIF
C
C=================================================
C---- Cf
C
      IF(ITYP.EQ.3) THEN
       CF    = 0.
       CF_HK = 0.
       CF_RT = 0.
       CF_MS = 0.
      ELSE IF(ITYP.EQ.1) THEN
       CALL CFL(HK, RT, MS,      CF, CF_HK, CF_RT, CF_MS )
      ELSE
       CALL CFT(HK, RT, MS, GAM, CF, CF_HK, CF_RT, CF_MS )
C
       CALL CFL(HK, RT, MS,      CFLT, CFLT_HK, CFLT_RT, CFLT_MS )
       IF(CFLT.GT.CF) THEN
C------ use laminar Cf if larger than turbulent Cf  (at very low Rtheta)
ccc         write(*,*) 'Using laminar Cf:', cf, cflt
        CF    = CFLT
        CF_HK = CFLT_HK
        CF_RT = CFLT_RT
        CF_MS = CFLT_MS
       ENDIF
      ENDIF
C
      CF_TH = CF_HK*HK_TH + CF_RT*RT_TH
      CF_DS = CF_HK*HK_DS
      CF_RH =               CF_RT*RT_RH
      CF_UE = CF_HK*HK_UE + CF_RT*RT_UE + CF_MS*MS_UE
      CF_MW = 0.
      CF_RR = CF_HK*HK_RR + CF_RT*RT_RR + CF_MS*MS_RR
      CF_SH = CF_HK*HK_SH + CF_RT*RT_SH + CF_MS*MS_SH
      CF_SR =               CF_RT*RT_SR
      CF_RE =               CF_RT*RT_RE
      CF_RO = CF_HK*HK_RO + CF_RT*RT_RO + CF_MS*MS_RO
C
C=================================================
C
C---- dissipation length scale factor
      IF(ITYP.EQ.3) THEN
C----- increased dissipation length in wake (decrease its reciprocal)
       ALD = DLCON
      ELSE
       ALD = 1.0
      ENDIF
C
C---- equilibrium velocity gradient (1/Ue dUe/dx)_EQ for lag equation
C
      GCC = 0.0
      HKC = HK - 1.0
      HKC_HK = 1.0
      HKC_RT = 0.0
      IF(ITYP.EQ.2) THEN
       GCC = GCCON
       HKC    = HK - 1.0 - GCC/RT
       HKC_HK = 1.0
       HKC_RT =            GCC/RT**2
       IF(HKC .LT. 0.01) THEN
        HKC = 0.01
        HKC_HK = 0.0
        HKC_RT = 0.0
       ENDIF
      ENDIF
C
      HRAT     = HKC    / (GACON*ALD*HK)
      HRAT_HK  = HKC_HK / (GACON*ALD*HK) - HRAT / HK
      HRAT_RT  = HKC_RT / (GACON*ALD*HK)
C
C---- CFS is subtracted from Cf--suction affects only the inner dissipation
      UQ    = (0.5*CF - HRAT**2)  / (GBCON*DS)
      UQ_CF =  0.5                / (GBCON*DS)
      UQ_HK =   -2.0*HRAT*HRAT_HK / (GBCON*DS)
      UQ_RT =   -2.0*HRAT*HRAT_RT / (GBCON*DS)
      UQ_DS = -UQ/DS
cccc      UQ_CFS= -0.5                / (GBCON*DS)
C
      UQ_TH = UQ_CF*CF_TH + UQ_HK*HK_TH + UQ_RT*RT_TH
      UQ_DS = UQ_CF*CF_DS + UQ_HK*HK_DS               + UQ_DS
      UQ_RH = UQ_CF*CF_RH               + UQ_RT*RT_RH
      UQ_UE = UQ_CF*CF_UE + UQ_HK*HK_UE + UQ_RT*RT_UE
      UQ_MW = 0.
      UQ_RR = UQ_CF*CF_RR + UQ_HK*HK_RR + UQ_RT*RT_RR
      UQ_SH = UQ_CF*CF_SH + UQ_HK*HK_SH + UQ_RT*RT_SH
      UQ_SR = UQ_CF*CF_SR               + UQ_RT*RT_SR
      UQ_RE = UQ_CF*CF_RE               + UQ_RT*RT_RE
      UQ_RO = UQ_CF*CF_RO + UQ_HK*HK_RO + UQ_RT*RT_RO
C
c      UQ_TH = UQ_TH + UQ_CFS*2.*CM*US_TH
c      UQ_DS = UQ_DS + UQ_CFS*2.*CM*US_DS
c      UQ_RH = UQ_RH + UQ_CFS*2.*CM*US_RH + UQ_CFS*2.*US*CM_RH
c      UQ_UE = UQ_UE + UQ_CFS*2.*CM*US_UE + UQ_CFS*2.*US*CM_UE
c      UQ_MW = UQ_MW                      + UQ_CFS*2.*US*CM_MW
c      UQ_RR = UQ_RR + UQ_CFS*2.*CM*US_RR + UQ_CFS*2.*US*CM_RR
c      UQ_SH = UQ_SH + UQ_CFS*2.*CM*US_SH + UQ_CFS*2.*US*CM_SH
c      UQ_SR = UQ_SR + UQ_CFS*2.*CM*US_SR + UQ_CFS*2.*US*CM_SR
c      UQ_RE = UQ_RE + UQ_CFS*2.*CM*US_RE
c      UQ_RO = UQ_RO + UQ_CFS*2.*CM*US_RO + UQ_CFS*2.*US*CM_RO
C
C=================================================
C---- equilibrium wake layer shear coefficient (Ctau)_EQ ^ 1/2
C
      HKB = HK - 1.0
      USB = 1.0 - US
C
      CQ    =
     &   SQRT( CTCON*HS*HKB*HKC**2 / (USB*H*HK**2) )
      CQ_HS =  CTCON   *HKB*HKC**2 / (USB*H*HK**2)       * 0.5/CQ
      CQ_US =  CTCON*HS*HKB*HKC**2 / (USB*H*HK**2) / USB * 0.5/CQ
      CQ_HK =  CTCON*HS    *HKC**2 / (USB*H*HK**2)       * 0.5/CQ
     &       - CTCON*HS*HKB*HKC**2 / (USB*H*HK**3) * 2.0 * 0.5/CQ
     &       + CTCON*HS*HKB*HKC    / (USB*H*HK**2) * 2.0 * 0.5/CQ
     &                         *HKC_HK
      CQ_RT =  CTCON*HS*HKB*HKC    / (USB*H*HK**2) * 2.0 * 0.5/CQ
     &                         *HKC_RT
      CQ_H  = -CTCON*HS*HKB*HKC**2 / (USB*H*HK**2) / H   * 0.5/CQ
C
      CQ_TH = CQ_HS*HS_TH + CQ_US*US_TH + CQ_HK*HK_TH + CQ_RT*RT_TH
      CQ_DS = CQ_HS*HS_DS + CQ_US*US_DS + CQ_HK*HK_DS
      CQ_RH = CQ_HS*HS_RH + CQ_US*US_RH               + CQ_RT*RT_RH
      CQ_UE = CQ_HS*HS_UE + CQ_US*US_UE + CQ_HK*HK_UE + CQ_RT*RT_UE
      CQ_RR = CQ_HS*HS_RR + CQ_US*US_RR + CQ_HK*HK_RR + CQ_RT*RT_RR
      CQ_SH = CQ_HS*HS_SH + CQ_US*US_SH + CQ_HK*HK_SH + CQ_RT*RT_SH
      CQ_SR = CQ_HS*HS_SR + CQ_US*US_SR               + CQ_RT*RT_SR
      CQ_RE = CQ_HS*HS_RE + CQ_US*US_RE               + CQ_RT*RT_RE
      CQ_RO = CQ_HS*HS_RO + CQ_US*US_RO + CQ_HK*HK_RO + CQ_RT*RT_RO
C
      CQ_TH = CQ_TH + CQ_H*H_TH
      CQ_DS = CQ_DS + CQ_H*H_DS
      CQ_RR = CQ_RR + CQ_H*H_RR
C
C=================================================
C---- dissipation coefficient function 2 CD / H*
C
      IF(ITYP.EQ.1) THEN
C
       CALL DIL( HK, RT, DI, DI_HK, DI_RT )
C
       DI_TH = DI_HK*HK_TH + DI_RT*RT_TH
       DI_DS = DI_HK*HK_DS
       DI_RH =               DI_RT*RT_RH
       DI_UE = DI_HK*HK_UE + DI_RT*RT_UE
       DI_MW = 0.
       DI_RR = DI_HK*HK_RR + DI_RT*RT_RR
       DI_SH = DI_HK*HK_SH + DI_RT*RT_SH
       DI_SR =               DI_RT*RT_SR
       DI_RE =               DI_RT*RT_RE
       DI_RO = DI_HK*HK_RO + DI_RT*RT_RO
       DI_CT = 0.
C
      ELSE
C
       CALL DIT(    HS,    US,    CF,    CT,  DI,
     &           DI_HS, DI_US, DI_CF, DI_CT       )
C
       DI_TH = DI_HS*HS_TH + DI_US*US_TH + DI_CF*CF_TH
       DI_DS = DI_HS*HS_DS + DI_US*US_DS + DI_CF*CF_DS
       DI_RH = DI_HS*HS_RH + DI_US*US_RH + DI_CF*CF_RH
       DI_UE = DI_HS*HS_UE + DI_US*US_UE + DI_CF*CF_UE
       DI_MW =                             DI_CF*CF_MW
       DI_RR = DI_HS*HS_RR + DI_US*US_RR + DI_CF*CF_RR
       DI_SH = DI_HS*HS_SH + DI_US*US_SH + DI_CF*CF_SH
       DI_SR = DI_HS*HS_SR + DI_US*US_SR + DI_CF*CF_SR
       DI_RE = DI_HS*HS_RE + DI_US*US_RE + DI_CF*CF_RE
       DI_RO = DI_HS*HS_RO + DI_US*US_RO + DI_CF*CF_RO
CCC    DI_CT =                                           DI_CT
C
C----- add on suction contribution to inner dissipation
       DDI    = CM*US**2 / HS
       DDI_CM =    US**2 / HS
       DDI_US = CM*US*2.0/ HS
       DDI_HS = -DDI/HS
C
       DI    = DI    + DDI
       DI_TH = DI_TH                + DDI_US*US_TH + DDI_HS*HS_TH
       DI_DS = DI_DS                + DDI_US*US_DS + DDI_HS*HS_DS
       DI_RH = DI_RH + DDI_CM*CM_RH + DDI_US*US_RH + DDI_HS*HS_RH
       DI_UE = DI_UE + DDI_CM*CM_UE + DDI_US*US_UE + DDI_HS*HS_UE
       DI_MW = DI_MW + DDI_CM*CM_MW
       DI_RR = DI_RR + DDI_CM*CM_RR + DDI_US*US_RR + DDI_HS*HS_RR
       DI_SH = DI_SH + DDI_CM*CM_SH + DDI_US*US_SH + DDI_HS*HS_SH
       DI_SR = DI_SR + DDI_CM*CM_SR + DDI_US*US_SR + DDI_HS*HS_SR
       DI_RE = DI_RE                + DDI_US*US_RE + DDI_HS*HS_RE
       DI_RO = DI_RO + DDI_CM*CM_RO + DDI_US*US_RO + DDI_HS*HS_RO
C          
C----- add on CD contribution of inner shear layer
       IF(ITYP.EQ.3 .AND. DW.GT.0.0) THEN
        DKON = 2.0*CTCON*0.75**3
        DDI    =     DKON*US**3
        DDI_US = 3.0*DKON*US**2
C
        DI    = DI    + DDI          * DW/DWTE
        DI_TH = DI_TH + DDI_US*US_TH * DW/DWTE
        DI_DS = DI_DS + DDI_US*US_DS * DW/DWTE
        DI_RH = DI_RH + DDI_US*US_RH * DW/DWTE
        DI_UE = DI_UE + DDI_US*US_UE * DW/DWTE
CCC     DI_MW = DI_MW
        DI_RR = DI_RR + DDI_US*US_RR * DW/DWTE
        DI_SH = DI_SH + DDI_US*US_SH * DW/DWTE
        DI_SR = DI_SR + DDI_US*US_SR * DW/DWTE
        DI_RE = DI_RE + DDI_US*US_RE * DW/DWTE
        DI_RO = DI_RO + DDI_US*US_RO * DW/DWTE
       ENDIF
C
       IF(ITYP.EQ.2) THEN
C
C------ use laminar CD if larger than turbulent CD  (at very low Rtheta)
        CALL DIL( HK, RT, DILT, DILT_HK, DILT_RT )
C
        IF(DILT .GT. DI) THEN
ccc          write(*,*) 'Using laminar Cd:', DI, DILT
          DI    = DILT
          DI_TH = DILT_HK*HK_TH + DILT_RT*RT_TH
          DI_DS = DILT_HK*HK_DS
          DI_RH =                 DILT_RT*RT_RH
          DI_UE = DILT_HK*HK_UE + DILT_RT*RT_UE
          DI_MW = 0.
          DI_RR = DILT_HK*HK_RR + DILT_RT*RT_RR
          DI_SH = DILT_HK*HK_SH + DILT_RT*RT_SH
          DI_SR =                 DILT_RT*RT_SR
          DI_RE =                 DILT_RT*RT_RE
          DI_RO = DILT_HK*HK_RO + DILT_RT*RT_RO
          DI_CT = 0.
        ENDIF
C
       ENDIF
C
       IF(ITYP.EQ.3) THEN
C------- make sure turbulent wake CD is not smaller than laminar wake CD
         CALL DILW( HK, RT, DILT, DILT_HK, DILT_RT )
         IF(DILT .GT. DI) THEN
ccc          write(*,*) 'CDt CDl Rt Hk:', DI, DILT, RT, HK
          DI    = DILT
          DI_TH = DILT_HK*HK_TH + DILT_RT*RT_TH
          DI_DS = DILT_HK*HK_DS
          DI_RH =                 DILT_RT*RT_RH
          DI_UE = DILT_HK*HK_UE + DILT_RT*RT_UE
          DI_MW = 0.
          DI_RR = DILT_HK*HK_RR + DILT_RT*RT_RR
          DI_SH = DILT_HK*HK_SH + DILT_RT*RT_SH
          DI_SR =                 DILT_RT*RT_SR
          DI_RE =                 DILT_RT*RT_RE
          DI_RO = DILT_HK*HK_RO + DILT_RT*RT_RO
          DI_CT = 0.
         ENDIF
       ENDIF
C
       IF(ITYP.EQ.3) THEN
C------ double dissipation for the wake (two half-wake profiles)
        DI    = DI   *2.0
        DI_TH = DI_TH*2.0
        DI_DS = DI_DS*2.0
        DI_RH = DI_RH*2.0
        DI_UE = DI_UE*2.0
        DI_MW = DI_MW*2.0
        DI_RR = DI_RR*2.0
        DI_SH = DI_SH*2.0
        DI_SR = DI_SR*2.0
        DI_RE = DI_RE*2.0
        DI_RO = DI_RO*2.0
        DI_CT = DI_CT*2.0
       ENDIF
C
      ENDIF
C
C-----add on Cf contribution due to suction 
      IF (ITYP.NE.1 .OR. ITYP.NE.3) THEN
       CF    = CF    + 2.0*CM*US
       CF_TH = CF_TH + 2.0*CM*US_TH
       CF_DS = CF_DS + 2.0*CM*US_DS
       CF_RH = CF_RH + 2.0*CM*US_RH + 2.0*US*CM_RH
       CF_UE = CF_UE + 2.0*CM*US_UE + 2.0*US*CM_UE
       CF_MW = CF_MW                + 2.0*US*CM_MW
       CF_RR = CF_RR + 2.0*CM*US_RR + 2.0*US*CM_RR
       CF_SH = CF_SH + 2.0*CM*US_SH + 2.0*US*CM_SH
       CF_SR = CF_SR + 2.0*CM*US_SR + 2.0*US*CM_SR
       CF_RE = CF_RE + 2.0*CM*US_RE
       CF_RO = CF_RO + 2.0*CM*US_RO + 2.0*US*CM_RO
      ENDIF
C
C============================================================
C---- fill Jacobian array for passing back to calling routine
C
      DO J=1, JTOT
        DO L=0, ITOT
          VJ(L,J) = 0.
        ENDDO
        DO L=1, LTOT
          PJ(L,J) = 0.
        ENDDO
      ENDDO
C
C      Output:  VJ(0,JRH)  Rhoe
C               VJ(0,JMS)  Me^2
C               VJ(0,JHK)  Hk
C               VJ(0,JHS)  H*
C               VJ(0,JHC)  H**
C               VJ(0,JHD)  Hr
C               VJ(0,JUS)  Us
C               VJ(0,JCF)  Cf
C               VJ(0,JDI)  2CD/H*
C               VJ(0,JUQ)  (1/qe dqe/ds)EQ
C               VJ(0,JDE)  Delta
C               VJ(0,JRT)  Rtheta
C
      VJ(  0,JRH) = RHO
      VJ(IRH,JRH) = RHO_RH
      VJ(IUE,JRH) = RHO_UE
      VJ(IRR,JRH) = RHO_RR
      PJ(LSH,JRH) = RHO_SH
      PJ(LSR,JRH) = RHO_SR
      PJ(LRO,JRH) = RHO_RO
C
      VJ(  0,JMS) = MS
      VJ(IUE,JMS) = MS_UE
      VJ(IRR,JMS) = MS_RR
      PJ(LSH,JMS) = MS_SH
      PJ(LRO,JMS) = MS_RO
C
      VJ(  0,JHK) = HK
      VJ(ITH,JHK) = HK_TH
      VJ(IDS,JHK) = HK_DS
      VJ(IUE,JHK) = HK_UE
      VJ(IRR,JHK) = HK_RR
      PJ(LSH,JHK) = HK_SH
      PJ(LRO,JHK) = HK_RO
C
      VJ(  0,JHS) = HS
      VJ(ITH,JHS) = HS_TH
      VJ(IDS,JHS) = HS_DS
      VJ(IRH,JHS) = HS_RH
      VJ(IUE,JHS) = HS_UE
      VJ(IRR,JHS) = HS_RR
      PJ(LSH,JHS) = HS_SH
      PJ(LSR,JHS) = HS_SR
      PJ(LRE,JHS) = HS_RE
      PJ(LRO,JHS) = HS_RO
C
      VJ(  0,JHC) = HC
      VJ(ITH,JHC) = HC_TH
      VJ(IDS,JHC) = HC_DS
      VJ(IRH,JHC) = HC_RH
      VJ(IUE,JHC) = HC_UE
      VJ(IRR,JHC) = HC_RR
      PJ(LSH,JHC) = HC_SH
      PJ(LSR,JHC) = HC_SR
      PJ(LRE,JHC) = HC_RE
      PJ(LRO,JHC) = HC_RO
C
      VJ(  0,JHD) = HD
      VJ(ITH,JHD) = HD_TH
      VJ(IDS,JHD) = HD_DS
      VJ(IUE,JHD) = HD_UE
      VJ(IRR,JHD) = HD_RR
      PJ(LSH,JHD) = HD_SH
      PJ(LRO,JHD) = HD_RO
C
      VJ(  0,JUS) = US
      VJ(ITH,JUS) = US_TH
      VJ(IDS,JUS) = US_DS
      VJ(IRH,JUS) = US_RH
      VJ(IUE,JUS) = US_UE
      VJ(IRR,JUS) = US_RR
      PJ(LSH,JUS) = US_SH
      PJ(LSR,JUS) = US_SR
      PJ(LRE,JUS) = US_RE
      PJ(LRO,JUS) = US_RO
C
      VJ(  0,JCF) = CF
      VJ(ITH,JCF) = CF_TH
      VJ(IDS,JCF) = CF_DS
      VJ(IRH,JCF) = CF_RH
      VJ(IUE,JCF) = CF_UE
      VJ(IMW,JCF) = CF_MW
      VJ(IRR,JCF) = CF_RR
      PJ(LSH,JCF) = CF_SH
      PJ(LSR,JCF) = CF_SR
      PJ(LRE,JCF) = CF_RE
      PJ(LRO,JCF) = CF_RO
C
      VJ(  0,JDI) = DI
      VJ(ICT,JDI) = DI_CT
      VJ(ITH,JDI) = DI_TH
      VJ(IDS,JDI) = DI_DS
      VJ(IRH,JDI) = DI_RH
      VJ(IUE,JDI) = DI_UE
      VJ(IMW,JDI) = DI_MW
      VJ(IRR,JDI) = DI_RR
      PJ(LSH,JDI) = DI_SH
      PJ(LSR,JDI) = DI_SR
      PJ(LRE,JDI) = DI_RE
      PJ(LRO,JDI) = DI_RO
C
      VJ(  0,JCM) = CM
      VJ(IRH,JCM) = CM_RH
      VJ(IUE,JCM) = CM_UE
      VJ(IMW,JCM) = CM_MW
      VJ(IRR,JCM) = CM_RR
      PJ(LSH,JCM) = CM_SH
      PJ(LSR,JCM) = CM_SR
      PJ(LRO,JCM) = CM_RO
C
      VJ(  0,JUQ) = UQ
      VJ(ITH,JUQ) = UQ_TH
      VJ(IDS,JUQ) = UQ_DS
      VJ(IRH,JUQ) = UQ_RH
      VJ(IUE,JUQ) = UQ_UE
      VJ(IMW,JUQ) = UQ_MW
      VJ(IRR,JUQ) = UQ_RR
      PJ(LSH,JUQ) = UQ_SH
      PJ(LSR,JUQ) = UQ_SR
      PJ(LRE,JUQ) = UQ_RE
      PJ(LRO,JUQ) = UQ_RO
C
      VJ(  0,JCQ) = CQ
      VJ(ITH,JCQ) = CQ_TH
      VJ(IDS,JCQ) = CQ_DS
      VJ(IRH,JCQ) = CQ_RH
      VJ(IUE,JCQ) = CQ_UE
      VJ(IRR,JCQ) = CQ_RR
      PJ(LSH,JCQ) = CQ_SH
      PJ(LSR,JCQ) = CQ_SR
      PJ(LRE,JCQ) = CQ_RE
      PJ(LRO,JCQ) = CQ_RO
C
      VJ(  0,JDE) = DE
      VJ(ITH,JDE) = DE_TH
      VJ(IDS,JDE) = DE_DS
      VJ(IUE,JDE) = DE_UE
      VJ(IRR,JDE) = DE_RR
      PJ(LSH,JDE) = DE_SH
      PJ(LRO,JDE) = DE_RO
C
      VJ(  0,JRT) = RT
      VJ(ITH,JRT) = RT_TH
      VJ(IRH,JRT) = RT_RH
      VJ(IUE,JRT) = RT_UE
      VJ(IRR,JRT) = RT_RR
      PJ(LSH,JRT) = RT_SH
      PJ(LSR,JRT) = RT_SR
      PJ(LRE,JRT) = RT_RE
      PJ(LRO,JRT) = RT_RO
C
      RETURN
      END ! BLVAR
