
      SUBROUTINE MRCHBL(IMODE,
     &                  HTOT,RTOT,RETOT,ROT,
     &                  GAM,HSUTH, AMCRIT,XITRIP,DWTE,
     &                  LWALL,IPRNT1,
     &                  NPTS,XI,RAD,BST,WGAP,
     &                  UEDG,CTAU,THET,DSTR,RHOE,CURV,UWAL,
     &                  MWAL,
     &                  TAU,
     &                  VISRES,DVISR1,DVISR2,DVISPR,DVISXT,
     &                  ITRAN,KTRAN,XITRAN,DHSDHK)
C----------------------------------------------------------------------
C
C     Marches the BLs and wake in one of three modes:
C
C   IMODE = 0:  All primary and transition variables are input, 
C               and only the residuals and Jacobians are returned.
C               No marching calculation in the usual sense is performed.
C
C   IMODE = 1:  Only Ue is taken as input.  The BL is marched in
C               a standard "direct" mode with the prescribed Ue. 
C               If separation occurs, a plausible value of the 
C               kinematic shape parameter Hk is prescribed until
C               reattachment occurs.
C
C   IMODE = 2:  Both Ue and Delta* are taken as input.  The BL is marched
C               in a "mixed" mode, with a linear combination of the
C               deviations from the input Ue and Delta* being prescribed.
C               The linear combination is set up to intersect the
C               BL's Ue-Delta* dependence curve, so that the Goldstein
C               or Levy-Lees singularities are never encountered.  
C
C
C     For IMODE=1,2, continuous checking of transition onset is performed.
C
C     For IMODE=1,2, the BL residuals will normally be returned as zero,
C        except for any interval which suffers converence failure.
C
C
C     This version implements quasi-3D  and  rotation effects.
C
C       - The radius RAD(i) and rotation rate ROT appear in centrifugal
C         BL equation terms.
C
C       - The streamtube thickness BST(i) appears in the momentum equation
C         via continuity.
C
C       - The BL coordinate XI(i) is arc length in m'-theta space.
C         A physical length increment is  dS = RAD * dXI.  The BL
C         thickness variables DSTR,THET are lengths.
C
C
C     Two types of starting solutions can be used:
C
C     1) If    XI(1) = 0 
C         and  XI(1) < XITRIP
C         and  LWALL = t      
C        Then a similar laminar (Falkner-Skan) flow is assumed 
C           to exist over XI(1) .. XI(2).  Starting variables
C           are NOT necessary, except for UEDG(1).
C
C     2) Otherwise, the flow is started from i=1, and hence
C           CTAU(1), THET(1), DSTR(1), UEDG(1), RHOE(1) must all be input.
C      
C
C  Input:
C  ------
C    IMODE       (see above)
C
C    stagnation quantities are in non-rotating frame...
C
C    HTOT        stagnation enthalpy  h_o    (rothalpy)
C    RTOT        stagnation density   rho_o
C    RETOT       stagnation Reynolds number/unit-length
C                  = rho_o a_o / mu_o     ;  a_o = sqrt((gam-1) h_o) 
C    ROT         rotation rate  (rad/unit-time = rad/(unit-length/a_o) )
C
C    GAM         cp/cv
C    HSUTH       cp x 110 K  (Sutherland's constant for air)
C
C    AMCRIT      critical amplification "n"
C    XITRIP      trip XI location
C    DWTE        value of WGAP at trailing edge point (used only for wake)
C                (DWTE*RAD is physical TE base thickness)
C
C    LWALL       T = wall      F = wake
C    IPRNT1      index shift for info printout (i is printed as i+IPRNT1-1)
C    NPTS        number of points  1 ... i ... NPTS
C    XI  (i)     BL coordinate in m'-theta space, must not be negative
C    RAD (i)     streamtube radius      ( = constant for 2D )
C    BST (i)     streamtube thickness   ( = constant for 2D )
C    WGAP(i)     wake gap width for blunt-TE model (used only for wake)
C                (WGAP*RAD is physical wake gap width)
C    CURV(i)     displacement-surface curvature
C    UWAL(i)     wall velocity (or wake centerline velocity)
C    MWAL(i)     wall suction mass flow rate
C
C  Input AND Output for IMODE shown, just Output otherwise:
C  ----------------------------------------------------
C  0 2  CTAU(i)     amplification ratio   (    1 < i < ITRAN-1)
C                   max shear coefficient (ITRAN < i < NPTS   )
C  0 2  THET(i)     Theta     mom.  thickness (in same length units as RAD)
C  0 2  DSTR(i)     Delta*    disp. thickness (in same length units as RAD)
C  012  UEDG(i)     Ue        edge velocity
C  012  RHOE(i)     Rhoe      edge density
C  0 2  ITRAN       transition interval index,  > NPTS  if no transition
C  0    XITRAN      transition XI location,  > XI(NPTS) if no transition
C
C  Output:
C  -------
C    TAU(i)    tau       skin friction, in same units as (RTOT*HTOT)
C
C Residuals and Jacobians are for the 3 BL equations (first index):
C           1       Ctau lag equation,  e^n equation in laminar region 
C           2       integral momentum equation
C           3       integral kinetic energy shape parameter equation
C    VISRES(3,i)    viscous equation residuals for  i-1 .. i  interval
C    DVISR1(3,11,i) Jacobian wrt i-1 variables  |  variable and parameter
C    DVISR2(3,11,i) Jacobian wrt  i  variables  |     indices are defined
C    DVISPR(3,4,i)  Jacobian wrt parameters     |        in INDEX.INC
C    DVISXT(3)      Jacobian wrt XITRIP for  i = ITRAN-1..ITRAN interval
C                   ( = 0 indicates free transition )
C    KTRAN          = 0  no  criterion triggered
C                   = 1  e^n criterion triggered
C                   = 2  AGS criterion triggered
C    DHSDHK(i)      d(H*)/d(Hk) ( > 0 indicates separated flow)
C
C........................................................................
C
C    Typical direct-march usage (IMODE=1) for airfoil/wake combination:
C
C    1) CALL MRCHUE   for side 1,  UEDG specified, XI(1) = 0
C    2) CALL MRCHUE   for side 2,  UEDG specified, XI(1) = 0
C    3) Set initial wake point values using the side 1,2 TE-point values:
C        THw = TH1 + TH2                (Theta simply adds)
C        DSw = DS1 + DS2 + DWTE*RR      (Delta* has TE base height added)
C        CTw = (TH1*CT1 + TH2*CT2)/THw  (Ctau is Theta-averaged)
C    4) CALL MRCHUE   for wake  ,  UEDG specified, XI(1) > 0
C
C      Calculating CTw requires that the TE values CT1,CT2 be valid,
C      i.e. the TE points must be turbulent.  This can be ensured by
C      running steps 1),2) with a trip set at the TE: XITRIP = XI(NPTS)
C
C      The wake gap WGAP decreases from  WGAP(1) = DWTE  to  zero
C      over a short distance -- the standard ISES model uses a smooth
C      cubic decrease over a distance of L = 2.5 DWTE.  The gap
C      closure rate matches the airfoil TE and is zero at the end:
C
C      At X_TE     ...   d(WGAP)/dXI = d(thickness)/dx  of airfoil
C      At X_TE + L ...   d(WGAP)/dXI = 0
C
C
C    Note: There are two exit points in the internal Newton loop.
C          The best exit point depends on how the MRCHBL is being used.
C          (see comments below).
C
C........................................................................
C
C    Conversion from static reference quantities to the stagnation
C    quantities required by MRCHBL:
C
C      For a case with reference (e.g. freestream) velocity V, 
C      Mach number M, static density and viscosity rho, mu,  
C      the correct set of MRCHBL input parameters is as follows:
C
C    u_e   = (u_e/V) V                                        = UEDG(i)
C
C    h_o   = (V/M)^2 (1 + 0.5 gm1 M^2) / gm1                  = HTOT
C    rho_o = rho (1 + 0.5 gm1 M^2)^(1/gm1)                    = RTOT
C
C    h_o/h = 1 + 0.5 gm1 M^2
C    mu_o  = mu (h_o/h)^1.5  (h/h_o + h_s/h_o) / (1 + h_s/h_o)
C    a_o   = sqrt(gm1 h_o)
C    Re_o  = rho_o a_o / mu_o                                 = RETOT
C
C     where  h_s = HSUTH ,  gm1 = GAM - 1
C
C----------------------------------------------------------------------
      IMPLICIT REAL (A-H,M,O-Z)
      INCLUDE 'INDEX.INC'
C
C---- passed variables
      DIMENSION XI(NPTS),RAD(NPTS),BST(NPTS),WGAP(NPTS),
     &          UEDG(NPTS),RHOE(NPTS),
     &          CTAU(NPTS),THET(NPTS),
     &          DSTR(NPTS),CURV(NPTS),
     &          UWAL(NPTS),MWAL(NPTS),TAU(NPTS)
      DIMENSION VISRES(3,NPTS),
     &          DVISR1(3,ITOT,NPTS),
     &          DVISR2(3,ITOT,NPTS),
     &          DVISPR(3,LTOT,NPTS), 
     &          DVISXT(3)
      DIMENSION DHSDHK(NPTS)
      LOGICAL LWALL
C
C---- local arrays
      DIMENSION PAR(LTOT)
      DIMENSION VAR1(ITOT), VJ1(0:ITOT,JTOT), PJ1(LTOT,JTOT),
     &          VAR2(ITOT), VJ2(0:ITOT,JTOT), PJ2(LTOT,JTOT)
      DIMENSION XIT_VAR1(ITOT),XIT_VAR2(ITOT),XIT_PAR(LTOT)
      DIMENSION BLRES(3), BLRV1(3,ITOT), BLRV2(3,ITOT), 
     &          BLRPAR(3,LTOT), BLRXIT(3)
      DIMENSION ABL(4,4), DVAR(4)
      LOGICAL SIMI, TRAN, TURB, WAKE, DIRECT, NEWBLV, LAMINR
      CHARACTER*4 CTYPE
C
C============================================================
C
C---- convergence tolerance
      DATA EPS /1.0E-6/
c###      DATA EPS /1.0D-14/
C
C---- assume that forced transition, if any, is at trip location
      XIFORC = XITRIP
C
C---- fill parameter array for passing to other routines
      PAR(LSH) = HTOT
      PAR(LSR) = RTOT
      PAR(LRE) = RETOT
      PAR(LRO) = ROT
C
C---- save old transition interval index
      ITROLD = ITRAN
C
C---- assume there will be no transition for direct march
      IF(IMODE.EQ.0) THEN
       ITRAN = NPTS + 2
      ENDIF
      XITRAN = XI(NPTS) + 1.0
C
C---- set wake flag
      WAKE = .NOT. LWALL
C
C============================================================
C---- set stuff for first point
C
      IO = 1
C
C---- interval-type flags 
      SIMI = XI(IO) .EQ. 0.0
      TRAN = .FALSE.
      TURB = XIFORC .LE. XI(IO)  .OR.  WAKE
C
C---- set laminar/turbulent/wake variable-type indicator
      IF(WAKE) THEN
       IVTYP1 = 3
      ELSE IF(TURB) THEN
       IVTYP1 = 2
      ELSE
       IVTYP1 = 1
      ENDIF
C
      IF(SIMI) THEN
C----- set initial guess for similarity station
C
C----- set similarity parameter x/Ue dUe/dx ...
C
C      ... using downstream interval (not linearized!)
       BULE = LOG(UEDG(IO+2)/UEDG(IO+1))
     &      / LOG(  XI(IO+2)/  XI(IO+1))
C
cccC      ... using a fixed value (should be 1.0 for densly-resolved stag. pt.)
ccc       BULE = 0.5
C
       BULE = MAX( -.08 , BULE )
C
       XSI = XI(IO+1)
       UEI = UEDG(IO+1)
       VEI = SQRT((GAM-1.0)*HTOT) / RETOT
C
C----- set initial Theta^2 using Thwaites' formula
       TSQ = 0.45 * VEI * XSI/(UEI*(5.0*BULE+1.0))
C
       VAR1(ICT) = 0.
       VAR1(ITH) = SQRT(TSQ)
       VAR1(IDS) = SQRT(TSQ) * 2.2
       VAR1(IAM) = 0.
C
C----- fudge initial Delta* to get initial specified _kinematic_ Hk
       UROT = VAR1(IRR)*ROT
       MSQ = UEI**2 / ((GAM-1.0)*(HTOT - 0.5*UEI**2 + 0.5*UROT**2))
       CALL DSLIM(VAR1(IDS),VAR1(ITH),VAR1(IUE),MSQ,GAM,2.2 )
C
      ELSE
C
C----- use passed-in first point as starting solution
       VAR1(ICT) = CTAU(IO)
       VAR1(ITH) = THET(IO)
       VAR1(IDS) = DSTR(IO)
       VAR1(IUE) = UEDG(IO)
       VAR1(IRH) = RHOE(IO)
       VAR1(ICV) = CURV(IO)
       VAR1(IXI) =   XI(IO)
       VAR1(IRR) =  RAD(IO)
       VAR1(IBB) =  BST(IO)
       VAR1(IAM) = CTAU(IO)
       VAR1(IMW) = MWAL(IO)
       DW1       = WGAP(IO)
C
C----- calculate secondary BL variables
       CALL BLVAR( IVTYP1, PAR, GAM,HSUTH,
     &             DW1, DWTE, VAR1,VJ1,PJ1 )
C
      ENDIF
C
C============================================================
C---- march downstream ...
C
      DO 1000 IO=2, NPTS
        IM = IO-1
C
C------ set similarity flag
        SIMI = XI(IM) .EQ. 0.0
C
C------ initialize current station to passed-in variables
        VAR2(ICT) = CTAU(IO)
        VAR2(ITH) = THET(IO)
        VAR2(IDS) = DSTR(IO)
        VAR2(IUE) = UEDG(IO)
        VAR2(IRH) = RHOE(IO)
        VAR2(ICV) = CURV(IO)
        VAR2(IXI) =   XI(IO)
        VAR2(IRR) =  RAD(IO)
        VAR2(IBB) =  BST(IO)
        VAR2(IAM) = CTAU(IO)
        VAR2(IMW) = MWAL(IO)
        DW2       = WGAP(IO)
C
        IF     (IMODE.EQ.0) THEN
C
          TRAN = IO .EQ. ITROLD
C
        ELSE IF(IMODE.EQ.1) THEN
C
C-------- IMODE=1: flow variables aren't passed in, so use upstream station
          VAR2(ICT) = VAR1(ICT)
          VAR2(ITH) = VAR1(ITH)
          VAR2(IDS) = VAR1(IDS)
          VAR2(IAM) = VAR1(IAM)
C
C-------- try specified Ue first
          DIRECT = .TRUE.
C
        ELSE IF(IMODE.EQ.2) THEN
C
          TRAN = IO.EQ.ITROLD .AND. .NOT.TURB
C
C-------- if we're in laminar flow which was turbulent, initialize AMPL
          IF(IO.LT.ITRAN .AND. IO.GE.ITROLD) VAR2(IAM) = VAR1(IAM)
C
C-------- if we're in turbulent flow which was laminar, initialize CTAU
          IF(IO.GE.ITRAN .AND. IO.LT.ITROLD) VAR2(ICT) = VAR1(ICT)
C
        ENDIF
C
C------ set laminar/turbulent/wake variable-type indicator
        IF(WAKE) THEN
         IVTYP2 = 3
        ELSE IF(TURB .OR. TRAN) THEN
         IVTYP2 = 2
        ELSE
         IVTYP2 = 1
        ENDIF
C
C------ calculate secondary BL variables VJ2
        CALL BLVAR( IVTYP2, PAR, GAM,HSUTH,
     &              DW2, DWTE, VAR2,VJ2,PJ2 )
C
C
        IF(IMODE.EQ.0) THEN
C======== just set up Newton system with passed-in variables (no iteration)
C
          IF(IO.EQ.ITROLD) THEN
            CALL TRCHEK(VAR1,VAR2, VJ1,VJ2, PJ1,PJ2, 
     &                  AMCRIT, XIFORC,
     &                  KTRAN,
     &                  XIT, XIT_VAR1, XIT_VAR2, XIT_PAR, XIT_XIF,
     &                  AMPL2 )
            TRAN = KTRAN .NE. 0
C
C---------- if .NOT.TRAN, then ITROLD doesn't match passed-in solution
            IF(.NOT.TRAN)
     &         WRITE(*,*) '? MRCHBL: TRAN = f  for transition interval'
          ELSE
            TRAN = .FALSE.
          ENDIF
C
C-------- set up BL equation system
          CALL BLSYS(SIMI, TRAN, TURB, WAKE,
     &           PAR, GAM, HSUTH,
     &           AMCRIT, XIT,
     &           BULE,
     &           VAR1 , VAR2 ,
     &           VJ1  , VJ2  ,
     &           PJ1  , PJ2  ,
     &           BLRV1, BLRV2, BLRPAR, BLRXIT, BLRES )
C
          IF(TRAN) THEN
C---------- add on senstivities of residual to XIT to get total Jacobians
            DO 10 K=1, 3
              DO 102 L=1, ITOT
                BLRV1(K,L) = BLRV1(K,L) + BLRXIT(K)*XIT_VAR1(L)
                BLRV2(K,L) = BLRV2(K,L) + BLRXIT(K)*XIT_VAR2(L)
 102          CONTINUE
              DO 104 L=1, LTOT
                BLRPAR(K,L) = BLRPAR(K,L) + BLRXIT(K)*XIT_PAR(L)
 104          CONTINUE
 10         CONTINUE
          ENDIF
C
C-------- skip Newton iteration for current station
          GO TO 115
C
        ENDIF
C
C
C------ save current variables for constructing Ue-Delta* relation for IMODE=2
        UEREF = VAR2(IUE)
        DSREF = VAR2(IDS)
        HKREF = VJ2(0,JHK)
C
C------ Newton iteration loop
        DO 100 ITER = 1, 20
C
C-------- if interval is currently laminar or contains transition...
          IF(.NOT.(TURB.OR.SIMI)) THEN
C
C---------- ... check or re-check for transition
            CALL TRCHEK(VAR1,VAR2, VJ1,VJ2, PJ1,PJ2, 
     &                  AMCRIT, XIFORC,
     &                  KTRAN,
     &                  XIT, XIT_VAR1, XIT_VAR2, XIT_PAR, XIT_XIF,
     &                  AMPL2 )
            TRAN = KTRAN .NE. 0
C
            IF(TRAN) THEN
C----------- transition is inside interval... 
C
C----------- if secondary variables are not turbulent, must recalculate them
             NEWBLV = IVTYP2 .NE. 2
C
C----------- initialize Ctau for iteration?
             IF(VAR2(ICT) .LE. 0.0 .OR.
     &          VAR2(ICT) .GE. 0.5      ) THEN
              VAR2(ICT) = 0.03
              NEWBLV = .TRUE.
             ENDIF
C
             IVTYP2 = 2
            ELSE
C----------- no transition inside interval...
C
C----------- if secondary variables are not laminar, must recalculate them
             NEWBLV = IVTYP2 .NE. 1
C
C----------- store integrated amplification ratio
             VAR2(IAM)  = AMPL2
C
             IVTYP2 = 1
            ENDIF
C
C---------- recalculate secondary BL variables if necessary
            IF(NEWBLV)
     &        CALL BLVAR( IVTYP2, PAR, GAM,HSUTH,
     &                    DW2,DWTE, VAR2,VJ2,PJ2  )
C
          ENDIF
C

c          if(iter.eq.19) then
c            write(8,*) '      IVTYP1 =', IVTYP1
c            write(8,*) '      IVTYP2 =', IVTYP2
c            write(8,*) '      RTOT   =', RTOT
c            write(8,*) '      HTOT   =', HTOT
c            write(8,*) '      RETOT  =', RETOT
c            write(8,*) '      GAM    =', GAM
c            write(8,*) '      HSUTH  =', HSUTH
c            write(8,*) '      AMCRIT =', AMCRIT
c            write(8,*) '      XIFORC =', XIFORC
c            write(8,*) '      XIT    =', XIT
c            write(8,*) '      ROT    =', ROT
c            write(8,*) '      VAR1(ICT) =',VAR1(ICT)
c            write(8,*) '      VAR1(ITH) =',VAR1(ITH)
c            write(8,*) '      VAR1(IDS) =',VAR1(IDS)
c            write(8,*) '      VAR1(IUE) =',VAR1(IUE)
c            write(8,*) '      VAR1(IRH) =',VAR1(IRH)
c            write(8,*) '      VAR1(ICV) =',VAR1(ICV)
c            write(8,*) '      VAR1(IXI) =',VAR1(IXI)
c            write(8,*) '      VAR1(IRR) =',VAR1(IRR)
c            write(8,*) '      VAR1(IBB) =',VAR1(IBB)
c            write(8,*) '      VAR1(IAM) =',VAR1(IAM)
c            write(8,*) '      VAR1(IMW) =',VAR1(IMW)
c            write(8,*) '      VAR2(ICT) =',VAR2(ICT)
c            write(8,*) '      VAR2(ITH) =',VAR2(ITH)
c            write(8,*) '      VAR2(IDS) =',VAR2(IDS)
c            write(8,*) '      VAR2(IUE) =',VAR2(IUE)
c            write(8,*) '      VAR2(IRH) =',VAR2(IRH)
c            write(8,*) '      VAR2(ICV) =',VAR2(ICV)
c            write(8,*) '      VAR2(IXI) =',VAR2(IXI)
c            write(8,*) '      VAR2(IRR) =',VAR2(IRR)
c            write(8,*) '      VAR2(IBB) =',VAR2(IBB)
c            write(8,*) '      VAR2(IAM) =',VAR2(IAM)
c            write(8,*) '      VAR2(IMW) =',VAR2(IMW)
c          endif

c          iprt = IO+IPRNT1-1
c          if(iprt.ge.152 .and. iprt.le.154) then
c             write(8,*)
c             write(8,*) iprt, iter, tran, turb
c             do i=1, itot
c               write(8,*) i, var2(i)
c             enddo
c             write(8,*)
c             do j=1, jtot
c               write(8,*) j, vj2(0,j)
c             enddo
c             write(8,*)
c             btmp = 2.0*vj2(0,jhc)/vj2(0,jhs)
c     &            + 1.0 - var2(ids)/var2(ith)
c             write(8,*) btmp, bule
c          endif


C-------- set up BL equation system
          CALL BLSYS(SIMI, TRAN, TURB, WAKE,
     &           PAR, GAM, HSUTH,
     &           AMCRIT, XIT,
     &           BULE,
     &           VAR1 , VAR2 ,
     &           VJ1  , VJ2  ,
     &           PJ1  , PJ2  ,
     &           BLRV1, BLRV2, BLRPAR, BLRXIT, BLRES )
C
          IF(TRAN) THEN
C---------- add on senstivities of residual to XIT to get total Jacobians
            DO 15 K=1, 3
              DO 152 L=1, ITOT
                BLRV1(K,L) = BLRV1(K,L) + BLRXIT(K)*XIT_VAR1(L)
                BLRV2(K,L) = BLRV2(K,L) + BLRXIT(K)*XIT_VAR2(L)
 152          CONTINUE
              DO 154 L=1, LTOT
                BLRPAR(K,L) = BLRPAR(K,L) + BLRXIT(K)*XIT_PAR(L)
 154          CONTINUE
 15         CONTINUE
          ENDIF
C
C
C-------- put "2" system into first 3 lines of work matrix for solution
          DO 20 K=1, 3
            DVAR(K) = -BLRES(K)
            DO 202 L=1, 4
              ABL(K,L) = BLRV2(K,L)
 202        CONTINUE
C---------- if station 2 is laminar:  put IAM column into ICT column
            IF(.NOT.(TURB .OR. TRAN)) ABL(K,ICT) = BLRV2(K,IAM)
 20       CONTINUE
C
C
C-------- add 4th line depending on situation and solve 4x4 system...
C
          IF(SIMI) THEN
C----------- for similiarity station, always prescribe Ue
             CALL UESOL(VAR2,VJ2, UEREF, ABL, DVAR)
C
          ELSE IF(IMODE.EQ.1) THEN
C----------- direct marching... prescribe Ue, or Hk if separated
             IF(DIRECT) THEN
C------------- prescribed Ue
               CALL UESOL(VAR2,VJ2, UEREF, ABL, DVAR)
C
C------------- see if prescribing Hk will be necessary
               CALL HKTEST(HTOT,GAM,ROT, TURB,TRAN,WAKE, XIT,
     &                     VAR1,VJ1, VAR2,VJ2, DVAR,
     &                     DIRECT,HKTARG)
               IF(.NOT.DIRECT) THEN
                 IF    (WAKE) THEN
                  CTYPE = 'wake'
                 ELSEIF(TURB) THEN
                  CTYPE = 'turb'
                 ELSEIF(TRAN) THEN
                  CTYPE = 'tran'
                 ELSE
                  CTYPE = 'lam '
                 ENDIF
                 WRITE(*,1440) IO+IPRNT1-1, HKTARG, CTYPE
 1440            FORMAT(1X,
     &            'MRCHBL: Inverse mode at',I4,'   Hk =', F9.4,3X,A4)
               ENDIF
             ENDIF
C
             IF(.NOT.DIRECT) THEN
C------------- prescribe Hk
               CALL HKSOL(VAR2,VJ2, HKTARG, ABL, DVAR)
             ENDIF
C
          ELSE IF(IMODE.EQ.2) THEN
C----------- mixed-mode marching... prescribe Ue-Delta* relation
C-           using calculated d(Ue)/d(Delta*) from current BL Jacobian
             CALL SENSET(ABL,UE_DS)
             SNEW = UE_DS * (DSREF/UEREF)
             IF    (ITER.LE.4) THEN
               SENS = SNEW
             ELSEIF(ITER.LE.8) THEN
               SENS = 0.5*(SENS + SNEW)
             ENDIF
             CALL DUSOL(VAR2,VJ2, UEREF,DSREF,HKREF, SENS, ABL, DVAR )
C
          ENDIF
C
C
C-------- determine max changes and underrelax if necessary
          LAMINR = .NOT.(TURB .OR. TRAN)
          HSTREL = HTOT + 0.5*(VAR2(IRR)*ROT)**2
          CALL VRLX(VAR2, DVAR, HSTREL, LAMINR, DMAX, RLX)
C
C------------------------------------------------------------------
C-        If the calling routine requires the residuals VISRES,
C-        it is necessary to exit here before the update, 
C-        so that VISRES corresponds to the returned solution
C-         (see alternative exit point below).
C
          IF(DMAX.LE.EPS) GO TO 110
C
C
C-------- Newton update, except for AM2
C-        (AM2 must be set only by TRCHEK for consistency with TRAN, XIT)
          IF(TURB.OR.TRAN) VAR2(ICT) = VAR2(ICT) + RLX*DVAR(ICT)
          VAR2(ITH) = VAR2(ITH) + RLX*DVAR(ITH)
          VAR2(IDS) = VAR2(IDS) + RLX*DVAR(IDS)
          VAR2(IUE) = VAR2(IUE) + RLX*DVAR(IUE)
          IF(SIMI) VAR2(IAM) = 0.
C
C-------- eliminate absurd transients
          IF(WAKE) THEN
            DSI  = VAR2(IDS) - VAR2(IRR)*DW2
            THI  = VAR2(ITH)
            UEI  = VAR2(IUE)
            UROT = VAR2(IRR)*ROT
            MSQ = UEI**2 / ((GAM-1.0)*(HTOT - 0.5*UEI**2 + 0.5*UROT**2))
            CALL DSLIM(DSI,THI,UEI,MSQ,GAM,1.00001 )
            VAR2(IDS) = DSI +  VAR2(IRR)*DW2
          ELSE
            UEI  = VAR2(IUE)
            UROT = VAR2(IRR)*ROT
            MSQ = UEI**2 / ((GAM-1.0)*(HTOT - 0.5*UEI**2 + 0.5*UROT**2))
            CALL DSLIM(VAR2(IDS),VAR2(ITH),UEI,MSQ,GAM,1.1 )
          ENDIF
C
          IF(TURB .OR. TRAN) THEN
            VAR2(ICT) = MIN(VAR2(ICT) , 0.30   )
            VAR2(ICT) = MAX(VAR2(ICT) , 0.00001)
          ENDIF
C
C-------- set new secondary variables after update
          CALL BLVAR( IVTYP2, PAR, GAM,HSUTH,
     &                DW2,DWTE, VAR2,VJ2,PJ2  )
C
C---------------------------------------------------------------------
C-        If MRCHBL is run as a stand-alone BL code, it is slightly
C-        more efficient to exit here, after the Newton update 
C-        (saves one Newton iteration).  The returned residuals
C-        VISRES will *not* correspond to the returned solution.
C
ccc          IF(DMAX.LE.EPS) GO TO 110
C
  100   CONTINUE
        WRITE(*,1450) IO+IPRNT1-1, VJ2(0,JHK), VJ2(0,JRT), DMAX
 1450   FORMAT(1X,'MRCHBL: Conv. failed at', I4,
     &         '   Hk =', F9.4,'   Rt =', F9.1,
     &         '   max(dV) =', E12.4)
C
C------ the current unconverged solution might still be reasonable...
        IF(DMAX .LE. 0.1) GO TO 110
C
C------- the current solution is garbage --> extrapolate values instead
         IF(IO.GT.2) THEN
           VAR2(ICT) = VAR1(ICT)
           VAR2(ITH) = VAR1(ITH)
           VAR2(IDS) = VAR1(IDS)
           VAR2(IUE) = VAR1(IUE)
           IF(TRAN) VAR2(ICT) = 0.7*VJ1(0,JCQ)
         ENDIF
C
C------- set all other extrapolated values for current station
         CALL BLVAR( IVTYP2, PAR, GAM,HSUTH,
     &               DW2,DWTE, VAR2,VJ2,PJ2  )
C
C
C------ pick up here after the Newton iterations
  110   CONTINUE
C
C------ store primary variables
        CTAU(IO) = VAR2(ICT)
        THET(IO) = VAR2(ITH)
        DSTR(IO) = VAR2(IDS)
        UEDG(IO) = VAR2(IUE)
C
C------ CTAU(.) is amplification ratio in laminar regions
        IF(.NOT.(TURB .OR. TRAN)) CTAU(IO) = VAR2(IAM)
C
C
C------ pick up here from IMODE=0 setup block
 115    CONTINUE
C
C
C------ store residuals and final Jacobians
        DO 120 K=1, 3
          VISRES(K,IO) = BLRES(K)
          DO 1205 L=1, ITOT
            DVISR1(K,L,IO) = BLRV1(K,L)
            DVISR2(K,L,IO) = BLRV2(K,L)
 1205     CONTINUE
          DO 1207 L=1, LTOT
            DVISPR(K,L,IO) = BLRPAR(K,L)
 1207     CONTINUE
          IF(TRAN) THEN
            DVISXT(K) = BLRXIT(K)*XIT_XIF
          ELSE
            DVISXT(K) = 0.
          ENDIF
 120    CONTINUE
C
C
C------ store wall shear stress for passing back
        TAU(IO) = VJ2(0,JCF) * 0.5*VJ2(0,JRH)*VAR2(IUE)**2
C
C------ store dH*/dHk for passing back
        HK2 = VJ2(0,JHK)
        RT2 = VJ2(0,JRT)
        MS2 = VJ2(0,JMS)
        CALL HST( HK2, RT2, MS2 , HS2, DHSDHK(IO), HS_RT, HS_MS  )
C
C
C------ move "2" variables into "1" variables for next streamwise station
        DO 140 L=1, ITOT
          VAR1(L) = VAR2(L)
 140    CONTINUE
C
        DO 150 J=1, JTOT
          DO 1505 L=0, ITOT
            VJ1(L,J) = VJ2(L,J)
 1505     CONTINUE
          DO 1507 L=1, LTOT
            PJ1(L,J) = PJ2(L,J)
 1507     CONTINUE
 150    CONTINUE
C
        IVTYP1 = IVTYP2
        DW1 = DW2
C
C
        IF(TRAN) THEN
C------- save transition-interval index and transition location
         ITRAN = IO
         XITRAN = XIT
        ENDIF
C
C------ turbulent intervals will follow transition interval
        IF(TRAN) TURB = .TRUE.
        TRAN = .FALSE.
C
 1000 CONTINUE
C
      RETURN
      END ! MRCHBL



      SUBROUTINE UESOL(VAR,VJ,
     &                 UETARG,
     &                 ABL, DVAR)
      INCLUDE 'INDEX.INC'
      DIMENSION VAR(ITOT), VJ(0:ITOT,JTOT)
      DIMENSION ABL(4,4), DVAR(4)
C
      ABL(4,ICT) = 0.
      ABL(4,ITH) = 0.
      ABL(4,IDS) = 0.
      ABL(4,IUE) = 1.0
      DVAR(4)  = -(VAR(IUE) - UETARG)
C
C---- solve Newton system for current "2" station
      CALL GAUSSN(4,4,ABL,DVAR,1)
C
      RETURN
      END ! UESOL


      SUBROUTINE HKSOL(VAR,VJ,
     &                 HKTARG,
     &                 ABL, DVAR)
      INCLUDE 'INDEX.INC'
      DIMENSION VAR(ITOT), VJ(0:ITOT,JTOT)
      DIMENSION ABL(4,4), DVAR(4)
C
C---- force Hk to prescribed value HKTARG
      ABL(4,ICT) = VJ(ICT,JHK)
      ABL(4,ITH) = VJ(ITH,JHK)
      ABL(4,IDS) = VJ(IDS,JHK)
      ABL(4,IUE) = VJ(IUE,JHK)
      DVAR(4)  = -(VJ(  0,JHK) - HKTARG)
C
      CALL GAUSSN(4,4,ABL,DVAR,1)
C
      RETURN
      END ! HKSOL


      SUBROUTINE DUSOL(VAR,VJ,
     &                 UEREF,DSREF,HKREF,
     &                 SENS, 
     &                 ABL, DVAR)
      INCLUDE 'INDEX.INC'
      DIMENSION VAR(ITOT), VJ(0:ITOT,JTOT)
      DIMENSION ABL(4,4), DVAR(4)
C
C---- constant controlling how far Delta* is allowed to deviate
C-    from the specified value.
ccc   SWT = 1000.0
ccc   SWT = 100.0
      SWT = 20.0
ccc   SWT =  2.5
C
C---- set prescribed Ue-Delta* combination in 4th line
      DS = VAR(IDS)
      UE = VAR(IUE)
      RES  = HKREF**2 * LOG(DS/DSREF)
     &     + SWT*SENS * LOG(UE/UEREF)
      Z_DS = HKREF**2 / DS 
      Z_UE = SWT*SENS / UE
C
      ABL(4,ICT) = 0.
      ABL(4,ITH) = 0.
      ABL(4,IDS) = Z_DS
      ABL(4,IUE) = Z_UE
      DVAR(4)    = -RES
C
C---- solve 4x4 system
      CALL GAUSSN(4,4,ABL,DVAR,1)
C
      RETURN
      END ! DUSOL


      SUBROUTINE SENSET(A,SENS)
      DIMENSION A(4,4)
      DIMENSION B(4,4), D(4)
C
C---- fill first 3 lines of system into work matrix
      DO K=1, 3
        DO L=1, 4
          B(K,L) = A(K,L)
        ENDDO
        D(K) = 0.
      ENDDO
C
C---- set unit dDelta* in 4th line
      B(4,1) = 0.
      B(4,2) = 0.
      B(4,3) = 1.0
      B(4,4) = 0.
      D(4)   = 1.0
C
C---- calculate response of all BL variables
      CALL GAUSSN(4,4,B,D,1)
C
C---- set  SENS = dUe/dDelta*
      SENS = D(4)/D(3)
C
      RETURN
      END ! SENSET


      SUBROUTINE HKTEST(HTOT,GAM,ROT, TURB,TRAN,WAKE, XIT,
     &                  VAR1,VJ1, VAR2,VJ2, DVAR,
     &                  DIRECT,HKTARG)
C-----------------------------------------------------------
C     Performs a dummy update for current Newton iteration.
C     If Hk will exceed separation threshold, then the 
C     specified-Hk toggle is set, and a plausible value
C     for the specified Hk is extrapolated from upstream.
C-----------------------------------------------------------
      IMPLICIT REAL (A-H,M,O-Z)
      INCLUDE 'INDEX.INC'
      DIMENSION VAR1(ITOT), VJ1(0:ITOT,JTOT)
      DIMENSION VAR2(ITOT), VJ2(0:ITOT,JTOT)
      DIMENSION DVAR(4)
      LOGICAL TURB,TRAN,WAKE, DIRECT
      LOGICAL LAMINR
C
C---- turbulent, laminar Hk for separation criterion
      DATA HLMAX, HTMAX / 3.8 , 2.7 /
C
C---- determine under-relaxation factor, if necessary
      LAMINR = .NOT.(TURB .OR. TRAN)
      HSTREL = HTOT + 0.5*(VAR2(IRR)*ROT)**2
      CALL VRLX(VAR2, DVAR, HSTREL, LAMINR, DMAX, RLX)
C
C---- calculate resulting kinematic shape parameter Hk
      TH = VAR2(ITH) + RLX*DVAR(ITH)
      DS = VAR2(IDS) + RLX*DVAR(IDS)
      UE = VAR2(IUE) + RLX*DVAR(IUE)
      UROT = ROT*VAR2(IRR)
      MSQ = UE**2/((GAM-1.0)*(HTOT - 0.5*UE**2 + 0.5*UROT**2))
C
      HNEW = DS/TH
      CALL HKIN( HNEW, MSQ, GAM, HKNEW, DUMMY, DUMMY)
C
C---- decide whether to do direct or inverse problem based on Hk
      IF(LAMINR) THEN
        DIRECT = HKNEW.LT.HLMAX
      ELSE
        DIRECT = HKNEW.LT.HTMAX
      ENDIF
C
C---- don't do anything else if Ue is to be specified
      IF(DIRECT) RETURN
C
C---- unpack some variables
      HK1 = VJ1(0,JHK)
      HK2 = VJ2(0,JHK)
      TH1 = VAR1(ITH)
      TH2 = VAR2(ITH)
      XI1 = VAR1(IXI)
      XI2 = VAR2(IXI)
C
C---- set prescribed Hk for inverse mode at the current station
      IF(.NOT.(TURB.OR.TRAN)) THEN
C----- laminar case: relatively slow increase in Hk downstream
       HKTARG = HK1 + 0.03*(XI2-XI1)/TH1
      ELSE IF(TRAN) THEN
C----- transition interval: weighted laminar and turbulent case
       HKTARG = HK1 + (0.03*(XIT-XI1) - 0.20*(XI2-XIT))/TH1
      ELSE IF(WAKE) THEN
C----- turbulent wake case:
C      asymptotic wake behavior with approx. Backward Euler,
       HK2 = HK1
       HK2 = HK1 - 0.03*(HK2-1.0)**3 * (XI2-XI1)/TH1
       HK2 = HK1 - 0.03*(HK2-1.0)**3 * (XI2-XI1)/TH1
       HK2 = HK1 - 0.03*(HK2-1.0)**3 * (XI2-XI1)/TH1
       HKTARG = HK2
      ELSE
C----- turbulent case: relatively fast decrease in Hk downstream
       HKTARG = HK1 - 0.20*(XI2-XI1)/TH1
      ENDIF
C
C---- limit specified Hk to something reasonable
      IF(WAKE) THEN
       HKTARG = MAX( HKTARG , 1.01 )
      ELSE
       HKTARG = MAX( HKTARG , 2.5  )
      ENDIF
C
      HKTARG = MIN( HKTARG , 8.0  )
C
      RETURN
      END ! HKTEST


      SUBROUTINE VRLX(VAR,DVAR, HSTREL, LAMINR, DMAX, RLX )
C
C---- Returns reasonable under-relaxation factor for viscous update
C
      INCLUDE 'INDEX.INC'
      DIMENSION VAR(ITOT), DVAR(4)
      LOGICAL LAMINR
C
      DMAX = MAX( ABS(DVAR(ITH)) / VAR(ITH) ,
     &            ABS(DVAR(IDS)) / VAR(IDS) ,
     &            ABS(DVAR(IUE)) / VAR(IUE)  )
C
      IF(LAMINR) THEN
        DMAX = MAX( DMAX , ABS(DVAR(ICT)) / 50.0 )
      ELSE
        DMAX = MAX( DMAX , ABS(DVAR(ICT)) /(VAR(ICT)*5.0) )
      ENDIF
C
      RLX = 1.0
      IF(DMAX.GT.0.3) RLX = 0.3/DMAX
C
      QVAC = SQRT(2.0*HSTREL)
      QMAX = MIN( 0.5*(VAR(IUE) + QVAC) , 2.0*VAR(IUE) )
      IF(VAR(IUE)+RLX*DVAR(IUE).GT.QMAX) RLX = (QMAX-VAR(IUE))/DVAR(IUE)
C
      RETURN
      END ! VRLX


      SUBROUTINE DSLIM(DSTR,THET,UEDG,MSQ,GAM,HKLIM)
      IMPLICIT REAL (A-H,M,O-Z)
C
C---- modifies DSTR so that Hk doesn't fall below HKLIM
C
      H = DSTR/THET
      CALL HKIN(H,MSQ,GAM,HK,HK_H,HK_M)
      IF(HK .GT. HKLIM) RETURN
C
      DH = (HKLIM - HK) / HK_H
      DSTR = DSTR + DH*THET
ccc      write(*,*) 'DSLIM:  Hk  dH = ', HK, DH
C
      RETURN
      END
