
      BLVJ(VAR,PAR,GAM,HSUTH,DHW,TRF,RK, VJ,PJ)
C------------------------------------------------------------
C     Calculates turbulent wall velocity, shear profiles,
C     and related parameters.
C
C     Input: primary variables and parameters
C
C       VAR(ICQ)    Cq^1/2     r.m.s. turbulent velocity scale
C       VAR(ITO)    Theta_o    outer-layer momentum thickness
C       VAR(IDO)    Delta*_o   outer-layer displacement thickness
C       VAR(IUE)    u*         EIF velocity scale
C       VAR(IUW)    u_w        wall velocity (or at wake centerline)
C       VAR(IRH)    rho*       EIF density scale
C       VAR(ICV)    kappa      displacement surface curvature, + concave 
C       VAR(IXI)    xi         BL coordinate (in m'-theta space)
C       VAR(IRR)    r          radius
C       VAR(IBB)    b          streamtube thickness
C       VAR(IAM)    n          ln(amplification ratio)
C       VAR(IMW)    m_wall     wall suction mass flow rate
C
C       PAR(LSH)    Htot      total enthalpy  (or rothalpy)
C       PAR(LSR)    Rtot      total density
C       PAR(LRE)    REtot     Reynolds number at stagnation
C       PAR(LRO)    rot       rotation rate
C                
C       GAM         cp/cv
C       HSUTH       Sutherland's enthalpy
C       DHW         (1 - hoa/ho)_wall   wall heating ratio
C       TRF         temperature recovery factor
C       RK          wall roughness height  (for log-law correction)
C
C       NDIM        array dimension
C
C
C     Output: secondary variables and derivatives
C
C       VJ(0,JRH)  Rhoe         edge density
C       VJ(0,JMS)  Me^2         edge Mach^2
C       VJ(0,JHK)  Hk           kinematic shape parameter
C       VJ(0,JHS)  H*           KE shape parameter
C       VJ(0,JHC)  H**          density-flux shape parameter
C       VJ(0,JHD)  Hr           density shape parameter
C       VJ(0,JUS)  Us           apparent slip velocity
C       VJ(0,JCF)  Cf           skin friction coefficient
C       VJ(0,JDI)  2CD/H*       dissipation coefficient x 2/H*
C       VJ(0,JUQ)  (ln Ue)_x    equilibrium pressure gradient
C       VJ(0,JCQ)  CtauEQ^1/2   equilibrium shear stress coefficient
C       VJ(0,JDE)  Delta        BL thickness
C       VJ(0,JRT)  Rtheta       momentum-thickness Reynolds number
C
C       VJ(IUE,JRH)  d(Rhoe)/d(Ue)
C       VJ(IRR,JRH)  d(Rhoe)/d(r)
C
C       VJ(ITH,JHK)  d(Hk)/d(Theta)
C       VJ(IDS,JHK)  d(Hk)/d(Delta*)
C       ... etc
C
C
C       PJ(LSH,JRH)   d(Rhoe)/dHTOT
C       PJ(LSR,JRH)   d(Rhoe)/dRTOT
C       PJ(LRE,JRH)   d(Rhoe)/dREYN
C       PJ(LRO,JRH)   d(Rhoe)/dROT
C       ... etc
C
C
C
C       N           number of profile points
C
C       Y(.)        y coordinate
C       Y_VAR(i,.)  dY(.)/dVAR(i)
C       Y_PAR(l,.)  dY(.)/dPAR(l)
C                          _
C       U(.)        u/u* = U                 normalized velocity profile
C       U_VAR(i,.)  dU(.)/dVAR(i)
C       U_PAR(l,.)  dU(.)/dPAR(l)
C
C       T(.)        tau/(rho* u*^2) = C_tau  normalized shear profile
C       T_VAR(i,.)  dT(.)/dVAR(i)
C       T_PAR(l,.)  dT(.)/dPAR(l)
C
C       UTAU        sqrt( tau_w / (rho* u*^2) )  shear velocity
C       UT_VAR(i)   dUT/dVAR(i)
C       UT_PAR(l)   dUT/dPAR(l)
C
C------------------------------------------------------------
      IMPLICIT REAL (A-H,M,O-Z)
C
      INCLUDE 'INDEX.INC'
      INCLUDE 'BLPAR.INC'
C
      DIMENSION VAR(ITOT), PAR(LTOT)
      DIMENSION VJ(0:ITOT,JTOT), PJ(LTOT,JTOT)
C
C
c     &      NDIM, N,
c     &      Y , Y_VAR, Y_PAR,
c     &      U , U_VAR, U_PAR,
c     &      T , T_VAR, T_PAR,
c     &    UTAU,UT_VAR,UT_PAR )


      DIMENSION YY(NDIM), YY_VAR(NDIM,ITOT), YY_PAR(NDIM,LTOT)
      DIMENSION UU(NDIM), UU_VAR(NDIM,ITOT), UU_PAR(NDIM,LTOT)
      DIMENSION UT_VAR(ITOT), UT_PAR(LTOT)
C
C---- local arrays
      PARAMETER (KK=128)
      DIMENSION U(0:KK), U_VAR(ITOT,0:KK), U_PAR(LTOT,0:KK),
     &          R(0:KK), R_VAR(ITOT,0:KK), R_PAR(LTOT,0:KK),
     &          T(0:KK), T_VAR(ITOT,0:KK), T_PAR(LTOT,0:KK),
     &          Q(0:KK), Q_VAR(ITOT,0:KK), Q_PAR(LTOT,0:KK)
C
      DIMENSION U1_VAR(ITOT),U2_VAR(ITOT),U3_VAR(ITOT),
     &          M1_VAR(ITOT),M2_VAR(ITOT),M3_VAR(ITOT),
     &          CD_VAR(ITOT),TU_VAR(ITOT)

C
C---- Utau convergence tolerance, Romberg convergence tolerance
      DATA UEPS, REPS / 1.0E-6 , 1.0E-4 /
C
C---- decay constant for laminar sublayer function
      DATA FPCON  / 20.0 /
C
C
C---- unpack parameters and variables
      HTOT  = PAR(LSH)
      RTOT  = PAR(LSR)
      RETOT = PAR(LRE)
      ROT   = PAR(LRO)
C
      CQ = VAR(ICQ)
      TO = VAR(ITO)
      DO = VAR(IDO)
      UE = VAR(IUE)
      UW = VAR(IUW)
      RH = VAR(IRH)
      CV = VAR(ICV)
      XI = VAR(IXI)
      RR = VAR(IRR)
CCC   BB = VAR(IBB)
      AM = VAR(IAM)
      MW = VAR(IMW)
C
      GM1 = GAM - 1.0
C
C---- static enthalpy  HE( HTOT UE RR )
      HE    = HTOT - 0.5*UE**2 + 0.5*(RR*ROT)**2
      HE_UE =      -     UE
      HE_RR =                         RR*ROT**2
      HE_SH = 1.0
      HE_RO =                         RR**2*ROT
C
C---- edge Mach^2   MS( UE RR SH RO )
      MS    = UE**2 /(GM1*HE)
      MS_UE = UE*2.0/(GM1*HE) - (MS/HE)*HE_UE
      MS_RR =                 - (MS/HE)*HE_RR
      MS_SH =                 - (MS/HE)*HE_SH
      MS_RO =                 - (MS/HE)*HE_RO
C
C---- edge density  RHO( RH UE RR SH SR RO )
C
c----- rho is isentropically related to Ue
c      RHO = RTOT*(HE/HTOT)**(1.0/GM1)
c      RHO_RH = 0.
c      RHO_UE = RHO/(GM1*HE) * HE_UE
c      RHO_RR = RHO/(GM1*HE) * HE_RR
c      RHO_SH = RHO*(HE_SH/HE - 1.0/HTOT) / GM1
c      RHO_SR = RHO/RTOT
c      RHO_RO = RHO*(HE_RO/HE           ) / GM1
C
C---- rho is an independent variable
      RHO    = RH
      RHO_RH = 1.0
      RHO_UE = 0.
      RHO_RR = 0.
      RHO_SH = 0.
      RHO_SR = 0.
      RHO_RO = 0.
C
C---- stagnation viscosity
      MUTOT = RTOT*SQRT(GM1*HTOT)/RETOT
      MT_SR =     MUTOT/RTOT
      MT_SH = 0.5*MUTOT/HTOT
      MT_RE =    -MUTOT/RETOT

C---- edge viscosity MU( UE RR SR SH RE RO )
      MU    = SQRT((HE/HTOT)**3) * (HTOT+HSUTH)/(HE+HSUTH) * MUTOT
      MU_MT = SQRT((HE/HTOT)**3) * (HTOT+HSUTH)/(HE+HSUTH)
C
      MU_UE = MU*(1.5/HE   - 1.0/(HE  +HSUTH)) * HE_UE
      MU_RR = MU*(1.5/HE   - 1.0/(HE  +HSUTH)) * HE_RR
      MU_SR =                                            MU_MT*MT_SR
      MU_SH = MU*(1.5/HE   - 1.0/(HE  +HSUTH)) * HE_SH + MU_MT*MT_SH
     &      - MU*(1.5/HTOT - 1.0/(HTOT+HSUTH))
      MU_RE =                                            MU_MT*MT_RE
      MU_RO = MU*(1.5/HE   - 1.0/(HE  +HSUTH)) * HE_RO
C
C---- suction coefficient (Rho x v)wall / (Rho x U)edge
      CM    = -MW/(UE*RHO)
      CM_RH = -(CM/RHO)*RHO_RH
      CM_UE = -(CM/RHO)*RHO_UE - CM/UE
      CM_MW =                  - 1.0/(UE*RHO)
      CM_RR = -(CM/RHO)*RHO_RR
      CM_SH = -(CM/RHO)*RHO_SH
      CM_SR = -(CM/RHO)*RHO_SR
      CM_RO = -(CM/RHO)*RHO_RO
C           _
C---- 1 - 1/Ho
      HMI    = 1.0 - TO/DO
      HMI_TO =     - 1./DO
      HMI_DO =       TO/DO**2
C
C---- profile exponent b(To Do)
      BB     = 1.0 / (1.0 - 5.0*HMI**4 + 4.0*HMI**5)
      BB_HMI = BB*BB * 20.0*(HMI**3 - HMI**4)
C
      BB_TO = BB_HMI*HMI_TO
      BB_DO = BB_HMI*HMI_DO
C
C---- profile coefficients b1(b), b2(b), b3(b)
      B1 =  2.0/BB
      B2 = -3.0/BB + BB + 1.0
      B3 =  1.0/BB - BB
C
      B1_BB = -2.0/BB**2
      B2_BB =  3.0/BB**2 + 1.0
      B3_BB = -1.0/BB**2 - 1.0
C
C
C---- F1 = Int[ f(eta) ]
      F1    = B1   *2.0/5.0 + B2   /(3.0+BB)    + B3   /(4.0*BB)
      F1_BB = B1_BB*2.0/5.0 + B2_BB/(3.0+BB)    + B3_BB/(4.0*BB)
     &      +       2.0/5.0 - B2   /(3.0+BB)**2 - B3   /(4.0*BB)**2
C
      F1_TO = F1_BB*BB_TO
      F1_DO = F1_BB*BB_DO
C
C---- F2 = Int[ f(eta)^2 ]
      BB5 = 2.0*BB + 5.0
      BB6 = 2.0*BB + 6.0
      BB7 = 2.0*BB + 7.0
      F2    =   B1*B1/ 4.0     +    B2*B2/BB5      +    B3*B3/BB7
     &   + 2.0*(B1*B2/(4.5+BB) +    B1*B3/(5.5+BB) +    B2*B3/BB6)
C
      F2_BB = (2.0*B1/ 4.0 + 2.0*B2/(4.5+BB) + B3/(5.5+BB))*B1_BB
     &      + (2.0*B2/ BB5 + 2.0*B1/(4.5+BB) + B3/ BB6    )*B2_BB
     &      + (2.0*B3/ BB7 + 2.0*B1/(5.5+BB) + B2/ BB6    )*B3_BB
     &      -  2.0*(B2*B2/BB5**2 + B3*B3/BB7**2 + B2*B3/BB6**2)
     &      -  2.0*(B1*B2/(4.5+BB)**2 + B1*B3/(5.5+BB)**2)
C
      F2_TO = F2_BB*BB_TO
      F2_DO = F2_BB*BB_DO
C
C---- max velocity defect DU(HMI F1 F2)
      DU     =   HMI * (1.0 - F1  ) / (1.0 - 2.0*F1 + F2)
      DU_HMI =         (1.0 - F1  ) / (1.0 - 2.0*F1 + F2)
      DU_F1  = (-HMI + 2.0*DU     ) / (1.0 - 2.0*F1 + F2)
      DU_F2  =            -DU       / (1.0 - 2.0*F1 + F2)
C
C---- DU( TO DO )
      DU_TO = DU_HMI*HMI_TO + DU_F1*F1_TO + DU_F2*F2_TO
      DU_DO = DU_HMI*HMI_DO + DU_F1*F1_DO + DU_F2*F2_DO
C
C
C---- delta = DE
      DE    =  DO / (DU*(1.0 - F1))
      DE_DO = 1.0 / (DU*(1.0 - F1))
      DE_DU = -DE /  DU
      DE_F1 =  DE /     (1.0 - F1)
C
C---- DE( TO DO )
      DE_TO = DE_DU*DU_TO + DE_F1*F1_TO
      DE_DO = DE_DU*DU_DO + DE_F1*F1_DO  +  DE_DO
C
C
C---- R_delta = RD( RHO UE DE MU )
      RD     = RHO*UE*DE/MU
      RD_RHO =     UE*DE/MU
      RD_UE  = RHO   *DE/MU
      RD_DE  = RHO*UE   /MU
      RD_MU  =       -RD/MU
C
C---- RD( RH UE TO DO RR SR SH RE RO )
      RD_RH = RD_RHO*RHO_RH
      RD_UE = RD_RHO*RHO_UE + RD_UE               + RD_MU*MU_UE
      RD_TO =                       + RD_DE*DE_TO
      RD_DO =                       + RD_DE*DE_DO
      RD_RR = RD_RHO*RHO_RR                       + RD_MU*MU_RR
      RD_SR = RD_RHO*RHO_SR                       + RD_MU*MU_SR
      RD_SH = RD_RHO*RHO_SH                       + RD_MU*MU_SH
      RD_RE =                                     + RD_MU*MU_RE
      RD_RO = RD_RHO*RHO_RO                       + RD_MU*MU_RO
C
C
C---- first guess for Utau, B(k+)
      UT = 0.04*(1.0-DU)
      SGN = SIGN( 1.0 , UT )
C
      RKP = MIN( RK * SGN*UT*RD/DE , 60.0 )
      VB = VBLL - LOG(1.0 + 0.3*RKP)/VKAP
C
      UT = (1.0-DU) / ((LOG(SGN*UT*RD/CC) - 1.0)/VKAP + VB )
C
C---- converge on exact Utau
      DO ITER=1, 10
C
C------ set k+, limiting it to 60  (fully rough)
        THK = TANH(RK * SGN*UT*RD/DE / 60.0 )
        RKP    = 60.0 * THK
        RKP_UT = (1.0 - THK**2) * RK * SGN *RD/DE
C
C------ set log-profile intercept "constant"  VB( UT RD DE )
        VB    = VBLL     - LOG(1.0 + 0.3*RKP)/VKAP
        VB_UT = -0.3*RKP_UT / (1.0 + 0.3*RKP)/VKAP
C
C------ set d+
        DP    = SGN*UT*RD
        DP_UT = SGN   *RD
C
C------ set u+
        UP    = SGN*(1.0-DU)/UT
        UP_UT = -UP/UT
C
C------ set exp(k(u+ - B))
        EUB    = EXP(VKAP*(UP    - VB   ))
        EUB_UT = EUB*VKAP*(UP_UT - VB_UT)
C
        EB    = EXP(-VKAP*VB   )
        EB_UT = EB*(-VKAP*VB_UT)
C
C------ set ys+
        YP    = UP    + EUB
     &   - (1.0 + VKAP*UP + (VKAP*UP)**2/2.0 + (VKAP*UP)**3/6.0)*EB
        YP_UT = UP_UT + EUB_UT
     &   - (1.0 + VKAP*UP + (VKAP*UP)**2/2.0)*EB*VKAP*UP_UT
     &   - (1.0 + VKAP*UP + (VKAP*UP)**2/2.0 + (VKAP*UP)**3/6.0)*EB_UT
C
        REZ  = DP   *(1.0+(CC-1.0)**3) - 3.0*CC*YP
        Z_UT = DP_UT*(1.0+(CC-1.0)**3) - 3.0*CC*YP_UT
C
        DUT = -REZ/Z_UT
        UT = UT + DUT
C
        IF(ABS(DUT) .LT. UEPS) GO TO 5
      ENDDO
      WRITE(*,*) 'UWALL: Utau convergence failed.  dUtau =', DUT
C
 5    CONTINUE
C
C
C---- get UT(RD DU) sensitivities by holding dRes(UT RD DE DU) = 0
      UT_RD = -Z_RD/Z_UT
      UT_DE = -Z_DE/Z_UT
      UT_DU = -Z_DU/Z_UT
C
C---- set final UT(DO TO RD) sensitivities
      UT_TO = UT_DU*DU_TO + Z_DE*DE_TO
      UT_DO = UT_DU*DU_DO + Z_DE*DE_DO
ccc   UT_RD = UT_RD
C
C
C
C---- calculate location of max f'(eta) value
ccc   FDD ~ B1*B*(B+1.0) + B2*(B+1.0)*(B+2.0)*ET + B3*(B+2.0)*(B+3.0)*ET**2 = 0
      ET = 1.0 - 0.6/BB
      DO ITER=1, 10
        ETB = ET**BB
        ETH = SQRT(ET)
        G  = 1.5 *B1*ETH + ( (2.0+BB)         *B2*ET
     &                      +(3.0+BB)         *B3*ET**2)*ETB
        GE = 0.75*B1/ETH + ( (2.0+BB)*(1.0+BB)*B2
     &                      +(3.0+BB)*(2.0*BB)*B3*ET   )*ETB
C
        DET = -G/GE
C
        ET = ET + DET
        IF(ABS(DET) .LT. EEPS) GO TO 8
C
      ENDDO
      WRITE(*,*) 'UWALL: Gmax convergence failed.  dEta =', DET
C
 8    CONTINUE
C
C
C---- set max f'(eta) value
      ETB = ET**BB
      ETH = SQRT(ET)
      GMAX    =    1.5    *B1   *ETH
     &        + ( (2.0+BB)*B2   *ET
     &          + (3.0+BB)*B3   *ET**2 )*ETB
      GMAX_BB =    1.5    *B1_BB*ETH
     &        + (((2.0+BB)*B2_BB + B2)*ET
     &          +((3.0+BB)*B3_BB + B3)*ET**2)*ETB
     &        + ( (2.0+BB)*B2   *ET
     &          + (3.0+BB)*B3   *ET**2 )*ETB*LOG(ET)
C
C
C---- density profile coefficients r1, r2, r3
      R1    = DHW
      R1_MS = 0.
C
      R2    = TRF*0.5*GM1*MS
      R2_MS = TRF*0.5*GM1
C
      R3    = 0.
      R3_MS = 0.
C
C
C=============================================
C---- Profile setup section
C
      DO 10 K=0, KK
        DO 102 I=1, ITOT
          U_VAR(I,K) = 0.
          R_VAR(I,K) = 0.
          T_VAR(I,K) = 0.
          Q_VAR(I,K) = 0.
 102    CONTINUE
        DO 104 L=1, LTOT
          U_PAR(L,K) = 0.
          R_PAR(L,K) = 0.
          T_PAR(L,K) = 0.
          Q_PAR(L,K) = 0.
 104    CONTINUE
 10   CONTINUE
C
C
C---- set wall values
      K = 0
C
      U(K) = 0.
C
      R(K) = 1.0 + R1 + R2 + R3
      R_MS = R_R1*R1_MS + R_R2*R2_MS + R_R3*R3_MS
      R_VAR(IUE,K) = R_MS*MS_UE 
C
      S(K) = 0.
C
      T(K) = 0.
C
C---- set edge values
      K = KK
C
      U(K) = 1.0
C
      R(K) = 1.0
C
      S(K) = 0.
C
      T(K) = 0.
C
C
C---- set k+, limiting it to 60  (fully rough)
      THK = TANH( RK * SGN*UT*RD/DE / 60.0 )
      RKP    = 60.0 * THK
      RKP_UT =  (1.0 - THK**2) * RK * SGN   *RD/DE
      RKP_RD =  (1.0 - THK**2) * RK * SGN*UT   /DE
      RKP_DE = -(1.0 - THK**2) * RK * SGN*UT*RD/DE**2
C
C---- set log-profile intercept "constant"  VB( UT RD DE )
      VB    = VBLL     - LOG(1.0 + 0.3*RKP)/VKAP
      VB_UT = -0.3*RKP_UT / (1.0 + 0.3*RKP)/VKAP
      VB_RD = -0.3*RKP_RD / (1.0 + 0.3*RKP)/VKAP
      VB_DE = -0.3*RKP_DE / (1.0 + 0.3*RKP)/VKAP
C
C---- set exp(-kB)
      EB    = EXP(-VKAP*VB   )
      EB_UT = EB*(-VKAP*VB_UT)
      EB_RD = EB*(-VKAP*VB_RD)
      EB_DE = EB*(-VKAP*VB_DE)
C
C---- set d+
      DP    = SGN*UT*RD
      DP_UT = SGN   *RD
      DP_RD = SGN*UT
C
C
      ALC = LOG(CC)
C
      DET = 1.0 / FLOAT(KK)
C
      DO 20 K=1, KK-1
C 
C------ eta = y/delta  and  ln(eta)
        ET = DET*FLOAT(K)
        ALE = LOG(ET)
C 
C------ set fw and initialize  u+ = UP(UT RD CC)
        IF(ET .GE. 1.0/CC) THEN
C 
          FW    = 0.
          FW_CC = 0.
C 
          UP    = (LOG(SGN*UT*RD) - ALC - 1.0)/VKAP + VB
C 
        ELSE
C 
          FW    = 1.0 - CC*ET
          FW_CC =     -    ET
C 
          FP    =        TANH(SGN*UT*RD*ET/(FPCON*FW))
          UP    = FP*((LOG(SGN*UT*RD) + ALE - CC*ET)/ VKAP + VB)
        ENDIF
C 
C
        DO ITER=1, 10
C-------- set exp(k(u+ - B))
          EUB    = EXP(VKAP*(UP    - VB   ))
          EUB_UP = EUB*VKAP
C
C-------- set ys+
          UPK = VKAP*UP
          YP    = UP    + EUB
     &     - (1.0 + UPK + UPK**2/2.0 + UPK**3/6.0)*EB
          YP_UP = 1.0   + EUB_UP
     &     - (1.0 + UPK + UPK**2/2.0             )*EB*VKAP
C
C
          REZ  = ET*DP*(1.0+(CC-1.0)**3) - 3.0*CC*YP
          Z_UP =                         - 3.0*CC*YP_UP
C
          DUP = -REZ/Z_UP
          UP = UP + DUP
C
          IF(ABS(DUP) .LT. EEPS) GO TO 205
        ENDDO
        WRITE(*,*) 'UWALL:  u+ convergence failed.  du+ = ', DUP
C
 205    CONTINUE
C
C------ set exp(k(u+ - B))
        EUB    = EXP(VKAP*(UP    - VB   ))
        EUB_UP = EUB*VKAP
        EUB_UT = EUB*VKAP*(      - VB_UT)
        EUB_RD = EUB*VKAP*(      - VB_RD)
        EUB_DE = EUB*VKAP*(      - VB_DE)
C
C------ set ys+
        UPK = VKAP*UP
        YP    = UP + EUB    - (1. + UPK + UPK**2/2. + UPK**3/6.)*EB
        YP_UP = 1. + EUB_UP - (1. + UPK + UPK**2/2.)*EB*VKAP
        YP_UT =      EUB_UT - (1. + UPK + UPK**2/2. + UPK**3/6.)*EB_UT
        YP_RD =      EUB_RD - (1. + UPK + UPK**2/2. + UPK**3/6.)*EB_RD
        YP_DE =      EUB_DE - (1. + UPK + UPK**2/2. + UPK**3/6.)*EB_DE
C
C
CCC     REZ  = ET*DP   *(1.0+(CC-1.0)**3) - 3.0*CC*YP
        Z_UP =                            - 3.0*CC*YP_UP
        Z_UT = ET*DP_UT*(1.0+(CC-1.0)**3) - 3.0*CC*YP_UT
        Z_RD = ET*DP_RD*(1.0+(CC-1.0)**3) - 3.0*CC*YP_RD
        Z_DE =                            - 3.0*CC*YP_DE
C
C------ set sensitivities for UP(UT RD DE)  by holding  dREZ(UP UT RD DE) = 0
        UP_UT = -Z_UT/Z_UP
        UP_RD = -Z_RD/Z_UP
        UP_DE = -Z_DE/Z_UP
C
C
C 
        ETB = ET**BB
        ETH = SQRT(ET)
C 
C------ wake function f(eta;b) = F(BB)
        F    = B1   *ETH**3 + (B2   *ET**2 + B3   *ET**3)*ETB
        F_BB = B1_BB*ETH**3 + (B2_BB*ET**2 + B3_BB*ET**3)*ETB
     &                      + (B2   *ET**2 + B3   *ET**3)*ETB*ALE
C 
C------ derivative of wake function f'(eta;b) = G(BB)
        G    = 1.5 *B1   *ETH + ( (2.0+BB)*B2         *ET
     &                          + (3.0+BB)*B3         *ET**2)*ETB
        G_BB = 1.5 *B1_BB*ETH + (((2.0+BB)*B2_BB + B2)*ET
     &                          +((3.0+BB)*B3_BB + B3)*ET**2)*ETB
     &                        + ( (2.0+BB)*B2         *ET
     &                          + (3.0+BB)*B3         *ET**2)*ETB*ALE
C
C
C
C---- BB( TO DO )
C---- DU( TO DO )
C---- DE( TO DO )
C---- RD( TO DO RH UE RR SR SH RE RO )
C---- UT( TO DO                          ; RD )
C---- MS(          UE RR SH       RO )
C
C
C------ velocity profile  u/ui = U
        U(K) = UT*UP         + (1.0-DU)*F
        U_UT = UT*UP_UT + UP
C     
        U_RD = UT*UP_RD + U_UT*UT_RD
        U_DE = UT*UP_DE
        U_CC = UT*UP_CC
        U_DU =                     -    F
        U_BB =                 (1.0-DU)*F_BB
C     
        U_VAR(ITO,K) = U_UT*UT_TO + U_RD*RD_TO + U_DU*DU_TO + U_BB*BB_TO
        U_VAR(IDO,K) = U_UT*UT_DO + U_RD*RD_DO + U_DU*DU_DO + U_BB*BB_DO
        U_VAR(IUE,K) =              U_RD*RD_UE
        U_VAR(IRH,K) =              U_RD*RD_RH
C
C------ reciprocal density profile  rhoi/rho = R
        R(K) = 1.0 + R1*(1.0-U(K)) + R2*(1.0-U(K)**2) + R3*(1.0-U(K)**3)
        R_R1 =           1.0-U(K)
        R_R2 =                           1.0-U(K)**2
        R_R3 =                                              1.0-U(K)**3
        R_U  =     - R1            - R2* 2.0*U(K)     - R3* 3.0*U(K)**2
C
        R_MS = R_R1*R1_MS + R_R2*R2_MS + R_R3*R3_MS
C     
        R_VAR(ITO,K) = R_U*U_VAR(ITO)
        R_VAR(IDO,K) = R_U*U_VAR(IDO) 
        R_VAR(IUE,K) = R_U*U_VAR(IUE) + R_MS*MS_UE 
        R_VAR(IRH,K) = R_U*U_VAR(IRH)
C
C------ turbulent KE profile  q'/u* = Q
        Q(K) =  G/GMAX
        Q_BB = (G_BB - Q(K)*GMAX_BB)/GMAX
C     
        Q_VAR(ITO,K) =                                        Q_BB*BB_TO
        Q_VAR(IDO,K) =                                        Q_BB*BB_DO
C     
C------ outer-layer strain rate / R  =  dU/deta  =  STR(CV DE UT RD CC DU BB)
        STR    = (1.0-DU)*G   + 2.0*U(K)*CV*DE*R(K)
        STR_U  =                2.0     *CV*DE*R(K)
        STR_R  =                2.0*U(K)*CV*DE
C     
        STR_CV =                2.0*U(K)   *DE*R(K)
        STR_DE =                2.0*U(K)*CV   *R(K)
        STR_UT =                 STR_U*U_UT + STR_R*R_UT
        STR_RD =                 STR_U*U_RD + STR_R*R_RD
        STR_CC =                 STR_U*U_CC + STR_R*R_CC
        STR_DU =     -    G    + STR_U*U_DU + STR_R*R_DU
        STR_BB = (1.0-DU)*G_BB + STR_U*U_BB + STR_R*R_BB
C     
C------ total shear stress  Ctau / R
        T(K) = SGN*(UT*FW)**2 + VA12*(CQ*Q(K))**2 + STR   /RD
        T_Q  =                  VA12*(CQ*Q(K))*2.0*CQ
C     
        T_UT = SGN*(UT*FW)*2.0*FW           + STR_UT/RD
C     
        T_CQ =                  VA12*(CQ*Q(K))*2.0*Q(K)
        T_CV =                                STR_CV/RD
        T_DE =                                STR_DE/RD
        T_RD = T_UT*UT_RD                   + STR_RD/RD - STR/RD**2
        T_CC = SGN*(UT*FW)*2.0*UT*FW_CC     + STR_CC/RD
        T_DU =                                STR_DU/RD
        T_BB =                     T_Q*Q_BB + STR_BB/RD
C     
        T_VAR(ICQ,K) = T_CQ
        T_VAR(ICV,K) = T_CV
C
        T_VAR(ITO,K) = T_UT*UT_TO + T_RD*RD_TO + T_DU*DU_TO + T_BB*BB_TO
     &               + T_DE*DE_TO
        T_VAR(IDO,K) = T_UT*UT_DO + T_RD*RD_DO + T_DU*DU_DO + T_BB*BB_DO
     &               + T_DE*DE_DO
        T_VAR(IUE,K) =              T_RD*RD_UE
        T_VAR(IRH,K) =              T_RD*RD_RH
C
 20   CONTINUE
C
C=============================================
C---- profile integration section
C
      U1INT = 0.
      U2INT = 0.
      U3INT = 0.
      M1INT = 0.
      M2INT = 0.
      M3INT = 0.
      DIINT = 0.
      TUINT = 0.
C   
      DO 30 I=1, ITOT
        U1_VAR(I) = 0. 
        U2_VAR(I) = 0. 
        U3_VAR(I) = 0. 
        M1_VAR(I) = 0. 
        M2_VAR(I) = 0. 
        M3_VAR(I) = 0. 
        DI_VAR(I) = 0. 
        TU_VAR(I) = 0.
 30   CONTINUE
C                                            
C
      DO 40 K=1, KK
        ET   =  DET*(FLOAT(K) - 0.5)
C
        H = 1.0 - CV*ET*DE
C
        DELU =  U(K) - U(K-1)
        UA   = (U(K) + U(K-1))*0.5
        RA   = (R(K) + R(K-1))*0.5
        TA   = (T(K) + T(K-1))*0.5
C
        U1INT = U1INT + DET*(1. - UA      )
        U2INT = U2INT + DET*(UA - UA*UA   )
        U3INT = U3INT + DET*(UA - UA*UA*UA)
C
        M1INT = M1INT + DET*(1. - UA      )*ET
        M2INT = M2INT + DET*(UA - UA*UA   )*ET
        M3INT = M3INT + DET*(UA - UA*UA*UA)*ET
C
        DUINT = DUINT + DUA * TA
        TUINT = TUINT + DET * TA*UA*RA
C
        DO 402 I=1, ITOT
C
          DUA_VAR =  U_VAR(I,K) - U_VAR(I,K-1)
          UA_VAR  = (U_VAR(I,K) + U_VAR(I,K-1))*0.5
          RA_VAR  = (R_VAR(I,K) + R_VAR(I,K-1))*0.5
          TA_VAR  = (T_VAR(I,K) + T_VAR(I,K-1))*0.5
C
          U1_VAR(I) = U1_VAR(I) - DET                 *UA_VAR
          U2_VAR(I) = U2_VAR(I) + DET*(1. - 2.0*UA   )*UA_VAR
          U3_VAR(I) = U3_VAR(I) + DET*(1. - 3.0*UA*UA)*UA_VAR
C
          M1_VAR(I) = M1_VAR(I) - DET                 *ET*UA_VAR
          M2_VAR(I) = M2_VAR(I) + DET*(1. - 2.0*UA   )*ET*UA_VAR
          M3_VAR(I) = M3_VAR(I) + DET*(1. - 3.0*UA*UA)*ET*UA_VAR
C                               
          CD_VAR(I) = CD_VAR(I) + DUA_VAR*TA
     &                          + DUA    *TA_VAR
C
          TU_VAR(I) = TU_VAR(I) + DET*TA_VAR*UA    *RA
     &                          + DET*TA    *UA_VAR*RA
     &                          + DET*TA    *UA    *RA_VAR
 402    CONTINUE
 40   CONTINUE
C   
C
      CDINT = CDINT  + 2.0*CV*DE*DET*TUINT
      DO 45 I=1, ITOT
        CD_VAR(I) = CD_VAR(I)  +  2.0*CV*DE*DET*TU_VAR(I)
 45   CONTINUE
C
      CD_VAR(ICV) = CD_VAR(ICV) + 2.0   *DE   *DET*TUINT
      CD_VAR(ITO) = CD_VAR(ITO) + 2.0*CV*DE_TO*DET*TUINT
      CD_VAR(IDO) = CD_VAR(IDO) + 2.0*CV*DE_DO*DET*TUINT
C
C
      RETURN
      END
