C
      SUBROUTINE PANGEN(NDIM,XB,XBP,YB,YBP,SB,NB,NBEL,NPAN,CVPAR,CTERAT)
C
C---------------------------------------------------------------
C     Set paneling distribution from input airfoil geometry.
C     The results are kept in the COMMON block in PANEL.INC
C     for the other panel routines.
C 
C     Input:
C
C      NDIM      first dimension of XB,YB,SB
C      XB,YB(..) input airfoil coordinates (2nd index goes over elements)
C      XBP,YBP   derivative arrays for spline evaluation
C      SB(..)    arc length for XB,YB
C      NB(.)     number of points in XB,YB
C      NBEL      number of elements
C
C      NPAN(.)   number of panel nodes to be generated for each element
C      CVPAR(.)  sets how much surface curvature attracts nodes
C      CTERAT(.) ratio of TE to LE panel spacing (approximately)
C
C
C     Output:
C
C      NDIM      first dimension of XA,YA,SA
C      XA,YA(..) panel node coordinates  (2nd index goes over elements)
C      SA(..)    arc length for XA,YA
C---------------------------------------------------------------
      INCLUDE 'PANEL.INC'
C
      DIMENSION XB(NDIM,NBEL), XBP(NDIM,NBEL), 
     &          YB(NDIM,NBEL), YBP(NDIM,NBEL), SB(NDIM,NBEL)
      DIMENSION NB(NBEL), NPAN(NBEL)
      DIMENSION CVPAR(NBEL), CTERAT(NBEL)
C
C---- Number of temporary nodes for panel distribution calculation
C       exceeds the specified panel number by factor of IPFAC.
      PARAMETER (IPFACX=3)
C
C---- allocate work arrays for the paneling algorithm
      PARAMETER (IWX=IPFACX*IQX)
      DIMENSION W1(IWX),W2(IWX),W3(IWX),W4(IWX),W5(IWX),W6(IWX)
      DIMENSION SNEW(IWX)
C
      IF(NBEL.GT.IEX)
     &  STOP 'PANGEN: Too many elements. Increase IEX in PANEL.INC'
C
      NEL = NBEL
C
      DO 9000 IEL=1, NEL
C
      IF(NB(IEL).GT.IWX) 
     & STOP 'PANGEN: Too many airfoil points. Increase IQX in PANEL.INC'
C
      DO 1 I=1, IQX
        LINVRS(I,IEL) = .FALSE.
    1 CONTINUE
C
C---- locate LE point arc length value
      CALL LEFIND(SBLE,XB(1,IEL),XBP(1,IEL),
     &                 YB(1,IEL),YBP(1,IEL),SB(1,IEL),NB(IEL))
C
C---- set element leading and trailing edge coordinates
      XLE = SEVAL(SBLE,XB(1,IEL),XBP(1,IEL),SB(1,IEL),NB(IEL))
      YLE = SEVAL(SBLE,YB(1,IEL),YBP(1,IEL),SB(1,IEL),NB(IEL))
      XTE = 0.5*(XB(1,IEL)+XB(NB(IEL),IEL))
      YTE = 0.5*(YB(1,IEL)+YB(NB(IEL),IEL))
C
C---- set element chord
      CHORD(IEL) = SQRT( (XTE-XLE)**2 + (YTE-YLE)**2 )
C
C---- set number of panels
      NA(IEL) = NPAN(IEL)
      IF(NA(IEL) .GE. IQX)
     &  STOP 'PANGEN: Too many panels. Increase IQX in PANEL.INC'
C
C---- remove one slot for each corner point added later
      DO 2 IB=1, NB(IEL)-1
        IF(SB(IB,IEL) .EQ. SB(IB+1,IEL)) NA(IEL) = NA(IEL)-1
    2 CONTINUE
C
C---- set up non-dimensionalized curvature array
      DO 3 I=1, NB(IEL)
        W5(I) = ABS( 
     &    CURV(SB(I,IEL),
     &         XB(1,IEL),XBP(1,IEL),
     &         YB(1,IEL),YBP(1,IEL),SB(1,IEL),NB(IEL)) )*CHORD(IEL)
    3 CONTINUE
C
C---- nondimensionalized curvature at LE point
      CVLE = ABS( CURV(SBLE,
     &                 XB(1,IEL),XBP(1,IEL),
     &                 YB(1,IEL),YBP(1,IEL),
     &                 SB(1,IEL),NB(IEL)) )*CHORD(IEL)
C
C
C---- set curvature attraction coefficient actually used
      CC = 5.0 * CVPAR(IEL)
C
C---- set artificial curvature at TE to bunch panels there
      CVTE = CVLE * CTERAT(IEL)
      W5(1)       = CVTE
      W5(NB(IEL)) = CVTE
C
C
C**** smooth curvature array for smoother panel size distribution  ****
C
      SMOOL = 0.015
CCC      CALL ASK('Enter curvature smoothing length/c\',3,SMOOL)
      SMOOSQ = (SMOOL*CHORD(IEL))**2
C
C---- set up tri-diagonal system for smoothed curvatures
      W2(1) = 1.0
      W3(1) = 0.0
      DO 4 I=2, NB(IEL)-1
        DSM = SB(I  ,IEL) - SB(I-1,IEL)
        DSP = SB(I+1,IEL) - SB(I  ,IEL)
        DSO = 0.5*(SB(I+1,IEL) - SB(I-1,IEL))
C
        IF(DSM.EQ.0.0 .OR. DSP.EQ.0.0) THEN
         W1(I) = 0.0
         W2(I) = 1.0
         W3(I) = 0.0
        ELSE
         W1(I) =  SMOOSQ * (         - 1.0/DSM) / DSO
         W2(I) =  SMOOSQ * ( 1.0/DSP + 1.0/DSM) / DSO  +  1.0
         W3(I) =  SMOOSQ * (-1.0/DSP          ) / DSO
        ENDIF
    4 CONTINUE
C
      W1(NB(IEL)) = 0.0
      W2(NB(IEL)) = 1.0
C
C---- fix curvature at LE point by modifying equations adjacent to LE
      DO 5 I=2, NB(IEL)-1
        IF(SB(I,IEL).EQ.SBLE) THEN
C------- if node falls right on LE point, fix curvature there
         W1(I) = 0.
         W2(I) = 1.0
         W3(I) = 0.
         W5(I) = CVLE
         GO TO 51
        ELSE IF(SB(I-1,IEL).LT.SBLE .AND. SB(I,IEL).GT.SBLE) THEN
C------- modify equation at node just before LE point
         DSM = SB(I-1,IEL) - SB(I-2,IEL)
         DSP = SBLE        - SB(I-1,IEL)
         DSO = 0.5*(SBLE - SB(I-2,IEL))
C
         W1(I-1) =  SMOOSQ * (         - 1.0/DSM) / DSO
         W2(I-1) =  SMOOSQ * ( 1.0/DSP + 1.0/DSM) / DSO  +  1.0
         W3(I-1) =  0.
         W5(I-1) = W5(I-1) + SMOOSQ*CVLE/(DSP*DSO)
C
C------- modify equation at node just after LE point
         DSM = SB(I  ,IEL) - SBLE
         DSP = SB(I+1,IEL) - SB(I,IEL)
         DSO = 0.5*(SB(I+1,IEL) - SBLE)
         W1(I) =  0.
         W2(I) =  SMOOSQ * ( 1.0/DSP + 1.0/DSM) / DSO  +  1.0
         W3(I) =  SMOOSQ * (-1.0/DSP          ) / DSO
         W5(I) = W5(I) + SMOOSQ*CVLE/(DSM*DSO)
C
         GO TO 51
        ENDIF
    5 CONTINUE
   51 CONTINUE
C
C---- solve for smoothed curvature array W5
      CALL TRISOL(W2,W1,W3,W5,NB(IEL))
C
C---- find max curvature
      CVMAX = 0.
      DO 61 I=1, NB(IEL)
        CVMAX = MAX( CVMAX , ABS(W5(I)) )
   61 CONTINUE
C
C---- normalize curvature array
      DO 62 I=1, NB(IEL)
        W5(I) = W5(I) / CVMAX
   62 CONTINUE
C
C---- spline curvature array
      CALL SEGSPL(W5,W6,SB(1,IEL),NB(IEL))
C
C---- Set initial guess for node positions uniform in s.
C     More nodes than specified (by factor of IPFAC) are 
C     temporarily used  for more reliable convergence.
      DO 7 K=1, IPFACX
        IPFAC = IPFACX - K + 1
        NN = IPFAC*(NA(IEL)-1)+1
        IF(NN .LE. IWX) GO TO 71
 7    CONTINUE
      STOP 'PANGEN: Work array overflow.  Increase IQX.'
 71   CONTINUE
C
      DSAVG = SB(NB(IEL),IEL)/FLOAT(NN-1)
      DO 8 I=1, NN
        SNEW(I) = DSAVG * FLOAT(I-1)
    8 CONTINUE
C
C---- Newton iteration loop for new node positions
      DO 10 ITER=1, 20
C
C------ set up tri-diagonal system for node position deltas
        CV1  = SEVAL(SNEW(1),W5,W6,SB(1,IEL),NB(IEL))
        CV2  = SEVAL(SNEW(2),W5,W6,SB(1,IEL),NB(IEL))
        CVS1 = DEVAL(SNEW(1),W5,W6,SB(1,IEL),NB(IEL))
        CVS2 = DEVAL(SNEW(2),W5,W6,SB(1,IEL),NB(IEL))
        CAVM = SQRT(CV1**2 + CV2**2)
        DO 110 I=2, NN-1
          DSM = SNEW(I) - SNEW(I-1)
          DSP = SNEW(I) - SNEW(I+1)
          CV3  = SEVAL(SNEW(I+1),W5,W6,SB(1,IEL),NB(IEL))
          CVS3 = DEVAL(SNEW(I+1),W5,W6,SB(1,IEL),NB(IEL))
          CAVP = SQRT(CV3**2 + CV2**2)
          FM = CC*CAVM + 1.0
          FP = CC*CAVP + 1.0
C
C-------- lower, main, and upper diagonals
          W1(I) =      -FM  +  CC*            DSM/CAVM  * CV1*CVS1
          W2(I) =  FP + FM  +  CC*(DSP/CAVP + DSM/CAVM) * CV2*CVS2
          W3(I) = -FP       +  CC* DSP/CAVP             * CV3*CVS3
C
C-------- righthand side, requiring that
C         (1 + C*curv)*deltaS is equal on both sides of node i
          W4(I) = -DSP*FP - DSM*FM
C
          CV1 = CV2
          CV2 = CV3
          CVS1 = CVS2
          CVS2 = CVS3
          CAVM = CAVP
  110   CONTINUE
C
C------ fix endpoints (at TE)
        W2(1) = 1.0
        W3(1) = 0.0
        W4(1) = 0.0
        W1(NN) = 0.0
        W2(NN) = 1.0
        W4(NN) = 0.0
C
C------ solve for changes W4 in node position arc length values
        CALL TRISOL(W2,W1,W3,W4,NN)
C
C------ find under-relaxation factor so as to keep nodes from changing order
        RLX = 1.0
        DMAX = 0.0
        DO 12 I=1, NN-1
          DS  = SNEW(I+1) - SNEW(I)
          DDS = W4(I+1) - W4(I)
          DSRAT = 1.0 + RLX*DDS/DS
          IF(DSRAT.GT.4.0) RLX = (4.0-1.0)*DS/DDS
          IF(DSRAT.LT.0.2) RLX = (0.2-1.0)*DS/DDS
          DMAX = MAX(ABS(W4(I)),DMAX)
   12   CONTINUE
C
C------ update node position
        DO 13 I=2, NN-1
          SNEW(I) = SNEW(I) + RLX*W4(I)
   13   CONTINUE
C
CCC        IF(RLX.EQ.1.0) WRITE(*,*) DMAX
CCC        IF(RLX.NE.1.0) WRITE(*,*) DMAX,'    RLX =',RLX
        IF(ABS(DMAX).LT.1.E-3) GO TO 19
   10 CONTINUE
      WRITE(*,*) 'Paneling convergence failed.  Continuing anyway...'
C
   19 CONTINUE
C
C---- set new panel node coordinates
      DO 20 I=1, NA(IEL)
        IND = IPFAC*(I-1) + 1
        SA(I,IEL)  = SNEW(IND)
        XA(I,IEL)  = SEVAL(SNEW(IND),XB(1,IEL),XBP(1,IEL),
     &                    SB(1,IEL),NB(IEL))
        YA(I,IEL)  = SEVAL(SNEW(IND),YB(1,IEL),YBP(1,IEL),
     &                    SB(1,IEL),NB(IEL))
        XAP(I,IEL) = DEVAL(SNEW(IND),XB(1,IEL),XBP(1,IEL),
     &                    SB(1,IEL),NB(IEL))
        YAP(I,IEL) = DEVAL(SNEW(IND),YB(1,IEL),YBP(1,IEL),
     &                    SB(1,IEL),NB(IEL))
   20 CONTINUE
C
C---- set arc length array for panel node coordinates
ccc%%%   CALL SCALC(XA(1,IEL),YA(1,IEL),SA(1,IEL),NA(IEL))
C
C---- go over buffer airfoil, checking for corners (double points)
      NCORN = 0
      DO 25 IB=1, NB(IEL)-1
        IF(SB(IB,IEL) .EQ. SB(IB+1,IEL)) THEN
C------- found one ! ... Find panel which contains corner
         NCORN = NCORN+1
         XBCORN = XB(IB,IEL)
         YBCORN = YB(IB,IEL)
         SBCORN = SB(IB,IEL)
         XPCORN = XBP(IB,IEL)
         YPCORN = YBP(IB,IEL)
         DO 252 I=1, NA(IEL)
C
C--------- keep stepping until first node past corner
           IF(SA(I,IEL) .LE. SBCORN) GO TO 252
C
C---------- move remainder of panel nodes to make room for additional node
            DO 2522 J=NA(IEL), I, -1
              XA(J+1,IEL)  = XA(J,IEL)
              YA(J+1,IEL)  = YA(J,IEL)
              SA(J+1,IEL)  = SA(J,IEL)
              XAP(J+1,IEL) = XAP(J,IEL)
              YAP(J+1,IEL) = YAP(J,IEL)
              LINVRS(J+1,IEL) = LINVRS(J,IEL)
 2522       CONTINUE
            NA(IEL) = NA(IEL)+1
            IF(NA(IEL) .GT. IQX-1)
     &       STOP 'PANGEN: Too many panels. Increase IQX in PANEL.INC'
            XA(I,IEL) = XBCORN
            YA(I,IEL) = YBCORN
            SA(I,IEL) = SBCORN
            XAP(I,IEL) = XPCORN
            YAP(I,IEL) = YPCORN
C
C---------- shift nodes adjacent to corner to keep panel sizes comparable
            IF(I-2 .GE. 1) THEN
             SA(I-1,IEL)  = 0.5*(SA(I,IEL) + SA(I-2,IEL))
             XA(I-1,IEL)  = SEVAL(SA(I-1,IEL),XB(1,IEL),XBP(1,IEL),
     &                            SB(1,IEL),NB(IEL))
             YA(I-1,IEL)  = SEVAL(SA(I-1,IEL),YB(1,IEL),YBP(1,IEL),
     &                            SB(1,IEL),NB(IEL))
             XAP(I-1,IEL) = DEVAL(SA(I-1,IEL),XB(1,IEL),XBP(1,IEL),
     &                            SB(1,IEL),NB(IEL))
             YAP(I-1,IEL) = DEVAL(SA(I-1,IEL),YB(1,IEL),YBP(1,IEL),
     &                            SB(1,IEL),NB(IEL))
            ENDIF
C
            IF(I+2 .LE. NA(IEL)) THEN
             SA(I+1,IEL)  = 0.5*(SA(I,IEL) + SA(I+2,IEL))
             XA(I+1,IEL)  = SEVAL(SA(I+1,IEL),XB(1,IEL),XBP(1,IEL),
     &                            SB(1,IEL),NB(IEL))
             YA(I+1,IEL)  = SEVAL(SA(I+1,IEL),YB(1,IEL),YBP(1,IEL),
     &                            SB(1,IEL),NB(IEL))
             XAP(I+1,IEL) = DEVAL(SA(I+1,IEL),XB(1,IEL),XBP(1,IEL),
     &                            SB(1,IEL),NB(IEL))
             YAP(I+1,IEL) = DEVAL(SA(I+1,IEL),YB(1,IEL),YBP(1,IEL),
     &                            SB(1,IEL),NB(IEL))
            ENDIF
C
C---------- mark panel nodes from corner to 80% to TE as inverse nodes
            DOTP = XBP(IB,IEL)*(XTE-XLE) + YBP(IB,IEL)*(YTE-YLE)
            IF(DOTP .GT. 0.0) THEN
C----------- bottom surface ...
             DO 2524 J=I+1, NA(IEL)-2
               FRAC = (SA(J,IEL)-SA(I,IEL))/(SA(NA(IEL),IEL)-SA(I,IEL))
               IF(FRAC .LT. 0.80) LINVRS(J,IEL) = .TRUE.
 2524        CONTINUE
            ELSE
C----------- top surface
             DO 2526 J=I-1, 3, -1
               FRAC = (SA(J,IEL)-SA(I,IEL))/(SA(1,IEL)-SA(I,IEL))
               IF(FRAC .LT. 0.80) LINVRS(J,IEL) = .TRUE.
 2526        CONTINUE
            ENDIF
C
C---------- go on to next input geometry point to check for corner
            GO TO 25
C
  252    CONTINUE
        ENDIF
   25 CONTINUE
C
ccc   CALL SCALC (XA(1,IEL), YA(1,IEL),SA(1,IEL),NA(IEL))
ccc   CALL SPLINE(XA(1,IEL),XAP(1,IEL),SA(1,IEL),NA(IEL))
ccc   CALL SPLINE(YA(1,IEL),YAP(1,IEL),SA(1,IEL),NA(IEL))
C
C---- set normal vectors
      DO 30 I=1, NA(IEL)
        NX(I,IEL) =  YAP(I,IEL)
        NY(I,IEL) = -XAP(I,IEL)
   30 CONTINUE
C
C---- calculate panel size ratios (user info)
      DSMIN =  1000.0
      DSMAX = -1000.0
      DO 40 I=1, NA(IEL)-1
        DS = SA(I+1,IEL) - SA(I,IEL)
        DSMIN = MIN(DSMIN,DS)
        DSMAX = MAX(DSMAX,DS)
   40 CONTINUE
C
      DSMIN = DSMIN*FLOAT(NA(IEL)-1)/SA(NA(IEL),IEL)
      DSMAX = DSMAX*FLOAT(NA(IEL)-1)/SA(NA(IEL),IEL)
CCC      WRITE(*,*) 'DSmin/DSavg = ',DSMIN,'     DSmax/DSavg = ',DSMAX
C
 9000 CONTINUE ! with next element
C
C---- set up system pointers
      IQ0(1) = 0
      DO 50 IEL=1, NEL-1
        IQ0(IEL+1) = IQ0(IEL) + NA(IEL)
   50 CONTINUE
C
      NTOT = 0
      DO 52 IEL=1, NEL
        NTOT = NTOT + NA(IEL)
   52 CONTINUE
C
C---- check if arrays are big enough for panel matrix solution
      NMAT = NTOT + NEL
      IF(NMAT .GT. IAX) THEN
       WRITE(*,*) 'Arrays will overflow'
       WRITE(*,*) 'Increase IAX in PANEL.INC to ', NMAT
      ENDIF
C
C---- determine if TE is blunt or sharp, calculate TE geometry parameters
      CALL TECALC
C
C---- calculate position vector angles for panel routines
      CALL AVAIRF
C
C
C      DO 60 IEL=1, NEL
C        IF(SHARP(IEL)) THEN
C         WRITE(*,1101) IEL
CCCC         WRITE(*,1200) NA(IEL), CVPAR(IEL), CTERAT(IEL), CTRRAT(IEL),
CCCC     &              XSREF1(IEL), XSREF2(IEL), XPREF1(IEL), XPREF2(IEL)
C        ELSE
C         WRITE(*,1102) IEL
CCCC         WRITE(*,1200) NA(IEL), CVPAR(IEL), CTERAT(IEL), CTRRAT(IEL),
CCCC     &              XSREF1(IEL), XSREF2(IEL), XPREF1(IEL), XPREF2(IEL)
C        ENDIF
C   60 CONTINUE
C 1101 FORMAT(' Element',I2,'   Sharp trailing edge')
C 1102 FORMAT(' Element',I2,'   Blunt trailing edge')
CC
C 1200 FORMAT(/' Paneling parameters used:'
C     &       /' Number of panel nodes     ' , I3
C     &       /' Panel bunching parameter  ' , F6.3
C     &       /' TE/LE panel density ratio ' , F6.3
C     &       /' Refined area/LE panel density ratio  ' , F6.3
C     &       /' Suction  side refined area x/c limits' , 2F6.3
C     &       /' Pressure side refined area x/c limits' , 2F6.3 /)
CC
      RETURN
      END ! PANGEN


      SUBROUTINE TECALC
C--------------------------------------------
C     Calculates total and projected TE gap 
C     areas and TE panel strengths.
C--------------------------------------------
      INCLUDE 'PANEL.INC'
C
      DO 900 IEL=1, NEL
C
      DXTE(IEL) = XA(1,IEL) - XA(NA(IEL),IEL)
      DYTE(IEL) = YA(1,IEL) - YA(NA(IEL),IEL)
      DXS = 0.5*(-XAP(1,IEL) + XAP(NA(IEL),IEL))
      DYS = 0.5*(-YAP(1,IEL) + YAP(NA(IEL),IEL))
C
C---- normal and streamwise projected TE gap areas
      ANTE(IEL) = DXS*DYTE(IEL) - DYS*DXTE(IEL)
      ASTE(IEL) = DXS*DXTE(IEL) + DYS*DYTE(IEL)
C
C---- total TE gap area
      DSTE(IEL) = SQRT(DXTE(IEL)**2 + DYTE(IEL)**2)
C
      SHARP(IEL) = DSTE(IEL) .LT. 0.0001*CHORD(IEL)
C
      IF(SHARP(IEL)) THEN
       SCS = 1.0
       SDS = 0.0
      ELSE
       SCS = ANTE(IEL)/DSTE(IEL)
       SDS = ASTE(IEL)/DSTE(IEL)
      ENDIF
C
C---- TE panel source and vorticity strengths
CCC      SIGTE(IEL) = 0.5*(GAM(1,IEL) - GAM(NA(IEL),IEL))*SCS
CCC      GAMTE(IEL) = -.5*(GAM(1,IEL) - GAM(NA(IEL),IEL))*SDS
C
  900 CONTINUE
C
      RETURN
      END ! TECALC


 
      SUBROUTINE AVAIRF
      INCLUDE 'PANEL.INC'
C------------------------------------------------
C     Sets angles of vectors from first airfoil 
C     contour points J=1 to all surface points.
C------------------------------------------------
C
C---- go over all J=1 panel points on all elements
      DO 900 JEL=1, NEL
C
      XJ1 = XA(1,JEL)
      YJ1 = YA(1,JEL)
C
C---- go over all field points (actually, airfoil surface nodes)
      DO 100 IEL=1, NEL
C
C------ set angle from J=1 point to first field point
        I = 1
        RXN = XA(I,IEL) - XJ1
        RYN = YA(I,IEL) - YJ1
C
        IF(IEL.EQ.1) THEN
C
C------- first element --- just use ATAN2
         IF(IEL.EQ.JEL) THEN
          AVECA(I,IEL,JEL) = ATAN2(NX(1,IEL),NY(1,IEL))
         ELSE
          AVECA(I,IEL,JEL) = ATAN2(RXN,RYN)
         ENDIF
C
        ELSE
C
C------- lower elements --- make sure they use the same branch as upper element
         IELM = IEL-1
         ATMP = AVECA(NA(IELM),IELM,JEL)
C
         IF(IEL.EQ.JEL) THEN
          AVECA(I,IEL,JEL) = ATANC(NX(I,IEL),NY(I,IEL),ATMP)
         ELSE
          AVECA(I,IEL,JEL) = ATANC(RXN,RYN,ATMP)
         ENDIF
C
        ENDIF
C
C------ set angles to remaining field points so that a branch cut is not crossed
        IF(SHARP(IEL) .AND. IEL.EQ.JEL) THEN
C
         DO 10 I=2, NA(IEL)-1
           RXN = XA(I,IEL) - XJ1
           RYN = YA(I,IEL) - YJ1
           AVECA(I,IEL,JEL) = ATANC(RXN,RYN,AVECA(I-1,IEL,JEL))
   10    CONTINUE
         I = NA(IEL)
         AVECA(I,IEL,JEL) = ATANC(NX(I,IEL),NY(I,IEL),
     &                            AVECA(I-1,IEL,JEL))
        ELSE
C
         DO 12 I=2, NA(IEL)
           RXN = XA(I,IEL) - XJ1
           RYN = YA(I,IEL) - YJ1
           AVECA(I,IEL,JEL) = ATANC(RXN,RYN,AVECA(I-1,IEL,JEL))
   12    CONTINUE
C
        ENDIF
C
  100 CONTINUE
  900 CONTINUE
C
      RETURN
      END ! AVAIRF



      SUBROUTINE PANSOL(ANG,IFFTYP)
C     ........................................................
C      Calculates two surface vorticity (gamma) distributions
C      for alpha = ANG, ANG+90  degrees.  These can be later 
C      superimposed for specified alpha or CL.
C     ........................................................
      INCLUDE 'PANEL.INC'
      DIMENSION ANGL1(IEX)
C
      AGEN = ANG
      KFFTYP = IFFTYP
C
      QINF = 1.0
      ARAD = AGEN
C
      COSA = COS(ARAD)
      SINA = SIN(ARAD)
C
      WRITE(*,*)
      WRITE(*,*) 'Calculating unit vorticity distributions ...'
C
      DO 6000 IEL=1, NEL
        DO 10 I=1, NA(IEL)
          GAM(I,IEL) = 0.
          GAMU(I,1,IEL) = 0.
          GAMU(I,2,IEL) = 0.
C
          SIG(I,IEL) = 0.
          SIGU(I,1,IEL) = 0.
          SIGU(I,2,IEL) = 0.
   10   CONTINUE
C
        I = NA(IEL)+1
        PSIO(IEL) = 0.
        GAMU(I,1,IEL) = 0.
        GAMU(I,2,IEL) = 0.
 6000 CONTINUE
C
C---- go over alements
      DO 9000 IEL=1, NEL
C
C---- system row indices for 1st and last point (panel node) on this element
      JQ1 = IQ0(IEL) + 1
      JQN = IQ0(IEL) + NA(IEL)
C
C---- Set up matrix system for  Psi = Psio  on airfoil surface.
C-    The unknowns are (dGamma)i  or  (dSigma)i ,  and dPsio.
      DO 20 I=1, NA(IEL)
C
C------ system row index for this control point (panel node)
        IQ = IQ0(IEL) + I
C
C------ set angles from all first points on all elements to this control point
        DO 205 JEL=1, NEL
          ANGL1(JEL) = AVECA(I,IEL,JEL)
  205   CONTINUE
C
C------ calculate Psi and dPsi/dGamma array for current node
        CALL PSIPHI(I,IEL,XA(I,IEL),YA(I,IEL),ANGL1,PSII,PHII,UII,VII)
C
        PSIINF = QINF*(COSA*YA(I,IEL) - SINA*XA(I,IEL))
C
C------ set  Res = Psi( 0) - PsiO
        RES1 =  QINF*YA(I,IEL)
C
C------ set  Res = Psi(90) - PsiO
        RES2 = -QINF*XA(I,IEL)
C
C------ dRes/dGamma  or  dRes/dSigma
        DO 210 JEL=1, NEL
          DO 2105 J=1, NA(JEL)
            JQ = IQ0(JEL) + J
            AIJ(IQ,JQ) = PSI_G(J,JEL)
 2105     CONTINUE
  210   CONTINUE
C
C------ dRes/dPsio
        DO 220 JEL=1, NEL
          JQ = NTOT + JEL
          AIJ(IQ,JQ) = 0.0
  220   CONTINUE
        AIJ(IQ,NTOT+IEL) = -1.0
C
C------ local source contribution to Psi
        IF(LINVRS(I,IEL)) AIJ(IQ,IQ) = AIJ(IQ,IQ) - 1.0
C
        DQ(IQ,1) = -RES1
        DQ(IQ,2) = -RES2
C
C------ save influence of Gamma on Phi
        DO 230 JEL=1, NEL
          DO 2305 J=1, NA(JEL)
            PHAIC(I,IEL,J,JEL) = PHI_G(J,JEL)
 2305     CONTINUE
  230   CONTINUE
   20 CONTINUE
C
C
C---- set Kutta condition
      RES = 0.0
CCC       = GAM(1) + GAM(N)
C
      IQ = NTOT + IEL
C
      DO 30 JEL=1, NEL
        DO 301 J=1, NA(JEL)
          JQ = IQ0(JEL) + J
          AIJ(IQ,JQ) = 0.0
  301   CONTINUE
        JQ = NTOT + JEL
        AIJ(IQ,JQ) = 0.0
   30 CONTINUE
C
      AIJ(IQ,JQ1) = 1.0
      AIJ(IQ,JQN) = 1.0
C
      DQ(IQ,1) = -RES
      DQ(IQ,2) = -RES
C
      IF(SHARP(IEL)) THEN
C----- TE Gamma extrapolation if  1st and Nth  points coincide
       IQ = IQ0(IEL) + NA(IEL)
C
       TM = XA(3,IEL) + XA(NA(IEL)-2,IEL)
       TO = XA(2,IEL) + XA(NA(IEL)-1,IEL)
       TP = XA(1,IEL) + XA(NA(IEL)  ,IEL)
C
       DM = 1.0 / (TP - TO)
       DP = 1.0 / (TO - TM)
C
       DO 40 JEL=1, NEL
         DO 401 J=1, NA(JEL)
           JQ = IQ0(IEL) + J
           AIJ(IQ,JQ) = 0.0
  401    CONTINUE
         JQ = NTOT + JEL
         AIJ(IQ,JQ) = 0.0
   40  CONTINUE
C
       RES = 0.0
C          = ( (GAM(1)-GAM(N  )) - (GAM(2)-GAM(N-1)) ) * DP 
C    &     - ( (GAM(2)-GAM(N-1)) - (GAM(3)-GAM(N-2)) ) * DM
C
       AIJ(IQ,JQ1  ) =   DP
       AIJ(IQ,JQ1+1) = -(DP+DM)
       AIJ(IQ,JQ1+2) =   DM
       AIJ(IQ,JQN  ) = -DP
       AIJ(IQ,JQN-1) =  DP+DM
       AIJ(IQ,JQN-2) = -DM
C
C----- redefine influence coefficients for any inverse points
       DO 42 J = 1, 3
         JQ = IQ0(IEL) + J
         IF(LINVRS(J,IEL)) THEN
          CALL GMILIN(J,IEL,JL,JR,GMI_GL,GMI_GR)
          JQL = IQ0(IEL) + JL
          JQR = IQ0(IEL) + JR
          AIJ(IQ,JQL) = AIJ(IQ,JQL) + AIJ(IQ,JQ)*GMI_GL
          AIJ(IQ,JQR) = AIJ(IQ,JQR) + AIJ(IQ,JQ)*GMI_GR
          AIJ(IQ,JQ) = 0.0
         ENDIF
   42  CONTINUE
C
       DO 43 J = NA(IEL)-2, NA(IEL)
         JQ = IQ0(IEL) + J
         IF(LINVRS(J,IEL)) THEN
          CALL GMILIN(J,IEL,JL,JR,GMI_GL,GMI_GR)
          JQL = IQ0(IEL) + JL
          JQR = IQ0(IEL) + JR
          AIJ(IQ,JQL) = AIJ(IQ,JQL) + AIJ(IQ,JQ)*GMI_GL
          AIJ(IQ,JQR) = AIJ(IQ,JQR) + AIJ(IQ,JQ)*GMI_GR
          AIJ(IQ,JQ) = 0.0
         ENDIF
   43  CONTINUE
C
       DQ(IQ,1) = -RES
       DQ(IQ,2) = -RES
C
      ENDIF
C
 9000 CONTINUE
C
C---- solve system for the two vorticity distributions
      NMAT = NTOT + NEL
      CALL GAUSSN(IAX,NMAT,AIJ,DQ,2)
C
C---- set alpha=0,90 surface vorticity for this geometry
      DO 50 IEL=1, NEL
        DO 501 I=1, NA(IEL)
          IQ = IQ0(IEL) + I
          GAMU(I,1,IEL) = DQ(IQ,1)
          GAMU(I,2,IEL) = DQ(IQ,2)
          SIGU(I,1,IEL) = 0.0
          SIGU(I,2,IEL) = 0.0
  501   CONTINUE
C
C------ reset source and vorticity at any inverse points
        DO 502 I=1, NA(IEL)
          IF(LINVRS(I,IEL)) THEN
           CALL GMILIN(I,IEL,JL,JR,GMI_GL,GMI_GR)
           SIGU(I,1,IEL) = GAMU(I,1,IEL)
           SIGU(I,2,IEL) = GAMU(I,2,IEL)
           GAMU(I,1,IEL) = GMI_GL*GAMU(JL,1,IEL) + GMI_GR*GAMU(JR,1,IEL)
           GAMU(I,2,IEL) = GMI_GL*GAMU(JL,2,IEL) + GMI_GR*GAMU(JR,2,IEL)
          ENDIF
  502   CONTINUE
C
C------ set Psio0 and Psio90
        I = NA(IEL)+1
        IQ = NTOT + IEL
        GAMU(I,1,IEL)  = DQ(IQ,1)
        GAMU(I,2,IEL)  = DQ(IQ,2)
   50 CONTINUE
C
C---- set circulations for alpha=0,90
      DO 60 IEL=1, NEL
        CIRCU(1,IEL) = 0.0
        CIRCU(2,IEL) = 0.0
        DO 601 I=2, NA(IEL)
          DS = SA(I,IEL) - SA(I-1,IEL)
          CIRCU(1,IEL) = CIRCU(1,IEL)
     &                 + DS*0.5*(GAMU(I,1,IEL)+GAMU(I-1,1,IEL))
          CIRCU(2,IEL) = CIRCU(2,IEL)
     &                 + DS*0.5*(GAMU(I,2,IEL)+GAMU(I-1,2,IEL))
  601   CONTINUE
C
        IF(SHARP(IEL)) THEN
         SCS = 1.0
         SDS = 0.0
        ELSE
         SCS = ANTE(IEL)/DSTE(IEL)
         SDS = ASTE(IEL)/DSTE(IEL)
        ENDIF
C
C------ TE panel source and vorticity strengths
        GAMTE1 = -.5*(GAMU(1,1,IEL) - GAMU(NA(IEL),1,IEL))*SDS
        GAMTE2 = -.5*(GAMU(1,2,IEL) - GAMU(NA(IEL),2,IEL))*SDS
C
        CIRCU(1,IEL) = CIRCU(1,IEL) + GAMTE1*DSTE(IEL)
        CIRCU(2,IEL) = CIRCU(2,IEL) + GAMTE2*DSTE(IEL)
C
   60 CONTINUE
C
      RETURN
      END ! PANSOL



      SUBROUTINE GMILIN(I,IEL,JL,JR,GMI_GL,GMI_GR)
      INCLUDE 'PANEL.INC'
C
C---- find "direct" endpoints JL,JR of the inverse segment containing I
      DO 10 JL=I-1, 1, -1
        IF(.NOT. LINVRS(JL,IEL)) GO TO 11
   10 CONTINUE
   11 CONTINUE
C
      DO 20 JR=I+1, NA(IEL)
        IF(.NOT. LINVRS(JR,IEL)) GO TO 21
   20 CONTINUE
   21 CONTINUE
C
C---- GAM(I,IEL) is defined to be linearly interpolated between JL, JR
      GMI_GL = (SA(JR,IEL) - SA(I ,IEL)) / (SA(JR,IEL) - SA(JL,IEL))
      GMI_GR = (SA(I ,IEL) - SA(JL,IEL)) / (SA(JR,IEL) - SA(JL,IEL))
C
cC---- GAM(I,IEL) is interpolated between JL, JR via high-order polynomial
c      FLIN = (SA(I ,IEL) - SA(JL,IEL)) / (SA(JR,IEL) - SA(JL,IEL))
c      FCUB = 3.0*FLIN**2 - 2.0*FLIN**3
cC
c      IF(XA(JR,IEL) .GT. XA(JL,IEL)) THEN
cC----- lower surface
c       GMI_GL = 1.0 - FCUB**6
c       GMI_GR =       FCUB**6
c      ELSE
cC----- upper surface
c       GMI_GL =       (1.0 - FCUB)**6
c       GMI_GR = 1.0 - (1.0 - FCUB)**6
c      ENDIF
C
      RETURN
      END




      SUBROUTINE PSIPHI(I,IE,XI,YI,ANGL1,PSII,PHII,UI,VI)
C....................................................................
C     Calculates current streamfunction Psi and potential Phi at 
C     point XI,YI due to freestream and all bound vorticity Gam 
C     on the airfoil. 
C
C     Input:
C      I      node index of panel node at XI,YI      ( 0 if not a panel node)
C      IE     element index of panel node at XI,YI   ( 0 if not a panel node)
C      XI,YI  point coordinates
C      ANGL1  angles from point XI,YI to J=1 points on all elements
C
C     Output:
C      PSII   streamfunction at XI,YI
C      PHII   potential at XI,YI
C      UI,VI  velocity components at XI,YI
C
C      PSI_G   streamfunction / surface vorticity influence coeff. dPsi/dGam
C      PHI_G   vel. potential / surface vorticity influence coeff. dPhi/dGam
C
C....................................................................
      INCLUDE 'PANEL.INC'
      DIMENSION ANGL1(IEX)
      DIMENSION AVEC(IQX)
      COMPLEX*8 ZI,DZ,DZINV,ZJO,ZJP,DZ1,DZ2,DZLOG1,DZLOG2,
     &          FIS,FIS_ZI,FID,FID_ZI,FI,FI_ZI,COMSUM,COMDIF,
     &          FI_SD,FI_GS,FI_GD,FI_SIG,FI_GAM
C
      DATA PI, QOPI / 3.141592654, 0.07957747151 /
C
      DATA EPS / 0.001 /
C
      IO = I
      IEL = IE
C
      COSA = COS(ARAD)
      SINA = SIN(ARAD)
C
      PSII = 0.
      PHII = 0.
      UI   = 0.
      VI   = 0.
C
      DO 3 JEL=1, NEL
        DO 31 JO=1, NA(JEL)
          PSI_G(JO,JEL) = 0.0
          PHI_G(JO,JEL) = 0.0
   31   CONTINUE
    3 CONTINUE
C
C
      ZI = CMPLX( XI , YI )
C
      DO 900 JEL=1, NEL
C
      IF(SHARP(JEL)) THEN
       SCS = 1.0
       SDS = 0.0
      ELSE
       SCS = ANTE(JEL)/DSTE(JEL)
       SDS = ASTE(JEL)/DSTE(JEL)
      ENDIF
C
C---- set angle AVEC of each vector from (IO,IEL) point to all (JO,JEL) points 
C     (this array is set now to allow the  DO 10  loop to vectorize)
      IF(IEL.EQ.JEL .AND. IO.NE.0) THEN
C
       AVEC(1) = ANGL1(JEL)
       DO 61 JO=2, IO-1
         RX1 = XI - XA(JO,IEL)
         RY1 = YI - YA(JO,IEL)
         AVEC(JO) = ATANC(RX1,RY1,AVEC(JO-1))
   61  CONTINUE
C
       IF(IO.GT.1) AVEC(IO) = ATANC(NX(IO,IEL),NY(IO,IEL),AVEC(IO-1))
C
       IF(SHARP(IEL) .AND. (IO.EQ.1 .OR. IO.EQ.NA(IEL))) THEN
        DO 62 JO=IO+1, NA(IEL)-1
          RX1 = XI - XA(JO,IEL)
          RY1 = YI - YA(JO,IEL)
          AVEC(JO) = ATANC(RX1,RY1,AVEC(JO-1))
   62   CONTINUE
        JO = NA(IEL)
        AVEC(JO) = ATANC(NX(JO,IEL),NY(JO,IEL),AVEC(JO-1))
       ELSE
        DO 63 JO=IO+1, NA(IEL)
          RX1 = XI - XA(JO,IEL)
          RY1 = YI - YA(JO,IEL)
          AVEC(JO) = ATANC(RX1,RY1,AVEC(JO-1))
   63   CONTINUE
       ENDIF
C
      ELSE
C
       AVEC(1) = ANGL1(JEL)

c       if(i.eq.0 .and. ie.eq.0)
c     &       write(*,*) 1, jel, avec(1)*180.0/3.14159

       DO 8 JO=2, NA(JEL)
         RX1 = XI - XA(JO,JEL)
         RY1 = YI - YA(JO,JEL)
         AVEC(JO) = ATANC(RX1,RY1,AVEC(JO-1))

c       if(i.eq.0 .and. ie.eq.0) 
c     &       write(*,*) jo, jel, avec(jo)*180.0/3.14159

    8  CONTINUE
C
      ENDIF
C
C---- set first point angle again for TE panel streamfunction evaluation
      IF(IO .EQ. 0) THEN
       JO = 1
       RX1 = XI - XA(JO,JEL)
       RY1 = YI - YA(JO,JEL)
       JO = NA(JEL)+1
       AVEC(JO) = ATANC(RX1,RY1,AVEC(JO-1))
      ELSE
       JO = NA(JEL)+1
       AVEC(JO) = AVEC(1)
      ENDIF
C
C---- see if first and last angles are the same
      DAVEC = AVEC(NA(JEL)+1) - AVEC(1) 
      IF(ABS(DAVEC) .GT. 0.01) THEN
C----- they are not ... we're inside element JEL
       UI = 0.0
       VI = 0.0
       RETURN
      ENDIF
C
      DO 10 JO=1, NA(JEL)
        JP = JO+1
C
        JM = JO-1
        JQ = JP+1
C
        IF(JO.EQ.1) THEN
         JM = JO
        ELSE IF(JO.EQ.NA(JEL)-1) THEN
         JQ = JP
        ELSE IF(JO.EQ.NA(JEL)  ) THEN
         JP = 1
         SATOT = SA(NA(JEL),JEL) - SA(1,JEL)
         IF((XA(JO,JEL)-XA(JP,JEL))**2 + (YA(JO,JEL)-YA(JP,JEL))**2
     &      .LT. (EPS*SATOT)**2) GO TO 12
        ENDIF
C
        DZ = CMPLX( (XA(JP,JEL)-XA(JO,JEL)) , (YA(JP,JEL)-YA(JO,JEL)) )
        DS = ABS(DZ)
C
        DZINV = 1.0 / DZ
C
        ZJO = CMPLX( XA(JO,JEL) , YA(JO,JEL) )
        ZJP = CMPLX( XA(JP,JEL) , YA(JP,JEL) )
C
        DZ1 = ZJO - ZI
        DZ2 = ZJP - ZI
C
        DZ1SQ =  REAL(DZ1)**2 + IMAG(DZ1)**2
        DZ2SQ =  REAL(DZ2)**2 + IMAG(DZ2)**2
C
        G1 = 0.
        G2 = 0.
        IF(DZ1SQ .NE. 0.0) G1 = 0.5*LOG( DZ1SQ )
        IF(DZ2SQ .NE. 0.0) G2 = 0.5*LOG( DZ2SQ )
C
        T1 = -AVEC(JO)
        T2 = -AVEC(JO+1)
C
        DZLOG1 = CMPLX( G1 , T1 )
        DZLOG2 = CMPLX( G2 , T2 )
C
        IF(JO.EQ.NA(JEL)) GO TO 11
C
C------ calculate vortex panel contribution to complex potential
        FIS    = (DZ2*DZLOG2 - DZ1*DZLOG1)*DZINV  -  1.0
        FIS_ZI = (   -DZLOG2 +     DZLOG1)*DZINV
C
        FID    =  -(DZ1+DZ2)*FIS*DZINV
     &         + (DZLOG2 - 0.5)*(DZ2*DZINV)**2
     &         - (DZLOG1 - 0.5)*(DZ1*DZINV)**2
        FID_ZI = -((DZ1+DZ2)*FIS_ZI + 2.0)*DZINV
C
        SDIF = SIG(JP,JEL) - SIG(JO,JEL)
        GSUM = GAM(JP,JEL) + GAM(JO,JEL)
        GDIF = GAM(JP,JEL) - GAM(JO,JEL)
C
C------ set complex sources
        COMSUM = CMPLX( 2.0*SDIF , GSUM*DS )
        COMDIF = CMPLX( 0.0      , GDIF*DS )
C
        FI    = QOPI*(COMSUM*FIS    + COMDIF*FID   )
        FI_ZI = QOPI*(COMSUM*FIS_ZI + COMDIF*FID_ZI)
C
        FI_SD = QOPI*CMPLX(2.0  ,0.0  )*FIS
        FI_GS = QOPI*CMPLX(0.0  ,DS   )*FIS
        FI_GD = QOPI*CMPLX(0.0  ,DS   )*FID
C
        PSII = PSII + IMAG(FI)
        PHII = PHII + REAL(FI)
C
        PSI_SP =             + IMAG(FI_SD)
        PSI_SO =             - IMAG(FI_SD)
        PSI_GP = IMAG(FI_GS) + IMAG(FI_GD)
        PSI_GO = IMAG(FI_GS) - IMAG(FI_GD)
C
        PHI_SP =             + REAL(FI_SD)
        PHI_SO =             - REAL(FI_SD)
        PHI_GP = REAL(FI_GS) + REAL(FI_GD)
        PHI_GO = REAL(FI_GS) - REAL(FI_GD)
C
C------ increment  dPsi/dGam  and/or  dPsi/dSig  influence coefficients
        IF(LINVRS(JP,JEL)) THEN
         CALL GMILIN(JP,JEL,JL,JR,GP_GL,GP_GR)
C
         PSI_G(JP,JEL) = PSI_G(JP,JEL) + PSI_SP
         PSI_G(JL,JEL) = PSI_G(JL,JEL) + PSI_GP*GP_GL
         PSI_G(JR,JEL) = PSI_G(JR,JEL) + PSI_GP*GP_GR
C
         PHI_G(JP,JEL) = PHI_G(JP,JEL) + PHI_SP
         PHI_G(JL,JEL) = PHI_G(JL,JEL) + PHI_GP*GP_GL
         PHI_G(JR,JEL) = PHI_G(JR,JEL) + PHI_GP*GP_GR
        ELSE
         PSI_G(JP,JEL) = PSI_G(JP,JEL) + PSI_GP
         PHI_G(JP,JEL) = PHI_G(JP,JEL) + PHI_GP
        ENDIF
C
        IF(LINVRS(JO,JEL)) THEN
         CALL GMILIN(JO,JEL,JL,JR,GO_GL,GO_GR)
C
         PSI_G(JO,JEL) = PSI_G(JO,JEL) + PSI_SO
         PSI_G(JL,JEL) = PSI_G(JL,JEL) + PSI_GO*GO_GL
         PSI_G(JR,JEL) = PSI_G(JR,JEL) + PSI_GO*GO_GR
C
         PHI_G(JO,JEL) = PHI_G(JO,JEL) + PHI_SO
         PHI_G(JL,JEL) = PHI_G(JL,JEL) + PHI_GO*GO_GL
         PHI_G(JR,JEL) = PHI_G(JR,JEL) + PHI_GO*GO_GR
        ELSE
         PSI_G(JO,JEL) = PSI_G(JO,JEL) + PSI_GO
         PHI_G(JO,JEL) = PHI_G(JO,JEL) + PHI_GO
        ENDIF
C
        UI   = UI   + REAL(FI_ZI)
        VI   = VI   - IMAG(FI_ZI)
C
   10 CONTINUE ! with next panel
C
   11 CONTINUE
      FIS    = (DZ2*DZLOG2 - DZ1*DZLOG1)*DZINV  -  1.0
      FIS_ZI = (   -DZLOG2 +     DZLOG1)*DZINV
C
      SIGTE(JEL) =  0.5*SCS*(GAM(JP,JEL) - GAM(JO,JEL))
      GAMTE(JEL) = -0.5*SDS*(GAM(JP,JEL) - GAM(JO,JEL))
C
      COMSUM = CMPLX( SIGTE(JEL)*DS , GAMTE(JEL)*DS )
C
      FI    = QOPI*2.0*COMSUM*FIS
      FI_ZI = QOPI*2.0*COMSUM*FIS_ZI
C
      FI_SIG = QOPI*2.0*CMPLX( DS  ,0.0  )*FIS
      FI_GAM = QOPI*2.0*CMPLX(0.0  ,   DS)*FIS
C
C---- TE panel contribution to Psi and Phi
      PSII = PSII + IMAG(FI)
      PHII = PHII + REAL(FI)
C
C---- increment dPsi/dGam influence coefficients
      PSI_G(JP,JEL) = PSI_G(JP,JEL) + 0.5*SCS*IMAG(FI_SIG)
     &                              - 0.5*SDS*IMAG(FI_GAM)
      PSI_G(JO,JEL) = PSI_G(JO,JEL) - 0.5*SCS*IMAG(FI_SIG)
     &                              + 0.5*SDS*IMAG(FI_GAM)
C
      PHI_G(JP,JEL) = PHI_G(JP,JEL) + 0.5*SCS*REAL(FI_SIG)
     &                              - 0.5*SDS*REAL(FI_GAM)
      PHI_G(JO,JEL) = PHI_G(JO,JEL) - 0.5*SCS*REAL(FI_SIG)
     &                              + 0.5*SDS*REAL(FI_GAM)
C
      UI   = UI   + REAL(FI_ZI)
      VI   = VI   - IMAG(FI_ZI)
C
   12 CONTINUE
C
  900 CONTINUE ! with next element
C
C**** Freestream terms
      FI    = QINF*CMPLX(COSA,-SINA)*ZI
      FI_ZI = QINF*CMPLX(COSA,-SINA)
C
      PSII = PSII + IMAG(FI)
      PHII = PHII + REAL(FI)
C
      UI   = UI   + REAL(FI_ZI)
      VI   = VI   - IMAG(FI_ZI)
C
      RETURN
      END ! PSIPHI


      SUBROUTINE ALSET(ANG)
C----------------------------------------------------------
C     Sets surface vorticity distribution for specified
C     freestream flow angle ANG.  The total configuration
C     circulation is returned in CIRTOT.
C----------------------------------------------------------
      INCLUDE 'PANEL.INC'
C
      ARAD = ANG - AGEN
C
      SINA = SIN(ARAD)
      COSA = COS(ARAD)
C
C---- set surface vorticity distributions and add up circulation
      CIRC = 0.0
      DO 20 IEL=1, NEL
C
C------ superimpose  alpha = 0,90  distributions for this alpha
        DO 210 I=1, NA(IEL)
          GAM(I,IEL) = GAMU(I,1,IEL)*COSA + GAMU(I,2,IEL)*SINA
          SIG(I,IEL) = SIGU(I,1,IEL)*COSA + SIGU(I,2,IEL)*SINA
ccc          write(*,*) i, xa(i,iel), ya(i,iel), gam(i,iel), sig(i,iel)
  210   CONTINUE
C
        I = NA(IEL)+1
        PSIO(IEL) = GAMU(I,1,IEL)*COSA + GAMU(I,2,IEL)*SINA
C
        IF(SHARP(IEL)) THEN
         SCS = 1.0
         SDS = 0.0
        ELSE
         SCS = ANTE(IEL)/DSTE(IEL)
         SDS = ASTE(IEL)/DSTE(IEL)
        ENDIF
C
C------ TE panel source and vorticity strengths
        SIGTE(IEL) = 0.5*(GAM(1,IEL) - GAM(NA(IEL),IEL))*SCS
        GAMTE(IEL) = -.5*(GAM(1,IEL) - GAM(NA(IEL),IEL))*SDS
C
C------ element circulation
        CIRCEL(IEL) = CIRCU(1,IEL)*COSA + CIRCU(2,IEL)*SINA
C
        CIRC = CIRC + CIRCEL(IEL)
C
   20 CONTINUE
C
      CIRTOT = CIRC
C
C
C**** set surface potentials from saved influence matrix ...
C
C---- freestream contribution
      DO 30 IEL=1, NEL
        DO 310 I=1, NA(IEL)
          PHIA(I,IEL) = QINF*(COSA*XA(I,IEL) + SINA*YA(I,IEL))
  310   CONTINUE
   30 CONTINUE
C
C---- surface singularity contributions
      DO 60 JEL=1, NEL
        DO 650 J=1, NA(JEL)
C
          IF(LINVRS(J,JEL)) THEN
           GAMSIG = SIG(J,JEL)
          ELSE
           GAMSIG = GAM(J,JEL)
          ENDIF
C
          DO 6510 IEL=1, NEL
            DO 6520 I=1, NA(IEL)
              PHIA(I,IEL) = PHIA(I,IEL) + PHAIC(I,IEL,J,JEL)*GAMSIG
 6520       CONTINUE
 6510     CONTINUE
C
  650   CONTINUE
   60 CONTINUE
C
      RETURN
      END ! ALSET


      SUBROUTINE CLSET(CLSPEC,ANG,MINF)
      INCLUDE 'PANEL.INC'
C
C---- find alpha needed to achieve specified CL
C
      ARAD = AGEN
      ITMAX = 10
C
C---- Newton loop for alpha to get specified inviscid CL
      DO 100 ITAL=1, 20
C
        COSA = COS(ARAD)
        SINA = SIN(ARAD)
        DO 10 IEL=1, NEL
          DO 101 I=1, NA(IEL)
            GAM(I,IEL) = COSA*GAMU(I,1,IEL) + SINA*GAMU(I,2,IEL)
  101     CONTINUE
   10   CONTINUE
C
C------ get corresponding CL and CL_alpha
        CALL CLCALC(CL,CL_ALF,CM,MINF,CLEL,CDEL,CMEL)
C
        DARAD = (CLSPEC - CL) / CL_ALF
        ARAD = ARAD + DARAD
C
        IF(ABS(DARAD).LE.1.0E-6) GO TO 110
  100 CONTINUE
      WRITE(*,*) 'CLSET:  CL convergence failed'
  110 CONTINUE
C
      ANG = ARAD + AGEN
C
      RETURN
      END ! CLSET


      SUBROUTINE CLCALC(CLT,CLT_ALF,CMT,MINF,CLELT,CDELT,CMELT)
C...........................................................
C     Integrates surface pressures to get CL and CM
C     Also calculates dCL/dAlpha for prescribed-CL routines
C...........................................................
      INCLUDE 'PANEL.INC'
      DIMENSION CLELT(IEX),CDELT(IEX),CMELT(IEX)
C
      SINA = SIN(ARAD)
      COSA = COS(ARAD)
C
      BETA = SQRT(1.0 - MINF**2)
      BFAC = 0.5*MINF**2 / (1.0 + BETA)
C
      CL = 0.
      CD = 0.
      CM = 0.
C
      CL_ALF = 0.
C
      DO 9000 IEL=1, NEL
C
      CLEL(IEL) = 0.0
      CDEL(IEL) = 0.0
      CMEL(IEL) = 0.0
C
      I = 1
      CGINC = 1.0 - (GAM(I,IEL)/QINF)**2
      CPG1 = CGINC / (BETA + BFAC*CGINC)
C
      GAM_ALF = -SINA*GAMU(I,1,IEL) + COSA*GAMU(I,2,IEL)
      CPI_GAM = -2.0*GAM(I,IEL)/QINF**2
      CPC_CPI = (1.0 - BFAC*CPG1)/ (BETA + BFAC*CGINC)
      CPG1_A = CPC_CPI*CPI_GAM*GAM_ALF
C
      DO 10 I=1, NA(IEL)
        IP = I+1
        IF(I.EQ.NA(IEL)) IP = 1
C
        CGINC = 1.0 - (GAM(IP,IEL)/QINF)**2
        CPG2 = CGINC / (BETA + BFAC*CGINC)
C
        GAM_ALF = -SINA*GAMU(IP,1,IEL) + COSA*GAMU(IP,2,IEL)
        CPI_GAM = -2.0*GAM(IP,IEL)/QINF**2
        CPC_CPI = (1.0 - BFAC*CPG2)/ (BETA + BFAC*CGINC)
        CPG2_A = CPC_CPI*CPI_GAM*GAM_ALF
C
        DX = (XA(IP,IEL)-XA(I,IEL))*COSA + (YA(IP,IEL)-YA(I,IEL))*SINA
        DY = (YA(IP,IEL)-YA(I,IEL))*COSA - (XA(IP,IEL)-XA(I,IEL))*SINA
        DG = CPG2 - CPG1
C
        AX = (0.5*(XA(IP,IEL)+XA(I,IEL)) - 0.25)*COSA
     &      + 0.5*(YA(IP,IEL)+YA(I,IEL))*SINA
        AY = (0.5*(YA(IP,IEL)+YA(I,IEL))       )*COSA
     &      - 0.5*(XA(IP,IEL)+XA(I,IEL))*SINA
        AG = 0.5*(CPG2 + CPG1)
C
        DX_A = (YA(IP,IEL)-YA(I,IEL))*COSA
     &       - (XA(IP,IEL)-XA(I,IEL))*SINA
        AG_A = 0.5*(CPG2_A + CPG1_A)
C
        CLEL(IEL) = CLEL(IEL) + DX* AG
        CDEL(IEL) = CDEL(IEL) - DY* AG
        CMEL(IEL) = CMEL(IEL) - DX*(AG*AX + DG*DX/12.0)
     &                        - DY*(AG*AY + DG*DY/12.0)
C
        CL_ALF = CL_ALF + DX*AG_A + AG*DX_A
C
        CPG1 = CPG2
        CPG1_A = CPG2_A
   10 CONTINUE
C
      CL = CL + CLEL(IEL)
      CD = CD + CDEL(IEL)
      CM = CM + CMEL(IEL)
C
      CLELT(IEL) = CLEL(IEL)
      CDELT(IEL) = CDEL(IEL)
      CMELT(IEL) = CMEL(IEL)
C
 9000 CONTINUE
C
      CLT = CL
      CMT = CM
      CLT_ALF = CL_ALF
C
      RETURN
      END ! CLCALC


      SUBROUTINE VCENT(XCEN,YCEN,DOUX,DOUY)
      INCLUDE 'PANEL.INC'
C
C**** Calculates vorticity centroid coordinates
C
      V1SUM = 0.0
      V2SUM = 0.0
      X1SUM = 0.0
      X2SUM = 0.0
      Y1SUM = 0.0
      Y2SUM = 0.0
C
      DO 100 IEL=1, NEL
        DO 10 I=2, NA(IEL)
          DS = SA(I,IEL) - SA(I-1,IEL)
          V1AVG = DS * (GAMU(I,1,IEL) + GAMU(I-1,1,IEL) )*0.5
          V2AVG = DS * (GAMU(I,2,IEL) + GAMU(I-1,2,IEL) )*0.5
          XAVG = ( XA(I,IEL) +  XA(I-1,IEL) )*0.5
          YAVG = ( YA(I,IEL) +  YA(I-1,IEL) )*0.5
C
          V1SUM = V1SUM + V1AVG
          V2SUM = V2SUM + V2AVG
          X1SUM = X1SUM + V1AVG*XAVG
          X2SUM = X2SUM + V2AVG*XAVG
          Y1SUM = Y1SUM + V1AVG*YAVG
          Y2SUM = Y2SUM + V2AVG*YAVG
   10   CONTINUE
C
C------ TE panel vorticity
        VTE1 = -.5*(GAMU(1,1,IEL) - GAMU(NA(IEL),1,IEL))*ASTE(IEL)
        VTE2 = -.5*(GAMU(1,2,IEL) - GAMU(NA(IEL),2,IEL))*ASTE(IEL)
C
        XAVG = ( XA(NA(IEL),IEL) +  XA(1,IEL) )*0.5
        YAVG = ( YA(NA(IEL),IEL) +  YA(1,IEL) )*0.5
C
        V1SUM = V1SUM + VTE1
        V2SUM = V2SUM + VTE2
        X1SUM = X1SUM + VTE1*XAVG
        X2SUM = X2SUM + VTE2*XAVG
        Y1SUM = Y1SUM + VTE1*YAVG
        Y2SUM = Y2SUM + VTE2*YAVG
  100 CONTINUE
C
      A11 =  V1SUM
      A12 = -V2SUM
      A21 =  V2SUM
      A22 =  V1SUM
C
      RHS1 = X1SUM - Y2SUM
      RHS2 = X2SUM + Y1SUM
C
      DET   =  A11 *A22  - A12 *A21
      IF(DET.EQ.0.0) RETURN
C
      VCENX = (RHS1*A22  - A12 *RHS2) / DET
      VCENY = (A11 *RHS2 - RHS1*A21 ) / DET
C
      SINA = SIN(ARAD)
      COSA = COS(ARAD)
C
      DOUX =   (Y1SUM - VCENY*V1SUM)*COSA**2
     &       - (X2SUM - VCENX*V2SUM)*SINA**2
     &       + (Y2SUM - VCENY*V2SUM)*SINA*COSA
     &       - (X1SUM - VCENX*V1SUM)*COSA*SINA
      DOUY = - (X1SUM - VCENX*V1SUM)*COSA**2
     &       - (Y2SUM - VCENY*V2SUM)*SINA**2
     &       - (X2SUM - VCENX*V2SUM)*SINA*COSA
     &       - (Y1SUM - VCENY*V1SUM)*COSA*SINA
C
      XCEN = VCENX
      YCEN = VCENY
C
      RETURN
      END ! VCENT



      SUBROUTINE STRGEN(NSX,NST,ISTLE,ISTTE,XST,YST,PHIST,PSIST,
     &                  PHINL,PHOUT,PSBOT,PSTOP,PHIEND)
      INCLUDE 'PANEL.INC'
      DIMENSION NST(2*IEX),ISTLE(2*IEX),ISTTE(2*IEX)
      DIMENSION XST(NSX,2*IEX),YST(NSX,2*IEX),
     &          PHIST(NSX,2*IEX),PSIST(2*IEX),PHIEND(2*IEX)
C
      DIMENSION ANGL1(IEX)
C
      IF(NEL.LT.1) RETURN
C
      PHIINL = PHINL
      PHIOUT = PHOUT
      PSIBOT = PSBOT
      PSITOP = PSTOP
C
      CALL WAKSET
      CALL INLSET
      CALL FFSSET
C
C---- set terminating potential on each streamline
      PHIEND(1) = PHIOUT
      DO 4 IEL=1, NEL
        IS = 2*IEL
        PHIEND(IS)   = PHIEND(IS-1) - CIRCEL(IEL)
        PHIEND(IS+1) = PHIEND(IS)
    4 CONTINUE
      PHIEND(2*NEL+2) = PHIEND(1)
C
C---------
C-    potential PHIA on airfoil surfaces has been set in ALSET
C---------
C
C---- set streamlines for passing back to calling routine
      DO 20 IEL=1, NEL
        IS1 = 2*IEL-1
        IS2 = 2*IEL
        I1 = 0
        I2 = 0
        DO 210 IU=NU(IEL), 1, -1
          I1 = I1 + 1
          I2 = I2 + 1
          XST(I1,IS1) = XU(IU,IEL)
          XST(I2,IS2) = XU(IU,IEL)
          YST(I1,IS1) = YU(IU,IEL)
          YST(I2,IS2) = YU(IU,IEL)
          PHIST(I1,IS1) = PHIU(IU,IEL)
          PHIST(I2,IS2) = PHIU(IU,IEL)
  210   CONTINUE
C
        ISTLE(IS1) = I1
        ISTLE(IS2) = I2
C
C------ double up stagnation point
        I1 = I1 + 1
        I2 = I2 + 1
        XST(I1,IS1) = XST(I1-1,IS1)
        XST(I2,IS2) = XST(I2-1,IS2)
        YST(I1,IS1) = YST(I1-1,IS1)
        YST(I2,IS2) = YST(I2-1,IS2)
        PHIST(I1,IS1) = PHIST(I1-1,IS1)
        PHIST(I2,IS2) = PHIST(I2-1,IS2)
C
C------ find stagnation point index
        DO 220 I=2, NA(IEL)
          IF(GAM(I,IEL) .LT. 0.0  .AND.  GAM(I-1,IEL) .GE. 0.0) THEN
           IST = I
           GO TO 221
          ENDIF
  220   CONTINUE
        STOP 'STRGEN: Stagnation point not found'
  221   CONTINUE
C
C------ set top of element IEL
        DO 230 I=IST-1, 1, -1
          I1 = I1 + 1
          XST(I1,IS1) = XA(I,IEL)
          YST(I1,IS1) = YA(I,IEL)
          PHIST(I1,IS1) = PHIA(I,IEL)
          IF(LINVRS(I,IEL)) CALL OFFPNT(I,IEL,XST(I1,IS1),YST(I1,IS1))
  230   CONTINUE
C
C------ set bottom of element IEL
        DO 240 I=IST, NA(IEL)
          I2 = I2 + 1
          XST(I2,IS2) = XA(I,IEL)
          YST(I2,IS2) = YA(I,IEL)
          PHIST(I2,IS2) = PHIA(I,IEL)
          IF(LINVRS(I,IEL)) CALL OFFPNT(I,IEL,XST(I2,IS2),YST(I2,IS2))
  240   CONTINUE
C
        ISTTE(IS1) = I1
        ISTTE(IS2) = I2
C
C------ double TE point
        I1 = I1 + 1
        I2 = I2 + 1
        XST(I1,IS1) = XST(I1-1,IS1)
        XST(I2,IS2) = XST(I2-1,IS2)
        YST(I1,IS1) = YST(I1-1,IS1)
        YST(I2,IS2) = YST(I2-1,IS2)
        PHIST(I1,IS1) = PHIST(I1-1,IS1)
        PHIST(I2,IS2) = PHIST(I2-1,IS2)
C
C------ set rest of wake
        DO 250 IW=2, NW(IEL)
          I1 = I1 + 1
          I2 = I2 + 1
          XST(I1,IS1) = XW(IW,IEL)
          XST(I2,IS2) = XW(IW,IEL) + DXW(IW,IEL)
          YST(I1,IS1) = YW(IW,IEL)
          YST(I2,IS2) = YW(IW,IEL) + DYW(IW,IEL)
          PHIST(I1,IS1) = PHIW(IW,IEL)
          PHIST(I2,IS2) = PHIW(IW,IEL) - CIRCEL(IEL)
  250   CONTINUE
C
        IF(I1.GT.NSX .OR. 
     &     I2.GT.NSX     ) STOP 'STRGEN: Overflow of passed arrays.'
C
        NST(IS1) = I1
        NST(IS2) = I2
C
        IW = NW(IEL)
        PSIST(IS1) = PSIO(IEL)
        PSIST(IS2) = PSIO(IEL)
C
   20 CONTINUE
C
C---- set top and bottom streamlines
      IS1 = 2*NEL + 1
      IS2 = 2*NEL + 2
C
      I1 = 0
      DO 31 I=1, NF(1)
        I1 = I1+1
        XST(I1,IS1) = XF(I,1)
        YST(I1,IS1) = YF(I,1)
        PHIST(I1,IS1) = PHIF(I,1)
  31  CONTINUE
      PSIST(IS1) = PSIBOT
C
      I2 = 0
      DO 32 I=1, NF(2)
        I2 = I2+1
        XST(I2,IS2) = XF(I,2)
        YST(I2,IS2) = YF(I,2)
        PHIST(I2,IS2) = PHIF(I,2)
  32  CONTINUE
      PSIST(IS2) = PSITOP
C
      NST(IS1) = I1
      NST(IS2) = I2
C
C---- set LE and TE indices so streamlines look like airfoil sides to NORGEN
      ISTLE(IS1) = 1
      ISTTE(IS1) = NST(IS1)
C
      ISTLE(IS2) = 1
      ISTTE(IS2) = NST(IS2)
C
      RETURN
      END ! STRGEN


      SUBROUTINE OFFPNT(I,IEL,XSTR,YSTR)
      INCLUDE 'PANEL.INC'
      DIMENSION ANGL1(IEX)
C
C---- specified Psi Phi at free streamline point
      PSII = PSIO(IEL)
      PHII = PHIA(I,IEL)
C
C---- make sure potential is monotonically increasing
C-    (might not be the case if cove surface doubles back)
      IF(GAM(I,IEL) .LT. 0.0) THEN
       IF(PHII .LT. PHIA(I-1,IEL))
     &  PHII = PHIA(I-1,IEL) + 0.05*(PHIA(I-1,IEL)-PHIA(I-2,IEL))
      ELSE
       IF(PHII .LT. PHIA(I+1,IEL))
     &  PHII = PHIA(I+1,IEL) + 0.05*(PHIA(I+1,IEL)-PHIA(I+2,IEL))
      ENDIF
C
C---- implied Dstar
      DSTR = -SIG(I,IEL)/GAM(I,IEL)
C
      IF(DSTR .LE. 0.0) THEN
C----- don't offset for negative Dstar
       XSTR = XA(I,IEL)
       YSTR = YA(I,IEL)
       RETURN
      ENDIF
C
C---- initial guess for free streamline using Dstar offset
      XSTR = XA(I,IEL) - NX(I,IEL)*SIG(I,IEL)/GAM(I,IEL)
      YSTR = YA(I,IEL) - NY(I,IEL)*SIG(I,IEL)/GAM(I,IEL)
C
C------ Newton iteration to get specified Psi and Phi at current point
        ITMAX = 15
        DO 210 ITER=1, ITMAX
C
C-------- set angles from first point to all J=1 TE points (used by PSIPHI)
          DO 2110 JEL=1, NEL
            RX1 = XSTR - XA(1,JEL)
            RY1 = YSTR - YA(1,JEL)
            ANGL1(JEL) = ATANC(RX1,RY1,AVECA(I,IEL,JEL))
 2110     CONTINUE
C
          CALL PSIPHI(0,0,XSTR,YSTR,ANGL1,
     &                  PSISTR,PHISTR,UWAKE,VWAKE)
C
C-------- if both velocities returned as zero, we've moved inside element IEL
C-        ... subdivide change or just return surface node coordinates
          IF(UWAKE .EQ. 0.0  .AND.  VWAKE .EQ. 0.0) THEN
           IF(ITER.EQ.1 .OR. ITER.EQ.ITMAX) THEN
            XSTR = XA(I,IEL)
            YSTR = YA(I,IEL)
            RETURN
           ELSE
            DX = 0.5*DX
            DY = 0.5*DY
            XSTR = XSTR - DX
            YSTR = YSTR - DY
            GO TO 210
           ENDIF
          ENDIF 
C
C-------- fill 2x2 Jacobian for  Psi(x,y), Phi(x,y)
          Z11 = -VWAKE
          Z12 =  UWAKE
          Z21 =  UWAKE
          Z22 =  VWAKE
C
C-------- residuals
          D1 = PSII - PSISTR
          D2 = PHII - PHISTR
C
C-------- solve for Newton deltas with Cramer's rule
          DET = Z11*Z22 - Z12*Z21
          DX = ( D1*Z22 - Z12*D2 )/DET
          DY = (Z11*D2  - D1 *Z21)/DET
C
          DS = SQRT(DX*DX + DY*DY)
C
C-------- update x and y
          XSTR = XSTR + DX
          YSTR = YSTR + DY
C
C-------- convergence check
          IF(DS .LT. 5.0E-5) GO TO 211
C
  210   CONTINUE
        WRITE(*,*) 'OFFPNT:  Convergence failed.  ds =', DS
  211 CONTINUE
C
      RETURN
      END ! OFFPNT



      SUBROUTINE NORSET(XNOR,YNOR,XSNOR,YSNOR,
     &                  PHINOR,PSINOR,NNOR,IS, IEL)
C.....................................................
C     Sets iso-potential coordinate array for current 
C     surface vorticity distributions.
C.....................................................
      INCLUDE 'PANEL.INC'
      DIMENSION XNOR(*), YNOR(*)
      DIMENSION XSNOR(*), YSNOR(*), PSINOR(*)
C
      DIMENSION ANGL1(IEX)
C
      IF(MOD(IS,2).EQ.1) THEN
C----- top surface
       SGN = -1.0
       IA = 1
      ELSE
C----- bot surface
       SGN = +1.0
       IA = NA(IEL)
      ENDIF
C
C---- set angles from first point to all J=1 TE points (used by PSIPHI)
      DO JEL=1, NEL
        IF((IS+1)/2 .EQ. IEL) THEN
         ANGL1(JEL) = AVECA(IA,IEL,JEL)
C
        ELSE
C
         DSQMIN = (XNOR(1) - XW(1,IEL))**2
     &          + (YNOR(1) - YW(1,IEL))**2 + 1.0
C
         DO IW = 1, NW(IEL)
           DSQ = (XNOR(1) - XW(IW,IEL))**2
     &         + (YNOR(1) - YW(IW,IEL))**2
           IF(DSQ.LT.DSQMIN) THEN
            DSQMIN = DSQ
            ANGL1(JEL) = AVECW(IW,IEL,JEL)
           ENDIF
         ENDDO
C
         DO IU = 1, NU(IEL)
           DSQ = (XNOR(1) - XU(IU,IEL))**2
     &         + (YNOR(1) - YU(IU,IEL))**2
           IF(DSQ.LT.DSQMIN) THEN
            DSQMIN = DSQ
            ANGL1(JEL) = AVECU(IU,IEL,JEL)
           ENDIF
         ENDDO
C
        ENDIF
      ENDDO
C
C
      DO 20 K = 2, NNOR
C
        IF(K.NE.2) THEN
C------- initial guess for this point
         XNOR(K) = XNOR(K-1)
         YNOR(K) = YNOR(K-1)
        ENDIF
C
C------ converge exact point with specified Psi, Phi
C
C------ Newton iteration to get specified Psi and Phi at current point
        DO 210 ITER=1, 6
C
C-------- set angles from first point to all J=1 TE points (used by PSIPHI)
          DO 2110 JEL=1, NEL
            RX1 = XNOR(K) - XA(1,JEL)
            RY1 = YNOR(K) - YA(1,JEL)
            ANGL1(JEL) = ATANC(RX1,RY1,ANGL1(JEL))
 2110     CONTINUE
C
          CALL PSIPHI(0,0,XNOR(K),YNOR(K),ANGL1,
     &                  PSII,PHII,UNOR,VNOR)

C
c          IF(UNOR.EQ.0.0 .AND. VNOR.EQ.0.0) THEN
cC--------- we landed inside another element... try again with halved step
c           DSFAC = DSFAC * 0.5
c           IF(DSFAC .LT. 0.001) STOP 'NORSET: Ran inside element'
c           GO TO 18
c          ENDIF
C
C-------- fill 2x2 Jacobian for  Psi(x,y), Phi(x,y)
          Z11 = -VNOR
          Z12 =  UNOR
          Z21 =  UNOR
          Z22 =  VNOR
C
C-------- residuals
          D1 = PSINOR(K) - PSII
          D2 = PHINOR    - PHII
C
C-------- solve for Newton deltas with Cramer's rule
          DET = Z11*Z22 - Z12*Z21
          DX = ( D1*Z22 - Z12*D2 )/DET
          DY = (Z11*D2  - D1 *Z21)/DET
C
C-------- update x and y
          XNOR(K) = XNOR(K) + DX
          YNOR(K) = YNOR(K) + DY
C
C-------- convergence check
          IF((DX*DX + DY*DY) .LT. 1.0E-8) GO TO 211
C
  210   CONTINUE
        WRITE(*,*) 'NORSET: Convergence failed.', D1, D2

  211   CONTINUE
C
        XSNOR(K) =  SGN*VNOR
        YSNOR(K) = -SGN*UNOR
   20 CONTINUE
C
      RETURN
      END ! NORSET



      SUBROUTINE WAKSET
C.....................................................
C     Sets wake coordinate array for current 
C     surface vorticity distributions.
C.....................................................
      INCLUDE 'PANEL.INC'
      DIMENSION ANGL1(IEX)
C
C---- find foremost and aftmost TE positions among all elements
      XMIN = XA(1,1)
      XMAX = XA(1,1)
      DO 5 IEL=1, NEL
        DO 51 I=1, NA(IEL)
          XMIN = MIN( XMIN , XA(I,IEL) )
          XMAX = MAX( XMAX , XA(I,IEL) )
   51   CONTINUE
    5 CONTINUE
C
C---- set maximum wake length increment within chord limits
      DSMAX = 0.04*(XMAX-XMIN)
C
      STOL = 0.00005*(XMAX-XMIN)
C
C---- set stretching factor for increments beyond chord limits
      SGEOM = 1.2
C
      PHIEND = PHIOUT
C
      DO 1000 IEL=1, NEL
C
      WRITE(*,9010) IEL
C
      IW = 1
C
C---- set first wake point at upper TE point
      XW(IW,IEL) = XA(1,IEL)
      YW(IW,IEL) = YA(1,IEL)
      SW(IW,IEL) = 0.0
C
C---- set offsets to wake trajectory springing from opposite TE point
      DXW(IW,IEL) = XA(NA(IEL),IEL) - XA(1,IEL)
      DYW(IW,IEL) = YA(NA(IEL),IEL) - YA(1,IEL)
C
C---- set angles from first point to all J=1 TE points (used by PSIPHI)
      DO 10 JEL=1, NEL
        AVECW(IW,IEL,JEL) = AVECA(1,IEL,JEL)
        ANGL1(JEL) = AVECW(IW,IEL,JEL)
   10 CONTINUE
C
C---- set potential at first wake point
      CALL PSIPHI(1,IEL,XW(IW,IEL),YW(IW,IEL),ANGL1,
     &              PSIW,PHIW(IW,IEL),UWAKE,VWAKE)
      QWAKE = GAM(1,IEL)
      UWAKE = QWAKE*(-XAP(1,IEL))
      VWAKE = QWAKE*(-YAP(1,IEL))
C
C---- set streamfunction jump across TE gap
      DPSIEL(IEL) = SIGTE(IEL)*DSTE(IEL)
C
C---- set specified streamfunction along current wake
      PSII = PSIW
C
C---- set arc length to next wake point
      DS = 0.5*(  SA(2      ,IEL) - SA(1        ,IEL)
     &          + SA(NA(IEL),IEL) - SA(NA(IEL)-1,IEL))
      DS = MIN(DS, DSMAX)
C
      DO 20 IW=2, IQX
        DSFAC = 1.0
C
 18     DSF = DS * DSFAC
C
C------ specified potential at current wake point
        PHII = PHIW(IW-1,IEL) + DSF*QWAKE
C
C------ set initial guess for current wake point using u,v from previous point
        XW(IW,IEL) = XW(IW-1,IEL) + DSF*UWAKE/QWAKE
        YW(IW,IEL) = YW(IW-1,IEL) + DSF*VWAKE/QWAKE
C
C------ converge exact wake point with specified Psi, Phi
C
C------ Newton iteration to get specified Psi and Phi at current point
        DO 210 ITER=1, 5
C
C-------- set angles from first point to all J=1 TE points (used by PSIPHI)
          DO 2110 JEL=1, NEL
            RX1 = XW(IW,IEL) - XA(1,JEL)
            RY1 = YW(IW,IEL) - YA(1,JEL)
            ANGL1(JEL) = ATANC(RX1,RY1,AVECW(IW-1,IEL,JEL))
 2110     CONTINUE
C
          CALL PSIPHI(0,0,XW(IW,IEL),YW(IW,IEL),ANGL1,
     &                  PSIW,PHIW(IW,IEL),UWAKE,VWAKE)
C
          IF(UWAKE.EQ.0.0 .AND. VWAKE.EQ.0.0) THEN
C--------- we landed inside another element... try again with halved step
           DSFAC = DSFAC * 0.5
           IF(DSFAC .LT. 0.001) STOP 'WAKSET: Wake ran inside element'
           GO TO 18
          ENDIF

C-------- fill 2x2 Jacobian for  Psi(x,y), Phi(x,y)
          Z11 = -VWAKE
          Z12 =  UWAKE
          Z21 =  UWAKE
          Z22 =  VWAKE
C
C-------- residuals
          D1 = PSII - PSIW
          D2 = PHII - PHIW(IW,IEL)
C
C-------- solve for Newton deltas with Cramer's rule
          DET = Z11*Z22 - Z12*Z21
          DX = ( D1*Z22 - Z12*D2 )/DET
          DY = (Z11*D2  - D1 *Z21)/DET
C
C-------- update x and y
          XW(IW,IEL) = XW(IW,IEL) + DX
          YW(IW,IEL) = YW(IW,IEL) + DY
C
C-------- convergence check
          IF((DX*DX + DY*DY) .LT. STOL**2) GO TO 211
C
  210   CONTINUE
  211   CONTINUE
C
C------ set arc length
        SX = XW(IW,IEL) - XW(IW-1,IEL)
        SY = YW(IW,IEL) - YW(IW-1,IEL)
        SW(IW,IEL) = SW(IW-1,IEL) + SQRT(SX*SX + SY*SY)
C
C------ save final angles from converged wake point to first airfoil field points
        DO 30 JEL=1, NEL
          RX1 = XW(IW,IEL) - XA(1,JEL)
          RY1 = YW(IW,IEL) - YA(1,JEL)
          AVECW(IW,IEL,JEL) = ATANC(RX1,RY1,AVECW(IW-1,IEL,JEL))
   30   CONTINUE
C
C------ set position increments to adjacent wake from opposite TE point
        DXW(IW,IEL) =  DPSIEL(IEL) * VWAKE/(UWAKE**2 + VWAKE**2)
        DYW(IW,IEL) = -DPSIEL(IEL) * UWAKE/(UWAKE**2 + VWAKE**2)
C
        QWAKE = SQRT(UWAKE**2 + VWAKE**2)
C
C------ jump out if wake is long enough, setting new terminating potential
        IF(PHIW(IW,IEL) .GT. PHIEND) THEN
         NW(IEL) = IW
         PHIEND = PHIEND - CIRCEL(IEL)
         GO TO 21
        ENDIF
C
C------ set arc length increment for next point
        DS = DS * SGEOM
        IF(XW(IW,IEL) .LT. XMAX) DS = MIN(DS, DSMAX)
C
   20 CONTINUE ! with next wake point
      STOP 'WAKSET: Wake array overflow.  Increase IQX in PANEL.INC'
   21 CONTINUE
C
 1000 CONTINUE ! with next element
C
C---- spline wake trajectories
      DO 2000 IEL=1, NEL
        CALL SPLINE(XW(1,IEL),XWP(1,IEL),SW(1,IEL),NW(IEL))
        CALL SPLINE(YW(1,IEL),YWP(1,IEL),SW(1,IEL),NW(IEL))
 2000 CONTINUE
      RETURN
C
 9010 FORMAT(' Setting wake for element',I3)
      END ! WAKSET



      SUBROUTINE INLSET
C.....................................................
C     Sets wake coordinate array for current 
C     surface vorticity distributions.
C.....................................................
      INCLUDE 'PANEL.INC'
      DIMENSION ANGL1(IEX)
C
C---- find foremost and aftmost TE positions among all elements
      XMIN = XA(1,1)
      XMAX = XA(1,1)
      DO 5 IEL=1, NEL
        DO 51 I=1, NA(IEL)
          XMIN = MIN( XMIN , XA(I,IEL) )
          XMAX = MAX( XMAX , XA(I,IEL) )
   51   CONTINUE
    5 CONTINUE
C
C---- set maximum streamline length increment within chord limits
      DSMAX = 0.04*(XMAX-XMIN)
C
      STOL = 0.00005*(XMAX-XMIN)
C
C---- set stretching factor for increments beyond chord limits
      SGEOM = 1.2
C
      DO 1000 IEL=1, NEL
C
      WRITE(*,9010) IEL
C
C---- find stagnation point index
      DO 7 I=2, NA(IEL)
        IF(GAM(I,IEL) .LT. 0.0  .AND.  GAM(I-1,IEL) .GE. 0.0) THEN
         IST = I
         GO TO 8
        ENDIF
    7 CONTINUE
      STOP 'INLSET: Stagnation point not found'
    8 CONTINUE
C
      FRAC = GAM(IST,IEL)/(GAM(IST,IEL)-GAM(IST-1,IEL))
      DS  = SA(IST,IEL) - SA(IST-1,IEL)
      DX  = XA(IST,IEL) - XA(IST-1,IEL)
      DY  = YA(IST,IEL) - YA(IST-1,IEL)
C
C---- set stagnation point by linear interpolation from bounding nodes
ccc   XA1 = XA(IST,IEL) - FRAC*DX
ccc   YA1 = YA(IST,IEL) - FRAC*DY
C
C---- set stagnation point from spline rather than linear interpolation
      SA1 = SA(IST,IEL) - FRAC*DS
      XA1 = SEVAL(SA1,XA(1,IEL),XAP(1,IEL),SA(1,IEL),NA(IEL))
      YA1 = SEVAL(SA1,YA(1,IEL),YAP(1,IEL),SA(1,IEL),NA(IEL))
C
C---- first-point offset (fraction of stagnation panel length)
      DSNF = 0.0001
C
 10   IU = 1
      XU(IU,IEL) = XA1 + DSNF*DY
      YU(IU,IEL) = YA1 - DSNF*DX
      SU(IU,IEL) = 0.0
C
C---- set angles to all J=1 TE points (used by PSIPHI)
      DO 11 JEL=1, NEL
        RX1 = XU(IU,IEL) - XA(1,JEL)
        RY1 = YU(IU,IEL) - YA(1,JEL)
        AVECU(IU,IEL,JEL) = ATANC(RX1,RY1,AVECA(IST,IEL,JEL))
        ANGL1(JEL) = AVECU(IU,IEL,JEL)
   11 CONTINUE
C
C---- calculate streamfunction at first point
      CALL PSIPHI(0,0,XU(IU,IEL),YU(IU,IEL),ANGL1,
     &                PSIU,PHIU(IU,IEL),UWAKE,VWAKE)
      QWAKE = SQRT(UWAKE**2 + VWAKE**2)
C
      IF(QWAKE.EQ.0.0 .AND. DSNF.LT.1.0) THEN
C------ PSIPHI thought we were inside airfoil (numerical roundoff ?) ...
C-      so go out a bit further
        DSNF = DSNF*2.0
        GO TO 10
      ENDIF
C
C---- set specified streamfunction along current wake
      PSII = PSIU
C
C
C---- initialize second point off the surface DS away
      IU = 2
      DS = MIN( (SA(IST,IEL) - SA(IST-1,IEL)) , DSMAX )
      XU(IU,IEL) = XU(IU-1,IEL) + DY * DS/(SA(IST,IEL) - SA(IST-1,IEL))
      YU(IU,IEL) = YU(IU-1,IEL) - DX * DS/(SA(IST,IEL) - SA(IST-1,IEL))
C
C---- set angles to all J=1 TE points (used by PSIPHI)
      DO 12 JEL=1, NEL
        RX1 = XU(IU,IEL) - XA(1,JEL)
        RY1 = YU(IU,IEL) - YA(1,JEL)
        AVECU(IU,IEL,JEL) = ATANC(RX1,RY1,AVECU(IU-1,IEL,JEL))
        ANGL1(JEL) = AVECU(IU,IEL,JEL)
   12 CONTINUE
C
C---- calculate potential at second point
      CALL PSIPHI(0,0,XU(IU,IEL),YU(IU,IEL),ANGL1,
     &                PSIU,PHIU(IU,IEL),UWAKE,VWAKE)
      QWAKE = SQRT(UWAKE**2 + VWAKE**2)
C
C---- set specified potential for second point
      PHII = PHIU(IU,IEL)
C
C
C---- set remainder of points on inlet streamline
      DO 20 IU=2, IQX
C
C------ set specified potential at current point
        IF(IU.GT.2) THEN
         PHII = PHIU(IU-1,IEL) - DS*QWAKE
C
C------- set initial guess for current wake point using u,v from previous point
         XU(IU,IEL) = XU(IU-1,IEL) - DS*UWAKE/QWAKE
         YU(IU,IEL) = YU(IU-1,IEL) - DS*VWAKE/QWAKE
        ENDIF
C
C------ Newton iteration to get specified Psi and Phi at current point
        DO 210 ITER=1, 5
C
C-------- set angles from first point to all J=1 TE points (used by PSIPHI)
          DO 2110 JEL=1, NEL
            RX1 = XU(IU,IEL) - XA(1,JEL)
            RY1 = YU(IU,IEL) - YA(1,JEL)
            ANGL1(JEL) = ATANC(RX1,RY1,AVECU(IU-1,IEL,JEL))
 2110     CONTINUE
C
          CALL PSIPHI(0,0,XU(IU,IEL),YU(IU,IEL),ANGL1,
     &                  PSIU,PHIU(IU,IEL),UWAKE,VWAKE)
C
C-------- fill 2x2 Jacobian for  Psi(x,y), Phi(x,y)
          Z11 = -VWAKE
          Z12 =  UWAKE
          Z21 =  UWAKE
          Z22 =  VWAKE
C
C-------- residuals
          D1 = PSII - PSIU
          D2 = PHII - PHIU(IU,IEL)
C
C-------- solve for Newton deltas with Cramer's rule
          DET = Z11*Z22 - Z12*Z21
          DX = ( D1*Z22 - Z12*D2 )/DET
          DY = (Z11*D2  - D1 *Z21)/DET
C
C-------- update x and y
          XU(IU,IEL) = XU(IU,IEL) + DX
          YU(IU,IEL) = YU(IU,IEL) + DY
C
C-------- convergence check
          IF((DX*DX + DY*DY) .LT. STOL**2) GO TO 211
C
  210   CONTINUE
  211   CONTINUE
C
C------ set arc length
        SX = XU(IU,IEL) - XU(IU-1,IEL)
        SY = YU(IU,IEL) - YU(IU-1,IEL)
        SU(IU,IEL) = SU(IU-1,IEL) + SQRT(SX*SX + SY*SY)
C
C------ save final angles from converged wake point to first airfoil field points
        DO 30 JEL=1, NEL
          RX1 = XU(IU,IEL) - XA(1,JEL)
          RY1 = YU(IU,IEL) - YA(1,JEL)
          AVECU(IU,IEL,JEL) = ATANC(RX1,RY1,AVECU(IU-1,IEL,JEL))
   30   CONTINUE
C
        QWAKE = SQRT(UWAKE**2 + VWAKE**2)
C
C------ jump out if wake is long enough
        IF(PHIU(IU,IEL) .LT. PHIINL) THEN
         NU(IEL) = IU
         GO TO 21
        ENDIF
C
C------ set arc length increment for next point
        DS = DS * SGEOM
        IF(XU(IU,IEL) .GT. XMIN) DS = MIN(DS, DSMAX)
C
   20 CONTINUE ! with next point
      STOP 'INLSET: Inlet array overflow.  Increase IQX in PANEL.INC'
   21 CONTINUE
C
 1000 CONTINUE ! with next element
C
C---- spline trajectories
      DO 2000 IEL=1, NEL
        CALL SPLINE(XU(1,IEL),XUP(1,IEL),SU(1,IEL),NU(IEL))
        CALL SPLINE(YU(1,IEL),YUP(1,IEL),SU(1,IEL),NU(IEL))
 2000 CONTINUE
      RETURN
C
 9010 FORMAT(' Setting inlet streamline for element',I3)
      END ! INLSET


      SUBROUTINE FFSSET
C.....................................................
C     Sets farfield streamline array for current 
C     surface vorticity distributions.
C.....................................................
      INCLUDE 'PANEL.INC'
      DIMENSION ANGL1(IEX)
C
      IF(NEL.LT.1) RETURN
C
C---- set streamline increment 
      DS = 0.02*(PHIOUT-PHIINL)
C
      STOL = 0.00005*(PHIOUT - PHIINL)
C
C---- calculate top and bottom streamlines
      DO 1000 IST=1, 2
C
      WRITE(*,9010) IST
C
C**** calculate location of first point on streamline
C
      I = 1
      SF(I,IST) = 0.0
C
C---- set initial x,y guesses
      COSA = COS(ARAD)
      SINA = SIN(ARAD)
      IF(IST.EQ.1) THEN
C
C----- streamline at bottom of domain
       IEL = 1
       DPSI = PSIBOT - PSIO(  IEL)
       DPHI = PHIINL - PHIU(1,IEL)
       XF(I,IST) = XU(1,IEL) + DPHI*COSA - DPSI*SINA
       YF(I,IST) = YU(1,IEL) + DPSI*COSA + DPHI*SINA
       PSII = PSIBOT
       PHII = PHIINL
       PHIEND = PHIOUT - CIRC
C
C----- set angles to all J=1 TE points for ATANC in 2110 loop below
       DO 11 JEL=1, NEL
         RX1 = XF(I,IST) - XA(1,JEL)
         RY1 = YF(I,IST) - YA(1,JEL)
         AVECF(I-1,IST,JEL) = ATANC(RX1,RY1,AVECA(NA(NEL),NEL,JEL))
   11  CONTINUE
C
      ELSE
C
C----- streamline at top of domain
       IEL = 1
       DPSI = PSITOP - PSIO(  IEL)
       DPHI = PHIINL - PHIU(1,IEL)
       XF(I,IST) = XU(1,IEL) + DPHI*COSA - DPSI*SINA
       YF(I,IST) = YU(1,IEL) + DPSI*COSA + DPHI*SINA
       PSII = PSITOP
       PHII = PHIINL
       PHIEND = PHIOUT
C
C----- set angles to all J=1 TE points for ATANC in 2110 loop below
       DO 12 JEL=1, NEL
         RX1 = XF(I,IST) - XA(1,JEL)
         RY1 = YF(I,IST) - YA(1,JEL)
         AVECF(I-1,IST,JEL) = ATANC(RX1,RY1,AVECA(1,1,JEL))
   12  CONTINUE
C
      ENDIF
C
C
C---- converge on remainder of points at specified Phi, Psi
      DO 20 I=1, IQX
C
        IF(I.GT.1) THEN
C------- specified potential at current wake point
         PHII = PHIF(I-1,IST) + DS*QWAKE
C
C------- set initial guess for current wake point using u,v from previous point
         XF(I,IST) = XF(I-1,IST) + DS*UWAKE/QWAKE
         YF(I,IST) = YF(I-1,IST) + DS*VWAKE/QWAKE
        ENDIF
C
C------ Newton iteration to get exact specified Psi and Phi at current point
        DO 210 ITER=1, 5
C
C-------- set angles from first point to all J=1 TE points (used by PSIPHI)
          DO 2110 JEL=1, NEL
            RX1 = XF(I,IST) - XA(1,JEL)
            RY1 = YF(I,IST) - YA(1,JEL)
            ANGL1(JEL) = ATANC(RX1,RY1,AVECF(I-1,IST,JEL))
 2110     CONTINUE
C
          CALL PSIPHI(0,0,XF(I,IST),YF(I,IST),ANGL1,
     &                  PSIF,PHIF(I,IST),UWAKE,VWAKE)
C
C-------- fill 2x2 Jacobian for  Psi(x,y), Phi(x,y)
          Z11 = -VWAKE
          Z12 =  UWAKE
          Z21 =  UWAKE
          Z22 =  VWAKE
C
C-------- residuals
          D1 = PSII - PSIF
          D2 = PHII - PHIF(I,IST)
C
C-------- solve for Newton deltas with Cramer's rule
          DET = Z11*Z22 - Z12*Z21
          DX = ( D1*Z22 - Z12*D2 )/DET
          DY = (Z11*D2  - D1 *Z21)/DET
C
C-------- update x and y
          XF(I,IST) = XF(I,IST) + DX
          YF(I,IST) = YF(I,IST) + DY
C
C-------- convergence check
          IF((DX*DX + DY*DY) .LT. STOL**2) GO TO 211
C
  210   CONTINUE
  211   CONTINUE
C
        IF(i.GT.1) THEN
C------- set arc length
         SX = XF(I,IST) - XF(I-1,IST)
         SY = YF(I,IST) - YF(I-1,IST)
         SF(I,IST) = SF(I-1,IST) + SQRT(SX*SX + SY*SY)
        ENDIF
C
C------ save final angles from converged wake point to first airfoil field points
        DO 230 JEL=1, NEL
          RX1 = XF(I,IST) - XA(1,JEL)
          RY1 = YF(I,IST) - YA(1,JEL)
          AVECF(I,IST,JEL) = ATANC(RX1,RY1,AVECF(I-1,IST,JEL))
  230   CONTINUE
C
        QWAKE = SQRT(UWAKE**2 + VWAKE**2)
C
C------ jump out if wake is long enough
        IF(PHIF(I,IST) .GT. PHIEND) THEN
         NF(IST) = I
         GO TO 21
        ENDIF
C
   20 CONTINUE ! with next wake point
      STOP 'FFSSET: Farfield streamline array overflow.  Increase IQX.'
   21 CONTINUE
C
 1000 CONTINUE ! with next streamline
C
C---- spline trajectories
      DO 2000 IST=1, 2
        CALL SPLINE(XF(1,IST),XFP(1,IST),SF(1,IST),NF(IST))
        CALL SPLINE(YF(1,IST),YFP(1,IST),SF(1,IST),NF(IST))
 2000 CONTINUE
      RETURN
C
 9010 FORMAT(' Setting farfield streamline', I3)
      END ! FFSSET


      SUBROUTINE STFIND(SST)
      INCLUDE 'PANEL.INC'
      DIMENSION SST(IEX)
C
      DO 100 IEL=1, NEL
C
C------ find stagnation point index
        DO 7 IST=2, NA(IEL)
          IF((GAM(IST  ,IEL) .LT. 0.0) .AND.  
     &       (GAM(IST-1,IEL) .GE. 0.0)       ) GO TO 8
    7   CONTINUE
        STOP 'STFIND: Stagnation point not found'
C
    8   FRAC = GAM(IST,IEL)/(GAM(IST,IEL)-GAM(IST-1,IEL))
        SST(IEL) = SA(IST,IEL) - FRAC*(SA(IST,IEL) - SA(IST-1,IEL))
C
  100 CONTINUE
C
      RETURN
      END ! STFIND



      SUBROUTINE NORGEN(NSX,NST,ISTLE,ISTTE,
     &                  XST,YST,SST,XSTP,YSTP,PHIST,PSIST,
     &                  PHINL,PHIEND,
     &                  NLX,NLIN,SSTL,ISOR,LEOR)
      INCLUDE 'PANEL.INC'
C
      INTEGER NST(2*IEX),ISTLE(2*IEX),ISTTE(2*IEX)
      DIMENSION  XST(NSX,2*IEX), YST(NSX,2*IEX),SST(NSX,2*IEX),
     &          XSTP(NSX,2*IEX),YSTP(NSX,2*IEX),
     &         PHIST(NSX,2*IEX),PSIST(2*IEX),PHIEND(2*IEX)
C
      DIMENSION NLIN(NLX),ISOR(NLX,NLX)
      DIMENSION SSTL(NLX,NLX)
      LOGICAL LEOR(NLX,NLX)
C
C-----------------------------------------------------------
C     Divides up the inter-element streamtubes into blocks,
C     with each block boundary emanating from a LE or TE
C     on some other element (not necessarily one of the two
C     bounding the inter-element steramtube.  The big
C     streamtubes above and below the configuration are
C     also divided up in the same manner.
C-----------------------------------------------------------
      DIMENSION ANGL1(IEX)
C
      PARAMETER (NLDIM = 2*IEX+2)
      DIMENSION ISORT(NLDIM), IWORK(NLDIM)
      DIMENSION RWORK(NLDIM)
      LOGICAL LWORK(NLDIM)
C
      IF(NEL.LT.1) RETURN
C
      PHIINL = PHINL
C
      DO 5 IS=1, 2*NEL+2
        NLIN(IS) = 0
        DO 51 IL=1, NLX
          ISOR(IL,IS) = 0
   51   CONTINUE
    5 CONTINUE
C
C---- set normal line points at inlet boundary
      DO 7 IS=1, 2*NEL+2
        I = 1
        FRAC = (PHIINL - PHIST(I,IS))/(PHIST(I+1,IS) - PHIST(I,IS))
        NLIN(IS) = NLIN(IS) + 1
        SSTL(NLIN(IS),IS) = SST(I,IS) + FRAC*(SST(I+1,IS) - SST(I,IS))
    7 CONTINUE
C
C---- set normal lines at exit boundary
      DO 9 IS=1, 2*NEL+2
        I = NST(IS)-1
        FRAC = (PHIEND(IS) - PHIST(I,IS))/(PHIST(I+1,IS) - PHIST(I,IS))
        NLIN(IS) = NLIN(IS) + 1
        SSTL(NLIN(IS),IS) = SST(I,IS) + FRAC*(SST(I+1,IS) - SST(I,IS))
    9 CONTINUE
C
C
C---- go over streamlines, setting normal lines from LE and TE
      DO 1000 IS=1, 2*NEL
C
C------ increment normal line counters for current streamlines
        NLIN(IS) = NLIN(IS) + 1
C
C------ set position on current streamline (at LE)
        SSTL(NLIN(IS),IS) = SST(ISTLE(IS),IS)
C
C------ set index of "originating" streamline (itself in this case)
        ISOR(NLIN(IS),IS) = IS
        LEOR(NLIN(IS),IS) = .TRUE.
C
        NLIN(IS) = NLIN(IS) + 1
C
C------ set position on current streamline (at TE)
        SSTL(NLIN(IS),IS) = SST(ISTTE(IS),IS)
C
        ISOR(NLIN(IS),IS) = IS
        LEOR(NLIN(IS),IS) = .FALSE.
C
C------ do the trailing edge and leading edge
        DO 500 ILETE=1, 2
C
        IF(MOD(IS,2) .EQ. 1) THEN
C------- we're on upper airfoil side -- go up
C
C------- set potential above streamline
         IF(ILETE.EQ.1) PHILIN = PHIST(ISTLE(IS),IS)
         IF(ILETE.EQ.2) PHILIN = PHIST(ISTTE(IS),IS)
C
C------- go over all streamlines above current streamline
         DO 10 JST=IS-1, 0, -2
           JS2 = JST
           IF(JST.EQ.0) JS2 = 2*NEL+2
           JS1 = JS2-1
C
C--------- find interval with the same potential as point on streamline below
           DO 110 I=2, NST(JS2)
             IF(PHIST(I,JS2) .GE. PHILIN) GO TO 111
  110      CONTINUE
           WRITE(*,*) 'NORGEN: potential intersection not found'
  111      CONTINUE
C
C--------- set arc length of point with the same potential as point below
           FRAC = (PHILIN       - PHIST(I-1,JS2))
     &          / (PHIST(I,JS2) - PHIST(I-1,JS2))
           NLIN(JS2) = NLIN(JS2) + 1
           SSTL(NLIN(JS2),JS2) = SST(I-1,JS2)
     &        + FRAC*(SST(I,JS2)-SST(I-1,JS2))
C--------- set index of "originating" streamline
           ISOR(NLIN(JS2),JS2) = IS
           LEOR(NLIN(JS2),JS2) = ILETE.EQ.1
C
           IF(SSTL(NLIN(JS2),JS2) .LE. SST(ISTLE(JS2),JS2)) THEN
C---------- new point falls before LE: set side 1 streamline and keep going
            NLIN(JS1) = NLIN(JS1) + 1
            SSTL(NLIN(JS1),JS1) = SSTL(NLIN(JS2),JS2)
            ISOR(NLIN(JS1),JS1) = IS
            LEOR(NLIN(JS1),JS1) = ILETE.EQ.1
           ELSE IF(SSTL(NLIN(JS2),JS2) .GE. SST(ISTTE(JS2),JS2)) THEN
C---------- new point falls after TE: set side 1 streamline and keep going
            NLIN(JS1) = NLIN(JS1) + 1
            SSTL(NLIN(JS1),JS1) = SSTL(NLIN(JS2),JS2)
     &              + SST(NST(JS1),JS1) - SST(NST(JS2),JS2)
            ISOR(NLIN(JS1),JS1) = IS
            LEOR(NLIN(JS1),JS1) = ILETE.EQ.1
C---------- set new potential to account for jump across wake
            IOPP = I + NST(JS1) - NST(JS2)
            FRAC = (SSTL(NLIN(JS1),JS1) - SST(IOPP-1,JS1))
     &           / (SST(IOPP,JS1)       - SST(IOPP-1,JS1))
            PHILIN = PHIST(IOPP-1,JS1)
     &             + FRAC*(PHIST(IOPP,JS1) - PHIST(IOPP-1,JS1))
           ELSE
C---------- new point falls on element surface: stop normal line here
            GO TO 11
           ENDIF
C
   10    CONTINUE
   11    CONTINUE
C
C
        ELSE
C------- we're on airfoil lower side -- go down
C
C------- set potential below streamline
         IF(ILETE.EQ.1) PHILIN = PHIST(ISTLE(IS),IS)
         IF(ILETE.EQ.2) PHILIN = PHIST(ISTTE(IS),IS)
C
C------- go over all streamlines below current streamline
         DO 20 JS1=IS+1, 2*NEL+2, 2
           JS2 = JS1+1
C
C--------- find interval with the same potential as point on streamline above
           DO 210 I=2, NST(JS1)
             IF(PHIST(I,JS1) .GE. PHILIN) GO TO 211
  210      CONTINUE
           WRITE(*,*) 'NORGEN: potential intersection not found'
  211      CONTINUE
C
C--------- set arc length of point with the same potential as point above
           FRAC = (PHILIN       - PHIST(I-1,JS1))
     &          / (PHIST(I,JS1) - PHIST(I-1,JS1))
           NLIN(JS1) = NLIN(JS1) + 1
           SSTL(NLIN(JS1),JS1) = SST(I-1,JS1)
     &             + FRAC*(SST(I,JS1)-SST(I-1,JS1))
           ISOR(NLIN(JS1),JS1) = IS
           LEOR(NLIN(JS1),JS1) = ILETE.EQ.1
C
           IF(SSTL(NLIN(JS1),JS1) .LE. SST(ISTLE(JS1),JS1)) THEN
C---------- new point falls before LE: set side 2 streamline and keep going
            NLIN(JS2) = NLIN(JS2) + 1
            SSTL(NLIN(JS2),JS2) = SSTL(NLIN(JS1),JS1)
            ISOR(NLIN(JS2),JS2) = IS
            LEOR(NLIN(JS2),JS2) = ILETE.EQ.1
           ELSE IF(SSTL(NLIN(JS1),JS1) .GE. SST(ISTTE(JS1),JS1)) THEN
C---------- new point falls after TE: set side 2 streamline and keep going
            NLIN(JS2) = NLIN(JS2) + 1
            SSTL(NLIN(JS2),JS2) = SSTL(NLIN(JS1),JS1)
     &          + SST(NST(JS2),JS2) - SST(NST(JS1),JS1)
            ISOR(NLIN(JS2),JS2) = IS
            LEOR(NLIN(JS2),JS2) = ILETE.EQ.1
C---------- set new potential to account for jump across wake
            IOPP = I + NST(JS2) - NST(JS1)
            FRAC = (SSTL(NLIN(JS2),JS2) - SST(IOPP-1,JS2))
     &           / (SST(IOPP,JS2)       - SST(IOPP-1,JS2))
            PHILIN = PHIST(IOPP-1,JS2)
     &             + FRAC*(PHIST(IOPP,JS2)-PHIST(IOPP-1,JS2))
           ELSE
C---------- new point falls on element surface: stop normal line here
            GO TO 21
           ENDIF
C
   20    CONTINUE ! with next streamline/normal line intersection
   21    CONTINUE
C
        ENDIF
C
  500   CONTINUE ! with TE line
C
 1000 CONTINUE ! with next element
C
C
C---- sort normal line intersections along each streamline
      DO 2000 IS=1, 2*NEL+2
        CALL SORT(SSTL(1,IS),NLIN(IS),ISORT)
C
        DO 40 IL=1, NLIN(IS)
          RWORK(IL) = SSTL(ISORT(IL),IS)
          IWORK(IL) = ISOR(ISORT(IL),IS)
          LWORK(IL) = LEOR(ISORT(IL),IS)
   40   CONTINUE
C
        DO 45 IL=1, NLIN(IS)
          SSTL(IL,IS) = RWORK(IL)
          ISOR(IL,IS) = IWORK(IL)
          LEOR(IL,IS) = LWORK(IL)
   45   CONTINUE
C
 2000 CONTINUE
C
      RETURN
      END



      SUBROUTINE SORT(A,N,ISORT)
      DIMENSION A(N)
      DIMENSION ISORT(N)
      LOGICAL DONE
C
      DO 1 I=1, N
        ISORT(I) = I
    1 CONTINUE
C
C
      DO 2 IPASS=1, 2*N
C
        DONE = .TRUE.
C
        DO 20 I=2, N
          ISM = ISORT(I-1)
          ISO = ISORT(I)
          IF(A(ISO) .LT. A(ISM)) THEN
           ISORT(I)   = ISM
           ISORT(I-1) = ISO
           DONE = .FALSE.
          ENDIF
   20   CONTINUE
C
        IF(DONE) RETURN
C
    2 CONTINUE
C
      WRITE(*,*) 'Sort failed.  Continuing ...'
      RETURN
      END



      SUBROUTINE LEDPSI(NDIM,XB,XBP,YB,YBP,SB,NB,SBLE, 
     &                  IGX,SG0,DPSILE,CELLAR)
      INCLUDE 'PANEL.INC'
C
      DIMENSION XB(NDIM,*), XBP(NDIM,*), 
     &          YB(NDIM,*), YBP(NDIM,*), SB(NDIM,*), SBLE(*)
      DIMENSION NB(*)
      DIMENSION SG0(IGX,*),DPSILE(*)
C---------------------------------------------------------------
C     Calculates streamfunction values at the two points in the 
C     stagnation-point "corners".  The points are placed on the 
C     corner bisectors at a distance which is a multiple CELLAR
C     of the average surface spacing at the stagnation point.
C---------------------------------------------------------------
      DIMENSION ANGL1(IEX)
C
      DO 1000 IEL=1, NEL
C
        XX = SEVAL(SBLE(IEL),XB(1,IEL),XBP(1,IEL),SB(1,IEL),NB(IEL))
        YY = SEVAL(SBLE(IEL),YB(1,IEL),YBP(1,IEL),SB(1,IEL),NB(IEL))
        XD = DEVAL(SBLE(IEL),XB(1,IEL),XBP(1,IEL),SB(1,IEL),NB(IEL))
        YD = DEVAL(SBLE(IEL),YB(1,IEL),YBP(1,IEL),SB(1,IEL),NB(IEL))
C
        IS1 = 2*IEL-1
        IS2 = 2*IEL
C
        SSURF1 = ABS( SB(      1,IEL) - SBLE(IEL) )
        SSURF2 = ABS( SB(NB(IEL),IEL) - SBLE(IEL) )
C
C------ set surface node spacing on each side at LE
        DS1 = SSURF1*(SG0(2,IS1) - SG0(1,IS1))
        DS2 = SSURF2*(SG0(2,IS2) - SG0(1,IS2))
C
        DSAVG = 0.5*(DS1 + DS2)
C
C------ set points in stagnation corners 45 degrees off the surface
        XX1 = XX + CELLAR*DSAVG*(-XD + YD)*0.707
        YY1 = YY + CELLAR*DSAVG*(-XD - YD)*0.707
        XX2 = XX + CELLAR*DSAVG*( XD + YD)*0.707
        YY2 = YY + CELLAR*DSAVG*(-XD + YD)*0.707
C
C------ find the airfoil surface node nearest the stagnation point
        DSMIN = (XA(1,IEL)-XX)**2 + (YA(1,IEL)-YY)**2
        IAMIN = 1
        DO I=1, NA(IEL)
          DS = (XA(I,IEL)-XX)**2 + (YA(I,IEL)-YY)**2
          IF(DS .LT. DSMIN) THEN
            DSMIN = DS
            IAMIN = I
          ENDIF
        ENDDO
C
        I = IAMIN
C
C------ set initial angles to all J=1 TE points
        DO 10 JEL=1, NEL
          RX1 = XX1 - XA(1,JEL)
          RY1 = YY1 - YA(1,JEL)
          ANGL1(JEL) = ATANC(RX1,RY1,AVECA(I,IEL,JEL))
   10   CONTINUE
C
C------ calculate streamfunction at point 1
        CALL PSIPHI(0,0,XX1,YY1,ANGL1,PSII,PHII,UII,VII)
        DPSILE(IS1) = PSII - PSIO(IEL)
C
C
C------ set initial angles to all J=1 TE points
        DO 20 JEL=1, NEL
          RX1 = XX2 - XA(1,JEL)
          RY1 = YY2 - YA(1,JEL)
          ANGL1(JEL) = ATANC(RX1,RY1,AVECA(I,IEL,JEL))
   20   CONTINUE
C
C------ calculate streamfunction at point 2
        CALL PSIPHI(0,0,XX2,YY2,ANGL1,PSII,PHII,UII,VII)
        DPSILE(IS2) = PSIO(IEL) - PSII
C
C------ average spacings to make stagnation streamtubes equal  (22 Aug 92)
        DPSAVG = 0.5*(DPSILE(IS1) + DPSILE(IS2))
        DPSILE(IS1) = DPSAVG
        DPSILE(IS2) = DPSAVG
C
 1000 CONTINUE
C
      RETURN
      END


      SUBROUTINE CPELPL(IEL,XOFF,POFF,XFAC,PFAC)
      INCLUDE 'PANEL.INC'
C
      IPEN = 3
      DO 10 I=1, NA(IEL)
        XPLT = (     XA(I,IEL)    - XOFF)*XFAC
        YPLT = (1.0-GAM(I,IEL)**2 - POFF)*PFAC
        CALL PLOT(XPLT,YPLT,IPEN)
        IPEN = 2
   10 CONTINUE
C
      RETURN
      END


      SUBROUTINE CPLIMS(IEL,CPMIN,CPMAX)
      INCLUDE 'PANEL.INC'
C
      IF(IEL.EQ.0) THEN
       IEL1 = 1
       IEL2 = NEL
      ELSE
       IEL1 = IEL
       IEL2 = IEL
      ENDIF
C
      CPMIN = 1.0-GAM(1,IEL1)**2
      CPMAX = 1.0-GAM(1,IEL1)**2
C
      DO 10 JEL=IEL1, IEL2
        DO 105 I=1, NA(JEL)
          CPMIN = MIN( CPMIN , 1.0-GAM(I,JEL)**2 )
          CPMAX = MAX( CPMAX , 1.0-GAM(I,JEL)**2 )
 105    CONTINUE
   10 CONTINUE
C
      RETURN
      END



      SUBROUTINE CPPLOT(XOFF,XFAC,PFAC,MINF)
      INCLUDE 'PANEL.INC'
C
C---- plot inviscid -Cp distribution on airfoil elements
C
      BETA = SQRT(1.0 - MINF**2)
      BFAC = 0.5*MINF**2 / (1.0 + BETA)
C
      CALL NEWPEN(2)
      DO 900 L=1, NEL
C
        IPEN = 3
        DO 20 I=1, NA(L)
          CPINC = 1.0 - GAM(I,L)**2
          CPCOM = CPINC / (BETA + BFAC*CPINC)
          CALL PLOT((XA(I,L)+XOFF)*XFAC,-PFAC*CPCOM,IPEN)
          IPEN = 2
 20     CONTINUE
C
 900  CONTINUE
C
      RETURN
      END ! CPPLOT



      SUBROUTINE PANPLT(SFAC,XOFF,YOFF)
C-----------------------------------------------------
C     Shows panel nodes on current airfoil geometry.
C-----------------------------------------------------
      INCLUDE 'PANEL.INC'
      XMOD(XTMP) = SFAC * (XTMP - XOFF)
      YMOD(YTMP) = SFAC * (YTMP - YOFF)
C
      DSN = 0.0075
C
      CALL PLOT(0.05,0.05,-3)
C
      DO 10 L=1, NEL
        DO 110 I=2, NA(L)
          CALL PLOT(XMOD(XA(I-1,L)),YMOD(YA(I-1,L)),3)
          CALL PLOT(XMOD(XA(I  ,L)),YMOD(YA(I  ,L)),2)
          CALL PLOT(XMOD(XA(I  ,L)) + DSN*NX(I,L),
     &              YMOD(YA(I  ,L)) + DSN*NY(I,L),2)
 110    CONTINUE
   10 CONTINUE
C
      RETURN
      END
