C
      SUBROUTINE BLSYS(NBL)
C.......................................................................
C
C     Sets up the Newton block system governing the current i interval.
C     Each block row (with 3 sub-rows detailed below), governs one layer.
C
C      |VS1      ||dV1|   |VS2      ||dV2|   |VSR|        |VSREZ|
C      |  .      || . |   |  .      || . |   | . |        |  .  |
C      |   VS1   ||dV1| + |   VS2   ||dV2| + |VSR|dRe  =  |VSREZ|
C      |      .  || . |   |      .  || . |   | . |        |  .  |
C      |        .|| . |   |        .|| . |   | . |        |  .  |
C
C       VV1 | VVX1         VV2 | VVX2         VVR          VVREZ
C
C     Each block row subsystem looks like:
C
C     |       ||dA1|     |       ||dA2|     |   |           |     |
C     |  VS1  ||dT1|  +  |  VS2  ||dT2|  +  |VSR| dRe   =   |VSREZ|
C     |       ||dD1|     |       ||dD2|     |   |           |     |
C              |dU1|              |dU2|
C              |dX1|              |dX2|
C              |dN1|              |dN2|
C              |dR1|              |dR2|
C
C        3x7    7x1         3x7    7x1       3x1              3x1
C
C
C     A   amplification variable (laminar)  or max shear stress (turbulent)
C     T   Theta
C     D   Dstar
C     U   Uedge
C     X   Xi  (arc-length BL coordinate)
C     N   dUe/dy  edge velocity gradient due to longitudinal curvature
C     R   Rho     edge density
C
C
C     The three rows in the subsystem contain:
C   1) the amplification equation (or lag equation for turbulent stations)
C   2) the momentum equation
C   3) the shape parameter equation
C
C     The subsystem as shown corresponds to a laminar station.
C       If TRAN, then  dS2  replaces  dA2
C       If TURB, then  dS1, dS2  replace  dA1, dA2
C
C     The subsystem also contains an empty 4th row which is only used 
C       to impose some Ue constraint during a marching calculation. 
C
C..................................................................
      IMPLICIT REAL (A-H,M,O-Z)
      INCLUDE 'MBL.INC'
C
C---- zero out overall system
      DO 1 N=1, NBL
        DO 10 KV=1, 8
          VVREZ(KV,N) = 0.0
          VVR(KV,N)   = 0.0
          DO 101 LV=1, 8
            VV1(KV,LV,N) = 0.0
            VV2(KV,LV,N) = 0.0
  101     CONTINUE
          DO 102 LX=1, 2
            VVX1(KV,LX,N) = 0.0
            VVX2(KV,LX,N) = 0.0
            VVN1(KV,LX,N) = 0.0
            VVN2(KV,LX,N) = 0.0
            VVR1(KV,LX,N) = 0.0
            VVR2(KV,LX,N) = 0.0
            VVXF(KV,LX,N) = 0.0
  102     CONTINUE
   10   CONTINUE
    1 CONTINUE
C
C
C---- go over elements
      DO 1000 N=1, NBL
C
C---- skip this element if we converged
      IF(CONVB(N)) GO TO 1000
C
C---- go over the two sides of this element
      DO 900 KS=1, 2
C
      IS = 2*(N-1) + KS
C
C---- if we're upstream of LE at this layer, put in dummy coefficients and go on
      IF(UPLE(IS)) THEN
       DO 15 KV=1, 8
         VV2(KV,KV,N) = 1.0
   15  CONTINUE
C
       GO TO 1000
      ENDIF
C
C---- recall all variables and sensitivities
      DO 20 NC=1, NCOM
        COM1(NC) = V1SAV(NC,IS)
        COM2(NC) = V2SAV(NC,IS)
   20 CONTINUE
C
      DO 22 NC=1, NCOMA
        COMA(NC) = VASAV(NC,IS)
   22 CONTINUE
C
C**** set up appropriate finite difference subsystem for current layer
      ISIDE = IS
      IF(SIMI(IS)) THEN
C
C----- similarity station -- "1", "2", and midpoint variables are the same
       DO 30 NC=1, NCOM
         COM1(NC) = COM2(NC)
   30  CONTINUE
       CFM    = CF2
       CFM_RE = CF2_RE
       CFM_R1 = 0.
       CFM_U1 = 0.
       CFM_T1 = 0.
       CFM_D1 = 0.
       CFM_R2 = CF2_R2
       CFM_U2 = CF2_U2
       CFM_T2 = CF2_T2
       CFM_D2 = CF2_D2
C
       CALL BLDIF(0)
C
C----- "1" variables are really "2" variables, so lump derivatives together
       DO 32 K=1, 4
         DO 321 L=1, 7
           VS2(K,L) = VS1(K,L) + VS2(K,L)
           VS1(K,L) = 0.
  321    CONTINUE
   32  CONTINUE
C
      ELSE IF(TRAN(IS)) THEN
C
C----- transition interval
       CALL TRDIF
C
      ELSE IF(.NOT.TURB(IS)) THEN
C
C----- laminar interval
       CALL BLDIF(1)
C
      ELSE IF(WAKE(IS)) THEN
C
C----- wake interval
       CALL BLDIF(3)
C
      ELSE IF(TURB(IS)) THEN
C
C----- turbulent interval
       CALL BLDIF(2)
C
      ENDIF
C
C---- put system for current layer into overall system
      KOFF = 4*(KS-1)
      LOFF = 4*(KS-1)
      LX = KS
      DO 40 K=1, 4
        KV = KOFF + K
        VVREZ(KV,N) = VSREZ(K)
        VVR(KV,N)   = VSR(K)
        DO 401 L=1, 4
          LV = LOFF + L
          VV1(KV,LV,N) = VS1(K,L)
          VV2(KV,LV,N) = VS2(K,L)
  401   CONTINUE
C
C------ the xi and dudn columns are stored in separate arrays
        VVX1(KV,LX,N) = VS1(K,5)
        VVX2(KV,LX,N) = VS2(K,5)
        VVN1(KV,LX,N) = VS1(K,6)
        VVN2(KV,LX,N) = VS2(K,6)
        VVR1(KV,LX,N) = VS1(K,7)
        VVR2(KV,LX,N) = VS2(K,7)
        VVXF(KV,LX,N) = VSX(K)
   40 CONTINUE
C
  900 CONTINUE ! with next side of this element
C
C---- add on entrainment and mixing layer dissipation terms
      IF(WAKE(2*N)) CALL BLCENT(N)
C
 1000 CONTINUE ! with next element
C
      RETURN
      END ! BLSYS



      SUBROUTINE BLCENT(NAIR)
C.........................................
C     Adds entrainment terms to momentum 
C     and shape parameter equations
C.........................................
      IMPLICIT REAL (A-H,M,O-Z)
      INCLUDE 'MBL.INC'
ccc      common /yy/ tyysav(nex), uyysav(nex), ussav(nex), cesav(nex),
ccc     &            dussav(nex)
C
      N = NAIR
C
C---- go over upper (KSP) and lower (KSM) wake decks for element NAIR
      KSP = 1
      KSM = 2
C
      ISP = 2*(N-1) + KSP
      ISM = 2*(N-1) + KSM
C
C---- recall saved variables for lower layer
      DO 10 NC=1, NCOM
        COM1(NC) = V1SAV(NC,ISM)
        COM2(NC) = V2SAV(NC,ISM)
   10 CONTINUE
C
      DXM    = X2 - X1
C
      DUM    = US2    - US1
      DUM_R2 = US2_R2
      DUM_U2 = US2_U2
      DUM_T2 = US2_T2
      DUM_D2 = US2_D2
      DUM_R1 =        - US1_R1
      DUM_U1 =        - US1_U1
      DUM_T1 =        - US1_T1
      DUM_D1 =        - US1_D1
      DUM_RE = US2_RE - US1_RE
C
      XIM    = X2
      THM    = T2
      CTM    = S2
      UEM    = U2
C
      HKM    = HK2
      HKM_UM = HK2_U2
      HKM_TM = HK2_T2
      HKM_DM = HK2_D2
C
      RTM    = RT2
      RTM_RM = RT2_R2
      RTM_UM = RT2_U2
      RTM_TM = RT2_T2
      RTM_RE = RT2_RE
C
      USM    = US2
      USM_RM = US2_R2
      USM_UM = US2_U2
      USM_TM = US2_T2
      USM_DM = US2_D2
      USM_RE = US2_RE
C
      HSM    = HS2
      HSM_RM = HS2_R2
      HSM_UM = HS2_U2
      HSM_TM = HS2_T2
      HSM_DM = HS2_D2
      HSM_RE = HS2_RE
C
      CQM    = CQ2
      CQM_RM = CQ2_R2
      CQM_UM = CQ2_U2
      CQM_TM = CQ2_T2
      CQM_DM = CQ2_D2
      CQM_NM = CQ2_UN2
      CQM_RE = CQ2_RE
C
      DEM    = DE2
      DEM_UM = DE2_U2
      DEM_TM = DE2_T2
      DEM_DM = DE2_D2
C
      DWM    = 0.5*(DW1+DW2)
C
C
C---- recall saved variables for upper layer
      DO 20 NC=1, NCOM
        COM1(NC) = V1SAV(NC,ISP)
        COM2(NC) = V2SAV(NC,ISP)
   20 CONTINUE
C
      DXP    = X2 - X1
C
      DUP    = US2    - US1
      DUP_R2 = US2_R2
      DUP_U2 = US2_U2
      DUP_T2 = US2_T2
      DUP_D2 = US2_D2
      DUP_R1 =        - US1_R1
      DUP_U1 =        - US1_U1
      DUP_T1 =        - US1_T1
      DUP_D1 =        - US1_D1
      DUP_RE = US2_RE - US1_RE
C
      XIP    = X2
      THP    = T2
      CTP    = S2
      UEP    = U2
C
      HKP    = HK2
      HKP_UP = HK2_U2
      HKP_TP = HK2_T2
      HKP_DP = HK2_D2
C
      RTP    = RT2
      RTP_RP = RT2_R2
      RTP_UP = RT2_U2
      RTP_TP = RT2_T2
      RTP_RE = RT2_RE
C
      USP    = US2
      USP_RP = US2_R2
      USP_UP = US2_U2
      USP_TP = US2_T2
      USP_DP = US2_D2
      USP_RE = US2_RE
C
      HSP    = HS2
      HSP_RP = HS2_R2
      HSP_UP = HS2_U2
      HSP_TP = HS2_T2
      HSP_DP = HS2_D2
      HSP_RE = HS2_RE
C
      CQP    = CQ2
      CQP_RP = CQ2_R2
      CQP_UP = CQ2_U2
      CQP_TP = CQ2_T2
      CQP_DP = CQ2_D2
      CQP_NP = CQ2_UN2
      CQP_RE = CQ2_RE
C
      DEP    = DE2
      DEP_UP = DE2_U2
      DEP_TP = DE2_T2
      DEP_DP = DE2_D2
C
      DWP    = 0.5*(DW1+DW2)
C
C
C          2      2
C---- set d U / dy   ( without scale factor )
      UYY     =      (1.0-USP)/DEP**2 +     (1.0-USM)/DEM**2
      UYY_USP =          -1.0 /DEP**2
      UYY_USM =                                 -1.0 /DEM**2
      UYY_DEP = -2.0*(1.0-USP)/DEP**3
      UYY_DEM =                       - 2.0*(1.0-USM)/DEM**3
C
C          2        2
C---- set d Tau / dy   ( without scale factor )
      TYY     =      CTP**2/DEP**2 -     CTM**2/DEM**2
      TYY_CTP =  2.0*CTP   /DEP**2
      TYY_CTM =                    - 2.0*CTM   /DEM**2
      TYY_DEP = -2.0*CTP**2/DEP**3
      TYY_DEM =                      2.0*CTM**2/DEM**3
C
C---- set inner shear layer thickness
      DDK = 1000.0
      IF(HKM .GT. HKP) THEN
       DEI    = DDK*THP/RTP + DICON*0.5*(XIM-XITE(ISM) + XIP-XITE(ISP))
       DEI_TP = DDK    /RTP
     &        - DDK*THP/RTP**2 * RTP_TP
       DEI_UP =-DDK*THP/RTP**2 * RTP_UP
       DEI_RP =-DDK*THP/RTP**2 * RTP_RP
       DEI_TM = 0.
       DEI_UM = 0.
       DEI_RM = 0.
       DEI_RE =-DDK*THP/RTP**2 * RTP_RE
      ELSE
       DEI    = DDK*THM/RTM + DICON*0.5*(XIM-XITE(ISM) + XIP-XITE(ISP))
       DEI_TM = DDK    /RTM
     &        - DDK*THM/RTM**2 * RTM_TM
       DEI_UM =-DDK*THM/RTM**2 * RTM_UM
       DEI_RM =-DDK*THM/RTM**2 * RTM_RM
       DEI_TP = 0.
       DEI_UP = 0.
       DEI_RP = 0.
       DEI_RE =-DDK*THM/RTM**2 * RTM_RE
      ENDIF
C
C---- limit inner shear layer thickness to multiple of momentum thickness
      IF(DEI .GT. 0.4*THP) THEN
       DEI    = 0.4*THP
       DEI_TP = 0.4
       DEI_UP = 0.
       DEI_RP = 0.
       DEI_TM = 0.
       DEI_UM = 0.
       DEI_RM = 0.
       DEI_RE = 0.
      ENDIF
C
C---- limit inner shear layer thickness to multiple of momentum thickness
      IF(DEI .GT. 0.4*THM) THEN
       DEI    = 0.4*THM
       DEI_TP = 0.
       DEI_UP = 0.
       DEI_RP = 0.
       DEI_TM = 0.4
       DEI_UM = 0.
       DEI_RM = 0.
       DEI_RE = 0.
      ENDIF
C
C---- add on contributions from inner shear layer
      IF(USP .GT. USM) THEN
C
C----- inner layer is above zero shear line
       US    = USM
       US_RM = USM_RM
       US_UM = USM_UM
       US_TM = USM_TM
       US_DM = USM_DM
       US_RP = 0.
       US_UP = 0.
       US_TP = 0.
       US_DP = 0.
       US_RE = USM_RE
C
C           2      2
C----- add d U / dy  from inner layer
       UYY     = UYY     + (USP - USM)/DEI**2
       UYY_DEI =         - (USP - USM)/DEI**3 * 2.0
       UYY_USP = UYY_USP +  1.0       /DEI**2
       UYY_USM = UYY_USM        - 1.0 /DEI**2
C
       TYY = TYY + 4.0*CTCON*(USP-USM)**2/DEI**2
       TYY_DEI = - 8.0*CTCON*(USP-USM)**2/DEI**3
       TYY_USP =   8.0*CTCON*(USP-USM)   /DEI**2
       TYY_USM = - 8.0*CTCON*(USP-USM)   /DEI**2
C
      ELSE
C
C----- inner layer is below zero shear line
       US    = USP
       US_RM = 0.
       US_UM = 0.
       US_TM = 0.
       US_DM = 0.
       US_RP = USP_RP
       US_UP = USP_UP
       US_TP = USP_TP
       US_DP = USP_DP
       US_RE = USP_RE
C
C           2      2
C----- add d U / dy  from inner layer
       UYY     = UYY     + (USM - USP)/DEI**2
       UYY_DEI =         - (USM - USP)/DEI**3 * 2.0
       UYY_USP = UYY_USP        - 1.0 /DEI**2
       UYY_USM = UYY_USM +  1.0       /DEI**2
C
       TYY = TYY - 4.0*CTCON*(USM-USP)**2/DEI**2
       TYY_DEI =   8.0*CTCON*(USM-USP)**2/DEI**3
       TYY_USP =   8.0*CTCON*(USM-USP)   /DEI**2
       TYY_USM = - 8.0*CTCON*(USM-USP)   /DEI**2
C
      ENDIF
C
C                                       2      2   2    2
C---- set entrainment coefficient ( = -d Tau/dy / d U/dy  )  at centerline
      CE     = -CECON*TYY/UYY
      CE_TYY = -CECON    /UYY
      CE_UYY = -CE/UYY
C
      CE_DEI = CE_TYY*TYY_DEI + CE_UYY*UYY_DEI
      CE_CTM = CE_TYY*TYY_CTM
      CE_CTP = CE_TYY*TYY_CTP
      CE_DEM = CE_TYY*TYY_DEM + CE_UYY*UYY_DEM
      CE_DEP = CE_TYY*TYY_DEP + CE_UYY*UYY_DEP
      CE_USM = CE_TYY*TYY_USM + CE_UYY*UYY_USM
      CE_USP = CE_TYY*TYY_USP + CE_UYY*UYY_USP
C
ccc      tyysav(n) = tyy
ccc      uyysav(n) = uyy
ccc      ussav(n) = us
ccc      cesav(n) = ce
ccc      dussav(n) = usm - usp
c
C---- set entrainment term
      CEU    = (1.0-US)* CE
      CEU_SM = (1.0-US)* CE_CTM
      CEU_TM = (1.0-US)*(CE_DEM*DEM_TM + CE_USM*USM_TM) - CE*US_TM
      CEU_DM = (1.0-US)*(CE_DEM*DEM_DM + CE_USM*USM_DM) - CE*US_DM
      CEU_UM = (1.0-US)*(CE_DEM*DEM_UM + CE_USM*USM_UM) - CE*US_UM
      CEU_RM = (1.0-US)*(                CE_USM*USM_RM) - CE*US_RM
      CEU_SP = (1.0-US)* CE_CTP
      CEU_TP = (1.0-US)*(CE_DEP*DEP_TP + CE_USP*USP_TP) - CE*US_TP
      CEU_DP = (1.0-US)*(CE_DEP*DEP_DP + CE_USP*USP_DP) - CE*US_DP
      CEU_UP = (1.0-US)*(CE_DEP*DEP_UP + CE_USP*USP_UP) - CE*US_UP
      CEU_RP = (1.0-US)*(                CE_USP*USP_RP) - CE*US_RP
      CEU_RE = (1.0-US)*(CE_USM*USM_RE + CE_USP*USP_RE) - CE*US_RE
C
      CEU_TM = (1.0-US)*(CE_DEI*DEI_TM)  +  CEU_TM
      CEU_UM = (1.0-US)*(CE_DEI*DEI_UM)  +  CEU_UM
      CEU_RM = (1.0-US)*(CE_DEI*DEI_RM)  +  CEU_RM
      CEU_TP = (1.0-US)*(CE_DEI*DEI_TP)  +  CEU_TP
      CEU_UP = (1.0-US)*(CE_DEI*DEI_UP)  +  CEU_UP
      CEU_RP = (1.0-US)*(CE_DEI*DEI_RP)  +  CEU_RP
      CEU_RE = (1.0-US)*(CE_DEI*DEI_RE)  +  CEU_RE
C
C
      VCENI = -CEU*0.5*(UEM+UEP)
C
C---- set velocity at zero shear line and inner layer dissipation coefficient
      CDCON = 2.0*CTCON*0.78**2
      IF(USP .GT. USM) THEN
C
       CDP     =  CDCON*(USP-USM)**3
       CDP_USP =  CDCON*(USP-USM)**2 * 3.0
       CDP_USM = -CDCON*(USP-USM)**2 * 3.0
C
       CDP_TM = CDP_USM*USM_TM
       CDP_DM = CDP_USM*USM_DM
       CDP_UM = CDP_USM*USM_UM
       CDP_RM = CDP_USM*USM_RM
       CDP_TP = CDP_USP*USP_TP
       CDP_DP = CDP_USP*USP_DP
       CDP_UP = CDP_USP*USP_UP
       CDP_RP = CDP_USP*USP_RP
       CDP_RE = CDP_USM*USM_RE + CDP_USP*USP_RE
C
       CDM    = 0.
       CDM_TM = 0.
       CDM_DM = 0.
       CDM_UM = 0.
       CDM_RM = 0.
       CDM_TP = 0.
       CDM_DP = 0.
       CDM_UP = 0.
       CDM_RP = 0.
       CDM_RE = 0.
C
      ELSE
C
       CDM     =  CDCON*(USM-USP)**3
       CDM_USP = -CDCON*(USM-USP)**2 * 3.0
       CDM_USM =  CDCON*(USM-USP)**2 * 3.0
C
       CDM_TM = CDM_USM*USM_TM
       CDM_DM = CDM_USM*USM_DM
       CDM_UM = CDM_USM*USM_UM
       CDM_RM = CDM_USM*USM_RM
       CDM_TP = CDM_USP*USP_TP
       CDM_DP = CDM_USP*USP_DP
       CDM_UP = CDM_USP*USP_UP
       CDM_RP = CDM_USP*USP_RP
       CDM_RE = CDM_USM*USM_RE + CDM_USP*USP_RE
C
       CDP    = 0.
       CDP_TM = 0.
       CDP_DM = 0.
       CDP_UM = 0.
       CDP_RM = 0.
       CDP_TP = 0.
       CDP_DP = 0.
       CDP_UP = 0.
       CDP_RP = 0.
       CDP_RE = 0.
C
      ENDIF
C
C---- set column indices for BL variables
      LXM = 2
      LXP = 1
C
      LCM = 5
      LTM = 6
      LDM = 7
      LUM = 8
C
      LCP = 1
      LTP = 2
      LDP = 3
      LUP = 4
C
C
C**** add corrections to lag equation
      KM = 5
      KP = 1
C
C---- inner shear layer correction for CtauEQ
      DZCON = 4.0*SQRT(CTCON)
C
      VVREZ(KM,N)   = VVREZ(KM,N)   - DZCON*(USM    - US   )*DXM
      VVR(KM,N)     = VVR(KM,N)     + DZCON*(USM_RE - US_RE)*DXM
      VVX1(KM,LXM,N)= VVX1(KM,LXM,N)- DZCON*(USM    - US   )
      VVX2(KM,LXM,N)= VVX2(KM,LXM,N)+ DZCON*(USM    - US   )
C
      VV2(KM,LTM,N) = VV2(KM,LTM,N) + DZCON*(USM_TM - US_TM)*DXM
      VV2(KM,LDM,N) = VV2(KM,LDM,N) + DZCON*(USM_DM - US_DM)*DXM
      VV2(KM,LUM,N) = VV2(KM,LUM,N) + DZCON*(USM_UM - US_UM)*DXM
      VVR2(KM,LXM,N)= VVR2(KM,LXM,N)+ DZCON*(USM_RM - US_RM)*DXM
C
      VV2(KM,LTP,N) = VV2(KM,LTP,N) + DZCON*(       - US_TP)*DXM
      VV2(KM,LDP,N) = VV2(KM,LDP,N) + DZCON*(       - US_DP)*DXM
      VV2(KM,LUP,N) = VV2(KM,LUP,N) + DZCON*(       - US_UP)*DXM
      VVR2(KM,LXP,N)= VVR2(KM,LXP,N)+ DZCON*(       - US_RP)*DXM
C
C
      VVREZ(KP,N)   = VVREZ(KP,N)   - DZCON*(USP    - US   )*DXP
      VVR(KP,N)     = VVR(KP,N)     + DZCON*(USP_RE - US_RE)*DXP
      VVX1(KP,LXP,N)= VVX1(KP,LXP,N)- DZCON*(USP    - US   )
      VVX2(KP,LXP,N)= VVX2(KP,LXP,N)+ DZCON*(USP    - US   )
C
      VV2(KP,LTM,N) = VV2(KP,LTM,N) + DZCON*(       - US_TM)*DXP
      VV2(KP,LDM,N) = VV2(KP,LDM,N) + DZCON*(       - US_DM)*DXP
      VV2(KP,LUM,N) = VV2(KP,LUM,N) + DZCON*(       - US_UM)*DXP
      VVR2(KP,LXM,N)= VVR2(KP,LXM,N)+ DZCON*(       - US_RM)*DXP
C
      VV2(KP,LTP,N) = VV2(KP,LTP,N) + DZCON*(USP_TP - US_TP)*DXP
      VV2(KP,LDP,N) = VV2(KP,LDP,N) + DZCON*(USP_DP - US_DP)*DXP
      VV2(KP,LUP,N) = VV2(KP,LUP,N) + DZCON*(USP_UP - US_UP)*DXP
      VVR2(KP,LXP,N)= VVR2(KP,LXP,N)+ DZCON*(USP_RP - US_RP)*DXP
C
C
C
C---- add cross-layer turb. KE diffusion term (drives Ctau symmetric)
      SCCON = 0.5*SCC
ccc      SCCON = 0.2*SCC
C
      VVREZ(KM,N) = VVREZ(KM,N) - SCCON*(CTP-CTM)*DXM
      Z_CTM = -SCCON*DXM
      Z_CTP =  SCCON*DXM
      VV2(KM,LCM,N) = VV2(KM,LCM,N) + Z_CTM
      VV2(KM,LCP,N) = VV2(KM,LCP,N) + Z_CTP
      VVX1(KM,LXM,N) = VVX1(KM,LXM,N) - SCCON*(CTP-CTM)
      VVX2(KM,LXM,N) = VVX2(KM,LXM,N) + SCCON*(CTP-CTM)
C
      VVREZ(KP,N) = VVREZ(KP,N) - SCCON*(CTM-CTP)*DXP
      Z_CTP = -SCCON*DXP
      Z_CTM =  SCCON*DXP
      VV2(KP,LCP,N) = VV2(KP,LCP,N) + Z_CTP
      VV2(KP,LCM,N) = VV2(KP,LCM,N) + Z_CTM
      VVX1(KP,LXP,N) = VVX1(KP,LXP,N) - SCCON*(CTM-CTP)
      VVX2(KP,LXP,N) = VVX2(KP,LXP,N) + SCCON*(CTM-CTP)
C
C
C---- additional CtauEQ on thinner side due to increased mixing length
C
c      IF(THM.LT.THP) THEN
cC
c       VVREZ(KM,N) = VVREZ(KM,N) - SCC*CQM*0.25*(SQRT(THP/THM)-1.0)*DXM
c       Z_CQM =  SCC    *0.25*(SQRT(THP/THM)-1.0)*DXM
c       Z_TM  = -SCC*CQM*0.25*(SQRT(THP/THM)/THM )*DXM * 0.5
c       Z_TP  =  SCC*CQM*0.25*(SQRT(THP/THM)/THP )*DXM * 0.5
cC
c       VV2(KM,LTM,N) = VV2(KM,LTM,N) + Z_CQM*CQM_TM + Z_TM
c       VV2(KM,LDM,N) = VV2(KM,LDM,N) + Z_CQM*CQM_DM
c       VV2(KM,LUM,N) = VV2(KM,LUM,N) + Z_CQM*CQM_UM
c       VVR(KM,N)     = VVR(KM,N)     + Z_CQM*CQM_RE
c       VV2(KM,LTP,N) = VV2(KM,LTP,N)                + Z_TP
cC
c      ELSE
cC
c       VVREZ(KP,N) = VVREZ(KP,N) - SCC*CQP*0.25*(SQRT(THM/THP)-1.0)*DXP
c       Z_CQP =  SCC    *0.25*(SQRT(THM/THP)-1.0)*DXP
c       Z_TP  = -SCC*CQP*0.25*(SQRT(THM/THP)/THP )*DXP * 0.5
c       Z_TM  =  SCC*CQP*0.25*(SQRT(THM/THP)/THM )*DXP * 0.5
cC
c       VV2(KP,LTP,N) = VV2(KP,LTP,N) + Z_CQP*CQP_TP + Z_TP
c       VV2(KP,LDP,N) = VV2(KP,LDP,N) + Z_CQP*CQP_DP
c       VV2(KP,LUP,N) = VV2(KP,LUP,N) + Z_CQP*CQP_UP
c       VVR(KP,N)     = VVR(KP,N)     + Z_CQP*CQP_RE
c       VV2(KP,LTM,N) = VV2(KP,LTM,N)                + Z_TM
cC
c      ENDIF
C
C
C**** add/subtract entrainment term to momentum equations
      KM = 6
      KP = 2
C
      XOTM = DXM/THM
      XOTP = DXP/THP
C
      DREZ           =                - CEU   *XOTM
      VVR(KM,N)      = VVR(KM,N)      - CEU_RE*XOTM
      VVX1(KM,LXM,N) = VVX1(KM,LXM,N)               + CEU /THM
      VVX2(KM,LXM,N) = VVX2(KM,LXM,N)               - CEU /THM
C
      VV2(KM,LCM,N)  = VV2(KM,LCM,N)  - CEU_SM*XOTM
      VV2(KM,LTM,N)  = VV2(KM,LTM,N)  - CEU_TM*XOTM - DREZ/THM
      VV2(KM,LDM,N)  = VV2(KM,LDM,N)  - CEU_DM*XOTM
      VV2(KM,LUM,N)  = VV2(KM,LUM,N)  - CEU_UM*XOTM
      VVR2(KM,LXM,N) = VVR2(KM,LXM,N) - CEU_RM*XOTM
C
      VV2(KM,LCP,N)  = VV2(KM,LCP,N)  - CEU_SP*XOTM
      VV2(KM,LTP,N)  = VV2(KM,LTP,N)  - CEU_TP*XOTM
      VV2(KM,LDP,N)  = VV2(KM,LDP,N)  - CEU_DP*XOTM
      VV2(KM,LUP,N)  = VV2(KM,LUP,N)  - CEU_UP*XOTM
      VVR2(KM,LXP,N) = VVR2(KM,LXP,N) - CEU_RP*XOTM
C
      VVREZ(KM,N)    = VVREZ(KM,N)    - DREZ
C
C
      DREZ           =                  CEU   *XOTP
      VVR(KP,N)      = VVR(KP,N)      + CEU_RE*XOTP
      VVX1(KP,LXP,N) = VVX1(KP,LXP,N)               - CEU /THP
      VVX2(KP,LXP,N) = VVX2(KP,LXP,N)               + CEU /THP
C
      VV2(KP,LCM,N)  = VV2(KP,LCM,N)  + CEU_SM*XOTP
      VV2(KP,LTM,N)  = VV2(KP,LTM,N)  + CEU_TM*XOTP
      VV2(KP,LDM,N)  = VV2(KP,LDM,N)  + CEU_DM*XOTP
      VV2(KP,LUM,N)  = VV2(KP,LUM,N)  + CEU_UM*XOTP
      VVR2(KP,LXM,N) = VVR2(KP,LXM,N) + CEU_RM*XOTP
C
      VV2(KP,LCP,N)  = VV2(KP,LCP,N)  + CEU_SP*XOTP
      VV2(KP,LTP,N)  = VV2(KP,LTP,N)  + CEU_TP*XOTP - DREZ/THP
      VV2(KP,LDP,N)  = VV2(KP,LDP,N)  + CEU_DP*XOTP
      VV2(KP,LUP,N)  = VV2(KP,LUP,N)  + CEU_UP*XOTP
      VVR2(KP,LXP,N) = VVR2(KP,LXP,N) + CEU_RP*XOTP
C
      VVREZ(KP,N)    = VVREZ(KP,N)    - DREZ
C
C
C**** add/subtract entrainment and dissipation terms to KE equation
      KM = 7
      KP = 3
C
C
c      FCEM = (1.0+US)/HSM - 1.0
cC
c      DREZM  =  (-CEU*FCEM - CDM*2.0/HSM)*XOTM
c      Z_XOTM =  (-CEU*FCEM - CDM*2.0/HSM)
c      Z_CEU  =  (-    FCEM              )*XOTM
c      Z_CDM  =  (          -     2.0/HSM)*XOTM
c      Z_US   =  (-CEU/HSM               )*XOTM
c      Z_HSM  = -(-CEU*(1.0+US) - CDM*2.0)*XOTM/HSM**2
cC
c      VVREZ(KM,N)   = VVREZ(KM,N)   - DREZM
c      VVX1(KM,LXM,N)= VVX1(KM,LXM,N)               - Z_XOTM/THM
c      VVX2(KM,LXM,N)= VVX2(KM,LXM,N)               + Z_XOTM/THM
c      VV2(KM,LTM,N) = VV2(KM,LTM,N)                + Z_XOTM*(-XOTM/THM)
c      VVR(KM,N)     = VVR(KM,N)     + Z_CEU*CEU_RE + Z_US*US_RE
c      VV2(KM,LCM,N) = VV2(KM,LCM,N) + Z_CEU*CEU_SM
c      VV2(KM,LTM,N) = VV2(KM,LTM,N) + Z_CEU*CEU_TM + Z_US*US_TM
c      VV2(KM,LDM,N) = VV2(KM,LDM,N) + Z_CEU*CEU_DM + Z_US*US_DM
c      VV2(KM,LUM,N) = VV2(KM,LUM,N) + Z_CEU*CEU_UM + Z_US*US_UM
c      VV2(KM,LCP,N) = VV2(KM,LCP,N) + Z_CEU*CEU_SP
c      VV2(KM,LTP,N) = VV2(KM,LTP,N) + Z_CEU*CEU_TP + Z_US*US_TP
c      VV2(KM,LDP,N) = VV2(KM,LDP,N) + Z_CEU*CEU_DP + Z_US*US_DP
c      VV2(KM,LUP,N) = VV2(KM,LUP,N) + Z_CEU*CEU_UP + Z_US*US_UP
cC
c      VVR(KM,N)     = VVR(KM,N)     + Z_CDM*CDM_RE + Z_HSM*HSM_RE
c      VV2(KM,LTM,N) = VV2(KM,LTM,N) + Z_CDM*CDM_TM + Z_HSM*HSM_TM
c      VV2(KM,LDM,N) = VV2(KM,LDM,N) + Z_CDM*CDM_DM + Z_HSM*HSM_DM
c      VV2(KM,LUM,N) = VV2(KM,LUM,N) + Z_CDM*CDM_UM + Z_HSM*HSM_UM
c      VV2(KM,LTP,N) = VV2(KM,LTP,N) + Z_CDM*CDM_TP
c      VV2(KM,LDP,N) = VV2(KM,LDP,N) + Z_CDM*CDM_DP
c      VV2(KM,LUP,N) = VV2(KM,LUP,N) + Z_CDM*CDM_UP
cC
cC
c      FCEP = (1.0+US)/HSP - 1.0
cC
c      DREZP  =  (-CEU*FCEP + CDP*2.0/HSP)*XOTP
c      Z_XOTP =  (-CEU*FCEP + CDP*2.0/HSP)
c      Z_CEU  =  (-    FCEP              )*XOTP
c      Z_CDP  =  (                2.0/HSP)*XOTP
c     Z_US   =  (-CEU/HSP               )*XOTP
c      Z_HSP  = -(-CEU*(1.0+US) + CDP*2.0)*XOTP/HSP**2
cC
c      VVREZ(KP,N)   = VVREZ(KP,N)   + DREZP
c      VVX1(KP,LXP,N)= VVX1(KP,LXP,N)               + Z_XOTP/THP
c      VVX2(KP,LXP,N)= VVX2(KP,LXP,N)               - Z_XOTP/THP
c      VV2(KP,LTP,N) = VV2(KP,LTP,N)                - Z_XOTP*(-XOTP/THP)
c      VVR(KP,N)     = VVR(KP,N)     - Z_CEU*CEU_RE - Z_US*US_RE
c      VV2(KP,LCM,N) = VV2(KP,LCM,N) - Z_CEU*CEU_SM
c      VV2(KP,LTM,N) = VV2(KP,LTM,N) - Z_CEU*CEU_TM - Z_US*US_TM
c      VV2(KP,LDM,N) = VV2(KP,LDM,N) - Z_CEU*CEU_DM - Z_US*US_DM
c      VV2(KP,LUM,N) = VV2(KP,LUM,N) - Z_CEU*CEU_UM - Z_US*US_UM
c      VV2(KP,LCP,N) = VV2(KP,LCP,N) - Z_CEU*CEU_SP
c      VV2(KP,LTP,N) = VV2(KP,LTP,N) - Z_CEU*CEU_TP - Z_US*US_TP
c      VV2(KP,LDP,N) = VV2(KP,LDP,N) - Z_CEU*CEU_DP - Z_US*US_DP
c      VV2(KP,LUP,N) = VV2(KP,LUP,N) - Z_CEU*CEU_UP - Z_US*US_UP
cC
c      VVR(KP,N)     = VVR(KP,N)     - Z_CDP*CDP_RE - Z_HSP*HSP_RE
c      VV2(KP,LTM,N) = VV2(KP,LTM,N) - Z_CDP*CDP_TM
c      VV2(KP,LDM,N) = VV2(KP,LDM,N) - Z_CDP*CDP_DM
c      VV2(KP,LUM,N) = VV2(KP,LUM,N) - Z_CDP*CDP_UM
c      VV2(KP,LTP,N) = VV2(KP,LTP,N) - Z_CDP*CDP_TP - Z_HSP*HSP_TP
c      VV2(KP,LDP,N) = VV2(KP,LDP,N) - Z_CDP*CDP_DP - Z_HSP*HSP_DP
c      VV2(KP,LUP,N) = VV2(KP,LUP,N) - Z_CDP*CDP_UP - Z_HSP*HSP_UP
C
C
C---- add up the two KE equations
      REZP = -VVREZ(KP,N)
      REZM = -VVREZ(KM,N)
C
      VVREZ(KP,N) = -(HSP*THP*REZP          + HSM*THM*REZM)
      Z_HSP       =       THP*REZP
      Z_HSM       =                               THM*REZM
      Z_THP       =   HSP    *REZP
      Z_THM       =                           HSM    *REZM
C
      VVR(KP,N)     = HSP*THP*VVR(KP,N)     + HSM*THM*VVR(KM,N)
     &              + Z_HSP*HSP_RE          + Z_HSM*HSM_RE
      VVXF(KP,LXM,N)= HSP*THP*VVXF(KP,LXM,N)+ HSM*THM*VVXF(KM,LXM,N)
      VVXF(KP,LXP,N)= HSP*THP*VVXF(KP,LXP,N)+ HSM*THM*VVXF(KM,LXP,N)
C
      VV1(KP,LCM,N) = HSP*THP*VV1(KP,LCM,N) + HSM*THM*VV1(KM,LCM,N)
      VV1(KP,LTM,N) = HSP*THP*VV1(KP,LTM,N) + HSM*THM*VV1(KM,LTM,N)
      VV1(KP,LDM,N) = HSP*THP*VV1(KP,LDM,N) + HSM*THM*VV1(KM,LDM,N)
      VV1(KP,LUM,N) = HSP*THP*VV1(KP,LUM,N) + HSM*THM*VV1(KM,LUM,N)
      VVX1(KP,LXM,N)= HSP*THP*VVX1(KP,LXM,N)+ HSM*THM*VVX1(KM,LXM,N)
      VVN1(KP,LXM,N)= HSP*THP*VVN1(KP,LXM,N)+ HSM*THM*VVN1(KM,LXM,N)
      VVR1(KP,LXM,N)= HSP*THP*VVR1(KP,LXM,N)+ HSM*THM*VVR1(KM,LXM,N)

      VV1(KP,LCP,N) = HSP*THP*VV1(KP,LCP,N) + HSM*THM*VV1(KM,LCP,N)
      VV1(KP,LTP,N) = HSP*THP*VV1(KP,LTP,N) + HSM*THM*VV1(KM,LTP,N)
      VV1(KP,LDP,N) = HSP*THP*VV1(KP,LDP,N) + HSM*THM*VV1(KM,LDP,N)
      VV1(KP,LUP,N) = HSP*THP*VV1(KP,LUP,N) + HSM*THM*VV1(KM,LUP,N)
      VVX1(KP,LXP,N)= HSP*THP*VVX1(KP,LXP,N)+ HSM*THM*VVX1(KM,LXP,N)
      VVN1(KP,LXP,N)= HSP*THP*VVN1(KP,LXP,N)+ HSM*THM*VVN1(KM,LXP,N)
      VVR1(KP,LXP,N)= HSP*THP*VVR1(KP,LXP,N)+ HSM*THM*VVR1(KM,LXP,N)
C
      VV2(KP,LCM,N) = HSP*THP*VV2(KP,LCM,N) + HSM*THM*VV2(KM,LCM,N)
      VV2(KP,LTM,N) = HSP*THP*VV2(KP,LTM,N) + HSM*THM*VV2(KM,LTM,N)
     &                                      + Z_THM
     &                                      + Z_HSM*HSM_TM
      VV2(KP,LDM,N) = HSP*THP*VV2(KP,LDM,N) + HSM*THM*VV2(KM,LDM,N)
     &                                      + Z_HSM*HSM_DM
      VV2(KP,LUM,N) = HSP*THP*VV2(KP,LUM,N) + HSM*THM*VV2(KM,LUM,N)
     &                                      + Z_HSM*HSM_UM
      VVX2(KP,LXM,N)= HSP*THP*VVX2(KP,LXM,N)+ HSM*THM*VVX2(KM,LXM,N)
      VVN2(KP,LXM,N)= HSP*THP*VVN2(KP,LXM,N)+ HSM*THM*VVN2(KM,LXM,N)
      VVR2(KP,LXM,N)= HSP*THP*VVR2(KP,LXM,N)+ HSM*THM*VVR2(KM,LXM,N)
     &                                      + Z_HSM*HSM_RM
C
      VV2(KP,LCP,N) = HSP*THP*VV2(KP,LCP,N) + HSM*THM*VV2(KM,LCP,N)
      VV2(KP,LTP,N) = HSP*THP*VV2(KP,LTP,N) + HSM*THM*VV2(KM,LTP,N)
     &              + Z_THP
     &              + Z_HSP*HSP_TP
      VV2(KP,LDP,N) = HSP*THP*VV2(KP,LDP,N) + HSM*THM*VV2(KM,LDP,N)
     &              + Z_HSP*HSP_DP
      VV2(KP,LUP,N) = HSP*THP*VV2(KP,LUP,N) + HSM*THM*VV2(KM,LUP,N)
     &              + Z_HSP*HSP_UP
      VVX2(KP,LXP,N)= HSP*THP*VVX2(KP,LXP,N)+ HSM*THM*VVX2(KM,LXP,N)
      VVN2(KP,LXP,N)= HSP*THP*VVN2(KP,LXP,N)+ HSM*THM*VVN2(KM,LXP,N)
      VVR2(KP,LXP,N)= HSP*THP*VVR2(KP,LXP,N)+ HSM*THM*VVR2(KM,LXP,N)
     &              + Z_HSP*HSP_RP
C
C
C---- set evolution equation for wake centerline velocity mismatch
      UCON = 0.10
C
      XOTM = DXM/THM
      XOTP = DXP/THP
C
      XOT = XOTP + XOTM
C
      VVREZ(KM,N)   =-(DUP    - DUM    + UCON*(USP    - USM   )*XOT)
      VVR(KM,N)     =  DUP_RE - DUM_RE
      VVXF(KM,LXM,N)= 0.
      VVXF(KM,LXP,N)= 0.
C
      VV1(KM,LCM,N) = 0.0
      VV1(KM,LTM,N) =         - DUM_T1
      VV1(KM,LDM,N) =         - DUM_D1
      VV1(KM,LUM,N) =         - DUM_U1
      VVR1(KM,LXM,N)=         - DUM_R1
      VVN1(KM,LXM,N)= 0.0
      VVX1(KM,LXM,N)=                  - UCON*(USP    - USM   )/THM
C
      VV1(KM,LCP,N) = 0.0
      VV1(KM,LTP,N) =  DUP_T1
      VV1(KM,LDP,N) =  DUP_D1
      VV1(KM,LUP,N) =  DUP_U1
      VVR1(KM,LXP,N)=  DUP_R1
      VVN1(KM,LXP,N)= 0.0
      VVX1(KM,LXP,N)=                  - UCON*(USP    - USM   )/THP
C
      VV2(KM,LCM,N) = 0.0
      VV2(KM,LTM,N) =         - DUM_T2 + UCON*(       - USM_TM)*XOT
     &                                 - UCON*(USP    - USM   )*XOTM/THM
      VV2(KM,LDM,N) =         - DUM_D2 + UCON*(       - USM_DM)*XOT
      VV2(KM,LUM,N) =         - DUM_U2 + UCON*(       - USM_UM)*XOT
      VVR2(KM,LXM,N)=         - DUM_R2 + UCON*(       - USM_RM)*XOT
      VVN2(KM,LXM,N)= 0.0
      VVX2(KM,LXM,N)=                    UCON*(USP    - USM   )/THM
C
      VV2(KM,LCP,N) = 0.0
      VV2(KM,LTP,N) =  DUP_T2          + UCON*(USP_TP         )*XOT
     &                                 - UCON*(USP    - USM   )*XOTP/THP
      VV2(KM,LDP,N) =  DUP_D2          + UCON*(USP_DP         )*XOT
      VV2(KM,LUP,N) =  DUP_U2          + UCON*(USP_UP         )*XOT
      VVR2(KM,LXP,N)=  DUP_R2          + UCON*(USP_RP         )*XOT
      VVN2(KM,LXP,N)= 0.0
      VVX2(KM,LXP,N)=                    UCON*(USP    - USM   )/THP
C
C
      RETURN
      END ! BLCENT


      SUBROUTINE SETKIN(XSI,AMI,CTI,UEI,THI,DSI,DSWAKI,UNI,RHI)
C............................................
C     Calculates secondary variables at "2"
C...........................................
      IMPLICIT REAL (A-H,M,O-Z)
      INCLUDE 'MBL.INC'
C
C---- set primary BL variables from current values
      X2 = XSI
      AMPL2 = AMI
      S2 = CTI
      U2 = UEI
      T2 = THI
      D2 = DSI
      DW2 = DSWAKI
      UN2 = UNI
      R2 = RHI
C
C---- set edge Mach number ** 2
      M2    = U2*U2 / (GM1BL*(HSTBL-0.5*U2*U2))
      TR2   = 1.0 + 0.5*GM1BL*M2
      M2_U2 = 2.0*M2*TR2/U2
C
C---- set edge static density (isentropic relation)
ccc      R2    = RSTBL*TR2**(-1.0/GM1BL)
ccc      R2_U2 = -R2/TR2 * 0.5*M2_U2
C
C---- set shape parameter
      H2    = D2/T2
      H2_D2 = 1.0/T2
      H2_T2 = -H2/T2
C
C---- set edge static enthalpy and molecular viscosity
      HEDGE = HSTBL - 0.5*U2*U2
      V2 = SQRT((HEDGE/HSTBL)**3) * (HSTBL+HVISBL)/(HEDGE+HVISBL)/REYBL
      V2_U2 = V2*U2*(1.0/(HEDGE+HVISBL)-1.5/HEDGE)
      V2_RE = -V2/REYBL
C
C---- set kinematic shape parameter
      CALL HKIN( H2, M2, GAMBL, HK2, HK2_H2, HK2_M2 )
C
      HK2_U2 =                HK2_M2*M2_U2
      HK2_T2 = HK2_H2*H2_T2
      HK2_D2 = HK2_H2*H2_D2
C
C---- set momentum thickness Reynolds number
      RT2    = R2*U2*T2/V2
      RT2_R2 =    U2*T2/V2
      RT2_U2 = R2   *T2/V2 - RT2/V2 * V2_U2
      RT2_T2 = R2*U2   /V2
      RT2_RE =             - RT2/V2 * V2_RE
C
      RETURN
      END ! SETKIN


      SUBROUTINE SETVAR(IS)
C.........................................
C     Sets up calls to BLVAR for side IS
C.........................................
      IMPLICIT REAL (A-H,M,O-Z)
      INCLUDE 'MBL.INC'
C
C---- recall all primary and secondary variables for deck IS
      DO 10 NC=1, NCOM
        COM1(NC) = V1SAV(NC,IS)
        COM2(NC) = V2SAV(NC,IS)
   10 CONTINUE
C
      DO 15 NC=1, NCOMA
        COMA(NC) = VASAV(NC,IS)
   15 CONTINUE
C
C---- clear DS2, DS2_T2, ...   (triggered by CF2 = 0)
      CF2 = 0.0
      CALL DSTSET
C
C---- calculate tertiary BL variables and their sensitivities
      ISIDE = IS
      IF(WAKE(IS)) THEN
       CALL BLVAR(3)
      ELSE IF(TURB(IS).OR.TRAN(IS)) THEN
       CALL BLVAR(2)
      ELSE
       CALL BLVAR(1)
      ENDIF
C
C---- for the similarity station, "1" and "2" variables are the same
      IF(SIMI(IS)) THEN
       DO 20 NC=1, NCOM
         COM1(NC) = COM2(NC)
   20  CONTINUE
C
       CFM    = CF2
       CFM_RE = CF2_RE
       CFM_R1 = 0.
       CFM_U1 = 0.
       CFM_T1 = 0.
       CFM_D1 = 0.
       CFM_R2 = CF2_R2
       CFM_U2 = CF2_U2
       CFM_T2 = CF2_T2
       CFM_D2 = CF2_D2
      ENDIF
C
C---- save all variables
      DO 30 NC=1, NCOM
        V1SAV(NC,IS) = COM1(NC)
        V2SAV(NC,IS) = COM2(NC)
   30 CONTINUE
C
      DO 40 NC=1, NCOMA
        VASAV(NC,IS) = COMA(NC)
   40 CONTINUE
C
C---- set wall shear for possible use in calling routine
      TAUWI  = 0.5*R2*U2*U2*CF2
      TAU_RHI = 0.5*R2*U2*U2*CF2_R2 + 0.5   *U2*U2*CF2
      TAU_UEI = 0.5*R2*U2*U2*CF2_U2 +     R2*U2   *CF2
      TAU_THI = 0.5*R2*U2*U2*CF2_T2
      TAU_DSI = 0.5*R2*U2*U2*CF2_D2
      TAU_REY = 0.5*R2*U2*U2*CF2_RE
C
      RETURN
      END ! SETVAR


      SUBROUTINE BLVAR(ITYP)
C...................................................
C     Calculates all tertiary "2" variables from
C     the primary "2" variables X2, U2, T2, D2, S2,
C     and the secondary "2" variables HK2, RT2, ...
C     Also calculates the sensitivities of the
C     secondary variables wrt the primary variables.
C
C      ITYP = 1 :  laminar
C      ITYP = 2 :  turbulent
C      ITYP = 3 :  turbulent wake
C....................................................
      IMPLICIT REAL (A-H,M,O-Z)
      INCLUDE 'MBL.INC'
C
C---- limit wall Hk transients for robustness
      IF(ITYP.NE.3) THEN
       HK2 = MAX(HK2,1.05)
      ENDIF
C
C---- set KE thickness shape parameter from  H - H*  correlations
      IF(ITYP.EQ.1) THEN
       CALL HSL( HK2, RT2, M2, HS2, HS2_HK2, HS2_RT2, HS2_M2 )
      ELSE
       CALL HST( HK2, RT2, M2, HS2, HS2_HK2, HS2_RT2, HS2_M2 )
      ENDIF
C
      HS2_R2 =                  HS2_RT2*RT2_R2
      HS2_U2 = HS2_HK2*HK2_U2 + HS2_RT2*RT2_U2 + HS2_M2*M2_U2
      HS2_T2 = HS2_HK2*HK2_T2 + HS2_RT2*RT2_T2
      HS2_D2 = HS2_HK2*HK2_D2
      HS2_RE =                  HS2_RT2*RT2_RE
C
C---- density thickness shape parameter     ( H** )
c      CALL HCT( HK2, M2, GAMBL, HC2, HC2_HK2, HC2_M2 )
c      HC2_R2 = 0.
c      HC2_U2 = HC2_HK2*HK2_U2 + HC2_M2*M2_U2
c      HC2_T2 = HC2_HK2*HK2_T2
c      HC2_D2 = HC2_HK2*HK2_D2
c      HC2_RE = 0.
C
C---- exact relation for adiabatic flows
      HC2     = 0.5*GM1BL*M2*HS2
      HC2_M2  = 0.5*GM1BL   *HS2
      HC2_HS2 = 0.5*GM1BL*M2
C
      HC2_R2 = HC2_HS2*HS2_R2
      HC2_T2 = HC2_HS2*HS2_T2
      HC2_D2 = HC2_HS2*HS2_D2
      HC2_U2 = HC2_HS2*HS2_U2 + HC2_M2*M2_U2
      HC2_RE = HC2_HS2*HS2_RE
C
C---- BL thickness (Delta) from simplified Green's correlation
      IF(HK2 .LE. 1.0) THEN
       DE2    = 15.0*T2
       DE2_U2 =  0.0
       DE2_T2 = 15.0
       DE2_D2 =  0.0
      ELSE
       DE2     = (3.15 + 1.72/(HK2-1.0)   )*T2  +  D2
       DE2_HK2 = (     - 1.72/(HK2-1.0)**2)*T2
C
       DE2_U2 = DE2_HK2*HK2_U2
       DE2_T2 = DE2_HK2*HK2_T2 + (3.15 + 1.72/(HK2-1.0))
       DE2_D2 = DE2_HK2*HK2_D2 + 1.0
C
cccc       IF(DE2 .GT. 15.0*T2) THEN
       IF(DE2 .GT. 15.0*T2 .AND. (HK2 .GT. 4.0 .OR. ITYP.EQ.3)) THEN
        DE2    = 15.0*T2
        DE2_U2 =  0.0
        DE2_T2 = 15.0
        DE2_D2 =  0.0
       ENDIF
      ENDIF
C
C
C==============================
ccc      RTMIN = 250.0
      RTMIN = 1.0
C==============================
C
C---- define stuff for midpoint CF
      HKA = 0.5*(HK1 + HK2)
      RTA = 0.5*(RT1 + RT2)
      MA  = 0.5*(M1  + M2 )
C
C---- skin friction coefficient  (zero in wake)
      IF(ITYP.EQ.3) THEN
C
       CF2     = 0.
       CF2_HK2 = 0.
       CF2_RT2 = 0.
       CF2_M2  = 0.
C
       CFM     = 0.
       CFM_HKA = 0.
       CFM_RTA = 0.
       CFM_MA  = 0.
C
      ELSE IF(ITYP.EQ.1) THEN
C
       CALL CFL( HK2, RT2, M2, CF2, CF2_HK2, CF2_RT2, CF2_M2 )
       CALL CFL( HKA, RTA, MA, CFM, CFM_HKA, CFM_RTA, CFM_MA )
C
      ELSE
C
C----- use laminar CF if larger than turbulent CF  (at very low Rtheta)
       CALL CFT( HK2, RT2, M2, GAMBL, CF2, CF2_HK2, CF2_RT2, CF2_M2 )
       CALL CFL(HK2, RT2, M2,       CF2l, CF2l_HK2, CF2l_RT2, CF2l_M2)
C
       IF(CF2l.GT.CF2) THEN
ccc         write(*,*) 'Using laminar Cf:', cf2, cf2l
         CF2     = CF2l
         CF2_HK2 = CF2l_HK2
         CF2_RT2 = CF2l_RT2
         CF2_M2  = CF2l_M2
       ENDIF
C
C
       CALL CFT( HKA, RTA, MA, GAMBL, CFM, CFM_HKA, CFM_RTA, CFM_MA )
       CALL CFL(HKA, RTA, MA,       CFMl, CFMl_HKA, CFMl_RTA, CFMl_MA)
c
       IF(CFMl.GT.CFM) THEN
         CFM     = CFMl
         CFM_HKA = CFMl_HKA
         CFM_RTA = CFMl_RTA
         CFM_MA  = CFMl_MA
       ENDIF
C
      ENDIF
C
      CF2_R2 =                  CF2_RT2*RT2_R2
      CF2_U2 = CF2_HK2*HK2_U2 + CF2_RT2*RT2_U2 + CF2_M2*M2_U2
      CF2_T2 = CF2_HK2*HK2_T2 + CF2_RT2*RT2_T2
      CF2_D2 = CF2_HK2*HK2_D2
      CF2_RE =                  CF2_RT2*RT2_RE
C
      CFM_R1 = 0.5*(                                CFM_RTA*RT1_R1)
      CFM_U1 = 0.5*(CFM_HKA*HK1_U1 + CFM_MA*M1_U1 + CFM_RTA*RT1_U1)
      CFM_T1 = 0.5*(CFM_HKA*HK1_T1 +                CFM_RTA*RT1_T1)
      CFM_D1 = 0.5*(CFM_HKA*HK1_D1                                )
      CFM_R2 = 0.5*(                                CFM_RTA*RT2_R2)
      CFM_U2 = 0.5*(CFM_HKA*HK2_U2 + CFM_MA*M2_U2 + CFM_RTA*RT2_U2)
      CFM_T2 = 0.5*(CFM_HKA*HK2_T2 +                CFM_RTA*RT2_T2)
      CFM_D2 = 0.5*(CFM_HKA*HK2_D2                                )
      CFM_RE = 0.5*CFM_RTA*(RT1_RE + RT2_RE)
C
C
C---- normalized slip velocity  Us  (centerline velocity for wake)
      US2     = 0.5*HS2*( 1.0 - (HK2-1.0)/(GBCON*H2) )
      US2_HS2 = 0.5  *  ( 1.0 - (HK2-1.0)/(GBCON*H2) )
      US2_HK2 = 0.5*HS2*(     -  1.0     /(GBCON*H2) )
      US2_H2  = 0.5*HS2*        (HK2-1.0)/(GBCON*H2**2)
C
      US2_R2 = US2_HS2*HS2_R2
      US2_U2 = US2_HS2*HS2_U2 + US2_HK2*HK2_U2
      US2_T2 = US2_HS2*HS2_T2 + US2_HK2*HK2_T2 + US2_H2*H2_T2
      US2_D2 = US2_HS2*HS2_D2 + US2_HK2*HK2_D2 + US2_H2*H2_D2
      US2_RE = US2_HS2*HS2_RE
C
      IF(ITYP.LE.2 .AND. US2.GT.0.97) THEN
CCC       WRITE(*,*) 'BLVAR: Us clamped:', US2
       US2 = 0.97
       US2_R2 = 0.
       US2_U2 = 0.
       US2_T2 = 0.
       US2_D2 = 0.
       US2_RE = 0.
      ENDIF
C
c      IF(ITYP.EQ.3 .AND. US2.GT.0.99995) THEN
cCCC       WRITE(*,*) 'BLVAR: Wake Us clamped:', US2
c       US2 = 0.99995
c       US2_R2 = 0.
c       US2_U2 = 0.
c       US2_T2 = 0.
c       US2_D2 = 0.
c       US2_RE = 0.
c      ENDIF
C
C---- equilibrium wake layer shear coefficient (Ctau)EQ ** 1/2
C   ...  NEW  1 May 93
      GCC = 0.0
      HKC = HK2 - 1.0
      HKC_HK2 = 1.0
      HKC_RT2 = 0.0
      IF(ITYP.EQ.2) THEN
       GCC = GCCON
       HKC     = HK2 - 1.0 - GCC/RT2
       HKC_HK2 = 1.0
       HKC_RT2 =             GCC/RT2**2
       IF(HKC .LT. 0.01) THEN
        HKC = 0.01
        HKC_HK2 = 0.0
        HKC_RT2 = 0.0
       ENDIF
      ENDIF
C
      HKB = HK2 - 1.0
      USB = 1.0 - US2
      CQ2     =
     &    SQRT( CTCON*HS2*HKB*HKC**2 / (USB*H2*HK2**2) )
      CQ2_HS2 = CTCON    *HKB*HKC**2 / (USB*H2*HK2**2)       * 0.5/CQ2
      CQ2_US2 = CTCON*HS2*HKB*HKC**2 / (USB*H2*HK2**2) / USB * 0.5/CQ2
      CQ2_HK2 = CTCON*HS2    *HKC**2 / (USB*H2*HK2**2)       * 0.5/CQ2
     &        - CTCON*HS2*HKB*HKC**2 / (USB*H2*HK2**3) * 2.0 * 0.5/CQ2
     &        + CTCON*HS2*HKB*HKC    / (USB*H2*HK2**2) * 2.0 * 0.5/CQ2
     &         *HKC_HK2
      CQ2_RT2 = CTCON*HS2*HKB*HKC    / (USB*H2*HK2**2) * 2.0 * 0.5/CQ2
     &         *HKC_RT2
      CQ2_H2  =-CTCON*HS2*HKB*HKC**2 / (USB*H2*HK2**2) / H2  * 0.5/CQ2
C
      CQ2_R2 = CQ2_HS2*HS2_R2 + CQ2_US2*US2_R2
      CQ2_U2 = CQ2_HS2*HS2_U2 + CQ2_US2*US2_U2 + CQ2_HK2*HK2_U2
      CQ2_T2 = CQ2_HS2*HS2_T2 + CQ2_US2*US2_T2 + CQ2_HK2*HK2_T2
      CQ2_D2 = CQ2_HS2*HS2_D2 + CQ2_US2*US2_D2 + CQ2_HK2*HK2_D2
      CQ2_RE = CQ2_HS2*HS2_RE + CQ2_US2*US2_RE
C
      CQ2_R2 = CQ2_R2                + CQ2_RT2*RT2_R2
      CQ2_U2 = CQ2_U2                + CQ2_RT2*RT2_U2
      CQ2_T2 = CQ2_T2 + CQ2_H2*H2_T2 + CQ2_RT2*RT2_T2
      CQ2_D2 = CQ2_D2 + CQ2_H2*H2_D2
      CQ2_RE = CQ2_RE                + CQ2_RT2*RT2_RE
C
C
C---- longitudinal streamline curvature correction to (Ctau)EQ ** 1/2
C-    (altered strain rate dU/dy)
      DCQCON = 2.0*SQRT(CTCON)
ccc      DCQCON = 0.0*SQRT(CTCON)
C
      DCQ     = DCQCON*UN2*DE2/U2
      DCQ_UN2 = DCQCON    *DE2/U2
      DCQ_DE2 = DCQCON*UN2    /U2
      DCQ_U2  = -DCQ/U2
C
      CQ2     = CQ2    + DCQ
      CQ2_U2  = CQ2_U2 + DCQ_DE2*DE2_U2 + DCQ_U2
      CQ2_T2  = CQ2_T2 + DCQ_DE2*DE2_T2
      CQ2_D2  = CQ2_D2 + DCQ_DE2*DE2_D2
      CQ2_UN2 =          DCQ_UN2
C
      IF(CQ2 .LT. 0.001) THEN
C----- in a pathological case (Ctau)EQ**1/2 may be negative (nonsense)
       CQ2     = 0.001
       CQ2_R2  = 0.
       CQ2_U2  = 0.
       CQ2_T2  = 0.
       CQ2_D2  = 0.
       CQ2_UN2 = 0.
       CQ2_RE  = 0.
      ENDIF
C
C
C---- dissipation function
      IF(ITYP.EQ.1) THEN
C
C----- laminar
       CALL DIL( HK2, RT2, DI2, DI2_HK2, DI2_RT2 )
C
       DI2_S2 = 0.
       DI2_R2 =                  DI2_RT2*RT2_R2
       DI2_U2 = DI2_HK2*HK2_U2 + DI2_RT2*RT2_U2
       DI2_T2 = DI2_HK2*HK2_T2 + DI2_RT2*RT2_T2
       DI2_D2 = DI2_HK2*HK2_D2
       DI2_RE =                  DI2_RT2*RT2_RE
C
      ELSE IF(ITYP.EQ.2) THEN
C
CCC       CALL DIT(     HS2,     US2,     CF2,     S2, DI2,
CCC     &           DI2_HS2, DI2_US2, DI2_CF2, DI2_S2      )
C
C
C----- turbulent wall contribution
       CALL CFT(HK2, RT2, M2,GAMBL, CF2t, CF2t_HK2, CF2t_RT2, CF2t_M2)
       DI2      =  ( 0.5*CF2t*US2 ) * 2.0/HS2
       DI2_HS2  = -( 0.5*CF2t*US2 ) * 2.0/HS2**2
       DI2_US2  =  ( 0.5*CF2t     ) * 2.0/HS2
       DI2_CF2t =  ( 0.5     *US2 ) * 2.0/HS2
C
       CF2t_R2 =                   CF2t_RT2*RT2_R2
       CF2t_U2 = CF2t_HK2*HK2_U2 + CF2t_RT2*RT2_U2 + CF2t_M2*M2_U2
       CF2t_T2 = CF2t_HK2*HK2_T2 + CF2t_RT2*RT2_T2
       CF2t_D2 = CF2t_HK2*HK2_D2
       CF2t_RE =                   CF2t_RT2*RT2_RE
C

       DI2_S2 = 0.0
       DI2_R2 = DI2_HS2*HS2_R2 + DI2_US2*US2_R2 + DI2_CF2t*CF2t_R2
       DI2_U2 = DI2_HS2*HS2_U2 + DI2_US2*US2_U2 + DI2_CF2t*CF2t_U2
       DI2_T2 = DI2_HS2*HS2_T2 + DI2_US2*US2_T2 + DI2_CF2t*CF2t_T2
       DI2_D2 = DI2_HS2*HS2_D2 + DI2_US2*US2_D2 + DI2_CF2t*CF2t_D2
       DI2_RE = DI2_HS2*HS2_RE + DI2_US2*US2_RE + DI2_CF2t*CF2t_RE
C
C
C----- set minimum Hk for wake layer to still exist
       GRT = LOG(RT2)
       HMIN = 1.0 + 2.1/GRT
       HM_RT2 = -(2.1/GRT**2) / RT2
C
C----- set factor DFAC for correcting wall dissipation for very low Hk
       FL = (HK2-1.0)/(HMIN-1.0)
       FL_HK2 =   1.0/(HMIN-1.0)
       FL_RT2 = ( -FL/(HMIN-1.0) ) * HM_RT2
C
       TFL = TANH(FL)
       DFAC  = 0.5 + 0.5* TFL
       DF_FL =       0.5*(1.0 - TFL**2)
C
       DF_HK2 = DF_FL*FL_HK2
       DF_RT2 = DF_FL*FL_RT2
C
       DI2_S2 = DI2_S2*DFAC
       DI2_R2 = DI2_R2*DFAC + DI2*(                DF_RT2*RT2_R2)
       DI2_U2 = DI2_U2*DFAC + DI2*(DF_HK2*HK2_U2 + DF_RT2*RT2_U2)
       DI2_T2 = DI2_T2*DFAC + DI2*(DF_HK2*HK2_T2 + DF_RT2*RT2_T2)
       DI2_D2 = DI2_D2*DFAC + DI2*(DF_HK2*HK2_D2                )
       DI2_RE = DI2_RE*DFAC + DI2*(                DF_RT2*RT2_RE)
       DI2    = DI2   *DFAC
C
      ELSE
C
C----- zero wall contribution for wake
       DI2    = 0.0
       DI2_S2 = 0.0
       DI2_R2 = 0.0
       DI2_U2 = 0.0
       DI2_T2 = 0.0
       DI2_D2 = 0.0
       DI2_RE = 0.0
C
      ENDIF
C
C
C---- Add on turbulent outer layer contribution
      IF(ITYP.NE.1) THEN
C
       DD     =  (S2+DS2)**2 * (1.0-US2) * 2.0/HS2
       DD_HS2 = -(S2+DS2)**2 * (1.0-US2) * 2.0/HS2**2
       DD_US2 = -(S2+DS2)**2               * 2.0/HS2
       DD_S2  =  (S2+DS2)*2.0* (1.0-US2) * 2.0/HS2
       DD_DS2 =  (S2+DS2)*2.0* (1.0-US2) * 2.0/HS2
C
       DI2    = DI2    + DD
       DI2_S2 =          DD_S2
       DI2_R2 = DI2_R2 + DD_HS2*HS2_R2 + DD_US2*US2_R2 + DD_DS2*DS2_R2
       DI2_U2 = DI2_U2 + DD_HS2*HS2_U2 + DD_US2*US2_U2 + DD_DS2*DS2_U2
       DI2_T2 = DI2_T2 + DD_HS2*HS2_T2 + DD_US2*US2_T2 + DD_DS2*DS2_T2
       DI2_D2 = DI2_D2 + DD_HS2*HS2_D2 + DD_US2*US2_D2 + DD_DS2*DS2_D2
       DI2_RE = DI2_RE + DD_HS2*HS2_RE + DD_US2*US2_RE + DD_DS2*DS2_RE
C
      ENDIF
C
      IF(ITYP.EQ.2) THEN
C----- use laminar CD if larger than turbulent CD  (at very low Rtheta)
       CALL DIL( HK2, RT2, DI2l, DI2l_HK2, DI2l_RT2 )
C
       IF(DI2l.GT.DI2) THEN
cc          write(*,*) 'Using laminar Cd:', DI2, DI2l
         DI2     = DI2l
         DI2_S2 = 0.
         DI2_R2 =                   DI2l_RT2*RT2_R2
         DI2_U2 = DI2l_HK2*HK2_U2 + DI2l_RT2*RT2_U2
         DI2_T2 = DI2l_HK2*HK2_T2 + DI2l_RT2*RT2_T2
         DI2_D2 = DI2l_HK2*HK2_D2
         DI2_RE =                   DI2l_RT2*RT2_RE
       ENDIF
C
      ENDIF
C
C---- add on CD contribution of inner shear layer
c      IF(ITYP.EQ.3 .AND. DW2.GT.0.0) THEN
c       DKON = 2.0*CTCON*0.75**3
c       DD = DKON*US2**3
c       DD_US2 = 3.0*DKON*US2**2
c       DI2    = DI2    + DD            * DW2/DWTE(ISIDE)
c       DI2_R2 = DI2_R2 + DD_US2*US2_R2 * DW2/DWTE(ISIDE)
c       DI2_U2 = DI2_U2 + DD_US2*US2_U2 * DW2/DWTE(ISIDE)
c       DI2_T2 = DI2_T2 + DD_US2*US2_T2 * DW2/DWTE(ISIDE)
c       DI2_D2 = DI2_D2 + DD_US2*US2_D2 * DW2/DWTE(ISIDE)
c       DI2_RE = DI2_RE + DD_US2*US2_RE * DW2/DWTE(ISIDE)
c      ENDIF
C
C
      RETURN
      END ! BLVAR
 

 
      SUBROUTINE TRDIF
C...............................................
C     Sets up the Newton system governing the
C     transition interval.  Equations governing
C     the  laminar  part  X1 < xi < XT  and
C     the turbulent part  XT < xi < X2
C     are simply summed.
C...............................................
      IMPLICIT REAL (A-H,M,O-Z)
      INCLUDE 'MBL.INC'
      DIMENSION  BL1(4,7), BL2(4,7), BLREZ(4), BLR(4), BLX(4)
     &         , BT1(4,7), BT2(4,7), BTREZ(4), BTR(4), BTX(4)
      DIMENSION C1SAV(NCOM), C2SAV(NCOM)
C
C---- save variables and sensitivities for future restoration
      DO 5 NC=1, NCOM
        C1SAV(NC) = COM1(NC)
        C2SAV(NC) = COM2(NC)
    5 CONTINUE
C
C
C---- weighting factors for interpolation to transition point
      SF    = (XT-X1)/(X2-X1)
      SF_XT = 1.0/(X2-X1)
      SF_X1 = (SF - 1.0)/(X2-X1)
      SF_X2 = (   - SF )/(X2-X1)
C
ccc      WF2    = 3.0*SF**2 - 2.0*SF**3
ccc      WF2_SF = 6.0*SF    - 6.0*SF**2
C
C---- linear interpolation
      WF2    = SF
      WF2_SF = 1.0
C
C
      WF2_XT = WF2_SF*SF_XT
C
      WF2_A1 = WF2_XT*XT_A1
      WF2_X1 = WF2_XT*XT_X1 + WF2_SF*SF_X1
      WF2_X2 = WF2_XT*XT_X2 + WF2_SF*SF_X2
      WF2_T1 = WF2_XT*XT_T1
      WF2_T2 = WF2_XT*XT_T2
      WF2_D1 = WF2_XT*XT_D1
      WF2_D2 = WF2_XT*XT_D2
      WF2_U1 = WF2_XT*XT_U1
      WF2_U2 = WF2_XT*XT_U2
      WF2_R1 = WF2_XT*XT_R1
      WF2_R2 = WF2_XT*XT_R2
      WF2_RE = WF2_XT*XT_RE
      WF2_NC = WF2_XT*XT_NC
      WF2_XF = WF2_XT*XT_XF
C
      WF1    = 1.0 - WF2
      WF1_A1 = -WF2_A1
      WF1_X1 = -WF2_X1
      WF1_X2 = -WF2_X2
      WF1_T1 = -WF2_T1
      WF1_T2 = -WF2_T2
      WF1_D1 = -WF2_D1
      WF1_D2 = -WF2_D2
      WF1_U1 = -WF2_U1
      WF1_U2 = -WF2_U2
      WF1_R1 = -WF2_R1
      WF1_R2 = -WF2_R2
      WF1_RE = -WF2_RE
      WF1_NC = -WF2_NC
      WF1_XF = -WF2_XF
C
C
C**** FIRST,  do laminar part between X1 and XT
C
C-----interpolate primary variables to transition point
      TT    = T1*WF1    + T2*WF2
      TT_A1 = T1*WF1_A1 + T2*WF2_A1
      TT_X1 = T1*WF1_X1 + T2*WF2_X1
      TT_X2 = T1*WF1_X2 + T2*WF2_X2
      TT_T1 = T1*WF1_T1 + T2*WF2_T1 + WF1
      TT_T2 = T1*WF1_T2 + T2*WF2_T2 + WF2
      TT_D1 = T1*WF1_D1 + T2*WF2_D1
      TT_D2 = T1*WF1_D2 + T2*WF2_D2
      TT_U1 = T1*WF1_U1 + T2*WF2_U1
      TT_U2 = T1*WF1_U2 + T2*WF2_U2
      TT_R1 = T1*WF1_R1 + T2*WF2_R1
      TT_R2 = T1*WF1_R2 + T2*WF2_R2
      TT_RE = T1*WF1_RE + T2*WF2_RE
      TT_NC = T1*WF1_NC + T2*WF2_NC
      TT_XF = T1*WF1_XF + T2*WF2_XF
C
      DT    = D1*WF1    + D2*WF2
      DT_A1 = D1*WF1_A1 + D2*WF2_A1
      DT_X1 = D1*WF1_X1 + D2*WF2_X1
      DT_X2 = D1*WF1_X2 + D2*WF2_X2
      DT_T1 = D1*WF1_T1 + D2*WF2_T1
      DT_T2 = D1*WF1_T2 + D2*WF2_T2
      DT_D1 = D1*WF1_D1 + D2*WF2_D1 + WF1
      DT_D2 = D1*WF1_D2 + D2*WF2_D2 + WF2
      DT_U1 = D1*WF1_U1 + D2*WF2_U1
      DT_U2 = D1*WF1_U2 + D2*WF2_U2
      DT_R1 = D1*WF1_R1 + D2*WF2_R1
      DT_R2 = D1*WF1_R2 + D2*WF2_R2
      DT_RE = D1*WF1_RE + D2*WF2_RE
      DT_NC = D1*WF1_NC + D2*WF2_NC
      DT_XF = D1*WF1_XF + D2*WF2_XF
C
      UT    = U1*WF1    + U2*WF2
      UT_A1 = U1*WF1_A1 + U2*WF2_A1
      UT_X1 = U1*WF1_X1 + U2*WF2_X1
      UT_X2 = U1*WF1_X2 + U2*WF2_X2
      UT_T1 = U1*WF1_T1 + U2*WF2_T1
      UT_T2 = U1*WF1_T2 + U2*WF2_T2
      UT_D1 = U1*WF1_D1 + U2*WF2_D1
      UT_D2 = U1*WF1_D2 + U2*WF2_D2
      UT_U1 = U1*WF1_U1 + U2*WF2_U1 + WF1
      UT_U2 = U1*WF1_U2 + U2*WF2_U2 + WF2
      UT_R1 = U1*WF1_R1 + U2*WF2_R1
      UT_R2 = U1*WF1_R2 + U2*WF2_R2
      UT_RE = U1*WF1_RE + U2*WF2_RE
      UT_NC = U1*WF1_NC + U2*WF2_NC
      UT_XF = U1*WF1_XF + U2*WF2_XF
C
      RT    = R1*WF1    + R2*WF2
      RT_A1 = R1*WF1_A1 + R2*WF2_A1
      RT_X1 = R1*WF1_X1 + R2*WF2_X1
      RT_X2 = R1*WF1_X2 + R2*WF2_X2
      RT_T1 = R1*WF1_T1 + R2*WF2_T1
      RT_T2 = R1*WF1_T2 + R2*WF2_T2
      RT_D1 = R1*WF1_D1 + R2*WF2_D1
      RT_D2 = R1*WF1_D2 + R2*WF2_D2
      RT_U1 = R1*WF1_U1 + R2*WF2_U1
      RT_U2 = R1*WF1_U2 + R2*WF2_U2
      RT_R1 = R1*WF1_R1 + R2*WF2_R1 + WF1
      RT_R2 = R1*WF1_R2 + R2*WF2_R2 + WF2
      RT_RE = R1*WF1_RE + R2*WF2_RE
      RT_NC = R1*WF1_NC + R2*WF2_NC
      RT_XF = R1*WF1_XF + R2*WF2_XF
C
      UNT    = UN1*WF1    + UN2*WF2
      UNT_A1 = UN1*WF1_A1 + UN2*WF2_A1
      UNT_X1 = UN1*WF1_X1 + UN2*WF2_X1
      UNT_X2 = UN1*WF1_X2 + UN2*WF2_X2
      UNT_T1 = UN1*WF1_T1 + UN2*WF2_T1
      UNT_T2 = UN1*WF1_T2 + UN2*WF2_T2
      UNT_D1 = UN1*WF1_D1 + UN2*WF2_D1
      UNT_D2 = UN1*WF1_D2 + UN2*WF2_D2
      UNT_U1 = UN1*WF1_U1 + UN2*WF2_U1
      UNT_U2 = UN1*WF1_U2 + UN2*WF2_U2
      UNT_R1 = UN1*WF1_R1 + UN2*WF2_R1
      UNT_R2 = UN1*WF1_R2 + UN2*WF2_R2
      UNT_RE = UN1*WF1_RE + UN2*WF2_RE
      UNT_NC = UN1*WF1_NC + UN2*WF2_NC
      UNT_XF = UN1*WF1_XF + UN2*WF2_XF
      UNT_UN1 =    WF1
      UNT_UN2 =                 WF2
C
C---- set "2" variables to primary "T" variables at XT
      XSI = XT
      AMI = AMCRIT
      CTI = 0.
      UEI = UT
      THI = TT
      DSI = DT
      DSWAKI = 0.0
      UNI = UNT
      RHI = RT
C
C---- calculate kinematic "T" variables
      CALL SETKIN(XSI,AMI,CTI,UEI,THI,DSI,DSWAKI,UNI,RHI)
C
C---- calculate laminar secondary "T" variables
      CALL BLVAR(1)
c
c      write(*,2233) 1000*t2, 1000*d2, hk2, rt2, s2, cq2, u2, un2
c      write(*,*) 'L --- T'
C=
C=    at this point, all "2" variables are really "T" variables at XT
C=
C
C---- set up Newton system for dAm, dTh, dDs, dUe, dXi  at  X1 and XT
      CALL BLDIF(1)
C
C---- The current Newton system is in terms of "1" and "T" variables,
C-    so calculate its equivalent in terms of "1" and "2" variables.
C-    In other words, convert residual sensitivities wrt "T" variables
C-    into sensitivities wrt "1" and "2" variables.  The amplification
C-    equation is unnecessary here, so the K=1 row is left empty.
      DO 10 K=2, 3
        BLREZ(K) = VSREZ(K)
c        BLN(K)   = VS2(K,2)*TT_NC
c     &           + VS2(K,3)*DT_NC
c     &           + VS2(K,4)*UT_NC
c     &           + VS2(K,5)*XT_NC
c     &           + VS2(K,6)*UNT_NC
c     &           + VS2(K,7)*RT_NC
        BLR(K)   = VSR(K)
     &           + VS2(K,2)*TT_RE
     &           + VS2(K,3)*DT_RE
     &           + VS2(K,4)*UT_RE
     &           + VS2(K,5)*XT_RE
     &           + VS2(K,6)*UNT_RE
     &           + VS2(K,7)*RT_RE
        BLX(K)   = VSX(K)
     &           + VS2(K,2)*TT_XF
     &           + VS2(K,3)*DT_XF
     &           + VS2(K,4)*UT_XF
     &           + VS2(K,5)*XT_XF
     &           + VS2(K,6)*UNT_XF
     &           + VS2(K,7)*RT_XF
C
        BL1(K,1) = VS1(K,1)
     &           + VS2(K,2)*TT_A1
     &           + VS2(K,3)*DT_A1
     &           + VS2(K,4)*UT_A1
     &           + VS2(K,5)*XT_A1
     &           + VS2(K,6)*UNT_A1
     &           + VS2(K,7)*RT_A1
        BL1(K,2) = VS1(K,2)
     &           + VS2(K,2)*TT_T1
     &           + VS2(K,3)*DT_T1
     &           + VS2(K,4)*UT_T1
     &           + VS2(K,5)*XT_T1
     &           + VS2(K,6)*UNT_T1
     &           + VS2(K,7)*RT_T1
        BL1(K,3) = VS1(K,3)
     &           + VS2(K,2)*TT_D1
     &           + VS2(K,3)*DT_D1
     &           + VS2(K,4)*UT_D1
     &           + VS2(K,5)*XT_D1
     &           + VS2(K,6)*UNT_D1
     &           + VS2(K,7)*RT_D1
        BL1(K,4) = VS1(K,4)
     &           + VS2(K,2)*TT_U1
     &           + VS2(K,3)*DT_U1
     &           + VS2(K,4)*UT_U1
     &           + VS2(K,5)*XT_U1
     &           + VS2(K,6)*UNT_U1
     &           + VS2(K,7)*RT_U1
        BL1(K,5) = VS1(K,5)
     &           + VS2(K,2)*TT_X1
     &           + VS2(K,3)*DT_X1
     &           + VS2(K,4)*UT_X1
     &           + VS2(K,5)*XT_X1
     &           + VS2(K,6)*UNT_X1
     &           + VS2(K,7)*RT_X1
        BL1(K,6) = VS1(K,6)
     &           + VS2(K,6)*UNT_UN1
        BL1(K,7) = VS1(K,7)
     &           + VS2(K,2)*TT_R1
     &           + VS2(K,3)*DT_R1
     &           + VS2(K,4)*UT_R1
     &           + VS2(K,5)*XT_R1
     &           + VS2(K,6)*UNT_R1
     &           + VS2(K,7)*RT_R1
C
        BL2(K,1) = 0.
        BL2(K,2) = VS2(K,2)*TT_T2
     &           + VS2(K,3)*DT_T2
     &           + VS2(K,4)*UT_T2
     &           + VS2(K,5)*XT_T2
     &           + VS2(K,6)*UNT_T2
     &           + VS2(K,7)*RT_T2
        BL2(K,3) = VS2(K,2)*TT_D2
     &           + VS2(K,3)*DT_D2
     &           + VS2(K,4)*UT_D2
     &           + VS2(K,5)*XT_D2
     &           + VS2(K,6)*UNT_D2
     &           + VS2(K,7)*RT_D2
        BL2(K,4) = VS2(K,2)*TT_U2
     &           + VS2(K,3)*DT_U2
     &           + VS2(K,4)*UT_U2
     &           + VS2(K,5)*XT_U2
     &           + VS2(K,6)*UNT_U2
     &           + VS2(K,7)*RT_U2
        BL2(K,5) = VS2(K,2)*TT_X2
     &           + VS2(K,3)*DT_X2
     &           + VS2(K,4)*UT_X2
     &           + VS2(K,5)*XT_X2
     &           + VS2(K,6)*UNT_X2
     &           + VS2(K,7)*RT_X2
        BL2(K,6) = VS2(K,6)*UNT_UN2
        BL2(K,7) = VS2(K,2)*TT_R2
     &           + VS2(K,3)*DT_R2
     &           + VS2(K,4)*UT_R2
     &           + VS2(K,5)*XT_R2
     &           + VS2(K,6)*UNT_R2
     &           + VS2(K,7)*RT_R2
C
   10 CONTINUE
C
C
C**** SECOND, set up turbulent part between XT and X2  ****
C
C---- calculate equilibrium shear coefficient CQT at transition point
      CALL BLVAR(2)
C
C---- set initial shear coefficient value ST at transition point
C-    ( note that CQ2, CQ2_T2, etc. are really "CQT", "CQT_TT", etc.)
C
      CTR     = 1.8*EXP(-3.3/(HK2-1.0))
      CTR_HK2 = CTR * 3.3/(HK2-1.0)**2
C
CCC      CTR = 1.2
CCC      CTR = 0.7
CCC      CTR_HK2 = 0.0
C
      ST    = CTR*CQ2
      ST_RT = CTR*CQ2_R2
      ST_UT = CTR*CQ2_U2 + CQ2*CTR_HK2*HK2_U2
      ST_TT = CTR*CQ2_T2 + CQ2*CTR_HK2*HK2_T2
      ST_DT = CTR*CQ2_D2 + CQ2*CTR_HK2*HK2_D2
      ST_UNT= CTR*CQ2_UN2
      ST_RE = CTR*CQ2_RE
C
C---- calculate ST sensitivities wrt the actual "1" and "2" variables
      ST_A1 = ST_TT*TT_A1 + ST_DT*DT_A1 + ST_UT*UT_A1
      ST_X1 = ST_TT*TT_X1 + ST_DT*DT_X1 + ST_UT*UT_X1
      ST_X2 = ST_TT*TT_X2 + ST_DT*DT_X2 + ST_UT*UT_X2
      ST_T1 = ST_TT*TT_T1 + ST_DT*DT_T1 + ST_UT*UT_T1
      ST_T2 = ST_TT*TT_T2 + ST_DT*DT_T2 + ST_UT*UT_T2
      ST_D1 = ST_TT*TT_D1 + ST_DT*DT_D1 + ST_UT*UT_D1
      ST_D2 = ST_TT*TT_D2 + ST_DT*DT_D2 + ST_UT*UT_D2
      ST_U1 = ST_TT*TT_U1 + ST_DT*DT_U1 + ST_UT*UT_U1
      ST_U2 = ST_TT*TT_U2 + ST_DT*DT_U2 + ST_UT*UT_U2
      ST_R1 = ST_TT*TT_R1 + ST_DT*DT_R1 + ST_UT*UT_R1
      ST_R2 = ST_TT*TT_R2 + ST_DT*DT_R2 + ST_UT*UT_R2
      ST_RE = ST_TT*TT_RE + ST_DT*DT_RE + ST_UT*UT_RE + ST_RE
      ST_NC = ST_TT*TT_NC + ST_DT*DT_NC + ST_UT*UT_NC
      ST_XF = ST_TT*TT_XF + ST_DT*DT_XF + ST_UT*UT_XF
C
      ST_A1 = ST_RT*RT_A1 + ST_UNT*UNT_A1  + ST_A1
      ST_X1 = ST_RT*RT_X1 + ST_UNT*UNT_X1  + ST_X1
      ST_X2 = ST_RT*RT_X2 + ST_UNT*UNT_X2  + ST_X2
      ST_T1 = ST_RT*RT_T1 + ST_UNT*UNT_T1  + ST_T1
      ST_T2 = ST_RT*RT_T2 + ST_UNT*UNT_T2  + ST_T2
      ST_D1 = ST_RT*RT_D1 + ST_UNT*UNT_D1  + ST_D1
      ST_D2 = ST_RT*RT_D2 + ST_UNT*UNT_D2  + ST_D2
      ST_U1 = ST_RT*RT_U1 + ST_UNT*UNT_U1  + ST_U1
      ST_U2 = ST_RT*RT_U2 + ST_UNT*UNT_U2  + ST_U2
      ST_R1 = ST_RT*RT_R1 + ST_UNT*UNT_R1  + ST_R1
      ST_R2 = ST_RT*RT_R2 + ST_UNT*UNT_R2  + ST_R2
      ST_RE = ST_RT*RT_RE + ST_UNT*UNT_RE  + ST_RE
      ST_NC = ST_RT*RT_NC + ST_UNT*UNT_NC  + ST_NC
      ST_XF = ST_RT*RT_XF + ST_UNT*UNT_XF  + ST_XF
      ST_UN1 =              ST_UNT*UNT_UN1
      ST_UN2 =              ST_UNT*UNT_UN2
C
      CTI = ST
C
C---- recalculate turbulent secondary "T" variables using proper Ctau
      S2 = CTI
      CALL BLVAR(2)
C
C---- set "1" variables to "T" variables and reset "2" variables
C-    to their saved turbulent values
      DO 30 NC=1, NCOM
        COM1(NC) = COM2(NC)
        COM2(NC) = C2SAV(NC)
   30 CONTINUE
C
C---- recalculate midpoint CFM value
      CALL BLVAR(2)
C
C---- set up Newton system for dCt, dTh, dDs, dUe, dXi  at  XT and X2
      CALL BLDIF(2)
C
C---- convert sensitivities wrt "T" variables into sensitivities
C-    wrt "1" and "2" variables as done before for the laminar part
      DO 40 K=1, 3
        BTREZ(K) = VSREZ(K)
c        BTN(K)   = VS1(K,1)*ST_NC
c     &           + VS1(K,2)*TT_NC
c     &           + VS1(K,3)*DT_NC
c     &           + VS1(K,4)*UT_NC
c     &           + VS1(K,5)*XT_NC
c     &           + VS1(K,6)*UNT_NC
c     &           + VS1(K,7)*RT_NC
        BTR(K)   = VSR(K)
     &           + VS1(K,1)*ST_RE
     &           + VS1(K,2)*TT_RE
     &           + VS1(K,3)*DT_RE
     &           + VS1(K,4)*UT_RE
     &           + VS1(K,5)*XT_RE
     &           + VS1(K,6)*UNT_RE
     &           + VS1(K,7)*RT_RE
        BTX(K)   = VSX(K)
     &           + VS1(K,1)*ST_XF
     &           + VS1(K,2)*TT_XF
     &           + VS1(K,3)*DT_XF
     &           + VS1(K,4)*UT_XF
     &           + VS1(K,5)*XT_XF
     &           + VS1(K,6)*UNT_XF
     &           + VS1(K,7)*RT_XF
C
        BT1(K,1) = VS1(K,1)*ST_A1
     &           + VS1(K,2)*TT_A1
     &           + VS1(K,3)*DT_A1
     &           + VS1(K,4)*UT_A1
     &           + VS1(K,5)*XT_A1
     &           + VS1(K,6)*UNT_A1
     &           + VS1(K,7)*RT_A1
        BT1(K,2) = VS1(K,1)*ST_T1
     &           + VS1(K,2)*TT_T1
     &           + VS1(K,3)*DT_T1
     &           + VS1(K,4)*UT_T1
     &           + VS1(K,5)*XT_T1
     &           + VS1(K,6)*UNT_T1
     &           + VS1(K,7)*RT_T1
        BT1(K,3) = VS1(K,1)*ST_D1
     &           + VS1(K,2)*TT_D1
     &           + VS1(K,3)*DT_D1
     &           + VS1(K,4)*UT_D1
     &           + VS1(K,5)*XT_D1
     &           + VS1(K,6)*UNT_D1
     &           + VS1(K,7)*RT_D1
        BT1(K,4) = VS1(K,1)*ST_U1
     &           + VS1(K,2)*TT_U1
     &           + VS1(K,3)*DT_U1
     &           + VS1(K,4)*UT_U1
     &           + VS1(K,5)*XT_U1
     &           + VS1(K,6)*UNT_U1
     &           + VS1(K,7)*RT_U1
        BT1(K,5) = VS1(K,1)*ST_X1
     &           + VS1(K,2)*TT_X1
     &           + VS1(K,3)*DT_X1
     &           + VS1(K,4)*UT_X1
     &           + VS1(K,5)*XT_X1
     &           + VS1(K,6)*UNT_X1
     &           + VS1(K,7)*RT_X1
        BT1(K,6) = VS1(K,1)*ST_UN1
     &           + VS1(K,6)*UNT_UN1
        BT1(K,7) = VS1(K,1)*ST_R1
     &           + VS1(K,2)*TT_R1
     &           + VS1(K,3)*DT_R1
     &           + VS1(K,4)*UT_R1
     &           + VS1(K,5)*XT_R1
     &           + VS1(K,6)*UNT_R1
     &           + VS1(K,7)*RT_R1
C
        BT2(K,1) = VS2(K,1)
        BT2(K,2) = VS2(K,2)
     &           + VS1(K,1)*ST_T2
     &           + VS1(K,2)*TT_T2
     &           + VS1(K,3)*DT_T2
     &           + VS1(K,4)*UT_T2
     &           + VS1(K,5)*XT_T2
     &           + VS1(K,6)*UNT_T2
     &           + VS1(K,7)*RT_T2
        BT2(K,3) = VS2(K,3)
     &           + VS1(K,1)*ST_D2
     &           + VS1(K,2)*TT_D2
     &           + VS1(K,3)*DT_D2
     &           + VS1(K,4)*UT_D2
     &           + VS1(K,5)*XT_D2
     &           + VS1(K,6)*UNT_D2
     &           + VS1(K,7)*RT_D2
        BT2(K,4) = VS2(K,4)
     &           + VS1(K,1)*ST_U2
     &           + VS1(K,2)*TT_U2
     &           + VS1(K,3)*DT_U2
     &           + VS1(K,4)*UT_U2
     &           + VS1(K,5)*XT_U2
     &           + VS1(K,6)*UNT_U2
     &           + VS1(K,7)*RT_U2
        BT2(K,5) = VS2(K,5)
     &           + VS1(K,1)*ST_X2
     &           + VS1(K,2)*TT_X2
     &           + VS1(K,3)*DT_X2
     &           + VS1(K,4)*UT_X2
     &           + VS1(K,5)*XT_X2
     &           + VS1(K,6)*UNT_X2
     &           + VS1(K,7)*RT_X2
        BT2(K,6) = VS2(K,6)
     &           + VS1(K,1)*ST_UN2
     &           + VS1(K,6)*UNT_UN2
        BT2(K,7) = VS2(K,7)
     &           + VS1(K,1)*ST_R2
     &           + VS1(K,2)*TT_R2
     &           + VS1(K,3)*DT_R2
     &           + VS1(K,4)*UT_R2
     &           + VS1(K,5)*XT_R2
     &           + VS1(K,6)*UNT_R2
     &           + VS1(K,7)*RT_R2
C
   40 CONTINUE
C
C---- Add up laminar and turbulent parts to get final system
C-    in terms of honest-to-God "1" and "2" variables.
      VSREZ(1) =            BTREZ(1)
      VSREZ(2) = BLREZ(2) + BTREZ(2)
      VSREZ(3) = BLREZ(3) + BTREZ(3)
c      VSN(1)   =            BTN(1)
c      VSN(2)   = BLN(2)   + BTN(2)
c      VSN(3)   = BLN(3)   + BTN(3)
      VSR(1)   =            BTR(1)
      VSR(2)   = BLR(2)   + BTR(2)
      VSR(3)   = BLR(3)   + BTR(3)
      VSX(1)   =            BTX(1)
      VSX(2)   = BLX(2)   + BTX(2)
      VSX(3)   = BLX(3)   + BTX(3)
      DO 60 L=1, 7
        VS1(1,L) =            BT1(1,L)
        VS2(1,L) =            BT2(1,L)
        VS1(2,L) = BL1(2,L) + BT1(2,L)
        VS2(2,L) = BL2(2,L) + BT2(2,L)
        VS1(3,L) = BL1(3,L) + BT1(3,L)
        VS2(3,L) = BL2(3,L) + BT2(3,L)
   60 CONTINUE
C
C---- To be sanitary, restore "1" quantities which got clobbered
C-    in all of the numerical gymnastics above.  The "2" variables
C-    were already restored for the XT-X2 differencing part.
      DO 70 NC=1, NCOM
        COM1(NC) = C1SAV(NC)
   70 CONTINUE
C
C---- restore input "2" variables
      XSI = X2
      CTI = S2
      UEI = U2
      THI = T2
      DSI = D2
      RHI = R2
      DSWAKI = DW2
      UNI = UN2
C
      RETURN
      END ! TRDIF
 
 
      SUBROUTINE BLDIF(ITYP)
C............................................................
C     Sets up the Newton system coefficients and residuals
C
C        ITYP = 0 :  similarity station
C        ITYP = 1 :  laminar interval
C        ITYP = 2 :  turbulent interval
C        ITYP = 3 :  wake interval
C
C     This routine knows nothing about a transition interval,
C     which is taken care of by TRDIF.
C............................................................
      IMPLICIT REAL (A-H,M,O-Z)
      INCLUDE 'MBL.INC'
C
C---- limits of dissipation length factor
      DATA ALDMIN, ALDMAX / 0.4 , 2.5 /
ccc      DATA ALDMIN, ALDMAX / 0.99, 1.01 /
ccc      DATA ALDMIN, ALDMAX / 0.66667, 1.5 /
C
      IF(ITYP.EQ.0) THEN
C----- similarity logarithmic differences  (prescribed)
       XLOG = 1.0
       ULOG = BULE(ISIDE)
       TLOG = 0.5*(1.0 - BULE(ISIDE))
       RLOG = 0.
       HLOG = 0.
       DDLOG = 0.
      ELSE
C----- usual logarithmic differences
       XLOG = LOG(X2/X1)
       ULOG = LOG(U2/U1)
       TLOG = LOG(T2/T1)
       RLOG = LOG(R2/R1)
       HLOG = LOG(HS2/HS1)
       DDLOG = 1.0
      ENDIF
C
C---- zero out system arrays
      DO 55 K=1, 4
        VSREZ(K) = 0.
        VSR(K) = 0.
        VSX(K) = 0.
        DO 551 L=1, 7
          VS1(K,L) = 0.
          VS2(K,L) = 0.
  551   CONTINUE
   55 CONTINUE
C
C---- set triggering constant for local upwinding
ccc      HDCON = 5.0
ccc      HD_HK1 = 0.0
ccc      HD_HK2 = 0.0
C
C---- new formulation: cut down upwinding in separations (large Hk)
C%%%      HDCON  =  5.0/HK2**2
      HDCON  =  1.0/HK2**2
      HD_HK1 =  0.0
      HD_HK2 = -HDCON*2.0/HK2
C
C---- use less upwinding in the wake
      IF(ITYP.EQ.3) THEN
C%%%       HDCON  =  1.0/HK2**2
       HDCON  =  0.2/HK2**2
       HD_HK1 =  0.0
       HD_HK2 = -HDCON*2.0/HK2
      ENDIF
C
C---- local upwinding is based on local change in  log(Hk-1)
C-    (mainly kicks in at transition)
      ARG = ABS((HK2-1.0)/(HK1-1.0))
      HL = LOG(ARG)
      HL_HK1 = -1.0/(HK1-1.0)
      HL_HK2 =  1.0/(HK2-1.0)
C
C---- set local upwinding parameter UPW and linearize it
C
C       UPW = 0.5   Trapezoidal
C       UPW = 1.0   Backward Euler
C
      HLSQ = MIN( HL**2 , 15.0 )
      EHH = EXP(-HLSQ*HDCON)
      UPW = 1.0 - 0.5*EHH
      UPW_HL =        EHH * HL  *HDCON
      UPW_HD =    0.5*EHH * HLSQ
C
      UPW_HK1 = UPW_HL*HL_HK1 + UPW_HD*HD_HK1
      UPW_HK2 = UPW_HL*HL_HK2 + UPW_HD*HD_HK2
C
      UPW_U1 = UPW_HK1*HK1_U1
      UPW_T1 = UPW_HK1*HK1_T1
      UPW_D1 = UPW_HK1*HK1_D1
      UPW_U2 = UPW_HK2*HK2_U2
      UPW_T2 = UPW_HK2*HK2_T2
      UPW_D2 = UPW_HK2*HK2_D2
C
C
      IF(ITYP.EQ.0) THEN
C
C***** LE point -->  set zero amplification factor
       VS2(1,1) = 1.0
       VSREZ(1) = -AMPL2
C
      ELSE IF(ITYP.EQ.1) THEN
C
C***** laminar part -->  set amplification equation
       CALL AXSET( HK1,    T1,    RT1, AMPL1,
     &             HK2,    T2,    RT2, AMPL2, AMCRIT,
     &      AX, AX_HK1, AX_T1, AX_RT1, AX_A1,
     &          AX_HK2, AX_T2, AX_RT2, AX_A2, AX_NC )
C
       REZC = AMPL2 - AMPL1 - AX*(X2-X1)
       Z_AX = -(X2-X1)
C
       VS1(1,1) = Z_AX* AX_A1  -  1.0
       VS1(1,2) = Z_AX*(AX_HK1*HK1_T1 + AX_T1 + AX_RT1*RT1_T1)
       VS1(1,3) = Z_AX*(AX_HK1*HK1_D1                        )
       VS1(1,4) = Z_AX*(AX_HK1*HK1_U1         + AX_RT1*RT1_U1)
       VS1(1,5) =  AX
       VS1(1,7) = Z_AX*(                        AX_RT1*RT1_R1)
C
       VS2(1,1) = Z_AX* AX_A2  +  1.0
       VS2(1,2) = Z_AX*(AX_HK2*HK2_T2 + AX_T2 + AX_RT2*RT2_T2)
       VS2(1,3) = Z_AX*(AX_HK2*HK2_D2                        )
       VS2(1,4) = Z_AX*(AX_HK2*HK2_U2         + AX_RT2*RT2_U2)
       VS2(1,5) = -AX
       VS2(1,7) = Z_AX*(                        AX_RT2*RT2_R2)
C
       VSR(1)   = Z_AX*(AX_RT1*RT1_RE         + AX_RT2*RT2_RE)
       VSREZ(1) = -REZC
C
      ELSE
C
C***** turbulent wall or wake part -->  set shear lag equation
C
       CQA = (1.0-UPW)*CQ1 + UPW*CQ2
       CFA = (1.0-UPW)*CF1 + UPW*CF2
       HKA = (1.0-UPW)*HK1 + UPW*HK2
       SA  = (1.0-UPW)*S1  + UPW*S2
     &     + (1.0-UPW)*DS1
C
       TA  = 0.5*(T1  + T2 )
       DA  = 0.5*(D1  + D2 )
       UA  = 0.5*(U1  + U2 )
       DEA = 0.5*(DE1 + DE2)
       RTA = 0.5*(RT1 + RT2)
C
       UNA = 0.5*(UN1 + UN2)
C
       IF(UNA .LT. 0.0) THEN
        BET = 0.0
        IF(ITYP.EQ.3) BET = 7.0     ! convex 
       ELSE
        BET = 4.5     ! concave
       ENDIF
C
C
C%%%-- disable streamline-curvature correction to dissipation length
       BET = 0.0
c
c
C----- set scaling factor for dissipation length
       ALD     = -BET*0.667*UNA*(DEA/DA - 0.7)*DEA/UA  +  1.0
       ALD_UNA = -BET*0.667    *(DEA/DA - 0.7)*DEA/UA
       ALD_DEA = -BET*0.667*UNA*(DEA/DA - 0.7)    /UA
     &           -BET*0.667*UNA*(1.0/DA      )*DEA/UA
       ALD_DA  =  BET*0.667*UNA*(DEA/DA**2   )*DEA/UA
       ALD_UA  =  BET*0.667*UNA*(DEA/DA - 0.7)*DEA/UA**2
       ALD_TA  = 0.0
C
cC----- alternative form using 10xTheta instead of Delta
c       ALD     = -BET*0.667*UNA*(10.0*TA/DA - 0.7)*10.0*TA/UA  +  1.0
c       ALD_UNA = -BET*0.667    *(10.0*TA/DA - 0.7)*10.0*TA/UA
c       ALD_TA  = -BET*0.667*UNA*(10.0*TA/DA - 0.7)*10.0   /UA
c     &           -BET*0.667*UNA*(10.0   /DA      )*10.0*TA/UA
c       ALD_DA  =  BET*0.667*UNA*(10.0*TA/DA**2   )*10.0*TA/UA
c       ALD_UA  =  BET*0.667*UNA*(10.0*TA/DA - 0.7)*10.0*TA/UA**2
c       ALD_DEA =  0.0
C
       DLC = 1.0
       IF(ITYP.EQ.3) DLC = DLCON
C
       IF(ITYP.EQ.3) THEN
C------ increased dissipation length in wake (decrease its reciprocal)
        ALD     = DLCON*ALD
        ALD_UNA = DLCON*ALD_UNA
        ALD_DEA = DLCON*ALD_DEA
        ALD_DA  = DLCON*ALD_DA
        ALD_UA  = DLCON*ALD_UA
        ALD_TA  = DLCON*ALD_TA
       ENDIF
C
C----- limit ALD to ALDMIN...ALDMAX
       IF(ALD .LT. 0.99999) THEN
C
        TRAT = TANH((ALD-1.0)/(ALDMIN-1.0))
        ALD  = (ALDMIN-1.0)*TRAT  +  1.0
        ALD_UNA = (1.0 - TRAT**2)*ALD_UNA
        ALD_DEA = (1.0 - TRAT**2)*ALD_DEA
        ALD_DA  = (1.0 - TRAT**2)*ALD_DA
        ALD_UA  = (1.0 - TRAT**2)*ALD_UA
        ALD_TA  = (1.0 - TRAT**2)*ALD_TA
C
       ELSE IF(ALD .GT. 1.00001) THEN
C
        TRAT = TANH((ALD-1.0)/(ALDMAX-1.0))
        ALD  = (ALDMAX-1.0)*TRAT  +  1.0
        ALD_UNA = (1.0 - TRAT**2)*ALD_UNA
        ALD_DEA = (1.0 - TRAT**2)*ALD_DEA
        ALD_DA  = (1.0 - TRAT**2)*ALD_DA
        ALD_UA  = (1.0 - TRAT**2)*ALD_UA
        ALD_TA  = (1.0 - TRAT**2)*ALD_TA
C
       ENDIF
C
       ALD_T1 = 0.5*(ALD_DEA*DE1_T1 + ALD_TA)
       ALD_D1 = 0.5*(ALD_DEA*DE1_D1 + ALD_DA)
       ALD_U1 = 0.5*(ALD_DEA*DE1_U1 + ALD_UA)
       ALD_T2 = 0.5*(ALD_DEA*DE2_T2 + ALD_TA)
       ALD_D2 = 0.5*(ALD_DEA*DE2_D2 + ALD_DA)
       ALD_U2 = 0.5*(ALD_DEA*DE2_U2 + ALD_UA)
C
       ALD_UN1 = 0.5*ALD_UNA
       ALD_UN2 = 0.5*ALD_UNA
C
C
C----- set and linearize  equilibrium 1/Ue dUe/dx   ...  NEW  1 May 93
       GCC = 0.0
       HKC = HKA - 1.0
       HKC_HKA = 1.0
       HKC_RTA = 0.0
       IF(ITYP.EQ.2) THEN
        GCC = GCCON
        HKC     = HKA - 1.0 - GCC/RTA
        HKC_HKA = 1.0
        HKC_RTA =             GCC/RTA**2
        IF(HKC .LT. 0.01) THEN
         HKC = 0.01
         HKC_HKA = 0.0
         HKC_RTA = 0.0
        ENDIF
       ENDIF
C
       HR     = HKC     / (GACON*ALD*HKA)
       HR_HKA = HKC_HKA / (GACON*ALD*HKA) - HR / HKA
       HR_RTA = HKC_RTA / (GACON*ALD*HKA)
       HR_ALD =                           - HR / ALD
C
       UQ     = (0.5*CFA - HR**2) / (GBCON*DA)
       UQ_CFA =  0.5              / (GBCON*DA)
       UQ_HKA =   -2.0*HR*HR_HKA  / (GBCON*DA)
       UQ_RTA =   -2.0*HR*HR_RTA  / (GBCON*DA)
       UQ_ALD =   -2.0*HR*HR_ALD  / (GBCON*DA)
       UQ_DA  = -UQ/DA
C
       UQ_UPW = UQ_CFA*(CF2-CF1) + UQ_HKA*(HK2-HK1)
C
       UQ_T1 = (1.0-UPW)*(UQ_CFA*CF1_T1 + UQ_HKA*HK1_T1) + UQ_UPW*UPW_T1
       UQ_D1 = (1.0-UPW)*(UQ_CFA*CF1_D1 + UQ_HKA*HK1_D1) + UQ_UPW*UPW_D1
       UQ_U1 = (1.0-UPW)*(UQ_CFA*CF1_U1 + UQ_HKA*HK1_U1) + UQ_UPW*UPW_U1
       UQ_R1 = (1.0-UPW)*(UQ_CFA*CF1_R1                )
       UQ_T2 =      UPW *(UQ_CFA*CF2_T2 + UQ_HKA*HK2_T2) + UQ_UPW*UPW_T2
       UQ_D2 =      UPW *(UQ_CFA*CF2_D2 + UQ_HKA*HK2_D2) + UQ_UPW*UPW_D2
       UQ_U2 =      UPW *(UQ_CFA*CF2_U2 + UQ_HKA*HK2_U2) + UQ_UPW*UPW_U2
       UQ_R2 =      UPW *(UQ_CFA*CF2_R2                )
       UQ_RE = (1.0-UPW)* UQ_CFA*CF1_RE
     &       +      UPW * UQ_CFA*CF2_RE
C
       UQ_T1 = UQ_T1 + UQ_ALD*ALD_T1             + 0.5*UQ_RTA*RT1_T1
       UQ_D1 = UQ_D1 + UQ_ALD*ALD_D1 + 0.5*UQ_DA
       UQ_U1 = UQ_U1 + UQ_ALD*ALD_U1             + 0.5*UQ_RTA*RT1_U1
       UQ_R1 = UQ_R1                             + 0.5*UQ_RTA*RT1_R1
       UQ_T2 = UQ_T2 + UQ_ALD*ALD_T2             + 0.5*UQ_RTA*RT2_T2
       UQ_D2 = UQ_D2 + UQ_ALD*ALD_D2 + 0.5*UQ_DA
       UQ_U2 = UQ_U2 + UQ_ALD*ALD_U2             + 0.5*UQ_RTA*RT2_U2
       UQ_R2 = UQ_R2                             + 0.5*UQ_RTA*RT2_R2
       UQ_RE = UQ_RE                             + 0.5*UQ_RTA*RT1_RE
     &                                           + 0.5*UQ_RTA*RT2_RE
       UQ_UN1 =        UQ_ALD*ALD_UN1
       UQ_UN2 =        UQ_ALD*ALD_UN2
C
C
       SLOG = LOG(S2/(S1+DS1))
       DXI = X2 - X1
C
       REZC = SCC*(CQA - SA*ALD)*DXI  -  DEA*2.0*SLOG
     &      + DEA*2.0*(UQ*DXI - ULOG)
C
       Z_SL = -DEA*2.0
       Z_UL = -DEA*2.0
       Z_DXI = SCC*(CQA - SA*ALD) + DEA*2.0*UQ
       Z_CQA = SCC*DXI
       Z_SA = -SCC*DXI*ALD
       Z_ALD = -SCC*SA*DXI
       Z_DEA = 2.0*(UQ*DXI - ULOG - SLOG)
       Z_UQ = DEA*2.0*DXI
C
       Z_DS1 = Z_SA*(1.0-UPW)              - Z_SL/(S1+DS1)
       Z_UPW = Z_CQA*(CQ2-CQ1) + Z_SA*(S2-S1-DS1)
       Z_DE1 = Z_DEA*0.5
       Z_DE2 = Z_DEA*0.5
       Z_UN1 = Z_UQ*UQ_UN1 + Z_ALD*ALD_UN1
       Z_UN2 = Z_UQ*UQ_UN2 + Z_ALD*ALD_UN2
       Z_T1  = Z_UQ*UQ_T1  + Z_ALD*ALD_T1            + Z_DS1*DS1_T1
       Z_T2  = Z_UQ*UQ_T2  + Z_ALD*ALD_T2 
       Z_D1  = Z_UQ*UQ_D1  + Z_ALD*ALD_D1            + Z_DS1*DS1_D1
       Z_D2  = Z_UQ*UQ_D2  + Z_ALD*ALD_D2 
       Z_U1  = Z_UQ*UQ_U1  + Z_ALD*ALD_U1  - Z_UL/U1 + Z_DS1*DS1_U1
       Z_U2  = Z_UQ*UQ_U2  + Z_ALD*ALD_U2  + Z_UL/U2
       Z_R1  = Z_UQ*UQ_R1
       Z_R2  = Z_UQ*UQ_R2
       Z_RE  = Z_UQ*UQ_RE                            + Z_DS1*DS1_RE
       Z_X1  = -Z_DXI
       Z_X2  =  Z_DXI
       Z_S1  = Z_SA *(1.0-UPW)             - Z_SL/(S1+DS1)
       Z_S2  = Z_SA *     UPW              + Z_SL/S2
       Z_CQ1 = Z_CQA*(1.0-UPW)
       Z_CQ2 = Z_CQA*     UPW
C
       VS1(1,1) = Z_S1
       VS1(1,2) = Z_T1 + Z_UPW*UPW_T1 + Z_DE1*DE1_T1 + Z_CQ1*CQ1_T1
       VS1(1,3) = Z_D1 + Z_UPW*UPW_D1 + Z_DE1*DE1_D1 + Z_CQ1*CQ1_D1
       VS1(1,4) = Z_U1 + Z_UPW*UPW_U1 + Z_DE1*DE1_U1 + Z_CQ1*CQ1_U1
       VS1(1,5) = Z_X1
       VS1(1,6) = Z_UN1                              + Z_CQ1*CQ1_UN1
       VS1(1,7) = Z_R1                               + Z_CQ1*CQ1_R1
       VS2(1,1) = Z_S2
       VS2(1,2) = Z_T2 + Z_UPW*UPW_T2 + Z_DE2*DE2_T2 + Z_CQ2*CQ2_T2
       VS2(1,3) = Z_D2 + Z_UPW*UPW_D2 + Z_DE2*DE2_D2 + Z_CQ2*CQ2_D2
       VS2(1,4) = Z_U2 + Z_UPW*UPW_U2 + Z_DE2*DE2_U2 + Z_CQ2*CQ2_U2
       VS2(1,5) = Z_X2
       VS2(1,6) = Z_UN2                              + Z_CQ2*CQ2_UN2
       VS2(1,7) = Z_R2                               + Z_CQ2*CQ2_R2
C
       VSR(1)   = Z_RE + Z_CQ1*CQ1_RE + Z_CQ2*CQ2_RE
       VSREZ(1) = -REZC
C
      ENDIF
C
C
C**** Set up momentum equation
      XA = 0.5*(X1 + X2)
      TA = 0.5*(T1 + T2)
      HWA = 0.5*(DW1/T1 + DW2/T2)
C
C
      HA    = SQRT(H1*H2)
      HA_T1 = 0.5*H2/HA * H1_T1
      HA_D1 = 0.5*H2/HA * H1_D1
      HA_T2 = 0.5*H1/HA * H2_T2
      HA_D2 = 0.5*H1/HA * H2_D2
C
c
c      HA    = 0.5*(H1 + H2   )
c      HA_T1 = 0.5* H1_T1
c      HA_D1 = 0.5* H1_D1
c      HA_T2 = 0.5*      H2_T2
c      HA_D2 = 0.5*      H2_D2
c
C
C
C---- set skin friction term, using central value CFM for better accuracy
      CFX     = 0.50*CFM*XA/TA  +  0.25*(CF1*X1/T1 + CF2*X2/T2)
      CFX_XA  = 0.50*CFM   /TA
      CFX_TA  = -.50*CFM*XA/TA**2
C
      CFX_X1  = 0.25*CF1   /T1     + CFX_XA*0.5
      CFX_X2  = 0.25*CF2   /T2     + CFX_XA*0.5
      CFX_T1  = -.25*CF1*X1/T1**2  + CFX_TA*0.5
      CFX_T2  = -.25*CF2*X2/T2**2  + CFX_TA*0.5
      CFX_CF1 = 0.25*    X1/T1
      CFX_CF2 = 0.25*    X2/T2
      CFX_CFM = 0.50*    XA/TA
C
      BTMP = HA + 2.0 + HWA
C
      REZT  = TLOG  + BTMP*ULOG + RLOG  -  0.5*CFX*XLOG
      Z_CFX = -0.5*XLOG
      Z_HA  =  ULOG
      Z_HWA =  ULOG
      Z_XL  =-DDLOG * 0.5*CFX
      Z_UL  = DDLOG * BTMP
      Z_TL  = DDLOG
      Z_RL  = DDLOG
C
      Z_CFM = Z_CFX*CFX_CFM
      Z_CF1 = Z_CFX*CFX_CF1
      Z_CF2 = Z_CFX*CFX_CF2
C
      Z_T1 = -Z_TL/T1 + Z_CFX*CFX_T1 + Z_HWA*0.5*(-DW1/T1**2)    
      Z_T2 =  Z_TL/T2 + Z_CFX*CFX_T2 + Z_HWA*0.5*(-DW2/T2**2)    
      Z_X1 = -Z_XL/X1 + Z_CFX*CFX_X1
      Z_X2 =  Z_XL/X2 + Z_CFX*CFX_X2
      Z_U1 = -Z_UL/U1
      Z_U2 =  Z_UL/U2
      Z_R1 = -Z_RL/R1
      Z_R2 =  Z_RL/R2
C
      VS1(2,2) = Z_HA*HA_T1 + Z_CFM*CFM_T1 + Z_CF1*CF1_T1 + Z_T1
      VS1(2,3) = Z_HA*HA_D1 + Z_CFM*CFM_D1 + Z_CF1*CF1_D1
      VS1(2,4) =              Z_CFM*CFM_U1 + Z_CF1*CF1_U1 + Z_U1
      VS1(2,5) =                                            Z_X1
      VS1(2,7) =              Z_CFM*CFM_R1 + Z_CF1*CF1_R1 + Z_R1
      VS2(2,2) = Z_HA*HA_T2 + Z_CFM*CFM_T2 + Z_CF2*CF2_T2 + Z_T2
      VS2(2,3) = Z_HA*HA_D2 + Z_CFM*CFM_D2 + Z_CF2*CF2_D2
      VS2(2,4) =              Z_CFM*CFM_U2 + Z_CF2*CF2_U2 + Z_U2
      VS2(2,5) =                                            Z_X2
      VS2(2,7) =              Z_CFM*CFM_R2 + Z_CF2*CF2_R2 + Z_R2
C
      VSR(2)   = Z_CFM*CFM_RE + Z_CF1*CF1_RE + Z_CF2*CF2_RE
      VSREZ(2) = -REZT
C
C**** Set up shape parameter equation
C
      XOT1 = X1/T1
      XOT2 = X2/T2
C
      HA  = 0.5*(H1  + H2 )
      HSA = 0.5*(HS1 + HS2)
      HCA = 0.5*(HC1 + HC2)
      HWA = 0.5*(DW1/T1 + DW2/T2)             
C
C
c      HA    = SQRT(H1*H2)
c      HA_T1 = 0.5*H2/HA * H1_T1
c      HA_D1 = 0.5*H2/HA * H1_D1
c      HA_T2 = 0.5*H1/HA * H2_T2
c      HA_D2 = 0.5*H1/HA * H2_D2
C
      HA_T1 = 0.5 * H1_T1
      HA_D1 = 0.5 * H1_D1
      HA_T2 = 0.5 * H2_T2
      HA_D2 = 0.5 * H2_D2
c
C
C------------------------------
      RTA = 0.5*(RT1 + RT2)
C
      IF(ITYP.EQ.2 .AND. (ULOG .GT. 0.0)) THEN
C----- inner dissipation dp/dx correction
       DDIX   = -ULOG*75.0/RTA
       DD_UL  = -     75.0/RTA
       DD_RTA = -DDIX/RTA
      ELSE
       DDIX   = 0.0
       DD_UL  = 0.0
       DD_RTA = 0.0
      ENDIF
C
C---- set dissipation and skin friction terms
      DIX = (1.0-UPW)*DI1*XOT1 + UPW*DI2*XOT2
      CFX = (1.0-UPW)*CF1*XOT1 + UPW*CF2*XOT2
      DIX_UPW = DI2*XOT2 - DI1*XOT1
      CFX_UPW = CF2*XOT2 - CF1*XOT1
C
      BTMP = 2.0*HCA/HSA + 1.0 - HA - HWA
C
      REZH  = HLOG + BTMP*ULOG  - DDIX  +  (0.5*CFX - DIX)*XLOG
      Z_CFX =  XLOG * 0.5
      Z_DIX = -XLOG
      Z_HCA =  ULOG * 2.0    /HSA
      Z_HSA = -ULOG * 2.0*HCA/HSA**2
      Z_HA  = -ULOG
      Z_HWA = -ULOG
      Z_RTA =               - DD_RTA
      Z_UL  = DDLOG * (BTMP - DD_UL )
      Z_HL  = DDLOG
      Z_XL  = DDLOG * (0.5*CFX - DIX)
C
C---- eliminate dependence on CFX, DIX, XOT1, XOT2, XL, UL
      Z_UPW = Z_CFX*CFX_UPW + Z_DIX*DIX_UPW
C
      Z_CF1 = (1.0-UPW)*Z_CFX*XOT1
      Z_CF2 =      UPW *Z_CFX*XOT2
      Z_DI1 = (1.0-UPW)*Z_DIX*XOT1
      Z_DI2 =      UPW *Z_DIX*XOT2
C
      Z_T1 = (1.0-UPW)*(Z_CFX*CF1 + Z_DIX*DI1)*(-XOT1/T1)
      Z_T2 =      UPW *(Z_CFX*CF2 + Z_DIX*DI2)*(-XOT2/T2)
      Z_X1 = (1.0-UPW)*(Z_CFX*CF1 + Z_DIX*DI1)/ T1        - Z_XL/X1
      Z_X2 =      UPW *(Z_CFX*CF2 + Z_DIX*DI2)/ T2        + Z_XL/X2
      Z_U1 =                                              - Z_UL/U1
      Z_U2 =                                                Z_UL/U2
C
C---- eliminate dependence on HSA, HL, HWA, RTA
      Z_HS1 = 0.5*Z_HSA - Z_HL/HS1
      Z_HS2 = 0.5*Z_HSA + Z_HL/HS2
C
      Z_T1 = Z_T1 + Z_HWA*0.5*(-DW1/T1**2)  +  Z_RTA*0.5*RT1_T1
      Z_T2 = Z_T2 + Z_HWA*0.5*(-DW2/T2**2)  +  Z_RTA*0.5*RT2_T2
      Z_U1 = Z_U1                           +  Z_RTA*0.5*RT1_U1
      Z_U2 = Z_U2                           +  Z_RTA*0.5*RT2_U2
      Z_R1 =                                   Z_RTA*0.5*RT1_R1
      Z_R2 =                                   Z_RTA*0.5*RT2_R2
      Z_RE =                                   Z_RTA*0.5*RT1_RE
     &                                      +  Z_RTA*0.5*RT2_RE
C
C---- set all derivatives
      VS1(3,1) =                               Z_DI1*DI1_S1
      VS1(3,2) = Z_HS1*HS1_T1 + Z_CF1*CF1_T1 + Z_DI1*DI1_T1 + Z_T1
      VS1(3,3) = Z_HS1*HS1_D1 + Z_CF1*CF1_D1 + Z_DI1*DI1_D1
      VS1(3,4) = Z_HS1*HS1_U1 + Z_CF1*CF1_U1 + Z_DI1*DI1_U1 + Z_U1
      VS1(3,5) =                                              Z_X1
      VS1(3,7) = Z_HS1*HS1_R1 + Z_CF1*CF1_R1 + Z_DI1*DI1_R1 + Z_R1
      VS2(3,1) =                               Z_DI2*DI2_S2
      VS2(3,2) = Z_HS2*HS2_T2 + Z_CF2*CF2_T2 + Z_DI2*DI2_T2 + Z_T2
      VS2(3,3) = Z_HS2*HS2_D2 + Z_CF2*CF2_D2 + Z_DI2*DI2_D2
      VS2(3,4) = Z_HS2*HS2_U2 + Z_CF2*CF2_U2 + Z_DI2*DI2_U2 + Z_U2
      VS2(3,5) =                                              Z_X2
      VS2(3,7) = Z_HS2*HS2_R2 + Z_CF2*CF2_R2 + Z_DI2*DI2_R2 + Z_R2
C
      VS1(3,2) = VS1(3,2) + 0.5*Z_HCA*HC1_T1 + Z_HA*HA_T1 + Z_UPW*UPW_T1
      VS1(3,3) = VS1(3,3) + 0.5*Z_HCA*HC1_D1 + Z_HA*HA_D1 + Z_UPW*UPW_D1
      VS1(3,4) = VS1(3,4) + 0.5*Z_HCA*HC1_U1              + Z_UPW*UPW_U1
      VS1(3,7) = VS1(3,7) + 0.5*Z_HCA*HC1_R1
      VS2(3,2) = VS2(3,2) + 0.5*Z_HCA*HC2_T2 + Z_HA*HA_T2 + Z_UPW*UPW_T2
      VS2(3,3) = VS2(3,3) + 0.5*Z_HCA*HC2_D2 + Z_HA*HA_D2 + Z_UPW*UPW_D2
      VS2(3,4) = VS2(3,4) + 0.5*Z_HCA*HC2_U2              + Z_UPW*UPW_U2
      VS2(3,7) = VS2(3,7) + 0.5*Z_HCA*HC2_R2
C
      VSR(3)   = Z_HS1*HS1_RE + Z_CF1*CF1_RE + Z_DI1*DI1_RE
     &         + Z_HS2*HS2_RE + Z_CF2*CF2_RE + Z_DI2*DI2_RE
     &         + 0.5*Z_HCA*HC1_RE
     &         + 0.5*Z_HCA*HC2_RE
     &         + Z_RE
      VSREZ(3) = -REZH
C
      RETURN
      END ! BLDIF



      SUBROUTINE AXSET( HK1,    T1,    RT1,    A1,
     &                  HK2,    T2,    RT2,    A2, ACRIT,
     &           AX, AX_HK1, AX_T1, AX_RT1, AX_A1,
     &               AX_HK2, AX_T2, AX_RT2, AX_A2, AX_AC )
C----------------------------------------------------------
C     Returns average amplification AX over interval 1..2
C----------------------------------------------------------
C
cC==========================
cC---- 1st-order -- based on "1" quantities only
c      CALL DAMPL( HK1, T1, RT1, AX1, AX1_HK1, AX1_T1, AX1_RT1 )
c      AX2_HK2 = 0.0
c      AX2_T2  = 0.0
c      AX2_RT2 = 0.0
cC
c      AX1_A1 = 0.0
c      AX2_A2 = 0.0
cC
c      AX     = AX1
c      AX_AX1 = 1.0
c      AX_AX2 = 0.0
cC
c      ARG = MIN( 20.0*(ACRIT-A1) , 20.0 )
c      EXN    = EXP(-ARG)
c      EXN_A1 =  20.0*EXN
c      EXN_A2 = 0.
c      EXN_AC = -20.0*EXN
cC
c      DAX    = EXN   * 0.0004/T1
c      DAX_A1 = EXN_A1* 0.0004/T1
c      DAX_A2 = 0.
c      DAX_AC = EXN_AC* 0.0004/T1
c      DAX_T1 = -DAX/T1
c      DAX_T2 = 0.
C
C==========================
C---- 2nd-order
      CALL DAMPL( HK1, T1, RT1, AX1, AX1_HK1, AX1_T1, AX1_RT1 )
      CALL DAMPL( HK2, T2, RT2, AX2, AX2_HK2, AX2_T2, AX2_RT2 )
C
CC---- simple-average version
C      AX = 0.5*(AX1 + AX2)
C      IF(AX .LE. 0.0) THEN
C       AX = 0.0
C       AX_AX1 = 0.0
C       AX_AX2 = 0.0
C      ELSE
C       AX_AX1 = 0.5
C       AX_AX2 = 0.5
C      ENDIF
C
C---- rms-average version (seems a little better on coarse grids)
      AXSQ = 0.5*(AX1**2 + AX2**2)
      IF(AXSQ .LE. 0.0) THEN
       AX = 0.0
       AX_AX1 = 0.0
       AX_AX2 = 0.0
      ELSE
       AX = SQRT(AXSQ)
       AX_AX1 = 0.5*AX1/AX
       AX_AX2 = 0.5*AX2/AX
      ENDIF
C
C----- small additional term to ensure  dN/dx > 0  near  N = Ncrit
ccc    AAFAC = 20.0
       AAFAC = 15.0
       ARG = MIN( AAFAC*(ACRIT-0.5*(A1+A2)) , AAFAC )
       IF(ARG.LE.0.0) THEN
        EXN    = 1.0
        EXN_A1 = 0.
        EXN_A2 = 0.
        EXN_AC = 0.
       ELSE
        EXN    = EXP(-ARG)
        EXN_A1 =  AAFAC*0.5*EXN
        EXN_A2 =  AAFAC*0.5*EXN
        EXN_AC = -AAFAC    *EXN
       ENDIF
C
       DAX    = EXN    * 0.002/(T1+T2)
       DAX_A1 = EXN_A1 * 0.002/(T1+T2)
       DAX_A2 = EXN_A2 * 0.002/(T1+T2)
       DAX_AC = EXN_AC * 0.002/(T1+T2)
       DAX_T1 = -DAX/(T1+T2)
       DAX_T2 = -DAX/(T1+T2)
C
c        DAX    = 0.
c        DAX_A1 = 0.
c        DAX_A2 = 0.
c        DAX_AC = 0.
c        DAX_T1 = 0.
c        DAX_T2 = 0.
C==========================
C
      AX     = AX             + DAX
C
      AX_HK1 = AX_AX1*AX1_HK1
      AX_T1  = AX_AX1*AX1_T1  + DAX_T1
      AX_RT1 = AX_AX1*AX1_RT1
      AX_A1  =                  DAX_A1
C
      AX_HK2 = AX_AX2*AX2_HK2
      AX_T2  = AX_AX2*AX2_T2  + DAX_T2
      AX_RT2 = AX_AX2*AX2_RT2
      AX_A2  =                  DAX_A2
C
      AX_AC  =                  DAX_AC
C
      RETURN
      END

 
 
      SUBROUTINE DAMPL( HK, TH, RT, AX, AX_HK, AX_TH, AX_RT )
C==============================================================
C     Amplification rate routine for envelope e^n method.
C     Reference: 
C                Drela, M., Giles, M.,
C               "Viscous/Inviscid Analysis of Transonic and 
C                Low Reynolds Number Airfoils", 
C                AIAA Journal, Oct. 1987.
C
C     NEW VERSION.   March 1991
C          - m(H) correlation made valid up to H=20
C          - non-similar profiles are used for H > 5 
C            in lieu of Falkner-Skan profiles.  These are
C            more representative of separation bubble profiles.
C--------------------------------------------------------------
C
C     input :   HK     kinematic shape parameter
C               TH     momentum thickness
C               RT     momentum-thickness Reynolds number
C
C     output:   AX     envelope spatial amplification rate
C               AX_(.) sensitivity of AX to parameter (.)
C
C
C     Usage: The log of the envelope amplitude N(x) is 
C            calculated by integrating AX (= dN/dx) with 
C            respect to the streamwise distance x.
C                      x
C                     /
C              N(x) = | AX(H(x),Th(x),Rth(x)) dx
C                     /
C                      0
C            The integration can be started from the leading
C            edge since AX will be returned as zero when RT
C            is below the critical Rtheta.  Transition occurs
C            when N(x) reaches Ncrit (Ncrit= 9 is "standard").
C==============================================================
      IMPLICIT REAL (A-H,M,O-Z)
ccc   DATA DGR / 0.04 /
ccc   DATA DGR / 0.08 /
      DATA DGR / 0.16 /
C
      HMI = 1.0/(HK - 1.0)
      HMI_HK = -HMI**2
C
C---- log10(Critical Rth) - H   correlation for Falkner-Skan profiles
      AA    = 2.492*HMI**0.43
      AA_HK =   (AA/HMI)*0.43 * HMI_HK
C
      BB    = TANH(14.0*HMI - 9.24)
      BB_HK = (1.0 - BB*BB) * 14.0 * HMI_HK
C
      GRCRIT = AA    + 0.7*(BB + 1.0)
      GRC_HK = AA_HK + 0.7* BB_HK
C
C
      GR = LOG10(RT)
      GR_RT = 1.0 / (2.3025851*RT)
C
      IF(GR .LT. GRCRIT-DGR) THEN
C
C----- no amplification for Rtheta < Rcrit
       AX    = 0.
       AX_HK = 0.
       AX_TH = 0.
       AX_RT = 0.
C
      ELSE
C
C----- Set steep cubic ramp used to turn on AX smoothly as Rtheta 
C-     exceeds Rcrit (previously, this was done discontinuously).
C-     The ramp goes between  -DGR < log10(Rtheta/Rcrit) < DGR
C
       RNORM = (GR - (GRCRIT-DGR)) / (2.0*DGR)
       RN_HK =     -  GRC_HK       / (2.0*DGR)
       RN_RT =  GR_RT              / (2.0*DGR)
C
       IF(RNORM .GE. 1.0) THEN
        RFAC    = 1.0
        RFAC_HK = 0.
        RFAC_RT = 0.
       ELSE
        RFAC    = 3.0*RNORM**2 - 2.0*RNORM**3
        RFAC_RN = 6.0*RNORM    - 6.0*RNORM**2
C
        RFAC_HK = RFAC_RN*RN_HK
        RFAC_RT = RFAC_RN*RN_RT
       ENDIF
C
C----- amplification envelope slope correlation  dn/dRt(H)
       ARG    = 3.87*HMI    - 2.52
       ARG_HK = 3.87*HMI_HK
C
       EX    = EXP(-ARG**2)
       EX_HK = EX * (-2.0*ARG*ARG_HK)
C
       DADR    = 0.028*(HK-1.0) - 0.0345*EX
       DADR_HK = 0.028          - 0.0345*EX_HK
C
C----- m(H) correlation
       BRG = -20.0*HMI
       AF = -0.05 + 2.7*HMI -  5.5*HMI**2 + 3.0*HMI**3 + 0.1*EXP(BRG)
       AF_HMI =     2.7     - 11.0*HMI    + 9.0*HMI**2 - 2.0*EXP(BRG)
       AF_HK = AF_HMI*HMI_HK
C
C----- amplification rate  dn/dx(H,theta,Rt)
       AX    = (AF   *DADR/TH                ) * RFAC
       AX_HK = (AF_HK*DADR/TH + AF*DADR_HK/TH) * RFAC
     &       + (AF   *DADR/TH                ) * RFAC_HK
       AX_TH = -AX/TH
       AX_RT = (AF   *DADR/TH                ) * RFAC_RT
C
      ENDIF
C
      RETURN
      END ! DAMPL
 
 
 
      SUBROUTINE HKIN( H, MSQ, GAM, HK, HK_H, HK_MSQ )
      IMPLICIT REAL (A-H,M,O-Z)
C
C---- calculate kinematic shape parameter
C     (from Whitfield )
      GM1 = GAM - 1.0
      HK     = (H - 0.725*GM1*MSQ) / (1.0 + 0.2825*GM1*MSQ)
      HK_H   =  1.0                / (1.0 + 0.2825*GM1*MSQ)
      HK_GM  = (-.725 - 0.2825*HK) / (1.0 + 0.2825*GM1*MSQ)
C
      HK_MSQ = HK_GM*GM1
      HK_GAM = HK_GM*MSQ
C
      RETURN
      END ! HKIN

 
      SUBROUTINE HCT( HK, MSQ, GAM, HC, HC_HK, HC_MSQ )
      IMPLICIT REAL (A-H,M,O-Z)
C
C---- density shape parameter    (from Whitfield)
      GM1 = GAM - 1.0
      HC     = GM1*MSQ * (0.16/(HK-0.8) + 0.6275)
      HC_HK  = GM1*MSQ * (-.16/(HK-0.8)**2      )
      HC_MSQ = GM1     * (0.16/(HK-0.8) + 0.6275)
      HC_GAM =     MSQ * (0.16/(HK-0.8) + 0.6275)
C
ccc      HC     = MSQ * (0.064/(HK-0.8) + 0.251)
ccc      HC_HK  = MSQ * (-.064/(HK-0.8)**2     )
ccc      HC_MSQ =        0.064/(HK-0.8) + 0.251
C
      RETURN
      END ! HCT
 


C======================================================================
C---- new correlations
C
      SUBROUTINE DIL( HK, RT, DI, DI_HK, DI_RT )
      IMPLICIT REAL (A-H,M,O-Z)
C
C---- Laminar dissipation function  ( 2 CD/H* )     (from Falkner-Skan)
      IF(HK.LT.4.0) THEN
       DI    = ( 0.00205  *  (4.0-HK)**5.5 + 0.207 ) / RT
       DI_HK = ( -.00205*5.5*(4.0-HK)**4.5         ) / RT
      ELSE
       HKB = HK - 4.0
       DEN = 1.0 + 0.02*HKB**2
       DI    = ( -.0016  *  HKB**2  /DEN   + 0.207             ) / RT
       DI_HK = ( -.0016*2.0*HKB*(1.0/DEN - 0.02*HKB**2/DEN**2) ) / RT
      ENDIF
      DI_RT = -DI/RT
C
      RETURN
      END


      SUBROUTINE HSL( HK, RT, MSQ, HS, HS_HK, HS_RT, HS_MSQ )
      IMPLICIT REAL (A-H,M,O-Z)
C
C---- Laminar HS correlation
      IF(HK.LT.4.35) THEN
       TMP = HK - 4.35
       HS    = 0.0111*TMP**2/(HK+1.0)
     &       - 0.0278*TMP**3/(HK+1.0)  + 1.528
     &       - 0.0002*(TMP*HK)**2
       HS_HK = 0.0111*(2.0*TMP    - TMP**2/(HK+1.0))/(HK+1.0)
     &       - 0.0278*(3.0*TMP**2 - TMP**3/(HK+1.0))/(HK+1.0)
     &       - 0.0002*2.0*TMP*HK * (TMP + HK)
      ELSE
       HS    = 0.015*    (HK-4.35)**2/HK + 1.528
       HS_HK = 0.015*2.0*(HK-4.35)   /HK
     &       - 0.015*    (HK-4.35)**2/HK**2
      ENDIF
C
      HS_RT  = 0.
      HS_MSQ = 0.
C
      RETURN
      END


      SUBROUTINE CFL( HK, RT, MSQ, CF, CF_HK, CF_RT, CF_MSQ )
      IMPLICIT REAL (A-H,M,O-Z)
C
C---- Laminar skin friction function  ( Cf )    ( from Falkner-Skan )
      IF(HK.LT.5.5) THEN
       TMP = (5.5-HK)**3 / (HK+1.0)
       CF    = ( 0.0727*TMP                      - 0.07       )/RT
       CF_HK = ( -.0727*TMP*3.0/(5.5-HK) - 0.0727*TMP/(HK+1.0))/RT
      ELSE
       TMP = 1.0 - 1.0/(HK-4.5)
       CF    = ( 0.015*TMP**2      - 0.07  ) / RT
       CF_HK = ( 0.015*TMP*2.0/(HK-4.5)**2 ) / RT
      ENDIF
      CF_RT = -CF/RT
      CF_MSQ = 0.0
C
      RETURN
      END
C
C=====================================================================

 
C=====================================================================
cC---- old correlations
cC
c      SUBROUTINE DIL( HK, RT, DI, DI_HK, DI_RT )
c
cC---- Laminar dissipation function  ( 2 CD/H* )     (from Falkner-Skan)
c      IF(HK.LT.4.0) THEN
c       DI    = ( 0.00205  *  (4.0-HK)**5.5 + 0.207 ) / RT
c       DI_HK = ( -.00205*5.5*(4.0-HK)**4.5         ) / RT
c      ELSE
c       HKB = HK - 4.0
c       DEN = 1.0 + 0.02*HKB**2
c       DI    = ( -.003  *  HKB**2  /DEN   + 0.207             ) / RT
c       DI_HK = ( -.003*2.0*HKB*(1.0/DEN - 0.02*HKB**2/DEN**2) ) / RT
c      ENDIF
c      DI_RT = -DI/RT
cC
c      RETURN
c      END
c
c
c      SUBROUTINE HSL( HK, RT, MSQ, HS, HS_HK, HS_RT, HS_MSQ )
c      REAL MSQ
cC
cC---- Laminar HS correlation    ( from Falkner-Skan )
c      IF(HK.LT.4.0) THEN
c       HS    = 0.076*(HK-4.0)**2/HK + 1.515
c       HS_HK = 0.076*(1.0-16.0/HK**2)
c      ELSE
c       HS    = 0.040*(HK-4.0)**2/HK + 1.515
c       HS_HK = 0.040*(1.0-16.0/HK**2)
c      ENDIF
cC
c      HS_RT  = 0.
c      HS_MSQ = 0.
cC
c      RETURN
c      END
c
c
c      SUBROUTINE CFL( HK, RT, MSQ, CF, CF_HK, CF_RT, CF_MSQ )
c      REAL MSQ
cC
cC---- Laminar skin friction function  ( Cf )    ( from Falkner-Skan )
c      IF(HK .LT. 7.4) THEN
c       TMP = (7.4-HK)**2   / (HK-1.0)
c       CF    = ( 0.03954*TMP                         - 0.134 ) / RT
c       CF_HK = ( -.03954*TMP * (2.0/(7.4-HK) + 1.0/(HK-1.0)) ) / RT
c       ELSE
c       TMP = 1.0 - 1.4/(HK-6.0)
c       CF    = ( 0.044*TMP**2      - 0.134 ) / RT
c       CF_HK = ( 0.088*TMP*1.4/(HK-6.0)**2 ) / RT
c      ENDIF
c      CF_RT = -CF/RT
c      CF_MSQ = 0.0
cC
c      RETURN
c      END
cC
C===================================================================== 

 
      SUBROUTINE DIT( HS, US, CF, ST, DI, DI_HS, DI_US, DI_CF, DI_ST )
      IMPLICIT REAL (A-H,M,O-Z)
C
C---- Turbulent dissipation function  ( 2Cd/H* )
      DI    =  ( 0.5*CF*US + ST*ST*(1.0-US) ) * 2.0/HS
      DI_HS = -( 0.5*CF*US + ST*ST*(1.0-US) ) * 2.0/HS**2
      DI_US =  ( 0.5*CF    - ST*ST          ) * 2.0/HS
      DI_CF =  ( 0.5   *US                  ) * 2.0/HS
      DI_ST =  (            2.0*ST*(1.0-US) ) * 2.0/HS
C
      RETURN
      END ! DIT



      SUBROUTINE HST( HK, RT, MSQ, HS, HS_HK, HS_RT, HS_MSQ )
      IMPLICIT REAL (A-H,M,O-Z)
C
C---- Turbulent HS correlation
C
      DATA HSMIN, DHSINF / 1.500, 0.015 /
C
      IF(RT.GT.400.0) THEN
       HO    = 3.0 + 400.0/RT
       HO_RT =     - 400.0/RT**2
      ELSE
       HO    = 4.0
       HO_RT = 0.
      ENDIF
C
C---- limited Rtheta dependence for Rtheta < 200    MD 12/4/94
      IF(RT.GT.200.0) THEN
       RTZ    = RT
       RTZ_RT = 1.
      ELSE
       RTZ    = 200.0
       RTZ_RT = 0.
      ENDIF
C
      IF(HK.LT.HO) THEN
C----- attached branch
C=======================================================
C----- old correlation
C-     (from Swafford profiles)
c       SRT = SQRT(RT)
c       HEX = (HO-HK)**1.6
c       RTMP = 0.165 - 1.6/SRT
c       HS    = HSMIN + 4.0/RT + RTMP*HEX/HK
c       HS_HK = RTMP*HEX/HK*(-1.6/(HO-HK) - 1.0/HK)
c       HS_RT = -4.0/RT**2 + HEX/HK*0.8/SRT/RT
c     &             + RTMP*HEX/HK*1.6/(HO-HK)*HO_RT
C=======================================================
C----- new correlation  29 Nov 91
C-     (from  arctan(y+) + Schlichting  profiles)
       HR    = ( HO - HK)/(HO-1.0)
       HR_HK =      - 1.0/(HO-1.0)
       HR_RT = (1.0 - HR)/(HO-1.0) * HO_RT
       HS    = (2.0-HSMIN-4.0/RTZ)*HR**2  * 1.5/(HK+0.5) + HSMIN
     &       + 4.0/RTZ
       HS_HK =-(2.0-HSMIN-4.0/RTZ)*HR**2  * 1.5/(HK+0.5)**2
     &       + (2.0-HSMIN-4.0/RTZ)*HR*2.0 * 1.5/(HK+0.5) * HR_HK
       HS_RT = (2.0-HSMIN-4.0/RTZ)*HR*2.0 * 1.5/(HK+0.5) * HR_RT
     &       + (HR**2 * 1.5/(HK+0.5) - 1.0)*4.0/RTZ**2 * RTZ_RT
C
      ELSE
C
C----- separated branch
       GRT = LOG(RTZ)
       HDIF = HK - HO 
       RTMP = HK - HO + 4.0/GRT
       HTMP    = 0.007*GRT/RTMP**2 + DHSINF/HK
       HTMP_HK = -.014*GRT/RTMP**3 - DHSINF/HK**2
       HTMP_RT = -.014*GRT/RTMP**3 * (-HO_RT - 4.0/GRT**2/RTZ * RTZ_RT)
     &         + 0.007    /RTMP**2 / RTZ * RTZ_RT
       HS    = HDIF**2 * HTMP + HSMIN + 4.0/RTZ
       HS_HK = HDIF*2.0* HTMP
     &       + HDIF**2 * HTMP_HK
       HS_RT = HDIF**2 * HTMP_RT      - 4.0/RTZ**2 * RTZ_RT
     &       + HDIF*2.0* HTMP * (-HO_RT)
C
      ENDIF
C
C---- fudge HS slightly to make sure   HS -> 2   as   HK -> 1
C-    (unnecessary with new correlation)
c      HTF    = 0.485/9.0 * (HK-4.0)**2/HK  +  1.515
c      HTF_HK = 0.485/9.0 * (1.0-16.0/HK**2)
c      ARG = MAX( 10.0*(1.0 - HK) , -15.0 )
c      HXX = EXP(ARG)
c      HXX_HK = -10.0*HXX
cC
c      HS_HK  = (1.0-HXX)*HS_HK  +  HXX*HTF_HK
c     &       + (        -HS     +      HTF    )*HXX_HK
c      HS_RT  = (1.0-HXX)*HS_RT
c      HS     = (1.0-HXX)*HS     +  HXX*HTF
C
C---- Whitfield's minor additional compressibility correction
      FM = 1.0 + 0.014*MSQ
      HS     = ( HS + 0.028*MSQ ) / FM
      HS_HK  = ( HS_HK          ) / FM
      HS_RT  = ( HS_RT          ) / FM
      HS_MSQ = 0.028/FM  -  0.014*HS/FM
C
      RETURN
      END
 
 
 
      SUBROUTINE CFT( HK, RT, MSQ, GAM, CF, CF_HK, CF_RT, CF_MSQ )
      IMPLICIT REAL (A-H,M,O-Z)
C
C---- Turbulent skin friction function  ( Cf )    (Swafford)
      GM1 = GAM - 1.0
      FC = SQRT(1.0 + 0.5*GM1*MSQ)
      GRT = LOG(RT/FC)
      GRT = MAX(GRT,3.0)
C
      GEX = -1.74 - 0.31*HK
C
      ARG = -1.33*HK
      ARG = MAX(-20.0, ARG )
C
      THK = TANH(4.0 - HK/0.875)
C
      CFO =  0.3*EXP(ARG) * (GRT/2.3026)**GEX
      CF     = ( CFO  +  1.1E-4*(THK-1.0) ) / FC
      CF_HK  = (-1.33*CFO - 0.31*LOG(GRT/2.3026)*CFO
     &         - 1.1E-4*(1.0-THK**2) / 0.875    ) / FC
      CF_RT  = GEX*CFO/(FC*GRT) / RT
      CF_MSQ = GEX*CFO/(FC*GRT) * (-0.25*GM1/FC**2) - 0.25*GM1*CF/FC**2
C
      RETURN
      END ! CFT



      SUBROUTINE DSTSET
      IMPLICIT REAL (A-H,M,O-Z)
      INCLUDE 'MBL.INC'
C
cccc      IF(CF2.LE.0.0) THEN
C
       DS2    = 0.0
       DS2_R2 = 0.0
       DS2_U2 = 0.0
       DS2_T2 = 0.0
       DS2_D2 = 0.0
       DS2_RE = 0.0
C
cccc      ELSE
C
c       SCF = SQRT(0.5*CF2)
c       SCF_CF2 = 0.25/SCF
cC
c       DS2    = US2*SCF
c       DS2_R2 = US2*SCF_CF2*CF2_R2 + US2_R2*SCF
c       DS2_U2 = US2*SCF_CF2*CF2_U2 + US2_U2*SCF
c       DS2_T2 = US2*SCF_CF2*CF2_T2 + US2_T2*SCF
c       DS2_D2 = US2*SCF_CF2*CF2_D2 + US2_D2*SCF
c       DS2_RE = US2*SCF_CF2*CF2_RE + US2_RE*SCF
cC
c       YPLUS = 50.0
c       DS2    = (YPLUS/RT2)*T2/DE2
c       DS2_R2 = -DS2/RT2*RT2_R2
c       DS2_U2 = -DS2/RT2*RT2_U2 - DS2/DE2*DE2_U2
c       DS2_T2 = -DS2/RT2*RT2_T2 - DS2/DE2*DE2_T2 + (YPLUS/RT2)/DE2
c       DS2_D2 =                 - DS2/DE2*DE2_D2
c       DS2_RE = -DS2/RT2*RT2_RE
C
cccc      ENDIF
C
      RETURN
      END
