
      SUBROUTINE FFBC1(LFFXY)
C----------------------------------------------------------------------
C     Sets solid wall far field (wind tunnel) BC's
C     Requires an extra MAS1 global variable to be active (explicitly set)
C
C     Normally, flat walls are imposed, possibly rotated by ALFA.
C
C     If LFFXY=.TRUE., a specified y(x) is set for either wall.
C     The specified y(x) is read in as a set of x,y points from
C     file xywall.xxx, with the following format:
C
C      XTOP1  YTOP1
C      XTOP2  YTOP2
C       .      .
C       .      .
C      999.0  999.0
C      XBOT1  YBOT1
C      XBOT2  YBOT2
C       .      .
C       .      .
C
C     These points are defined in freestream-aligned coordinates,
C     centered on the blade.xxx coordinate point XCROT,YCROT 
C     (defined below), and rotated counterclockwise by +ALFA.
C
C     If either the top or bottom wall points are omitted, then
C     the default flat wall is substituted.  It is necessary that
C     the specified wall x values above extend beyond the grid,
C     otherwise the splined wall y(x) will be extrapolated, with 
C     unpredictable results.
C
C----------------------------------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MSES.INC'
      LOGICAL LFFXY
C
C
      CHARACTER*80 ARGP1, FNAME
      DIMENSION XWTOP(IX), YWTOP(IX), YWTOPS(IX)
      DIMENSION XWBOT(IX), YWBOT(IX), YWBOTS(IX)
C
      COSA = COS(ALFA)
      SINA = SIN(ALFA)
C
C---- point about which outer walls are rotated by alpha
      XCROT = 0.25
      YCROT = 0.0
ccc      XCROT = XCENT
ccc      YCROT = YCENT
C
C
C---- first assume flat walls will be used
      NWTOP = 0
      NWBOT = 0
C
      IF(LFFXY) THEN
C
C------ Try to read in XWTOP,YWTOP and XWBOT,YWBOT distributions from file
        CALL GETARG(1,ARGP1)
        FNAME = 'xywall.' // ARGP1
        OPEN(8,FILE=FNAME,STATUS='OLD',FORM='FORMATTED',ERR=50)
C
        DO K=1, IX
          READ(8,*,END=31) XWTOP(K),YWTOP(K)
          IF(XWTOP(K) .EQ. 999.0 .AND. YWTOP(K) .EQ. 999.0) GO TO 31
        ENDDO
        K = K+1
   31   CONTINUE
        NWTOP = K-1
C
        DO K=1, IX
          READ(8,*,END=32) XWBOT(K),YWBOT(K)
          IF(XWBOT(K) .EQ. 999.0 .AND. YWBOT(K) .EQ. 999.0) GO TO 32
        ENDDO
        K = K+1
   32   CONTINUE
        NWBOT = K-1
C
        CLOSE(8)
C
      ENDIF
C
   50 CONTINUE
C
C---- spline wall shapes y(x) if available
      IF(NWTOP.GE.2) CALL SPLIND(YWTOP,YWTOPS,XWTOP,NWTOP,-999.0,-999.0)
      IF(NWBOT.GE.2) CALL SPLIND(YWBOT,YWBOTS,XWBOT,NWBOT,-999.0,-999.0)
C
C
C---- top streamline
      J = JJ
      DO 110 I=2, II-1
C------ for local wall location XWLOC, set YWLOC from spline or use default
	IF(NWTOP.GE.2) THEN
         XWLOC = (X(I,J)-XCROT)*COSA + (Y(I,J)-YCROT)*SINA
         YWLOC = SEVAL(XWLOC,YWTOP,YWTOPS,XWTOP,NWTOP)
        ELSE
         YWLOC = YBTOP
        ENDIF
C
C------ set wall BC residual
        CALL CLROW(I,J)
        RES = (Y(I,J)-YCROT)*COSA - (X(I,J)-XCROT)*SINA - YWLOC
C
        A2(J,I) = COSA*NY(I,J) - SINA*NX(I,J)
        DO 104 N = 1, NBL
          DR(J,LSBLE(N),I) = COSA*NYG(I,J,N) - SINA*NXG(I,J,N)
  104   CONTINUE
        DO 108 N = 1, NPOSN
          K = KPOSN(N)
          DR(J,LPOSN(K),I) = COSA*NYP(I,J,K) - SINA*NXP(I,J,K)
  108   CONTINUE
        DR(J,LALFA,I) = -(Y(I,J)-YCROT)*SINA - (X(I,J)-XCROT)*COSA
     &                + COSA*NYA(I,J) - SINA*NXA(I,J)
        DR(J,1,I) = -RES
  110 CONTINUE
C
C---- bottom streamline
      J = 1
      DO 120 I=2, II-1
C------ for local wall location XWLOC, set YWLOC from spline or use default
	IF(NWBOT.GE.2) THEN
         XWLOC = (X(I,J)-XCROT)*COSA + (Y(I,J)-YCROT)*SINA
         YWLOC = SEVAL(XWLOC,YWBOT,YWBOTS,XWBOT,NWBOT)
        ELSE
         YWLOC = YBBOT
        ENDIF
C
C------ set wall BC residual
        CALL CLROW(I,J)
        RES = (Y(I,J)-YCROT)*COSA - (X(I,J)-XCROT)*SINA - YWLOC
C
        A2(J,I) = COSA*NY(I,J) - SINA*NX(I,J)
        DO 112 N = 1, NBL
          DR(J,LSBLE(N),I) = COSA*NYG(I,J,N) - SINA*NXG(I,J,N)
  112   CONTINUE
        DO 118 N = 1, NPOSN
          K = KPOSN(N)
          DR(J,LPOSN(K),I) = COSA*NYP(I,J,K) - SINA*NXP(I,J,K)
  118   CONTINUE
        DR(J,LALFA,I) = -(Y(I,J)-YCROT)*SINA - (X(I,J)-XCROT)*COSA
     &                + COSA*NYA(I,J) - SINA*NXA(I,J)
        DR(J,1,I) = -RES
  120 CONTINUE
C
C
C
C---- set top and bottom wall angles over inlet plane
      IO = 2
      IM = 1
      IF(NWTOP.GE.2) THEN
       J = JJ
       XWLOC = (0.5*(X(IO,J)+X(IM,J))-XCROT)*COSA
     &       + (0.5*(Y(IO,J)+Y(IM,J))-YCROT)*SINA
       SLOPE = DEVAL(XWLOC,YWTOP,YWTOPS,XWTOP,NWTOP)
       TTOP = ATAN(SLOPE)
      ELSE
       TTOP = 0.0
      ENDIF
C
      IF(NWBOT.GE.2) THEN
       J = 1
       XWLOC = (0.5*(X(IO,J)+X(IM,J))-XCROT)*COSA
     &       + (0.5*(Y(IO,J)+Y(IM,J))-YCROT)*SINA
       SLOPE = DEVAL(XWLOC,YWBOT,YWBOTS,XWBOT,NWBOT)
       TBOT = ATAN(SLOPE)
      ELSE
       TBOT = 0.0
      ENDIF
C
C---- set flow angles over inlet plane
      MSUM = 0.
      DO 130 J = 1, JJ
C------ interpolate flow angle between top and bottom
        TLOC = TBOT + (TTOP-TBOT)*MSUM
        MSUM = MSUM + MFRACT(J)
C
        SINT = SIN(TLOC + ALFA)
        COST = COS(TLOC + ALFA)
C
        CALL CLROW(1,J)
        RES = (X(IO,J) - X(IM,J))*SINT - (Y(IO,J) - Y(IM,J))*COST
        A2(J,1) = -NX(IM,J)*SINT + NY(IM,J)*COST
        C2(J,1) =  NX(IO,J)*SINT - NY(IO,J)*COST
        DO 1302 N = 1, NBL
          DR(J,LSBLE(N),1) =-NXG(IM,J,N)*SINT + NYG(IM,J,N)*COST
     &                     + NXG(IO,J,N)*SINT - NYG(IO,J,N)*COST
 1302   CONTINUE
        DO 1304 N = 1, NPOSN
          K = KPOSN(N)
          DR(J,LPOSN(K),1) =-NXP(IM,J,K)*SINT + NYP(IM,J,K)*COST
     &                     + NXP(IO,J,K)*SINT - NYP(IO,J,K)*COST
 1304   CONTINUE
        DR(J,LALFA,1) = (X(IO,J) - X(IM,J))*COST
     &                + (Y(IO,J) - Y(IM,J))*SINT
        DR(J,1,1) = -RES
 130  CONTINUE
C
C
C---- set top and bottom wall angles over inlet plane
      IO = II
      IM = II-1
      IF(NWTOP.GE.2) THEN
       J = JJ
       XWLOC = (0.5*(X(IO,J)+X(IM,J))-XCROT)*COSA
     &       + (0.5*(Y(IO,J)+Y(IM,J))-YCROT)*SINA
       SLOPE = DEVAL(XWLOC,YWTOP,YWTOPS,XWTOP,NWTOP)
       TTOP = ATAN(SLOPE)
      ELSE
       TTOP = 0.0
      ENDIF
C
      IF(NWBOT.GE.2) THEN
       J = 1
       XWLOC = (0.5*(X(IO,J)+X(IM,J))-XCROT)*COSA
     &       + (0.5*(Y(IO,J)+Y(IM,J))-YCROT)*SINA
       SLOPE = DEVAL(XWLOC,YWBOT,YWBOTS,XWBOT,NWBOT)
       TBOT = ATAN(SLOPE)
      ELSE
       TBOT = 0.0
      ENDIF
C
C---- set flow angles over outlet plane
      MSUM = 0.
      DO 140 J = 1, JJ
C------ interpolate flow angle between top and bottom
        TLOC = TBOT + (TTOP-TBOT)*MSUM
        MSUM = MSUM + MFRACT(J)
C
        SINT = SIN(TLOC + ALFA)
        COST = COS(TLOC + ALFA)
C
        CALL CLROW(II,J)
        RES = (X(IO,J) - X(IM,J))*SINT - (Y(IO,J) - Y(IM,J))*COST
        A2(J,II) = NX(IO,J)*SINT - NY(IO,J)*COST
        B2(J,II) =-NX(IM,J)*SINT + NY(IM,J)*COST
        DO 1402 N = 1, NBL
          DR(J,LSBLE(N),II) = NXG(IO,J,N)*SINT - NYG(IO,J,N)*COST
     &                      - NXG(IM,J,N)*SINT + NYG(IM,J,N)*COST
 1402   CONTINUE
        DO 1404 N = 1, NPOSN
          K = KPOSN(N)
          DR(J,LPOSN(K),II) = NXP(IO,J,K)*SINT - NYP(IO,J,K)*COST
     &                      - NXP(IM,J,K)*SINT + NYP(IM,J,K)*COST
 1404   CONTINUE
        DR(J,LALFA,II) = (X(IO,J)-X(IM,J))*COST
     &                 + (Y(IO,J)-Y(IM,J))*SINT
        DR(J,1,II) = -RES
 140  CONTINUE
C
      RETURN
      END ! FFBC1

 
  
      SUBROUTINE FFBC2(LFFCP)
C-------------------------------------------------------------------------
C     Sets  vortex + source + doublet  far field  BC's.
C
C     If LFFCP=.FALSE., the farfield potential singularity 
C     expansion is used to derive pressures which are 
C     imposed on the upper and lower steamlines, and to
C     derive flow angles which are imposed at the inlet 
C     and outlet planes.
C
C     If LFFCP=.TRUE., the pressure distributions on the 
C     upper and lower streamlines are instead taken from 
C     a "trace" in the domain along which the pressure is 
C     specified.  The upper and lower trace shapes Y(X) 
C     and their pressure distribution CP(X) are input via 
C     the file FFBC.xxx with the following format:
C
C     CLspec
C     Xupp1  Yupp1  CPupp1
C     Xupp2  Yupp2  CPupp2
C     Xupp3  Yupp3  CPupp3
C       .      .
C       .      .
C     999.0  999.0
C     Xlow1  Ylow1  CPlow1
C     Xlow2  Ylow2  CPlow2
C     Xlow3  Ylow3  CPlow3
C       .      .
C       .      .
C     999.0  999.0   (optional)
C
C     X and Y are x/c,y/c relative to the airfoil chord line,
C     and x/c must increase monotonically since it is used to
C     as a spline parameter for the Y and CP distributions.
C     If the number of upper and/or lower points is less than 
C     two, then the usual farfield pressures are imposed there.
C     It is thus possible to specify pressures on either the 
C     upper or the lower streamline alone.
C
C     The pressure Pspec imposed at a grid node on the upper 
C     or lower streamline is taken from the CP(X) spline at
C     the node's x/c location.  The difference between the
C     node's y/c position and Y(X) is used to correct the
C     CP(X) using the known dp/dy at the grid node:
C
C      Pspec(x/c,y/x)  =  Pinf  +  q CP(X)  +  dp/dy [y/x - Y(X)]   (*)
C
C     This linear correction allows some flexibility in choosing 
C     the domain size, since the upper and lower streamlines don't 
C     exactly have to lie on the trace Y(X) where the pressures 
C     are specified.  Naturally, the dp/dy correction term 
C     will be inaccurate if the trace and the streamline are
C     far apart.
C     
C     The CLspec value on the first line of FFBC.xxx is the 
C     airfoil CL corresponding to the specified CP distributions.  
C     It is needed to set up the farfield boundary conditions 
C     in the calculation.  The pressure P actually imposed at
C     each grid node on the outermost streamlines is given by
C
C        P  =  Pspec  -  P(Gspec) + P(G)                   (**)
C
C     where Pspec is the specified pressure defined by (*),
C     P(Gspec) is the farfield pressure corresponding to 
C     CLspec (actually, the corresponding circulation 
C     Gspec = CLspec/2), and P(G) is the pressure corresponding 
C     to the ISES circulation variable G (Fortran name: CIRC).  
C     Note that the two last terms in (**) will cancel exactly if 
C     ISES is run in the specified-CL mode, and the CL is specified 
C     to be the same as CLspec above.  The reason for including the 
C     last two terms in (**) is to introduce the circulation variable 
C     G into the equation system.  This DOF is necessary to allow 
C     the Kutta condition to be set exactly.
C
C     If LFFCP=.TRUE., the doublet strengths in the farfield expansion
C     are not updated, but are kept frozen at the initial guess or 
C     previously-converged values.  This is necessary since the doublet 
C     strengths are updated explicitly from the geometry of the outer 
C     streamlines, and can easily attain very non-physical values if 
C     the specified CP distributions do not closely correspond to an 
C     infinite domain.
C
C     Note:  In this routine, the file FFBC.xxx is read and 
C     processed every Newton iteration.  Although this incurs
C     extra disk traffic, it is a small price to pay for the
C     simplicity of keeping everything local.
C-------------------------------------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MSES.INC'
      LOGICAL LFFCP
C
      DIMENSION CIRCB(NBX), XCIRCB(NBX), YCIRCB(NBX)
      DIMENSION DOUX_NP(NPOSX), DOUY_NP(NPOSX)
C
C
C==========================================================
      CHARACTER*80 FNAME, ARGP1
      DIMENSION XFF(IX,2), YFF(IX,2),YFFX(IX,2), CPFF(IX,2),CPFFX(IX,2)
      DIMENSION NFF(2)
      LOGICAL LCPSET(2)
C
      LCPSET(1) = .FALSE.
      LCPSET(2) = .FALSE.
C
      IF(LFFCP) THEN
C
C---- Read in FFBC.xxx for experimental CL and wall Cp distributions
C
      CALL GETARG(1,ARGP1)
      FNAME = 'ffbc.'//ARGP1
      OPEN(11,FILE=FNAME,STATUS='OLD',FORM='FORMATTED',ERR=19)
C
      READ(11,*) CLFF
      CIRCFF = 0.5*CLFF
C
      DO 10 IFF=1, IX-1
        READ(11,*,END=11) XFF(IFF,1), YFF(IFF,1), CPFF(IFF,1)
        IF(XFF(IFF,1) .EQ. 999.0) GO TO 11
   10 CONTINUE
      WRITE(*,*) 'FFBC2: Farfield Cp array overflow.'
   11 CONTINUE
      NFF(1) = IFF-1
      LCPSET(1) = NFF(1) .GT. 1
C
      DO 15 IFF=1, IX-1
        READ(11,*,END=16) XFF(IFF,2), YFF(IFF,2), CPFF(IFF,2)
        IF(XFF(IFF,2) .EQ. 999.0) GO TO 16
 15   CONTINUE
      WRITE(*,*) 'FFBC2: Farfield Cp array overflow.'
 16   CONTINUE
      NFF(2) = IFF-1
      LCPSET(2) = NFF(2) .GT. 1
C
      CLOSE(11)
C
      IF(LCPSET(1)) THEN
       CALL SPLINE(CPFF(1,1),CPFFX(1,1),XFF(1,1),NFF(1))
       CALL SPLINE( YFF(1,1), YFFX(1,1),XFF(1,1),NFF(1))
      ENDIF
C
      IF(LCPSET(2)) THEN
       CALL SPLINE(CPFF(1,2),CPFFX(1,2),XFF(1,2),NFF(2))
       CALL SPLINE( YFF(1,2), YFFX(1,2),XFF(1,2),NFF(2))
      ENDIF
C
 19   CONTINUE
C
      ENDIF
C=========================================================
C
      COSA = COS(ALFA)
      SINA = SIN(ALFA)
C
C---- calculate far-field source strength from entropy and BL wake(s)
C
C---- First, the element wake(s)
      SRCE = 0.
      DO 20 N = 1, NBL
        J1 = JS1(N)
        J2 = JS2(N)
        SRCE = SRCE + Y(II,J1) - Y(II,J2)
   20 CONTINUE
C
C---- Then, shock entropy wakes for SETUP case     
      IF(ISSET.EQ.1) THEN
C
       I = II-1
       DO 25 J=1, JJ-1
         IF(JSTAG(J).GT.0) GO TO 25
         P1 = GM1/GAM * R(1,J)*(HINF - 0.5*Q(1,J)**2)
         P2 = GM1/GAM * R(I,J)*(HINF - 0.5*Q(I,J)**2)
         PST1 = P1 / (1.0 - 0.5/HINF*Q(1,J)**2)**(GAM/GM1)
         PST2 = P2 / (1.0 - 0.5/HINF*Q(I,J)**2)**(GAM/GM1)
         PX1 = (PINF/PST1)**(GM1/GAM)
         PX2 = (PINF/PST2)**(GM1/GAM)
         USQ1 = MAX( 2.0*HINF*(1.0 - PX1) , 0.0 )
         USQ2 = MAX( 2.0*HINF*(1.0 - PX2) , 0.0 )
         RU1 = GAM*PINF/(GM1*HINF*PX1) * SQRT(USQ1)
         RU2 = GAM*PINF/(GM1*HINF*PX2) * SQRT(USQ2)
         IF(RU1.EQ.0.0 .OR. RU2.EQ.0.0) THEN
           WRITE(*,*) 'SETBC: Negative mass flux transient, J = ',J
         ELSE
           SRCE = SRCE + (1.0/RU2 - 1.0/RU1)*M(J)
         ENDIF
   25  CONTINUE
C
      ENDIF
C
C==================================
C
C---- Vortex + Doublet farfield
      IF(ICOUNT.GT.2 .AND. .NOT.LFFCP) CALL DCORR
C
C==================================
C
C---- Vortex-only farfield... set doublet strengths to zero
ccc      DOUX = 0.0
ccc      DOUY = 0.0
C
C==================================
C
C---- set sensitivities to farfield doublets wrt element position
      CALL ELCIRC(CIRCB,XCIRCB,YCIRCB)
      DO 30 IPOS=1, NPOSN
        K = KPOSN(IPOS)
        DOUX_NP(K) = 0.0
        DOUY_NP(K) = 0.0
        DO 302 NN=1, NPOSEL(K)
          N = NBPOS(NN,K)
          IF(ABPOS(NN,K).EQ.0.0) THEN
            DOUX_NP(K) = DOUX_NP(K)
     &       + (-XBPOS(NN,K)*SINA + YBPOS(NN,K)*COSA)*CIRCB(N)
            DOUY_NP(K) = DOUY_NP(K)
     &       + (-XBPOS(NN,K)*COSA - YBPOS(NN,K)*SINA)*CIRCB(N)
          ELSE
            DOUX_NP(K) = DOUX_NP(K)
     &       + (-(XCIRCB(N) - XBPOS(NN,K)*CIRCB(N))*SINA
     &          +(YCIRCB(N) - YBPOS(NN,K)*CIRCB(N))*COSA)*ABPOS(NN,K)
            DOUY_NP(K) = DOUY_NP(K)
     &       + (-(XCIRCB(N) - XBPOS(NN,K)*CIRCB(N))*COSA
     &          -(YCIRCB(N) - YBPOS(NN,K)*CIRCB(N))*SINA)*ABPOS(NN,K)
          ENDIF
 302    CONTINUE
 30   CONTINUE
C
C---- set up top and bottom streamline source and vortex conditions
      MSQ_MS = 1.0/MS_MSQ
C
      DPINF = PINF - PSTOUT
      DO 50 I=2, II-1
        J = JJ
        XXX = X(I,J) - XCENT
        YYY = Y(I,J) - YCENT
        CALL PIFAR(XXX, YYY, GAM, PSTOUT, HINF,
     &             CIRC, ALFA, MINF, QINF, SRCE, DOUX, DOUY,
     &             PIFF, P_X, P_Y, P_CIRC, P_ALFA, P_MSQ, P_QINF,
     &                   P_SRCE, P_DOUX, P_DOUY )
        P_MS = (P_MSQ + P_QINF*QI_MSQ)*MSQ_MS
        A2(J,I) = A2(J,I) - 2.0*(P_X*NX(I,J) + P_Y*NY(I,J))
        DR(J,LCIRC,I) = DR(J,LCIRC,I) - 2.0*P_CIRC
        DR(J,LALFA,I) = DR(J,LALFA,I) - 2.0*P_ALFA
        DR(J,LMASS,I) = DR(J,LMASS,I) - 2.0*P_MS
        DO 42 N = 1, NBL
          DR(J,LSBLE(N),I) = DR(J,LSBLE(N),I)
     &                     - 2.0*(P_X*NXG(I,J,N) + P_Y*NYG(I,J,N))
   42   CONTINUE
        DO 43 N = 1, NPOSN
          K = KPOSN(N)
          DR(J,LPOSN(K),I) = DR(J,LPOSN(K),I)
     &                     - 2.0*( P_X*NXP(I,J,K)
     &                           + P_Y*NYP(I,J,K)
     &                           + P_DOUX*DOUX_NP(K)
     &                           + P_DOUY*DOUY_NP(K) )
   43   CONTINUE
        DR(J,LALFA,I) = DR(J,LALFA,I)
     &                - 2.0*( P_X*NXA(I,J)
     &                      + P_Y*NYA(I,J) )
        DR(J,1,I) = DR(J,1,I) + 2.0*PIFF
C
        IF(LCPSET(1)) THEN
C------- add terms to residual to drive Cp to prescribed value
         CPFORC = SEVAL(X(I,J),CPFF(1,1),CPFFX(1,1),XFF(1,1),NFF(1))
         CPF_X  = DEVAL(X(I,J),CPFF(1,1),CPFFX(1,1),XFF(1,1),NFF(1))
         YFORC  = SEVAL(X(I,J), YFF(1,1), YFFX(1,1),XFF(1,1),NFF(1))
         YF_X   = DEVAL(X(I,J), YFF(1,1), YFFX(1,1),XFF(1,1),NFF(1))
         CALL PIFAR(XXX, YYY, GAM, PSTOUT, HINF,
     &          CIRCFF, ALFA, MINF, QINF, SRCE, DOUX, DOUY,
     &          PICIRC, PC_X, PC_Y, PC_CIRCFF, PC_ALFA, PC_MSQ, PC_QINF,
     &                  PC_SRCE, PC_DOUX, PC_DOUY )
         PC_MS = (PC_MSQ + PC_QINF*QI_MSQ)*MSQ_MS
C
         PIFORC = DPINF + QU*CPFORC + P_Y*(Y(I,J)-YFORC)
         PIF_X  =         QU*CPF_X  + P_Y*(      -YF_X )
         PIF_Y  =                     P_Y
         PIF_MS = (PI_MSQ + QU_MSQ*CPFORC)*MSQ_MS
C
         A2(J,I) = A2(J,I) - 2.0*(PIF_X - PC_X)*NX(I,J)
     &                     - 2.0*(PIF_Y - PC_Y)*NY(I,J)
         DR(J,LMASS,I) = DR(J,LMASS,I) - 2.0*(PIF_MS - PC_MS  )
         DR(J,LALFA,I) = DR(J,LALFA,I) - 2.0*(       - PC_ALFA)
         DO 44 N=1, NBL
           DR(J,LSBLE(N),I) = DR(J,LSBLE(N),I)
     &                      - 2.0*(PIF_X - PC_X)*NXG(I,J,N)
     &                      - 2.0*(PIF_Y - PC_Y)*NYG(I,J,N)
 44      CONTINUE
C
         DO 45 N = 1, NPOSN
           K = KPOSN(N)
           DR(J,LPOSN(K),I) = DR(J,LPOSN(K),I)
     &                      - 2.0*(PIF_X - PC_X)*NXP(I,J,K)
     &                      - 2.0*(PIF_Y - PC_Y)*NYP(I,J,K)
     &                      - 2.0*(      - PC_DOUX)*DOUX_NP(K)
     &                      - 2.0*(      - PC_DOUY)*DOUY_NP(K)
 45      CONTINUE
         DR(J,LALFA,I) = DR(J,LALFA,I)
     &                 - 2.0*(    - PC_X)*NXA(I,J)
     &                 - 2.0*(    - PC_Y)*NYA(I,J)
C
	 DR(J,1,I) = DR(J,1,I) + 2.0*(PIFORC - PICIRC)
        ENDIF
C
C
        J = 1
        XXX = X(I,J) - XCENT
        YYY = Y(I,J) - YCENT
        CALL PIFAR(XXX, YYY, GAM, PSTOUT, HINF,
     &             CIRC, ALFA, MINF, QINF, SRCE, DOUX, DOUY,
     &             PIFF, P_X, P_Y, P_CIRC, P_ALFA, P_MSQ, P_QINF,
     &                   P_SRCE, P_DOUX, P_DOUY )
        P_MS = (P_MSQ + P_QINF*QI_MSQ)*MSQ_MS
        A2(J,I) = A2(J,I) + 2.0*(P_X*NX(I,J) + P_Y*NY(I,J))
        DR(J,LCIRC,I) = DR(J,LCIRC,I) + 2.0*P_CIRC
        DR(J,LALFA,I) = DR(J,LALFA,I) + 2.0*P_ALFA
        DR(J,LMASS,I) = DR(J,LMASS,I) + 2.0*P_MS
        DO 46 N = 1, NBL
          DR(J,LSBLE(N),I) = DR(J,LSBLE(N),I)
     &                     + 2.0*(P_X*NXG(I,J,N) + P_Y*NYG(I,J,N))
 46     CONTINUE
        DO 47 N = 1, NPOSN
          K = KPOSN(N)
          DR(J,LPOSN(K),I) = DR(J,LPOSN(K),I)
     &                     + 2.0*( P_X*NXP(I,J,K)
     &                           + P_Y*NYP(I,J,K)
     &                           + P_DOUX*DOUX_NP(K)
     &                           + P_DOUY*DOUY_NP(K) )
 47     CONTINUE
        DR(J,LALFA,I) = DR(J,LALFA,I)
     &                + 2.0*( P_X*NXA(I,J)
     &                      + P_Y*NYA(I,J) )
        DR(J,1,I) = DR(J,1,I) - 2.0*PIFF
C
        IF(LCPSET(2)) THEN
C------- add terms to residual to drive Cp to prescribed value
         CPFORC = SEVAL(X(I,J),CPFF(1,2),CPFFX(1,2),XFF(1,2),NFF(2))
         CPF_X  = DEVAL(X(I,J),CPFF(1,2),CPFFX(1,2),XFF(1,2),NFF(2))
         YFORC  = SEVAL(X(I,J), YFF(1,2), YFFX(1,2),XFF(1,2),NFF(2))
         YF_X   = DEVAL(X(I,J), YFF(1,2), YFFX(1,2),XFF(1,2),NFF(2))
         CALL PIFAR(XXX, YYY, GAM, PSTOUT, HINF,
     &          CIRCFF, ALFA, MINF, QINF, SRCE, DOUX, DOUY,
     &          PICIRC, PC_X, PC_Y, PC_CIRCFF, PC_ALFA, PC_MSQ, PC_QINF,
     &                  PC_SRCE, PC_DOUX, PC_DOUY )
         PC_MS = (PC_MSQ + PC_QINF*QI_MSQ)*MSQ_MS
C
         PIFORC = DPINF + QU*CPFORC + P_Y*(Y(I,J)-YFORC)
         PIF_X  =         QU*CPF_X  + P_Y*(      -YF_X )
         PIF_Y  =                     P_Y
         PIF_MS = (PI_MSQ + QU_MSQ*CPFORC)*MSQ_MS
C
         A2(J,I) = A2(J,I) + 2.0*(PIF_X - PC_X)*NX(I,J)
     &                     + 2.0*(PIF_Y - PC_Y)*NY(I,J)
         DR(J,LMASS,I) = DR(J,LMASS,I) + 2.0*(PIF_MS - PC_MS  )
         DR(J,LALFA,I) = DR(J,LALFA,I) + 2.0*(       - PC_ALFA)
         DO 48 N=1, NBL
           DR(J,LSBLE(N),I) = DR(J,LSBLE(N),I)
     &                      + 2.0*(PIF_X - PC_X)*NXG(I,J,N)
     &                      + 2.0*(PIF_Y - PC_Y)*NYG(I,J,N)
 48      CONTINUE
C
         DO 49 N = 1, NPOSN
           K = KPOSN(N)
           DR(J,LPOSN(K),I) = DR(J,LPOSN(K),I)
     &                      + 2.0*(PIF_X - PC_X)*NXP(I,J,K)
     &                      + 2.0*(PIF_Y - PC_Y)*NYP(I,J,K)
     &                      + 2.0*(      - PC_DOUX)*DOUX_NP(K)
     &                      + 2.0*(      - PC_DOUY)*DOUY_NP(K)
 49      CONTINUE
         DR(J,LALFA,I) = DR(J,LALFA,I)
     &                 + 2.0*(    - PC_X)*NXA(I,J)
     &                 + 2.0*(    - PC_Y)*NYA(I,J)
C
	 DR(J,1,I) = DR(J,1,I) - 2.0*(PIFORC - PICIRC)
        ENDIF
C
   50 CONTINUE
C
C---- set up inlet-outlet source and vortex conditions
      DO 60 J=1, JJ
C
        I = 1
        CALL CLROW(I,J)
C
        XXX = 0.5*(X(I+1,J) + X(I,J)) - XCENT
        YYY = 0.5*(Y(I+1,J) + Y(I,J)) - YCENT
        CALL PHIXY(XXX, YYY, GAM,
     &             CIRC, ALFA, MINF, SRCE, DOUX, DOUY,
     &    PHIX, PX_X, PX_Y, PX_CIRC, PX_ALFA, PX_DOUX, PX_DOUY, PX_MSQ,
     &    PHIY, PY_X, PY_Y, PY_CIRC, PY_ALFA, PY_DOUX, PY_DOUY, PY_MSQ )
C
        SX = X(I+1,J) - X(I,J)
        SY = Y(I+1,J) - Y(I,J)
        RES = SX*(PHIY+SINA) - SY*(PHIX+COSA)
        Z_XO = 0.5*(SX*PY_X - SY*PX_X) - (PHIY+SINA)
        Z_XP = 0.5*(SX*PY_X - SY*PX_X) + (PHIY+SINA)
        Z_YO = 0.5*(SX*PY_Y - SY*PX_Y) + (PHIX+COSA)
        Z_YP = 0.5*(SX*PY_Y - SY*PX_Y) - (PHIX+COSA)
        Z_ALFA = SX*COSA    + SY*SINA
        Z_DOUX = SX*PY_DOUX - SY*PX_DOUX
        Z_DOUY = SX*PY_DOUY - SY*PX_DOUY
        A2(J,I) = Z_XO*NX(I  ,J) + Z_YO*NY(I  ,J)
        C2(J,I) = Z_XP*NX(I+1,J) + Z_YP*NY(I+1,J)
        DR(J,LCIRC,I) = SX*PY_CIRC - SY*PX_CIRC
        DR(J,LALFA,I) = SX*PY_ALFA - SY*PX_ALFA + Z_ALFA
        DR(J,LMASS,I) =(SX*PY_MSQ  - SY*PX_MSQ ) * MSQ_MS
        DO 52 N = 1, NBL
          DR(J,LSBLE(N),I) = Z_XO*NXG(I  ,J,N) + Z_YO*NYG(I  ,J,N)
     &                     + Z_XP*NXG(I+1,J,N) + Z_YP*NYG(I+1,J,N)
   52   CONTINUE
        DO 54 N = 1, NPOSN
          K = KPOSN(N)
          DR(J,LPOSN(K),I) = Z_XO*NXP(I  ,J,K) + Z_YO*NYP(I  ,J,K)
     &                     + Z_XP*NXP(I+1,J,K) + Z_YP*NYP(I+1,J,K)
     &                     + Z_DOUX*DOUX_NP(K)
     &                     + Z_DOUY*DOUY_NP(K)
   54   CONTINUE
        DR(J,LALFA,I) = DR(J,LALFA,I)
     &                + Z_XO*NXA(I  ,J) + Z_YO*NYA(I  ,J)
     &                + Z_XP*NXA(I+1,J) + Z_YP*NYA(I+1,J)
        DR(J,1,I) = -RES
C
        I = II
        CALL CLROW(I,J)
C
        XXX = 0.5*(X(I-1,J) + X(I,J)) - XCENT
        YYY = 0.5*(Y(I-1,J) + Y(I,J)) - YCENT
        CALL PHIXY(XXX, YYY, GAM,
     &             CIRC, ALFA, MINF, SRCE, DOUX, DOUY,
     &    PHIX, PX_X, PX_Y, PX_CIRC, PX_ALFA, PX_DOUX, PX_DOUY, PX_MSQ,
     &    PHIY, PY_X, PY_Y, PY_CIRC, PY_ALFA, PY_DOUX, PY_DOUY, PY_MSQ )
C
        SX = X(I-1,J) - X(I,J)
        SY = Y(I-1,J) - Y(I,J)
        RES = SX*(PHIY+SINA) - SY*(PHIX+COSA)
        Z_XO = 0.5*(SX*PY_X - SY*PX_X) - (PHIY+SINA)
        Z_XM = 0.5*(SX*PY_X - SY*PX_X) + (PHIY+SINA)
        Z_YO = 0.5*(SX*PY_Y - SY*PX_Y) + (PHIX+COSA)
        Z_YM = 0.5*(SX*PY_Y - SY*PX_Y) - (PHIX+COSA)
        Z_ALFA = SX*COSA    + SY*SINA
        Z_DOUX = SX*PY_DOUX - SY*PX_DOUX
        Z_DOUY = SX*PY_DOUY - SY*PX_DOUY
        A2(J,I) = Z_XO*NX(I  ,J) + Z_YO*NY(I  ,J)
        B2(J,I) = Z_XM*NX(I-1,J) + Z_YM*NY(I-1,J)
        DR(J,LCIRC,I) = SX*PY_CIRC - SY*PX_CIRC
        DR(J,LALFA,I) = SX*PY_ALFA - SY*PX_ALFA + Z_ALFA
        DR(J,LMASS,I) =(SX*PY_MSQ  - SY*PX_MSQ ) * MSQ_MS
        DO 56 N = 1, NBL
          DR(J,LSBLE(N),I) = Z_XO*NXG(I  ,J,N) + Z_YO*NYG(I  ,J,N)
     &                     + Z_XM*NXG(I-1,J,N) + Z_YM*NYG(I-1,J,N)
   56   CONTINUE
        DO 58 N = 1, NPOSN
          K = KPOSN(N)
          DR(J,LPOSN(K),I) = Z_XO*NXP(I  ,J,K) + Z_YO*NYP(I  ,J,K)
     &                     + Z_XM*NXP(I-1,J,K) + Z_YM*NYP(I-1,J,K)
     &                     + Z_DOUX*DOUX_NP(K)
     &                     + Z_DOUY*DOUY_NP(K)
   58   CONTINUE
        DR(J,LALFA,I) = DR(J,LALFA,I)
     &                + Z_XO*NXA(I  ,J) + Z_YO*NYA(I  ,J)
     &                + Z_XM*NXA(I-1,J) + Z_YM*NYA(I-1,J)
        DR(J,1,I) = -RES
C
   60 CONTINUE
C
      RETURN
      END ! FFBC2
 
  
      SUBROUTINE FFBC3
C----------------------------------------------------------
C     Sets constant pressure far field (jet) BC's
C     Freestream pressure is imposed at J=1,JJ
C     Inlet and outlet angles are obtained from the
C     circulation dof CIRC.
C----------------------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MSES.INC'
C
C---- point from which gravitational pressure contribution is measured
C-    (for free-surface cases, RHOGEE .ne. 0)
      XCROT = 0.25
      YCROT = 0.0
ccc      XCROT = XCENT
ccc      YCROT = YCENT
C
C
C%%%   Changed XCENT,YCENT  to  XCROT,YCROT.     MD  23 Sep 94
C
      COSA = COS(ALFA)
      SINA = SIN(ALFA)
C
      RHOGEE = 0.0
C
CC---- constant for gravitational contribution to free-surface BC
C      RHOGEE = RHOINF*QINF**2 * 9.8*0.25/3.0**2      !  water in SI units
CC                                g    c /  V^2
C
C---- set freestream pressure sensitivity to mass flow
      PINF_MS = PI_MSQ/MS_MSQ
C
      DPINF = PINF - PSTOUT
      DO 30 I=2, II-1
        J = JJ
        DR(J,1,I)     = DR(J,1,I)     + 2.0*DPINF
        DR(J,LMASS,I) = DR(J,LMASS,I) - 2.0* PINF_MS
C
C------ add gravity contribution to top streamline BC  (constant REAL pressure)
        YBAR   =  ( Y(I,J)-YCROT)*COSA - ( X(I,J)-XCROT)*SINA
        YBAR_A = -( Y(I,J)-YCROT)*SINA - ( X(I,J)-XCROT)*COSA
     &           -(NYA(I,J)     )*SINA - (NXA(I,J)     )*COSA
        YBAR_N = -(NY(I,J)      )*SINA - (NX(I,J)      )*COSA
        DO 12 N=1, NPOSN
          K = KPOSN(N)
          DR(J,LPOSN(K),I) = DR(J,LPOSN(K),I)
     &           -(NYP(I,J,K)   )*SINA - (NXP(I,J,K)   )*COSA
 12     CONTINUE
        DR(J,1,I)     = DR(J,1,I)     + 2.0*RHOGEE*(YBAR  - YBTOP)
        DR(J,LALFA,I) = DR(J,LALFA,I) - 2.0*RHOGEE* YBAR_A
        A2(J,I)       = A2(J,I)       - 2.0*RHOGEE* YBAR_N
C
        J = 1
        DR(J,1,I)     = DR(J,1,I)     - 2.0*DPINF
        DR(J,LMASS,I) = DR(J,LMASS,I) + 2.0* PINF_MS
   30 CONTINUE
C
C---- set inlet and outlet angles
      DO 50 J = 1, JJ
C
C===================================================
C------ optional rotation to equalize free jet inlet & outlet angles
ccc        DA      = 0.5*CIRC/AINF
ccc        DA_CIRC = 0.5     /AINF
        DA      = 0.0
        DA_CIRC = 0.0
C===================================================
C
C------ outlet angle superimposed on alpha
        AO      = -CIRC/AINF + DA
        AO_CIRC = -1.0 /AINF + DA_CIRC
C
        RES = (X(2,J) - X(1,J))*(SINA+DA) - (Y(2,J) - Y(1,J))*COSA
        A2(J,1) = -NX(1,J)*(SINA+DA) + NY(1,J)*COSA
        C2(J,1) =  NX(2,J)*(SINA+DA) - NY(2,J)*COSA
        DO 32 N = 1, NBL
          DR(J,LSBLE(N),1) =-NXG(1,J,N)*(SINA+DA) + NYG(1,J,N)*COSA
     &                     + NXG(2,J,N)*(SINA+DA) - NYG(2,J,N)*COSA
   32   CONTINUE
        DO 33 N = 1, NPOSN
          K = KPOSN(N)
          DR(J,LPOSN(K),1) =-NXP(1,J,K)*(SINA+DA) + NYP(1,J,K)*COSA
     &                     + NXP(2,J,K)*(SINA+DA) - NYP(2,J,K)*COSA
   33   CONTINUE
        DR(J,LALFA,1) = (X(2,J)-X(1,J))*COSA + (Y(2,J)-Y(1,J))*SINA
     &                - NXA(1,J)*(SINA+DA) + NYA(1,J)*COSA
     &                + NXA(2,J)*(SINA+DA) - NYA(2,J)*COSA
        DR(J,LCIRC,1) = (X(2,J)-X(1,J))*DA_CIRC
        DR(J,1,1) = -RES
C
        IM = II-1
        RES = (X(II,J) - X(IM,J))*(SINA+AO) - (Y(II,J) - Y(IM,J))*COSA
        A2(J,II) = NX(II,J)*(SINA+AO) - NY(II,J)*COSA
        B2(J,II) =-NX(IM,J)*(SINA+AO) + NY(IM,J)*COSA
        DO 34 N = 1, NBL
          DR(J,LSBLE(N),II) = NXG(II,J,N)*(SINA+AO) - NYG(II,J,N)*COSA
     &                      - NXG(IM,J,N)*(SINA+AO) + NYG(IM,J,N)*COSA
   34   CONTINUE
        DO 35 N = 1, NPOSN
          K = KPOSN(N)
          DR(J,LPOSN(K),II) = NXP(II,J,K)*(SINA+AO) - NYP(II,J,K)*COSA
     &                      - NXP(IM,J,K)*(SINA+AO) + NYP(IM,J,K)*COSA
   35   CONTINUE
        DR(J,LALFA,II) = (X(II,J)-X(IM,J))*COSA + (Y(II,J)-Y(IM,J))*SINA
     &                 + NXA(II,J)*(SINA+AO) - NYA(II,J)*COSA
     &                 - NXA(IM,J)*(SINA+AO) + NYA(IM,J)*COSA
        DR(J,LCIRC,II) = (X(II,J)-X(IM,J))*AO_CIRC
        DR(J,1,II) = -RES
   50 CONTINUE
C
      RETURN
      END ! FFBC3
 

      SUBROUTINE FFBC4
C----------------------------------------------------------
C     Sets supersonic freestream far field  BC's
C     Pressure derived from the local flow angle
C     and far field flow is imposed at J=1, JJ
C----------------------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MSES.INC'
C
      COSA = COS(ALFA)
      SINA = SIN(ALFA)
C
CCC   PSTAR = PSTOUT * (1.0 + 0.5*GM1)**(-GAM/GM1)
C
      GRAT = SQRT(GM1/GP1)
      BET  = SQRT(MACHIN**2 - 1.0)
      RNU0 = ATAN(GRAT*BET)/GRAT - ATAN(BET)
C
      WM = -1.0
      WO =  1.0
      WP =  0.
C
C---- set up top Pi(theta) conditions
      J = JJ
      DO 10 I=2, II-1
        IP = I+1
        IM = I-1
C
        PB = ( (PI(I,J)+PSTOUT)/PSTOUT )**(-GM1/GAM)
        MSQ    = (2.0/GM1) * ( PB - 1.0 )
        MSQ_PI = -2.0/GAM * PB/(PI(I,J)+PSTOUT)
C
        IF(MSQ.GT.1.0) THEN
         BET    = SQRT(MSQ - 1.0)
         BET_PI = (0.5/BET) * MSQ_PI
         IF(I.GT.2 .AND. PI(I-1,J)+PSTOUT .GE. PSTAR)
     &    WRITE(*,*) 'Subsonic top boundary ends   at I = ', I
        ELSE
         BET    = 0.
         BET_PI = 0.
         IF(I.GT.2 .AND. PI(I-1,J)+PSTOUT .LT. PSTAR)
     &    WRITE(*,*) 'Subsonic top boundary starts at I = ', I
        ENDIF
C
        RNU     = ATAN(GRAT*BET)/GRAT - ATAN(BET)
        RNU_BET = 1.0/((MSQ-1.0)*GM1/GP1 + 1.0)  -  1.0/MSQ
        RNU_PI  = RNU_BET*BET_PI
C
        DX = WP*X(IP,J) + WO*X(I,J) + WM*X(IM,J)
        DY = WP*Y(IP,J) + WO*Y(I,J) + WM*Y(IM,J)
        THETA = ATAN2(DY,DX)
        TH_DX = -DY/(DX*DX+DY*DY)
        TH_DY =  DX/(DX*DX+DY*DY)
C
        RES = (THETA + RNU) - (ALFA + RNU0)
C
        Z_PI = RNU_PI
C
        IF(IFFBC.EQ.5) THEN
         RES = THETA - ALFA
         Z_PI = 0.
        ENDIF
C
        Z1(J,I) = 0.5*Z_PI*Z1(J,I)
        B1(J,I) = 0.5*Z_PI*B1(J,I)
        A1(J,I) = 0.5*Z_PI*A1(J,I)
        C1(J,I) = 0.5*Z_PI*C1(J,I)
C
        Z2(J,I) = 0.5*Z_PI*Z2(J,I)
        B2(J,I) = 0.5*Z_PI*B2(J,I) + WM*(TH_DX*NX(IM,J)+TH_DY*NY(IM,J))
        A2(J,I) = 0.5*Z_PI*A2(J,I) + WO*(TH_DX*NX(I ,J)+TH_DY*NY(I ,J))
        C2(J,I) = 0.5*Z_PI*C2(J,I) + WP*(TH_DX*NX(IP,J)+TH_DY*NY(IP,J))
C
        Z4(J,I) = 0.5*Z_PI*Z4(J,I)
        B4(J,I) = 0.5*Z_PI*B4(J,I)
        A4(J,I) = 0.5*Z_PI*A4(J,I)
C
        DR(J,1,I) = -RES
        DR(J,LALFA,I) = -1.0
        DR(J,LMASS,I) = 0.5*Z_PI*DR(J,LMASS,I)
        DO 6 N = 1, NBL
          DR(J,LMAS1(N),I) = 0.5*Z_PI*DR(J,LMAS1(N),I)
          DR(J,LSBLE(N),I) = WM*(TH_DX*NXG(IM,J,N)+TH_DY*NYG(IM,J,N))
     &                     + WO*(TH_DX*NXG(I ,J,N)+TH_DY*NYG(I ,J,N))
     &                     + WP*(TH_DX*NXG(IP,J,N)+TH_DY*NYG(IP,J,N))
    6   CONTINUE
        DO 8 N = 1, NPOSN
          K = KPOSN(N)
          DR(J,LPOSN(K),I) = WM*(TH_DX*NXP(IM,J,K)+TH_DY*NYP(IM,J,K))
     &                     + WO*(TH_DX*NXP(I ,J,K)+TH_DY*NYP(I ,J,K))
     &                     + WP*(TH_DX*NXP(IP,J,K)+TH_DY*NYP(IP,J,K))
    8   CONTINUE
        DR(J,LALFA,I) = DR(J,LALFA,I) 
     &                + WM*(TH_DX*NXA(IM,J)+TH_DY*NYA(IM,J))
     &                + WO*(TH_DX*NXA(I ,J)+TH_DY*NYA(I ,J))
     &                + WP*(TH_DX*NXA(IP,J)+TH_DY*NYA(IP,J))
C
CCC        CALL CLROW(I,J)
CCC        A2(J,I) = 1.
   10 CONTINUE
C
C---- set up bottom Pi(theta) conditions
      J = 1
      DO 20 I=2, II-1
        IP = I+1
        IM = I-1
C
        PB = ( (PI(I,J)+PSTOUT)/PSTOUT )**(-GM1/GAM)
        MSQ    = (2.0/GM1) * ( PB - 1.0 )
        MSQ_PI = -2.0/GAM * PB/(PI(I,J)+PSTOUT)
C
        IF(MSQ.GT.1.0) THEN
         BET    = SQRT(MSQ - 1.0)
         BET_PI = (0.5/BET) * MSQ_PI
         IF(I.GT.2 .AND. PI(I-1,J)+PSTOUT .GE. PSTAR)
     &    WRITE(*,*) 'Subsonic bottom boundary ends   at I = ', I
        ELSE
         BET    = 0.
         BET_PI = 0.
         IF(I.GT.2 .AND. PI(I-1,J)+PSTOUT .LT. PSTAR)
     &    WRITE(*,*) 'Subsonic bottom boundary starts at I = ', I
        ENDIF
C
        RNU     = ATAN(GRAT*BET)/GRAT - ATAN(BET)
        RNU_BET = 1.0/((MSQ-1.0)*GM1/GP1 + 1.0)  -  1.0/MSQ
        RNU_PI  = RNU_BET*BET_PI
C
        DX = WP*X(IP,J) + WO*X(I,J) + WM*X(IM,J)
        DY = WP*Y(IP,J) + WO*Y(I,J) + WM*Y(IM,J)
        THETA = ATAN2(DY,DX)
        TH_DX = -DY/(DX*DX+DY*DY)
        TH_DY =  DX/(DX*DX+DY*DY)
C
        RES = (THETA - RNU) - (ALFA - RNU0)
C
        Z_PI = -RNU_PI
C
        IF(IFFBC.EQ.5) THEN
         RES = THETA - ALFA
         Z_PI = 0.
        ENDIF
C
        Z2(J,I) = -.5*Z_PI*Z2(J,I)
        B2(J,I) = -.5*Z_PI*B2(J,I) + WM*(TH_DX*NX(IM,J)+TH_DY*NY(IM,J))
        A2(J,I) = -.5*Z_PI*A2(J,I) + WO*(TH_DX*NX(I ,J)+TH_DY*NY(I ,J))
        C2(J,I) = -.5*Z_PI*C2(J,I) + WP*(TH_DX*NX(IP,J)+TH_DY*NY(IP,J))
C
        Z3(J,I) = -.5*Z_PI*Z3(J,I)
        B3(J,I) = -.5*Z_PI*B3(J,I)
        A3(J,I) = -.5*Z_PI*A3(J,I)
        C3(J,I) = -.5*Z_PI*C3(J,I)
C
        Z5(J,I) = -.5*Z_PI*Z5(J,I)
        B5(J,I) = -.5*Z_PI*B5(J,I)
        A5(J,I) = -.5*Z_PI*A5(J,I)
C
        DR(J,1,I) = -RES
        DR(J,LALFA,I) = -1.0
        DR(J,LMASS,I) = -.5*Z_PI*DR(J,LMASS,I)
        DO 16 N = 1, NBL
          DR(J,LMAS1(N),I) = -0.5*Z_PI*DR(J,LMAS1(N),I)
          DR(J,LSBLE(N),I) = WM*(TH_DX*NXG(IM,J,N)+TH_DY*NYG(IM,J,N))
     &                     + WO*(TH_DX*NXG(I ,J,N)+TH_DY*NYG(I ,J,N))
     &                     + WP*(TH_DX*NXG(IP,J,N)+TH_DY*NYG(IP,J,N))
   16   CONTINUE
        DO 18 N = 1, NPOSN
          K = KPOSN(N)
          DR(J,LPOSN(K),I) = WM*(TH_DX*NXP(IM,J,K)+TH_DY*NYP(IM,J,K))
     &                     + WO*(TH_DX*NXP(I ,J,K)+TH_DY*NYP(I ,J,K))
     &                     + WP*(TH_DX*NXP(IP,J,K)+TH_DY*NYP(IP,J,K))
   18   CONTINUE
        DR(J,LALFA,I) = DR(J,LALFA,I) 
     &                + WM*(TH_DX*NXA(IM,J)+TH_DY*NYA(IM,J))
     &                + WO*(TH_DX*NXA(I ,J)+TH_DY*NYA(I ,J))
     &                + WP*(TH_DX*NXA(IP,J)+TH_DY*NYA(IP,J))
C
CCC        CALL CLROW(I,J)
CCC        A2(J,I) = 1.
   20 CONTINUE
C
C---- set capture area of topmost streamtube
      I = 1
      J = JJ
      JM = J-1
      CALL CLROW(I,J)
C
      DX = X(I,J) - X(I,JM)
      DY = Y(I,J) - Y(I,JM)
C
      RES = COSA*DY - SINA*DX - AINF*MFRACT(JM)
C
      A2(J,I) =  COSA*NY(I,J ) - SINA*NX(I,J )
      A1(J,I) = -COSA*NY(I,JM) + SINA*NX(I,JM)
C
      DR(J,1,I) = -RES
      DR(J,LALFA,I) = - SINA*DY - COSA*DX
      DO 22 N = 1, NBL
        DR(J,LSBLE(N),I) = COSA*NYG(I,J, N) - SINA*NXG(I,J, N)
     &                   - COSA*NYG(I,JM,N) + SINA*NXG(I,JM,N)
   22 CONTINUE
      DO 23 N = 1, NPOSN
        K = KPOSN(N)
        DR(J,LPOSN(K),I) = COSA*NYP(I,J, K) - SINA*NXP(I,J, K)
     &                   - COSA*NYP(I,JM,K) + SINA*NXP(I,JM,K)
   23 CONTINUE
      DR(J,LALFA,I) = DR(J,LALFA,I)
     &              + COSA*NYA(I,J ) - SINA*NXA(I,J )
     &              - COSA*NYA(I,JM) + SINA*NXA(I,JM)
C
C---- set capture area of bottommost streamtube
      J = 1
      JP = J+1
      CALL CLROW(I,J)
C
      DX = X(I,JP) - X(I,J)
      DY = Y(I,JP) - Y(I,J)
C
      RES = COSA*DY - SINA*DX - AINF*MFRACT(J)
C
      A2(J,I) = -COSA*NY(I,J ) + SINA*NX(I,J )
      A3(J,I) =  COSA*NY(I,JP) - SINA*NX(I,JP)
C
      DR(J,1,I) = -RES
      DR(J,LALFA,I) = - SINA*DY - COSA*DX
      DO 24 N = 1, NBL
        DR(J,LSBLE(N),I) = -COSA*NYG(I,J, N) + SINA*NXG(I,J, N)
     &                    + COSA*NYG(I,JP,N) - SINA*NXG(I,JP,N)
   24 CONTINUE
      DO 25 N = 1, NPOSN
        K = KPOSN(N)
        DR(J,LPOSN(K),I) = -COSA*NYP(I,J, K) + SINA*NXP(I,J, K)
     &                    + COSA*NYP(I,JP,K) - SINA*NXP(I,JP,K)
   25 CONTINUE
      DR(J,LALFA,I) = DR(J,LALFA,I)
     &              - COSA*NYA(I,J ) + SINA*NXA(I,J )
     &              + COSA*NYA(I,JP) - SINA*NXA(I,JP)
C
C
C---- set inlet slopes of all interior streamlines
      I = 1
      DO 36 J=2, JJ-1
C
        CALL CLROW(I,J)
C
        SX = X(I+1,J) - X(I,J)
        SY = Y(I+1,J) - Y(I,J)
        RES = SX*SINA - SY*COSA
        A2(J,I) = -SINA*NX(I  ,J) + COSA*NY(I  ,J)
        C2(J,I) =  SINA*NX(I+1,J) - COSA*NY(I+1,J)
        DR(J,LALFA,I) = SX*COSA + SY*SINA
        DO 32 N = 1, NBL
          DR(J,LSBLE(N),I) = -SINA*NXG(I  ,J,N) + COSA*NYG(I  ,J,N)
     &                     +  SINA*NXG(I+1,J,N) - COSA*NYG(I+1,J,N)
   32   CONTINUE
        DO 34 N = 1, NPOSN
          K = KPOSN(N)
          DR(J,LPOSN(K),I) = -SINA*NXP(I  ,J,K) + COSA*NYP(I  ,J,K)
     &                     +  SINA*NXP(I+1,J,K) - COSA*NYP(I+1,J,K)
   34   CONTINUE
        DR(J,LALFA,I) = DR(J,LALFA,I)
     &                - SINA*NXA(I  ,J) + COSA*NYA(I  ,J)
     &                + SINA*NXA(I+1,J) - COSA*NYA(I+1,J)
        DR(J,1,I) = -RES
   36 CONTINUE
C
C
C**** supersonic exit conditions...
C
C---- shift equations down for TE Kutta condition
      CALL SKUTTA
C
C---- set supersonic outlet conditions
      I = II-1
      DO 40 J=2, JJ-1
        IF(JSTAG(J).NE.0) GO TO 40
C
        V2(J,II) = V6(J,II)
        V3(J,II) = V7(J,II)
        Z2(J,II) = Z6(J,II)
        Z3(J,II) = Z7(J,II)
        B2(J,II) = B6(J,II)
        B3(J,II) = B7(J,II)
        A2(J,II) = A6(J,II)
        A3(J,II) = A7(J,II)
C
        V5(J,II) = V8(J,II)
        Z5(J,II) = Z8(J,II)
        B5(J,II) = B8(J,II)
C
        DR(J,LMASS,II) = DR(J+JJ,LMASS,II)
        DO 38 N = 1, NBL
          DR(J,LMAS1(N),II) = DR(J+JJ,LMAS1(N),II)
          DR(J,LSBLE(N),II) = DR(J+JJ,LSBLE(N),II)
   38   CONTINUE
C
        DR(J,1,II) = DR(J+JJ,1,II)
C
   40 CONTINUE
C
C---- extrapolate top and bottom streamlines
      IM = II-1
      IL = II-2
C
      JO = JJ
      DXO = X(II,JO) - X(IM,JO)
      DYO = Y(II,JO) - Y(IM,JO)
      DXM = X(IM,JO) - X(IL,JO)
      DYM = Y(IM,JO) - Y(IL,JO)
C
      RES = DXO*DYM - DYO*DXM
C
      Z2(JO,II) = -DXO*NY(IL,JO) + DYO*NX(IL,JO)
      B2(JO,II) =  DXO*NY(IM,JO) - DYO*NX(IM,JO)
     &           + DXM*NY(IM,JO) - DYM*NX(IM,JO)
      A2(JO,II) = -DXM*NY(II,JO) + DYM*NX(II,JO)
      DO 50 N=1, NPOSN
        K = KPOSN(N)
        DR(JO,LPOSN(K),II) = 
     &           - DXO*NYP(IL,JO,K) + DYO*NXP(IL,JO,K)
     &           + DXO*NYP(IM,JO,K) - DYO*NXP(IM,JO,K)
     &           + DXM*NYP(IM,JO,K) - DYM*NXP(IM,JO,K)
     &           - DXM*NYP(II,JO,K) + DYM*NXP(II,JO,K)
 50   CONTINUE
      DR(JO,LALFA,II) = 
     &           - DXO*NYA(IL,JO) + DYO*NXA(IL,JO)
     &           + DXO*NYA(IM,JO) - DYO*NXA(IM,JO)
     &           + DXM*NYA(IM,JO) - DYM*NXA(IM,JO)
     &           - DXM*NYA(II,JO) + DYM*NXA(II,JO)
      DR(JO,1,II) = -RES
C
      JO = 1
      DXO = X(II,JO) - X(IM,JO)
      DYO = Y(II,JO) - Y(IM,JO)
      DXM = X(IM,JO) - X(IL,JO)
      DYM = Y(IM,JO) - Y(IL,JO)
C
      RES = DXO*DYM - DYO*DXM
C
      Z2(JO,II) = -DXO*NY(IL,JO) + DYO*NX(IL,JO)
      B2(JO,II) =  DXO*NY(IM,JO) - DYO*NX(IM,JO)
     &           + DXM*NY(IM,JO) - DYM*NX(IM,JO)
      A2(JO,II) = -DXM*NY(II,JO) + DYM*NX(II,JO)
      DO 52 N=1, NPOSN
        K = KPOSN(N)
        DR(JO,LPOSN(K),II) = 
     &           - DXO*NYP(IL,JO,K) + DYO*NXP(IL,JO,K)
     &           + DXO*NYP(IM,JO,K) - DYO*NXP(IM,JO,K)
     &           + DXM*NYP(IM,JO,K) - DYM*NXP(IM,JO,K)
     &           - DXM*NYP(II,JO,K) + DYM*NXP(II,JO,K)
 52   CONTINUE
      DR(JO,LALFA,II) = 
     &           - DXO*NYA(IL,JO) + DYO*NXA(IL,JO)
     &           + DXO*NYA(IM,JO) - DYO*NXA(IM,JO)
     &           + DXM*NYA(IM,JO) - DYM*NXA(IM,JO)
     &           - DXM*NYA(II,JO) + DYM*NXA(II,JO)
      DR(JO,1,II) = -RES
C
C---- enforce j=1 and j=J streamlines to be parallel at exit
      N = 1
      IM = II-1
      JO = JS1(N)
      JP = JS2(N)
C
      DXO = X(II,JO) - X(IM,JO)
      DYO = Y(II,JO) - Y(IM,JO)
      DXP = X(II,JP) - X(IM,JP)
      DYP = Y(II,JP) - Y(IM,JP)
C
      RES = DXO*DYP - DYO*DXP
C
      B2(JO,II) = -DYP*NX(IM,JO) + DXP*NY(IM,JO)
      BT( 1,II) = -DXO*NY(IM,JP) + DYO*NX(IM,JP)
      A2(JO,II) =  DYP*NX(II,JO) - DXP*NY(II,JO)
      AT( 1,II) =  DXO*NY(II,JP) - DYO*NX(II,JP)
      DO 60 N=1, NPOSN
        K = KPOSN(K)
        DR(JO,LPOSN(K),II) = 
     &          - DYP*NXP(IM,JO,K) + DXP*NYP(IM,JO,K)
     &          + DYP*NXP(II,JO,K) - DXP*NYP(II,JO,K)
        DR( 1,LPOSN(K),II) = 
     &          - DXO*NYP(IM,JP,K) + DYO*NXP(IM,JP,K)
     &          + DXO*NYP(II,JP,K) - DYO*NXP(II,JP,K)
 60   CONTINUE
      DR(JO,LALFA,II) = 
     &          - DYP*NXA(IM,JO) + DXP*NYA(IM,JO)
     &          + DYP*NXA(II,JO) - DXP*NYA(II,JO)
      DR( 1,LALFA,II) = 
     &          - DXO*NYA(IM,JP) + DYO*NXA(IM,JP)
     &          + DXO*NYA(II,JP) - DYO*NXA(II,JP)
      DR(JO,1,II) = -RES
C
      RETURN
      END ! FFBC4




      SUBROUTINE DCORR
C-------------------------------------------------------
C     Updates the two doublet components DOUX and DOUY
C     by minimizing the difference between the farfield
C     potential gradient locus and streamline geometry
C-------------------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MSES.INC'
C
      SINA = SIN(ALFA)
      COSA = COS(ALFA)
      HOPI = 0.5/PIE
C
C---- set up least-squares 2x2 system for dDOUX, dDOUY
      ZX      = 0.
      ZX_DOUX = 0.
      ZX_DOUY = 0.
C
      ZY      = 0.
      ZY_DOUX = 0.
      ZY_DOUY = 0.
C
      DO 10 J=1, JJ, (JJ-1)
C
      DO 101 I=2, II
        IM = I-1
        XXX = 0.5*(X(I,J) + X(IM,J)) - XCENT
        YYY = 0.5*(Y(I,J) + Y(IM,J)) - YCENT
        DX = X(I,J) - X(IM,J)
        DY = Y(I,J) - Y(IM,J)
        DS = SQRT(DX*DX + DY*DY)
        SX = DX/DS
        SY = DY/DS
C
        CALL PHIXY(XXX, YYY, GAM,
     &             CIRC, ALFA, MINF, SRCE, DOUX, DOUY,
     &    PHIX, PX_X, PX_Y, PX_CIRC, PX_ALFA, PX_DOUX, PX_DOUY, PX_MSQ,
     &    PHIY, PY_X, PY_Y, PY_CIRC, PY_ALFA, PY_DOUX, PY_DOUY, PY_MSQ )
C
        CROSSP  = (PHIX+COSA)*SY    - (PHIY+SINA)*SX
        CP_DOUX = PX_DOUX*SY - PY_DOUX*SX
        CP_DOUY = PX_DOUY*SY - PY_DOUY*SX
C
        ZX      = ZX      + CP_DOUX*CROSSP  * DS
        ZX_DOUX = ZX_DOUX + CP_DOUX**2      * DS
        ZX_DOUY = ZX_DOUY + CP_DOUX*CP_DOUY * DS
C
        ZY      = ZY      + CP_DOUY*CROSSP  * DS
        ZY_DOUX = ZY_DOUX + CP_DOUY*CP_DOUX * DS
        ZY_DOUY = ZY_DOUY + CP_DOUY**2      * DS
C
  101 CONTINUE
   10 CONTINUE
C
C---- solve 2x2 system by Cramer's rule
      A11 = ZX_DOUX
      A12 = ZX_DOUY
      A21 = ZY_DOUX
      A22 = ZY_DOUY
C
      DET = A11*A22 - A12*A21
C
      DDX = -( ZX*A22 - A12*ZY )/DET
      DDY = -(A11*ZY  -  ZX*A21)/DET
C
cc      write(*,7777) doux, ddx, douy, ddy
cc 7777 format(/1x,'    Dx dDx =', 2f10.5,3x,'   Dy dDy =', 2f10.5)
c
C---- underrelax doublet strength changes
      DOUX = DOUX + 0.8*DDX
      DOUY = DOUY + 0.4*DDY
C
      RETURN
      END ! DCORR


      SUBROUTINE ELCIRC(CIRCB,XCIRCB,YCIRCB)
C---------------------------------------------
C     Calculates element circulations.
C---------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MSES.INC'
      DIMENSION CIRCB(NBX), XCIRCB(NBX), YCIRCB(NBX)
C
      DO 100 N=1, NBL
C
        J1 = JS1(N)
        J2 = JS2(N)
C
        CIRCB(N) = 0.0
        XCIRCB(N) = 0.0
        YCIRCB(N) = 0.0
C
        DO 10 I=ILEB(N), ITEB(N)-1
          IO = I
          IP = I+1
C
          JO = J1
          JP = J1+1
          X1 = 0.25*(X(IP,JP)+X(IP,JO) + X(IO,JP)+X(IO,JO))
          Y1 = 0.25*(Y(IP,JP)+Y(IP,JO) + Y(IO,JP)+Y(IO,JO))
          DX = 0.50*(X(IP,JP)+X(IP,JO) - X(IO,JP)-X(IO,JO))
          DY = 0.50*(Y(IP,JP)+Y(IP,JO) - Y(IO,JP)-Y(IO,JO))
          DS1 = SQRT(DX*DX + DY*DY)
          Q1  = Q(IO,JO)/QINF
C
          JO = J2-1
          JP = J2
          X2 = 0.25*(X(IP,JP)+X(IP,JO) + X(IO,JP)+X(IO,JO))
          Y2 = 0.25*(Y(IP,JP)+Y(IP,JO) + Y(IO,JP)+Y(IO,JO))
          DX = 0.50*(X(IP,JP)+X(IP,JO) - X(IO,JP)-X(IO,JO))
          DY = 0.50*(Y(IP,JP)+Y(IP,JO) - Y(IO,JP)-Y(IO,JO))
          DS2 = SQRT(DX*DX + DY*DY)
          Q2  = Q(IO,JO)/QINF
C
          CIRCB(N)  = CIRCB(N)  + Q1*DS1    - Q2*DS2
          XCIRCB(N) = XCIRCB(N) + Q1*DS1*X1 - Q2*DS2*X2
          YCIRCB(N) = YCIRCB(N) + Q1*DS1*Y1 - Q2*DS2*Y2
 10     CONTINUE
 100  CONTINUE
C
      RETURN
      END ! ELCIRC
